/* test.c
 *
 * Copyright (C) 2006-2023 wolfSSL Inc.
 *
 * This file is part of wolfSSL.
 *
 * wolfSSL is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * wolfSSL is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1335, USA
 */

/*
 * Some common, optional build settings:
 * these can also be set in wolfssl/options.h or user_settings.h
 * -------------------------------------------------------------
 *
 * set the default devId for cryptocb to the value instead of INVALID_DEVID
 * WC_USE_DEVID=0x1234
 */

#ifdef HAVE_CONFIG_H
    #include <config.h>
#endif

#ifndef WOLFSSL_USER_SETTINGS
    #include <wolfssl/options.h>
#endif
#include <wolfssl/wolfcrypt/settings.h>

#ifndef NO_CRYPT_TEST

#include <wolfssl/version.h>
#include <wolfssl/wolfcrypt/types.h>
#include <wolfssl/wolfcrypt/wc_port.h>
#include <wolfssl/wolfcrypt/mem_track.h>

#if defined(HAVE_WOLFCRYPT_TEST_OPTIONS)
    #include <wolfssl/ssl.h>
    #define err_sys err_sys_remap /* remap err_sys */
    #include <wolfssl/test.h>
    #undef err_sys
#endif

#if defined(WC_ECC_NONBLOCK) && defined(WOLFSSL_PUBLIC_MP) && \
    defined(HAVE_ECC_SIGN) && defined(HAVE_ECC_VERIFY)
    #include <stdint.h>
#endif

#ifdef HAVE_STACK_SIZE_VERBOSE
#ifdef WOLFSSL_TEST_MAX_RELATIVE_STACK_BYTES
    static WC_MAYBE_UNUSED ssize_t max_relative_stack =
        WOLFSSL_TEST_MAX_RELATIVE_STACK_BYTES;
#else
    static WC_MAYBE_UNUSED ssize_t max_relative_stack = -1;
#endif
#endif

const byte const_byte_array[] = "A+Gd\0\0\0";
#define CBPTR_EXPECTED 'A'

#if defined(WOLFSSL_TRACK_MEMORY_VERBOSE) && !defined(WOLFSSL_STATIC_MEMORY)
#ifdef WOLFSSL_TEST_MAX_RELATIVE_HEAP_ALLOCS
    static ssize_t max_relative_heap_allocs = WOLFSSL_TEST_MAX_RELATIVE_HEAP_ALLOCS;
#else
    static ssize_t max_relative_heap_allocs = -1;
#endif
#ifdef WOLFSSL_TEST_MAX_RELATIVE_HEAP_BYTES
    static ssize_t max_relative_heap_bytes = WOLFSSL_TEST_MAX_RELATIVE_HEAP_BYTES;
#else
    static ssize_t max_relative_heap_bytes = -1;
#endif
#define PRINT_HEAP_CHECKPOINT() {                                            \
    const ssize_t _rha = wolfCrypt_heap_peakAllocs_checkpoint() - heap_baselineAllocs; \
    const ssize_t _rhb = wolfCrypt_heap_peakBytes_checkpoint() - heap_baselineBytes;   \
    printf("    relative heap peak usage: %ld alloc%s, %ld bytes\n",         \
           (long int)_rha,                                                   \
           _rha == 1 ? "" : "s",                                             \
           (long int)_rhb);                                                  \
    if ((max_relative_heap_allocs > 0) && (_rha > max_relative_heap_allocs)) \
        return err_sys("heap allocs exceed designated max.",                 \
                       WC_TEST_RET_ENC_NC);                                  \
    if ((max_relative_heap_bytes > 0) && (_rhb > max_relative_heap_bytes))   \
        return err_sys("heap bytes exceed designated max.",                  \
                       WC_TEST_RET_ENC_NC);                                  \
    heap_baselineAllocs = wolfCrypt_heap_peakAllocs_checkpoint();            \
    heap_baselineBytes = wolfCrypt_heap_peakBytes_checkpoint();              \
    }
#else
#define PRINT_HEAP_CHECKPOINT() WC_DO_NOTHING
#endif /* WOLFSSL_TRACK_MEMORY_VERBOSE && !WOLFSSL_STATIC_MEMORY */

#ifdef USE_FLAT_TEST_H
    #ifdef HAVE_CONFIG_H
        #include "test_paths.h"
    #endif
    #include "test.h"
#else
    #ifdef HAVE_CONFIG_H
        #include "wolfcrypt/test/test_paths.h"
    #endif
    #include "wolfcrypt/test/test.h"
#endif

/* printf mappings */
#ifndef WOLFSSL_LOG_PRINTF
#if defined(FREESCALE_MQX) || defined(FREESCALE_KSDK_MQX)
    #include <mqx.h>
    #include <stdlib.h>
    /* see wc_port.h for fio.h and nio.h includes */
#elif defined(FREESCALE_KSDK_BM)
    #include "fsl_debug_console.h"
    #undef printf
    #define printf PRINTF
#elif defined(WOLFSSL_APACHE_MYNEWT)
    #include <assert.h>
    #include <string.h>
    #include "sysinit/sysinit.h"
    #include "os/os.h"
    #ifdef ARCH_sim
    #include "mcu/mcu_sim.h"
    #endif
    #include "os/os_time.h"
#elif defined(WOLFSSL_ESPIDF)
    #include <time.h>
    #include <sys/time.h>
    #include <esp_log.h>
    #include <wolfcrypt/port/Espressif/esp32-crypt.h> /* */
#elif defined(WOLFSSL_ZEPHYR)
    #include <stdio.h>

    #define printf printk
#elif defined(MICRIUM)
    #include <os.h>
    #if (OS_VERSION < 50000)
        #include <bsp_ser.h>
        void BSP_Ser_Printf (CPU_CHAR* format, ...);
        #undef printf
        #define printf BSP_Ser_Printf
    #else
        #include <stdio.h>
    #endif
#elif defined(WOLFSSL_PB)
    #include <stdarg.h>
    int wolfssl_pb_print(const char*, ...);
    #undef printf
    #define printf wolfssl_pb_print
#elif defined(WOLFSSL_TELIT_M2MB)
    #include "wolfssl/wolfcrypt/wc_port.h" /* for m2mb headers */
    #include "m2m_log.h" /* for M2M_LOG_INFO - not standard API */
    /* remap printf */
    #undef printf
    #define printf M2M_LOG_INFO
    /* OS requires occasional sleep() */
    #ifndef TEST_SLEEP_MS
        #define TEST_SLEEP_MS 50
    #endif
    #define TEST_SLEEP() m2mb_os_taskSleep(M2MB_OS_MS2TICKS(TEST_SLEEP_MS))
    /* don't use file system for these tests, since ./certs dir isn't loaded */
    #undef  NO_FILESYSTEM
    #define NO_FILESYSTEM
#elif defined(THREADX) && !defined(WOLFSSL_WICED) && \
      !defined(THREADX_NO_DC_PRINTF)
    #ifndef NETOS
        /* since just testing, use THREADX log printf instead (NETOS prototypes
         * this elsewhere) */
        int dc_log_printf(char*, ...);
    #endif
    #undef printf
    #define printf dc_log_printf
#elif defined(ANDROID)
    #ifdef XMALLOC_USER
        #include <stdlib.h>  /* we're using malloc / free direct here */
    #endif
    #ifndef STRING_USER
        #include <stdio.h>
    #endif
    #include <android/log.h>

    #ifdef ANDROID_V454 /* See fips/android/wolfCrypt_v454_android */
        #ifndef NO_FILESYSTEM
            #define NO_FILESYSTEM /* Turn off tests that want to call SaveDerAndPem() */
        #endif
    #else
        #define printf(...)       \
              __android_log_print(ANDROID_LOG_DEBUG, "[WOLFCRYPT]", __VA_ARGS__)
        #define fprintf(fp, ...)  \
              __android_log_print(ANDROID_LOG_DEBUG, "[WOLFCRYPT]", __VA_ARGS__)
    #endif
#elif defined(WOLFSSL_DEOS)
    #include <printx.h>
    #undef printf
    #define printf printx
#elif defined(WOLFSSL_RENESAS_RSIP)
    #ifndef TEST_SLEEP
        #define TEST_SLEEP() vTaskDelay(50)
    #endif
    #undef vprintf
    #define vprintf rsip_vprintf
    #include <stdarg.h> /* for var args */
    int rsip_vprintf(const char* restrict format, va_list args)
    {
        int ret;
        char tmpBuf[80];

        ret = XSNPRINTF(tmpBuf, sizeof(tmpBuf), format, args);
        printf(tmpBuf);

    return ret;
    }
#else
    #ifdef XMALLOC_USER
        #include <stdlib.h>  /* we're using malloc / free direct here */
    #endif
    #if !defined(STRING_USER) && !defined(WOLFSSL_LINUXKM)
        #include <stdio.h>
    #endif

    #if defined(WOLFSSL_LINUXKM) && !defined(WOLFSSL_LINUXKM_VERBOSE_DEBUG)
        #undef printf
        #define printf(...) ({})
    #endif

    /* enable way for customer to override test/bench printf */
    #ifdef XPRINTF
        #undef  printf
        #define printf XPRINTF
    #elif !defined(printf)
        /* arrange for printf() to flush after every message -- this assures
         * redirected output (to a log file) records progress right up to the
         * moment of a crash/abort(); otherwise anything queued in stdout would
         * be lost.
         */
        #define printf(...) ( printf(__VA_ARGS__), fflush(stdout) )
    #endif
#endif
#endif /* !WOLFSSL_LOG_PRINTF */

#include <wolfssl/wolfcrypt/memory.h>
#include <wolfssl/wolfcrypt/wc_port.h>
#include <wolfssl/wolfcrypt/logging.h>
#include <wolfssl/wolfcrypt/types.h>
#include <wolfssl/wolfcrypt/asn.h>
#include <wolfssl/wolfcrypt/md2.h>
#include <wolfssl/wolfcrypt/md5.h>
#include <wolfssl/wolfcrypt/md4.h>
#include <wolfssl/wolfcrypt/sha.h>
#include <wolfssl/wolfcrypt/sha256.h>
#include <wolfssl/wolfcrypt/sha512.h>
#include <wolfssl/wolfcrypt/rc2.h>
#include <wolfssl/wolfcrypt/arc4.h>
#if !defined(WC_NO_RNG)
    #include <wolfssl/wolfcrypt/random.h>
#endif
#include <wolfssl/wolfcrypt/wolfmath.h>
#include <wolfssl/wolfcrypt/coding.h>
#include <wolfssl/wolfcrypt/signature.h>
#include <wolfssl/wolfcrypt/rsa.h>
#include <wolfssl/wolfcrypt/des3.h>
#include <wolfssl/wolfcrypt/aes.h>
#include <wolfssl/wolfcrypt/wc_encrypt.h>
#include <wolfssl/wolfcrypt/cmac.h>
#include <wolfssl/wolfcrypt/siphash.h>
#include <wolfssl/wolfcrypt/poly1305.h>
#include <wolfssl/wolfcrypt/camellia.h>
#include <wolfssl/wolfcrypt/hmac.h>
#include <wolfssl/wolfcrypt/kdf.h>
#include <wolfssl/wolfcrypt/dh.h>
#include <wolfssl/wolfcrypt/dsa.h>
#include <wolfssl/wolfcrypt/srp.h>
#include <wolfssl/wolfcrypt/chacha.h>
#include <wolfssl/wolfcrypt/chacha20_poly1305.h>
#include <wolfssl/wolfcrypt/pwdbased.h>
#include <wolfssl/wolfcrypt/ripemd.h>
#include <wolfssl/wolfcrypt/error-crypt.h>
#ifdef HAVE_ECC
    #include <wolfssl/wolfcrypt/ecc.h>
#endif
#ifdef WOLFSSL_SM2
    #include <wolfssl/wolfcrypt/sm2.h>
#endif
#ifdef HAVE_HPKE
    #include <wolfssl/wolfcrypt/hpke.h>
#endif
#ifdef HAVE_CURVE25519
    #include <wolfssl/wolfcrypt/curve25519.h>
#endif
#ifdef HAVE_ED25519
    #include <wolfssl/wolfcrypt/ed25519.h>
#endif
#ifdef HAVE_CURVE448
    #include <wolfssl/wolfcrypt/curve448.h>
#endif
#ifdef HAVE_ED448
    #include <wolfssl/wolfcrypt/ed448.h>
#endif
#ifdef WOLFSSL_HAVE_KYBER
    #include <wolfssl/wolfcrypt/kyber.h>
#ifdef WOLFSSL_WC_KYBER
    #include <wolfssl/wolfcrypt/wc_kyber.h>
#endif
#if defined(HAVE_LIBOQS) || defined(HAVE_PQM4)
    #include <wolfssl/wolfcrypt/ext_kyber.h>
#endif
#endif
#if defined(WOLFSSL_HAVE_XMSS)
    #include <wolfssl/wolfcrypt/xmss.h>
#ifdef HAVE_LIBXMSS
    #include <wolfssl/wolfcrypt/ext_xmss.h>
#endif
#endif
#if defined(WOLFSSL_HAVE_LMS)
    #include <wolfssl/wolfcrypt/lms.h>
#ifdef HAVE_LIBLMS
    #include <wolfssl/wolfcrypt/ext_lms.h>
#endif
#endif
#ifdef WOLFCRYPT_HAVE_ECCSI
    #include <wolfssl/wolfcrypt/eccsi.h>
#endif
#ifdef WOLFCRYPT_HAVE_SAKKE
    #include <wolfssl/wolfcrypt/sakke.h>
#endif
#if defined(HAVE_BLAKE2) || defined(HAVE_BLAKE2S)
    #include <wolfssl/wolfcrypt/blake2.h>
#endif
#ifdef WOLFSSL_SHA3
    #include <wolfssl/wolfcrypt/sha3.h>
#endif
#ifdef WOLFSSL_SM3
    #include <wolfssl/wolfcrypt/sm3.h>
#endif
#ifdef WOLFSSL_SM4
    #include <wolfssl/wolfcrypt/sm4.h>
#endif
#ifdef HAVE_LIBZ
    #include <wolfssl/wolfcrypt/compress.h>
#endif
#ifdef HAVE_PKCS7
    #include <wolfssl/wolfcrypt/pkcs7.h>
#endif
#ifdef HAVE_FIPS
    #include <wolfssl/wolfcrypt/fips_test.h>
#endif
#ifdef HAVE_SELFTEST
    #include <wolfssl/wolfcrypt/selftest.h>
#endif
#ifdef WOLFSSL_ASYNC_CRYPT
    #include <wolfssl/wolfcrypt/async.h>
#endif
#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
    #include <wolfssl/wolfcrypt/logging.h>
#endif
#ifdef WOLFSSL_CAAM
    #include <wolfssl/wolfcrypt/port/caam/wolfcaam.h>
#endif
#ifdef WOLF_CRYPTO_CB
    #include <wolfssl/wolfcrypt/cryptocb.h>
    #ifdef HAVE_INTEL_QA_SYNC
        #include <wolfssl/wolfcrypt/port/intel/quickassist_sync.h>
    #endif
    #ifdef HAVE_CAVIUM_OCTEON_SYNC
        #include <wolfssl/wolfcrypt/port/cavium/cavium_octeon_sync.h>
    #endif
    #ifdef HAVE_RENESAS_SYNC
        #include <wolfssl/wolfcrypt/port/renesas/renesas_sync.h>
    #endif
#endif

#ifdef _MSC_VER
    /* 4996 warning to use MS extensions e.g., strcpy_s instead of strncpy */
    #pragma warning(disable: 4996)
#endif

#ifdef OPENSSL_EXTRA
  #ifndef WOLFCRYPT_ONLY
    #include <wolfssl/openssl/evp.h>
    #include <wolfssl/openssl/hmac.h>
  #endif
    #include <wolfssl/openssl/rand.h>
    #include <wolfssl/openssl/aes.h>
    #include <wolfssl/openssl/des.h>
#endif

#if defined(NO_FILESYSTEM) || defined(WC_NO_RNG)
    #if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
        !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
        #define USE_CERT_BUFFERS_2048
    #endif
    #if !defined(USE_CERT_BUFFERS_256)
        #define USE_CERT_BUFFERS_256
    #endif
#endif

#if defined(WOLFSSL_CERT_GEN) && (defined(HAVE_ECC384) || defined(HAVE_ALL_CURVES))
    #define ENABLE_ECC384_CERT_GEN_TEST
#endif

#include <wolfssl/certs_test.h>

#ifdef DEVKITPRO
    #include <wiiuse/wpad.h>
#endif

#ifndef WOLFSSL_HAVE_ECC_KEY_GET_PRIV
    /* FIPS build has replaced ecc.h. */
    #define wc_ecc_key_get_priv(key) (&((key)->k))
    #define WOLFSSL_HAVE_ECC_KEY_GET_PRIV
#endif

#ifdef WOLFSSL_STATIC_MEMORY
    static WOLFSSL_HEAP_HINT* HEAP_HINT;
#else
    #define HEAP_HINT NULL
#endif /* WOLFSSL_STATIC_MEMORY */

/* these cases do not have intermediate hashing support */
#if (defined(WOLFSSL_AFALG_XILINX_SHA3) && !defined(WOLFSSL_AFALG_HASH_KEEP)) \
        && !defined(WOLFSSL_XILINX_CRYPT) || defined(WOLFSSL_XILINX_CRYPT_VERSAL)
    #define NO_INTM_HASH_TEST
#endif

#if defined(WOLFSSL_RENESAS_TSIP) || defined(WOLFSSL_RENESAS_SCEPROTECT) || \
    defined(WOLFSSL_RENESAS_SCEPROTECT_CRYPTONLY) || \
    defined(WOLFSSL_SECO_CAAM)
    #define HASH_SIZE_LIMIT
#endif

#if defined(WOLFSSL_CERT_GEN) && (!defined(NO_RSA) || defined(HAVE_ECC)) || \
  (defined(WOLFSSL_TEST_CERT) && (defined(HAVE_ED25519) || defined(HAVE_ED448)))
static void initDefaultName(void);
#endif

/* for async devices */
#ifdef WOLFSSL_CAAM_DEVID
static int devId = WOLFSSL_CAAM_DEVID;
#else
  #ifdef WC_USE_DEVID
static int devId = WC_USE_DEVID;
  #else
static int devId = INVALID_DEVID;
  #endif
#endif

#ifdef HAVE_WNR
    const char* wnrConfigFile = "wnr-example.conf";
#endif

#define TEST_STRING    "Everyone gets Friday off."
#define TEST_STRING_SZ 25

typedef struct testVector {
    const char*  input;
    const char*  output;
    size_t inLen;
    size_t outLen;
} testVector;

#ifndef WOLFSSL_TEST_SUBROUTINE
#define WOLFSSL_TEST_SUBROUTINE
#endif

PRAGMA_GCC("GCC diagnostic ignored \"-Wunused-function\"")
PRAGMA_CLANG("clang diagnostic ignored \"-Wunused-function\"")

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  error_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  base64_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  base16_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  asn_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  md2_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  md5_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  md4_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sha_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sha224_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sha256_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sha512_test(void);
#if !defined(WOLFSSL_NOSHA512_224) && \
   (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sha512_224_test(void);
#endif
#if !defined(WOLFSSL_NOSHA512_256) && \
   (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sha512_256_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sha384_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sha3_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  shake128_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  shake256_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sm3_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hash_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hmac_md5_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hmac_sha_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hmac_sha224_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hmac_sha256_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hmac_sha384_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hmac_sha512_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hmac_sha3_test(void);
#if defined(HAVE_HKDF) && !defined(NO_HMAC)
#if defined(WOLFSSL_AFALG_XILINX) || defined(WOLFSSL_AFALG_XILINX_AES) ||     \
    defined(WOLFSSL_AFALG_XILINX_SHA3) || defined(WOLFSSL_AFALG_HASH_KEEP) || \
    defined(WOLFSSL_AFALG_XILINX_RSA)
/* hkdf_test has issue with WOLFSSL_TEST_SUBROUTINE set on Xilinx with afalg */
static                  wc_test_ret_t  hkdf_test(void);
#else
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hkdf_test(void);
#endif
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sshkdf_test(void);
#ifdef WOLFSSL_TLS13
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  tls13_kdf_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  x963kdf_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  hpke_test(void);
#ifdef WC_SRTP_KDF
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  srtpkdf_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  arc4_test(void);
#ifdef WC_RC2
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  rc2_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  chacha_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  XChaCha_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  chacha20_poly1305_aead_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  XChaCha20Poly1305_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  des_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  des3_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  aes_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  aes192_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  aes256_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  aesofb_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  cmac_test(void);
#if defined(WOLFSSL_SIPHASH)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  siphash_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  poly1305_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  aesgcm_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  aesgcm_default_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  gmac_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  aesccm_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  aeskeywrap_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  camellia_test(void);
#ifdef WOLFSSL_SM4
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sm4_test(void);
#endif
#ifdef WC_RSA_NO_PADDING
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  rsa_no_pad_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  rsa_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  dh_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  dsa_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  srp_test(void);
#ifndef WC_NO_RNG
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  random_test(void);
#endif /* WC_NO_RNG */
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  pwdbased_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  ripemd_test(void);
#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  openssl_test(void);   /* test mini api */

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  openssl_pkey0_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  openssl_pkey1_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  openSSL_evpMD_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  openssl_evpSig_test(void);
#endif

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pbkdf1_test(void);
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pkcs12_test(void);
#if defined(HAVE_PBKDF2) && !defined(NO_SHA256) && !defined(NO_HMAC)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pbkdf2_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t scrypt_test(void);
#ifdef HAVE_ECC
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  ecc_test(void);
    #if defined(HAVE_ECC_ENCRYPT) && defined(HAVE_AES_CBC) && \
        (defined(WOLFSSL_AES_128) || defined(WOLFSSL_AES_256))
        WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  ecc_encrypt_test(void);
    #endif
    #if defined(USE_CERT_BUFFERS_256) && !defined(WOLFSSL_ATECC508A) && \
        !defined(WOLFSSL_ATECC608A) && !defined(NO_ECC256) && \
        defined(HAVE_ECC_VERIFY) && defined(HAVE_ECC_SIGN) && \
        !defined(WOLF_CRYPTO_CB_ONLY_ECC) && !defined(NO_ECC_SECP)
        /* skip for ATECC508/608A, cannot import private key buffers */
        WOLFSSL_TEST_SUBROUTINE wc_test_ret_t ecc_test_buffers(void);
    #endif
#endif
#ifdef HAVE_CURVE25519
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  curve25519_test(void);
#endif
#ifdef HAVE_ED25519
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  ed25519_test(void);
#endif
#ifdef HAVE_CURVE448
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  curve448_test(void);
#endif
#ifdef HAVE_ED448
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  ed448_test(void);
#endif
#ifdef WOLFSSL_HAVE_KYBER
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  kyber_test(void);
#endif
#if defined(WOLFSSL_HAVE_XMSS)
    #if !defined(WOLFSSL_XMSS_VERIFY_ONLY)
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  xmss_test(void);
    #endif
    #if defined(WOLFSSL_XMSS_VERIFY_ONLY) && !defined(WOLFSSL_SMALL_STACK)
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  xmss_test_verify_only(void);
    #endif
#endif
#if defined(WOLFSSL_HAVE_LMS)
    #if !defined(WOLFSSL_LMS_VERIFY_ONLY)
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  lms_test(void);
    #endif
    #if defined(WOLFSSL_LMS_VERIFY_ONLY) && !defined(WOLFSSL_SMALL_STACK)
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  lms_test_verify_only(void);
    #endif
#endif
#ifdef WOLFCRYPT_HAVE_ECCSI
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  eccsi_test(void);
#endif
#ifdef WOLFCRYPT_HAVE_SAKKE
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  sakke_test(void);
#endif
#ifdef HAVE_BLAKE2
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  blake2b_test(void);
#endif
#ifdef HAVE_BLAKE2S
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  blake2s_test(void);
#endif
#ifdef HAVE_LIBZ
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t compress_test(void);
#endif
#ifdef HAVE_PKCS7
    #ifndef NO_PKCS7_ENCRYPTED_DATA
        WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pkcs7encrypted_test(void);
    #endif
    #if defined(HAVE_LIBZ) && !defined(NO_PKCS7_COMPRESSED_DATA)
        WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pkcs7compressed_test(void);
    #endif
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pkcs7signed_test(void);
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pkcs7enveloped_test(void);
    #if defined(HAVE_AESGCM) || defined(HAVE_AESCCM)
        WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pkcs7authenveloped_test(void);
    #endif
    #if !defined(NO_AES) && defined(HAVE_AES_CBC)
        WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pkcs7callback_test(byte* cert, word32 certSz, byte* key,
                word32 keySz);
    #endif
#endif
#if !defined(NO_ASN_TIME) && !defined(NO_RSA) && defined(WOLFSSL_TEST_CERT) && \
    !defined(NO_FILESYSTEM)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t cert_test(void);
#endif
#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_TEST_CERT) && \
   !defined(NO_FILESYSTEM) && defined(WOLFSSL_CERT_GEN)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t  certext_test(void);
#endif
#if defined(WOLFSSL_CERT_GEN_CACHE) && defined(WOLFSSL_TEST_CERT) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t decodedCertCache_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t memory_test(void);
#if defined(WOLFSSL_PUBLIC_MP) && \
    ((defined(WOLFSSL_SP_MATH_ALL) && !defined(WOLFSSL_RSA_VERIFY_ONLY)) || \
     defined(USE_FAST_MATH))
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t mp_test(void);
#endif
#if defined(WOLFSSL_PUBLIC_MP) && defined(WOLFSSL_KEY_GEN)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t prime_test(void);
#endif
#if defined(ASN_BER_TO_DER) && \
    (defined(WOLFSSL_TEST_CERT) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_EXTRA_X509_SMALL))
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t berder_test(void);
#endif
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t logging_test(void);
#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t time_test(void);
#endif
#if defined(__INCLUDE_NUTTX_CONFIG_H)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t wolfcrypt_mutex_test(void);
#else
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t mutex_test(void);
#endif
#if defined(USE_WOLFSSL_MEMORY) && !defined(FREERTOS)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t memcb_test(void);
#endif
#ifdef WOLFSSL_CAAM_BLOB
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t blob_test(void);
#endif
#ifdef HAVE_ARIA
#include "wolfssl/wolfcrypt/port/aria/aria-crypt.h"
void printOutput(const char *strName, unsigned char *data, unsigned int dataSz);
WOLFSSL_TEST_SUBROUTINE int ariagcm_test(MC_ALGID);
#endif

#ifdef WOLF_CRYPTO_CB
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t cryptocb_test(void);
#endif
#ifdef WOLFSSL_CERT_PIV
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t certpiv_test(void);
#endif
#ifdef WOLFSSL_AES_SIV
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aes_siv_test(void);
#endif

#if defined(WOLFSSL_AES_EAX) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aes_eax_test(void);
#endif /* WOLFSSL_AES_EAX */

/* General big buffer size for many tests. */
#define FOURK_BUF 4096

#if defined(WOLFSSL_ESPIDF_ERROR_PAUSE)
    /* When defined, pause at error condition rather than exit with error. */
        #define ERROR_OUT(err, eLabel) \
            do { \
                ret = (err); \
                esp_ShowExtendedSystemInfo(); \
                ESP_LOGE("wolfcrypt_test", "ESP Error! ret = %d ", err); \
                while (1) { \
                    vTaskDelay(60000); \
                } \
                /* Just to appease compiler, don't actually go to eLabel */ \
                goto eLabel; \
            } while (0)
#else
    #define ERROR_OUT(err, eLabel) do { ret = (err); goto eLabel; } while (0)
#endif

/* Not all unexpected conditions are actually errors .*/
#define WARNING_OUT(err, eLabel) do { ret = (err); goto eLabel; } while (0)

static void render_error_message(const char* msg, wc_test_ret_t es)
{
    (void)msg;
    (void)es;

#ifdef WOLFSSL_LINUXKM
    #define err_sys_printf lkm_printf
#else
    #define err_sys_printf printf
#endif

    switch (WC_TEST_RET_DEC_TAG(es)) {
    case WC_TEST_RET_TAG_NC:
        err_sys_printf("%s error L=%d\n", msg, WC_TEST_RET_DEC_LN(es));
        break;
    case WC_TEST_RET_TAG_EC:
#ifdef NO_ERROR_STRINGS
        err_sys_printf("%s error L=%d code=%d\n", msg,
                       WC_TEST_RET_DEC_LN(es), -WC_TEST_RET_DEC_I(es));
#else
        err_sys_printf("%s error L=%d code=%d (%s)\n", msg,
                       WC_TEST_RET_DEC_LN(es), -WC_TEST_RET_DEC_I(es),
                       wc_GetErrorString(-WC_TEST_RET_DEC_I(es)));
#endif
        break;
    case WC_TEST_RET_TAG_ERRNO:
    {

/* strerror_r() comes in two mutually incompatible flavors, a native glibc
 * flavor that always returns a non-null char pointer that must be used
 * directly, and a POSIX flavor that returns an error int, and iff success,
 * stores an error string in the supplied buffer.  this is all most
 * infelicitous...
 */
#if !defined(STRING_USER) && !defined(NO_ERROR_STRINGS) &&      \
    (defined(__STDC_VERSION__) && (__STDC_VERSION__ > 199901L)) &&      \
    ((defined(__GLIBC__) && (__GLIBC__ >= 2)) ||                \
     (defined(__USE_XOPEN2K) &&                                 \
      defined(_POSIX_C_SOURCE) &&                               \
      (_POSIX_C_SOURCE >= 200112L)))

        char errno_buf[64], *errno_string;
        /* precisely mirror the gate used in glibc string.h */
#if defined __USE_XOPEN2K && !defined __USE_GNU
        if (strerror_r(WC_TEST_RET_DEC_I(es),
                       errno_buf, sizeof(errno_buf)) != 0)
            XSTRLCPY(errno_buf, "?", sizeof(errno_buf));
        errno_string = errno_buf;
#else
        errno_string = strerror_r(WC_TEST_RET_DEC_I(es),
                                  errno_buf, sizeof(errno_buf));
#endif
        err_sys_printf("%s error L=%d errno=%d (%s)\n", msg,
                       WC_TEST_RET_DEC_LN(es), WC_TEST_RET_DEC_I(es),
                       errno_string);

#else /* can't figure out how to strerror_r(), or don't want error strings */
        err_sys_printf("%s error L=%d errno=%d\n", msg,
                       WC_TEST_RET_DEC_LN(es), WC_TEST_RET_DEC_I(es));
#endif
        break;
    }
    case WC_TEST_RET_TAG_I:
        err_sys_printf("%s error L=%d i=%d\n", msg,
                       WC_TEST_RET_DEC_LN(es), WC_TEST_RET_DEC_I(es));
        break;
    }

#undef err_sys_printf
}

static void print_fiducials(void);

#ifdef HAVE_STACK_SIZE
static THREAD_RETURN err_sys(const char* msg, int es)
#else
static wc_test_ret_t err_sys(const char* msg, wc_test_ret_t es)
#endif
{
    render_error_message(msg, es);
    print_fiducials();
#ifdef WOLFSSL_LINUXKM
    EXIT_TEST(es);
#else
    EXIT_TEST(-1);
#endif
}

#ifndef HAVE_WOLFCRYPT_TEST_OPTIONS
/* func_args from test.h, so don't have to pull in other stuff */
typedef struct func_args {
    int    argc;
    char** argv;
    wc_test_ret_t return_code;
} func_args;
#endif /* !HAVE_WOLFCRYPT_TEST_OPTIONS */

#if defined(HAVE_FIPS) && !defined(WOLFSSL_LINUXKM)
static void myFipsCb(int ok, int err, const char* hash)
{
    printf("in my Fips callback, ok = %d, err = %d\n", ok, err);
    printf("message = %s\n", wc_GetErrorString(err));
    printf("hash = %s\n", hash);

    if (err == IN_CORE_FIPS_E) {
        printf("In core integrity hash check failure, copy above hash\n");
        printf("into verifyCore[] in fips_test.c and rebuild\n");
    }
}
#endif /* HAVE_FIPS && !WOLFSSL_LINUXKM */

#ifdef WOLFSSL_STATIC_MEMORY
    #if defined(WOLFSSL_STATIC_MEMORY_TEST_SZ)
        static byte gTestMemory[WOLFSSL_STATIC_MEMORY_TEST_SZ];
    #elif defined(BENCH_EMBEDDED)
        static byte gTestMemory[14000];
    #elif defined(WOLFSSL_CERT_EXT)
        static byte gTestMemory[140000];
    #elif (defined(WOLFSSL_SP_MATH_ALL) || defined(USE_FAST_MATH)) && \
          !defined(ALT_ECC_SIZE)
        static byte gTestMemory[160000];
    #else
        static byte gTestMemory[80000];
    #endif
#endif

#ifdef WOLFSSL_PB
static int wolfssl_pb_print(const char* msg, ...)
{
    int ret;
    va_list args;
    char tmpBuf[80];

    va_start(args, msg);
    ret = vsprint(tmpBuf, msg, args);
    va_end(args);

    fnDumpStringToSystemLog(tmpBuf);

    return ret;
}
#endif /* WOLFSSL_PB */


#if defined(WOLF_CRYPTO_CB) && !defined(HAVE_HASHDRBG) && \
    !defined(WC_NO_RNG) && !defined(CUSTOM_RAND_GENERATE_BLOCK)
/* Enable support for RNG with crypto callback */
static int rng_crypto_cb(int thisDevId, wc_CryptoInfo* info, void* ctx)
{
    int rc = CRYPTOCB_UNAVAILABLE;
    if (info->algo_type == WC_ALGO_TYPE_RNG) {
        rc = wc_GenerateSeed(&info->rng.rng->seed, info->rng.out, info->rng.sz);
    }
    (void)ctx;
    (void)thisDevId;
    return rc;
}
#endif

/* optional macro to add sleep between tests */
#ifndef TEST_SLEEP
#define TEST_SLEEP() WC_DO_NOTHING
#else
    #define TEST_PASS test_pass
    #include <stdarg.h> /* for var args */
    static WC_INLINE void test_pass(const char* fmt, ...)
    {
        va_list args;
        va_start(args, fmt);
        STACK_SIZE_CHECKPOINT_WITH_MAX_CHECK(max_relative_stack, vprintf(fmt, args));
        va_end(args);
        PRINT_HEAP_CHECKPOINT();
        TEST_SLEEP();
        ASSERT_RESTORED_VECTOR_REGISTERS(exit(1););
    }
#endif

/* set test pass output to printf if not overridden */
#ifndef TEST_PASS
    /* redirect to printf */
    #define TEST_PASS(...) {                                    \
        if (STACK_SIZE_CHECKPOINT_WITH_MAX_CHECK                \
            (max_relative_stack, printf(__VA_ARGS__)) < 0) {    \
            return err_sys("post-test check failed", WC_TEST_RET_ENC_NC);\
        }                                                       \
        PRINT_HEAP_CHECKPOINT();                                \
        ASSERT_RESTORED_VECTOR_REGISTERS(exit(1););             \
    }
#endif

#ifdef TEST_ALWAYS_RUN_TO_END
    #define TEST_FAIL(msg, retval) do { last_failed_test_ret = (retval); render_error_message(msg, retval); } while (0)
#elif !defined(TEST_FAIL)
    #define TEST_FAIL(msg, retval) return err_sys(msg, retval)
#endif

#ifdef HAVE_STACK_SIZE
THREAD_RETURN WOLFSSL_THREAD wolfcrypt_test(void* args)
#else
wc_test_ret_t wolfcrypt_test(void* args)
#endif
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_TRACK_MEMORY_VERBOSE) && !defined(WOLFSSL_STATIC_MEMORY)
    long heap_baselineAllocs, heap_baselineBytes;
#endif
#ifdef TEST_ALWAYS_RUN_TO_END
    int last_failed_test_ret = 0;
#endif
    STACK_SIZE_INIT();

#if defined(WOLFSSL_TRACK_MEMORY_VERBOSE) && !defined(WOLFSSL_STATIC_MEMORY)
    (void)wolfCrypt_heap_peakAllocs_checkpoint();
    heap_baselineAllocs = wolfCrypt_heap_peakAllocs_checkpoint();
    (void)wolfCrypt_heap_peakBytes_checkpoint();
    heap_baselineBytes = wolfCrypt_heap_peakBytes_checkpoint();
#endif

    printf("------------------------------------------------------------------------------\n");
    printf(" wolfSSL version %s\n", LIBWOLFSSL_VERSION_STRING);
#ifdef WOLF_CRYPTO_CB
    if (devId != INVALID_DEVID)
        printf("  CryptoCB with DevID:%X\n", devId);
#endif
    printf("------------------------------------------------------------------------------\n");

    if (args) {
#ifdef HAVE_WOLFCRYPT_TEST_OPTIONS
        int ch;
#endif
        ((func_args*)args)->return_code = -1; /* error state */
#ifdef HAVE_WOLFCRYPT_TEST_OPTIONS
        while ((ch = mygetopt(((func_args*)args)->argc, ((func_args*)args)->argv, "s:m:a:h")) != -1) {
            switch(ch) {
            case 's':
#ifdef HAVE_STACK_SIZE_VERBOSE
                max_relative_stack = (ssize_t)atoi(myoptarg);
                break;
#else
                return err_sys("-s (max relative stack bytes) requires HAVE_STACK_SIZE_VERBOSE (--enable-stacksize=verbose).", WC_TEST_RET_ENC_NC);
#endif
            case 'm':
#ifdef WOLFSSL_TRACK_MEMORY_VERBOSE
                max_relative_heap_bytes = (ssize_t)atoi(myoptarg);
                break;
#else
                return err_sys("-m (max relative heap memory bytes) requires WOLFSSL_TRACK_MEMORY_VERBOSE (--enable-trackmemory=verbose).", WC_TEST_RET_ENC_NC);
#endif
            case 'a':
#ifdef WOLFSSL_TRACK_MEMORY_VERBOSE
                max_relative_heap_allocs = (ssize_t)atoi(myoptarg);
                break;
#else
                return err_sys("-a (max relative heap allocs) requires WOLFSSL_TRACK_MEMORY_VERBOSE (--enable-trackmemory=verbose).", WC_TEST_RET_ENC_NC);
#endif
            case 'h':
                return err_sys("\
options: [-s max_relative_stack_bytes] [-m max_relative_heap_memory_bytes]\n\
         [-a max_relative_heap_allocs] [-h]\n", 0);
            default:
                return err_sys("unknown test option.  try -h.", WC_TEST_RET_ENC_NC);
            }
        }
#endif
    }

#ifdef WOLFSSL_STATIC_MEMORY
    if (wc_LoadStaticMemory(&HEAP_HINT, gTestMemory, sizeof(gTestMemory),
                                                WOLFMEM_GENERAL, 1) != 0) {
        printf("unable to load static memory.\n");
        return(EXIT_FAILURE);
    }
#endif

#if defined(DEBUG_WOLFSSL) && !defined(HAVE_VALGRIND)
    wolfSSL_Debugging_ON();
#endif

#if defined(OPENSSL_EXTRA) || defined(DEBUG_WOLFSSL_VERBOSE)
    wc_SetLoggingHeap(HEAP_HINT);
#endif

#if defined(HAVE_FIPS) && !defined(WOLFSSL_LINUXKM)
    wolfCrypt_SetCb_fips(myFipsCb);
#endif

#if !defined(NO_BIG_INT)
    if (CheckCtcSettings() != 1) {
        printf("Sizeof mismatch (build) %x != (run) %lx\n",
               CTC_SETTINGS, (unsigned long)CheckRunTimeSettings());
        return err_sys("Build vs runtime math mismatch\n", WC_TEST_RET_ENC_NC);
    }

#if defined(USE_FAST_MATH) && \
        (!defined(NO_RSA) || !defined(NO_DH) || defined(HAVE_ECC))
    if (CheckFastMathSettings() != 1)
        return err_sys("Build vs runtime fastmath FP_MAX_BITS mismatch\n",
                       WC_TEST_RET_ENC_NC);
#endif /* USE_FAST_MATH */
#endif /* !NO_BIG_INT */

#if defined(WOLFSSL_CERT_GEN) && (!defined(NO_RSA) || defined(HAVE_ECC)) || \
  (defined(WOLFSSL_TEST_CERT) && (defined(HAVE_ED25519) || defined(HAVE_ED448)))
    initDefaultName();
#endif

#ifdef WOLFSSL_ASYNC_CRYPT
    ret = wolfAsync_DevOpen(&devId);
    if (ret < 0) {
        printf("Async device open failed\nRunning without async\n");
    }
#else
    (void)devId;
#endif /* WOLFSSL_ASYNC_CRYPT */

#ifdef WOLF_CRYPTO_CB
#ifdef HAVE_INTEL_QA_SYNC
    devId = wc_CryptoCb_InitIntelQa();
    if (INVALID_DEVID == devId) {
        printf("Couldn't init the Intel QA\n");
    }
#endif
#ifdef HAVE_CAVIUM_OCTEON_SYNC
    devId = wc_CryptoCb_InitOcteon();
    if (INVALID_DEVID == devId) {
        printf("Couldn't init the Cavium Octeon\n");
    }
#endif
#ifdef HAVE_RENESAS_SYNC
    devId = wc_CryptoCb_CryptInitRenesasCmn(NULL, &guser_PKCbInfo);
    if (devId == INVALID_DEVID) {
        printf("Couldn't get the Renesas device ID\n");
    }
#endif
#endif

#if defined(WOLF_CRYPTO_CB) && !defined(HAVE_HASHDRBG) && \
    !defined(WC_NO_RNG) && !defined(CUSTOM_RAND_GENERATE_BLOCK)
    if (devId == INVALID_DEVID) {
        /* for testing RNG with crypto callback register function */
        devId = 100; /* any value beside -2 (INVALID_DEVID) */
        wc_CryptoCb_RegisterDevice(devId, rng_crypto_cb, NULL);
    }
#endif


#ifdef HAVE_SELFTEST
    if ( (ret = wolfCrypt_SelfTest()) != 0)
        TEST_FAIL("CAVP selftest failed!\n", ret);
    else
        TEST_PASS("CAVP selftest passed!\n");
#endif

    if ( (ret = error_test()) != 0)
        TEST_FAIL("error    test failed!\n", ret);
    else
        TEST_PASS("error    test passed!\n");

    if ( (ret = memory_test()) != 0)
        TEST_FAIL("MEMORY   test failed!\n", ret);
    else
        TEST_PASS("MEMORY   test passed!\n");

#ifndef NO_CODING
    if ( (ret = base64_test()) != 0)
        TEST_FAIL("base64   test failed!\n", ret);
    else
        TEST_PASS("base64   test passed!\n");
#ifdef WOLFSSL_BASE16
    if ( (ret = base16_test()) != 0)
        TEST_FAIL("base16   test failed!\n", ret);
    else
        TEST_PASS("base16   test passed!\n");
#endif
#endif /* !NO_CODING */

#ifndef NO_ASN
    if ( (ret = asn_test()) != 0)
        TEST_FAIL("asn      test failed!\n", ret);
    else
        TEST_PASS("asn      test passed!\n");
#endif

#ifndef WC_NO_RNG
    if ( (ret = random_test()) != 0)
        TEST_FAIL("RANDOM   test failed!\n", ret);
    else
        TEST_PASS("RANDOM   test passed!\n");
#endif /* WC_NO_RNG */

#ifndef NO_MD5
    if ( (ret = md5_test()) != 0)
        TEST_FAIL("MD5      test failed!\n", ret);
    else
        TEST_PASS("MD5      test passed!\n");
#endif

#ifdef WOLFSSL_MD2
    if ( (ret = md2_test()) != 0)
        TEST_FAIL("MD2      test failed!\n", ret);
    else
        TEST_PASS("MD2      test passed!\n");
#endif

#ifndef NO_MD4
    if ( (ret = md4_test()) != 0)
        TEST_FAIL("MD4      test failed!\n", ret);
    else
        TEST_PASS("MD4      test passed!\n");
#endif

#ifndef NO_SHA
    if ( (ret = sha_test()) != 0)
        TEST_FAIL("SHA      test failed!\n", ret);
    else
        TEST_PASS("SHA      test passed!\n");
#endif

#ifdef WOLFSSL_SHA224
    if ( (ret = sha224_test()) != 0)
        TEST_FAIL("SHA-224  test failed!\n", ret);
    else
        TEST_PASS("SHA-224  test passed!\n");
#endif

#ifndef NO_SHA256
    if ( (ret = sha256_test()) != 0)
        TEST_FAIL("SHA-256  test failed!\n", ret);
    else
        TEST_PASS("SHA-256  test passed!\n");
#endif

#ifdef WOLFSSL_SHA384
    if ( (ret = sha384_test()) != 0)
        TEST_FAIL("SHA-384  test failed!\n", ret);
    else
        TEST_PASS("SHA-384  test passed!\n");
#endif

#ifdef WOLFSSL_SHA512
    if ((ret = sha512_test()) != 0) {
        TEST_FAIL("SHA-512  test failed!\n", ret);
    }
    else {
        TEST_PASS("SHA-512  test passed!\n");
    }

#if !defined(WOLFSSL_NOSHA512_224) && \
   (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
    if ((ret = sha512_224_test()) != 0) {
        TEST_FAIL("SHA-512/224  test failed!\n", ret);
    }
    else
        TEST_PASS("SHA-512/224  test passed!\n");
#endif /* !defined(WOLFSSL_NOSHA512_224) && !FIPS ... */

#if !defined(WOLFSSL_NOSHA512_256) && \
   (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
    if ((ret = sha512_256_test()) != 0) {
        TEST_FAIL("SHA-512/256  test failed!\n", ret);
    }
    else
        TEST_PASS("SHA-512/256  test passed!\n");
#endif /* !defined(WOLFSSL_NOSHA512_256) & !FIPS ... */

#endif /* WOLFSSL_SHA512 */

#ifdef WOLFSSL_SHA3
    if ( (ret = sha3_test()) != 0)
        TEST_FAIL("SHA-3    test failed!\n", ret);
    else
        TEST_PASS("SHA-3    test passed!\n");
#endif

#ifdef WOLFSSL_SHAKE128
    if ( (ret = shake128_test()) != 0)
        TEST_FAIL("SHAKE128 test failed!\n", ret);
    else
        TEST_PASS("SHAKE128 test passed!\n");
#endif

#ifdef WOLFSSL_SHAKE256
    if ( (ret = shake256_test()) != 0)
        TEST_FAIL("SHAKE256 test failed!\n", ret);
    else
        TEST_PASS("SHAKE256 test passed!\n");
#endif

#ifdef WOLFSSL_SM3
    if ( (ret = sm3_test()) != 0)
        return err_sys("SM-3     test failed!\n", ret);
    else
        TEST_PASS("SM-3     test passed!\n");
#endif

#ifndef NO_HASH_WRAPPER
    if ( (ret = hash_test()) != 0)
        TEST_FAIL("Hash     test failed!\n", ret);
    else
        TEST_PASS("Hash     test passed!\n");
#endif

#ifdef WOLFSSL_RIPEMD
    if ( (ret = ripemd_test()) != 0)
        TEST_FAIL("RIPEMD   test failed!\n", ret);
    else
        TEST_PASS("RIPEMD   test passed!\n");
#endif

#ifdef HAVE_BLAKE2
    if ( (ret = blake2b_test()) != 0)
        TEST_FAIL("BLAKE2b  test failed!\n", ret);
    else
        TEST_PASS("BLAKE2b  test passed!\n");
#endif
#ifdef HAVE_BLAKE2S
    if ( (ret = blake2s_test()) != 0)
        TEST_FAIL("BLAKE2s  test failed!\n", ret);
    else
        TEST_PASS("BLAKE2s  test passed!\n");
#endif

#ifndef NO_HMAC
    #if !defined(NO_MD5) && !(defined(HAVE_FIPS) && defined(HAVE_FIPS_VERSION) \
                              && (HAVE_FIPS_VERSION >= 5))
        if ( (ret = hmac_md5_test()) != 0)
            TEST_FAIL("HMAC-MD5 test failed!\n", ret);
        else
            TEST_PASS("HMAC-MD5 test passed!\n");
    #endif

    #ifndef NO_SHA
    if ( (ret = hmac_sha_test()) != 0)
        TEST_FAIL("HMAC-SHA test failed!\n", ret);
    else
        TEST_PASS("HMAC-SHA test passed!\n");
    #endif

    #ifdef WOLFSSL_SHA224
        if ( (ret = hmac_sha224_test()) != 0)
            TEST_FAIL("HMAC-SHA224 test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA224 test passed!\n");
    #endif

    #ifndef NO_SHA256
        if ( (ret = hmac_sha256_test()) != 0)
            TEST_FAIL("HMAC-SHA256 test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA256 test passed!\n");
    #endif

    #ifdef WOLFSSL_SHA384
        if ( (ret = hmac_sha384_test()) != 0)
            TEST_FAIL("HMAC-SHA384 test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA384 test passed!\n");
    #endif

    #ifdef WOLFSSL_SHA512
        if ( (ret = hmac_sha512_test()) != 0)
            TEST_FAIL("HMAC-SHA512 test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA512 test passed!\n");
    #endif

    #if !defined(NO_HMAC) && defined(WOLFSSL_SHA3) && \
      !defined(WOLFSSL_NOSHA3_224) && !defined(WOLFSSL_NOSHA3_256) && \
      !defined(WOLFSSL_NOSHA3_384) && !defined(WOLFSSL_NOSHA3_512)
        if ( (ret = hmac_sha3_test()) != 0)
            TEST_FAIL("HMAC-SHA3   test failed!\n", ret);
        else
            TEST_PASS("HMAC-SHA3   test passed!\n");
    #endif

    #if defined(HAVE_HKDF) && !defined(NO_HMAC)
        PRIVATE_KEY_UNLOCK();
        if ( (ret = hkdf_test()) != 0)
            TEST_FAIL("HMAC-KDF    test failed!\n", ret);
        else
            TEST_PASS("HMAC-KDF    test passed!\n");
        PRIVATE_KEY_LOCK();
    #endif
#endif /* !NO_HMAC */

#ifdef WOLFSSL_WOLFSSH
    PRIVATE_KEY_UNLOCK();
    if ( (ret = sshkdf_test()) != 0)
        TEST_FAIL("SSH-KDF     test failed!\n", ret);
    else
        TEST_PASS("SSH-KDF     test passed!\n");
    PRIVATE_KEY_LOCK();
#endif /* WOLFSSL_WOLFSSH */

#ifdef WOLFSSL_TLS13
    PRIVATE_KEY_UNLOCK();
    if ( (ret = tls13_kdf_test()) != 0)
        TEST_FAIL("TLSv1.3 KDF test failed!\n", ret);
    else
        TEST_PASS("TLSv1.3 KDF test passed!\n");
    PRIVATE_KEY_LOCK();
#endif /* WOLFSSL_TLS13 */

#if defined(HAVE_X963_KDF) && defined(HAVE_ECC)
    if ( (ret = x963kdf_test()) != 0)
        TEST_FAIL("X963-KDF    test failed!\n", ret);
    else
        TEST_PASS("X963-KDF    test passed!\n");
#endif

#if defined(HAVE_HPKE) && defined(HAVE_ECC) && defined(HAVE_AESGCM)
    if ( (ret = hpke_test()) != 0)
        TEST_FAIL("HPKE     test failed!\n", ret);
    else
        TEST_PASS("HPKE     test passed!\n");
#endif

#if defined(WC_SRTP_KDF)
    if ( (ret = srtpkdf_test()) != 0)
        TEST_FAIL("SRTP KDF test failed!\n", ret);
    else
        TEST_PASS("SRTP KDF test passed!\n");
#endif

#if defined(HAVE_AESGCM) && defined(WOLFSSL_AES_128) && \
   !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT) && \
   !defined(WOLFSSL_RENESAS_FSPSM_CRYPTONLY)
    if ( (ret = gmac_test()) != 0)
        TEST_FAIL("GMAC     test failed!\n", ret);
    else
        TEST_PASS("GMAC     test passed!\n");
#endif

#ifdef WC_RC2
    if ( (ret = rc2_test()) != 0)
        TEST_FAIL("RC2      test failed!\n", ret);
    else
        TEST_PASS("RC2      test passed!\n");
#endif

#ifndef NO_RC4
    if ( (ret = arc4_test()) != 0)
        TEST_FAIL("ARC4     test failed!\n", ret);
    else
        TEST_PASS("ARC4     test passed!\n");
#endif

#ifdef HAVE_CHACHA
    if ( (ret = chacha_test()) != 0)
        TEST_FAIL("Chacha   test failed!\n", ret);
    else
        TEST_PASS("Chacha   test passed!\n");
#endif

#ifdef HAVE_XCHACHA
    if ( (ret = XChaCha_test()) != 0)
        TEST_FAIL("XChacha  test failed!\n", ret);
    else
        TEST_PASS("XChacha  test passed!\n");
#endif

#ifdef HAVE_POLY1305
    if ( (ret = poly1305_test()) != 0)
        TEST_FAIL("POLY1305 test failed!\n", ret);
    else
        TEST_PASS("POLY1305 test passed!\n");
#endif

#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
    if ( (ret = chacha20_poly1305_aead_test()) != 0)
        TEST_FAIL("ChaCha20-Poly1305 AEAD test failed!\n", ret);
    else
        TEST_PASS("ChaCha20-Poly1305 AEAD test passed!\n");
#endif

#if defined(HAVE_XCHACHA) && defined(HAVE_POLY1305)
    if ( (ret = XChaCha20Poly1305_test()) != 0)
        TEST_FAIL("XChaCha20-Poly1305 AEAD test failed!\n", ret);
    else
        TEST_PASS("XChaCha20-Poly1305 AEAD test passed!\n");
#endif

#ifndef NO_DES3
    if ( (ret = des_test()) != 0)
        TEST_FAIL("DES      test failed!\n", ret);
    else
        TEST_PASS("DES      test passed!\n");
#endif

#ifndef NO_DES3
    if ( (ret = des3_test()) != 0)
        TEST_FAIL("DES3     test failed!\n", ret);
    else
        TEST_PASS("DES3     test passed!\n");
#endif

#ifndef NO_AES
    if ( (ret = aes_test()) != 0)
        TEST_FAIL("AES      test failed!\n", ret);
    else
        TEST_PASS("AES      test passed!\n");

#if defined(WOLFSSL_AES_192)  && \
   !defined(WOLFSSL_RENESAS_FSPSM_CRYPTONLY)
    if ( (ret = aes192_test()) != 0)
        TEST_FAIL("AES192   test failed!\n", ret);
    else
        TEST_PASS("AES192   test passed!\n");
#endif

#if defined(WOLFSSL_AES_256)
    if ( (ret = aes256_test()) != 0)
        TEST_FAIL("AES256   test failed!\n", ret);
    else
        TEST_PASS("AES256   test passed!\n");
#endif

#ifdef WOLFSSL_AES_OFB
    if ( (ret = aesofb_test()) != 0)
        TEST_FAIL("AES-OFB  test failed!\n", ret);
    else
        TEST_PASS("AESOFB   test passed!\n");
#endif

#ifdef HAVE_AESGCM
    #if !defined(WOLFSSL_AFALG) && !defined(WOLFSSL_DEVCRYPTO)
    if ( (ret = aesgcm_test()) != 0)
        TEST_FAIL("AES-GCM  test failed!\n", ret);
    #endif
    #if !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT) && \
        !defined(WOLFSSL_RENESAS_FSPSM_CRYPTONLY) && \
        !defined(WOLFSSL_KCAPI_AES) && !(defined(WOLF_CRYPTO_CB) && \
            (defined(HAVE_INTEL_QA_SYNC) || defined(HAVE_CAVIUM_OCTEON_SYNC)))
    if ((ret = aesgcm_default_test()) != 0) {
        TEST_FAIL("AES-GCM  test failed!\n", ret);
    }
    #endif
    if (ret == 0) {
        TEST_PASS("AES-GCM  test passed!\n");
    }
#endif

#if defined(HAVE_AESCCM) && defined(WOLFSSL_AES_128)
    if ( (ret = aesccm_test()) != 0)
        TEST_FAIL("AES-CCM  test failed!\n", ret);
    else
        TEST_PASS("AES-CCM  test passed!\n");
#endif
#ifdef HAVE_AES_KEYWRAP
    if ( (ret = aeskeywrap_test()) != 0)
        TEST_FAIL("AES Key Wrap test failed!\n", ret);
    else
        TEST_PASS("AES Key Wrap test passed!\n");
#endif
#ifdef WOLFSSL_AES_SIV
    if ( (ret = aes_siv_test()) != 0)
        TEST_FAIL("AES-SIV  test failed!\n", ret);
    else
        TEST_PASS("AES-SIV  test passed!\n");
#endif
#endif

#if defined(WOLFSSL_AES_EAX) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
    if ( (ret = aes_eax_test()) != 0)
        TEST_FAIL("AES-EAX  test failed!\n", ret);
    else
        TEST_PASS("AES-EAX  test passed!\n");
#endif /* WOLFSSL_AES_EAX */

#ifdef HAVE_ARIA
    if ( (ret = ariagcm_test(MC_ALGID_ARIA_128BITKEY)) != 0)
        TEST_FAIL("ARIA128  test failed!\n", ret);
    else
        TEST_PASS("ARIA128   test passed!\n");

    if ( (ret = ariagcm_test(MC_ALGID_ARIA_192BITKEY)) != 0)
        TEST_FAIL("ARIA192  test failed!\n", ret);
    else
        TEST_PASS("ARIA192   test passed!\n");

    if ( (ret = ariagcm_test(MC_ALGID_ARIA_256BITKEY)) != 0)
        TEST_FAIL("ARIA256  test failed!\n", ret);
    else
        TEST_PASS("ARIA256   test passed!\n");
#endif

#ifdef HAVE_CAMELLIA
    if ( (ret = camellia_test()) != 0)
        TEST_FAIL("CAMELLIA test failed!\n", ret);
    else
        TEST_PASS("CAMELLIA test passed!\n");
#endif

#ifdef WOLFSSL_SM4
    if ( (ret = sm4_test()) != 0)
        return err_sys("SM-4     test failed!\n", ret);
    else
        TEST_PASS("SM-4     test passed!\n");
#endif

#if !defined(NO_RSA) && !defined(HAVE_RENESAS_SYNC)
    #ifdef WC_RSA_NO_PADDING
    if ( (ret = rsa_no_pad_test()) != 0)
        TEST_FAIL("RSA NOPAD test failed!\n", ret);
    else
        TEST_PASS("RSA NOPAD test passed!\n");
    #endif
    if ( (ret = rsa_test()) != 0)
        TEST_FAIL("RSA      test failed!\n", ret);
    else
        TEST_PASS("RSA      test passed!\n");
#endif

#ifndef NO_DH
    PRIVATE_KEY_UNLOCK();
    if ( (ret = dh_test()) != 0)
        TEST_FAIL("DH       test failed!\n", ret);
    else
        TEST_PASS("DH       test passed!\n");
    PRIVATE_KEY_LOCK();
#endif

#ifndef NO_DSA
    if ( (ret = dsa_test()) != 0)
        TEST_FAIL("DSA      test failed!\n", ret);
    else
        TEST_PASS("DSA      test passed!\n");
#endif

#ifdef WOLFCRYPT_HAVE_SRP
    if ( (ret = srp_test()) != 0)
        TEST_FAIL("SRP      test failed!\n", ret);
    else
        TEST_PASS("SRP      test passed!\n");
#endif

#ifndef NO_PWDBASED
    if ( (ret = pwdbased_test()) != 0)
        TEST_FAIL("PWDBASED test failed!\n", ret);
    else
        TEST_PASS("PWDBASED test passed!\n");
#endif

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
    if ( (ret = openssl_test()) != 0)
        TEST_FAIL("OPENSSL  test failed!\n", ret);
    else
        TEST_PASS("OPENSSL  test passed!\n");

    if ( (ret = openSSL_evpMD_test()) != 0)
        TEST_FAIL("OPENSSL (EVP MD) test failed!\n", ret);
    else
        TEST_PASS("OPENSSL (EVP MD) passed!\n");

    if ( (ret = openssl_pkey0_test()) != 0)
        TEST_FAIL("OPENSSL (PKEY0) test failed!\n", ret);
    else
        TEST_PASS("OPENSSL (PKEY0) passed!\n");

    if ( (ret = openssl_pkey1_test()) != 0)
        TEST_FAIL("OPENSSL (PKEY1) test failed!\n", ret);
    else
        TEST_PASS("OPENSSL (PKEY1) passed!\n");

    #if !defined(WOLF_CRYPTO_CB_ONLY_RSA) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    if ( (ret = openssl_evpSig_test()) != 0)
        TEST_FAIL("OPENSSL (EVP Sign/Verify) test failed!\n", ret);
    else
        TEST_PASS("OPENSSL (EVP Sign/Verify) passed!\n");
    #endif
#endif

#if defined(HAVE_ECC)
    PRIVATE_KEY_UNLOCK();
    if ( (ret = ecc_test()) != 0)
        TEST_FAIL("ECC      test failed!\n", ret);
    else
        TEST_PASS("ECC      test passed!\n");
    PRIVATE_KEY_LOCK();
    #if defined(HAVE_ECC_ENCRYPT) && defined(HAVE_AES_CBC) && \
        (defined(WOLFSSL_AES_128) || defined(WOLFSSL_AES_256))
        if ( (ret = ecc_encrypt_test()) != 0)
            TEST_FAIL("ECC Enc  test failed!\n", ret);
        else
            TEST_PASS("ECC Enc  test passed!\n");
    #endif
    #if defined(USE_CERT_BUFFERS_256) && !defined(WOLFSSL_ATECC508A) && \
        !defined(WOLFSSL_ATECC608A) && !defined(NO_ECC256) && \
        defined(HAVE_ECC_VERIFY) && defined(HAVE_ECC_SIGN) && \
        !defined(WOLF_CRYPTO_CB_ONLY_ECC) && !defined(NO_ECC_SECP)
        /* skip for ATECC508/608A, cannot import private key buffers */
        if ( (ret = ecc_test_buffers()) != 0)
            TEST_FAIL("ECC buffer test failed!\n", ret);
        else
            TEST_PASS("ECC buffer test passed!\n");
    #endif
#endif

#if !defined(NO_ASN_TIME) && !defined(NO_RSA) && defined(WOLFSSL_TEST_CERT) && \
    !defined(NO_FILESYSTEM)
    if ( (ret = cert_test()) != 0)
        TEST_FAIL("CERT     test failed!\n", ret);
    else
        TEST_PASS("CERT     test passed!\n");
#endif

#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_TEST_CERT) && \
   !defined(NO_FILESYSTEM) && !defined(NO_RSA) && defined(WOLFSSL_GEN_CERT)
    if ( (ret = certext_test()) != 0)
        TEST_FAIL("CERT EXT test failed!\n", ret);
    else
        TEST_PASS("CERT EXT test passed!\n");
#endif

#if defined(WOLFSSL_CERT_GEN_CACHE) && defined(WOLFSSL_TEST_CERT) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
    if ( (ret = decodedCertCache_test()) != 0)
        TEST_FAIL("DECODED CERT CACHE test failed!\n", ret);
    else
        TEST_PASS("DECODED CERT CACHE test passed!\n");
#endif

#ifdef HAVE_CURVE25519
    if ( (ret = curve25519_test()) != 0)
        TEST_FAIL("CURVE25519 test failed!\n", ret);
    else
        TEST_PASS("CURVE25519 test passed!\n");
#endif

#ifdef HAVE_ED25519
    if ( (ret = ed25519_test()) != 0)
        TEST_FAIL("ED25519  test failed!\n", ret);
    else
        TEST_PASS("ED25519  test passed!\n");
#endif

#ifdef HAVE_CURVE448
    if ( (ret = curve448_test()) != 0)
        TEST_FAIL("CURVE448 test failed!\n", ret);
    else
        TEST_PASS("CURVE448 test passed!\n");
#endif

#ifdef HAVE_ED448
    if ( (ret = ed448_test()) != 0)
        TEST_FAIL("ED448    test failed!\n", ret);
    else
        TEST_PASS("ED448    test passed!\n");
#endif

#ifdef WOLFSSL_HAVE_KYBER
    if ( (ret = kyber_test()) != 0)
        TEST_FAIL("KYBER    test failed!\n", ret);
    else
        TEST_PASS("KYBER    test passed!\n");
#endif

#if defined(WOLFSSL_HAVE_XMSS)
    #if !defined(WOLFSSL_XMSS_VERIFY_ONLY)
    if ( (ret = xmss_test()) != 0)
        TEST_FAIL("XMSS     test failed!\n", ret);
    else
        TEST_PASS("XMSS     test passed!\n");
    #endif

    #if defined(WOLFSSL_XMSS_VERIFY_ONLY) && !defined(WOLFSSL_SMALL_STACK)
    if ( (ret = xmss_test_verify_only()) != 0)
        TEST_FAIL("XMSS     test failed!\n", ret);
    else
        TEST_PASS("XMSS     test passed!\n");
    #endif
#endif /* if defined(WOLFSSL_HAVE_XMSS) */

#if defined(WOLFSSL_HAVE_LMS)
    #if !defined(WOLFSSL_LMS_VERIFY_ONLY)
    if ( (ret = lms_test()) != 0)
        TEST_FAIL("LMS      test failed!\n", ret);
    else
        TEST_PASS("LMS      test passed!\n");
    #endif

    #if defined(WOLFSSL_LMS_VERIFY_ONLY) && !defined(WOLFSSL_SMALL_STACK)
    if ( (ret = lms_test_verify_only()) != 0)
        TEST_FAIL("LMS      test failed!\n", ret);
    else
        TEST_PASS("LMS      test passed!\n");
    #endif
#endif /* if defined(WOLFSSL_HAVE_LMS) */

#ifdef WOLFCRYPT_HAVE_ECCSI
    if ( (ret = eccsi_test()) != 0)
        TEST_FAIL("ECCSI    test failed!\n", ret);
    else
        TEST_PASS("ECCSI    test passed!\n");
#endif
#ifdef WOLFCRYPT_HAVE_SAKKE
    if ( (ret = sakke_test()) != 0)
        TEST_FAIL("SAKKE    test failed!\n", ret);
    else
        TEST_PASS("SAKKE    test passed!\n");
#endif

#if defined(WOLFSSL_CMAC) && !defined(NO_AES)
    if ( (ret = cmac_test()) != 0)
        TEST_FAIL("CMAC     test failed!\n", ret);
    else
        TEST_PASS("CMAC     test passed!\n");
#endif

#if defined(WOLFSSL_SIPHASH)
    if ( (ret = siphash_test()) != 0)
        TEST_FAIL("SipHash  test failed!\n", ret);
    else
        TEST_PASS("SipHash  test passed!\n");
#endif

#ifdef HAVE_LIBZ
    if ( (ret = compress_test()) != 0)
        TEST_FAIL("COMPRESS test failed!\n", ret);
    else
        TEST_PASS("COMPRESS test passed!\n");
#endif

#ifdef HAVE_PKCS7
    #ifndef NO_PKCS7_ENCRYPTED_DATA
        if ( (ret = pkcs7encrypted_test()) != 0)
            TEST_FAIL("PKCS7encrypted  test failed!\n", ret);
        else
            TEST_PASS("PKCS7encrypted  test passed!\n");
    #endif
    #if defined(HAVE_LIBZ) && !defined(NO_PKCS7_COMPRESSED_DATA)
        if ( (ret = pkcs7compressed_test()) != 0)
            TEST_FAIL("PKCS7compressed test failed!\n", ret);
        else
            TEST_PASS("PKCS7compressed test passed!\n");
    #endif
    if ( (ret = pkcs7signed_test()) != 0)
        TEST_FAIL("PKCS7signed     test failed!\n", ret);
    else
        TEST_PASS("PKCS7signed     test passed!\n");

    if ( (ret = pkcs7enveloped_test()) != 0)
        TEST_FAIL("PKCS7enveloped  test failed!\n", ret);
    else
        TEST_PASS("PKCS7enveloped  test passed!\n");

    #if defined(HAVE_AESGCM) || defined(HAVE_AESCCM)
        if ( (ret = pkcs7authenveloped_test()) != 0)
            TEST_FAIL("PKCS7authenveloped  test failed!\n", ret);
        else
            TEST_PASS("PKCS7authenveloped  test passed!\n");
    #endif
#endif

#if defined(WOLFSSL_PUBLIC_MP) && \
    ((defined(WOLFSSL_SP_MATH_ALL) && !defined(WOLFSSL_RSA_VERIFY_ONLY)) || \
     defined(USE_FAST_MATH))
    if ( (ret = mp_test()) != 0)
        TEST_FAIL("mp       test failed!\n", ret);
    else
        TEST_PASS("mp       test passed!\n");
#endif

#if defined(WOLFSSL_PUBLIC_MP) && defined(WOLFSSL_KEY_GEN)
    if ( (ret = prime_test()) != 0)
        TEST_FAIL("prime    test failed!\n", ret);
    else
        TEST_PASS("prime    test passed!\n");
#endif

#if defined(ASN_BER_TO_DER) && \
    (defined(WOLFSSL_TEST_CERT) || defined(OPENSSL_EXTRA) || \
     defined(OPENSSL_EXTRA_X509_SMALL))
    if ( (ret = berder_test()) != 0)
        TEST_FAIL("ber-der  test failed!\n", ret);
    else
        TEST_PASS("ber-der  test passed!\n");
#endif

    if ( (ret = logging_test()) != 0)
        TEST_FAIL("logging  test failed!\n", ret);
    else
        TEST_PASS("logging  test passed!\n");

#if !defined(NO_ASN) && !defined(NO_ASN_TIME)
    if ( (ret = time_test()) != 0)
        TEST_FAIL("time test failed!\n", ret);
    else
        TEST_PASS("time test passed!\n");
#endif

#if defined(__INCLUDE_NUTTX_CONFIG_H)
    if ((ret = wolfcrypt_mutex_test()) != 0)
#else
    if ((ret = mutex_test()) != 0)
#endif
        TEST_FAIL("mutex    test failed!\n", ret);
    else
        TEST_PASS("mutex    test passed!\n");

#if defined(USE_WOLFSSL_MEMORY) && !defined(FREERTOS)
    if ( (ret = memcb_test()) != 0)
        TEST_FAIL("memcb    test failed!\n", ret);
    else
        TEST_PASS("memcb    test passed!\n");
#endif

#ifdef WOLFSSL_CAAM_BLOB
    if ( (ret = blob_test()) != 0)
        TEST_FAIL("blob     test failed!\n", ret);
    else
        TEST_PASS("blob     test passed!\n");
#endif

#if defined(WOLF_CRYPTO_CB) && \
    !(defined(HAVE_INTEL_QAT_SYNC) || defined(HAVE_CAVIUM_OCTEON_SYNC) || \
      defined(WOLFSSL_QNX_CAAM) || defined(HAVE_RENESAS_SYNC))
    if ( (ret = cryptocb_test()) != 0)
        TEST_FAIL("crypto callback test failed!\n", ret);
    else
        TEST_PASS("crypto callback test passed!\n");
#endif

#ifdef WOLFSSL_CERT_PIV
    if ( (ret = certpiv_test()) != 0)
        TEST_FAIL("cert piv test failed!\n", ret);
    else
        TEST_PASS("cert piv test passed!\n");
#endif

#ifdef WOLF_CRYPTO_CB
#ifdef HAVE_INTEL_QA_SYNC
    wc_CryptoCb_CleanupIntelQa(&devId);
#endif
#ifdef HAVE_CAVIUM_OCTEON_SYNC
    wc_CryptoCb_CleanupOcteon(&devId);
#endif
#endif

#ifdef WOLFSSL_ASYNC_CRYPT
    wolfAsync_DevClose(&devId);
#endif

    /* cleanup the thread if fixed point cache is enabled and have thread local */
#if defined(HAVE_THREAD_LS) && defined(HAVE_ECC) && defined(FP_ECC)
    wc_ecc_fp_free();
#endif

#ifdef TEST_ALWAYS_RUN_TO_END
    if (last_failed_test_ret != 0)
        ret = last_failed_test_ret;
#endif

    if (args)
        ((func_args*)args)->return_code = ret;

    TEST_PASS("Test complete\n");

    EXIT_TEST(ret);
}

#ifndef NO_MAIN_DRIVER

#ifdef HAVE_WOLFCRYPT_TEST_OPTIONS
    int myoptind = 0;
    char* myoptarg = NULL;
#endif

    /* so overall tests can pull in test function */
#if defined(WOLFSSL_ESPIDF) || defined(_WIN32_WCE)
    wc_test_ret_t wolf_test_task(void)
#else
#ifndef NO_MAIN_FUNCTION
    int main(int argc, char** argv)
    {
        return (int)wolfcrypt_test_main(argc, argv);
    }
#endif

    wc_test_ret_t wolfcrypt_test_main(int argc, char** argv)
#endif
    {
        wc_test_ret_t ret;
        func_args args = { 0, 0, 0 };
#if defined(WOLFSSL_ESPIDF) || defined(WOLFSSL_SE050)
        /* set dummy wallclock time. */
        struct timeval utctime;
        struct timezone tz;
        utctime.tv_sec = 1521725159; /* dummy time: 2018-03-22T13:25:59+00:00 */
        utctime.tv_usec = 0;
        tz.tz_minuteswest = 0;
        tz.tz_dsttime = 0;
        settimeofday(&utctime, &tz);
#endif
#ifdef WOLFSSL_APACHE_MYNEWT
        #ifdef ARCH_sim
        mcu_sim_parse_args(argc, argv);
        #endif
        sysinit();

        /* set dummy wallclock time. */
        struct os_timeval utctime;
        struct os_timezone tz;
        utctime.tv_sec = 1521725159; /* dummy time: 2018-03-22T13:25:59+00:00 */
        utctime.tv_usec = 0;
        tz.tz_minuteswest = 0;
        tz.tz_dsttime = 0;
        os_settimeofday(&utctime, &tz);
#endif
#ifdef WOLFSSL_ZEPHYR
        /* set dummy wallclock time. */
        struct timespec utctime;
        utctime.tv_sec = 1521725159; /* dummy time: 2018-03-22T13:25:59+00:00 */
        utctime.tv_nsec = 0;
        clock_settime(CLOCK_REALTIME, &utctime);
#endif
#ifdef DEVKITPRO
        void *framebuffer;
        GXRModeObj *rmode = NULL;

        VIDEO_Init();
        WPAD_Init();

        rmode = VIDEO_GetPreferredMode(NULL);
#pragma GCC diagnostic ignored "-Wbad-function-cast"
        framebuffer = MEM_K0_TO_K1(SYS_AllocateFramebuffer(rmode));
#pragma GCC diagnostic pop
        console_init(framebuffer,20,20,rmode->fbWidth,rmode->xfbHeight,rmode->fbWidth*VI_DISPLAY_PIX_SZ);

        VIDEO_Configure(rmode);
        VIDEO_SetNextFramebuffer(framebuffer);
        VIDEO_SetBlack(FALSE);
        VIDEO_Flush();
        VIDEO_WaitVSync();
        if(rmode->viTVMode&VI_NON_INTERLACE) VIDEO_WaitVSync();
#endif

#ifdef HAVE_WNR
        if ((ret = wc_InitNetRandom(wnrConfigFile, NULL, 5000)) != 0) {
            err_sys("Whitewood netRandom global config failed",
                    WC_TEST_RET_ENC_EC(ret));
            return WC_TEST_RET_ENC_EC(ret);
        }
#endif
#ifndef WOLFSSL_ESPIDF
        args.argc = argc;
        args.argv = argv;
#endif
        if ((ret = wolfCrypt_Init()) != 0) {
            printf("wolfCrypt_Init failed %d\n", (int)ret);
            err_sys("Error with wolfCrypt_Init!\n", WC_TEST_RET_ENC_EC(ret));
        }

    #ifdef HAVE_WC_INTROSPECTION
        printf("Math: %s\n", wc_GetMathInfo());
    #endif

#ifdef WC_RNG_SEED_CB
    wc_SetSeed_Cb(wc_GenerateSeed);
#endif

    #ifdef HAVE_STACK_SIZE
        StackSizeCheck(&args, wolfcrypt_test);
    #else
        wolfcrypt_test(&args);
    #endif

        if ((ret = wolfCrypt_Cleanup()) != 0) {
            printf("wolfCrypt_Cleanup failed %d\n", (int)ret);
            err_sys("Error with wolfCrypt_Cleanup!\n", WC_TEST_RET_ENC_EC(ret));
        }

#ifdef HAVE_WNR
        if ((ret = wc_FreeNetRandom()) < 0)
            err_sys("Failed to free netRandom context",
                    WC_TEST_RET_ENC_EC(ret));
#endif /* HAVE_WNR */
#ifdef DOLPHIN_EMULATOR
        /* Returning from main panics the emulator. Just hang
         * and let the user force quit the emulator window. */
        printf("args.return_code: %d\n", args.return_code);
        printf("Testing complete. You may close the window now\n");
        while (1);
#endif

        printf("Exiting main with return code: %ld\n", (long int)args.return_code);

        return args.return_code;
    } /* wolfcrypt_test_main or wolf_test_task */

#endif /* NO_MAIN_DRIVER */

/* helper to save DER, convert to PEM and save PEM */
#if !defined(NO_ASN) && (defined(HAVE_ECC) || !defined(NO_DSA) || \
(!defined(NO_RSA) && (defined(WOLFSSL_KEY_GEN) || defined(WOLFSSL_CERT_GEN)))) \
     && !defined(WOLF_CRYPTO_CB_ONLY_ECC)

#if !defined(NO_FILESYSTEM) && !defined(NO_WRITE_TEMP_FILES)
#define SaveDerAndPem(d, dSz, fD, fP, pT) _SaveDerAndPem(d, dSz, fD, fP, pT, WC_TEST_RET_LN)
#else
#define SaveDerAndPem(d, dSz, fD, fP, pT) _SaveDerAndPem(d, dSz, NULL, NULL, pT, WC_TEST_RET_LN)
#endif

static wc_test_ret_t _SaveDerAndPem(const byte* der, int derSz,
    const char* fileDer, const char* filePem, int pemType, int calling_line)
{
#if !defined(NO_FILESYSTEM) && !defined(NO_WRITE_TEMP_FILES)
    wc_test_ret_t ret;
    XFILE derFile;

    derFile = XFOPEN(fileDer, "wb");
    if (!derFile) {
        return WC_TEST_RET_ENC(calling_line, 0, WC_TEST_RET_TAG_I);
    }
    ret = (int)XFWRITE(der, 1, derSz, derFile);
    XFCLOSE(derFile);
    if (ret != derSz) {
        return WC_TEST_RET_ENC(calling_line, 1, WC_TEST_RET_TAG_I);
    }
#endif

#ifdef WOLFSSL_DER_TO_PEM
    if (filePem) {
    #if !defined(NO_FILESYSTEM) && !defined(NO_WRITE_TEMP_FILES)
        XFILE pemFile;
    #endif
        byte* pem;
        int pemSz;

        /* calculate PEM size */
        pemSz = wc_DerToPem(der, derSz, NULL, 0, pemType);
        if (pemSz < 0) {
            return WC_TEST_RET_ENC(calling_line, 2, WC_TEST_RET_TAG_I);
        }
        pem = (byte*)XMALLOC(pemSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if (pem == NULL) {
            return WC_TEST_RET_ENC(calling_line, 3, WC_TEST_RET_TAG_I);
        }
        /* Convert to PEM */
        pemSz = wc_DerToPem(der, derSz, pem, pemSz, pemType);
        if (pemSz < 0) {
            XFREE(pem, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            return WC_TEST_RET_ENC(calling_line, 4, WC_TEST_RET_TAG_I);
        }
    #if !defined(NO_FILESYSTEM) && !defined(NO_WRITE_TEMP_FILES)
        pemFile = XFOPEN(filePem, "wb");
        if (!pemFile) {
            XFREE(pem, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            return WC_TEST_RET_ENC(calling_line, 5, WC_TEST_RET_TAG_I);
        }
        ret = (int)XFWRITE(pem, 1, pemSz, pemFile);
        XFCLOSE(pemFile);
        if (ret != pemSz) {
            XFREE(pem, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            return WC_TEST_RET_ENC(calling_line, 6, WC_TEST_RET_TAG_I);
        }
    #endif
        XFREE(pem, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif /* WOLFSSL_DER_TO_PEM */

    /* suppress unused variable warnings */
    (void)der;
    (void)derSz;
    (void)filePem;
    (void)fileDer;
    (void)pemType;
    (void)calling_line;

    return 0;
}
#endif /* WOLFSSL_KEY_GEN || WOLFSSL_CERT_GEN */

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t error_test(void)
{
    const char* errStr;
    char        out[WOLFSSL_MAX_ERROR_SZ];
    const char* unknownStr = wc_GetErrorString(0);

#ifdef NO_ERROR_STRINGS
    /* Ensure a valid error code's string matches an invalid code's.
     * The string is that error strings are not available.
     */
    errStr = wc_GetErrorString(OPEN_RAN_E);
    wc_ErrorString(OPEN_RAN_E, out);
    if (XSTRCMP(errStr, unknownStr) != 0)
        return WC_TEST_RET_ENC_NC;
    if (XSTRCMP(out, unknownStr) != 0)
        return WC_TEST_RET_ENC_NC;
#else
    int i;
    int j = 0;
    /* Values that are not or no longer error codes. */
    int missing[] = { -123, -124, -128, -129, -159, -163, -164,
                      -165, -166, -167, -168, -169, -233,   0 };

    /* Check that all errors have a string and it's the same through the two
     * APIs. Check that the values that are not errors map to the unknown
     * string.
     */
    for (i = MAX_CODE_E-1; i >= WC_LAST_E; i--) {
        errStr = wc_GetErrorString(i);
        wc_ErrorString(i, out);

        if (i != missing[j]) {
            if (XSTRCMP(errStr, unknownStr) == 0)
                return WC_TEST_RET_ENC_NC;
            if (XSTRCMP(out, unknownStr) == 0)
                return WC_TEST_RET_ENC_NC;
            if (XSTRCMP(errStr, out) != 0)
                return WC_TEST_RET_ENC_NC;
            if (XSTRLEN(errStr) >= WOLFSSL_MAX_ERROR_SZ)
                return WC_TEST_RET_ENC_NC;
        }
        else {
            j++;
            if (XSTRCMP(errStr, unknownStr) != 0)
                return WC_TEST_RET_ENC_NC;
            if (XSTRCMP(out, unknownStr) != 0)
                return WC_TEST_RET_ENC_NC;
        }
    }

    /* Check if the next possible value has been given a string. */
    errStr = wc_GetErrorString(i);
    wc_ErrorString(i, out);
    if (XSTRCMP(errStr, unknownStr) != 0)
        return WC_TEST_RET_ENC_NC;
    if (XSTRCMP(out, unknownStr) != 0)
        return WC_TEST_RET_ENC_NC;
#endif

    return 0;
}

#ifndef NO_CODING

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t base64_test(void)
{
    wc_test_ret_t ret;
    WOLFSSL_SMALL_STACK_STATIC const byte good[] = "A+Gd\0\0\0";
    WOLFSSL_SMALL_STACK_STATIC const byte goodEnd[] = "A+Gd \r\n";
    WOLFSSL_SMALL_STACK_STATIC const byte good_spaces[] = " A + G d \0";
    byte       out[128];
    word32     outLen;
#ifdef WOLFSSL_BASE64_ENCODE
    byte       data[3];
    word32     dataLen;
    byte       longData[79] = { 0 };
    WOLFSSL_SMALL_STACK_STATIC const byte symbols[] = "+/A=";
#endif
    WOLFSSL_SMALL_STACK_STATIC const byte badSmall[] = "AAA!Gdj=";
    WOLFSSL_SMALL_STACK_STATIC const byte badLarge[] = "AAA~Gdj=";
    WOLFSSL_SMALL_STACK_STATIC const byte badEOL[] = "A+Gd!AA";
    WOLFSSL_SMALL_STACK_STATIC const byte badPadding[] = "AA=A";
    WOLFSSL_SMALL_STACK_STATIC const byte badChar[] = ",-.:;<=>?@[\\]^_`";
    byte goodChar[] =
        "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
        "abcdefghijklmnopqrstuvwxyz"
        "0123456789+/;";
    byte charTest[] = "A+Gd\0\0\0";
    int        i;

    /* Good Base64 encodings. */
    outLen = sizeof(out);
    ret = Base64_Decode(good, sizeof(good), out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_Decode(goodEnd, sizeof(goodEnd), out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(goodChar);
    ret = Base64_Decode(goodChar, sizeof(goodChar), goodChar, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (outLen != 64 / 4 * 3)
        return WC_TEST_RET_ENC_NC;
    outLen = sizeof(out);
    ret = Base64_Decode(good_spaces, sizeof(good_spaces), out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Bad parameters. */
    outLen = 1;
    ret = Base64_Decode(good, sizeof(good), out, &outLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    outLen = sizeof(out);
    ret = Base64_Decode(badEOL, sizeof(badEOL), out, &outLen);
    if (ret != ASN_INPUT_E)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_Decode(badPadding, sizeof(badPadding), out, &outLen);
    if (ret != ASN_INPUT_E)
        return WC_TEST_RET_ENC_EC(ret);
    /* Bad character at each offset 0-3. */
    for (i = 0; i < 4; i++) {
        outLen = sizeof(out);
        ret = Base64_Decode(badSmall + i, 4, out, &outLen);
        if (ret != ASN_INPUT_E)
            return WC_TEST_RET_ENC_I(i);
        ret = Base64_Decode(badLarge + i, 4, out, &outLen);
        if (ret != ASN_INPUT_E)
            return WC_TEST_RET_ENC_I(i);
    }
    /* Invalid character less than 0x2b */
    for (i = 1; i < 0x2b; i++) {
        outLen = sizeof(out);
        charTest[0] = (byte)i;
        ret = Base64_Decode(charTest, sizeof(charTest), out, &outLen);
        if (ret != ASN_INPUT_E)
            return WC_TEST_RET_ENC_I(i);
    }
    /* Bad characters in range 0x2b - 0x7a. */
    for (i = 0; i < (int)sizeof(badChar) - 1; i++) {
        outLen = sizeof(out);
        charTest[0] = badChar[i];
        ret = Base64_Decode(charTest, sizeof(charTest), out, &outLen);
        if (ret != ASN_INPUT_E)
            return WC_TEST_RET_ENC_I(i);
    }
    /* Invalid character greater than 0x7a */
    for (i = 0x7b; i < 0x100; i++) {
        outLen = sizeof(out);
        charTest[0] = (byte)i;
        ret = Base64_Decode(charTest, sizeof(charTest), out, &outLen);
        if (ret != ASN_INPUT_E)
            return WC_TEST_RET_ENC_I(i);
    }


#ifdef WOLFSSL_BASE64_ENCODE
    /* Decode and encode all symbols - non-alphanumeric. */
    dataLen = sizeof(data);
    ret = Base64_Decode(symbols, sizeof(symbols), data, &dataLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_Encode(data, dataLen, NULL, &outLen);
    if (ret != LENGTH_ONLY_E)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_Encode(data, dataLen, out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = 7;
    ret = Base64_EncodeEsc(data, dataLen, out, &outLen);
    if (ret != BUFFER_E)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_EncodeEsc(data, dataLen, NULL, &outLen);
    if (ret != LENGTH_ONLY_E)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_EncodeEsc(data, dataLen, out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_Encode_NoNl(data, dataLen, out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Data that results in an encoding longer than one line. */
    outLen = sizeof(out);
    dataLen = sizeof(longData);
    ret = Base64_Encode(longData, dataLen, out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_EncodeEsc(longData, dataLen, out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    outLen = sizeof(out);
    ret = Base64_Encode_NoNl(longData, dataLen, out, &outLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    return 0;
}

#ifdef WOLFSSL_BASE16
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t base16_test(void)
{
    wc_test_ret_t ret;
    WOLFSSL_SMALL_STACK_STATIC const byte testData[] = "SomeDataToEncode\n";
    WOLFSSL_SMALL_STACK_STATIC const byte encodedTestData[] = "536F6D6544617461546F456E636F64650A00";
    byte   encoded[40];
    word32 encodedLen;
    byte   plain[40];
    word32 len;

    /* length returned includes null termination */
    encodedLen = sizeof(encoded);
    ret = Base16_Encode(testData, sizeof(testData), encoded, &encodedLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    len = (word32)XSTRLEN((char*)encoded);
    if (len != encodedLen - 1)
        return WC_TEST_RET_ENC_NC;

    len = sizeof(plain);
    ret = Base16_Decode(encoded, encodedLen - 1, plain, &len);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (len != sizeof(testData) || XMEMCMP(testData, plain, len) != 0)
        return WC_TEST_RET_ENC_NC;

    if (encodedLen != sizeof(encodedTestData) ||
        XMEMCMP(encoded, encodedTestData, encodedLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }

    return 0;
}
#endif /* WOLFSSL_BASE16 */
#endif /* !NO_CODING */

#ifndef NO_ASN
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t asn_test(void)
{
    wc_test_ret_t ret;
    /* ASN1 encoded date buffer */
    WOLFSSL_SMALL_STACK_STATIC const byte dateBuf[] = {0x17, 0x0d, 0x31, 0x36, 0x30, 0x38, 0x31, 0x31,
                            0x32, 0x30, 0x30, 0x37, 0x33, 0x37, 0x5a};
    byte format;
    int length;
    const byte* datePart;
#ifndef NO_ASN_TIME
    struct tm timearg;
    time_t now;
#endif

    ret = wc_GetDateInfo(dateBuf, (int)sizeof(dateBuf), &datePart, &format,
                         &length);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

#ifndef NO_ASN_TIME
    /* Parameter Validation tests. */
    if ((ret = wc_GetTime(NULL, sizeof(now))) != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    if ((ret = wc_GetTime(&now, 0)) != BUFFER_E)
        return WC_TEST_RET_ENC_EC(ret);

    now = 0;
    if ((ret = wc_GetTime(&now, sizeof(now))) != 0) {
        return WC_TEST_RET_ENC_EC(ret);
    }
    if (now == 0) {
        printf("RTC/Time not set!\n");
        return WC_TEST_RET_ENC_NC;
    }

    ret = wc_GetDateAsCalendarTime(datePart, length, format, &timearg);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif /* !NO_ASN_TIME */

    return 0;
}
#endif /* !NO_ASN */

#ifdef WOLFSSL_MD2
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t md2_test(void)
{
    wc_test_ret_t ret = 0;
    Md2  md2;
    byte hash[MD2_DIGEST_SIZE];

    testVector a, b, c, d, e, f, g;
    testVector test_md2[7];
    int times = sizeof(test_md2) / sizeof(testVector), i;

    a.input  = "";
    a.output = "\x83\x50\xe5\xa3\xe2\x4c\x15\x3d\xf2\x27\x5c\x9f\x80\x69"
               "\x27\x73";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = MD2_DIGEST_SIZE;

    b.input  = "a";
    b.output = "\x32\xec\x01\xec\x4a\x6d\xac\x72\xc0\xab\x96\xfb\x34\xc0"
               "\xb5\xd1";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = MD2_DIGEST_SIZE;

    c.input  = "abc";
    c.output = "\xda\x85\x3b\x0d\x3f\x88\xd9\x9b\x30\x28\x3a\x69\xe6\xde"
               "\xd6\xbb";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = MD2_DIGEST_SIZE;

    d.input  = "message digest";
    d.output = "\xab\x4f\x49\x6b\xfb\x2a\x53\x0b\x21\x9f\xf3\x30\x31\xfe"
               "\x06\xb0";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = MD2_DIGEST_SIZE;

    e.input  = "abcdefghijklmnopqrstuvwxyz";
    e.output = "\x4e\x8d\xdf\xf3\x65\x02\x92\xab\x5a\x41\x08\xc3\xaa\x47"
               "\x94\x0b";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = MD2_DIGEST_SIZE;

    f.input  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz012345"
               "6789";
    f.output = "\xda\x33\xde\xf2\xa4\x2d\xf1\x39\x75\x35\x28\x46\xc3\x03"
               "\x38\xcd";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = MD2_DIGEST_SIZE;

    g.input  = "1234567890123456789012345678901234567890123456789012345678"
               "9012345678901234567890";
    g.output = "\xd5\x97\x6f\x79\xd8\x3d\x3a\x0d\xc9\x80\x6c\x3c\x66\xf3"
               "\xef\xd8";
    g.inLen  = XSTRLEN(g.input);
    g.outLen = MD2_DIGEST_SIZE;

    test_md2[0] = a;
    test_md2[1] = b;
    test_md2[2] = c;
    test_md2[3] = d;
    test_md2[4] = e;
    test_md2[5] = f;
    test_md2[6] = g;

    wc_InitMd2(&md2);

    for (i = 0; i < times; ++i) {
        wc_Md2Update(&md2, (byte*)test_md2[i].input, (word32)test_md2[i].inLen);
        wc_Md2Final(&md2, hash);

        if (XMEMCMP(hash, test_md2[i].output, MD2_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Md2Hash((byte*)test_md2[i].input, (word32)test_md2[i].inLen, hash);
        if (ret != 0) {
            return WC_TEST_RET_ENC_I(i);
        }

        if (XMEMCMP(hash, test_md2[i].output, MD2_DIGEST_SIZE) != 0) {
            return WC_TEST_RET_ENC_I(i);
        }
    }

    return 0;
}
#endif

#ifndef NO_MD5
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t md5_test(void)
{
    wc_test_ret_t ret = 0;
    wc_Md5 md5, md5Copy;
    byte hash[WC_MD5_DIGEST_SIZE];
    byte hashcopy[WC_MD5_DIGEST_SIZE];
    testVector a, b, c, d, e, f;
    testVector test_md5[6];
    int times = sizeof(test_md5) / sizeof(testVector), i;

    a.input  = "";
    a.output = "\xd4\x1d\x8c\xd9\x8f\x00\xb2\x04\xe9\x80\x09\x98\xec\xf8\x42"
               "\x7e";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_MD5_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\x90\x01\x50\x98\x3c\xd2\x4f\xb0\xd6\x96\x3f\x7d\x28\xe1\x7f"
               "\x72";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_MD5_DIGEST_SIZE;

    c.input  = "message digest";
    c.output = "\xf9\x6b\x69\x7d\x7c\xb7\x93\x8d\x52\x5a\x2f\x31\xaa\xf1\x61"
               "\xd0";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_MD5_DIGEST_SIZE;

    d.input  = "abcdefghijklmnopqrstuvwxyz";
    d.output = "\xc3\xfc\xd3\xd7\x61\x92\xe4\x00\x7d\xfb\x49\x6c\xca\x67\xe1"
               "\x3b";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_MD5_DIGEST_SIZE;

    e.input  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz012345"
               "6789";
    e.output = "\xd1\x74\xab\x98\xd2\x77\xd9\xf5\xa5\x61\x1c\x2c\x9f\x41\x9d"
               "\x9f";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_MD5_DIGEST_SIZE;

    f.input  = "1234567890123456789012345678901234567890123456789012345678"
               "9012345678901234567890";
    f.output = "\x57\xed\xf4\xa2\x2b\xe3\xc9\x55\xac\x49\xda\x2e\x21\x07\xb6"
               "\x7a";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_MD5_DIGEST_SIZE;

    test_md5[0] = a;
    test_md5[1] = b;
    test_md5[2] = c;
    test_md5[3] = d;
    test_md5[4] = e;
    test_md5[5] = f;

    ret = wc_InitMd5_ex(&md5, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitMd5_ex(&md5Copy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Md5Free(&md5);
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Md5Update(&md5, (byte*)test_md5[i].input,
            (word32)test_md5[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        ret = wc_Md5GetHash(&md5, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        ret = wc_Md5Copy(&md5, &md5Copy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        ret = wc_Md5Final(&md5, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        wc_Md5Free(&md5Copy);

        if (XMEMCMP(hash, test_md5[i].output, WC_MD5_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, hashcopy, WC_MD5_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
    const char* large_digest =
        "\x44\xd0\x88\xce\xf1\x36\xd1\x78\xe9\xc8\xba\x84\xc3\xfd\xf6\xca";

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
#ifdef WOLFSSL_PIC32MZ_HASH
    wc_Md5SizeSet(&md5, times * sizeof(large_input));
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Md5Update(&md5, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Md5Final(&md5, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_MD5_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:

    wc_Md5Free(&md5);
    wc_Md5Free(&md5Copy);

    return ret;
}
#endif /* NO_MD5 */


#ifndef NO_MD4

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t md4_test(void)
{
    Md4  md4;
    byte hash[MD4_DIGEST_SIZE];

    testVector a, b, c, d, e, f, g;
    testVector test_md4[7];
    int times = sizeof(test_md4) / sizeof(testVector), i;

    a.input  = "";
    a.output = "\x31\xd6\xcf\xe0\xd1\x6a\xe9\x31\xb7\x3c\x59\xd7\xe0\xc0\x89"
               "\xc0";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = MD4_DIGEST_SIZE;

    b.input  = "a";
    b.output = "\xbd\xe5\x2c\xb3\x1d\xe3\x3e\x46\x24\x5e\x05\xfb\xdb\xd6\xfb"
               "\x24";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = MD4_DIGEST_SIZE;

    c.input  = "abc";
    c.output = "\xa4\x48\x01\x7a\xaf\x21\xd8\x52\x5f\xc1\x0a\xe8\x7a\xa6\x72"
               "\x9d";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = MD4_DIGEST_SIZE;

    d.input  = "message digest";
    d.output = "\xd9\x13\x0a\x81\x64\x54\x9f\xe8\x18\x87\x48\x06\xe1\xc7\x01"
               "\x4b";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = MD4_DIGEST_SIZE;

    e.input  = "abcdefghijklmnopqrstuvwxyz";
    e.output = "\xd7\x9e\x1c\x30\x8a\xa5\xbb\xcd\xee\xa8\xed\x63\xdf\x41\x2d"
               "\xa9";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = MD4_DIGEST_SIZE;

    f.input  = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz012345"
               "6789";
    f.output = "\x04\x3f\x85\x82\xf2\x41\xdb\x35\x1c\xe6\x27\xe1\x53\xe7\xf0"
               "\xe4";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = MD4_DIGEST_SIZE;

    g.input  = "1234567890123456789012345678901234567890123456789012345678"
               "9012345678901234567890";
    g.output = "\xe3\x3b\x4d\xdc\x9c\x38\xf2\x19\x9c\x3e\x7b\x16\x4f\xcc\x05"
               "\x36";
    g.inLen  = XSTRLEN(g.input);
    g.outLen = MD4_DIGEST_SIZE;

    test_md4[0] = a;
    test_md4[1] = b;
    test_md4[2] = c;
    test_md4[3] = d;
    test_md4[4] = e;
    test_md4[5] = f;
    test_md4[6] = g;

    wc_InitMd4(&md4);

    for (i = 0; i < times; ++i) {
        wc_Md4Update(&md4, (byte*)test_md4[i].input, (word32)test_md4[i].inLen);
        wc_Md4Final(&md4, hash);

        if (XMEMCMP(hash, test_md4[i].output, MD4_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);
    }

    return 0;
}

#endif /* NO_MD4 */

#ifndef NO_SHA

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sha_test(void)
{
    wc_test_ret_t ret = 0;
    wc_Sha sha, shaCopy;
    byte hash[WC_SHA_DIGEST_SIZE];
    byte hashcopy[WC_SHA_DIGEST_SIZE];
    testVector a, b, c, d, e;
    testVector test_sha[5];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xda\x39\xa3\xee\x5e\x6b\x4b\x0d\x32\x55\xbf\xef\x95\x60\x18"
               "\x90\xaf\xd8\x07\x09";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xA9\x99\x3E\x36\x47\x06\x81\x6A\xBA\x3E\x25\x71\x78\x50\xC2"
               "\x6C\x9C\xD0\xD8\x9D";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x84\x98\x3E\x44\x1C\x3B\xD2\x6E\xBA\xAE\x4A\xA1\xF9\x51\x29"
               "\xE5\xE5\x46\x70\xF1";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA_DIGEST_SIZE;

    d.input  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaa";
    d.output = "\x00\x98\xBA\x82\x4B\x5C\x16\x42\x7B\xD7\xA1\x12\x2A\x5A\x44"
               "\x2A\x25\xEC\x64\x4D";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA_DIGEST_SIZE;

    e.input  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaaaaaa";
    e.output = "\xAD\x5B\x3F\xDB\xCB\x52\x67\x78\xC2\x83\x9D\x2F\x15\x1E\xA7"
               "\x53\x99\x5E\x26\xA0";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;
    test_sha[3] = d;
    test_sha[4] = e;

    ret = wc_InitSha_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitSha_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_ShaFree(&sha);
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_ShaUpdate(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_ShaGetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_ShaCopy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_ShaFinal(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        wc_ShaFree(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#if defined(WOLFSSL_RENESAS_TSIP) || defined(WOLFSSL_RENESAS_SCEPROTECT) || \
    defined(HASH_SIZE_LIMIT)
    const char* large_digest =
            "\x1d\x6a\x5a\xf6\xe5\x7c\x86\xce\x7f\x7c\xaf\xd5\xdb\x08\xcd\x59"
            "\x15\x8c\x6d\xb6";
#else
    const char* large_digest =
           "\x8b\x77\x02\x48\x39\xe8\xdb\xd3\x9a\xf4\x05\x24\x66\x12\x2d\x9e"
           "\xc5\xd9\x0a\xac";
#endif
    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#if defined(WOLFSSL_RENESAS_TSIP) || defined(WOLFSSL_RENESAS_SCEPROTECT) || \
    defined(HASH_SIZE_LIMIT)
    times = 20;
#else
    times = 100;
#endif
#ifdef WOLFSSL_PIC32MZ_HASH
    wc_ShaSizeSet(&sha, times * sizeof(large_input));
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_ShaUpdate(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_ShaFinal(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:

    wc_ShaFree(&sha);
    wc_ShaFree(&shaCopy);

    return ret;
}

#endif /* NO_SHA */

#ifdef WOLFSSL_RIPEMD
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t ripemd_test(void)
{
    RipeMd  ripemd;
    wc_test_ret_t ret;
    byte hash[RIPEMD_DIGEST_SIZE];

    testVector a, b, c, d;
    testVector test_ripemd[4];
    int times = sizeof(test_ripemd) / sizeof(struct testVector), i;

    a.input  = "abc";
    a.output = "\x8e\xb2\x08\xf7\xe0\x5d\x98\x7a\x9b\x04\x4a\x8e\x98\xc6"
               "\xb0\x87\xf1\x5a\x0b\xfc";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = RIPEMD_DIGEST_SIZE;

    b.input  = "message digest";
    b.output = "\x5d\x06\x89\xef\x49\xd2\xfa\xe5\x72\xb8\x81\xb1\x23\xa8"
               "\x5f\xfa\x21\x59\x5f\x36";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = RIPEMD_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x12\xa0\x53\x38\x4a\x9c\x0c\x88\xe4\x05\xa0\x6c\x27\xdc"
               "\xf4\x9a\xda\x62\xeb\x2b";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = RIPEMD_DIGEST_SIZE;

    d.input  = "12345678901234567890123456789012345678901234567890123456"
               "789012345678901234567890";
    d.output = "\x9b\x75\x2e\x45\x57\x3d\x4b\x39\xf4\xdb\xd3\x32\x3c\xab"
               "\x82\xbf\x63\x32\x6b\xfb";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = RIPEMD_DIGEST_SIZE;

    test_ripemd[0] = a;
    test_ripemd[1] = b;
    test_ripemd[2] = c;
    test_ripemd[3] = d;

    ret = wc_InitRipeMd(&ripemd);
    if (ret != 0) {
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_RipeMdUpdate(&ripemd, (byte*)test_ripemd[i].input,
                              (word32)test_ripemd[i].inLen);
        if (ret != 0) {
            return WC_TEST_RET_ENC_I(i);
        }

        ret = wc_RipeMdFinal(&ripemd, hash);
        if (ret != 0) {
            return WC_TEST_RET_ENC_I(i);
        }

        if (XMEMCMP(hash, test_ripemd[i].output, RIPEMD_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);
    }

    return 0;
}
#endif /* WOLFSSL_RIPEMD */


#ifdef HAVE_BLAKE2


#define BLAKE2B_TESTS 3

static const byte blake2b_vec[BLAKE2B_TESTS][BLAKE2B_OUTBYTES] =
{
  {
    0x78, 0x6A, 0x02, 0xF7, 0x42, 0x01, 0x59, 0x03,
    0xC6, 0xC6, 0xFD, 0x85, 0x25, 0x52, 0xD2, 0x72,
    0x91, 0x2F, 0x47, 0x40, 0xE1, 0x58, 0x47, 0x61,
    0x8A, 0x86, 0xE2, 0x17, 0xF7, 0x1F, 0x54, 0x19,
    0xD2, 0x5E, 0x10, 0x31, 0xAF, 0xEE, 0x58, 0x53,
    0x13, 0x89, 0x64, 0x44, 0x93, 0x4E, 0xB0, 0x4B,
    0x90, 0x3A, 0x68, 0x5B, 0x14, 0x48, 0xB7, 0x55,
    0xD5, 0x6F, 0x70, 0x1A, 0xFE, 0x9B, 0xE2, 0xCE
  },
  {
    0x2F, 0xA3, 0xF6, 0x86, 0xDF, 0x87, 0x69, 0x95,
    0x16, 0x7E, 0x7C, 0x2E, 0x5D, 0x74, 0xC4, 0xC7,
    0xB6, 0xE4, 0x8F, 0x80, 0x68, 0xFE, 0x0E, 0x44,
    0x20, 0x83, 0x44, 0xD4, 0x80, 0xF7, 0x90, 0x4C,
    0x36, 0x96, 0x3E, 0x44, 0x11, 0x5F, 0xE3, 0xEB,
    0x2A, 0x3A, 0xC8, 0x69, 0x4C, 0x28, 0xBC, 0xB4,
    0xF5, 0xA0, 0xF3, 0x27, 0x6F, 0x2E, 0x79, 0x48,
    0x7D, 0x82, 0x19, 0x05, 0x7A, 0x50, 0x6E, 0x4B
  },
  {
    0x1C, 0x08, 0x79, 0x8D, 0xC6, 0x41, 0xAB, 0xA9,
    0xDE, 0xE4, 0x35, 0xE2, 0x25, 0x19, 0xA4, 0x72,
    0x9A, 0x09, 0xB2, 0xBF, 0xE0, 0xFF, 0x00, 0xEF,
    0x2D, 0xCD, 0x8E, 0xD6, 0xF8, 0xA0, 0x7D, 0x15,
    0xEA, 0xF4, 0xAE, 0xE5, 0x2B, 0xBF, 0x18, 0xAB,
    0x56, 0x08, 0xA6, 0x19, 0x0F, 0x70, 0xB9, 0x04,
    0x86, 0xC8, 0xA7, 0xD4, 0x87, 0x37, 0x10, 0xB1,
    0x11, 0x5D, 0x3D, 0xEB, 0xBB, 0x43, 0x27, 0xB5
  }
};



WOLFSSL_TEST_SUBROUTINE wc_test_ret_t blake2b_test(void)
{
    Blake2b b2b;
    byte    digest[64];
    byte    input[64];
    int     i, ret;

    for (i = 0; i < (int)sizeof(input); i++)
        input[i] = (byte)i;

    for (i = 0; i < BLAKE2B_TESTS; i++) {
        ret = wc_InitBlake2b(&b2b, 64);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        ret = wc_Blake2bUpdate(&b2b, input, i);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        ret = wc_Blake2bFinal(&b2b, digest, 64);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        if (XMEMCMP(digest, blake2b_vec[i], 64) != 0) {
            return WC_TEST_RET_ENC_I(i);
        }
    }

    return 0;
}
#endif /* HAVE_BLAKE2 */

#ifdef HAVE_BLAKE2S


#define BLAKE2S_TESTS 3

static const byte blake2s_vec[BLAKE2S_TESTS][BLAKE2S_OUTBYTES] =
{
  {
    0x69, 0x21, 0x7a, 0x30, 0x79, 0x90, 0x80, 0x94,
    0xe1, 0x11, 0x21, 0xd0, 0x42, 0x35, 0x4a, 0x7c,
    0x1f, 0x55, 0xb6, 0x48, 0x2c, 0xa1, 0xa5, 0x1e,
    0x1b, 0x25, 0x0d, 0xfd, 0x1e, 0xd0, 0xee, 0xf9,
  },
  {
    0xe3, 0x4d, 0x74, 0xdb, 0xaf, 0x4f, 0xf4, 0xc6,
    0xab, 0xd8, 0x71, 0xcc, 0x22, 0x04, 0x51, 0xd2,
    0xea, 0x26, 0x48, 0x84, 0x6c, 0x77, 0x57, 0xfb,
    0xaa, 0xc8, 0x2f, 0xe5, 0x1a, 0xd6, 0x4b, 0xea,
  },
  {
    0xdd, 0xad, 0x9a, 0xb1, 0x5d, 0xac, 0x45, 0x49,
    0xba, 0x42, 0xf4, 0x9d, 0x26, 0x24, 0x96, 0xbe,
    0xf6, 0xc0, 0xba, 0xe1, 0xdd, 0x34, 0x2a, 0x88,
    0x08, 0xf8, 0xea, 0x26, 0x7c, 0x6e, 0x21, 0x0c,
  }
};



WOLFSSL_TEST_SUBROUTINE wc_test_ret_t blake2s_test(void)
{
    Blake2s b2s;
    byte    digest[32];
    byte    input[64];
    int     i, ret;

    for (i = 0; i < (int)sizeof(input); i++)
        input[i] = (byte)i;

    for (i = 0; i < BLAKE2S_TESTS; i++) {
        ret = wc_InitBlake2s(&b2s, 32);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        ret = wc_Blake2sUpdate(&b2s, input, i);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        ret = wc_Blake2sFinal(&b2s, digest, 32);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        if (XMEMCMP(digest, blake2s_vec[i], 32) != 0) {
            return WC_TEST_RET_ENC_I(i);
        }
    }

    return 0;
}
#endif /* HAVE_BLAKE2S */


#ifdef WOLFSSL_SHA224
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sha224_test(void)
{
    wc_Sha224 sha, shaCopy;
    byte      hash[WC_SHA224_DIGEST_SIZE];
    byte      hashcopy[WC_SHA224_DIGEST_SIZE];
    wc_test_ret_t ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xd1\x4a\x02\x8c\x2a\x3a\x2b\xc9\x47\x61\x02\xbb\x28\x82\x34"
               "\xc4\x15\xa2\xb0\x1f\x82\x8e\xa6\x2a\xc5\xb3\xe4\x2f";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA224_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\x23\x09\x7d\x22\x34\x05\xd8\x22\x86\x42\xa4\x77\xbd\xa2\x55"
               "\xb3\x2a\xad\xbc\xe4\xbd\xa0\xb3\xf7\xe3\x6c\x9d\xa7";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA224_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x75\x38\x8b\x16\x51\x27\x76\xcc\x5d\xba\x5d\xa1\xfd\x89\x01"
               "\x50\xb0\xc6\x45\x5c\xb4\xf5\x8b\x19\x52\x52\x25\x25";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA224_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha224_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitSha224_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha224Free(&sha);
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha224Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha224GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha224Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha224Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        wc_Sha224Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA224_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA224_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

exit:
    wc_Sha224Free(&sha);
    wc_Sha224Free(&shaCopy);

    return ret;
}
#endif


#ifndef NO_SHA256
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sha256_test(void)
{
    wc_Sha256 sha, shaCopy;
    byte      hash[WC_SHA256_DIGEST_SIZE];
    byte      hashcopy[WC_SHA256_DIGEST_SIZE];
    wc_test_ret_t ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xe3\xb0\xc4\x42\x98\xfc\x1c\x14\x9a\xfb\xf4\xc8\x99\x6f\xb9"
               "\x24\x27\xae\x41\xe4\x64\x9b\x93\x4c\xa4\x95\x99\x1b\x78\x52"
               "\xb8\x55";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA256_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xBA\x78\x16\xBF\x8F\x01\xCF\xEA\x41\x41\x40\xDE\x5D\xAE\x22"
               "\x23\xB0\x03\x61\xA3\x96\x17\x7A\x9C\xB4\x10\xFF\x61\xF2\x00"
               "\x15\xAD";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA256_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x24\x8D\x6A\x61\xD2\x06\x38\xB8\xE5\xC0\x26\x93\x0C\x3E\x60"
               "\x39\xA3\x3C\xE4\x59\x64\xFF\x21\x67\xF6\xEC\xED\xD4\x19\xDB"
               "\x06\xC1";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA256_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha256_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitSha256_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha256Free(&sha);
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha256Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        }
        ret = wc_Sha256GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha256Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha256Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        wc_Sha256Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA256_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA256_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#ifdef HASH_SIZE_LIMIT
    const char* large_digest =
            "\xa4\x75\x9e\x7a\xa2\x03\x38\x32\x88\x66\xa2\xea\x17\xea\xf8\xc7"
            "\xfe\x4e\xc6\xbb\xe3\xbb\x71\xce\xe7\xdf\x7c\x04\x61\xb3\xc2\x2f";
#else
    const char* large_digest =
           "\x27\x78\x3e\x87\x96\x3a\x4e\xfb\x68\x29\xb5\x31\xc9\xba\x57\xb4"
           "\x4f\x45\x79\x7f\x67\x70\xbd\x63\x7f\xbf\x0d\x80\x7c\xbd\xba\xe0";
#endif
    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#ifdef HASH_SIZE_LIMIT
    times = 20;
#else
    times = 100;
#endif
#ifdef WOLFSSL_PIC32MZ_HASH
    wc_Sha256SizeSet(&sha, times * sizeof(large_input));
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Sha256Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
        }
    ret = wc_Sha256Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA256_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:

    wc_Sha256Free(&sha);
    wc_Sha256Free(&shaCopy);

    return ret;
}
#endif


#ifdef WOLFSSL_SHA512
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sha512_test(void)
{
    /*
    ** See https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA512.pdf
    */
    wc_Sha512 sha, shaCopy;
    byte      hash[WC_SHA512_DIGEST_SIZE];
    byte      hashcopy[WC_SHA512_DIGEST_SIZE];
    wc_test_ret_t ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xcf\x83\xe1\x35\x7e\xef\xb8\xbd\xf1\x54\x28\x50\xd6\x6d\x80"
               "\x07\xd6\x20\xe4\x05\x0b\x57\x15\xdc\x83\xf4\xa9\x21\xd3\x6c"
               "\xe9\xce\x47\xd0\xd1\x3c\x5d\x85\xf2\xb0\xff\x83\x18\xd2\x87"
               "\x7e\xec\x2f\x63\xb9\x31\xbd\x47\x41\x7a\x81\xa5\x38\x32\x7a"
               "\xf9\x27\xda\x3e";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA512_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xdd\xaf\x35\xa1\x93\x61\x7a\xba\xcc\x41\x73\x49\xae\x20\x41"
               "\x31\x12\xe6\xfa\x4e\x89\xa9\x7e\xa2\x0a\x9e\xee\xe6\x4b\x55"
               "\xd3\x9a\x21\x92\x99\x2a\x27\x4f\xc1\xa8\x36\xba\x3c\x23\xa3"
               "\xfe\xeb\xbd\x45\x4d\x44\x23\x64\x3c\xe8\x0e\x2a\x9a\xc9\x4f"
               "\xa5\x4c\xa4\x9f";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA512_DIGEST_SIZE;

    c.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    c.output = "\x8e\x95\x9b\x75\xda\xe3\x13\xda\x8c\xf4\xf7\x28\x14\xfc\x14"
               "\x3f\x8f\x77\x79\xc6\xeb\x9f\x7f\xa1\x72\x99\xae\xad\xb6\x88"
               "\x90\x18\x50\x1d\x28\x9e\x49\x00\xf7\xe4\x33\x1b\x99\xde\xc4"
               "\xb5\x43\x3a\xc7\xd3\x29\xee\xb6\xdd\x26\x54\x5e\x96\xe5\x5b"
               "\x87\x4b\xe9\x09";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA512_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha512_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitSha512_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha512Free(&sha);
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha512Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        wc_Sha512Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA512_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA512_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#ifdef HASH_SIZE_LIMIT
    const char* large_digest =
        "\x30\x9B\x96\xA6\xE9\x43\x78\x30\xA3\x71\x51\x61\xC1\xEB\xE1\xBE"
        "\xC8\xA5\xF9\x13\x5A\xD6\x6D\x9E\x46\x31\x31\x67\x8D\xE2\xC0\x0B"
        "\x2A\x1A\x03\xE1\xF3\x48\xA7\x33\xBD\x49\xF8\xFF\xF1\xC2\xC2\x95"
        "\xCB\xF0\xAF\x87\x61\x85\x58\x63\x6A\xCA\x70\x9C\x8B\x83\x3F\x5D";
#else
    const char* large_digest =
        "\x5a\x1f\x73\x90\xbd\x8c\xe4\x63\x54\xce\xa0\x9b\xef\x32\x78\x2d"
        "\x2e\xe7\x0d\x5e\x2f\x9d\x15\x1b\xdd\x2d\xde\x65\x0c\x7b\xfa\x83"
        "\x5e\x80\x02\x13\x84\xb8\x3f\xff\x71\x62\xb5\x09\x89\x63\xe1\xdc"
        "\xa5\xdc\xfc\xfa\x9d\x1a\x4d\xc0\xfa\x3a\x14\xf6\x01\x51\x90\xa4";
#endif

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#ifdef HASH_SIZE_LIMIT
    times = 20;
#else
    times = 100;
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Sha512Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha512Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA512_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);

#ifndef NO_UNALIGNED_MEMORY_TEST
    /* Unaligned memory access test */
    for (i = 1; i < 16; i++) {
        ret = wc_Sha512Update(&sha, (byte*)large_input + i,
            (word32)sizeof(large_input) - i);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
        ret = wc_Sha512Final(&sha, hash);
    }
#endif
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:
    wc_Sha512Free(&sha);
    wc_Sha512Free(&shaCopy);

    return ret;
}

#if !defined(WOLFSSL_NOSHA512_224) && \
   (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sha512_224_test(void)
{
    /*
    ** See https://csrc.nist.gov/Projects/cryptographic-standards-and-guidelines/example-values
    **
    ** NIST SHA512/224 Document Example:
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA512_224.pdf
    */
    wc_Sha512 sha, shaCopy;
    byte      hash[WC_SHA512_224_DIGEST_SIZE];
    byte      hashcopy[WC_SHA512_224_DIGEST_SIZE];
    wc_test_ret_t ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\x6e\xd0\xdd\x02"
               "\x80\x6f\xa8\x9e" "\x25\xde\x06\x0c"
               "\x19\xd3\xac\x86" "\xca\xbb\x87\xd6"
               "\xa0\xdd\xd0\x5c" "\x33\x3b\x84\xf4";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA512_224_DIGEST_SIZE;

    /*
    ** See page 1 in above document for the SHA512/224 "abc" test:
    */
    b.input  = "abc";
    /*
    ** See page 1 in above document for the SHA512/224 "abc" test digest:
    */
    b.output = "\x46\x34\x27\x0f"
               "\x70\x7b\x6a\x54" "\xda\xae\x75\x30"
               "\x46\x08\x42\xe2" "\x0e\x37\xed\x26"
               "\x5c\xee\xe9\xa4" "\x3e\x89\x24\xaa";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA512_224_DIGEST_SIZE;
    /*
    ** See page 4 in above for the 2-block test:
    */
    c.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    /*
    ** see page 9 in above document for the 2-block test message digest:
    */
    c.output = "\x23\xfe\xc5\xbb"
               "\x94\xd6\x0b\x23" "\x30\x81\x92\x64"
               "\x0b\x0c\x45\x33" "\x35\xd6\x64\x73"
               "\x4f\xe4\x0e\x72" "\x68\x67\x4a\xf9";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA512_224_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha512_224_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitSha512_224_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha512_224Free(&sha);
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha512_224Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512_224GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512_224Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512_224Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        wc_Sha512_224Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA512_224_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);


        if (XMEMCMP(hash, hashcopy, WC_SHA512_224_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#ifdef HASH_SIZE_LIMIT
    const char* large_digest =
        "\x98\x68\xc3\xd9\xb9\xef\x17\x53"
        "\x43\x66\x0e\x60\xdf\x29\xf8\xef"
        "\x96\xe3\x93\x34\x8c\x6f\xc0\xeb"
        "\x14\x6c\xcf\x6a";
#else
    const char* large_digest =
        "\x26\x5f\x98\xd1\x76\x49\x71\x4e"
        "\x82\xb7\x9d\x52\x32\x67\x9d\x56"
        "\x91\xf5\x88\xc3\x05\xbb\x3f\x90"
        "\xe2\x4e\x85\x05";
#endif

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#ifdef HASH_SIZE_LIMIT
    times = 20;
#else
    times = 100;
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Sha512_224Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha512_224Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA512_224_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);

#ifndef NO_UNALIGNED_MEMORY_TEST
    /* Unaligned memory access test */
    for (i = 1; i < 16; i++) {
        ret = wc_Sha512_224Update(&sha, (byte*)large_input + i,
            (word32)sizeof(large_input) - i);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
        ret = wc_Sha512_224Final(&sha, hash);
    }
#endif
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:
    wc_Sha512_224Free(&sha);
    wc_Sha512_224Free(&shaCopy);

    return ret;
} /* sha512_224_test */
#endif /* !defined(WOLFSSL_NOSHA512_224) && !FIPS ... */


#if !defined(WOLFSSL_NOSHA512_256) && \
   (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sha512_256_test(void)
{
    /*
    ** See https://csrc.nist.gov/Projects/cryptographic-standards-and-guidelines/example-values
    ** NIST SHA512/256 Document Example:
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA512_256.pdf
    */
    wc_Sha512 sha, shaCopy;
    byte      hash[WC_SHA512_256_DIGEST_SIZE];
    byte      hashcopy[WC_SHA512_256_DIGEST_SIZE];
    wc_test_ret_t ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\xc6\x72\xb8\xd1" "\xef\x56\xed\x28"
               "\xab\x87\xc3\x62" "\x2c\x51\x14\x06"
               "\x9b\xdd\x3a\xd7" "\xb8\xf9\x73\x74"
               "\x98\xd0\xc0\x1e" "\xce\xf0\x96\x7a";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA512_256_DIGEST_SIZE;

    /*
    ** See page 1 of above document for "abc" example:
    */
    b.input  = "abc";
    /*
    ** See page 4 of above document for "abc" example digest:
    */
    b.output = "\x53\x04\x8e\x26" "\x81\x94\x1e\xf9"
               "\x9b\x2e\x29\xb7" "\x6b\x4c\x7d\xab"
               "\xe4\xc2\xd0\xc6" "\x34\xfc\x6d\x46"
               "\xe0\xe2\xf1\x31" "\x07\xe7\xaf\x23";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA512_256_DIGEST_SIZE;

    /*
    ** See page 4 of above document for Two Block Message Sample:
    */
    c.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    /*
    ** See page 10 of above document for Two Block Message Sample digest:
    */
    c.output = "\x39\x28\xe1\x84" "\xfb\x86\x90\xf8"
               "\x40\xda\x39\x88" "\x12\x1d\x31\xbe"
               "\x65\xcb\x9d\x3e" "\xf8\x3e\xe6\x14"
               "\x6f\xea\xc8\x61" "\xe1\x9b\x56\x3a";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA512_256_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha512_256_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitSha512_256_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha512_256Free(&sha);
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha512_256Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512_256GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512_256Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha512_256Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        wc_Sha512_256Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA512_256_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);


        if (XMEMCMP(hash, hashcopy, WC_SHA512_256_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#ifdef HASH_SIZE_LIMIT
    const char* large_digest =
        "\x49\xcc\xbc\x7a\x93\x0b\x02\xb8"
        "\xad\x9a\x46\x51\x00\x1f\x13\x80"
        "\x35\x84\x36\xf1\xf2\x3c\xeb\xd8"
        "\x41\xd4\x06\x8b\x1d\x19\xad\x72";
#else
    const char* large_digest =
        "\x7a\xe3\x84\x05\xcb\x06\x22\x08"
        "\x7e\x2c\x65\x89\x1f\x26\x45\xfd"
        "\xad\xbc\x2e\x29\x83\x12\x84\x4b"
        "\xf2\xa0\xde\xbe\x06\x11\xd7\x44";
#endif

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#ifdef HASH_SIZE_LIMIT
    times = 20;
#else
    times = 100;
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Sha512_256Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha512_256Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA512_256_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);

#ifndef NO_UNALIGNED_MEMORY_TEST
    /* Unaligned memory access test */
    for (i = 1; i < 16; i++) {
        ret = wc_Sha512_256Update(&sha, (byte*)large_input + i,
            (word32)sizeof(large_input) - i);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
        ret = wc_Sha512_256Final(&sha, hash);
    }
#endif
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:
    wc_Sha512_256Free(&sha);
    wc_Sha512_256Free(&shaCopy);

    return ret;
} /* sha512_256_test */
#endif /* !defined(WOLFSSL_NOSHA512_256) && !FIPS ... */

#endif /* WOLFSSL_SHA512 */


#ifdef WOLFSSL_SHA384
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sha384_test(void)
{
    wc_Sha384 sha, shaCopy;
    byte      hash[WC_SHA384_DIGEST_SIZE];
    byte      hashcopy[WC_SHA384_DIGEST_SIZE];
    wc_test_ret_t ret = 0;

    testVector a, b, c;
    testVector test_sha[3];
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";

    a.output = "\x38\xb0\x60\xa7\x51\xac\x96\x38\x4c\xd9\x32\x7e\xb1\xb1\xe3"
               "\x6a\x21\xfd\xb7\x11\x14\xbe\x07\x43\x4c\x0c\xc7\xbf\x63\xf6"
               "\xe1\xda\x27\x4e\xde\xbf\xe7\x6f\x65\xfb\xd5\x1a\xd2\xf1\x48"
               "\x98\xb9\x5b";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA384_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xcb\x00\x75\x3f\x45\xa3\x5e\x8b\xb5\xa0\x3d\x69\x9a\xc6\x50"
               "\x07\x27\x2c\x32\xab\x0e\xde\xd1\x63\x1a\x8b\x60\x5a\x43\xff"
               "\x5b\xed\x80\x86\x07\x2b\xa1\xe7\xcc\x23\x58\xba\xec\xa1\x34"
               "\xc8\x25\xa7";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA384_DIGEST_SIZE;

    c.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    c.output = "\x09\x33\x0c\x33\xf7\x11\x47\xe8\x3d\x19\x2f\xc7\x82\xcd\x1b"
               "\x47\x53\x11\x1b\x17\x3b\x3b\x05\xd2\x2f\xa0\x80\x86\xe3\xb0"
               "\xf7\x12\xfc\xc7\xc7\x1a\x55\x7e\x2d\xb9\x66\xc3\xe9\xfa\x91"
               "\x74\x60\x39";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA384_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha384_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitSha384_ex(&shaCopy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sha384Free(&sha);
        return WC_TEST_RET_ENC_EC(ret);
    }

    for (i = 0; i < times; ++i) {
        ret = wc_Sha384Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha384GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha384Copy(&sha, &shaCopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha384Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        wc_Sha384Free(&shaCopy);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA384_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA384_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
#ifdef HASH_SIZE_LIMIT
    const char* large_digest =
        "\xB5\xAD\x66\x6F\xD9\x58\x5E\x68\xDD\x5E\x30\xD3\x95\x72\x33\xA4"
        "\xE9\x4B\x99\x3A\xEF\xF8\xE1\xBF\x1F\x05\x32\xAA\x16\x00\x82\xEC"
        "\x15\xDA\xF2\x75\xEE\xE9\x06\xAF\x52\x8A\x5C\xEF\x72\x81\x80\xD6";
#else
    const char* large_digest =
        "\x37\x01\xdb\xff\x1e\x40\x4f\xe1\xe2\xea\x0b\x40\xbb\x3b\x39\x9a"
        "\xcc\xe8\x44\x8e\x7e\xe5\x64\xb5\x6b\x7f\x56\x64\xa7\x2b\x84\xe3"
        "\xc5\xd7\x79\x03\x25\x90\xf7\xa4\x58\xcb\x97\xa8\x8b\xb1\xa4\x81";
#endif

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
#ifdef HASH_SIZE_LIMIT
    times = 20;
#else
    times = 100;
#endif
    for (i = 0; i < times; ++i) {
        ret = wc_Sha384Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha384Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA384_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:

    wc_Sha384Free(&sha);
    wc_Sha384Free(&shaCopy);

    return ret;
}
#endif /* WOLFSSL_SHA384 */

#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
static wc_test_ret_t sha3_224_test(void)
{
    wc_Sha3  sha;
    byte  hash[WC_SHA3_224_DIGEST_SIZE];
    byte  hashcopy[WC_SHA3_224_DIGEST_SIZE];

    testVector a, b, c;
    testVector test_sha[3];
    wc_test_ret_t ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\x6b\x4e\x03\x42\x36\x67\xdb\xb7\x3b\x6e\x15\x45\x4f\x0e\xb1"
               "\xab\xd4\x59\x7f\x9a\x1b\x07\x8e\x3f\x5b\x5a\x6b\xc7";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_224_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xe6\x42\x82\x4c\x3f\x8c\xf2\x4a\xd0\x92\x34\xee\x7d\x3c\x76"
               "\x6f\xc9\xa3\xa5\x16\x8d\x0c\x94\xad\x73\xb4\x6f\xdf";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_224_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x8a\x24\x10\x8b\x15\x4a\xda\x21\xc9\xfd\x55\x74\x49\x44\x79"
               "\xba\x5c\x7e\x7a\xb7\x6e\xf2\x64\xea\xd0\xfc\xce\x33";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_224_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha3_224(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_224_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha3_224_GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha3_224_Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA3_224_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA3_224_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
    const char* large_digest =
        "\x13\xe5\xd3\x98\x7b\x94\xda\x41\x12\xc7\x1e\x92\x3a\x19"
        "\x21\x20\x86\x6f\x24\xbf\x0a\x31\xbc\xfd\xd6\x70\x36\xf3";

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_224_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha3_224_Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA3_224_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:
    wc_Sha3_224_Free(&sha);

    return ret;
}
#endif /* WOLFSSL_NOSHA3_224 */

#ifndef WOLFSSL_NOSHA3_256
static wc_test_ret_t sha3_256_test(void)
{
    wc_Sha3  sha;
    byte  hash[WC_SHA3_256_DIGEST_SIZE];
    byte  hashcopy[WC_SHA3_256_DIGEST_SIZE];

    testVector a, b, c;
    testVector test_sha[3];
    wc_test_ret_t ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    byte large_input[1024];
    const char* large_digest =
        "\xdc\x90\xc0\xb1\x25\xdb\x2c\x34\x81\xa3\xff\xbc\x1e\x2e\x87\xeb"
        "\x6d\x70\x85\x61\xe0\xe9\x63\x61\xff\xe5\x84\x4b\x1f\x68\x05\x15";

#if defined(WOLFSSL_HASH_FLAGS) && !defined(WOLFSSL_ASYNC_CRYPT)
    /* test vector with hash of empty string */
    const char* Keccak256EmptyOut =
        "\xc5\xd2\x46\x01\x86\xf7\x23\x3c\x92\x7e\x7d\xb2\xdc\xc7\x03\xc0"
        "\xe5\x00\xb6\x53\xca\x82\x27\x3b\x7b\xfa\xd8\x04\x5d\x85\xa4\x70";
#endif

    /*
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA3-256_Msg0.pdf
    */
    a.input  = "";
    a.output = "\xa7\xff\xc6\xf8\xbf\x1e\xd7\x66\x51\xc1\x47\x56\xa0\x61\xd6"
               "\x62\xf5\x80\xff\x4d\xe4\x3b\x49\xfa\x82\xd8\x0a\x4b\x80\xf8"
               "\x43\x4a";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_256_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\x3a\x98\x5d\xa7\x4f\xe2\x25\xb2\x04\x5c\x17\x2d\x6b\xd3\x90"
               "\xbd\x85\x5f\x08\x6e\x3e\x9d\x52\x5b\x46\xbf\xe2\x45\x11\x43"
               "\x15\x32";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_256_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x41\xc0\xdb\xa2\xa9\xd6\x24\x08\x49\x10\x03\x76\xa8\x23\x5e"
               "\x2c\x82\xe1\xb9\x99\x8a\x99\x9e\x21\xdb\x32\xdd\x97\x49\x6d"
               "\x33\x76";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_256_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha3_256(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_256_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha3_256_GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha3_256_Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA3_256_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA3_256_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_256_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha3_256_Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA3_256_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

    /* this is a software only variant of SHA3 not supported by external hardware devices */
#if defined(WOLFSSL_HASH_FLAGS) && !defined(WOLFSSL_ASYNC_CRYPT)
    /* Test for Keccak256 */
    ret = wc_Sha3_SetFlags(&sha, WC_HASH_SHA3_KECCAK256);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha3_256_Update(&sha, (byte*)"", 0);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha3_256_Final(&sha, hash);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    if (XMEMCMP(hash, Keccak256EmptyOut, WC_SHA3_256_DIGEST_SIZE) != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    }
#endif /* WOLFSSL_HASH_FLAGS && !WOLFSSL_ASYNC_CRYPT */

exit:
    wc_Sha3_256_Free(&sha);

    return ret;
}
#endif /* WOLFSSL_NOSHA3_256 */

#ifndef WOLFSSL_NOSHA3_384
static wc_test_ret_t sha3_384_test(void)
{
    wc_Sha3  sha;
    byte  hash[WC_SHA3_384_DIGEST_SIZE];
    byte  buf[64];
#ifndef NO_INTM_HASH_TEST
    byte  hashcopy[WC_SHA3_384_DIGEST_SIZE];
#endif

    testVector a, b, c;
    testVector test_sha[3];
    wc_test_ret_t ret;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    /*
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA3-384_Msg0.pdf
    */
    a.input  = "";
    a.output = "\x0c\x63\xa7\x5b\x84\x5e\x4f\x7d\x01\x10\x7d\x85\x2e\x4c\x24"
               "\x85\xc5\x1a\x50\xaa\xaa\x94\xfc\x61\x99\x5e\x71\xbb\xee\x98"
               "\x3a\x2a\xc3\x71\x38\x31\x26\x4a\xdb\x47\xfb\x6b\xd1\xe0\x58"
               "\xd5\xf0\x04";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_384_DIGEST_SIZE;

#if defined(WOLFSSL_AFALG_XILINX_SHA3) || defined(WOLFSSL_XILINX_CRYPT) && !defined(WOLFSSL_XILINX_CRYPT_VERSAL)
    /* NIST test vector with a length that is a multiple of 4 */
    b.input  = "\x7d\x80\xb1\x60\xc4\xb5\x36\xa3\xbe\xb7\x99\x80\x59\x93\x44"
               "\x04\x7c\x5f\x82\xa1\xdf\xc3\xee\xd4";
    b.output = "\x04\x1c\xc5\x86\x1b\xa3\x34\x56\x3c\x61\xd4\xef\x97\x10\xd4"
               "\x89\x6c\x31\x1c\x92\xed\xbe\x0d\x7c\xd5\x3e\x80\x3b\xf2\xf4"
               "\xeb\x60\x57\x23\x55\x70\x77\x0c\xe8\x7c\x55\x20\xd7\xec\x14"
               "\x19\x87\x22";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_384_DIGEST_SIZE;
#else
    b.input  = "abc";
    b.output = "\xec\x01\x49\x82\x88\x51\x6f\xc9\x26\x45\x9f\x58\xe2\xc6\xad"
               "\x8d\xf9\xb4\x73\xcb\x0f\xc0\x8c\x25\x96\xda\x7c\xf0\xe4\x9b"
               "\xe4\xb2\x98\xd8\x8c\xea\x92\x7a\xc7\xf5\x39\xf1\xed\xf2\x28"
               "\x37\x6d\x25";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_384_DIGEST_SIZE;
#endif
    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x99\x1c\x66\x57\x55\xeb\x3a\x4b\x6b\xbd\xfb\x75\xc7\x8a\x49"
               "\x2e\x8c\x56\xa2\x2c\x5c\x4d\x7e\x42\x9b\xfd\xbc\x32\xb9\xd4"
               "\xad\x5a\xa0\x4a\x1f\x07\x6e\x62\xfe\xa1\x9e\xef\x51\xac\xd0"
               "\x65\x7c\x22";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_384_DIGEST_SIZE;

#if defined(WOLFSSL_XILINX_CRYPT) && !defined(WOLFSSL_XILINX_CRYPT_VERSAL)
    test_sha[0] = b; /* hardware acc. pre-Versal can not handle "" string */
#else
    test_sha[0] = a;
#endif
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha3_384(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    for (i = 0; i < times; ++i) {
        XMEMCPY(buf, test_sha[i].input, test_sha[i].inLen);
        ret = wc_Sha3_384_Update(&sha, buf,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    #ifndef NO_INTM_HASH_TEST
        ret = wc_Sha3_384_GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    #endif
        ret = wc_Sha3_384_Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA3_384_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    #ifndef NO_INTM_HASH_TEST
        if (XMEMCMP(hash, hashcopy, WC_SHA3_384_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    #endif
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
    const char* large_digest =
        "\x30\x44\xec\x17\xef\x47\x9f\x55\x36\x11\xd6\x3f\x8a\x31\x5a\x71"
        "\x8a\x71\xa7\x1d\x8e\x84\xe8\x6c\x24\x02\x2f\x7a\x08\x4e\xea\xd7"
        "\x42\x36\x5d\xa8\xc2\xb7\x42\xad\xec\x19\xfb\xca\xc6\x64\xb3\xa4";

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_384_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha3_384_Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA3_384_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:
    wc_Sha3_384_Free(&sha);

    return ret;
}
#endif /* WOLFSSL_NOSHA3_384 */

#ifndef WOLFSSL_NOSHA3_512
static wc_test_ret_t sha3_512_test(void)
{
    wc_Sha3  sha;
    byte  hash[WC_SHA3_512_DIGEST_SIZE];
    byte  hashcopy[WC_SHA3_512_DIGEST_SIZE];

    testVector a, b, c;
    testVector test_sha[3];
    wc_test_ret_t ret;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    /*
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHA3-512_Msg0.pdf
    */
    a.input  = "";
    a.output = "\xa6\x9f\x73\xcc\xa2\x3a\x9a\xc5\xc8\xb5\x67\xdc\x18\x5a\x75"
               "\x6e\x97\xc9\x82\x16\x4f\xe2\x58\x59\xe0\xd1\xdc\xc1\x47\x5c"
               "\x80\xa6\x15\xb2\x12\x3a\xf1\xf5\xf9\x4c\x11\xe3\xe9\x40\x2c"
               "\x3a\xc5\x58\xf5\x00\x19\x9d\x95\xb6\xd3\xe3\x01\x75\x85\x86"
               "\x28\x1d\xcd\x26";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_512_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\xb7\x51\x85\x0b\x1a\x57\x16\x8a\x56\x93\xcd\x92\x4b\x6b\x09"
               "\x6e\x08\xf6\x21\x82\x74\x44\xf7\x0d\x88\x4f\x5d\x02\x40\xd2"
               "\x71\x2e\x10\xe1\x16\xe9\x19\x2a\xf3\xc9\x1a\x7e\xc5\x76\x47"
               "\xe3\x93\x40\x57\x34\x0b\x4c\xf4\x08\xd5\xa5\x65\x92\xf8\x27"
               "\x4e\xec\x53\xf0";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_512_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x04\xa3\x71\xe8\x4e\xcf\xb5\xb8\xb7\x7c\xb4\x86\x10\xfc\xa8"
               "\x18\x2d\xd4\x57\xce\x6f\x32\x6a\x0f\xd3\xd7\xec\x2f\x1e\x91"
               "\x63\x6d\xee\x69\x1f\xbe\x0c\x98\x53\x02\xba\x1b\x0d\x8d\xc7"
               "\x8c\x08\x63\x46\xb5\x33\xb4\x9c\x03\x0d\x99\xa2\x7d\xaf\x11"
               "\x39\xd6\xe7\x5e";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_512_DIGEST_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;

    ret = wc_InitSha3_512(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_512_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha3_512_GetHash(&sha, hashcopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Sha3_512_Final(&sha, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, test_sha[i].output, WC_SHA3_512_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashcopy, WC_SHA3_512_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    byte large_input[1024];
    const char* large_digest =
        "\x9c\x13\x26\xb6\x26\xb2\x94\x31\xbc\xf4\x34\xe9\x6f\xf2\xd6\x29"
        "\x9a\xd0\x9b\x32\x63\x2f\x18\xa7\x5f\x23\xc9\x60\xc2\x32\x0c\xbc"
        "\x57\x77\x33\xf1\x83\x81\x8a\xd3\x15\x7c\x93\xdc\x80\x9f\xed\x61"
        "\x41\xa7\x5b\xfd\x32\x0e\x38\x15\xb0\x46\x3b\x7a\x4f\xfd\x44\x88";

    for (i = 0; i < (int)sizeof(large_input); i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Sha3_512_Update(&sha, (byte*)large_input,
            (word32)sizeof(large_input));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Sha3_512_Final(&sha, hash);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, WC_SHA3_512_DIGEST_SIZE) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:
    wc_Sha3_512_Free(&sha);

    return ret;
}
#endif /* WOLFSSL_NOSHA3_512 */

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sha3_test(void)
{
    wc_test_ret_t ret;

    (void)ret;

#ifndef WOLFSSL_NOSHA3_224
    if ((ret = sha3_224_test()) != 0)
        return ret;
#endif
#ifndef WOLFSSL_NOSHA3_256
    if ((ret = sha3_256_test()) != 0)
        return ret;
#endif
#ifndef WOLFSSL_NOSHA3_384
    if ((ret = sha3_384_test()) != 0)
        return ret;
#endif
#ifndef WOLFSSL_NOSHA3_512
    if ((ret = sha3_512_test()) != 0)
        return ret;
#endif

    return 0;
}
#endif /* WOLFSSL_SHA3 */

#ifdef WOLFSSL_SHAKE128
static wc_test_ret_t shake128_absorb_test(wc_Shake* sha, byte *large_input_buf,
                                size_t large_input_buf_size)
{
    testVector a, b, c, d, e;
    testVector test_sha[5];
    wc_test_ret_t ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    static const char large_digest[] =
        "\x2b\xd1\x69\x9f\xb3\x75\x40\x74\xb8\xb2\xd2\x0b\x92\x47\x9b\xfe"
        "\xc9\x91\x48\xbe\xda\xa4\x09\xd7\x61\x35\x18\x05\x07\x71\xa5\x61"
        "\x4d\xc4\x94\xad\xbe\x04\x7d\xad\x95\x2f\xeb\x2c\xc0\x10\x67\x43"
        "\x40\xf1\x4a\x58\x1c\x54\xfa\x24\x1c\x1a\x4e\x8d\x9b\xbc\xea\xa7"
        "\x32\xf2\x4c\xc7\x86\x05\x36\xdc\xb4\x42\xd8\x35\xd1\xb4\xa2\x79"
        "\xa2\xe6\xee\x67\x4f\xbf\x2a\x93\x41\x88\x25\x56\x29\x90\x1a\x06"
        "\xba\xfe\x9f\xa6\x1a\x74\xe8\x7e\x85\x4a\xc8\x58\x60\xb1\x7b\x18"
        "\xdf\x77\x59\x46\x04\xc1\xff\x4b\x9b\xcb\xad\xfe\x91\x28\xf0\x01"
        "\xc1\x33\xd0\x99\x99\x2e\x0c\x86\x84\x67\x4d\x37\xa4\x42\x45\x10"
        "\xdc\x8f\xdb\x6f\xa6\x9b\xee\x8a\x60\xa5\x1f\x95\x3f\x8f\xf5\x31"
        "\x4b\x1d\x48\x1e\x45\xff\x79\x5c\xbe\x72\xfc\x56\xed\x6d\x1a\x99"
        "\x7f\x23\x7c\xd1\xa5\x50\x9e\xb0\x4d\x61\x37\xa5\xcb\x24\x71\x3b"
        "\xa3\x60\x51\x2e\x80\x83\x8b\xe0\x55\x50\xa7\x1e\xcc\x9f\xac\x41"
        "\x77\x2c\x79\x22\x30\x09\x1b\x1a\x83\x5b\x2c\x48\xdc\x09\x7d\x59"
        "\x0d\xf0\x54\x17\xfb\x5e\x38\x68\xde\xdb\xc5\x93\xab\x17\x5f\x4b"
        "\x4d\x6d\xf2\xc7\x4e\x15\x1e\x10\x76\xc4\xcb\x87\xd8\xb7\x9d\xa8"
        "\xbf\xc5\x2e\x5e\xfc\xd3\x6c\x45\xd4\x5d\x72\x0f\x66\xeb\x67\x86"
        "\xfa\x6c\xd6\x80\xa4\x23\xcb\x5d\xed\x3c\xde\xdc\x5b\x3d\xca\x95"
        "\x43\x4b\xdc\xe8\x49\xd3\xe1\x01\xd4\xf1\xe4\x47\xcf\x56\xba\x71"
        "\xb4\x69\xed\xe7\xdb\x0f\x89\xd6\xbb\xcd\x1a\xff\xb4\xbe\x72\x26"
        "\xdc\x76\x79\xb3\x1a\x4b\xe6\x8d\x9b\x8e\xd9\xe9\xe6\xf9\xff\xa5";

    byte  hash[sizeof(large_digest) - 1];

    /*
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE128_Msg0.pdf
    */
    a.input  = "";
    a.output = "\x7f\x9c\x2b\xa4\xe8\x8f\x82\x7d\x61\x60\x45\x50\x76\x05\x85"
               "\x3e\xd7\x3b\x80\x93\xf6\xef\xbc\x88\xeb\x1a\x6e\xac\xfa\x66"
               "\xef\x26\x3c\xb1\xee\xa9\x88\x00\x4b\x93\x10\x3c\xfb\x0a\xee"
               "\xfd\x2a\x68\x6e\x01\xfa\x4a\x58\xe8\xa3\x63\x9c\xa8\xa1\xe3"
               "\xf9\xae\x57\xe2\x35\xb8\xcc\x87\x3c\x23\xdc\x62\xb8\xd2\x60"
               "\x16\x9a\xfa\x2f\x75\xab\x91\x6a\x58\xd9\x74\x91\x88\x35\xd2"
               "\x5e\x6a\x43\x50\x85\xb2\xba\xdf\xd6\xdf\xaa\xc3\x59\xa5\xef"
               "\xbb\x7b\xcc\x4b\x59\xd5\x38\xdf\x9a\x04\x30\x2e\x10\xc8\xbc"
               "\x1c\xbf\x1a\x0b\x3a\x51\x20\xea\x17\xcd\xa7\xcf\xad\x76\x5f"
               "\x56\x23\x47\x4d\x36\x8c\xcc\xa8\xaf\x00\x07\xcd\x9f\x5e\x4c"
               "\x84\x9f\x16\x7a\x58\x0b\x14\xaa\xbd\xef\xae\xe7\xee\xf4\x7c"
               "\xb0\xfc\xa9";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_128_BLOCK_SIZE;

    b.input  = "abc";
    b.output = "\x58\x81\x09\x2d\xd8\x18\xbf\x5c\xf8\xa3\xdd\xb7\x93\xfb\xcb"
               "\xa7\x40\x97\xd5\xc5\x26\xa6\xd3\x5f\x97\xb8\x33\x51\x94\x0f"
               "\x2c\xc8\x44\xc5\x0a\xf3\x2a\xcd\x3f\x2c\xdd\x06\x65\x68\x70"
               "\x6f\x50\x9b\xc1\xbd\xde\x58\x29\x5d\xae\x3f\x89\x1a\x9a\x0f"
               "\xca\x57\x83\x78\x9a\x41\xf8\x61\x12\x14\xce\x61\x23\x94\xdf"
               "\x28\x6a\x62\xd1\xa2\x25\x2a\xa9\x4d\xb9\xc5\x38\x95\x6c\x71"
               "\x7d\xc2\xbe\xd4\xf2\x32\xa0\x29\x4c\x85\x7c\x73\x0a\xa1\x60"
               "\x67\xac\x10\x62\xf1\x20\x1f\xb0\xd3\x77\xcf\xb9\xcd\xe4\xc6"
               "\x35\x99\xb2\x7f\x34\x62\xbb\xa4\xa0\xed\x29\x6c\x80\x1f\x9f"
               "\xf7\xf5\x73\x02\xbb\x30\x76\xee\x14\x5f\x97\xa3\x2a\xe6\x8e"
               "\x76\xab\x66\xc4\x8d\x51\x67\x5b\xd4\x9a\xcc\x29\x08\x2f\x56"
               "\x47\x58\x4e";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_128_BLOCK_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x1a\x96\x18\x2b\x50\xfb\x8c\x7e\x74\xe0\xa7\x07\x78\x8f\x55"
               "\xe9\x82\x09\xb8\xd9\x1f\xad\xe8\xf3\x2f\x8d\xd5\xcf\xf7\xbf"
               "\x21\xf5\x4e\xe5\xf1\x95\x50\x82\x5a\x6e\x07\x00\x30\x51\x9e"
               "\x94\x42\x63\xac\x1c\x67\x65\x28\x70\x65\x62\x1f\x9f\xcb\x32"
               "\x01\x72\x3e\x32\x23\xb6\x3a\x46\xc2\x93\x8a\xa9\x53\xba\x84"
               "\x01\xd0\xea\x77\xb8\xd2\x64\x90\x77\x55\x66\x40\x7b\x95\x67"
               "\x3c\x0f\x4c\xc1\xce\x9f\xd9\x66\x14\x8d\x7e\xfd\xff\x26\xbb"
               "\xf9\xf4\x8a\x21\xc6\xda\x35\xbf\xaa\x54\x56\x54\xf7\x0a\xe5"
               "\x86\xff\x10\x13\x14\x20\x77\x14\x83\xec\x92\xed\xab\x40\x8c"
               "\x76\x7b\xf4\xc5\xb4\xff\xfa\xa8\x0c\x8c\xa2\x14\xd8\x4c\x4d"
               "\xc7\x00\xd0\xc5\x06\x30\xb2\xff\xc3\x79\x3e\xa4\xd8\x72\x58"
               "\xb4\xc9\x54";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_128_BLOCK_SIZE;

    /* Taken from NIST CAVP test vectors - full rate output. */
    d.input  = "\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb"
               "\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8"
               "\x85\xe0";
    d.output = "\x93\x68\xf0\x15\x10\x92\x44\xeb\x02\x47\xfa\x3a\x0e\x57\xf5"
               "\x2e\xa7\xd9\xeb\xa2\x3d\xae\x7a\x19\x7f\x0a\x29\xe9\x22\x55"
               "\x06\x05\x98\x16\xb7\x84\x48\xb6\x49\x7a\x76\xeb\x96\x2d\xb3"
               "\xf8\x4d\x37\x60\xf1\xfe\xb4\xbd\xc1\xfd\x4a\xc9\x4e\x91\x7a"
               "\xc2\xea\x5e\x4f\x38\x37\x4a\xa5\x6e\x4f\x47\x67\xb8\xd7\x83"
               "\x1b\x2d\x51\x49\x5a\xb8\xea\xb7\xc9\x82\x20\xaf\x13\x41\x5a"
               "\x59\xbb\x7c\x17\x7a\xcd\x62\x8e\xf0\xff\xe3\x6c\xeb\x18\x59"
               "\x5d\x14\x4c\xbf\x25\xef\xc0\x6c\xd9\x56\xa5\x78\x20\x6e\xa8"
               "\xf9\x14\x5e\xf9\xce\x19\x50\x6a\x9d\x04\x4e\xc7\x00\x79\x9f"
               "\xa1\x41\x9b\xaf\x60\x52\xc0\xc1\xb4\x45\xf8\x35\x17\x57\xb0"
               "\xd0\x22\x87\x21\x89\xe2\xc0\x27\x3f\x82\xd9\x69\x69\x66\x3e"
               "\x55\x4d\x09";
    d.inLen  = 32;
    d.outLen = WC_SHA3_128_BLOCK_SIZE;
    /* Taken from NIST CAVP test vectors - more than one output block. */
    e.input  = "\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef"
               "\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67"
               "\x47\xe4";
    e.output = "\xe1\x7e\xab\x0d\xa4\x04\xf9\xb6\xac\xc0\x84\x97\x2f\xc5\x79"
               "\xe8\x6d\xaa\x76\x10\xa5\xe1\x7c\x23\x2f\x79\x19\x83\x96\xfd"
               "\x01\xc2\x4c\x34\xbb\x54\xf4\xb0\x1e\xf7\x40\xb4\x25\x33\x4a"
               "\x55\xdd\x24\x81\x3d\xc8\xea\x86\xf5\x6e\xf7\x27\x67\x26\x2b"
               "\xf2\x25\x74\x8c\xcc\x3d\x9f\x48\x6f\xfb\x72\x8f\x4e\xad\x29"
               "\x60\xc9\x6c\x3e\x44\x63\x86\xea\xce\x21\x9c\x84\x28\x16\x11"
               "\x63\x58\xb0\xf4\x2d\x7d\xff\xf7\xdd\x24\x11\xfa\x2a\x56\x79"
               "\xfd\x7a\x94\x77\x45\x75\xba\xf9\xfc\xad\x68\xa1\x9e\x30\xd1"
               "\x49\xb0\x59\xb5\x9c\x44\x6c\x4e\xdc\xa5\x9b\xc5\xa4\x79\x9d"
               "\xc4\x65\xaa\x9e\x78\x2c\xed\x9f\x21\xc5\x5d\xe2\x42\xdd\x25"
               "\xd0\xd9\xde\x60\xd0\x9f\xf8\x6a\xba\xf3\xa0\x3a\x76\x71\xb3"
               "\x05\x42\xdf\xbe\x72\xfc\x56\xed\x6d\x1a\x99\x7f\x23\x7c\xd1"
               "\xa5\x50\x9e\xb0\x4d\x61\x37\xa5\xcb\x24\x71\x3b\xa3\x60\x51"
               "\x2e\x80\x83\x8b\xe0\x55\x50\xa7\x1e\xcc\x9f\xac\x41\x77\x2c"
               "\x79\x22\x30\x09\x1b\x1a\x83\x5b\x2c\x48\xdc\x09\x7d\x59\x0d"
               "\xf0\x54\x17\xfb\x5e\x38\x68\xde\xdb\xc5\x93\xab\x17\x5f\x4b"
               "\x4d\x6d\xf2\xc7\x4e\x15\x1e\x10\x76\xc4\xcb\x87\xd8\xb7\x9d"
               "\xa8\xbf\xc5\x2e\x5e\xfc\xd3\x6c\x45\xd4\x5d\x72\x0f\x66\xeb"
               "\x67\x86\xfa\x6c\xd6\x80\xa4\x23\xcb\x5d\xed\x3c\xde\xdc\x5b"
               "\x3d\xca\x95\x43\x4b\xdc\xe8\x49\xd3\xe1\x01\xd4\xf1\xe4\x47"
               "\xcf\x56\xba\x71\xb4\x69\xed\xe7\xdb\x0f\x89\xd6\xbb\xcd\x1a"
               "\xff\xb4\xbe\x72\x26\xdc\x76\x79\xb3\x1a\x4b\xe6\x8d\x9b\x8e"
               "\xd9\xe9\xe6\xf9\xff\xa5";
    e.inLen  = 32;
    e.outLen = 2 * WC_SHA3_128_BLOCK_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;
    test_sha[3] = d;
    test_sha[4] = e;

    for (i = 0; i < times; ++i) {
        ret = wc_InitShake128(sha, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Shake128_Absorb(sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Shake128_SqueezeBlocks(sha, hash,
            (word32)test_sha[i].outLen / WC_SHA3_128_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, test_sha[i].output, (word32)test_sha[i].outLen) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    for (i = 0; i < (int)large_input_buf_size; i++) {
        large_input_buf[i] = (byte)(i & 0xFF);
    }
    ret = wc_InitShake128(sha, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    /* Absorb is non-incremental. */
    ret = wc_Shake128_Absorb(sha, (byte*)large_input_buf,
        (word32)large_input_buf_size);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    /* Able to squeeze out blocks incrementally. */
    ret = wc_Shake128_SqueezeBlocks(sha, hash, 1);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    ret = wc_Shake128_SqueezeBlocks(sha, hash,
        ((word32)sizeof(hash) / WC_SHA3_128_BLOCK_SIZE) - 1);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, sizeof(hash)) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:
    return ret;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t shake128_test(void)
{
    wc_Shake  sha;
    byte  hash[250];

    testVector a, b, c, d, e;
    testVector test_sha[5];
    wc_test_ret_t ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

#define SHAKE128_LARGE_INPUT_BUFSIZ 1024
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte *large_input = NULL;
#else
    byte large_input[SHAKE128_LARGE_INPUT_BUFSIZ];
#endif
    static const char large_digest[] =
        "\x88\xd7\x0e\x86\x46\x72\x6b\x3d\x7d\x22\xe1\xa9\x2d\x02\xdb\x35"
        "\x92\x4f\x1b\x03\x90\xee\xa3\xce\xd1\x3a\x08\x3a\xd7\x4e\x10\xdf"
        "\x09\x67\x33\x35\x4f\xdd\x38\x50\x5b\xcb\x75\xc7\xba\x65\xe5\xe8"
        "\xb8\x76\xde\xc5\xee\xd7\xf1\x65\x93\x4e\x5e\xc4\xb1\xd7\x6b\xee"
        "\x4b\x57\x48\xf5\x38\x49\x9e\x45\xa0\xf7\x32\xe9\x05\x26\x6a\x10"
        "\x70\xd4\x7c\x19\x01\x1f\x6d\x37\xba\x7b\x74\xc2\xbc\xb6\xbc\x74"
        "\xa3\x66\x6c\x9b\x11\x84\x9d\x4a\x36\xbc\x8a\x0d\x4c\xe3\x39\xfa"
        "\xfa\x1b";


    /*
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE128_Msg0.pdf
    */
    a.input  = "";
    a.output = "\x7f\x9c\x2b\xa4\xe8\x8f\x82\x7d\x61\x60\x45\x50\x76\x05\x85"
               "\x3e\xd7\x3b\x80\x93\xf6\xef\xbc\x88\xeb\x1a\x6e\xac\xfa\x66"
               "\xef\x26\x3c\xb1\xee\xa9\x88\x00\x4b\x93\x10\x3c\xfb\x0a\xee"
               "\xfd\x2a\x68\x6e\x01\xfa\x4a\x58\xe8\xa3\x63\x9c\xa8\xa1\xe3"
               "\xf9\xae\x57\xe2\x35\xb8\xcc\x87\x3c\x23\xdc\x62\xb8\xd2\x60"
               "\x16\x9a\xfa\x2f\x75\xab\x91\x6a\x58\xd9\x74\x91\x88\x35\xd2"
               "\x5e\x6a\x43\x50\x85\xb2\xba\xdf\xd6\xdf\xaa\xc3\x59\xa5\xef"
               "\xbb\x7b\xcc\x4b\x59\xd5\x38\xdf\x9a";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = 114;

    b.input  = "abc";
    b.output = "\x58\x81\x09\x2d\xd8\x18\xbf\x5c\xf8\xa3\xdd\xb7\x93\xfb\xcb"
               "\xa7\x40\x97\xd5\xc5\x26\xa6\xd3\x5f\x97\xb8\x33\x51\x94\x0f"
               "\x2c\xc8\x44\xc5\x0a\xf3\x2a\xcd\x3f\x2c\xdd\x06\x65\x68\x70"
               "\x6f\x50\x9b\xc1\xbd\xde\x58\x29\x5d\xae\x3f\x89\x1a\x9a\x0f"
               "\xca\x57\x83\x78\x9a\x41\xf8\x61\x12\x14\xce\x61\x23\x94\xdf"
               "\x28\x6a\x62\xd1\xa2\x25\x2a\xa9\x4d\xb9\xc5\x38\x95\x6c\x71"
               "\x7d\xc2\xbe\xd4\xf2\x32\xa0\x29\x4c\x85\x7c\x73\x0a\xa1\x60"
               "\x67\xac\x10\x62\xf1\x20\x1f\xb0\xd3";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = 114;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x1a\x96\x18\x2b\x50\xfb\x8c\x7e\x74\xe0\xa7\x07\x78\x8f\x55"
               "\xe9\x82\x09\xb8\xd9\x1f\xad\xe8\xf3\x2f\x8d\xd5\xcf\xf7\xbf"
               "\x21\xf5\x4e\xe5\xf1\x95\x50\x82\x5a\x6e\x07\x00\x30\x51\x9e"
               "\x94\x42\x63\xac\x1c\x67\x65\x28\x70\x65\x62\x1f\x9f\xcb\x32"
               "\x01\x72\x3e\x32\x23\xb6\x3a\x46\xc2\x93\x8a\xa9\x53\xba\x84"
               "\x01\xd0\xea\x77\xb8\xd2\x64\x90\x77\x55\x66\x40\x7b\x95\x67"
               "\x3c\x0f\x4c\xc1\xce\x9f\xd9\x66\x14\x8d\x7e\xfd\xff\x26\xbb"
               "\xf9\xf4\x8a\x21\xc6\xda\x35\xbf\xaa";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = 114;

    /* Taken from NIST CAVP test vectors - full rate output. */
    d.input  = "\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb"
               "\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8"
               "\x85\xe0";
    d.output = "\x93\x68\xf0\x15\x10\x92\x44\xeb\x02\x47\xfa\x3a\x0e\x57\xf5"
               "\x2e\xa7\xd9\xeb\xa2\x3d\xae\x7a\x19\x7f\x0a\x29\xe9\x22\x55"
               "\x06\x05\x98\x16\xb7\x84\x48\xb6\x49\x7a\x76\xeb\x96\x2d\xb3"
               "\xf8\x4d\x37\x60\xf1\xfe\xb4\xbd\xc1\xfd\x4a\xc9\x4e\x91\x7a"
               "\xc2\xea\x5e\x4f\x38\x37\x4a\xa5\x6e\x4f\x47\x67\xb8\xd7\x83"
               "\x1b\x2d\x51\x49\x5a\xb8\xea\xb7\xc9\x82\x20\xaf\x13\x41\x5a"
               "\x59\xbb\x7c\x17\x7a\xcd\x62\x8e\xf0\xff\xe3\x6c\xeb\x18\x59"
               "\x5d\x14\x4c\xbf\x25\xef\xc0\x6c\xd9\x56\xa5\x78\x20\x6e\xa8"
               "\xf9\x14\x5e\xf9\xce\x19\x50\x6a\x9d\x04\x4e\xc7\x00\x79\x9f"
               "\xa1";
    d.inLen  = 32;
    d.outLen = 136;
    /* Taken from NIST CAVP test vectors - more than one output block. */
    e.input  = "\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef"
               "\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67"
               "\x47\xe4";
    e.output = "\xe1\x7e\xab\x0d\xa4\x04\xf9\xb6\xac\xc0\x84\x97\x2f\xc5\x79"
               "\xe8\x6d\xaa\x76\x10\xa5\xe1\x7c\x23\x2f\x79\x19\x83\x96\xfd"
               "\x01\xc2\x4c\x34\xbb\x54\xf4\xb0\x1e\xf7\x40\xb4\x25\x33\x4a"
               "\x55\xdd\x24\x81\x3d\xc8\xea\x86\xf5\x6e\xf7\x27\x67\x26\x2b"
               "\xf2\x25\x74\x8c\xcc\x3d\x9f\x48\x6f\xfb\x72\x8f\x4e\xad\x29"
               "\x60\xc9\x6c\x3e\x44\x63\x86\xea\xce\x21\x9c\x84\x28\x16\x11"
               "\x63\x58\xb0\xf4\x2d\x7d\xff\xf7\xdd\x24\x11\xfa\x2a\x56\x79"
               "\xfd\x7a\x94\x77\x45\x75\xba\xf9\xfc\xad\x68\xa1\x9e\x30\xd1"
               "\x49\xb0\x59\xb5\x9c\x44\x6c\x4e\xdc\xa5\x9b\xc5\xa4\x79\x9d"
               "\xc4\x65\xaa\x9e\x78\x2c\xed\x9f\x21\xc5\x5d\xe2\x42\xdd\x25"
               "\xd0\xd9\xde\x60\xd0\x9f\xf8\x6a\xba\xf3\xa0\x3a\x76\x71\xb3"
               "\x05\x42\xdf\xbe\x72\xfc\x56\xed\x6d\x1a\x99\x7f\x23\x7c\xd1"
               "\xa5\x50\x9e\xb0\x4d\x61\x37\xa5\xcb\x24\x71\x3b\xa3\x60\x51"
               "\x2e\x80\x83\x8b\xe0\x55\x50\xa7\x1e\xcc\x9f\xac\x41\x77\x2c"
               "\x79\x22\x30\x09\x1b\x1a\x83\x5b\x2c\x48\xdc\x09\x7d\x59\x0d"
               "\xf0\x54\x17\xfb\x5e\x38\x68\xde\xdb\xc5\x93\xab\x17\x5f\x4b"
               "\x4d\x6d\xf2\xc7\x4e\x15\x1e\x10\x76\xc4";
    e.inLen  = 32;
    e.outLen = 250;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;
    test_sha[3] = d;
    test_sha[4] = e;

    ret = wc_InitShake128(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    for (i = 0; i < times; ++i) {
        ret = wc_Shake128_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Shake128_Final(&sha, hash, (word32)test_sha[i].outLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, test_sha[i].output, test_sha[i].outLen) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    large_input = (byte *)XMALLOC(SHAKE128_LARGE_INPUT_BUFSIZ, NULL,
                                  DYNAMIC_TYPE_TMP_BUFFER);
    if (large_input == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit);
#endif

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    for (i = 0; i < SHAKE128_LARGE_INPUT_BUFSIZ; i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Shake128_Update(&sha, (byte*)large_input,
            SHAKE128_LARGE_INPUT_BUFSIZ);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Shake128_Final(&sha, hash, (word32)sizeof(hash));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, sizeof(large_digest) - 1) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

    ret = shake128_absorb_test(&sha, large_input, SHAKE128_LARGE_INPUT_BUFSIZ);

exit:
    wc_Shake128_Free(&sha);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (large_input != NULL)
        XFREE(large_input, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif

#ifdef WOLFSSL_SHAKE256
static wc_test_ret_t shake256_absorb_test(wc_Shake* sha, byte *large_input_buf,
                                size_t large_input_buf_size)
{
    testVector a, b, c, d, e;
    testVector test_sha[5];
    wc_test_ret_t ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

    static const char large_digest[] =
        "\x21\x25\x8e\xae\x6e\x4f\xa7\xe1\xb9\x6d\xa7\xc9\x7d\x46\x03\x69"
        "\x29\x0d\x81\x49\xba\x5d\xaf\x37\xfd\xeb\x25\x52\x1d\xd9\xbd\x65"
        "\xfa\x99\xb9\xd1\x70\x6b\xeb\xd4\xc1\x2c\xea\x24\x20\x27\xa7\xcd"
        "\xfa\xe1\x81\xd9\xd5\xc1\x1c\xc7\xe9\x70\xc3\xc7\x21\x6f\x32\x22"
        "\xe3\x27\xdb\x58\x5e\xea\x18\x2d\x63\x4d\x14\x6c\x94\xcf\x2b\x7e"
        "\x6e\x2a\x74\xf3\xe0\xac\xb3\xb2\xcc\xef\x38\xe9\xe7\x35\xb3\xc5"
        "\x77\x9d\xff\xe3\x08\x8e\xf8\x2c\x89\xbb\x45\x22\x16\x99\x91\xc0"
        "\xe7\x71\x57\x75\xc5\xb1\xc6\xaf\x27\xcb\x64\x8c\xc4\xee\x3d\x5f"
        "\x4c\x35\xfb\x1c\xf3\xf8\x0e\xfd\x5e\xfc\x07\xd8\x4d\x55\x32\x49"
        "\x45\x0d\xab\x4a\x49\xc4\x83\xde\xd2\x50\xc9\x33\x8f\x85\xcd\x93"
        "\x7a\xe6\x6b\xb4\x36\xf3\xb4\x02\x6e\x85\x9f\xda\x1c\xa5\x71\x43"
        "\x2f\x3b\xfc\x09\xe7\xc0\x3c\xa4\xd1\x83\xb7\x41\x11\x1c\xa0\x48"
        "\x3d\x0e\xda\xbc\x03\xfe\xb2\x3b\x17\xee\x48\xe8\x44\xba\x24\x08"
        "\xd9\xdc\xfd\x01\x39\xd2\xe8\xc7\x31\x01\x25\xae\xe8\x01\xc6\x1a"
        "\xb7\x90\x0d\x1e\xfc\x47\xc0\x78\x28\x17\x66\xf3\x61\xc5\xe6\x11"
        "\x13\x46\x23\x5e\x1d\xc3\x83\x25\x66\x6c\x68\x1b\x30\xdd\xc4\xe6"
        "\x83\x8b\x0f\x23\x58\x7e\x06\x5f\x4a\x2b\xed\xc9\x6c\x97\x68\x44";

    byte  hash[sizeof(large_digest) - 1];

    /*
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE256_Msg0.pdf
    */
    a.input  = "";
    a.output = "\x46\xb9\xdd\x2b\x0b\xa8\x8d\x13\x23\x3b\x3f\xeb\x74\x3e\xeb"
               "\x24\x3f\xcd\x52\xea\x62\xb8\x1b\x82\xb5\x0c\x27\x64\x6e\xd5"
               "\x76\x2f\xd7\x5d\xc4\xdd\xd8\xc0\xf2\x00\xcb\x05\x01\x9d\x67"
               "\xb5\x92\xf6\xfc\x82\x1c\x49\x47\x9a\xb4\x86\x40\x29\x2e\xac"
               "\xb3\xb7\xc4\xbe\x14\x1e\x96\x61\x6f\xb1\x39\x57\x69\x2c\xc7"
               "\xed\xd0\xb4\x5a\xe3\xdc\x07\x22\x3c\x8e\x92\x93\x7b\xef\x84"
               "\xbc\x0e\xab\x86\x28\x53\x34\x9e\xc7\x55\x46\xf5\x8f\xb7\xc2"
               "\x77\x5c\x38\x46\x2c\x50\x10\xd8\x46\xc1\x85\xc1\x51\x11\xe5"
               "\x95\x52\x2a\x6b\xcd\x16\xcf\x86\xf3\xd1\x22\x10\x9e\x3b\x1f"
               "\xdd";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA3_256_BLOCK_SIZE;

    b.input  = "abc";
    b.output = "\x48\x33\x66\x60\x13\x60\xa8\x77\x1c\x68\x63\x08\x0c\xc4\x11"
               "\x4d\x8d\xb4\x45\x30\xf8\xf1\xe1\xee\x4f\x94\xea\x37\xe7\x8b"
               "\x57\x39\xd5\xa1\x5b\xef\x18\x6a\x53\x86\xc7\x57\x44\xc0\x52"
               "\x7e\x1f\xaa\x9f\x87\x26\xe4\x62\xa1\x2a\x4f\xeb\x06\xbd\x88"
               "\x01\xe7\x51\xe4\x13\x85\x14\x12\x04\xf3\x29\x97\x9f\xd3\x04"
               "\x7a\x13\xc5\x65\x77\x24\xad\xa6\x4d\x24\x70\x15\x7b\x3c\xdc"
               "\x28\x86\x20\x94\x4d\x78\xdb\xcd\xdb\xd9\x12\x99\x3f\x09\x13"
               "\xf1\x64\xfb\x2c\xe9\x51\x31\xa2\xd0\x9a\x3e\x6d\x51\xcb\xfc"
               "\x62\x27\x20\xd7\xa7\x5c\x63\x34\xe8\xa2\xd7\xec\x71\xa7\xcc"
               "\x29";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA3_256_BLOCK_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x4d\x8c\x2d\xd2\x43\x5a\x01\x28\xee\xfb\xb8\xc3\x6f\x6f\x87"
               "\x13\x3a\x79\x11\xe1\x8d\x97\x9e\xe1\xae\x6b\xe5\xd4\xfd\x2e"
               "\x33\x29\x40\xd8\x68\x8a\x4e\x6a\x59\xaa\x80\x60\xf1\xf9\xbc"
               "\x99\x6c\x05\xac\xa3\xc6\x96\xa8\xb6\x62\x79\xdc\x67\x2c\x74"
               "\x0b\xb2\x24\xec\x37\xa9\x2b\x65\xdb\x05\x39\xc0\x20\x34\x55"
               "\xf5\x1d\x97\xcc\xe4\xcf\xc4\x91\x27\xd7\x26\x0a\xfc\x67\x3a"
               "\xf2\x08\xba\xf1\x9b\xe2\x12\x33\xf3\xde\xbe\x78\xd0\x67\x60"
               "\xcf\xa5\x51\xee\x1e\x07\x91\x41\xd4\x9d\xd3\xef\x7e\x18\x2b"
               "\x15\x24\xdf\x82\xea\x1c\xef\xe1\xc6\xc3\x96\x61\x75\xf0\x22"
               "\x8d";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA3_256_BLOCK_SIZE;

    /* Taken from NIST CAVP test vectors - full rate output. */
    d.input  = "\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb"
               "\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8"
               "\x85\xe0";
    d.output = "\x00\x64\x8a\xfb\xc5\xe6\x51\x64\x9d\xb1\xfd\x82\x93\x6b\x00"
               "\xdb\xbc\x12\x2f\xb4\xc8\x77\x86\x0d\x38\x5c\x49\x50\xd5\x6d"
               "\xe7\xe0\x96\xd6\x13\xd7\xa3\xf2\x7e\xd8\xf2\x63\x34\xb0\xcc"
               "\xc1\x40\x7b\x41\xdc\xcb\x23\xdf\xaa\x52\x98\x18\xd1\x12\x5c"
               "\xd5\x34\x80\x92\x52\x43\x66\xb8\x5f\xab\xb9\x7c\x6c\xd1\xe6"
               "\x06\x6f\x45\x9b\xcc\x56\x6d\xa8\x7e\xc9\xb7\xba\x36\x79\x2d"
               "\x11\x8a\xc3\x9a\x4c\xce\xf6\x19\x2b\xbf\x3a\x54\xaf\x18\xe5"
               "\x7b\x0c\x14\x61\x01\xf6\xae\xaa\x82\x2b\xc4\xb4\xc9\x70\x8b"
               "\x09\xf0\xb3\xba\xb4\x1b\xcc\xe9\x64\xd9\x99\xd1\x10\x7b\xd7"
               "\xc2";
    d.inLen  = 32;
    d.outLen = WC_SHA3_256_BLOCK_SIZE;
    /* Taken from NIST CAVP test vectors - more than one output block. */
    e.input  = "\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef"
               "\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67"
               "\x47\xe4";
    e.output = "\x2e\x97\x5f\x6a\x8a\x14\xf0\x70\x4d\x51\xb1\x36\x67\xd8\x19"
               "\x5c\x21\x9f\x71\xe6\x34\x56\x96\xc4\x9f\xa4\xb9\xd0\x8e\x92"
               "\x25\xd3\xd3\x93\x93\x42\x51\x52\xc9\x7e\x71\xdd\x24\x60\x1c"
               "\x11\xab\xcf\xa0\xf1\x2f\x53\xc6\x80\xbd\x3a\xe7\x57\xb8\x13"
               "\x4a\x9c\x10\xd4\x29\x61\x58\x69\x21\x7f\xdd\x58\x85\xc4\xdb"
               "\x17\x49\x85\x70\x3a\x6d\x6d\xe9\x4a\x66\x7e\xac\x30\x23\x44"
               "\x3a\x83\x37\xae\x1b\xc6\x01\xb7\x6d\x7d\x38\xec\x3c\x34\x46"
               "\x31\x05\xf0\xd3\x94\x9d\x78\xe5\x62\xa0\x39\xe4\x46\x95\x48"
               "\xb6\x09\x39\x5d\xe5\xa4\xfd\x43\xc4\x6c\xa9\xfd\x6e\xe2\x9a"
               "\xda\x5e\xfc\x07\xd8\x4d\x55\x32\x49\x45\x0d\xab\x4a\x49\xc4"
               "\x83\xde\xd2\x50\xc9\x33\x8f\x85\xcd\x93\x7a\xe6\x6b\xb4\x36"
               "\xf3\xb4\x02\x6e\x85\x9f\xda\x1c\xa5\x71\x43\x2f\x3b\xfc\x09"
               "\xe7\xc0\x3c\xa4\xd1\x83\xb7\x41\x11\x1c\xa0\x48\x3d\x0e\xda"
               "\xbc\x03\xfe\xb2\x3b\x17\xee\x48\xe8\x44\xba\x24\x08\xd9\xdc"
               "\xfd\x01\x39\xd2\xe8\xc7\x31\x01\x25\xae\xe8\x01\xc6\x1a\xb7"
               "\x90\x0d\x1e\xfc\x47\xc0\x78\x28\x17\x66\xf3\x61\xc5\xe6\x11"
               "\x13\x46\x23\x5e\x1d\xc3\x83\x25\x66\x6c\x68\x1b\x30\xdd\xc4"
               "\xe6\x83\x8b\x0f\x23\x58\x7e\x06\x5f\x4a\x2b\xed\xc9\x6c\x97"
               "\x68\x44";
    e.inLen  = 32;
    e.outLen = 2 * WC_SHA3_256_BLOCK_SIZE;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;
    test_sha[3] = d;
    test_sha[4] = e;

    for (i = 0; i < times; ++i) {
        ret = wc_InitShake256(sha, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Shake256_Absorb(sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Shake256_SqueezeBlocks(sha, hash,
            (word32)test_sha[i].outLen / WC_SHA3_256_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, test_sha[i].output, (word32)test_sha[i].outLen) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    for (i = 0; i < (int)large_input_buf_size; i++) {
        large_input_buf[i] = (byte)(i & 0xFF);
    }
    ret = wc_InitShake256(sha, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    /* Absorb is non-incremental. */
    ret = wc_Shake256_Absorb(sha, large_input_buf,
        (word32)large_input_buf_size);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    /* Able to squeeze out blocks incrementally. */
    ret = wc_Shake256_SqueezeBlocks(sha, hash, 1);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    ret = wc_Shake256_SqueezeBlocks(sha, hash,
        ((word32)sizeof(hash) / WC_SHA3_256_BLOCK_SIZE) - 1);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, sizeof(hash)) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

exit:
    return ret;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t shake256_test(void)
{
    wc_Shake  sha;
    byte  hash[250];

    testVector a, b, c, d, e;
    testVector test_sha[5];
    wc_test_ret_t ret = 0;
    int times = sizeof(test_sha) / sizeof(struct testVector), i;

#define SHAKE256_LARGE_INPUT_BUFSIZ 1024
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte *large_input = NULL;
#else
    byte large_input[SHAKE256_LARGE_INPUT_BUFSIZ];
#endif
    const char* large_digest =
        "\x90\x32\x4a\xcc\xd1\xdf\xb8\x0b\x79\x1f\xb8\xc8\x5b\x54\xc8\xe7"
        "\x45\xf5\x60\x6b\x38\x26\xb2\x0a\xee\x38\x01\xf3\xd9\xfa\x96\x9f"
        "\x6a\xd7\x15\xdf\xb6\xc2\xf4\x20\x33\x44\x55\xe8\x2a\x09\x2b\x68"
        "\x2e\x18\x65\x5e\x65\x93\x28\xbc\xb1\x9e\xe2\xb1\x92\xea\x98\xac"
        "\x21\xef\x4c\xe1\xb4\xb7\xbe\x81\x5c\x1d\xd3\xb7\x17\xe5\xbb\xc5"
        "\x8c\x68\xb7\xfb\xac\x55\x8a\x9b\x4d\x91\xe4\x9f\x72\xbb\x6e\x38"
        "\xaf\x21\x7d\x21\xaa\x98\x4e\x75\xc4\xb4\x1c\x7c\x50\x45\x54\xf9"
        "\xea\x26";
    /*
    ** https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Standards-and-Guidelines/documents/examples/SHAKE256_Msg0.pdf
    */
    a.input  = "";
    a.output = "\x46\xb9\xdd\x2b\x0b\xa8\x8d\x13\x23\x3b\x3f\xeb\x74\x3e\xeb"
               "\x24\x3f\xcd\x52\xea\x62\xb8\x1b\x82\xb5\x0c\x27\x64\x6e\xd5"
               "\x76\x2f\xd7\x5d\xc4\xdd\xd8\xc0\xf2\x00\xcb\x05\x01\x9d\x67"
               "\xb5\x92\xf6\xfc\x82\x1c\x49\x47\x9a\xb4\x86\x40\x29\x2e\xac"
               "\xb3\xb7\xc4\xbe\x14\x1e\x96\x61\x6f\xb1\x39\x57\x69\x2c\xc7"
               "\xed\xd0\xb4\x5a\xe3\xdc\x07\x22\x3c\x8e\x92\x93\x7b\xef\x84"
               "\xbc\x0e\xab\x86\x28\x53\x34\x9e\xc7\x55\x46\xf5\x8f\xb7\xc2"
               "\x77\x5c\x38\x46\x2c\x50\x10\xd8\x46";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = 114;

    b.input  = "abc";
    b.output = "\x48\x33\x66\x60\x13\x60\xa8\x77\x1c\x68\x63\x08\x0c\xc4\x11"
               "\x4d\x8d\xb4\x45\x30\xf8\xf1\xe1\xee\x4f\x94\xea\x37\xe7\x8b"
               "\x57\x39\xd5\xa1\x5b\xef\x18\x6a\x53\x86\xc7\x57\x44\xc0\x52"
               "\x7e\x1f\xaa\x9f\x87\x26\xe4\x62\xa1\x2a\x4f\xeb\x06\xbd\x88"
               "\x01\xe7\x51\xe4\x13\x85\x14\x12\x04\xf3\x29\x97\x9f\xd3\x04"
               "\x7a\x13\xc5\x65\x77\x24\xad\xa6\x4d\x24\x70\x15\x7b\x3c\xdc"
               "\x28\x86\x20\x94\x4d\x78\xdb\xcd\xdb\xd9\x12\x99\x3f\x09\x13"
               "\xf1\x64\xfb\x2c\xe9\x51\x31\xa2\xd0";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = 114;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x4d\x8c\x2d\xd2\x43\x5a\x01\x28\xee\xfb\xb8\xc3\x6f\x6f\x87"
               "\x13\x3a\x79\x11\xe1\x8d\x97\x9e\xe1\xae\x6b\xe5\xd4\xfd\x2e"
               "\x33\x29\x40\xd8\x68\x8a\x4e\x6a\x59\xaa\x80\x60\xf1\xf9\xbc"
               "\x99\x6c\x05\xac\xa3\xc6\x96\xa8\xb6\x62\x79\xdc\x67\x2c\x74"
               "\x0b\xb2\x24\xec\x37\xa9\x2b\x65\xdb\x05\x39\xc0\x20\x34\x55"
               "\xf5\x1d\x97\xcc\xe4\xcf\xc4\x91\x27\xd7\x26\x0a\xfc\x67\x3a"
               "\xf2\x08\xba\xf1\x9b\xe2\x12\x33\xf3\xde\xbe\x78\xd0\x67\x60"
               "\xcf\xa5\x51\xee\x1e\x07\x91\x41\xd4";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = 114;

    /* Taken from NIST CAVP test vectors - full rate output. */
    d.input  = "\xdc\x88\x6d\xf3\xf6\x9c\x49\x51\x3d\xe3\x62\x7e\x94\x81\xdb"
               "\x58\x71\xe8\xee\x88\xeb\x9f\x99\x61\x15\x41\x93\x0a\x8b\xc8"
               "\x85\xe0";
    d.output = "\x00\x64\x8a\xfb\xc5\xe6\x51\x64\x9d\xb1\xfd\x82\x93\x6b\x00"
               "\xdb\xbc\x12\x2f\xb4\xc8\x77\x86\x0d\x38\x5c\x49\x50\xd5\x6d"
               "\xe7\xe0\x96\xd6\x13\xd7\xa3\xf2\x7e\xd8\xf2\x63\x34\xb0\xcc"
               "\xc1\x40\x7b\x41\xdc\xcb\x23\xdf\xaa\x52\x98\x18\xd1\x12\x5c"
               "\xd5\x34\x80\x92\x52\x43\x66\xb8\x5f\xab\xb9\x7c\x6c\xd1\xe6"
               "\x06\x6f\x45\x9b\xcc\x56\x6d\xa8\x7e\xc9\xb7\xba\x36\x79\x2d"
               "\x11\x8a\xc3\x9a\x4c\xce\xf6\x19\x2b\xbf\x3a\x54\xaf\x18\xe5"
               "\x7b\x0c\x14\x61\x01\xf6\xae\xaa\x82\x2b\xc4\xb4\xc9\x70\x8b"
               "\x09\xf0\xb3\xba\xb4\x1b\xcc\xe9\x64\xd9\x99\xd1\x10\x7b\xd7"
               "\xc2";
    d.inLen  = 32;
    d.outLen = 136;
    /* Taken from NIST CAVP test vectors - more than one output block. */
    e.input  = "\x8d\x80\x01\xe2\xc0\x96\xf1\xb8\x8e\x7c\x92\x24\xa0\x86\xef"
               "\xd4\x79\x7f\xbf\x74\xa8\x03\x3a\x2d\x42\x2a\x2b\x6b\x8f\x67"
               "\x47\xe4";
    e.output = "\x2e\x97\x5f\x6a\x8a\x14\xf0\x70\x4d\x51\xb1\x36\x67\xd8\x19"
               "\x5c\x21\x9f\x71\xe6\x34\x56\x96\xc4\x9f\xa4\xb9\xd0\x8e\x92"
               "\x25\xd3\xd3\x93\x93\x42\x51\x52\xc9\x7e\x71\xdd\x24\x60\x1c"
               "\x11\xab\xcf\xa0\xf1\x2f\x53\xc6\x80\xbd\x3a\xe7\x57\xb8\x13"
               "\x4a\x9c\x10\xd4\x29\x61\x58\x69\x21\x7f\xdd\x58\x85\xc4\xdb"
               "\x17\x49\x85\x70\x3a\x6d\x6d\xe9\x4a\x66\x7e\xac\x30\x23\x44"
               "\x3a\x83\x37\xae\x1b\xc6\x01\xb7\x6d\x7d\x38\xec\x3c\x34\x46"
               "\x31\x05\xf0\xd3\x94\x9d\x78\xe5\x62\xa0\x39\xe4\x46\x95\x48"
               "\xb6\x09\x39\x5d\xe5\xa4\xfd\x43\xc4\x6c\xa9\xfd\x6e\xe2\x9a"
               "\xda\x5e\xfc\x07\xd8\x4d\x55\x32\x49\x45\x0d\xab\x4a\x49\xc4"
               "\x83\xde\xd2\x50\xc9\x33\x8f\x85\xcd\x93\x7a\xe6\x6b\xb4\x36"
               "\xf3\xb4\x02\x6e\x85\x9f\xda\x1c\xa5\x71\x43\x2f\x3b\xfc\x09"
               "\xe7\xc0\x3c\xa4\xd1\x83\xb7\x41\x11\x1c\xa0\x48\x3d\x0e\xda"
               "\xbc\x03\xfe\xb2\x3b\x17\xee\x48\xe8\x44\xba\x24\x08\xd9\xdc"
               "\xfd\x01\x39\xd2\xe8\xc7\x31\x01\x25\xae\xe8\x01\xc6\x1a\xb7"
               "\x90\x0d\x1e\xfc\x47\xc0\x78\x28\x17\x66\xf3\x61\xc5\xe6\x11"
               "\x13\x46\x23\x5e\x1d\xc3\x83\x25\x66\x6c";
    e.inLen  = 32;
    e.outLen = 250;

    test_sha[0] = a;
    test_sha[1] = b;
    test_sha[2] = c;
    test_sha[3] = d;
    test_sha[4] = e;

    ret = wc_InitShake256(&sha, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    for (i = 0; i < times; ++i) {
        ret = wc_Shake256_Update(&sha, (byte*)test_sha[i].input,
            (word32)test_sha[i].inLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        ret = wc_Shake256_Final(&sha, hash, (word32)test_sha[i].outLen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        if (XMEMCMP(hash, test_sha[i].output, test_sha[i].outLen) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    large_input = (byte *)XMALLOC(SHAKE256_LARGE_INPUT_BUFSIZ, NULL,
                                  DYNAMIC_TYPE_TMP_BUFFER);
    if (large_input == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit);
#endif

#ifndef NO_LARGE_HASH_TEST
    /* BEGIN LARGE HASH TEST */ {
    for (i = 0; i < SHAKE256_LARGE_INPUT_BUFSIZ; i++) {
        large_input[i] = (byte)(i & 0xFF);
    }
    times = 100;
    for (i = 0; i < times; ++i) {
        ret = wc_Shake256_Update(&sha, (byte*)large_input,
            SHAKE256_LARGE_INPUT_BUFSIZ);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    }
    ret = wc_Shake256_Final(&sha, hash, (word32)sizeof(hash));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
    if (XMEMCMP(hash, large_digest, 114) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit);
    } /* END LARGE HASH TEST */
#endif /* NO_LARGE_HASH_TEST */

    ret = shake256_absorb_test(&sha, large_input, SHAKE256_LARGE_INPUT_BUFSIZ);
exit:
    wc_Shake256_Free(&sha);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (large_input != NULL)
        XFREE(large_input, NULL, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif

#ifdef WOLFSSL_SM3
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sm3_test(void)
{
    wc_Sm3 sm3, sm3Copy;
    byte   hash[WC_SM3_DIGEST_SIZE];
    byte   hashGet[WC_SM3_DIGEST_SIZE];
    byte   hashCopy[WC_SM3_DIGEST_SIZE];
    wc_test_ret_t ret = 0;

    testVector a, b, c;
    testVector test_sm3[3];
    int times = sizeof(test_sm3) / sizeof(struct testVector), i;

    a.input  = "";
    a.output = "\x1a\xb2\x1d\x83\x55\xcf\xa1\x7f\x8e\x61\x19\x48\x31\xe8\x1a"
               "\x8f\x22\xbe\xc8\xc7\x28\xfe\xfb\x74\x7e\xd0\x35\xeb\x50\x82"
               "\xaa\x2b";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SM3_DIGEST_SIZE;

    b.input  = "abc";
    b.output = "\x66\xc7\xf0\xf4\x62\xee\xed\xd9\xd1\xf2\xd4\x6b\xdc\x10\xe4"
               "\xe2\x41\x67\xc4\x87\x5c\xf2\xf7\xa2\x29\x7d\xa0\x2b\x8f\x4b"
               "\xa8\xe0";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SM3_DIGEST_SIZE;

    c.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    c.output = "\x63\x9b\x6c\xc5\xe6\x4d\x9e\x37\xa3\x90\xb1\x92\xdf\x4f\xa1"
               "\xea\x07\x20\xab\x74\x7f\xf6\x92\xb9\xf3\x8c\x4e\x66\xad\x7b"
               "\x8c\x05";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SM3_DIGEST_SIZE;

    test_sm3[0] = a;
    test_sm3[1] = b;
    test_sm3[2] = c;

    ret = wc_InitSm3(&sm3, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_InitSm3(&sm3Copy, HEAP_HINT, devId);
    if (ret != 0) {
        wc_Sm3Free(&sm3);
        return WC_TEST_RET_ENC_EC(ret);
    }

    /* Test all the KATs. */
    for (i = 0; i < times; ++i) {
        ret = wc_Sm3Update(&sm3, (byte*)test_sm3[i].input,
            (word32)test_sm3[i].inLen);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        }
        /* Get the final hash but leave ready for more updates. */
        ret = wc_Sm3GetHash(&sm3, hashGet);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        /* Make a copy of the hash. */
        ret = wc_Sm3Copy(&sm3, &sm3Copy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        /* Get the final hash with original. */
        ret = wc_Sm3Final(&sm3, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);

        /* Get the final hash with copy. */
        ret = wc_Sm3Final(&sm3Copy, hashCopy);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        /* Dispose of copy. */
        wc_Sm3Free(&sm3Copy);

        /* Check hashes match expected. */
        if (XMEMCMP(hash, test_sm3[i].output, WC_SM3_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashGet, WC_SM3_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        if (XMEMCMP(hash, hashCopy, WC_SM3_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
    }

#ifndef NO_LARGE_HASH_TEST
    {
        word32 sz;
        byte large_input[1024];
    #ifdef HASH_SIZE_LIMIT
        const char* large_digest =
            "\x6c\x42\x57\x64\x8e\x45\xf3\xb6\xc0\x83\xd3\x41\x83\x66\x51\xb4"
            "\x50\xfe\x06\xb5\xb7\x1e\xd5\x0d\x41\xfc\x1e\xe5\xc6\x57\x95\x0f";

        times = 20;
    #else
        const char* large_digest =
            "\x34\x51\x3c\xde\x7c\x30\xb7\xc5\xaa\x97\x3b\xed\xb3\x16\xb9\x76"
            "\x35\x46\x14\x80\x2a\x57\xca\xd9\x48\xf9\x93\xcc\x1f\xdd\xab\x79";

        times = 100;
    #endif

        /* Set large input to something. */
        for (i = 0; i < (int)sizeof(large_input); i++) {
            large_input[i] = (byte)(i & 0xFF);
        }

        /* Hash a large number of times. */
        for (i = 0; i < times; ++i) {
            ret = wc_Sm3Update(&sm3, (byte*)large_input,
                (word32)sizeof(large_input));
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
        }
        /* Calculate hash and compare to expected. */
        ret = wc_Sm3Final(&sm3, hash);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);
        if (XMEMCMP(hash, large_digest, WC_SM3_DIGEST_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit);


        /* Check updating with various sizes works. */
        for (sz = 1; sz <= 64; sz++) {
            /* Hash a large number of times. */
            for (i = 0; i < times; ++i) {
                word32 o;

                /* Update sz bytes at a time from large input buffer. */
                for (o = 0; o + sz <= (word32)sizeof(large_input); o += sz) {
                    ret = wc_Sm3Update(&sm3, (byte*)(large_input + o), sz);
                    if (ret != 0)
                        ERROR_OUT(WC_TEST_RET_ENC_I(o), exit);
                }
                /* Check for left-overs. */
                if (o < (word32)sizeof(large_input)) {
                    ret = wc_Sm3Update(&sm3, (byte*)(large_input + o),
                        (word32)sizeof(large_input) - o);
                    if (ret != 0)
                        ERROR_OUT(WC_TEST_RET_ENC_I(i), exit);
                }
            }

            /* Calculate hash and compare to expected. */
            ret = wc_Sm3Final(&sm3, hash);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_I(sz), exit);
            if (XMEMCMP(hash, large_digest, WC_SM3_DIGEST_SIZE) != 0)
                ERROR_OUT(WC_TEST_RET_ENC_I(sz), exit);
        }
    }
#endif /* NO_LARGE_HASH_TEST */

exit:

    wc_Sm3Free(&sm3);
    wc_Sm3Free(&sm3Copy);

    return ret;
}
#endif

#ifndef NO_HASH_WRAPPER
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hash_test(void)
{
    wc_HashAlg       hash;
    int              ret, exp_ret;
    int              i, j;
    int              digestSz;
    byte             data[] = "0123456789abcdef0123456789abcdef0123456";
    byte             out[WC_MAX_DIGEST_SIZE];
    byte             hashOut[WC_MAX_DIGEST_SIZE];
#if !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC)
    enum wc_HashType hashType;
#endif
    enum wc_HashType typesGood[] = { WC_HASH_TYPE_MD5, WC_HASH_TYPE_SHA,
                                     WC_HASH_TYPE_SHA224, WC_HASH_TYPE_SHA256,
                                     WC_HASH_TYPE_SHA384, WC_HASH_TYPE_SHA512,
                                     WC_HASH_TYPE_SHA3_224,
                                     WC_HASH_TYPE_SHA3_256,
                                     WC_HASH_TYPE_SHA3_384,
                                     WC_HASH_TYPE_SHA3_512 };
    enum wc_HashType typesNoImpl[] = {
#ifdef NO_MD5
                                        WC_HASH_TYPE_MD5,
#endif
#ifdef NO_SHA
                                        WC_HASH_TYPE_SHA,
#endif
#ifndef WOLFSSL_SHA224
                                        WC_HASH_TYPE_SHA224,
#endif
#ifdef NO_SHA256
                                        WC_HASH_TYPE_SHA256,
#endif
#ifndef WOLFSSL_SHA384
                                        WC_HASH_TYPE_SHA384,
#endif
#ifndef WOLFSSL_SHA512
                                        WC_HASH_TYPE_SHA512,
#endif
#if !defined(WOLFSSL_SHA3) || defined(WOLFSSL_NOSHA3_224)
                                        WC_HASH_TYPE_SHA3_224,
#endif
#if !defined(WOLFSSL_SHA3) || defined(WOLFSSL_NOSHA3_256)
                                        WC_HASH_TYPE_SHA3_256,
#endif
#if !defined(WOLFSSL_SHA3) || defined(WOLFSSL_NOSHA3_384)
                                        WC_HASH_TYPE_SHA3_384,
#endif
#if !defined(WOLFSSL_SHA3) || defined(WOLFSSL_NOSHA3_512)
                                        WC_HASH_TYPE_SHA3_512,
#endif
                                        WC_HASH_TYPE_NONE
                                     };
    enum wc_HashType typesBad[]  = { WC_HASH_TYPE_NONE, WC_HASH_TYPE_MD5_SHA,
                                     WC_HASH_TYPE_MD2, WC_HASH_TYPE_MD4 };
    enum wc_HashType typesHashBad[] = { WC_HASH_TYPE_MD2, WC_HASH_TYPE_MD4,
                                        WC_HASH_TYPE_BLAKE2B,
                                        WC_HASH_TYPE_NONE };

    /* Parameter Validation testing. */
    ret = wc_HashInit(NULL, WC_HASH_TYPE_SHA256);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_HashUpdate(NULL, WC_HASH_TYPE_SHA256, NULL, sizeof(data));
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_HashUpdate(&hash, WC_HASH_TYPE_SHA256, NULL, sizeof(data));
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_HashUpdate(NULL, WC_HASH_TYPE_SHA256, data, sizeof(data));
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_HashFinal(NULL, WC_HASH_TYPE_SHA256, NULL);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_HashFinal(&hash, WC_HASH_TYPE_SHA256, NULL);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_HashFinal(NULL, WC_HASH_TYPE_SHA256, out);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    /* Try invalid hash algorithms. */
    for (i = 0; i < (int)(sizeof(typesBad)/sizeof(*typesBad)); i++) {
        ret = wc_HashInit(&hash, typesBad[i]);
        if (ret != BAD_FUNC_ARG)
            return WC_TEST_RET_ENC_I(i);
        ret = wc_HashUpdate(&hash, typesBad[i], data, sizeof(data));
        if (ret != BAD_FUNC_ARG)
            return WC_TEST_RET_ENC_I(i);
        ret = wc_HashFinal(&hash, typesBad[i], out);
        if (ret != BAD_FUNC_ARG)
            return WC_TEST_RET_ENC_I(i);
        wc_HashFree(&hash, typesBad[i]);
    }

    /* Try valid hash algorithms. */
    for (i = 0, j = 0; i < (int)(sizeof(typesGood)/sizeof(*typesGood)); i++) {
        exp_ret = 0;
        if (typesGood[i] == typesNoImpl[j]) {
            /* Recognized but no implementation compiled in. */
            exp_ret = HASH_TYPE_E;
            j++;
        }
        ret = wc_HashInit(&hash, typesGood[i]);
        if (ret != exp_ret)
            return WC_TEST_RET_ENC_I(i);
        ret = wc_HashUpdate(&hash, typesGood[i], data, sizeof(data));
        if (ret != exp_ret)
            return WC_TEST_RET_ENC_I(i);
        ret = wc_HashFinal(&hash, typesGood[i], out);
        if (ret != exp_ret)
            return WC_TEST_RET_ENC_I(i);
        wc_HashFree(&hash, typesGood[i]);

        digestSz = wc_HashGetDigestSize(typesGood[i]);
        if (exp_ret < 0 && digestSz != exp_ret)
            return WC_TEST_RET_ENC_I(i);
        if (exp_ret == 0 && digestSz < 0)
            return WC_TEST_RET_ENC_I(i);
        if (exp_ret == 0) {
            ret = wc_Hash(typesGood[i], data, sizeof(data), hashOut,
                                                                  digestSz - 1);
            if (ret != BUFFER_E)
                return WC_TEST_RET_ENC_I(i);
        }
        ret = wc_Hash(typesGood[i], data, sizeof(data), hashOut, digestSz);
        if (ret != exp_ret)
            return WC_TEST_RET_ENC_I(i);
        if (exp_ret == 0 && XMEMCMP(out, hashOut, digestSz) != 0)
            return WC_TEST_RET_ENC_I(i);

        ret = wc_HashGetBlockSize(typesGood[i]);
        if (exp_ret < 0 && ret != exp_ret)
            return WC_TEST_RET_ENC_I(i);
        if (exp_ret == 0 && ret < 0)
            return WC_TEST_RET_ENC_I(i);

#if !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC)
        ret = wc_HashGetOID(typesGood[i]);
        if (ret == BAD_FUNC_ARG ||
                (exp_ret == 0 && ret == HASH_TYPE_E) ||
                (exp_ret != 0 && ret != HASH_TYPE_E)) {
            return WC_TEST_RET_ENC_I(i);
        }

        hashType = wc_OidGetHash(ret);
        if (exp_ret == 0 && hashType != typesGood[i])
            return WC_TEST_RET_ENC_I(i);
#endif /* !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC) */
    }

    for (i = 0; i < (int)(sizeof(typesHashBad)/sizeof(*typesHashBad)); i++) {
        ret = wc_Hash(typesHashBad[i], data, sizeof(data), out, sizeof(out));
        if ((ret != BAD_FUNC_ARG) && (ret != BUFFER_E) && (ret != HASH_TYPE_E))
            return WC_TEST_RET_ENC_I(i);
    }

#if !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC)
    ret = wc_HashGetOID(WC_HASH_TYPE_MD2);
#ifdef WOLFSSL_MD2
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif
    hashType = wc_OidGetHash(646); /* Md2h */
#ifdef WOLFSSL_MD2
    if (hashType != WC_HASH_TYPE_MD2)
        return WC_TEST_RET_ENC_NC;
#else
    if (hashType != WC_HASH_TYPE_NONE)
        return WC_TEST_RET_ENC_NC;
#endif

    ret = wc_HashGetOID(WC_HASH_TYPE_MD5_SHA);
#ifndef NO_MD5
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif
    ret = wc_HashGetOID(WC_HASH_TYPE_MD4);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_HashGetOID(WC_HASH_TYPE_NONE);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    hashType = wc_OidGetHash(0);
    if (hashType != WC_HASH_TYPE_NONE)
        return WC_TEST_RET_ENC_NC;
#endif /* !defined(NO_ASN) || !defined(NO_DH) || defined(HAVE_ECC) */

    ret = wc_HashGetBlockSize(WC_HASH_TYPE_MD2);
#ifdef WOLFSSL_MD2
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif
    ret = wc_HashGetDigestSize(WC_HASH_TYPE_MD2);
#ifdef WOLFSSL_MD2
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    ret = wc_HashGetBlockSize(WC_HASH_TYPE_MD4);
#ifndef NO_MD4
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif
    ret = wc_HashGetDigestSize(WC_HASH_TYPE_MD4);
#ifndef NO_MD4
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif
    ret = wc_HashGetBlockSize(WC_HASH_TYPE_MD5_SHA);
#if !defined(NO_MD5) && !defined(NO_SHA)
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    ret = wc_HashGetBlockSize(WC_HASH_TYPE_BLAKE2B);
#if defined(HAVE_BLAKE2) || defined(HAVE_BLAKE2S)
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif
    ret = wc_HashGetDigestSize(WC_HASH_TYPE_BLAKE2B);
#if defined(HAVE_BLAKE2) || defined(HAVE_BLAKE2S)
    if (ret == HASH_TYPE_E || ret == BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#else
    if (ret != HASH_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    ret = wc_HashGetBlockSize(WC_HASH_TYPE_NONE);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_HashGetDigestSize(WC_HASH_TYPE_NONE);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

#if !defined(NO_CERTS) && !defined(NO_ASN)
#if defined(WOLFSSL_MD2) && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    ret = wc_GetCTC_HashOID(MD2);
    if (ret == 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif
#ifndef NO_MD5
    ret = wc_GetCTC_HashOID(WC_MD5);
    if (ret == 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif
#ifndef NO_SHA
    ret = wc_GetCTC_HashOID(WC_SHA);
    if (ret == 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif
#ifdef WOLFSSL_SHA224
    ret = wc_GetCTC_HashOID(WC_SHA224);
    if (ret == 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif
#ifndef NO_SHA256
    ret = wc_GetCTC_HashOID(WC_SHA256);
    if (ret == 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif
#ifdef WOLFSSL_SHA384
    ret = wc_GetCTC_HashOID(WC_SHA384);
    if (ret == 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif
#ifdef WOLFSSL_SHA512
    ret = wc_GetCTC_HashOID(WC_SHA512);
    if (ret == 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif
    ret = wc_GetCTC_HashOID(-1);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    return 0;
}
#endif /* !NO_HASH_WRAPPER */

#if !defined(NO_HMAC) && !defined(NO_MD5) && !(defined(HAVE_FIPS) && \
                                               defined(HAVE_FIPS_VERSION) && \
                                               (HAVE_FIPS_VERSION >= 5))
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hmac_md5_test(void)
{
    Hmac hmac;
    byte hash[WC_MD5_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
    };

    testVector a, b, c;
    testVector test_hmac[3];

    wc_test_ret_t ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\x92\x94\x72\x7a\x36\x38\xbb\x1c\x13\xf4\x8e\xf8\x15\x8b\xfc"
               "\x9d";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_MD5_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\x75\x0c\x78\x3e\x6a\xb0\xb5\x03\xea\xa8\x6e\x31\x0a\x5d\xb7"
               "\x38";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_MD5_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x56\xbe\x34\x52\x1d\x14\x4c\x88\xdb\xb8\xc7\x33\xf0\xe8\xb3"
               "\xf6";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_MD5_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;

    for (i = 0; i < times; ++i) {
    #if defined(HAVE_FIPS) || defined(HAVE_CAVIUM)
        if (i == 1) {
            continue; /* cavium can't handle short keys, fips not allowed */
        }
    #endif

        ret = wc_HmacInit(&hmac, HEAP_HINT, devId);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        ret = wc_HmacSetKey(&hmac, WC_MD5, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        if (XMEMCMP(hash, test_hmac[i].output, WC_MD5_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if ((ret = wc_HmacSizeByType(WC_MD5)) != WC_MD5_DIGEST_SIZE)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    return 0;
}
#endif /* !NO_HMAC && !NO_MD5 && (!HAVE_FIPS || (HAVE_FIPS_VERSION < 5)) */

#if !defined(NO_HMAC) && !defined(NO_SHA)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hmac_sha_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA"
    };

    testVector a, b, c;
    testVector test_hmac[3];

    wc_test_ret_t ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\xb6\x17\x31\x86\x55\x05\x72\x64\xe2\x8b\xc0\xb6\xfb\x37\x8c"
               "\x8e\xf1\x46\xbe\x00";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\xef\xfc\xdf\x6a\xe5\xeb\x2f\xa2\xd2\x74\x16\xd5\xf1\x84\xdf"
               "\x9c\x25\x9a\x7c\x79";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x12\x5d\x73\x42\xb9\xac\x11\xcd\x91\xa3\x9a\xf4\x8a\xa1\x7b"
               "\x4f\x63\xf1\x75\xd3";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS) || defined(HAVE_CAVIUM)
        if (i == 1)
            continue; /* cavium can't handle short keys, fips not allowed */
#endif

        if ((ret = wc_HmacInit(&hmac, HEAP_HINT, devId)) != 0)
            return WC_TEST_RET_ENC_EC(ret);

        ret = wc_HmacSetKey(&hmac, WC_SHA, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if ((ret = wc_HmacSizeByType(WC_SHA)) != WC_SHA_DIGEST_SIZE)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    return 0;
}
#endif


#if !defined(NO_HMAC) && defined(WOLFSSL_SHA224)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hmac_sha224_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA224_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
    };

    testVector a, b, c, d;
    testVector test_hmac[4];

    wc_test_ret_t ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\x89\x6f\xb1\x12\x8a\xbb\xdf\x19\x68\x32\x10\x7c\xd4\x9d\xf3"
               "\x3f\x47\xb4\xb1\x16\x99\x12\xba\x4f\x53\x68\x4b\x22";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA224_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\xa3\x0e\x01\x09\x8b\xc6\xdb\xbf\x45\x69\x0f\x3a\x7e\x9e\x6d"
               "\x0f\x8b\xbe\xa2\xa3\x9e\x61\x48\x00\x8f\xd0\x5e\x44";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA224_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x7f\xb3\xcb\x35\x88\xc6\xc1\xf6\xff\xa9\x69\x4d\x7d\x6a\xd2"
               "\x64\x93\x65\xb0\xc1\xf6\x5d\x69\xd1\xec\x83\x33\xea";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA224_DIGEST_SIZE;

    d.input  = "Big Key Input";
    d.output = "\xe7\x4e\x2b\x8a\xa9\xf0\x37\x2f\xed\xae\x70\x0c\x49\x47\xf1"
               "\x46\x54\xa7\x32\x6b\x55\x01\x87\xd2\xc8\x02\x0e\x3a";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA224_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;
    test_hmac[3] = d;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS) || defined(HAVE_CAVIUM)
        if (i == 1)
            continue; /* cavium can't handle short keys, fips not allowed */
#endif

        if ((ret = wc_HmacInit(&hmac, HEAP_HINT, devId)) != 0)
            return WC_TEST_RET_ENC_EC(ret);

        ret = wc_HmacSetKey(&hmac, WC_SHA224, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA224_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if ((ret = wc_HmacSizeByType(WC_SHA224)) != WC_SHA224_DIGEST_SIZE)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    return 0;
}
#endif


#if !defined(NO_HMAC) && !defined(NO_SHA256)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hmac_sha256_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA256_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
    };

    testVector a, b, c, d;
    testVector test_hmac[4];

    wc_test_ret_t ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\xb0\x34\x4c\x61\xd8\xdb\x38\x53\x5c\xa8\xaf\xce\xaf\x0b\xf1"
               "\x2b\x88\x1d\xc2\x00\xc9\x83\x3d\xa7\x26\xe9\x37\x6c\x2e\x32"
               "\xcf\xf7";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA256_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\x5b\xdc\xc1\x46\xbf\x60\x75\x4e\x6a\x04\x24\x26\x08\x95\x75"
               "\xc7\x5a\x00\x3f\x08\x9d\x27\x39\x83\x9d\xec\x58\xb9\x64\xec"
               "\x38\x43";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA256_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x77\x3e\xa9\x1e\x36\x80\x0e\x46\x85\x4d\xb8\xeb\xd0\x91\x81"
               "\xa7\x29\x59\x09\x8b\x3e\xf8\xc1\x22\xd9\x63\x55\x14\xce\xd5"
               "\x65\xfe";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA256_DIGEST_SIZE;

    d.input  = 0;
    d.output = "\x86\xe5\x4f\xd4\x48\x72\x5d\x7e\x5d\xcf\xe2\x23\x53\xc8\x28"
               "\xaf\x48\x78\x1e\xb4\x8c\xae\x81\x06\xa7\xe1\xd4\x98\x94\x9f"
               "\x3e\x46";
    d.inLen  = 0;
    d.outLen = WC_SHA256_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;
    test_hmac[3] = d;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS) || defined(HAVE_CAVIUM)
        if (i == 1)
            continue; /* cavium can't handle short keys, fips not allowed */
#endif
#if defined(HAVE_INTEL_QA) || defined(HAVE_CAVIUM)
        if (i == 3)
            continue; /* QuickAssist can't handle empty HMAC */
#endif

        if (wc_HmacInit(&hmac, HEAP_HINT, devId) != 0)
            return WC_TEST_RET_ENC_I(i);

        ret = wc_HmacSetKey(&hmac, WC_SHA256, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);
        if (test_hmac[i].input != NULL) {
            ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                       (word32)test_hmac[i].inLen);
            if (ret != 0)
                return WC_TEST_RET_ENC_I(i);
        }
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA256_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if ((ret = wc_HmacSizeByType(WC_SHA256)) != WC_SHA256_DIGEST_SIZE)
        return WC_TEST_RET_ENC_EC(ret);
    if ((ret = wc_HmacSizeByType(21)) != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#endif
    if ((ret = wolfSSL_GetHmacMaxSize()) != WC_MAX_DIGEST_SIZE)
        return WC_TEST_RET_ENC_EC(ret);

    return 0;
}
#endif


#if !defined(NO_HMAC) && defined(WOLFSSL_SHA384)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hmac_sha384_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA384_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
    };

    testVector a, b, c, d;
    testVector test_hmac[4];

    wc_test_ret_t ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\xaf\xd0\x39\x44\xd8\x48\x95\x62\x6b\x08\x25\xf4\xab\x46\x90"
               "\x7f\x15\xf9\xda\xdb\xe4\x10\x1e\xc6\x82\xaa\x03\x4c\x7c\xeb"
               "\xc5\x9c\xfa\xea\x9e\xa9\x07\x6e\xde\x7f\x4a\xf1\x52\xe8\xb2"
               "\xfa\x9c\xb6";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA384_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\xaf\x45\xd2\xe3\x76\x48\x40\x31\x61\x7f\x78\xd2\xb5\x8a\x6b"
               "\x1b\x9c\x7e\xf4\x64\xf5\xa0\x1b\x47\xe4\x2e\xc3\x73\x63\x22"
               "\x44\x5e\x8e\x22\x40\xca\x5e\x69\xe2\xc7\x8b\x32\x39\xec\xfa"
               "\xb2\x16\x49";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA384_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\x88\x06\x26\x08\xd3\xe6\xad\x8a\x0a\xa2\xac\xe0\x14\xc8\xa8"
               "\x6f\x0a\xa6\x35\xd9\x47\xac\x9f\xeb\xe8\x3e\xf4\xe5\x59\x66"
               "\x14\x4b\x2a\x5a\xb3\x9d\xc1\x38\x14\xb9\x4e\x3a\xb6\xe1\x01"
               "\xa3\x4f\x27";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA384_DIGEST_SIZE;

    d.input  = "Big Key Input";
    d.output = "\xd2\x3d\x29\x6e\xf5\x1e\x23\x23\x49\x18\xb3\xbf\x4c\x38\x7b"
               "\x31\x21\x17\xbb\x09\x73\x27\xf8\x12\x9d\xe9\xc6\x5d\xf9\x54"
               "\xd6\x38\x5a\x68\x53\x14\xee\xe0\xa6\x4f\x36\x7e\xb2\xf3\x1a"
               "\x57\x41\x69";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA384_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;
    test_hmac[3] = d;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS)
        if (i == 1)
            continue; /* fips not allowed */
#endif

        if ((ret = wc_HmacInit(&hmac, HEAP_HINT, devId)) != 0)
            return WC_TEST_RET_ENC_EC(ret);

        ret = wc_HmacSetKey(&hmac, WC_SHA384, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA384_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if ((ret = wc_HmacSizeByType(WC_SHA384)) != WC_SHA384_DIGEST_SIZE)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    return 0;
}
#endif


#if !defined(NO_HMAC) && defined(WOLFSSL_SHA512)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hmac_sha512_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA512_DIGEST_SIZE];

    const char* keys[]=
    {
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
                                                                "\x0b\x0b\x0b",
        "Jefe",
        "\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA\xAA"
                                                                "\xAA\xAA\xAA",
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
    };

    testVector a, b, c, d;
    testVector test_hmac[4];

    wc_test_ret_t ret;
    int times = sizeof(test_hmac) / sizeof(testVector), i;

    a.input  = "Hi There";
    a.output = "\x87\xaa\x7c\xde\xa5\xef\x61\x9d\x4f\xf0\xb4\x24\x1a\x1d\x6c"
               "\xb0\x23\x79\xf4\xe2\xce\x4e\xc2\x78\x7a\xd0\xb3\x05\x45\xe1"
               "\x7c\xde\xda\xa8\x33\xb7\xd6\xb8\xa7\x02\x03\x8b\x27\x4e\xae"
               "\xa3\xf4\xe4\xbe\x9d\x91\x4e\xeb\x61\xf1\x70\x2e\x69\x6c\x20"
               "\x3a\x12\x68\x54";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_SHA512_DIGEST_SIZE;

    b.input  = "what do ya want for nothing?";
    b.output = "\x16\x4b\x7a\x7b\xfc\xf8\x19\xe2\xe3\x95\xfb\xe7\x3b\x56\xe0"
               "\xa3\x87\xbd\x64\x22\x2e\x83\x1f\xd6\x10\x27\x0c\xd7\xea\x25"
               "\x05\x54\x97\x58\xbf\x75\xc0\x5a\x99\x4a\x6d\x03\x4f\x65\xf8"
               "\xf0\xe6\xfd\xca\xea\xb1\xa3\x4d\x4a\x6b\x4b\x63\x6e\x07\x0a"
               "\x38\xbc\xe7\x37";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA512_DIGEST_SIZE;

    c.input  = "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD\xDD"
               "\xDD\xDD\xDD\xDD\xDD\xDD";
    c.output = "\xfa\x73\xb0\x08\x9d\x56\xa2\x84\xef\xb0\xf0\x75\x6c\x89\x0b"
               "\xe9\xb1\xb5\xdb\xdd\x8e\xe8\x1a\x36\x55\xf8\x3e\x33\xb2\x27"
               "\x9d\x39\xbf\x3e\x84\x82\x79\xa7\x22\xc8\x06\xb4\x85\xa4\x7e"
               "\x67\xc8\x07\xb9\x46\xa3\x37\xbe\xe8\x94\x26\x74\x27\x88\x59"
               "\xe1\x32\x92\xfb";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_SHA512_DIGEST_SIZE;

    d.input  = "Big Key Input";
    d.output = "\x3f\xa9\xc9\xe1\xbd\xbb\x04\x55\x1f\xef\xcc\x92\x33\x08\xeb"
               "\xcf\xc1\x9a\x5b\x5b\xc0\x7c\x86\x84\xae\x8c\x40\xaf\xb1\x27"
               "\x87\x38\x92\x04\xa8\xed\xd7\xd7\x07\xa9\x85\xa0\xc2\xcd\x30"
               "\xc0\x56\x14\x49\xbc\x2f\x69\x15\x6a\x97\xd8\x79\x2f\xb3\x3b"
               "\x1e\x18\xfe\xfa";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA512_DIGEST_SIZE;

    test_hmac[0] = a;
    test_hmac[1] = b;
    test_hmac[2] = c;
    test_hmac[3] = d;

    for (i = 0; i < times; ++i) {
#if defined(HAVE_FIPS)
        if (i == 1)
            continue; /* fips not allowed */
#endif

        if ((ret = wc_HmacInit(&hmac, HEAP_HINT, devId)) != 0)
            return WC_TEST_RET_ENC_EC(ret);

        ret = wc_HmacSetKey(&hmac, WC_SHA512, (byte*)keys[i],
            (word32)XSTRLEN(keys[i]));
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacUpdate(&hmac, (byte*)test_hmac[i].input,
                   (word32)test_hmac[i].inLen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_HmacFinal(&hmac, hash);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        if (XMEMCMP(hash, test_hmac[i].output, WC_SHA512_DIGEST_SIZE) != 0)
            return WC_TEST_RET_ENC_I(i);

        wc_HmacFree(&hmac);
    }

#ifndef HAVE_FIPS
    if ((ret = wc_HmacSizeByType(WC_SHA512)) != WC_SHA512_DIGEST_SIZE)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    return 0;
}
#endif


#if !defined(NO_HMAC) && defined(WOLFSSL_SHA3) && \
    !defined(WOLFSSL_NOSHA3_224) && !defined(WOLFSSL_NOSHA3_256) && \
    !defined(WOLFSSL_NOSHA3_384) && !defined(WOLFSSL_NOSHA3_512)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hmac_sha3_test(void)
{
    Hmac hmac;
    byte hash[WC_SHA3_512_DIGEST_SIZE];

    const char* key[4] =
    {
        "Jefe",

        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b"
        "\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b\x0b",

        "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa"
        "\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa\xaa",

        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
        "\x01\x02\x03\x04\x05\x06\x07\x08\x01\x02\x03\x04\x05\x06\x07\x08"
    };

    const char* input[4] =
    {
        "what do ya want for nothing?",

        "Hi There",

        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd"
        "\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd\xdd",

        "Big Key Input"
    };

    const int hashType[4] =
    {
        WC_SHA3_224, WC_SHA3_256, WC_SHA3_384, WC_SHA3_512
    };

    const int hashSz[4] =
    {
        WC_SHA3_224_DIGEST_SIZE, WC_SHA3_256_DIGEST_SIZE,
        WC_SHA3_384_DIGEST_SIZE, WC_SHA3_512_DIGEST_SIZE
    };

    const char* output[16] =
    {
        /* key = jefe, input = what do ya want for nothing? */
            /* HMAC-SHA3-224 */
            "\x7f\xdb\x8d\xd8\x8b\xd2\xf6\x0d\x1b\x79\x86\x34\xad\x38\x68\x11"
            "\xc2\xcf\xc8\x5b\xfa\xf5\xd5\x2b\xba\xce\x5e\x66",
            /* HMAC-SHA3-256 */
            "\xc7\xd4\x07\x2e\x78\x88\x77\xae\x35\x96\xbb\xb0\xda\x73\xb8\x87"
            "\xc9\x17\x1f\x93\x09\x5b\x29\x4a\xe8\x57\xfb\xe2\x64\x5e\x1b\xa5",
            /* HMAC-SHA3-384 */
            "\xf1\x10\x1f\x8c\xbf\x97\x66\xfd\x67\x64\xd2\xed\x61\x90\x3f\x21"
            "\xca\x9b\x18\xf5\x7c\xf3\xe1\xa2\x3c\xa1\x35\x08\xa9\x32\x43\xce"
            "\x48\xc0\x45\xdc\x00\x7f\x26\xa2\x1b\x3f\x5e\x0e\x9d\xf4\xc2\x0a",
            /* HMAC-SHA3-512 */
            "\x5a\x4b\xfe\xab\x61\x66\x42\x7c\x7a\x36\x47\xb7\x47\x29\x2b\x83"
            "\x84\x53\x7c\xdb\x89\xaf\xb3\xbf\x56\x65\xe4\xc5\xe7\x09\x35\x0b"
            "\x28\x7b\xae\xc9\x21\xfd\x7c\xa0\xee\x7a\x0c\x31\xd0\x22\xa9\x5e"
            "\x1f\xc9\x2b\xa9\xd7\x7d\xf8\x83\x96\x02\x75\xbe\xb4\xe6\x20\x24",

        /* key = 0b..., input = Hi There */
            /* HMAC-SHA3-224 */
            "\x3b\x16\x54\x6b\xbc\x7b\xe2\x70\x6a\x03\x1d\xca\xfd\x56\x37\x3d"
            "\x98\x84\x36\x76\x41\xd8\xc5\x9a\xf3\xc8\x60\xf7",
            /* HMAC-SHA3-256 */
            "\xba\x85\x19\x23\x10\xdf\xfa\x96\xe2\xa3\xa4\x0e\x69\x77\x43\x51"
            "\x14\x0b\xb7\x18\x5e\x12\x02\xcd\xcc\x91\x75\x89\xf9\x5e\x16\xbb",
            /* HMAC-SHA3-384 */
            "\x68\xd2\xdc\xf7\xfd\x4d\xdd\x0a\x22\x40\xc8\xa4\x37\x30\x5f\x61"
            "\xfb\x73\x34\xcf\xb5\xd0\x22\x6e\x1b\xc2\x7d\xc1\x0a\x2e\x72\x3a"
            "\x20\xd3\x70\xb4\x77\x43\x13\x0e\x26\xac\x7e\x3d\x53\x28\x86\xbd",
            /* HMAC-SHA3-512 */
            "\xeb\x3f\xbd\x4b\x2e\xaa\xb8\xf5\xc5\x04\xbd\x3a\x41\x46\x5a\xac"
            "\xec\x15\x77\x0a\x7c\xab\xac\x53\x1e\x48\x2f\x86\x0b\x5e\xc7\xba"
            "\x47\xcc\xb2\xc6\xf2\xaf\xce\x8f\x88\xd2\x2b\x6d\xc6\x13\x80\xf2"
            "\x3a\x66\x8f\xd3\x88\x8b\xb8\x05\x37\xc0\xa0\xb8\x64\x07\x68\x9e",

        /* key = aa..., output = dd... */
            /* HMAC-SHA3-224 */
            "\x67\x6c\xfc\x7d\x16\x15\x36\x38\x78\x03\x90\x69\x2b\xe1\x42\xd2"
            "\xdf\x7c\xe9\x24\xb9\x09\xc0\xc0\x8d\xbf\xdc\x1a",
            /* HMAC-SHA3-256 */
            "\x84\xec\x79\x12\x4a\x27\x10\x78\x65\xce\xdd\x8b\xd8\x2d\xa9\x96"
            "\x5e\x5e\xd8\xc3\x7b\x0a\xc9\x80\x05\xa7\xf3\x9e\xd5\x8a\x42\x07",
            /* HMAC-SHA3-384 */
            "\x27\x5c\xd0\xe6\x61\xbb\x8b\x15\x1c\x64\xd2\x88\xf1\xf7\x82\xfb"
            "\x91\xa8\xab\xd5\x68\x58\xd7\x2b\xab\xb2\xd4\x76\xf0\x45\x83\x73"
            "\xb4\x1b\x6a\xb5\xbf\x17\x4b\xec\x42\x2e\x53\xfc\x31\x35\xac\x6e",
            /* HMAC-SHA3-512 */
            "\x30\x9e\x99\xf9\xec\x07\x5e\xc6\xc6\xd4\x75\xed\xa1\x18\x06\x87"
            "\xfc\xf1\x53\x11\x95\x80\x2a\x99\xb5\x67\x74\x49\xa8\x62\x51\x82"
            "\x85\x1c\xb3\x32\xaf\xb6\xa8\x9c\x41\x13\x25\xfb\xcb\xcd\x42\xaf"
            "\xcb\x7b\x6e\x5a\xab\x7e\xa4\x2c\x66\x0f\x97\xfd\x85\x84\xbf\x03",

        /* key = big key, input = Big Key Input */
            /* HMAC-SHA3-224 */
            "\x29\xe0\x5e\x46\xc4\xa4\x5e\x46\x74\xbf\xd7\x2d\x1a\xd8\x66\xdb"
            "\x2d\x0d\x10\x4e\x2b\xfa\xad\x53\x7d\x15\x69\x8b",
            /* HMAC-SHA3-256 */
            "\xb5\x5b\x8d\x64\xb6\x9c\x21\xd0\xbf\x20\x5c\xa2\xf7\xb9\xb1\x4e"
            "\x88\x21\x61\x2c\x66\xc3\x91\xae\x6c\x95\x16\x85\x83\xe6\xf4\x9b",
            /* HMAC-SHA3-384 */
            "\xaa\x91\xb3\xa6\x2f\x56\xa1\xbe\x8c\x3e\x74\x38\xdb\x58\xd9\xd3"
            "\x34\xde\xa0\x60\x6d\x8d\x46\xe0\xec\xa9\xf6\x06\x35\x14\xe6\xed"
            "\x83\xe6\x7c\x77\x24\x6c\x11\xb5\x90\x82\xb5\x75\xda\x7b\x83\x2d",
            /* HMAC-SHA3-512 */
            "\x1c\xc3\xa9\x24\x4a\x4a\x3f\xbd\xc7\x20\x00\x16\x9b\x79\x47\x03"
            "\x78\x75\x2c\xb5\xf1\x2e\x62\x7c\xbe\xef\x4e\x8f\x0b\x11\x2b\x32"
            "\xa0\xee\xc9\xd0\x4d\x64\x64\x0b\x37\xf4\xdd\x66\xf7\x8b\xb3\xad"
            "\x52\x52\x6b\x65\x12\xde\x0d\x7c\xc0\x8b\x60\x01\x6c\x37\xd7\xa8"

    };

    int i = 0, iMax = sizeof(input) / sizeof(input[0]),
        j, jMax = sizeof(hashType) / sizeof(hashType[0]),
        ret;

#ifdef HAVE_FIPS
    /* FIPS requires a minimum length for HMAC keys, and "Jefe" is too
     * short. Skip it in FIPS builds. */
    i = 1;
#endif
    for (; i < iMax; i++) {
        for (j = 0; j < jMax; j++) {
            if ((ret = wc_HmacInit(&hmac, HEAP_HINT, devId)) != 0)
                return WC_TEST_RET_ENC_EC(ret);

            ret = wc_HmacSetKey(&hmac, hashType[j], (byte*)key[i],
                                                       (word32)XSTRLEN(key[i]));
            if (ret != 0)
                return WC_TEST_RET_ENC_EC(ret);
            ret = wc_HmacUpdate(&hmac, (byte*)input[i],
                                                     (word32)XSTRLEN(input[i]));
            if (ret != 0)
                return WC_TEST_RET_ENC_EC(ret);
            ret = wc_HmacFinal(&hmac, hash);
            if (ret != 0)
                return WC_TEST_RET_ENC_EC(ret);
            if (XMEMCMP(hash, output[(i*jMax) + j], hashSz[j]) != 0)
                return WC_TEST_RET_ENC_NC;

            wc_HmacFree(&hmac);

            if (i > 0)
                continue;

        #ifndef HAVE_FIPS
            ret = wc_HmacSizeByType(hashType[j]);
            if (ret != hashSz[j])
                return WC_TEST_RET_ENC_EC(ret);
        #endif
        }
    }

    return 0;
}
#endif


#ifdef WC_RC2
typedef struct rc2TestVector {
    const char* input;
    const char* output;
    const char* key;        /* Key, variable up to 128 bytes */
    const char* iv;         /* IV, 8-bytes */
    int inLen;
    int outLen;
    int keyLen;
    int effectiveKeyBits;   /* Up to 1024 bits supported */
} rc2TestVector;

static wc_test_ret_t rc2_ecb_test(void)
{
    wc_test_ret_t ret = 0;
    byte cipher[RC2_BLOCK_SIZE];
    byte plain[RC2_BLOCK_SIZE];

    rc2TestVector a, b, c, d, e, f, g, h;
    rc2TestVector test_rc2[8];

    int times = sizeof(test_rc2) / sizeof(rc2TestVector), i;

    a.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.output = "\xeb\xb7\x73\xf9\x93\x27\x8e\xff";
    a.key    = "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.inLen  = RC2_BLOCK_SIZE;
    a.outLen = RC2_BLOCK_SIZE;
    a.keyLen = 8;
    a.effectiveKeyBits = 63;

    b.input  = "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.output = "\x27\x8b\x27\xe4\x2e\x2f\x0d\x49";
    b.key    = "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.inLen  = RC2_BLOCK_SIZE;
    b.outLen = RC2_BLOCK_SIZE;
    b.keyLen = 8;
    b.effectiveKeyBits = 64;

    c.input  = "\x10\x00\x00\x00\x00\x00\x00\x01";
    c.output = "\x30\x64\x9e\xdf\x9b\xe7\xd2\xc2";
    c.key    = "\x30\x00\x00\x00\x00\x00\x00\x00";
    c.inLen  = RC2_BLOCK_SIZE;
    c.outLen = RC2_BLOCK_SIZE;
    c.keyLen = 8;
    c.effectiveKeyBits = 64;

    d.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    d.output = "\x61\xa8\xa2\x44\xad\xac\xcc\xf0";
    d.key    = "\x88";
    d.inLen  = RC2_BLOCK_SIZE;
    d.outLen = RC2_BLOCK_SIZE;
    d.keyLen = 1;
    d.effectiveKeyBits = 64;

    e.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    e.output = "\x6c\xcf\x43\x08\x97\x4c\x26\x7f";
    e.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a";
    e.inLen  = RC2_BLOCK_SIZE;
    e.outLen = RC2_BLOCK_SIZE;
    e.keyLen = 7;
    e.effectiveKeyBits = 64;

    f.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    f.output = "\x1a\x80\x7d\x27\x2b\xbe\x5d\xb1";
    f.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2";
    f.inLen  = RC2_BLOCK_SIZE;
    f.outLen = RC2_BLOCK_SIZE;
    f.keyLen = 16;
    f.effectiveKeyBits = 64;

    g.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    g.output = "\x22\x69\x55\x2a\xb0\xf8\x5c\xa6";
    g.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2";
    g.inLen  = RC2_BLOCK_SIZE;
    g.outLen = RC2_BLOCK_SIZE;
    g.keyLen = 16;
    g.effectiveKeyBits = 128;

    h.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    h.output = "\x5b\x78\xd3\xa4\x3d\xff\xf1\xf1";
    h.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2"
               "\x16\xf8\x0a\x6f\x85\x92\x05\x84"
               "\xc4\x2f\xce\xb0\xbe\x25\x5d\xaf"
               "\x1e";
    h.inLen  = RC2_BLOCK_SIZE;
    h.outLen = RC2_BLOCK_SIZE;
    h.keyLen = 33;
    h.effectiveKeyBits = 129;

    a.iv = b.iv = c.iv = d.iv = e.iv = f.iv = g.iv = h.iv = NULL;

    test_rc2[0] = a;
    test_rc2[1] = b;
    test_rc2[2] = c;
    test_rc2[3] = d;
    test_rc2[4] = e;
    test_rc2[5] = f;
    test_rc2[6] = g;
    test_rc2[7] = h;

    for (i = 0; i < times; ++i) {
        Rc2 enc;

        XMEMSET(cipher, 0, RC2_BLOCK_SIZE);
        XMEMSET(plain, 0, RC2_BLOCK_SIZE);

        ret = wc_Rc2SetKey(&enc, (byte*)test_rc2[i].key, test_rc2[i].keyLen,
                           NULL, test_rc2[i].effectiveKeyBits);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        /* ECB encrypt */
        ret = wc_Rc2EcbEncrypt(&enc, cipher, (byte*)test_rc2[i].input,
                               (word32)test_rc2[i].outLen);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        if (XMEMCMP(cipher, test_rc2[i].output, test_rc2[i].outLen)) {
            return WC_TEST_RET_ENC_NC;
        }

        /* ECB decrypt */
        ret = wc_Rc2EcbDecrypt(&enc, plain, cipher, RC2_BLOCK_SIZE);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        if (XMEMCMP(plain, test_rc2[i].input, RC2_BLOCK_SIZE)) {
            return WC_TEST_RET_ENC_NC;
        }
    }

    return 0;
}

static wc_test_ret_t rc2_cbc_test(void)
{
    wc_test_ret_t ret = 0;
    byte cipher[128];
    byte plain[128];

    rc2TestVector a, b, c, d, e, f, g, h, i;
    rc2TestVector test_rc2[9];

    int times = sizeof(test_rc2) / sizeof(rc2TestVector), j;

    /* key length = 7, effective key bits = 63 */
    a.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.output = "\xEB\xB7\x73\xF9\x93\x27\x8E\xFF"
               "\xF0\x51\x77\x8B\x65\xDB\x13\x57";
    a.key    = "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    a.inLen  = RC2_BLOCK_SIZE*2;
    a.outLen = RC2_BLOCK_SIZE*2;
    a.keyLen = 8;
    a.effectiveKeyBits = 63;

    /* key length = 8, effective key bits = 64, all 0xFF */
    b.input  = "\xff\xff\xff\xff\xff\xff\xff\xff"
               "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.output = "\xA3\xA1\x12\x65\x4F\x81\xC5\xCD"
               "\xB6\x94\x3E\xEA\x3E\x8B\x9D\x1F";
    b.key    = "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.iv     = "\xff\xff\xff\xff\xff\xff\xff\xff";
    b.inLen  = RC2_BLOCK_SIZE*2;
    b.outLen = RC2_BLOCK_SIZE*2;
    b.keyLen = 8;
    b.effectiveKeyBits = 64;

    /* key length = 8, effective key bits = 64 */
    c.input  = "\x10\x00\x00\x00\x00\x00\x00\x01"
               "\x10\x00\x00\x00\x00\x00\x00\x01";
    c.output = "\xB5\x70\x14\xA2\x5F\x40\xE3\x6D"
               "\x81\x99\x8D\xE0\xB5\xD5\x3A\x05";
    c.key    = "\x30\x00\x00\x00\x00\x00\x00\x00";
    c.iv     = "\x30\x00\x00\x00\x00\x00\x00\x00";
    c.inLen  = RC2_BLOCK_SIZE*2;
    c.outLen = RC2_BLOCK_SIZE*2;
    c.keyLen = 8;
    c.effectiveKeyBits = 64;

    /* key length = 1, effective key bits = 64 */
    d.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    d.output = "\x61\xA8\xA2\x44\xAD\xAC\xCC\xF0"
               "\x6D\x19\xE8\xF1\xFC\xE7\x38\x87";
    d.key    = "\x88";
    d.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    d.inLen  = RC2_BLOCK_SIZE*2;
    d.outLen = RC2_BLOCK_SIZE*2;
    d.keyLen = 1;
    d.effectiveKeyBits = 64;

    /* key length = 7, effective key bits = 64 */
    e.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    e.output = "\x6C\xCF\x43\x08\x97\x4C\x26\x7F"
               "\xCC\x3C\x53\x57\x7C\xA1\xA4\x4B";
    e.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a";
    e.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    e.inLen  = RC2_BLOCK_SIZE*2;
    e.outLen = RC2_BLOCK_SIZE*2;
    e.keyLen = 7;
    e.effectiveKeyBits = 64;

    /* key length = 16, effective key bits = 64 */
    f.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    f.output = "\x1A\x80\x7D\x27\x2B\xBE\x5D\xB1"
               "\x64\xEF\xE1\xC3\xB8\xAD\xFB\xBA";
    f.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2";
    f.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    f.inLen  = RC2_BLOCK_SIZE*2;
    f.outLen = RC2_BLOCK_SIZE*2;
    f.keyLen = 16;
    f.effectiveKeyBits = 64;

    /* key length = 16, effective bits = 128 */
    g.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    g.output = "\x22\x69\x55\x2A\xB0\xF8\x5C\xA6"
               "\x53\x6E\xFD\x2D\x89\xE1\x2A\x73";
    g.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2";
    g.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    g.inLen  = RC2_BLOCK_SIZE*2;
    g.outLen = RC2_BLOCK_SIZE*2;
    g.keyLen = 16;
    g.effectiveKeyBits = 128;

    /* key length = 33, effective bits = 129 */
    h.input  = "\x00\x00\x00\x00\x00\x00\x00\x00"
               "\x00\x00\x00\x00\x00\x00\x00\x00";
    h.output = "\x5B\x78\xD3\xA4\x3D\xFF\xF1\xF1"
               "\x45\x30\xA8\xD5\xC7\x7C\x46\x19";
    h.key    = "\x88\xbc\xa9\x0e\x90\x87\x5a\x7f"
               "\x0f\x79\xc3\x84\x62\x7b\xaf\xb2"
               "\x16\xf8\x0a\x6f\x85\x92\x05\x84"
               "\xc4\x2f\xce\xb0\xbe\x25\x5d\xaf"
               "\x1e";
    h.iv     = "\x00\x00\x00\x00\x00\x00\x00\x00";
    h.inLen  = RC2_BLOCK_SIZE*2;
    h.outLen = RC2_BLOCK_SIZE*2;
    h.keyLen = 33;
    h.effectiveKeyBits = 129;

    /* key length = 10, effective bits = 40 */
    i.input  = "\x11\x22\x33\x44\x55\x66\x77\x88"
               "\x99\xAA\xBB\xCC\xDD\xEE\xFF\x00"
               "\x11\x22\x33\x44\x55\x66\x77\x88"
               "\x99\xAA\xBB\xCC\xDD\xEE\xFF\x00";
    i.output = "\x71\x2D\x11\x99\xC9\xA0\x78\x4F"
               "\xCD\xF1\x1E\x3D\xFD\x21\x7E\xDB"
               "\xB2\x6E\x0D\xA4\x72\xBC\x31\x51"
               "\x48\xEF\x4E\x68\x3B\xDC\xCD\x7D";
    i.key    = "\x26\x1E\x57\x8E\xC9\x62\xBF\xB8"
               "\x3E\x96";
    i.iv     = "\x01\x02\x03\x04\x05\x06\x07\x08";
    i.inLen  = RC2_BLOCK_SIZE*4;
    i.outLen = RC2_BLOCK_SIZE*4;
    i.keyLen = 10;
    i.effectiveKeyBits = 40;

    test_rc2[0] = a;
    test_rc2[1] = b;
    test_rc2[2] = c;
    test_rc2[3] = d;
    test_rc2[4] = e;
    test_rc2[5] = f;
    test_rc2[6] = g;
    test_rc2[7] = h;
    test_rc2[8] = i;

    for (j = 0; j < times; ++j) {
        Rc2 rc2;

        XMEMSET(cipher, 0, sizeof(cipher));
        XMEMSET(plain, 0, sizeof(plain));

        ret = wc_Rc2SetKey(&rc2, (byte*)test_rc2[j].key, test_rc2[j].keyLen,
                           (byte*)test_rc2[j].iv, test_rc2[j].effectiveKeyBits);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        ret = wc_Rc2CbcEncrypt(&rc2, cipher, (byte*)test_rc2[j].input,
                               test_rc2[j].inLen);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        if (XMEMCMP(cipher, (byte*)test_rc2[j].output, test_rc2[j].outLen)) {
            return WC_TEST_RET_ENC_NC;
        }

        /* reset IV for decrypt, since overridden by encrypt operation */
        ret = wc_Rc2SetIV(&rc2, (byte*)test_rc2[j].iv);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        ret = wc_Rc2CbcDecrypt(&rc2, plain, cipher, test_rc2[j].outLen);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        if (XMEMCMP(plain, (byte*)test_rc2[j].input, test_rc2[j].inLen)) {
            return WC_TEST_RET_ENC_NC;
        }
    }

    return 0;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t rc2_test(void)
{
    wc_test_ret_t ret = 0;

    ret = rc2_ecb_test();
    if (ret != 0) {
        return ret;
    }

    return rc2_cbc_test();
}
#endif


#ifndef NO_RC4
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t arc4_test(void)
{
    byte cipher[16];
    byte plain[16];
    wc_test_ret_t ret;

    const char* keys[] =
    {
        "\x01\x23\x45\x67\x89\xab\xcd\xef",
        "\x01\x23\x45\x67\x89\xab\xcd\xef",
        "\x00\x00\x00\x00\x00\x00\x00\x00",
        "\xef\x01\x23\x45"
    };

    testVector a, b, c, d;
    testVector test_arc4[4];

    int times = sizeof(test_arc4) / sizeof(testVector), i;

    a.input  = "\x01\x23\x45\x67\x89\xab\xcd\xef";
    a.output = "\x75\xb7\x87\x80\x99\xe0\xc5\x96";
    a.inLen  = 8;
    a.outLen = 8;

    b.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    b.output = "\x74\x94\xc2\xe7\x10\x4b\x08\x79";
    b.inLen  = 8;
    b.outLen = 8;

    c.input  = "\x00\x00\x00\x00\x00\x00\x00\x00";
    c.output = "\xde\x18\x89\x41\xa3\x37\x5d\x3a";
    c.inLen  = 8;
    c.outLen = 8;

    d.input  = "\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00";
    d.output = "\xd6\xa1\x41\xa7\xec\x3c\x38\xdf\xbd\x61";
    d.inLen  = 10;
    d.outLen = 10;

    test_arc4[0] = a;
    test_arc4[1] = b;
    test_arc4[2] = c;
    test_arc4[3] = d;

    for (i = 0; i < times; ++i) {
        Arc4 enc;
        Arc4 dec;
        int  keylen = 8;  /* XSTRLEN with key 0x00 not good */
        if (i == 3)
            keylen = 4;

        ret = wc_Arc4Init(&enc, HEAP_HINT, devId);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_Arc4Init(&dec, HEAP_HINT, devId);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        ret = wc_Arc4SetKey(&enc, (byte*)keys[i], keylen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_Arc4SetKey(&dec, (byte*)keys[i], keylen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        ret = wc_Arc4Process(&enc, cipher, (byte*)test_arc4[i].input,
                    (word32)test_arc4[i].outLen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_Arc4Process(&dec, plain,  cipher, (word32)test_arc4[i].outLen);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        if (XMEMCMP(plain, test_arc4[i].input, test_arc4[i].outLen))
            return WC_TEST_RET_ENC_I(i);

        if (XMEMCMP(cipher, test_arc4[i].output, test_arc4[i].outLen))
            return WC_TEST_RET_ENC_I(i);

        wc_Arc4Free(&enc);
        wc_Arc4Free(&dec);
    }

    return 0;
}
#endif

#ifdef HAVE_CHACHA
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t chacha_test(void)
{
    ChaCha enc;
    ChaCha dec;
    byte   cipher[128];
    byte   plain[128];
    byte   sliver[64];
    byte   input[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
    word32 keySz = 32;
    wc_test_ret_t ret = 0;
    int    i;
    int    times = 4;

    WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };

    /* 128 bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte key4[] =
    {
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
         0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };


    const byte* keys[] = {key1, key2, key3, key4};

    WOLFSSL_SMALL_STACK_STATIC const byte ivs1[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
    WOLFSSL_SMALL_STACK_STATIC const byte ivs2[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};
    WOLFSSL_SMALL_STACK_STATIC const byte ivs3[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x00,0x00,0x00,0x00};
    WOLFSSL_SMALL_STACK_STATIC const byte ivs4[] = {0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00};


    const byte* ivs[] = {ivs1, ivs2, ivs3, ivs4};

#ifndef BENCH_EMBEDDED
    WOLFSSL_SMALL_STACK_STATIC const byte cipher_big_result[] = {
        0x06, 0xa6, 0x5d, 0x31, 0x21, 0x6c, 0xdb, 0x37, 0x48, 0x7c, 0x01, 0x9d,
        0x72, 0xdf, 0x0a, 0x5b, 0x64, 0x74, 0x20, 0xba, 0x9e, 0xe0, 0x26, 0x7a,
        0xbf, 0xdf, 0x83, 0x34, 0x3b, 0x4f, 0x94, 0x3f, 0x37, 0x89, 0xaf, 0x00,
        0xdf, 0x0f, 0x2e, 0x75, 0x16, 0x41, 0xf6, 0x7a, 0x86, 0x94, 0x9d, 0x32,
        0x56, 0xf0, 0x79, 0x71, 0x68, 0x6f, 0xa6, 0x6b, 0xc6, 0x59, 0x49, 0xf6,
        0x10, 0x34, 0x03, 0x03, 0x16, 0x53, 0x9a, 0x98, 0x2a, 0x46, 0xde, 0x17,
        0x06, 0x65, 0x70, 0xca, 0x0a, 0x1f, 0xab, 0x80, 0x26, 0x96, 0x3f, 0x3e,
        0x7a, 0x3c, 0xa8, 0x87, 0xbb, 0x65, 0xdd, 0x5e, 0x07, 0x7b, 0x34, 0xe0,
        0x56, 0xda, 0x32, 0x13, 0x30, 0xc9, 0x0c, 0xd7, 0xba, 0xe4, 0x1f, 0xa6,
        0x91, 0x4f, 0x72, 0x9f, 0xd9, 0x5c, 0x62, 0x7d, 0xa6, 0xc2, 0xbc, 0x87,
        0xae, 0x64, 0x11, 0x94, 0x3b, 0xbc, 0x6c, 0x23, 0xbd, 0x7d, 0x00, 0xb4,
        0x99, 0xf2, 0x68, 0xb5, 0x59, 0x70, 0x93, 0xad, 0x69, 0xd0, 0xb1, 0x28,
        0x70, 0x92, 0xeb, 0xec, 0x39, 0x80, 0x82, 0xde, 0x44, 0xe2, 0x8a, 0x26,
        0xb3, 0xe9, 0x45, 0xcf, 0x83, 0x76, 0x9f, 0x6a, 0xa0, 0x46, 0x4a, 0x3d,
        0x26, 0x56, 0xaf, 0x49, 0x41, 0x26, 0x1b, 0x6a, 0x41, 0x37, 0x65, 0x91,
        0x72, 0xc4, 0xe7, 0x3c, 0x17, 0x31, 0xae, 0x2e, 0x2b, 0x31, 0x45, 0xe4,
        0x93, 0xd3, 0x10, 0xaa, 0xc5, 0x62, 0xd5, 0x11, 0x4b, 0x57, 0x1d, 0xad,
        0x48, 0x06, 0xd0, 0x0d, 0x98, 0xa5, 0xc6, 0x5b, 0xd0, 0x9e, 0x22, 0xc0,
        0x00, 0x32, 0x5a, 0xf5, 0x1c, 0x89, 0x6d, 0x54, 0x97, 0x55, 0x6b, 0x46,
        0xc5, 0xc7, 0xc4, 0x48, 0x9c, 0xbf, 0x47, 0xdc, 0x03, 0xc4, 0x1b, 0xcb,
        0x65, 0xa6, 0x91, 0x9d, 0x6d, 0xf1, 0xb0, 0x7a, 0x4d, 0x3b, 0x03, 0x95,
        0xf4, 0x8b, 0x0b, 0xae, 0x39, 0xff, 0x3f, 0xf6, 0xc0, 0x14, 0x18, 0x8a,
        0xe5, 0x19, 0xbd, 0xc1, 0xb4, 0x05, 0x4e, 0x29, 0x2f, 0x0b, 0x33, 0x76,
        0x28, 0x16, 0xa4, 0xa6, 0x93, 0x04, 0xb5, 0x55, 0x6b, 0x89, 0x3d, 0xa5,
        0x0f, 0xd3, 0xad, 0xfa, 0xd9, 0xfd, 0x05, 0x5d, 0x48, 0x94, 0x25, 0x5a,
        0x2c, 0x9a, 0x94, 0x80, 0xb0, 0xe7, 0xcb, 0x4d, 0x77, 0xbf, 0xca, 0xd8,
        0x55, 0x48, 0xbd, 0x66, 0xb1, 0x85, 0x81, 0xb1, 0x37, 0x79, 0xab, 0x52,
        0x08, 0x14, 0x12, 0xac, 0xcd, 0x45, 0x4d, 0x53, 0x6b, 0xca, 0x96, 0xc7,
        0x3b, 0x2f, 0x73, 0xb1, 0x5a, 0x23, 0xbd, 0x65, 0xd5, 0xea, 0x17, 0xb3,
        0xdc, 0xa1, 0x17, 0x1b, 0x2d, 0xb3, 0x9c, 0xd0, 0xdb, 0x41, 0x77, 0xef,
        0x93, 0x20, 0x52, 0x3e, 0x9d, 0xf5, 0xbf, 0x33, 0xf7, 0x52, 0xc1, 0x90,
        0xa0, 0x15, 0x17, 0xce, 0xf7, 0xf7, 0xd0, 0x3a, 0x3b, 0xd1, 0x72, 0x56,
        0x31, 0x81, 0xae, 0x60, 0xab, 0x40, 0xc1, 0xd1, 0x28, 0x77, 0x53, 0xac,
        0x9f, 0x11, 0x0a, 0x88, 0x36, 0x4b, 0xda, 0x57, 0xa7, 0x28, 0x5c, 0x85,
        0xd3, 0x85, 0x9b, 0x79, 0xad, 0x05, 0x1c, 0x37, 0x14, 0x5e, 0x0d, 0xd0,
        0x23, 0x03, 0x42, 0x1d, 0x48, 0x5d, 0xc5, 0x3c, 0x5a, 0x08, 0xa9, 0x0d,
        0x6e, 0x82, 0x7c, 0x2e, 0x3c, 0x41, 0xcc, 0x96, 0x8e, 0xad, 0xee, 0x2a,
        0x61, 0x0b, 0x16, 0x0f, 0xa9, 0x24, 0x40, 0x85, 0xbc, 0x9f, 0x28, 0x8d,
        0xe6, 0x68, 0x4d, 0x8f, 0x30, 0x48, 0xd9, 0x73, 0x73, 0x6c, 0x9a, 0x7f,
        0x67, 0xf7, 0xde, 0x4c, 0x0a, 0x8b, 0xe4, 0xb3, 0x08, 0x2a, 0x52, 0xda,
        0x54, 0xee, 0xcd, 0xb5, 0x62, 0x4a, 0x26, 0x20, 0xfb, 0x40, 0xbb, 0x39,
        0x3a, 0x0f, 0x09, 0xe8, 0x00, 0xd1, 0x24, 0x97, 0x60, 0xe9, 0x83, 0x83,
        0xfe, 0x9f, 0x9c, 0x15, 0xcf, 0x69, 0x03, 0x9f, 0x03, 0xe1, 0xe8, 0x6e,
        0xbd, 0x87, 0x58, 0x68, 0xee, 0xec, 0xd8, 0x29, 0x46, 0x23, 0x49, 0x92,
        0x72, 0x95, 0x5b, 0x49, 0xca, 0xe0, 0x45, 0x59, 0xb2, 0xca, 0xf4, 0xfc,
        0xb7, 0x59, 0x37, 0x49, 0x28, 0xbc, 0xf3, 0xd7, 0x61, 0xbc, 0x4b, 0xf3,
        0xa9, 0x4b, 0x2f, 0x05, 0xa8, 0x01, 0xa5, 0xdc, 0x00, 0x6e, 0x01, 0xb6,
        0x45, 0x3c, 0xd5, 0x49, 0x7d, 0x5c, 0x25, 0xe8, 0x31, 0x87, 0xb2, 0xb9,
        0xbf, 0xb3, 0x01, 0x62, 0x0c, 0xd0, 0x48, 0x77, 0xa2, 0x34, 0x0f, 0x16,
        0x22, 0x28, 0xee, 0x54, 0x08, 0x93, 0x3b, 0xe4, 0xde, 0x7e, 0x63, 0xf7,
        0x97, 0x16, 0x5d, 0x71, 0x58, 0xc2, 0x2e, 0xf2, 0x36, 0xa6, 0x12, 0x65,
        0x94, 0x17, 0xac, 0x66, 0x23, 0x7e, 0xc6, 0x72, 0x79, 0x24, 0xce, 0x8f,
        0x55, 0x19, 0x97, 0x44, 0xfc, 0x55, 0xec, 0x85, 0x26, 0x27, 0xdb, 0x38,
        0xb1, 0x42, 0x0a, 0xdd, 0x05, 0x99, 0x28, 0xeb, 0x03, 0x6c, 0x9a, 0xe9,
        0x17, 0xf6, 0x2c, 0xb0, 0xfe, 0xe7, 0xa4, 0xa7, 0x31, 0xda, 0x4d, 0xb0,
        0x29, 0xdb, 0xdd, 0x8d, 0x12, 0x13, 0x9c, 0xb4, 0xcc, 0x83, 0x97, 0xfb,
        0x1a, 0xdc, 0x08, 0xd6, 0x30, 0x62, 0xe8, 0xeb, 0x8b, 0x61, 0xcb, 0x1d,
        0x06, 0xe3, 0xa5, 0x4d, 0x35, 0xdb, 0x59, 0xa8, 0x2d, 0x87, 0x27, 0x44,
        0x6f, 0xc0, 0x38, 0x97, 0xe4, 0x85, 0x00, 0x02, 0x09, 0xf6, 0x69, 0x3a,
        0xcf, 0x08, 0x1b, 0x21, 0xbb, 0x79, 0xb1, 0xa1, 0x34, 0x09, 0xe0, 0x80,
        0xca, 0xb0, 0x78, 0x8a, 0x11, 0x97, 0xd4, 0x07, 0xbe, 0x1b, 0x6a, 0x5d,
        0xdb, 0xd6, 0x1f, 0x76, 0x6b, 0x16, 0xf0, 0x58, 0x84, 0x5f, 0x59, 0xce,
        0x62, 0x34, 0xc3, 0xdf, 0x94, 0xb8, 0x2f, 0x84, 0x68, 0xf0, 0xb8, 0x51,
        0xd9, 0x6d, 0x8e, 0x4a, 0x1d, 0xe6, 0x5c, 0xd8, 0x86, 0x25, 0xe3, 0x24,
        0xfd, 0x21, 0x61, 0x13, 0x48, 0x3e, 0xf6, 0x7d, 0xa6, 0x71, 0x9b, 0xd2,
        0x6e, 0xe6, 0xd2, 0x08, 0x94, 0x62, 0x6c, 0x98, 0xfe, 0x2f, 0x9c, 0x88,
        0x7e, 0x78, 0x15, 0x02, 0x00, 0xf0, 0xba, 0x24, 0x91, 0xf2, 0xdc, 0x47,
        0x51, 0x4d, 0x15, 0x5e, 0x91, 0x5f, 0x57, 0x5b, 0x1d, 0x35, 0x24, 0x45,
        0x75, 0x9b, 0x88, 0x75, 0xf1, 0x2f, 0x85, 0xe7, 0x89, 0xd1, 0x01, 0xb4,
        0xc8, 0x18, 0xb7, 0x97, 0xef, 0x4b, 0x90, 0xf4, 0xbf, 0x10, 0x27, 0x3c,
        0x60, 0xff, 0xc4, 0x94, 0x20, 0x2f, 0x93, 0x4b, 0x4d, 0xe3, 0x80, 0xf7,
        0x2c, 0x71, 0xd9, 0xe3, 0x68, 0xb4, 0x77, 0x2b, 0xc7, 0x0d, 0x39, 0x92,
        0xef, 0x91, 0x0d, 0xb2, 0x11, 0x50, 0x0e, 0xe8, 0xad, 0x3b, 0xf6, 0xb5,
        0xc6, 0x14, 0x4d, 0x33, 0x53, 0xa7, 0x60, 0x15, 0xc7, 0x27, 0x51, 0xdc,
        0x54, 0x29, 0xa7, 0x0d, 0x6a, 0x7b, 0x72, 0x13, 0xad, 0x7d, 0x41, 0x19,
        0x4e, 0x42, 0x49, 0xcc, 0x42, 0xe4, 0xbd, 0x99, 0x13, 0xd9, 0x7f, 0xf3,
        0x38, 0xa4, 0xb6, 0x33, 0xed, 0x07, 0x48, 0x7e, 0x8e, 0x82, 0xfe, 0x3a,
        0x9d, 0x75, 0x93, 0xba, 0x25, 0x4e, 0x37, 0x3c, 0x0c, 0xd5, 0x69, 0xa9,
        0x2d, 0x9e, 0xfd, 0xe8, 0xbb, 0xf5, 0x0c, 0xe2, 0x86, 0xb9, 0x5e, 0x6f,
        0x28, 0xe4, 0x19, 0xb3, 0x0b, 0xa4, 0x86, 0xd7, 0x24, 0xd0, 0xb8, 0x89,
        0x7b, 0x76, 0xec, 0x05, 0x10, 0x5b, 0x68, 0xe9, 0x58, 0x66, 0xa3, 0xc5,
        0xb6, 0x63, 0x20, 0x0e, 0x0e, 0xea, 0x3d, 0x61, 0x5e, 0xda, 0x3d, 0x3c,
        0xf9, 0xfd, 0xed, 0xa9, 0xdb, 0x52, 0x94, 0x8a, 0x00, 0xca, 0x3c, 0x8d,
        0x66, 0x8f, 0xb0, 0xf0, 0x5a, 0xca, 0x3f, 0x63, 0x71, 0xbf, 0xca, 0x99,
        0x37, 0x9b, 0x75, 0x97, 0x89, 0x10, 0x6e, 0xcf, 0xf2, 0xf5, 0xe3, 0xd5,
        0x45, 0x9b, 0xad, 0x10, 0x71, 0x6c, 0x5f, 0x6f, 0x7f, 0x22, 0x77, 0x18,
        0x2f, 0xf9, 0x99, 0xc5, 0x69, 0x58, 0x03, 0x12, 0x86, 0x82, 0x3e, 0xbf,
        0xc2, 0x12, 0x35, 0x43, 0xa3, 0xd9, 0x18, 0x4f, 0x41, 0x11, 0x6b, 0xf3,
        0x67, 0xaf, 0x3d, 0x78, 0xe4, 0x22, 0x2d, 0xb3, 0x48, 0x43, 0x31, 0x1d,
        0xef, 0xa8, 0xba, 0x49, 0x8e, 0xa9, 0xa7, 0xb6, 0x18, 0x77, 0x84, 0xca,
        0xbd, 0xa2, 0x02, 0x1b, 0x6a, 0xf8, 0x5f, 0xda, 0xff, 0xcf, 0x01, 0x6a,
        0x86, 0x69, 0xa9, 0xe9, 0xcb, 0x60, 0x1e, 0x15, 0xdc, 0x8f, 0x5d, 0x39,
        0xb5, 0xce, 0x55, 0x5f, 0x47, 0x97, 0xb1, 0x19, 0x6e, 0x21, 0xd6, 0x13,
        0x39, 0xb2, 0x24, 0xe0, 0x62, 0x82, 0x9f, 0xed, 0x12, 0x81, 0xed, 0xee,
        0xab, 0xd0, 0x2f, 0x19, 0x89, 0x3f, 0x57, 0x2e, 0xc2, 0xe2, 0x67, 0xe8,
        0xae, 0x03, 0x56, 0xba, 0xd4, 0xd0, 0xa4, 0x89, 0x03, 0x06, 0x5b, 0xcc,
        0xf2, 0x22, 0xb8, 0x0e, 0x76, 0x79, 0x4a, 0x42, 0x1d, 0x37, 0x51, 0x5a,
        0xaa, 0x46, 0x6c, 0x2a, 0xdd, 0x66, 0xfe, 0xc6, 0x68, 0xc3, 0x38, 0xa2,
        0xae, 0x5b, 0x98, 0x24, 0x5d, 0x43, 0x05, 0x82, 0x38, 0x12, 0xd3, 0xd1,
        0x75, 0x2d, 0x4f, 0x61, 0xbd, 0xb9, 0x10, 0x87, 0x44, 0x2a, 0x78, 0x07,
        0xff, 0xf4, 0x0f, 0xa1, 0xf3, 0x68, 0x9f, 0xbe, 0xae, 0xa2, 0x91, 0xf0,
        0xc7, 0x55, 0x7a, 0x52, 0xd5, 0xa3, 0x8d, 0x6f, 0xe4, 0x90, 0x5c, 0xf3,
        0x5f, 0xce, 0x3d, 0x23, 0xf9, 0x8e, 0xae, 0x14, 0xfb, 0x82, 0x9a, 0xa3,
        0x04, 0x5f, 0xbf, 0xad, 0x3e, 0xf2, 0x97, 0x0a, 0x60, 0x40, 0x70, 0x19,
        0x72, 0xad, 0x66, 0xfb, 0x78, 0x1b, 0x84, 0x6c, 0x98, 0xbc, 0x8c, 0xf8,
        0x4f, 0xcb, 0xb5, 0xf6, 0xaf, 0x7a, 0xb7, 0x93, 0xef, 0x67, 0x48, 0x02,
        0x2c, 0xcb, 0xe6, 0x77, 0x0f, 0x7b, 0xc1, 0xee, 0xc5, 0xb6, 0x2d, 0x7e,
        0x62, 0xa0, 0xc0, 0xa7, 0xa5, 0x80, 0x31, 0x92, 0x50, 0xa1, 0x28, 0x22,
        0x95, 0x03, 0x17, 0xd1, 0x0f, 0xf6, 0x08, 0xe5, 0xec
    };
#define CHACHA_BIG_TEST_SIZE 1305
#if !defined(WOLFSSL_SMALL_STACK) || defined(WOLFSSL_NO_MALLOC)
    byte   cipher_big[CHACHA_BIG_TEST_SIZE] = {0};
    byte   plain_big[CHACHA_BIG_TEST_SIZE] = {0};
    byte   input_big[CHACHA_BIG_TEST_SIZE] = {0};
#else
    byte*  cipher_big;
    byte*  plain_big;
    byte*  input_big;
#endif /* WOLFSSL_SMALL_STACK && !WOLFSSL_NO_MALLOC */
    int    block_size;
#endif /* BENCH_EMBEDDED */

    byte a[] = {0x76,0xb8,0xe0,0xad,0xa0,0xf1,0x3d,0x90};
    byte b[] = {0x45,0x40,0xf0,0x5a,0x9f,0x1f,0xb2,0x96};
    byte c[] = {0xde,0x9c,0xba,0x7b,0xf3,0xd6,0x9e,0xf5};
    byte d[] = {0x89,0x67,0x09,0x52,0x60,0x83,0x64,0xfd};

    byte* test_chacha[4];

    test_chacha[0] = a;
    test_chacha[1] = b;
    test_chacha[2] = c;
    test_chacha[3] = d;

#ifndef BENCH_EMBEDDED
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    cipher_big = (byte*)XMALLOC(CHACHA_BIG_TEST_SIZE, HEAP_HINT,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (cipher_big == NULL) {
        return MEMORY_E;
    }
    plain_big = (byte*)XMALLOC(CHACHA_BIG_TEST_SIZE, HEAP_HINT,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (plain_big == NULL) {
        return MEMORY_E;
    }
    input_big = (byte*)XMALLOC(CHACHA_BIG_TEST_SIZE, HEAP_HINT,
                                                       DYNAMIC_TYPE_TMP_BUFFER);
    if (input_big == NULL) {
        return MEMORY_E;
    }
    XMEMSET(cipher_big, 0, CHACHA_BIG_TEST_SIZE);
    XMEMSET(plain_big, 0, CHACHA_BIG_TEST_SIZE);
    XMEMSET(input_big, 0, CHACHA_BIG_TEST_SIZE);
#endif /* WOLFSSL_SMALL_STACK && !WOLFSSL_NO_MALLOC */
#endif /* BENCH_EMBEDDED */

    for (i = 0; i < times; ++i) {
        if (i < 3) {
            keySz = 32;
        }
        else {
            keySz = 16;
        }

        XMEMCPY(plain, keys[i], keySz);
        XMEMSET(cipher, 0, 32);
        XMEMCPY(cipher + 4, ivs[i], 8);

        ret |= wc_Chacha_SetKey(&enc, keys[i], keySz);
        ret |= wc_Chacha_SetKey(&dec, keys[i], keySz);
        if (ret != 0)
            return ret;

        ret |= wc_Chacha_SetIV(&enc, cipher, 0);
        ret |= wc_Chacha_SetIV(&dec, cipher, 0);
        if (ret != 0)
            return ret;
        XMEMCPY(plain, input, 8);

        ret |= wc_Chacha_Process(&enc, cipher, plain,  (word32)8);
        ret |= wc_Chacha_Process(&dec, plain,  cipher, (word32)8);
        if (ret != 0)
            return ret;

        if (XMEMCMP(test_chacha[i], cipher, 8))
            return WC_TEST_RET_ENC_I(i);

        if (XMEMCMP(plain, input, 8))
            return WC_TEST_RET_ENC_I(i);
    }

    /* test of starting at a different counter
       encrypts all of the information and decrypts starting at 2nd chunk */
    XMEMSET(plain,  0, sizeof(plain));
    XMEMSET(sliver, 1, sizeof(sliver)); /* set as 1's to not match plain */
    XMEMSET(cipher, 0, sizeof(cipher));
    XMEMCPY(cipher + 4, ivs[0], 8);

    ret |= wc_Chacha_SetKey(&enc, keys[0], keySz);
    ret |= wc_Chacha_SetKey(&dec, keys[0], keySz);
    if (ret != 0)
        return ret;

    ret |= wc_Chacha_SetIV(&enc, cipher, 0);
    ret |= wc_Chacha_SetIV(&dec, cipher, 1);
    if (ret != 0)
        return ret;

    ret |= wc_Chacha_Process(&enc, cipher, plain,  sizeof(plain));
    ret |= wc_Chacha_Process(&dec, sliver,  cipher + 64, sizeof(sliver));
    if (ret != 0)
        return ret;

    if (XMEMCMP(plain + 64, sliver, 64))
        return WC_TEST_RET_ENC_NC;

#ifndef BENCH_EMBEDDED
    /* test of encrypting more data */
    keySz = 32;

    ret |= wc_Chacha_SetKey(&enc, keys[0], keySz);
    ret |= wc_Chacha_SetKey(&dec, keys[0], keySz);
    if (ret != 0)
        return ret;

    ret |= wc_Chacha_SetIV(&enc, ivs[2], 0);
    ret |= wc_Chacha_SetIV(&dec, ivs[2], 0);
    if (ret != 0)
        return ret;

    ret |= wc_Chacha_Process(&enc, cipher_big, plain_big, CHACHA_BIG_TEST_SIZE);
    ret |= wc_Chacha_Process(&dec, plain_big,  cipher_big,
                                                          CHACHA_BIG_TEST_SIZE);
    if (ret != 0)
        return ret;

    if (XMEMCMP(plain_big, input_big, CHACHA_BIG_TEST_SIZE))
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(cipher_big, cipher_big_result, CHACHA_BIG_TEST_SIZE))
        return WC_TEST_RET_ENC_NC;

    for (i = 0; i < 18; ++i) {
        /* this will test all paths
         * block sizes: 1 3 7 15 31 63 127 255 511 (i = 0- 8)
         *              2 4 8 16 32 64 128 256 512 (i = 9-17)
         */
        block_size = (2 << (i%9)) - (i<9?1:0);
        keySz = 32;

        ret |= wc_Chacha_SetKey(&enc, keys[0], keySz);
        ret |= wc_Chacha_SetKey(&dec, keys[0], keySz);
        if (ret != 0)
            return ret;

        ret |= wc_Chacha_SetIV(&enc, ivs[2], 0);
        ret |= wc_Chacha_SetIV(&dec, ivs[2], 0);
        if (ret != 0)
            return ret;

        ret |= wc_Chacha_Process(&enc, cipher_big, plain_big , block_size);
        ret |= wc_Chacha_Process(&dec, plain_big , cipher_big, block_size);
        if (ret != 0)
            return ret;

        if (XMEMCMP(plain_big, input_big, block_size))
            return WC_TEST_RET_ENC_I(i);

        if (XMEMCMP(cipher_big, cipher_big_result, block_size))
            return WC_TEST_RET_ENC_I(i);
    }

    /* Streaming test */
    for (i = 1; i <= (int)CHACHA_CHUNK_BYTES + 1; i++) {
        int j, rem;

        ret = wc_Chacha_SetKey(&enc, keys[0], keySz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_Chacha_SetKey(&dec, keys[0], keySz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        ret = wc_Chacha_SetIV(&enc, ivs[2], 0);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_Chacha_SetIV(&dec, ivs[2], 0);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        for (j = 0; j < CHACHA_BIG_TEST_SIZE - i; j+= i) {
            ret = wc_Chacha_Process(&enc, cipher_big + j, plain_big + j, i);
            if (ret != 0)
                return WC_TEST_RET_ENC_EC(ret);
            ret = wc_Chacha_Process(&dec, plain_big + j, cipher_big + j, i);
            if (ret != 0)
                return WC_TEST_RET_ENC_EC(ret);
        }

        rem = CHACHA_BIG_TEST_SIZE - j;
        ret = wc_Chacha_Process(&enc, cipher_big + j, plain_big + j, rem);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        ret = wc_Chacha_Process(&dec, plain_big + j, cipher_big + j, rem);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        if (XMEMCMP(plain_big, input_big, CHACHA_BIG_TEST_SIZE))
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(cipher_big, cipher_big_result, CHACHA_BIG_TEST_SIZE))
            return WC_TEST_RET_ENC_NC;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XFREE(cipher_big, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(plain_big, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(input_big, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* WOLFSSL_SMALL_STACK && !WOLFSSL_NO_MALLOC */
#endif /* BENCH_EMBEDDED */

    return 0;
}
#endif /* HAVE_CHACHA */


#ifdef HAVE_POLY1305
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t poly1305_test(void)
{
    wc_test_ret_t ret = 0;
    int      i;
    byte     tag[16];
    Poly1305 enc;

    WOLFSSL_SMALL_STACK_STATIC const byte msg1[] =
    {
        0x43,0x72,0x79,0x70,0x74,0x6f,0x67,0x72,
        0x61,0x70,0x68,0x69,0x63,0x20,0x46,0x6f,
        0x72,0x75,0x6d,0x20,0x52,0x65,0x73,0x65,
        0x61,0x72,0x63,0x68,0x20,0x47,0x72,0x6f,
        0x75,0x70
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
    {
        0x48,0x65,0x6c,0x6c,0x6f,0x20,0x77,0x6f,0x72,
        0x6c,0x64,0x21
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg3[] =
    {
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg4[] =
    {
        0xd3,0x1a,0x8d,0x34,0x64,0x8e,0x60,0xdb,
        0x7b,0x86,0xaf,0xbc,0x53,0xef,0x7e,0xc2,
        0xa4,0xad,0xed,0x51,0x29,0x6e,0x08,0xfe,
        0xa9,0xe2,0xb5,0xa7,0x36,0xee,0x62,0xd6,
        0x3d,0xbe,0xa4,0x5e,0x8c,0xa9,0x67,0x12,
        0x82,0xfa,0xfb,0x69,0xda,0x92,0x72,0x8b,
        0x1a,0x71,0xde,0x0a,0x9e,0x06,0x0b,0x29,
        0x05,0xd6,0xa5,0xb6,0x7e,0xcd,0x3b,0x36,
        0x92,0xdd,0xbd,0x7f,0x2d,0x77,0x8b,0x8c,
        0x98,0x03,0xae,0xe3,0x28,0x09,0x1b,0x58,
        0xfa,0xb3,0x24,0xe4,0xfa,0xd6,0x75,0x94,
        0x55,0x85,0x80,0x8b,0x48,0x31,0xd7,0xbc,
        0x3f,0xf4,0xde,0xf0,0x8e,0x4b,0x7a,0x9d,
        0xe5,0x76,0xd2,0x65,0x86,0xce,0xc6,0x4b,
        0x61,0x16
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg5[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
    };

    WOLFSSL_SMALL_STACK_STATIC const byte msg6[] =
    {
        0xd3,0x1a,0x8d,0x34,0x64,0x8e,0x60,0xdb,
        0x7b,0x86,0xaf,0xbc,0x53,0xef,0x7e,0xc2,
        0xa4,0xad,0xed,0x51,0x29,0x6e,0x08,0xfe,
        0xa9,0xe2,0xb5,0xa7,0x36,0xee,0x62,0xd6,
        0x3d,0xbe,0xa4,0x5e,0x8c,0xa9,0x67,0x12,
        0x82,0xfa,0xfb,0x69,0xda,0x92,0x72,0x8b,
        0xfa,0xb3,0x24,0xe4,0xfa,0xd6,0x75,0x94,
        0x1a,0x71,0xde,0x0a,0x9e,0x06,0x0b,0x29,
        0xa9,0xe2,0xb5,0xa7,0x36,0xee,0x62,0xd6,
        0x3d,0xbe,0xa4,0x5e,0x8c,0xa9,0x67,0x12,
        0xfa,0xb3,0x24,0xe4,0xfa,0xd6,0x75,0x94,
        0x05,0xd6,0xa5,0xb6,0x7e,0xcd,0x3b,0x36,
        0x92,0xdd,0xbd,0x7f,0x2d,0x77,0x8b,0x8c,
        0x7b,0x86,0xaf,0xbc,0x53,0xef,0x7e,0xc2,
        0x98,0x03,0xae,0xe3,0x28,0x09,0x1b,0x58,
        0xfa,0xb3,0x24,0xe4,0xfa,0xd6,0x75,0x94,
        0x55,0x85,0x80,0x8b,0x48,0x31,0xd7,0xbc,
        0x3f,0xf4,0xde,0xf0,0x8e,0x4b,0x7a,0x9d,
        0xe5,0x76,0xd2,0x65,0x86,0xce,0xc6,0x4b,
        0x61,0x16
    };

    byte additional[] =
    {
        0x50,0x51,0x52,0x53,0xc0,0xc1,0xc2,0xc3,
        0xc4,0xc5,0xc6,0xc7
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct0[] =
    {
        0x01,0x03,0x80,0x8a,0xfb,0x0d,0xb2,0xfd,
        0x4a,0xbf,0xf6,0xaf,0x41,0x49,0xf5,0x1b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct1[] =
    {
        0xa8,0x06,0x1d,0xc1,0x30,0x51,0x36,0xc6,
        0xc2,0x2b,0x8b,0xaf,0x0c,0x01,0x27,0xa9
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct2[] =
    {
        0xa6,0xf7,0x45,0x00,0x8f,0x81,0xc9,0x16,
        0xa2,0x0d,0xcc,0x74,0xee,0xf2,0xb2,0xf0
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct3[] =
    {
        0x49,0xec,0x78,0x09,0x0e,0x48,0x1e,0xc6,
        0xc2,0x6b,0x33,0xb9,0x1c,0xcc,0x03,0x07
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct4[] =
    {
        0x1a,0xe1,0x0b,0x59,0x4f,0x09,0xe2,0x6a,
        0x7e,0x90,0x2e,0xcb,0xd0,0x60,0x06,0x91
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct5[] =
    {
        0x03,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
    };

    WOLFSSL_SMALL_STACK_STATIC const byte correct6[] =
    {
        0xea,0x11,0x5c,0x4f,0xd0,0xc0,0x10,0xae,
        0xf7,0xdf,0xda,0x77,0xa2,0xe9,0xaf,0xca
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key[] = {
        0x85,0xd6,0xbe,0x78,0x57,0x55,0x6d,0x33,
        0x7f,0x44,0x52,0xfe,0x42,0xd5,0x06,0xa8,
        0x01,0x03,0x80,0x8a,0xfb,0x0d,0xb2,0xfd,
        0x4a,0xbf,0xf6,0xaf,0x41,0x49,0xf5,0x1b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key2[] = {
        0x74,0x68,0x69,0x73,0x20,0x69,0x73,0x20,
        0x33,0x32,0x2d,0x62,0x79,0x74,0x65,0x20,
        0x6b,0x65,0x79,0x20,0x66,0x6f,0x72,0x20,
        0x50,0x6f,0x6c,0x79,0x31,0x33,0x30,0x35
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key4[] = {
        0x7b,0xac,0x2b,0x25,0x2d,0xb4,0x47,0xaf,
        0x09,0xb6,0x7a,0x55,0xa4,0xe9,0x55,0x84,
        0x0a,0xe1,0xd6,0x73,0x10,0x75,0xd9,0xeb,
        0x2a,0x93,0x75,0x78,0x3e,0xd5,0x53,0xff
    };

    WOLFSSL_SMALL_STACK_STATIC const byte key5[] = {
        0x02,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
        0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
    };

    const byte* msgs[]  = {NULL, msg1, msg2, msg3, msg5, msg6};
    word32      szm[]   = {0, sizeof(msg1), sizeof(msg2),
                           sizeof(msg3), sizeof(msg5), sizeof(msg6)};
    const byte* keys[]  = {key, key, key2, key2, key5, key};
    const byte* tests[] = {correct0, correct1, correct2, correct3, correct5,
                           correct6};

    for (i = 0; i < 6; i++) {
        ret = wc_Poly1305SetKey(&enc, keys[i], 32);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        ret = wc_Poly1305Update(&enc, msgs[i], szm[i]);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        ret = wc_Poly1305Final(&enc, tag);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);

        if (XMEMCMP(tag, tests[i], sizeof(tag)))
            return WC_TEST_RET_ENC_I(i);
    }

    /* Check TLS MAC function from 2.8.2 https://tools.ietf.org/html/rfc7539 */
    XMEMSET(tag, 0, sizeof(tag));
    ret = wc_Poly1305SetKey(&enc, key4, sizeof(key4));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Poly1305_MAC(&enc, additional, sizeof(additional),
                                   (byte*)msg4, sizeof(msg4), tag, sizeof(tag));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(tag, correct4, sizeof(tag)))
        return WC_TEST_RET_ENC_NC;

    /* Check fail of TLS MAC function if altering additional data */
    XMEMSET(tag, 0, sizeof(tag));
    additional[0]++;
    ret = wc_Poly1305_MAC(&enc, additional, sizeof(additional),
                                   (byte*)msg4, sizeof(msg4), tag, sizeof(tag));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(tag, correct4, sizeof(tag)) == 0)
        return WC_TEST_RET_ENC_NC;


    return 0;
}
#endif /* HAVE_POLY1305 */


#if defined(HAVE_CHACHA) && defined(HAVE_POLY1305)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t chacha20_poly1305_aead_test(void)
{
    /* Test #1 from Section 2.8.2 of draft-irtf-cfrg-chacha20-poly1305-10 */
    /* https://tools.ietf.org/html/draft-irtf-cfrg-chacha20-poly1305-10  */

    WOLFSSL_SMALL_STACK_STATIC const byte key1[] = {
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
        0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
        0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte plaintext1[] = {
        0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61,
        0x6e, 0x64, 0x20, 0x47, 0x65, 0x6e, 0x74, 0x6c,
        0x65, 0x6d, 0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20,
        0x74, 0x68, 0x65, 0x20, 0x63, 0x6c, 0x61, 0x73,
        0x73, 0x20, 0x6f, 0x66, 0x20, 0x27, 0x39, 0x39,
        0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63,
        0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6f, 0x66, 0x66,
        0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f,
        0x6e, 0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20,
        0x74, 0x69, 0x70, 0x20, 0x66, 0x6f, 0x72, 0x20,
        0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75,
        0x72, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x6e, 0x73,
        0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f,
        0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x69,
        0x74, 0x2e
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv1[] = {
        0x07, 0x00, 0x00, 0x00, 0x40, 0x41, 0x42, 0x43,
        0x44, 0x45, 0x46, 0x47
    };

    WOLFSSL_SMALL_STACK_STATIC const byte aad1[] = { /* additional data */
        0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3,
        0xc4, 0xc5, 0xc6, 0xc7
    };

    WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] = { /* expected output from operation */
        0xd3, 0x1a, 0x8d, 0x34, 0x64, 0x8e, 0x60, 0xdb,
        0x7b, 0x86, 0xaf, 0xbc, 0x53, 0xef, 0x7e, 0xc2,
        0xa4, 0xad, 0xed, 0x51, 0x29, 0x6e, 0x08, 0xfe,
        0xa9, 0xe2, 0xb5, 0xa7, 0x36, 0xee, 0x62, 0xd6,
        0x3d, 0xbe, 0xa4, 0x5e, 0x8c, 0xa9, 0x67, 0x12,
        0x82, 0xfa, 0xfb, 0x69, 0xda, 0x92, 0x72, 0x8b,
        0x1a, 0x71, 0xde, 0x0a, 0x9e, 0x06, 0x0b, 0x29,
        0x05, 0xd6, 0xa5, 0xb6, 0x7e, 0xcd, 0x3b, 0x36,
        0x92, 0xdd, 0xbd, 0x7f, 0x2d, 0x77, 0x8b, 0x8c,
        0x98, 0x03, 0xae, 0xe3, 0x28, 0x09, 0x1b, 0x58,
        0xfa, 0xb3, 0x24, 0xe4, 0xfa, 0xd6, 0x75, 0x94,
        0x55, 0x85, 0x80, 0x8b, 0x48, 0x31, 0xd7, 0xbc,
        0x3f, 0xf4, 0xde, 0xf0, 0x8e, 0x4b, 0x7a, 0x9d,
        0xe5, 0x76, 0xd2, 0x65, 0x86, 0xce, 0xc6, 0x4b,
        0x61, 0x16
    };

    WOLFSSL_SMALL_STACK_STATIC const byte authTag1[] = { /* expected output from operation */
        0x1a, 0xe1, 0x0b, 0x59, 0x4f, 0x09, 0xe2, 0x6a,
        0x7e, 0x90, 0x2e, 0xcb, 0xd0, 0x60, 0x06, 0x91
    };

    /* Test #2 from Appendix A.2 in draft-irtf-cfrg-chacha20-poly1305-10 */
    /* https://tools.ietf.org/html/draft-irtf-cfrg-chacha20-poly1305-10  */

    WOLFSSL_SMALL_STACK_STATIC const byte key2[] = {
        0x1c, 0x92, 0x40, 0xa5, 0xeb, 0x55, 0xd3, 0x8a,
        0xf3, 0x33, 0x88, 0x86, 0x04, 0xf6, 0xb5, 0xf0,
        0x47, 0x39, 0x17, 0xc1, 0x40, 0x2b, 0x80, 0x09,
        0x9d, 0xca, 0x5c, 0xbc, 0x20, 0x70, 0x75, 0xc0
    };

    WOLFSSL_SMALL_STACK_STATIC const byte plaintext2[] = {
        0x49, 0x6e, 0x74, 0x65, 0x72, 0x6e, 0x65, 0x74,
        0x2d, 0x44, 0x72, 0x61, 0x66, 0x74, 0x73, 0x20,
        0x61, 0x72, 0x65, 0x20, 0x64, 0x72, 0x61, 0x66,
        0x74, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65,
        0x6e, 0x74, 0x73, 0x20, 0x76, 0x61, 0x6c, 0x69,
        0x64, 0x20, 0x66, 0x6f, 0x72, 0x20, 0x61, 0x20,
        0x6d, 0x61, 0x78, 0x69, 0x6d, 0x75, 0x6d, 0x20,
        0x6f, 0x66, 0x20, 0x73, 0x69, 0x78, 0x20, 0x6d,
        0x6f, 0x6e, 0x74, 0x68, 0x73, 0x20, 0x61, 0x6e,
        0x64, 0x20, 0x6d, 0x61, 0x79, 0x20, 0x62, 0x65,
        0x20, 0x75, 0x70, 0x64, 0x61, 0x74, 0x65, 0x64,
        0x2c, 0x20, 0x72, 0x65, 0x70, 0x6c, 0x61, 0x63,
        0x65, 0x64, 0x2c, 0x20, 0x6f, 0x72, 0x20, 0x6f,
        0x62, 0x73, 0x6f, 0x6c, 0x65, 0x74, 0x65, 0x64,
        0x20, 0x62, 0x79, 0x20, 0x6f, 0x74, 0x68, 0x65,
        0x72, 0x20, 0x64, 0x6f, 0x63, 0x75, 0x6d, 0x65,
        0x6e, 0x74, 0x73, 0x20, 0x61, 0x74, 0x20, 0x61,
        0x6e, 0x79, 0x20, 0x74, 0x69, 0x6d, 0x65, 0x2e,
        0x20, 0x49, 0x74, 0x20, 0x69, 0x73, 0x20, 0x69,
        0x6e, 0x61, 0x70, 0x70, 0x72, 0x6f, 0x70, 0x72,
        0x69, 0x61, 0x74, 0x65, 0x20, 0x74, 0x6f, 0x20,
        0x75, 0x73, 0x65, 0x20, 0x49, 0x6e, 0x74, 0x65,
        0x72, 0x6e, 0x65, 0x74, 0x2d, 0x44, 0x72, 0x61,
        0x66, 0x74, 0x73, 0x20, 0x61, 0x73, 0x20, 0x72,
        0x65, 0x66, 0x65, 0x72, 0x65, 0x6e, 0x63, 0x65,
        0x20, 0x6d, 0x61, 0x74, 0x65, 0x72, 0x69, 0x61,
        0x6c, 0x20, 0x6f, 0x72, 0x20, 0x74, 0x6f, 0x20,
        0x63, 0x69, 0x74, 0x65, 0x20, 0x74, 0x68, 0x65,
        0x6d, 0x20, 0x6f, 0x74, 0x68, 0x65, 0x72, 0x20,
        0x74, 0x68, 0x61, 0x6e, 0x20, 0x61, 0x73, 0x20,
        0x2f, 0xe2, 0x80, 0x9c, 0x77, 0x6f, 0x72, 0x6b,
        0x20, 0x69, 0x6e, 0x20, 0x70, 0x72, 0x6f, 0x67,
        0x72, 0x65, 0x73, 0x73, 0x2e, 0x2f, 0xe2, 0x80,
        0x9d
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv2[] = {
        0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03, 0x04,
        0x05, 0x06, 0x07, 0x08
    };

    WOLFSSL_SMALL_STACK_STATIC const byte aad2[] = { /* additional data */
        0xf3, 0x33, 0x88, 0x86, 0x00, 0x00, 0x00, 0x00,
        0x00, 0x00, 0x4e, 0x91
    };

    WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] = { /* expected output from operation */
        0x64, 0xa0, 0x86, 0x15, 0x75, 0x86, 0x1a, 0xf4,
        0x60, 0xf0, 0x62, 0xc7, 0x9b, 0xe6, 0x43, 0xbd,
        0x5e, 0x80, 0x5c, 0xfd, 0x34, 0x5c, 0xf3, 0x89,
        0xf1, 0x08, 0x67, 0x0a, 0xc7, 0x6c, 0x8c, 0xb2,
        0x4c, 0x6c, 0xfc, 0x18, 0x75, 0x5d, 0x43, 0xee,
        0xa0, 0x9e, 0xe9, 0x4e, 0x38, 0x2d, 0x26, 0xb0,
        0xbd, 0xb7, 0xb7, 0x3c, 0x32, 0x1b, 0x01, 0x00,
        0xd4, 0xf0, 0x3b, 0x7f, 0x35, 0x58, 0x94, 0xcf,
        0x33, 0x2f, 0x83, 0x0e, 0x71, 0x0b, 0x97, 0xce,
        0x98, 0xc8, 0xa8, 0x4a, 0xbd, 0x0b, 0x94, 0x81,
        0x14, 0xad, 0x17, 0x6e, 0x00, 0x8d, 0x33, 0xbd,
        0x60, 0xf9, 0x82, 0xb1, 0xff, 0x37, 0xc8, 0x55,
        0x97, 0x97, 0xa0, 0x6e, 0xf4, 0xf0, 0xef, 0x61,
        0xc1, 0x86, 0x32, 0x4e, 0x2b, 0x35, 0x06, 0x38,
        0x36, 0x06, 0x90, 0x7b, 0x6a, 0x7c, 0x02, 0xb0,
        0xf9, 0xf6, 0x15, 0x7b, 0x53, 0xc8, 0x67, 0xe4,
        0xb9, 0x16, 0x6c, 0x76, 0x7b, 0x80, 0x4d, 0x46,
        0xa5, 0x9b, 0x52, 0x16, 0xcd, 0xe7, 0xa4, 0xe9,
        0x90, 0x40, 0xc5, 0xa4, 0x04, 0x33, 0x22, 0x5e,
        0xe2, 0x82, 0xa1, 0xb0, 0xa0, 0x6c, 0x52, 0x3e,
        0xaf, 0x45, 0x34, 0xd7, 0xf8, 0x3f, 0xa1, 0x15,
        0x5b, 0x00, 0x47, 0x71, 0x8c, 0xbc, 0x54, 0x6a,
        0x0d, 0x07, 0x2b, 0x04, 0xb3, 0x56, 0x4e, 0xea,
        0x1b, 0x42, 0x22, 0x73, 0xf5, 0x48, 0x27, 0x1a,
        0x0b, 0xb2, 0x31, 0x60, 0x53, 0xfa, 0x76, 0x99,
        0x19, 0x55, 0xeb, 0xd6, 0x31, 0x59, 0x43, 0x4e,
        0xce, 0xbb, 0x4e, 0x46, 0x6d, 0xae, 0x5a, 0x10,
        0x73, 0xa6, 0x72, 0x76, 0x27, 0x09, 0x7a, 0x10,
        0x49, 0xe6, 0x17, 0xd9, 0x1d, 0x36, 0x10, 0x94,
        0xfa, 0x68, 0xf0, 0xff, 0x77, 0x98, 0x71, 0x30,
        0x30, 0x5b, 0xea, 0xba, 0x2e, 0xda, 0x04, 0xdf,
        0x99, 0x7b, 0x71, 0x4d, 0x6c, 0x6f, 0x2c, 0x29,
        0xa6, 0xad, 0x5c, 0xb4, 0x02, 0x2b, 0x02, 0x70,
        0x9b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte authTag2[] = { /* expected output from operation */
        0xee, 0xad, 0x9d, 0x67, 0x89, 0x0c, 0xbb, 0x22,
        0x39, 0x23, 0x36, 0xfe, 0xa1, 0x85, 0x1f, 0x38
    };

    byte generatedCiphertext[265]; /* max plaintext2/cipher2 */
    byte generatedPlaintext[265];  /* max plaintext2/cipher2 */
    byte generatedAuthTag[CHACHA20_POLY1305_AEAD_AUTHTAG_SIZE];
    wc_test_ret_t err;

    ChaChaPoly_Aead aead;

#if !defined(USE_INTEL_CHACHA_SPEEDUP) && !defined(WOLFSSL_ARMASM)
    #define TEST_SMALL_CHACHA_CHUNKS 32
#else
    #define TEST_SMALL_CHACHA_CHUNKS 64
#endif
    #ifdef TEST_SMALL_CHACHA_CHUNKS
    word32 testLen;
    #endif

    XMEMSET(generatedCiphertext, 0, sizeof(generatedCiphertext));
    XMEMSET(generatedAuthTag, 0, sizeof(generatedAuthTag));
    XMEMSET(generatedPlaintext, 0, sizeof(generatedPlaintext));

    /* Parameter Validation testing */
    /* Encrypt */
    err = wc_ChaCha20Poly1305_Encrypt(NULL, iv1, aad1, sizeof(aad1), plaintext1,
            sizeof(plaintext1), generatedCiphertext, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Encrypt(key1, NULL, aad1, sizeof(aad1),
            plaintext1, sizeof(plaintext1), generatedCiphertext,
            generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1, aad1, sizeof(aad1), NULL,
            sizeof(plaintext1), generatedCiphertext, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1, aad1, sizeof(aad1), plaintext1,
            sizeof(plaintext1), NULL, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1, aad1, sizeof(aad1), plaintext1,
            sizeof(plaintext1), generatedCiphertext, NULL);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1, aad1, sizeof(aad1), NULL,
            sizeof(plaintext1), generatedCiphertext, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    /* Decrypt */
    err = wc_ChaCha20Poly1305_Decrypt(NULL, iv2, aad2, sizeof(aad2), cipher2,
            sizeof(cipher2), authTag2, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Decrypt(key2, NULL, aad2, sizeof(aad2), cipher2,
            sizeof(cipher2), authTag2, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2, aad2, sizeof(aad2), NULL,
            sizeof(cipher2), authTag2, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2, aad2, sizeof(aad2), cipher2,
            sizeof(cipher2), NULL, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2, aad2, sizeof(aad2), cipher2,
            sizeof(cipher2), authTag2, NULL);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2, aad2, sizeof(aad2), NULL,
            sizeof(cipher2), authTag2, generatedPlaintext);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);


    /* Test #1 */
    err = wc_ChaCha20Poly1305_Encrypt(key1, iv1,
                                       aad1, sizeof(aad1),
                                       plaintext1, sizeof(plaintext1),
                                       generatedCiphertext, generatedAuthTag);
    if (err) {
        return WC_TEST_RET_ENC_EC(err);
    }

    /* -- Check the ciphertext and authtag */
    if (XMEMCMP(generatedCiphertext, cipher1, sizeof(cipher1))) {
        return WC_TEST_RET_ENC_NC;
    }
    if (XMEMCMP(generatedAuthTag, authTag1, sizeof(authTag1))) {
        return WC_TEST_RET_ENC_NC;
    }

    /* -- Verify decryption works */
    err = wc_ChaCha20Poly1305_Decrypt(key1, iv1,
                                       aad1, sizeof(aad1),
                                       cipher1, sizeof(cipher1),
                                       authTag1, generatedPlaintext);
    if (err) {
        return err;
    }
    if (XMEMCMP(generatedPlaintext, plaintext1, sizeof(plaintext1))) {
        return WC_TEST_RET_ENC_NC;
    }


    XMEMSET(generatedCiphertext, 0, sizeof(generatedCiphertext));
    XMEMSET(generatedAuthTag, 0, sizeof(generatedAuthTag));
    XMEMSET(generatedPlaintext, 0, sizeof(generatedPlaintext));

    /* Test #2 */
    err = wc_ChaCha20Poly1305_Encrypt(key2, iv2,
                                       aad2, sizeof(aad2),
                                       plaintext2, sizeof(plaintext2),
                                       generatedCiphertext, generatedAuthTag);
    if (err) {
        return err;
    }

    /* -- Check the ciphertext and authtag */
    if (XMEMCMP(generatedCiphertext, cipher2, sizeof(cipher2))) {
        return WC_TEST_RET_ENC_NC;
    }
    if (XMEMCMP(generatedAuthTag, authTag2, sizeof(authTag2))) {
        return WC_TEST_RET_ENC_NC;
    }

    /* -- Verify decryption works */
    err = wc_ChaCha20Poly1305_Decrypt(key2, iv2,
                                      aad2, sizeof(aad2),
                                      cipher2, sizeof(cipher2),
                                      authTag2, generatedPlaintext);
    if (err) {
        return err;
    }

    if (XMEMCMP(generatedPlaintext, plaintext2, sizeof(plaintext2))) {
        return WC_TEST_RET_ENC_NC;
    }


    /* AEAD init/update/final - bad argument tests */
    err = wc_ChaCha20Poly1305_Init(NULL, key1, iv1,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Init(&aead, NULL, iv1,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Init(&aead, key1, NULL,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateAad(NULL, aad1, sizeof(aad1));
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, NULL, sizeof(aad1));
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateData(NULL, generatedPlaintext,
        generatedPlaintext, sizeof(plaintext1));
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateData(&aead, generatedPlaintext, NULL,
        sizeof(plaintext1));
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateData(&aead, NULL, generatedPlaintext,
        sizeof(plaintext1));
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Final(NULL, generatedAuthTag);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_Final(&aead, NULL);
    if (err != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(err);

    /* AEAD init/update/final - bad state tests */
    /* clear struct - make valgrind happy to resolve
        "Conditional jump or move depends on uninitialised value(s)".
        The enum is "int" size and aead.state is "byte" */
    /* The wc_ChaCha20Poly1305_Init function does this normally */
    XMEMSET(&aead, 0, sizeof(aead));
    aead.state = CHACHA20_POLY1305_STATE_INIT;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad1, sizeof(aad1));
    if (err != BAD_STATE_E)
        return WC_TEST_RET_ENC_EC(err);
    aead.state = CHACHA20_POLY1305_STATE_DATA;
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad1, sizeof(aad1));
    if (err != BAD_STATE_E)
        return WC_TEST_RET_ENC_EC(err);
    aead.state = CHACHA20_POLY1305_STATE_INIT;
    err = wc_ChaCha20Poly1305_UpdateData(&aead, generatedPlaintext,
        generatedPlaintext, sizeof(plaintext1));
    if (err != BAD_STATE_E)
        return WC_TEST_RET_ENC_EC(err);
    aead.state = CHACHA20_POLY1305_STATE_INIT;
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != BAD_STATE_E)
        return WC_TEST_RET_ENC_EC(err);
    aead.state = CHACHA20_POLY1305_STATE_READY;
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != BAD_STATE_E)
        return WC_TEST_RET_ENC_EC(err);

    XMEMSET(generatedCiphertext, 0, sizeof(generatedCiphertext));
    XMEMSET(generatedAuthTag, 0, sizeof(generatedAuthTag));
    XMEMSET(generatedPlaintext, 0, sizeof(generatedPlaintext));

    /* Test 1 - Encrypt */
    err = wc_ChaCha20Poly1305_Init(&aead, key1, iv1,
        CHACHA20_POLY1305_AEAD_ENCRYPT);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad1, sizeof(aad1));
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
#ifdef TEST_SMALL_CHACHA_CHUNKS
    /* test doing data in smaller chunks */
    for (testLen=0; testLen<sizeof(plaintext1); ) {
        word32 dataLen = sizeof(plaintext1) - testLen;
        if (dataLen > TEST_SMALL_CHACHA_CHUNKS)
            dataLen = TEST_SMALL_CHACHA_CHUNKS;
        err = wc_ChaCha20Poly1305_UpdateData(&aead, &plaintext1[testLen],
            &generatedCiphertext[testLen], dataLen);
        if (err != 0)
            return WC_TEST_RET_ENC_EC(err);
        testLen += dataLen;
    }
#else
    err = wc_ChaCha20Poly1305_UpdateData(&aead, plaintext1,
        generatedCiphertext, sizeof(plaintext1));
#endif
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_CheckTag(generatedAuthTag, authTag1);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    if (XMEMCMP(generatedCiphertext, cipher1, sizeof(cipher1))) {
        return WC_TEST_RET_ENC_NC;
    }

    /* Test 1 - Decrypt */
    err = wc_ChaCha20Poly1305_Init(&aead, key1, iv1,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad1, sizeof(aad1));
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
#ifdef TEST_SMALL_CHACHA_CHUNKS
    /* test doing data in smaller chunks */
    for (testLen=0; testLen<sizeof(plaintext1); ) {
        word32 dataLen = sizeof(plaintext1) - testLen;
        if (dataLen > TEST_SMALL_CHACHA_CHUNKS)
            dataLen = TEST_SMALL_CHACHA_CHUNKS;
        err = wc_ChaCha20Poly1305_UpdateData(&aead,
            &generatedCiphertext[testLen], &generatedPlaintext[testLen],
            dataLen);
        if (err != 0)
            return WC_TEST_RET_ENC_EC(err);
        testLen += dataLen;
    }
#else
    err = wc_ChaCha20Poly1305_UpdateData(&aead, generatedCiphertext,
        generatedPlaintext, sizeof(cipher1));
#endif
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_CheckTag(generatedAuthTag, authTag1);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    if (XMEMCMP(generatedPlaintext, plaintext1, sizeof(plaintext1))) {
        return WC_TEST_RET_ENC_NC;
    }

    XMEMSET(generatedCiphertext, 0, sizeof(generatedCiphertext));
    XMEMSET(generatedAuthTag, 0, sizeof(generatedAuthTag));
    XMEMSET(generatedPlaintext, 0, sizeof(generatedPlaintext));

    /* Test 2 - Encrypt */
    err = wc_ChaCha20Poly1305_Init(&aead, key2, iv2,
        CHACHA20_POLY1305_AEAD_ENCRYPT);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad2, sizeof(aad2));
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
#ifdef TEST_SMALL_CHACHA_CHUNKS
    /* test doing data in smaller chunks */
    for (testLen=0; testLen<sizeof(plaintext2); ) {
        word32 dataLen = sizeof(plaintext2) - testLen;
        if (dataLen > TEST_SMALL_CHACHA_CHUNKS)
            dataLen = TEST_SMALL_CHACHA_CHUNKS;
        err = wc_ChaCha20Poly1305_UpdateData(&aead, &plaintext2[testLen],
            &generatedCiphertext[testLen], dataLen);
        if (err != 0)
            return WC_TEST_RET_ENC_EC(err);
        testLen += dataLen;
    }
#else
    err = wc_ChaCha20Poly1305_UpdateData(&aead, plaintext2, generatedCiphertext,
        sizeof(plaintext2));
#endif
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_CheckTag(generatedAuthTag, authTag2);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    if (XMEMCMP(generatedCiphertext, cipher2, sizeof(cipher2))) {
        return WC_TEST_RET_ENC_NC;
    }

    /* Test 2 - Decrypt */
    err = wc_ChaCha20Poly1305_Init(&aead, key2, iv2,
        CHACHA20_POLY1305_AEAD_DECRYPT);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_UpdateAad(&aead, aad2, sizeof(aad2));
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
#ifdef TEST_SMALL_CHACHA_CHUNKS
    /* test doing data in smaller chunks */
    for (testLen=0; testLen<sizeof(plaintext2); ) {
        word32 dataLen = sizeof(plaintext2) - testLen;
        if (dataLen > TEST_SMALL_CHACHA_CHUNKS)
            dataLen = TEST_SMALL_CHACHA_CHUNKS;
        err = wc_ChaCha20Poly1305_UpdateData(&aead,
            &generatedCiphertext[testLen], &generatedPlaintext[testLen],
            dataLen);
        if (err != 0)
            return WC_TEST_RET_ENC_EC(err);
        testLen += dataLen;
    }
#else
    err = wc_ChaCha20Poly1305_UpdateData(&aead, generatedCiphertext,
        generatedPlaintext, sizeof(cipher2));
#endif
    err = wc_ChaCha20Poly1305_Final(&aead, generatedAuthTag);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    err = wc_ChaCha20Poly1305_CheckTag(generatedAuthTag, authTag2);
    if (err != 0)
        return WC_TEST_RET_ENC_EC(err);
    if (XMEMCMP(generatedPlaintext, plaintext2, sizeof(plaintext2))) {
        return WC_TEST_RET_ENC_NC;
    }

    return err;
}
#endif /* HAVE_CHACHA && HAVE_POLY1305 */


#ifndef NO_DES3
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t des_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte vector[] = { /* "now is the time for all " w/o trailing 0 */
        0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    byte plain[24];
    byte cipher[24];

    Des enc;
    Des dec;

    WOLFSSL_SMALL_STACK_STATIC const byte key[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
    {
        0x8b,0x7c,0x52,0xb0,0x01,0x2b,0x6c,0xb8,
        0x4f,0x0f,0xeb,0xf3,0xfb,0x5f,0x86,0x73,
        0x15,0x85,0xb3,0x22,0x4b,0x86,0x2b,0x4b
    };

    wc_test_ret_t ret;

    ret = wc_Des_SetKey(&enc, key, iv, DES_ENCRYPTION);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Des_CbcEncrypt(&enc, cipher, vector, sizeof(vector));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Des_SetKey(&dec, key, iv, DES_DECRYPTION);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Des_CbcDecrypt(&dec, plain, cipher, sizeof(cipher));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(plain, vector, sizeof(plain)))
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(cipher, verify, sizeof(cipher)))
        return WC_TEST_RET_ENC_NC;

    ret = wc_Des_CbcEncryptWithKey(cipher, vector, sizeof(vector), key, iv);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

#if defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_SHA)
    {
        EncryptedInfo info;
        XMEMSET(&info, 0, sizeof(EncryptedInfo));
        XMEMCPY(info.iv, iv, sizeof(iv));
        info.ivSz = sizeof(iv);
        info.keySz = sizeof(key);
        info.cipherType = WC_CIPHER_DES;

        ret = wc_BufferKeyEncrypt(&info, cipher, sizeof(cipher), key,
                sizeof(key), WC_HASH_TYPE_SHA);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        /* Test invalid info ptr */
        ret = wc_BufferKeyEncrypt(NULL, cipher, sizeof(cipher), key,
                sizeof(key), WC_HASH_TYPE_SHA);
        if (ret != BAD_FUNC_ARG)
            return WC_TEST_RET_ENC_EC(ret);

    #ifndef NO_PWDBASED
        /* Test invalid hash type - only applies to wc_PBKDF1 call */
        ret = wc_BufferKeyEncrypt(&info, cipher, sizeof(cipher), key,
                sizeof(key), WC_HASH_TYPE_NONE);
        if (ret == 0)
            return WC_TEST_RET_ENC_EC(ret);
    #endif /* !NO_PWDBASED */
    }
#endif

    return 0;
}
#endif /* !NO_DES3 */


#ifndef NO_DES3
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t des3_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte vector[] = { /* "Now is the time for all " w/o trailing 0 */
        0x4e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    byte plain[24];
    byte cipher[24];

    Des3 enc;
    Des3 dec;

    WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
    {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef,
        0xfe,0xde,0xba,0x98,0x76,0x54,0x32,0x10,
        0x89,0xab,0xcd,0xef,0x01,0x23,0x45,0x67
    };
    WOLFSSL_SMALL_STACK_STATIC const byte iv3[] =
    {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef,
        0x01,0x01,0x01,0x01,0x01,0x01,0x01,0x01,
        0x11,0x21,0x31,0x41,0x51,0x61,0x71,0x81

    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify3[] =
    {
        0x43,0xa0,0x29,0x7e,0xd1,0x84,0xf8,0x0e,
        0x89,0x64,0x84,0x32,0x12,0xd5,0x08,0x98,
        0x18,0x94,0x15,0x74,0x87,0x12,0x7d,0xb0
    };

    wc_test_ret_t ret;
#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
    size_t i;
#endif


    ret = wc_Des3Init(&enc, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Des3Init(&dec, HEAP_HINT, devId);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Des3_SetKey(&enc, key3, iv3, DES_ENCRYPTION);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Des3_SetKey(&dec, key3, iv3, DES_DECRYPTION);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Des3_CbcEncrypt(&enc, cipher, vector, sizeof(vector));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Des3_CbcDecrypt(&dec, plain, cipher, sizeof(cipher));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(plain, vector, sizeof(plain)))
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(cipher, verify3, sizeof(cipher)))
        return WC_TEST_RET_ENC_NC;

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
    /* test the same vectors with using compatibility layer */
    for (i = 0; i < sizeof(vector); i += DES_BLOCK_SIZE){
        DES_key_schedule ks1;
        DES_key_schedule ks2;
        DES_key_schedule ks3;
        DES_cblock iv4;
        byte tmp[sizeof(vector)];

        XMEMCPY(ks1, key3, sizeof(DES_key_schedule));
        XMEMCPY(ks2, key3 + 8, sizeof(DES_key_schedule));
        XMEMCPY(ks3, key3 + 16, sizeof(DES_key_schedule));

        XMEMSET(plain, 0, sizeof(plain));
        XMEMSET(cipher, 0, sizeof(cipher));

        /* Test in-place encrypt/decrypt */
        XMEMCPY(tmp, vector, sizeof(vector));

        /* Use i as the splitter */
        XMEMCPY(iv4, iv3, sizeof(DES_cblock));
        DES_ede3_cbc_encrypt(tmp, tmp, (long)i, &ks1, &ks2, &ks3,
                &iv4, DES_ENCRYPT);
        DES_ede3_cbc_encrypt(tmp + i, tmp + i, (long)(sizeof(vector) - i),
                &ks1, &ks2, &ks3, &iv4, DES_ENCRYPT);
        XMEMCPY(cipher, tmp, sizeof(cipher));
        XMEMCPY(iv4, iv3, sizeof(DES_cblock));
        DES_ede3_cbc_encrypt(tmp, tmp, (long)i, &ks1, &ks2, &ks3,
                &iv4, DES_DECRYPT);
        DES_ede3_cbc_encrypt(tmp + i, tmp + i, (long)(sizeof(cipher) - i),
                &ks1, &ks2, &ks3, &iv4, DES_DECRYPT);
        XMEMCPY(plain, tmp, sizeof(plain));

        if (XMEMCMP(plain, vector, sizeof(plain)))
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(cipher, verify3, sizeof(cipher)))
            return WC_TEST_RET_ENC_NC;
    }
#endif /* OPENSSL_EXTRA */

    wc_Des3Free(&enc);
    wc_Des3Free(&dec);

#if defined(WOLFSSL_ENCRYPTED_KEYS) && !defined(NO_SHA)
    {
        EncryptedInfo info;
        XMEMSET(&info, 0, sizeof(EncryptedInfo));
        XMEMCPY(info.iv, iv3, sizeof(iv3));
        info.ivSz = sizeof(iv3);
        info.keySz = sizeof(key3);
        info.cipherType = WC_CIPHER_DES3;

        ret = wc_BufferKeyEncrypt(&info, cipher, sizeof(cipher), key3,
                sizeof(key3), WC_HASH_TYPE_SHA);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
    }
#endif

    return 0;
}
#endif /* NO_DES3 */

static const int fiducial1 = WC_TEST_RET_LN; /* source code reference point --
                                              * see print_fiducials() below.
                                              */

#ifndef NO_AES

#if defined(WOLFSSL_AES_OFB) || defined(WOLFSSL_AES_CFB) || \
    defined(WOLFSSL_AES_XTS)
#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
    && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
/* pass in the function, key, iv, plain text and expected and this function
 * tests that the encryption and decryption is successful */
static wc_test_ret_t EVP_test(const WOLFSSL_EVP_CIPHER* type, const byte* key,
        const byte* iv, const byte* plain, int plainSz,
        const byte* expected, int expectedSz)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    EVP_CIPHER_CTX *ctx = NULL;
#else
    EVP_CIPHER_CTX ctx[1];
#endif
    int ctx_inited = 0;
    int idx, cipherSz;
    wc_test_ret_t ret = 0;
    byte* cipher;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((ctx = wolfSSL_EVP_CIPHER_CTX_new()) == NULL)
        return MEMORY_E;
#endif

    cipher = (byte*)XMALLOC(plainSz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (cipher == NULL) {
        ret = WC_TEST_RET_ENC_ERRNO;
        goto EVP_TEST_END;
    }

    /* test encrypt */
    EVP_CIPHER_CTX_init(ctx);
    ctx_inited = 1;
    if (EVP_CipherInit(ctx, type, key, iv, 1) == 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }

    if (EVP_CipherUpdate(ctx, cipher, &idx, plain, expectedSz) == 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }

    cipherSz = idx;
    if (EVP_CipherFinal(ctx, cipher + cipherSz, &idx) == 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }
    cipherSz += idx;

    if (XMEMCMP(cipher, expected, plainSz)) {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }

    ret = wolfSSL_EVP_CIPHER_CTX_cleanup(ctx);
    ctx_inited = 0;
    if (ret == WOLFSSL_SUCCESS)
        ret = 0;
    else {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }

    /* test decrypt */
    EVP_CIPHER_CTX_init(ctx);
    ctx_inited = 1;
    if (EVP_CipherInit(ctx, type, key, iv, 0) == 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }

    if (EVP_CipherUpdate(ctx, cipher, &idx, cipher, expectedSz) == 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }

    cipherSz = idx;
    if (EVP_CipherFinal(ctx, cipher + cipherSz, &idx) == 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }
    cipherSz += idx;

    if ((expectedSz != cipherSz) || XMEMCMP(plain, cipher, plainSz)) {
        ret = WC_TEST_RET_ENC_NC;
        goto EVP_TEST_END;
    }

EVP_TEST_END:
    if (cipher)
        XFREE(cipher, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    (void)cipherSz;

    if (ctx_inited) {
        int cleanup_ret = wolfSSL_EVP_CIPHER_CTX_cleanup(ctx);
        if (cleanup_ret != WOLFSSL_SUCCESS)
            ret = WC_TEST_RET_ENC_NC;
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif

    return ret;
}
#endif /* OPENSSL_EXTRA */
#endif /* WOLFSSL_AES_OFB || WOLFSSL_AES_CFB */

#ifdef WOLFSSL_AES_OFB
    /* test vector from https://csrc.nist.gov/Projects/cryptographic-algorithm-validation-program/Block-Ciphers */
    WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aesofb_test(void)
    {
    #ifdef WOLFSSL_AES_256
        WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
        {
            0xc4,0xc7,0xfa,0xd6,0x53,0x5c,0xb8,0x71,
            0x4a,0x5c,0x40,0x77,0x9a,0x8b,0xa1,0xd2,
            0x53,0x3e,0x23,0xb4,0xb2,0x58,0x73,0x2a,
            0x5b,0x78,0x01,0xf4,0xe3,0x71,0xa7,0x94
        };
        WOLFSSL_SMALL_STACK_STATIC const byte iv1[] =
        {
            0x5e,0xb9,0x33,0x13,0xb8,0x71,0xff,0x16,
            0xb9,0x8a,0x9b,0xcb,0x43,0x33,0x0d,0x6f
        };
        WOLFSSL_SMALL_STACK_STATIC const byte plain1[] =
        {
            0x6d,0x0b,0xb0,0x79,0x63,0x84,0x71,0xe9,
            0x39,0xd4,0x53,0x14,0x86,0xc1,0x4c,0x25,
            0x9a,0xee,0xc6,0xf3,0xc0,0x0d,0xfd,0xd6,
            0xc0,0x50,0xa8,0xba,0xa8,0x20,0xdb,0x71,
            0xcc,0x12,0x2c,0x4e,0x0c,0x17,0x15,0xef,
            0x55,0xf3,0x99,0x5a,0x6b,0xf0,0x2a,0x4c
        };
        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0x0f,0x54,0x61,0x71,0x59,0xd0,0x3f,0xfc,
            0x1b,0xfa,0xfb,0x60,0x29,0x30,0xd7,0x00,
            0xf4,0xa4,0xa8,0xe6,0xdd,0x93,0x94,0x46,
            0x64,0xd2,0x19,0xc4,0xc5,0x4d,0xde,0x1b,
            0x04,0x53,0xe1,0x73,0xf5,0x18,0x74,0xae,
            0xfd,0x64,0xa2,0xe1,0xe2,0x76,0x13,0xb0
        };
    #endif /* WOLFSSL_AES_256 */


    #ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
        {
            0x10,0xa5,0x88,0x69,0xd7,0x4b,0xe5,0xa3,
            0x74,0xcf,0x86,0x7c,0xfb,0x47,0x38,0x59
        };
        WOLFSSL_SMALL_STACK_STATIC const byte iv2[] =
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        };
        WOLFSSL_SMALL_STACK_STATIC const byte plain2[] =
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        };
        WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] =
        {
            0x6d,0x25,0x1e,0x69,0x44,0xb0,0x51,0xe0,
            0x4e,0xaa,0x6f,0xb4,0xdb,0xf7,0x84,0x65
        };
    #endif /* WOLFSSL_AES_128 */


    #ifdef WOLFSSL_AES_192
        WOLFSSL_SMALL_STACK_STATIC const byte key3[] = {
            0xd0,0x77,0xa0,0x3b,0xd8,0xa3,0x89,0x73,
            0x92,0x8c,0xca,0xfe,0x4a,0x9d,0x2f,0x45,
            0x51,0x30,0xbd,0x0a,0xf5,0xae,0x46,0xa9
        };
        WOLFSSL_SMALL_STACK_STATIC const byte iv3[] =
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        };
        WOLFSSL_SMALL_STACK_STATIC const byte cipher3[] =
        {
            0xab,0xc7,0x86,0xfb,0x1e,0xdb,0x50,0x45,
            0x80,0xc4,0xd8,0x82,0xef,0x29,0xa0,0xc7
        };
        WOLFSSL_SMALL_STACK_STATIC const byte plain3[] =
        {
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
            0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
        };
    #endif /* WOLFSSL_AES_192 */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        Aes *enc = NULL;
#else
        Aes enc[1];
#endif
        byte cipher[AES_BLOCK_SIZE * 4];
    #ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        Aes *dec = NULL;
#else
        Aes dec[1];
#endif
        byte plain [AES_BLOCK_SIZE * 4];
    #endif
        wc_test_ret_t ret = 0;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-1, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(-1, out);
#endif
#endif

        XMEMSET(enc, 0, sizeof *enc);
    #ifdef HAVE_AES_DECRYPT
        XMEMSET(dec, 0, sizeof *dec);
    #endif

#ifdef WOLFSSL_AES_128
        /* 128 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_128_ofb(), key2, iv2, plain2, sizeof(plain2),
                cipher2, sizeof(cipher2));
        if (ret != 0) {
            goto out;
        }
    #endif

        ret = wc_AesInit(enc, HEAP_HINT, INVALID_DEVID);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        ret = wc_AesInit(dec, HEAP_HINT, INVALID_DEVID);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain2, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher2, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher2, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, plain2, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
        /* 192 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_192_ofb(), key3, iv3, plain3, sizeof(plain3),
                cipher3, sizeof(cipher3));
        if (ret != 0) {
            goto out;
        }
    #endif

        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain3, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher3, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher3, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, plain3, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_256_ofb(), key1, iv1, plain1, sizeof(plain1),
                cipher1, sizeof(cipher1));
        if (ret != 0) {
            goto out;
        }
    #endif

        ret = wc_AesSetKey(enc, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain1, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesOfbEncrypt(enc, cipher + AES_BLOCK_SIZE,
                plain1 + AES_BLOCK_SIZE, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher + AES_BLOCK_SIZE, cipher1 + AES_BLOCK_SIZE,
                    AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher1, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, plain1, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesOfbDecrypt(dec, plain + AES_BLOCK_SIZE,
                cipher1 + AES_BLOCK_SIZE, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain + AES_BLOCK_SIZE, plain1 + AES_BLOCK_SIZE,
                    AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */

        /* multiple blocks at once */
        ret = wc_AesSetKey(enc, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain1, AES_BLOCK_SIZE * 3);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE * 3))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher1, AES_BLOCK_SIZE * 3);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, plain1, AES_BLOCK_SIZE * 3))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */

        /* inline decrypt/encrypt*/
        ret = wc_AesSetKey(enc, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMCPY(cipher, plain1, AES_BLOCK_SIZE * 2);
        ret = wc_AesOfbEncrypt(enc, cipher, cipher, AES_BLOCK_SIZE * 2);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE * 2))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, cipher, cipher, AES_BLOCK_SIZE * 2);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, plain1, AES_BLOCK_SIZE * 2))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */

        /* 256 key size test leftover support */
        ret = wc_AesSetKey(enc, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, sizeof(key1), iv1, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesOfbEncrypt(enc, cipher, plain1, 3);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher1, 3))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesOfbEncrypt(enc, cipher + 3, plain1 + 3, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher + 3, cipher1 + 3, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesOfbDecrypt(dec, plain, cipher1, 6);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, plain1, 6))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesOfbDecrypt(dec, plain + 6, cipher1 + 6, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain + 6, plain1 + 6, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */

  out:

        wc_AesFree(enc);
        wc_AesFree(dec);
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif

#endif /* WOLFSSL_AES_256 */

        return ret;
    }
#endif /* WOLFSSL_AES_OFB */

#if defined(WOLFSSL_AES_CFB)
    /* Test cases from NIST SP 800-38A, Recommendation for Block Cipher Modes of Operation Methods an*/
    static wc_test_ret_t aescfb_test(void)
    {
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        Aes *enc = NULL;
#else
        Aes enc[1];
#endif
        int enc_inited = 0;
        byte cipher[AES_BLOCK_SIZE * 4];
    #ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        Aes *dec = NULL;
#else
        Aes dec[1];
#endif
        int dec_inited = 0;
        byte plain [AES_BLOCK_SIZE * 4];
    #endif
        wc_test_ret_t ret = 0;

        WOLFSSL_SMALL_STACK_STATIC const byte iv[] = {
            0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
            0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f
        };

#ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
        {
            0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
            0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0x3b,0x3f,0xd9,0x2e,0xb7,0x2d,0xad,0x20,
            0x33,0x34,0x49,0xf8,0xe8,0x3c,0xfb,0x4a,
            0xc8,0xa6,0x45,0x37,0xa0,0xb3,0xa9,0x3f,
            0xcd,0xe3,0xcd,0xad,0x9f,0x1c,0xe5,0x8b,
            0x26,0x75,0x1f,0x67,0xa3,0xcb,0xb1,0x40,
            0xb1,0x80,0x8c,0xf1,0x87,0xa4,0xf4,0xdf
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg1[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef
        };
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
        /* 192 size key test */
        WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
        {
            0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
            0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
            0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] =
        {
            0xcd,0xc8,0x0d,0x6f,0xdd,0xf1,0x8c,0xab,
            0x34,0xc2,0x59,0x09,0xc9,0x9a,0x41,0x74,
            0x67,0xce,0x7f,0x7f,0x81,0x17,0x36,0x21,
            0x96,0x1a,0x2b,0x70,0x17,0x1d,0x3d,0x7a,
            0x2e,0x1e,0x8a,0x1d,0xd5,0x9b,0x88,0xb1,
            0xc8,0xe6,0x0f,0xed,0x1e,0xfa,0xc4,0xc9,
            0xc0,0x5f,0x9f,0x9c,0xa9,0x83,0x4f,0xa0,
            0x42,0xae,0x8f,0xba,0x58,0x4b,0x09,0xff
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 size key simple test */
        WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher3[] =
        {
            0xdc,0x7e,0x84,0xbf,0xda,0x79,0x16,0x4b,
            0x7e,0xcd,0x84,0x86,0x98,0x5d,0x38,0x60,
            0x39,0xff,0xed,0x14,0x3b,0x28,0xb1,0xc8,
            0x32,0x11,0x3c,0x63,0x31,0xe5,0x40,0x7b,
            0xdf,0x10,0x13,0x24,0x15,0xe5,0x4b,0x92,
            0xa1,0x3e,0xd0,0xa8,0x26,0x7a,0xe2,0xf9,
            0x75,0xa3,0x85,0x74,0x1a,0xb9,0xce,0xf8,
            0x20,0x31,0x62,0x3d,0x55,0xb1,0xe4,0x71
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg3[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };
#endif /* WOLFSSL_AES_256 */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif
#endif

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        enc_inited = 1;
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesInit(dec, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        dec_inited = 1;
#endif

#ifdef WOLFSSL_AES_128
        /* 128 key tests */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_128_cfb128(), key1, iv, msg1, sizeof(msg1),
                cipher1, sizeof(cipher1));
        if (ret != 0) {
            return ret;
        }
    #endif

        ret = wc_AesSetKey(enc, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfbEncrypt(enc, cipher, msg1, AES_BLOCK_SIZE * 2);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE * 2))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        /* test restarting encryption process */
        ret = wc_AesCfbEncrypt(enc, cipher + (AES_BLOCK_SIZE * 2),
                msg1 + (AES_BLOCK_SIZE * 2), AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher + (AES_BLOCK_SIZE * 2),
                    cipher1 + (AES_BLOCK_SIZE * 2), AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfbDecrypt(dec, plain, cipher, AES_BLOCK_SIZE * 3);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, msg1, AES_BLOCK_SIZE * 3))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
        /* 192 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_192_cfb128(), key2, iv, msg2, sizeof(msg2),
                cipher2, sizeof(cipher2));
        if (ret != 0) {
            return ret;
        }
    #endif

        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key2, sizeof(key2), iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfbEncrypt(enc, cipher, msg2, AES_BLOCK_SIZE * 4);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher2, AES_BLOCK_SIZE * 4))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfbDecrypt(dec, plain, cipher, AES_BLOCK_SIZE * 4);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, msg2, AES_BLOCK_SIZE * 4))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 key size test */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
        && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
        ret = EVP_test(EVP_aes_256_cfb128(), key3, iv, msg3, sizeof(msg3),
                cipher3, sizeof(cipher3));
        if (ret != 0) {
            return ret;
        }
    #endif
        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key3, sizeof(key3), iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        /* test with data left overs, magic lengths are checking near edges */
        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfbEncrypt(enc, cipher, msg3, 4);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher3, 4))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesCfbEncrypt(enc, cipher + 4, msg3 + 4, 27);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher + 4, cipher3 + 4, 27))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesCfbEncrypt(enc, cipher + 31, msg3 + 31,
                (AES_BLOCK_SIZE * 4) - 31);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher3, AES_BLOCK_SIZE * 4))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfbDecrypt(dec, plain, cipher, 4);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, msg3, 4))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesCfbDecrypt(dec, plain + 4, cipher + 4, 4);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        ret = wc_AesCfbDecrypt(dec, plain + 8, cipher + 8, 23);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain + 4, msg3 + 4, 27))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesCfbDecrypt(dec, plain + 31, cipher + 31,
                (AES_BLOCK_SIZE * 4) - 31);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, msg3, AES_BLOCK_SIZE * 4))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_256 */

  out:

    if (enc_inited)
        wc_AesFree(enc);
    if (dec_inited)
        wc_AesFree(dec);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif

        return ret;
    }

#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    static wc_test_ret_t aescfb1_test(void)
    {
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        Aes *enc = NULL;
#else
        Aes enc[1];
#endif
        int enc_inited = 0;
        byte cipher[AES_BLOCK_SIZE];
    #ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        Aes *dec = NULL;
#else
        Aes dec[1];
#endif
        int dec_inited = 0;
        byte plain [AES_BLOCK_SIZE];
    #endif
        wc_test_ret_t ret = 0;

#ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte iv[] = {
            0x4d,0xbb,0xdc,0xaa,0x59,0xf3,0x63,0xc9,
            0x2a,0x3b,0x98,0x43,0xad,0x20,0xe2,0xb7
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
        {
            0xcd,0xef,0x9d,0x06,0x61,0xba,0xe4,0x73,
            0x8d,0x1a,0x58,0xa2,0xa6,0x22,0x8b,0x66
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0x00
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg1[] =
        {
            0xC0
        };
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_192
        WOLFSSL_SMALL_STACK_STATIC const byte iv2[] = {
            0x57,0xc6,0x89,0x7c,0x99,0x52,0x28,0x13,
            0xbf,0x67,0x9c,0xe1,0x13,0x70,0xaf,0x5e
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
        {
            0xba,0xa1,0x58,0xa1,0x6b,0x50,0x4a,0x10,
            0x8e,0xd4,0x33,0x2e,0xe7,0xf2,0x9b,0xf6,
            0xd1,0xac,0x46,0xa8,0xde,0x5a,0xfe,0x7a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] =
        {
            0x30
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
        {
            0x80
        };
#endif /* WOLFSSL_AES_192 */
#ifdef WOLFSSL_AES_256
        WOLFSSL_SMALL_STACK_STATIC const byte iv3[] = {
            0x63,0x2e,0x9f,0x83,0x1f,0xa3,0x80,0x5e,
            0x52,0x02,0xbc,0xe0,0x6d,0x04,0xf9,0xa0
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
        {
            0xf6,0xfa,0xe4,0xf1,0x5d,0x91,0xfc,0x50,
            0x88,0x78,0x4f,0x84,0xa5,0x37,0x12,0x7e,
            0x32,0x63,0x55,0x9c,0x62,0x73,0x88,0x20,
            0xc2,0xcf,0x3d,0xe1,0x1c,0x2a,0x30,0x40
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher3[] =
        {
            0xF7, 0x00
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg3[] =
        {
            0x41, 0xC0
        };
#endif /* WOLFSSL_AES_256 */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif
#endif

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        enc_inited = 1;
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesInit(dec, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        dec_inited = 1;
#endif

#ifdef WOLFSSL_AES_128
        /* 128 key tests */
        ret = wc_AesSetKey(enc, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg1, 2);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (cipher[0] != cipher1[0])
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfb1Decrypt(dec, plain, cipher, 2);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (plain[0] != msg1[0])
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */

    #ifdef OPENSSL_EXTRA
        ret = wc_AesSetKey(enc, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg1,
                sizeof(msg1) * WOLFSSL_BIT_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    #ifndef WOLFCRYPT_ONLY
        ret = EVP_test(EVP_aes_128_cfb1(), key1, iv, msg1, sizeof(msg1),
                cipher, sizeof(msg1));
        if (ret != 0) {
            goto out;
        }
    #endif
    #endif
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_192
        /* 192 key tests */
        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg2, 4);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(cipher, cipher2, sizeof(cipher2)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef OPENSSL_EXTRA
        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg2,
                sizeof(msg2) * WOLFSSL_BIT_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        #ifndef WOLFCRYPT_ONLY
        ret = EVP_test(EVP_aes_192_cfb1(), key2, iv2, msg2, sizeof(msg2),
                cipher, sizeof(msg2));
        if (ret != 0) {
            goto out;
        }
        #endif
    #endif
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 key tests */
        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg3, 10);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(cipher, cipher3, sizeof(cipher3)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef OPENSSL_EXTRA
        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb1Encrypt(enc, cipher, msg3,
                sizeof(msg3) * WOLFSSL_BIT_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        #ifndef WOLFCRYPT_ONLY
        ret = EVP_test(EVP_aes_256_cfb1(), key3, iv3, msg3, sizeof(msg3),
                cipher, sizeof(msg3));
        if (ret != 0) {
            goto out;
        }
        #endif
    #endif
#endif /* WOLFSSL_AES_256 */

  out:

    if (enc_inited)
        wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    if (dec_inited)
        wc_AesFree(dec);
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif

        return ret;
    }

    static wc_test_ret_t aescfb8_test(void)
    {
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        Aes *enc = NULL;
#else
        Aes enc[1];
#endif
        int enc_inited = 0;
        byte cipher[AES_BLOCK_SIZE];
    #ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        Aes *dec = NULL;
#else
        Aes dec[1];
#endif
        int dec_inited = 0;
        byte plain [AES_BLOCK_SIZE];
    #endif
        wc_test_ret_t ret = 0;

#ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte iv[] = {
            0xf4,0x75,0xc6,0x49,0x91,0xb2,0x0e,0xae,
            0xe1,0x83,0xa2,0x26,0x29,0xe2,0x1e,0x22
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key1[] =
        {
            0xc8,0xfe,0x9b,0xf7,0x7b,0x93,0x0f,0x46,
            0xd2,0x07,0x8b,0x8c,0x0e,0x65,0x7c,0xd4
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0xd2,0x76,0x91
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg1[] =
        {
            0xc9,0x06,0x35
        };
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_192
        WOLFSSL_SMALL_STACK_STATIC const byte iv2[] = {
            0x0a,0x02,0x84,0x6b,0x62,0xab,0xb6,0x93,
            0xef,0x31,0xd7,0x54,0x84,0x2e,0xed,0x29
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key2[] =
        {
            0xba,0xf0,0x8b,0x76,0x31,0x7a,0x65,0xc5,
            0xf0,0x7a,0xe6,0xf5,0x7e,0xb0,0xe6,0x54,
            0x88,0x65,0x93,0x24,0xd2,0x97,0x09,0xe3
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher2[] =
        {
            0x72,0x9c,0x0b,0x6d,0xeb,0x75,0xfa,0x6e,
            0xb5,0xe8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
        {
            0x98,0x95,0x93,0x24,0x02,0x39,0x3d,0xc3,
            0x3a,0x60
        };
#endif
#ifdef WOLFSSL_AES_256
        WOLFSSL_SMALL_STACK_STATIC const byte iv3[] = {
            0x33,0x8c,0x55,0x2f,0xf1,0xec,0xa1,0x44,
            0x08,0xe0,0x5d,0x8c,0xf9,0xf3,0xb3,0x1b
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key3[] =
        {
            0x06,0x48,0x74,0x09,0x2f,0x7a,0x13,0xcc,
            0x44,0x62,0x24,0x7a,0xd4,0x23,0xd0,0xe9,
            0x6e,0xdf,0x42,0xe8,0xb6,0x7a,0x5a,0x23,
            0xb7,0xa0,0xa6,0x47,0x7b,0x09,0x8e,0x66
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher3[] =
        {
            0x1c,0xff,0x95
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg3[] =
        {
            0xb9,0x74,0xfa
        };
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif
#endif

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        enc_inited = 1;
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesInit(dec, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        dec_inited = 1;
#endif

#ifdef WOLFSSL_AES_128
        /* 128 key tests */
    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
        ret = EVP_test(EVP_aes_128_cfb8(), key1, iv, msg1, sizeof(msg1),
                cipher1, sizeof(cipher1));
        if (ret != 0) {
            return ret;
        }
    #endif
        ret = wc_AesSetKey(enc, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifdef HAVE_AES_DECRYPT
        /* decrypt uses AES_ENCRYPTION */
        ret = wc_AesSetKey(dec, key1, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #endif

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb8Encrypt(enc, cipher, msg1, sizeof(msg1));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(cipher, cipher1, sizeof(cipher1)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #ifdef HAVE_AES_DECRYPT
        ret = wc_AesCfb8Decrypt(dec, plain, cipher, sizeof(msg1));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        if (XMEMCMP(plain, msg1, sizeof(msg1)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_192
        /* 192 key tests */
        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb8Encrypt(enc, cipher, msg2, sizeof(msg2));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(cipher, cipher2, sizeof(msg2)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
        ret = EVP_test(EVP_aes_192_cfb8(), key2, iv2, msg2, sizeof(msg2),
                cipher2, sizeof(msg2));
        if (ret != 0) {
            return ret;
        }
#endif

#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* 256 key tests */
        ret = wc_AesSetKey(enc, key3, sizeof(key3), iv3, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(cipher, 0, sizeof(cipher));
        ret = wc_AesCfb8Encrypt(enc, cipher, msg3, sizeof(msg3));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(cipher, cipher3, sizeof(cipher3)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    #if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)
        ret = EVP_test(EVP_aes_256_cfb8(), key3, iv3, msg3, sizeof(msg3),
                cipher3, sizeof(msg3));
        if (ret != 0) {
            goto out;
        }
    #endif
#endif /* WOLFSSL_AES_256 */

      out:

        if (enc_inited)
            wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
        if (dec_inited)
            wc_AesFree(dec);
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif

        return ret;
    }
#endif /* !HAVE_SELFTEST && !HAVE_FIPS */
#endif /* WOLFSSL_AES_CFB */


static wc_test_ret_t aes_key_size_test(void)
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes    *aes;
#else
    Aes    aes[1];
#endif
    byte   key16[] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                       0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66 };
#ifndef WOLFSSL_CRYPTOCELL
    byte   key24[] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                       0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37 };
#endif
    byte   key32[] = { 0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                       0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66,
                       0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
                       0x38, 0x39, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66 };
    byte   iv[]    = "1234567890abcdef";
#ifndef HAVE_FIPS
    word32 keySize;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((aes = (Aes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        return WC_TEST_RET_ENC_ERRNO;
#endif

#if !defined(HAVE_FIPS) || \
    defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2)
    /* w/ FIPS v1 (cert 2425) wc_AesInit just returns 0 always as it's not
     * supported with that FIPS version */
    ret = wc_AesInit(NULL, HEAP_HINT, devId);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

    ret = wc_AesInit(aes, HEAP_HINT, devId);
    /* 0 check OK for FIPSv1 */
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

#ifndef HAVE_FIPS
    /* Parameter Validation testing. */
    ret = wc_AesGetKeySize(NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGetKeySize(aes, NULL);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGetKeySize(NULL, &keySize);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    /* Crashes in FIPS */
    ret = wc_AesSetKey(NULL, key16, sizeof(key16), iv, AES_ENCRYPTION);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif
    /* NULL IV indicates to use all zeros IV. */
    ret = wc_AesSetKey(aes, key16, sizeof(key16), NULL, AES_ENCRYPTION);
#ifdef WOLFSSL_AES_128
    if (ret != 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesSetKey(aes, key32, sizeof(key32) - 1, iv, AES_ENCRYPTION);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
/* CryptoCell handles rounds internally */
#if !defined(HAVE_FIPS) && !defined(WOLFSSL_CRYPTOCELL)
    /* PSA don't use aes->rounds */
#if !defined(WOLFSSL_HAVE_PSA) || defined(WOLFSSL_PSA_NO_AES)
    /* Force invalid rounds */
    aes->rounds = 16;
    ret = wc_AesGetKeySize(aes, &keySize);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif
#endif

    ret = wc_AesSetKey(aes, key16, sizeof(key16), iv, AES_ENCRYPTION);
#ifdef WOLFSSL_AES_128
    if (ret != 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#if !defined(HAVE_FIPS) && defined(WOLFSSL_AES_128)
    ret = wc_AesGetKeySize(aes, &keySize);
    if (ret != 0 || keySize != sizeof(key16))
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif
#ifndef WOLFSSL_CRYPTOCELL
/* Cryptocell only supports AES-128 key size */
    ret = wc_AesSetKey(aes, key24, sizeof(key24), iv, AES_ENCRYPTION);
#ifdef WOLFSSL_AES_192
    if (ret != 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#if !defined(HAVE_FIPS) && defined(WOLFSSL_AES_192)
    ret = wc_AesGetKeySize(aes, &keySize);
    if (ret != 0 || keySize != sizeof(key24))
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

    ret = wc_AesSetKey(aes, key32, sizeof(key32), iv, AES_ENCRYPTION);
#ifdef WOLFSSL_AES_256
    if (ret != 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#if !defined(HAVE_FIPS) && defined(WOLFSSL_AES_256)
    ret = wc_AesGetKeySize(aes, &keySize);
    if (ret != 0 || keySize != sizeof(key32))
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif
#endif /* !WOLFSSL_CRYPTOCELL */
    ret = 0; /* success */
  out:

    wc_AesFree(aes);
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}

#if defined(WOLFSSL_AES_XTS) && (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5,3))

/* test vectors from http://csrc.nist.gov/groups/STM/cavp/block-cipher-modes.html */
#ifdef WOLFSSL_AES_128
static wc_test_ret_t aes_xts_128_test(void)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XtsAes *aes = NULL;
#else
    XtsAes aes[1];
#endif
    int aes_inited = 0;
    wc_test_ret_t ret = 0;
    unsigned char buf[AES_BLOCK_SIZE * 2 + 8];
    unsigned char cipher[AES_BLOCK_SIZE * 2 + 8];

    /* 128 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k1[] = {
        0xa1, 0xb9, 0x0c, 0xba, 0x3f, 0x06, 0xac, 0x35,
        0x3b, 0x2c, 0x34, 0x38, 0x76, 0x08, 0x17, 0x62,
        0x09, 0x09, 0x23, 0x02, 0x6e, 0x91, 0x77, 0x18,
        0x15, 0xf2, 0x9d, 0xab, 0x01, 0x93, 0x2f, 0x2f
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char i1[] = {
        0x4f, 0xae, 0xf7, 0x11, 0x7c, 0xda, 0x59, 0xc6,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p1[] = {
        0xeb, 0xab, 0xce, 0x95, 0xb1, 0x4d, 0x3c, 0x8d,
        0x6f, 0xb3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1c
    };

    /* plain text test of partial block is not from NIST test vector list */
    WOLFSSL_SMALL_STACK_STATIC unsigned char pp[] = {
        0xeb, 0xab, 0xce, 0x95, 0xb1, 0x4d, 0x3c, 0x8d,
        0x6f, 0xb3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1c,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c1[] = {
        0x77, 0x8a, 0xe8, 0xb4, 0x3c, 0xb9, 0x8d, 0x5a,
        0x82, 0x50, 0x81, 0xd5, 0xbe, 0x47, 0x1c, 0x63
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char k2[] = {
        0x39, 0x25, 0x79, 0x05, 0xdf, 0xcc, 0x77, 0x76,
        0x6c, 0x87, 0x0a, 0x80, 0x6a, 0x60, 0xe3, 0xc0,
        0x93, 0xd1, 0x2a, 0xcf, 0xcb, 0x51, 0x42, 0xfa,
        0x09, 0x69, 0x89, 0x62, 0x5b, 0x60, 0xdb, 0x16
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char i2[] = {
        0x5c, 0xf7, 0x9d, 0xb6, 0xc5, 0xcd, 0x99, 0x1a,
        0x1c, 0x78, 0x81, 0x42, 0x24, 0x95, 0x1e, 0x84
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p2[] = {
        0xbd, 0xc5, 0x46, 0x8f, 0xbc, 0x8d, 0x50, 0xa1,
        0x0d, 0x1c, 0x85, 0x7f, 0x79, 0x1c, 0x5c, 0xba,
        0xb3, 0x81, 0x0d, 0x0d, 0x73, 0xcf, 0x8f, 0x20,
        0x46, 0xb1, 0xd1, 0x9e, 0x7d, 0x5d, 0x8a, 0x56
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c2[] = {
        0xd6, 0xbe, 0x04, 0x6d, 0x41, 0xf2, 0x3b, 0x5e,
        0xd7, 0x0b, 0x6b, 0x3d, 0x5c, 0x8e, 0x66, 0x23,
        0x2b, 0xe6, 0xb8, 0x07, 0xd4, 0xdc, 0xc6, 0x0e,
        0xff, 0x8d, 0xbc, 0x1d, 0x9f, 0x7f, 0xc8, 0x22
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char cp2[] = {
        0x2b, 0xf7, 0x2c, 0xf3, 0xeb, 0x85, 0xef, 0x7b,
        0x0b, 0x76, 0xa0, 0xaa, 0xf3, 0x3f, 0x25, 0x8b,
        0x77, 0x8a, 0xe8, 0xb4, 0x3c, 0xb9, 0x8d, 0x5a
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char k3[] = {
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    };
    WOLFSSL_SMALL_STACK_STATIC unsigned char i3[] = {
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
    };
    WOLFSSL_SMALL_STACK_STATIC unsigned char p3[] = {
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0xff, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
        0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20
    };
    WOLFSSL_SMALL_STACK_STATIC unsigned char c3[] = {
        0xA2, 0x07, 0x47, 0x76, 0x3F, 0xEC, 0x0C, 0x23,
        0x1B, 0xD0, 0xBD, 0x46, 0x9A, 0x27, 0x38, 0x12,
        0x95, 0x02, 0x3D, 0x5D, 0xC6, 0x94, 0x51, 0x36,
        0xA0, 0x85, 0xD2, 0x69, 0x6E, 0x87, 0x0A, 0xBF,
        0xB5, 0x5A, 0xDD, 0xCB, 0x80, 0xE0, 0xFC, 0xCD
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((aes = (XtsAes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
    && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    ret = EVP_test(EVP_aes_128_xts(), k2, i2, p2, sizeof(p2), c2, sizeof(c2));
    if (ret != 0) {
        printf("EVP_aes_128_xts failed!\n");
        goto out;
    }
#endif

    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsInit(aes, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        aes_inited = 1;

    ret = wc_AesXtsSetKeyNoInit(aes, k2, sizeof(k2), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesXtsEncrypt(aes, buf, p2, sizeof(p2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c2, buf, sizeof(c2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    ret = wc_AesXtsEncrypt(aes, buf, p2, sizeof(p2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c2, buf, sizeof(c2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

    XMEMSET(buf, 0, sizeof(buf));

    ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsEncrypt(aes, buf, p1, sizeof(p1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    ret = wc_AesXtsEncrypt(aes, buf, p1, sizeof(p1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

    /* partial block encryption test */
    XMEMSET(cipher, 0, sizeof(cipher));
    ret = wc_AesXtsEncrypt(aes, cipher, pp, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(cp2, cipher, sizeof(cp2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    XMEMSET(cipher, 0, sizeof(cipher));
    ret = wc_AesXtsEncrypt(aes, cipher, pp, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(cp2, cipher, sizeof(cp2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

    /* partial block decrypt test */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecrypt(aes, buf, cipher, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(pp, buf, sizeof(pp)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsDecrypt(aes, buf, cipher, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(pp, buf, sizeof(pp)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

    /* NIST decrypt test vector */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsDecrypt(aes, buf, c1, sizeof(c1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsDecrypt(aes, buf, c1, sizeof(c1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

    /* fail case with decrypting using wrong key */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsDecrypt(aes, buf, c2, sizeof(c2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p2, buf, sizeof(p2)) == 0) /* fail case with wrong key */
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* set correct key and retest */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k2, sizeof(k2), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecrypt(aes, buf, c2, sizeof(c2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p2, buf, sizeof(p2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* Test ciphertext stealing in-place. */
    XMEMCPY(buf, p3, sizeof(p3));
    ret = wc_AesXtsSetKeyNoInit(aes, k3, sizeof(k3), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesXtsEncrypt(aes, buf, buf, sizeof(p3), i3, sizeof(i3));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c3, buf, sizeof(c3)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = wc_AesXtsSetKeyNoInit(aes, k3, sizeof(k3), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecrypt(aes, buf, buf, sizeof(c3), i3, sizeof(i3));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p3, buf, sizeof(p3)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if !defined(BENCH_EMBEDDED) && !defined(HAVE_CAVIUM) && \
    !defined(WOLFSSL_AFALG)
    {
    #define LARGE_XTS_SZ        1024
    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        byte* large_input = (byte *)XMALLOC(LARGE_XTS_SZ, HEAP_HINT,
            DYNAMIC_TYPE_TMP_BUFFER);
    #else
        byte large_input[LARGE_XTS_SZ];
    #endif
        int i;
        int j;
    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if (large_input == NULL)
            ERROR_OUT(WC_TEST_RET_ENC_EC(MEMORY_E), out);
    #endif

        for (i = 0; i < (int)LARGE_XTS_SZ; i++)
            large_input[i] = (byte)i;

        for (j = 16; j < (int)LARGE_XTS_SZ; j++) {
            ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_ENCRYPTION);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
            ret = wc_AesXtsEncrypt(aes, large_input, large_input, j, i1,
                sizeof(i1));
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

            ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_DECRYPTION);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
            ret = wc_AesXtsDecrypt(aes, large_input, large_input, j, i1,
                sizeof(i1));
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
        #endif
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
            for (i = 0; i < j; i++) {
                if (large_input[i] != (byte)i) {
                    ERROR_OUT(WC_TEST_RET_ENC_NC, out);
                }
            }
        }
    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        XFREE(large_input, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
    }
#endif /* !BENCH_EMBEDDED && !HAVE_CAVIUM &&
        * !WOLFSSL_AFALG
        */

  out:

    if (aes_inited)
        wc_AesXtsFree(aes);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (aes)
        XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 */


#ifdef WOLFSSL_AES_256
static wc_test_ret_t aes_xts_256_test(void)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XtsAes *aes = NULL;
#else
    XtsAes aes[1];
#endif
    int aes_inited = 0;
    wc_test_ret_t ret = 0;
    unsigned char buf[AES_BLOCK_SIZE * 3];
    unsigned char cipher[AES_BLOCK_SIZE * 3];

    /* 256 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k1[] = {
        0x1e, 0xa6, 0x61, 0xc5, 0x8d, 0x94, 0x3a, 0x0e,
        0x48, 0x01, 0xe4, 0x2f, 0x4b, 0x09, 0x47, 0x14,
        0x9e, 0x7f, 0x9f, 0x8e, 0x3e, 0x68, 0xd0, 0xc7,
        0x50, 0x52, 0x10, 0xbd, 0x31, 0x1a, 0x0e, 0x7c,
        0xd6, 0xe1, 0x3f, 0xfd, 0xf2, 0x41, 0x8d, 0x8d,
        0x19, 0x11, 0xc0, 0x04, 0xcd, 0xa5, 0x8d, 0xa3,
        0xd6, 0x19, 0xb7, 0xe2, 0xb9, 0x14, 0x1e, 0x58,
        0x31, 0x8e, 0xea, 0x39, 0x2c, 0xf4, 0x1b, 0x08
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char i1[] = {
        0xad, 0xf8, 0xd9, 0x26, 0x27, 0x46, 0x4a, 0xd2,
        0xf0, 0x42, 0x8e, 0x84, 0xa9, 0xf8, 0x75, 0x64
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p1[] = {
        0x2e, 0xed, 0xea, 0x52, 0xcd, 0x82, 0x15, 0xe1,
        0xac, 0xc6, 0x47, 0xe8, 0x10, 0xbb, 0xc3, 0x64,
        0x2e, 0x87, 0x28, 0x7f, 0x8d, 0x2e, 0x57, 0xe3,
        0x6c, 0x0a, 0x24, 0xfb, 0xc1, 0x2a, 0x20, 0x2e
    };

    /* plain text test of partial block is not from NIST test vector list */
    WOLFSSL_SMALL_STACK_STATIC unsigned char pp[] = {
        0xeb, 0xab, 0xce, 0x95, 0xb1, 0x4d, 0x3c, 0x8d,
        0x6f, 0xb3, 0x50, 0x39, 0x07, 0x90, 0x31, 0x1c,
        0x6e, 0x4b, 0x92, 0x01, 0x3e, 0x76, 0x8a, 0xd5
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c1[] = {
        0xcb, 0xaa, 0xd0, 0xe2, 0xf6, 0xce, 0xa3, 0xf5,
        0x0b, 0x37, 0xf9, 0x34, 0xd4, 0x6a, 0x9b, 0x13,
        0x0b, 0x9d, 0x54, 0xf0, 0x7e, 0x34, 0xf3, 0x6a,
        0xf7, 0x93, 0xe8, 0x6f, 0x73, 0xc6, 0xd7, 0xdb
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char k2[] = {
        0xad, 0x50, 0x4b, 0x85, 0xd7, 0x51, 0xbf, 0xba,
        0x69, 0x13, 0xb4, 0xcc, 0x79, 0xb6, 0x5a, 0x62,
        0xf7, 0xf3, 0x9d, 0x36, 0x0f, 0x35, 0xb5, 0xec,
        0x4a, 0x7e, 0x95, 0xbd, 0x9b, 0xa5, 0xf2, 0xec,
        0xc1, 0xd7, 0x7e, 0xa3, 0xc3, 0x74, 0xbd, 0x4b,
        0x13, 0x1b, 0x07, 0x83, 0x87, 0xdd, 0x55, 0x5a,
        0xb5, 0xb0, 0xc7, 0xe5, 0x2d, 0xb5, 0x06, 0x12,
        0xd2, 0xb5, 0x3a, 0xcb, 0x47, 0x8a, 0x53, 0xb4
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char i2[] = {
        0xe6, 0x42, 0x19, 0xed, 0xe0, 0xe1, 0xc2, 0xa0,
        0x0e, 0xf5, 0x58, 0x6a, 0xc4, 0x9b, 0xeb, 0x6f
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p2[] = {
        0x24, 0xcb, 0x76, 0x22, 0x55, 0xb5, 0xa8, 0x00,
        0xf4, 0x6e, 0x80, 0x60, 0x56, 0x9e, 0x05, 0x53,
        0xbc, 0xfe, 0x86, 0x55, 0x3b, 0xca, 0xd5, 0x89,
        0xc7, 0x54, 0x1a, 0x73, 0xac, 0xc3, 0x9a, 0xbd,
        0x53, 0xc4, 0x07, 0x76, 0xd8, 0xe8, 0x22, 0x61,
        0x9e, 0xa9, 0xad, 0x77, 0xa0, 0x13, 0x4c, 0xfc
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c2[] = {
        0xa3, 0xc6, 0xf3, 0xf3, 0x82, 0x79, 0x5b, 0x10,
        0x87, 0xd7, 0x02, 0x50, 0xdb, 0x2c, 0xd3, 0xb1,
        0xa1, 0x62, 0xa8, 0xb6, 0xdc, 0x12, 0x60, 0x61,
        0xc1, 0x0a, 0x84, 0xa5, 0x85, 0x3f, 0x3a, 0x89,
        0xe6, 0x6c, 0xdb, 0xb7, 0x9a, 0xb4, 0x28, 0x9b,
        0xc3, 0xea, 0xd8, 0x10, 0xe9, 0xc0, 0xaf, 0x92
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((aes = (XtsAes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY) \
    && !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    ret = EVP_test(EVP_aes_256_xts(), k2, i2, p2, sizeof(p2), c2, sizeof(c2));
    if (ret != 0) {
        printf("EVP_aes_256_xts failed\n");
        goto out;
    }
#endif

    ret = wc_AesXtsInit(aes, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        aes_inited = 1;

    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k2, sizeof(k2), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesXtsEncrypt(aes, buf, p2, sizeof(p2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c2, buf, sizeof(c2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsEncrypt(aes, buf, p1, sizeof(p1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* partial block encryption test */
    XMEMSET(cipher, 0, sizeof(cipher));
    ret = wc_AesXtsEncrypt(aes, cipher, pp, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    /* partial block decrypt test */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecrypt(aes, buf, cipher, sizeof(pp), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(pp, buf, sizeof(pp)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* NIST decrypt test vector */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsDecrypt(aes, buf, c1, sizeof(c1), i1, sizeof(i1));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k2, sizeof(k2), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecrypt(aes, buf, c2, sizeof(c2), i2, sizeof(i2));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p2, buf, sizeof(p2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

  out:

    if (aes_inited)
        wc_AesXtsFree(aes);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (aes)
        XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_256 */


#if defined(WOLFSSL_AES_128) && defined(WOLFSSL_AES_256)
/* both 128 and 256 bit key test */
static wc_test_ret_t aes_xts_sector_test(void)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XtsAes *aes = NULL;
#else
    XtsAes aes[1];
#endif
    int aes_inited = 0;
    wc_test_ret_t ret = 0;
    unsigned char buf[AES_BLOCK_SIZE * 2];

    /* 128 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k1[] = {
        0xa3, 0xe4, 0x0d, 0x5b, 0xd4, 0xb6, 0xbb, 0xed,
        0xb2, 0xd1, 0x8c, 0x70, 0x0a, 0xd2, 0xdb, 0x22,
        0x10, 0xc8, 0x11, 0x90, 0x64, 0x6d, 0x67, 0x3c,
        0xbc, 0xa5, 0x3f, 0x13, 0x3e, 0xab, 0x37, 0x3c
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p1[] = {
        0x20, 0xe0, 0x71, 0x94, 0x05, 0x99, 0x3f, 0x09,
        0xa6, 0x6a, 0xe5, 0xbb, 0x50, 0x0e, 0x56, 0x2c
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c1[] = {
        0x74, 0x62, 0x35, 0x51, 0x21, 0x02, 0x16, 0xac,
        0x92, 0x6b, 0x96, 0x50, 0xb6, 0xd3, 0xfa, 0x52
    };
    word64 s1 = 141;

    /* 256 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k2[] = {
        0xef, 0x01, 0x0c, 0xa1, 0xa3, 0x66, 0x3e, 0x32,
        0x53, 0x43, 0x49, 0xbc, 0x0b, 0xae, 0x62, 0x23,
        0x2a, 0x15, 0x73, 0x34, 0x85, 0x68, 0xfb, 0x9e,
        0xf4, 0x17, 0x68, 0xa7, 0x67, 0x4f, 0x50, 0x7a,
        0x72, 0x7f, 0x98, 0x75, 0x53, 0x97, 0xd0, 0xe0,
        0xaa, 0x32, 0xf8, 0x30, 0x33, 0x8c, 0xc7, 0xa9,
        0x26, 0xc7, 0x73, 0xf0, 0x9e, 0x57, 0xb3, 0x57,
        0xcd, 0x15, 0x6a, 0xfb, 0xca, 0x46, 0xe1, 0xa0
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p2[] = {
        0xed, 0x98, 0xe0, 0x17, 0x70, 0xa8, 0x53, 0xb4,
        0x9d, 0xb9, 0xe6, 0xaa, 0xf8, 0x8f, 0x0a, 0x41,
        0xb9, 0xb5, 0x6e, 0x91, 0xa5, 0xa2, 0xb1, 0x1d,
        0x40, 0x52, 0x92, 0x54, 0xf5, 0x52, 0x3e, 0x75
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c2[] = {
        0xca, 0x20, 0xc5, 0x5e, 0x8d, 0xc1, 0x49, 0x68,
        0x7d, 0x25, 0x41, 0xde, 0x39, 0xc3, 0xdf, 0x63,
        0x00, 0xbb, 0x5a, 0x16, 0x3c, 0x10, 0xce, 0xd3,
        0x66, 0x6b, 0x13, 0x57, 0xdb, 0x8b, 0xd3, 0x9d
    };
    word64 s2 = 187;

#if !defined(BENCH_EMBEDDED) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
    /* Sector size for encrypt/decrypt consecutive sectors testcase */
    word32 sectorSz = 512;

    unsigned char data[550];

    WOLFSSL_SMALL_STACK_STATIC unsigned char k3[] = {
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
        0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
        0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22,
        0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22, 0x22
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p3[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
        0xfc, 0xfd, 0xfe, 0xff,

        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f, 0x20, 0x21, 0x22, 0x23,
        0x24, 0x25, 0x26, 0x27, 0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3a, 0x3b,
        0x3c, 0x3d, 0x3e, 0x3f, 0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, 0x50, 0x51, 0x52, 0x53,
        0x54, 0x55, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f,
        0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b,
        0x6c, 0x6d, 0x6e, 0x6f, 0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
        0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 0x81, 0x82, 0x83,
        0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b,
        0x9c, 0x9d, 0x9e, 0x9f, 0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
        0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf, 0xb0, 0xb1, 0xb2, 0xb3,
        0xb4, 0xb5, 0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7, 0xc8, 0xc9, 0xca, 0xcb,
        0xcc, 0xcd, 0xce, 0xcf, 0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
        0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf, 0xe0, 0xe1, 0xe2, 0xe3,
        0xe4, 0xe5, 0xe6, 0xe7, 0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
        0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9, 0xfa, 0xfb,
        0xfc, 0xfd, 0xfe, 0xff,

        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
        0x0c, 0x0d, 0x0e, 0x0f, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c3[] = {
        0xb9, 0x6b, 0x2b, 0xfd, 0x61, 0x87, 0x84, 0xd5, 0x26, 0xd2, 0x8c, 0x62,
        0x63, 0x01, 0xca, 0x46, 0xb1, 0x82, 0xfa, 0xdc, 0xbc, 0x32, 0x18, 0xe9,
        0xda, 0xe6, 0xda, 0xd1, 0x1a, 0x52, 0x77, 0xca, 0xdb, 0x0e, 0xbe, 0x37,
        0x88, 0x36, 0x1c, 0x87, 0x16, 0x60, 0xfe, 0xa8, 0x9e, 0xf6, 0x48, 0x64,
        0x94, 0x34, 0x64, 0xed, 0xf6, 0x9a, 0xc5, 0x28, 0xc9, 0xed, 0x64, 0x80,
        0x85, 0xd8, 0x93, 0xa7, 0x50, 0xb1, 0x9d, 0x2f, 0x1e, 0x34, 0xcc, 0xb4,
        0x03, 0xfb, 0x6b, 0x43, 0x21, 0xa8, 0x5b, 0xc6, 0x59, 0x13, 0xd2, 0xb5,
        0xf5, 0x7b, 0xf6, 0xb2, 0xa4, 0x7a, 0xd2, 0x50, 0x26, 0xcb, 0xa4, 0x83,
        0xc3, 0x56, 0xb0, 0xb1, 0x14, 0x34, 0x12, 0x1b, 0xea, 0x26, 0x97, 0x24,
        0x54, 0xcc, 0x32, 0x4c, 0xa4, 0xc2, 0xa3, 0x07, 0xfa, 0x30, 0xa9, 0xf0,
        0x91, 0x17, 0x60, 0x68, 0x88, 0x7f, 0x34, 0x7e, 0xbd, 0x20, 0x33, 0x95,
        0x6e, 0xc0, 0xb6, 0x2b, 0xff, 0x7e, 0x61, 0x35, 0x9a, 0x88, 0xff, 0xd9,
        0x69, 0x21, 0xe7, 0x8f, 0x45, 0x02, 0xf9, 0xd7, 0xeb, 0xa6, 0x53, 0xf1,
        0x73, 0x04, 0xf1, 0x0b, 0x85, 0xc6, 0x1f, 0x4a, 0x51, 0x2f, 0x95, 0x87,
        0x5a, 0x67, 0x37, 0xb2, 0x87, 0xf7, 0xbe, 0x2a, 0x17, 0x57, 0xca, 0xfc,
        0xdd, 0x5f, 0x37, 0x48, 0x78, 0xbd, 0xfa, 0x75, 0xc9, 0xfa, 0x86, 0x7e,
        0xc4, 0x0f, 0x60, 0x85, 0xce, 0x12, 0x44, 0x7c, 0xd9, 0xb2, 0x50, 0xd9,
        0x57, 0x85, 0xa5, 0xd7, 0x68, 0x59, 0x03, 0x09, 0x97, 0x2e, 0x8e, 0xa5,
        0xe3, 0x98, 0xac, 0x16, 0xfb, 0x6d, 0x54, 0xc5, 0x5d, 0x7a, 0x33, 0x44,
        0x0a, 0x39, 0x91, 0xcc, 0x9f, 0x67, 0xf9, 0x89, 0xbb, 0x62, 0x02, 0xc4,
        0x22, 0xec, 0xcf, 0x97, 0x69, 0x81, 0x3d, 0x00, 0xfd, 0xeb, 0x55, 0x08,
        0xa2, 0xff, 0x97, 0xaa, 0x79, 0xde, 0x3c, 0x8a, 0x78, 0x71, 0x73, 0xa2,
        0x98, 0x2f, 0xd8, 0x5c, 0x62, 0x1c, 0x5c, 0x23, 0x0a, 0xd1, 0xf1, 0x81,
        0x8a, 0x12, 0xe7, 0x4d, 0xdd, 0x4f, 0xd4, 0xf1, 0xe8, 0x0f, 0x25, 0x79,
        0x45, 0x4a, 0x49, 0x49, 0x7e, 0x56, 0x91, 0x4e, 0xaa, 0xba, 0x18, 0xe1,
        0xe4, 0xbe, 0x21, 0xdc, 0x58, 0x60, 0x6f, 0x6a, 0x7f, 0xdc, 0x5e, 0x74,
        0x47, 0xbf, 0xeb, 0x84, 0xc4, 0x1e, 0x5a, 0x61, 0x64, 0xc8, 0x63, 0x68,
        0xfa, 0x17, 0x9c, 0xac, 0x60, 0x1c, 0xa5, 0x6e, 0x00, 0x21, 0x93, 0x3c,
        0xd7, 0xbb, 0x73, 0x45, 0xf7, 0x34, 0x81, 0x6c, 0xfa, 0xf2, 0x33, 0xfd,
        0xb1, 0x40, 0x30, 0x6b, 0x30, 0xd1, 0x83, 0x5e, 0x2e, 0x7a, 0xce, 0xa6,
        0x12, 0x2a, 0x15, 0x03, 0x78, 0x29, 0xb9, 0x07, 0xae, 0xe7, 0xc2, 0x78,
        0x74, 0x72, 0xa5, 0x0e, 0x6b, 0x1f, 0x78, 0xf2, 0x5a, 0x69, 0xb6, 0x2b,
        0x99, 0x94, 0x1f, 0x89, 0xd1, 0x21, 0x14, 0x4a, 0x54, 0xab, 0x5a, 0x9f,
        0xaa, 0xa7, 0x96, 0x0a, 0x21, 0xce, 0x30, 0xb6, 0x70, 0x81, 0xe9, 0xd3,
        0x71, 0xc0, 0xf1, 0x15, 0xe2, 0xf6, 0xd3, 0xcc, 0x41, 0x15, 0x9d, 0xd5,
        0xa3, 0xa4, 0xe0, 0xf8, 0x62, 0xc4, 0x76, 0x65, 0x63, 0x89, 0xa7, 0xe2,
        0xfb, 0xf5, 0xc9, 0x80, 0x15, 0x5b, 0xc1, 0x59, 0xb2, 0xd0, 0x01, 0x3a,
        0xf9, 0xab, 0x5b, 0x79, 0x54, 0xed, 0x6b, 0xf9, 0x1d, 0x9d, 0x87, 0x63,
        0x80, 0x4f, 0xec, 0x9c, 0x4f, 0xad, 0x97, 0x04, 0xff, 0x62, 0x4a, 0x17,
        0xc0, 0x09, 0x2a, 0x2c, 0x23, 0x4b, 0xc3, 0xb6, 0x6d, 0xed, 0xdb, 0x1a,
        0x6f, 0x56, 0x2b, 0x78, 0x92, 0x3a, 0x5c, 0x7f, 0xb2, 0x63, 0xd3, 0xd5,
        0x1a, 0xbe, 0xc2, 0x34, 0xc8, 0xad, 0x36, 0xb7, 0x12, 0xb8, 0xe1, 0xb7,
        0x52, 0x7f, 0x16, 0x84, 0x2c, 0x47, 0x7e, 0xf2, 0xa5, 0x36, 0x2e, 0xad,
        0xe7, 0xbb, 0xc0, 0x6f, 0x27, 0x8e, 0x41, 0x08, 0x75, 0xe5, 0xff, 0xde,
        0x08, 0x9f, 0x8c, 0x91, 0xba, 0xc9, 0x9d, 0x9f, 0x27, 0x90, 0x50, 0x44,
        0x24, 0xe7, 0x3d, 0x6f
    };

    word64 s3 = 0x000000ffffffffff;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((aes = (XtsAes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif

    ret = wc_AesXtsInit(aes, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        aes_inited = 1;

    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesXtsEncryptSector(aes, buf, p1, sizeof(p1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* decrypt test */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecryptSector(aes, buf, c1, sizeof(c1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p1, buf, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* 256 bit key tests */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k2, sizeof(k2), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsEncryptSector(aes, buf, p2, sizeof(p2), s2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c2, buf, sizeof(c2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* decrypt test */
    XMEMSET(buf, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k2, sizeof(k2), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecryptSector(aes, buf, c2, sizeof(c2), s2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p2, buf, sizeof(p2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if !defined(BENCH_EMBEDDED) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
    /* encrypt consecutive sectors test */
    XMEMSET(data, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k3, sizeof(k3), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsEncryptConsecutiveSectors(aes, data, p3,
            sizeof(p3), s3, sectorSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c3, data, sizeof(c3)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* decrypt consecutive sectors test */
    XMEMSET(data, 0, sizeof(buf));
    ret = wc_AesXtsSetKeyNoInit(aes, k3, sizeof(k3), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecryptConsecutiveSectors(aes, data, c3,
            sizeof(c3), s3, sectorSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p3, data, sizeof(p3)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#endif /* !BENCH_EMBEDDED && (!HAVE_FIPS || FIPS_VERSION_GE(5, 3)) */

out:

    if (aes_inited)
        wc_AesXtsFree(aes);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (aes)
        XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 && WOLFSSL_AES_256 */


#ifdef WOLFSSL_AES_128
/* testing of bad arguments */
static wc_test_ret_t aes_xts_args_test(void)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XtsAes *aes = NULL;
#else
    XtsAes aes[1];
#endif
    int aes_inited = 0;
    wc_test_ret_t ret;
    unsigned char buf[AES_BLOCK_SIZE * 2];

    /* 128 key tests */
    WOLFSSL_SMALL_STACK_STATIC unsigned char k1[] = {
        0xa3, 0xe4, 0x0d, 0x5b, 0xd4, 0xb6, 0xbb, 0xed,
        0xb2, 0xd1, 0x8c, 0x70, 0x0a, 0xd2, 0xdb, 0x22,
        0x10, 0xc8, 0x11, 0x90, 0x64, 0x6d, 0x67, 0x3c,
        0xbc, 0xa5, 0x3f, 0x13, 0x3e, 0xab, 0x37, 0x3c
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char p1[] = {
        0x20, 0xe0, 0x71, 0x94, 0x05, 0x99, 0x3f, 0x09,
        0xa6, 0x6a, 0xe5, 0xbb, 0x50, 0x0e, 0x56, 0x2c
    };

    WOLFSSL_SMALL_STACK_STATIC unsigned char c1[] = {
        0x74, 0x62, 0x35, 0x51, 0x21, 0x02, 0x16, 0xac,
        0x92, 0x6b, 0x96, 0x50, 0xb6, 0xd3, 0xfa, 0x52
    };
    word64 s1 = 141;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((aes = (XtsAes *)XMALLOC(sizeof *aes, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif

    ret = wc_AesXtsInit(aes, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        aes_inited = 1;

    if (wc_AesXtsSetKeyNoInit(NULL, k1, sizeof(k1), AES_ENCRYPTION) == 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (wc_AesXtsSetKeyNoInit(aes, NULL, sizeof(k1), AES_ENCRYPTION) == 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* encryption operations */
    ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesXtsEncryptSector(NULL, buf, p1, sizeof(p1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret == 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = wc_AesXtsEncryptSector(aes, NULL, p1, sizeof(p1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret == 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* decryption operations */
    ret = wc_AesXtsSetKeyNoInit(aes, k1, sizeof(k1), AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesXtsDecryptSector(NULL, buf, c1, sizeof(c1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret == 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = wc_AesXtsDecryptSector(aes, NULL, c1, sizeof(c1), s1);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &aes->aes.asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret == 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = 0;

  out:

    if (aes_inited)
        wc_AesXtsFree(aes);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (aes)
        XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 */
#endif /* WOLFSSL_AES_XTS && (!HAVE_FIPS || FIPS_VERSION_GE(5,3)) */

#if defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
static wc_test_ret_t aes_cbc_test(void)
{
    byte cipher[AES_BLOCK_SIZE];
    byte plain[AES_BLOCK_SIZE];
    wc_test_ret_t ret;
    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = { /* "Now is the time for all " w/o trailing 0 */
        0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    byte key[] = "0123456789abcdef   ";  /* align */
    byte iv[]  = "1234567890abcdef   ";  /* align */

    XMEMSET(cipher, 0, AES_BLOCK_SIZE);
    XMEMSET(plain, 0, AES_BLOCK_SIZE);

    /* Parameter Validation testing. */
    ret = wc_AesCbcEncryptWithKey(cipher, msg, AES_BLOCK_SIZE, key, 17, NULL);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesCbcDecryptWithKey(plain, cipher, AES_BLOCK_SIZE, key, 17, NULL);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
#endif

    ret = wc_AesCbcEncryptWithKey(cipher, msg, AES_BLOCK_SIZE, key,
                                  AES_BLOCK_SIZE, iv);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesCbcDecryptWithKey(plain, cipher, AES_BLOCK_SIZE, key,
                                  AES_BLOCK_SIZE, iv);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(plain, msg, AES_BLOCK_SIZE) != 0)
        return WC_TEST_RET_ENC_NC;
#endif /* HAVE_AES_DECRYPT */

    (void)plain;
    return 0;
}
#endif

#if defined(HAVE_AES_ECB) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
static wc_test_ret_t aesecb_test(void)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
    Aes enc[1];
#endif
    int enc_inited = 0;
    byte cipher[AES_BLOCK_SIZE * 4];
#ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
    Aes dec[1];
#endif
    int dec_inited = 0;
    byte plain [AES_BLOCK_SIZE * 4];
#endif /* HAVE_AES_DECRYPT */
    wc_test_ret_t ret = 0;

#if defined(WOLFSSL_AES_256)
    {
        WOLFSSL_SMALL_STACK_STATIC const byte niPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte niCipher[] =
        {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte niKey[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        ret = wc_AesInit(enc, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        enc_inited = 1;
    #if defined(HAVE_AES_DECRYPT)
        ret = wc_AesInit(dec, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        dec_inited = 1;
    #endif

        XMEMSET(cipher, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(enc, niKey, sizeof(niKey), cipher, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (wc_AesEcbEncrypt(enc, cipher, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(cipher, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
        XMEMSET(cipher, 0, AES_BLOCK_SIZE);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        ret = wc_AesSetKey(enc, niKey, sizeof(niKey), cipher, AES_ENCRYPTION);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        ret = wc_AesEcbEncrypt(enc, cipher, niPlain, AES_BLOCK_SIZE);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(cipher, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        XMEMSET(cipher, 0, AES_BLOCK_SIZE);
        ret = wc_AesEcbEncrypt(enc, cipher, niPlain, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(cipher, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        XMEMSET(cipher, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(enc, niKey, sizeof(niKey), cipher, AES_ENCRYPTION);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        ret = wc_AesEcbEncrypt(enc, cipher, niPlain, AES_BLOCK_SIZE);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(cipher, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

        XMEMSET(plain, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(dec, niKey, sizeof(niKey), plain, AES_DECRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (wc_AesEcbDecrypt(dec, plain, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(plain, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
        XMEMSET(plain, 0, AES_BLOCK_SIZE);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        ret = wc_AesSetKey(dec, niKey, sizeof(niKey), plain, AES_DECRYPTION);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        ret = wc_AesEcbDecrypt(dec, plain, niCipher, AES_BLOCK_SIZE);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(plain, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        XMEMSET(plain, 0, AES_BLOCK_SIZE);
        ret = wc_AesEcbDecrypt(dec, plain, niCipher, AES_BLOCK_SIZE);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(plain, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        XMEMSET(plain, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(dec, niKey, sizeof(niKey), plain, AES_DECRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        ret = wc_AesEcbDecrypt(dec, plain, niCipher, AES_BLOCK_SIZE);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(plain, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif
    }

  out:
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc) {
        if (enc_inited)
            wc_AesFree(enc);
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
    }
#ifdef HAVE_AES_DECRYPT
    if (dec) {
        if (dec_inited)
            wc_AesFree(dec);
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
    }
#endif
#else
    if (enc_inited)
        wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    if (dec_inited)
        wc_AesFree(dec);
#endif
#endif

#endif /* WOLFSSL_AES_256 */

    return ret;
}
#endif /* HAVE_AES_ECB */

#ifdef WOLFSSL_AES_COUNTER
static wc_test_ret_t aesctr_test(Aes* enc, Aes* dec, byte* cipher, byte* plain)
{
    wc_test_ret_t ret = 0;

    /* test vectors from "Recommendation for Block Cipher Modes of
     * Operation" NIST Special Publication 800-38A */

    WOLFSSL_SMALL_STACK_STATIC const byte ctrIv[] =
    {
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctrPlain[] =
    {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
        0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
        0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
        0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
        0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
        0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
        0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
    };

#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap32[] =
    {
        0xff,0xff,0xff,0xff,0x0f,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap32_2[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0x0f,0xff,0xff,0xfe
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap64[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0x0f,0xff,0xff,0xff,0xff,0xff,0xff,0xff
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap64_2[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xf0,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap96[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0x0f,0xff,0xff,0xff
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap96_2[] =
    {
        0xff,0xff,0xff,0xf0,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe
    };
#endif
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap128[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
    };
#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctrIvWrap128_2[] =
    {
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
        0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe
    };
#endif

#ifdef WOLFSSL_AES_128
    WOLFSSL_SMALL_STACK_STATIC const byte oddCipher[] =
    {
        0xb9,0xd7,0xcb,0x08,0xb0,0xe1,0x7b,0xa0,
        0xc2
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Key[] =
    {
        0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
        0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Cipher[] =
    {
        0x87,0x4d,0x61,0x91,0xb6,0x20,0xe3,0x26,
        0x1b,0xef,0x68,0x64,0x99,0x0d,0xb6,0xce,
        0x98,0x06,0xf6,0x6b,0x79,0x70,0xfd,0xff,
        0x86,0x17,0x18,0x7b,0xb9,0xff,0xfd,0xff,
        0x5a,0xe4,0xdf,0x3e,0xdb,0xd5,0xd3,0x5e,
        0x5b,0x4f,0x09,0x02,0x0d,0xb0,0x3e,0xab,
        0x1e,0x03,0x1d,0xda,0x2f,0xbe,0x03,0xd1,
        0x79,0x21,0x70,0xa0,0xf3,0x00,0x9c,0xee
    };

#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap32Cipher[] =
    {
        0xb3,0x8b,0x58,0xbc,0xce,0xf4,0x71,0x78,
        0xf6,0x7c,0xdb,0xb4,0x27,0x2b,0x0a,0xbf,
        0x7e,0xad,0xea,0x5c,0xd1
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap32CipherLong[] =
    {
        0xb3,0x8b,0x58,0xbc,0xce,0xf4,0x71,0x78,
        0xf6,0x7c,0xdb,0xb4,0x27,0x2b,0x0a,0xbf,
        0x7e,0xad,0xea,0x5c,0xd1,0xb7,0x98,0xf0,
        0x22,0x20,0xfe,0x67,0xb0,0x02,0x23,0x50
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap32_2CipherLong[] =
    {
        0x6e,0xa1,0x27,0x4d,0xea,0x20,0x5f,0x39,
        0x68,0xc8,0xb6,0x78,0xde,0xfc,0x53,0x5c,
        0x90,0xc8,0xf6,0xc6,0xfa,0xe0,0x7b,0x09,
        0x7c,0xf8,0x9c,0x6a,0x5a,0xa5,0x17,0x7f,
        0x03,0x92,0x5f,0x4e,0x85,0xea,0x26,0xc9,
        0x5a,0xc2,0x74,0xe2,0xbf,0xe4,0x1b,0xd4
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap64Cipher[] =
    {
        0xdd,0x17,0x10,0x7c,0x45,0x04,0xac,0x43,
        0xef,0xa8,0xcc,0x32,0x34,0x87,0x88,0xd7,
        0xae,0x74,0x94,0x72,0x8e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap64CipherLong[] =
    {
        0xdd,0x17,0x10,0x7c,0x45,0x04,0xac,0x43,
        0xef,0xa8,0xcc,0x32,0x34,0x87,0x88,0xd7,
        0xae,0x74,0x94,0x72,0x8e,0xd0,0x71,0xc0,
        0x89,0x8a,0xa1,0xb0,0x29,0xa0,0x10,0x9e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap64_2CipherLong[] =
    {
        0x3f,0xe7,0xd5,0xf3,0xfa,0x09,0xfe,0x40,
        0xa6,0xa1,0x32,0x8b,0x57,0x12,0xb9,0xfa,
        0xf2,0x2d,0xe4,0x3c,0x66,0x1d,0x0a,0x8e,
        0x46,0xf8,0x2e,0x33,0xce,0x8d,0x4e,0x3b,
        0x17,0x67,0x9e,0x9f,0x76,0x9e,0xc2,0x99,
        0xd5,0xd4,0x71,0xed,0xb4,0x33,0xb2,0xcd
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap96Cipher[] =
    {
        0x55,0x24,0xc2,0x73,0xca,0xa3,0x48,0x03,
        0x0b,0x72,0x8d,0xd7,0x6c,0x99,0x8e,0x04,
        0x9d,0x77,0xc9,0x5f,0x38
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap96CipherLong[] =
    {
        0x55,0x24,0xc2,0x73,0xca,0xa3,0x48,0x03,
        0x0b,0x72,0x8d,0xd7,0x6c,0x99,0x8e,0x04,
        0x9d,0x77,0xc9,0x5f,0x38,0xb5,0x6e,0x44,
        0x21,0x8e,0xda,0x57,0xe0,0x41,0xc7,0x6a
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap96_2CipherLong[] =
    {
        0xc8,0x81,0x1a,0xbe,0xc7,0x5b,0x93,0x6f,
        0xe6,0x52,0xe4,0xb1,0x2d,0x1c,0x39,0xbc,
        0xeb,0x82,0x27,0x0a,0x7e,0xa5,0x0e,0x2d,
        0x32,0xda,0xbe,0x10,0x7a,0x10,0xcc,0xd3,
        0x6f,0xc6,0x83,0x28,0x05,0x57,0x8a,0x24,
        0x44,0x76,0x17,0x81,0xb9,0x5c,0x94,0x81
    };
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap128Cipher[] =
    {
        0xe1,0x33,0x38,0xe3,0x6c,0xb7,0x19,0x62,
        0xe0,0x0d,0x02,0x0b,0x4c,0xed,0xbd,0x86,
        0xd3,0xda,0xe1,0x5b,0x04
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap128CipherLong[] =
    {
        0xe1,0x33,0x38,0xe3,0x6c,0xb7,0x19,0x62,
        0xe0,0x0d,0x02,0x0b,0x4c,0xed,0xbd,0x86,
        0xd3,0xda,0xe1,0x5b,0x04,0xbb,0x35,0x2f,
        0xa0,0xf5,0x9f,0xeb,0xfc,0xb4,0xda,0x3e
    };
#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr128Wrap128_2CipherLong[] =
    {
        0xba,0x76,0xaa,0x54,0xd5,0xb5,0x60,0x67,
        0xc1,0xa7,0x90,0x3b,0x3f,0xdd,0xfa,0x89,
        0x24,0xdf,0x0c,0x56,0x5c,0xf4,0x2a,0x68,
        0x97,0x87,0x13,0xb6,0x7a,0xd1,0x24,0xfd,
        0x4d,0x3f,0x77,0x4a,0xb9,0xe4,0x7d,0xa2,
        0xdb,0xb9,0x31,0x5e,0xa3,0x11,0x06,0x80
    };
#endif
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Key[] =
    {
        0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
        0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
        0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Cipher[] =
    {
        0x1a,0xbc,0x93,0x24,0x17,0x52,0x1c,0xa2,
        0x4f,0x2b,0x04,0x59,0xfe,0x7e,0x6e,0x0b,
        0x09,0x03,0x39,0xec,0x0a,0xa6,0xfa,0xef,
        0xd5,0xcc,0xc2,0xc6,0xf4,0xce,0x8e,0x94,
        0x1e,0x36,0xb2,0x6b,0xd1,0xeb,0xc6,0x70,
        0xd1,0xbd,0x1d,0x66,0x56,0x20,0xab,0xf7,
        0x4f,0x78,0xa7,0xf6,0xd2,0x98,0x09,0x58,
        0x5a,0x97,0xda,0xec,0x58,0xc6,0xb0,0x50
    };

#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap32Cipher[] =
    {
        0x28,0xaa,0xfa,0x90,0x72,0x74,0x86,0xaf,
        0x72,0x73,0x35,0x17,0x70,0x4e,0x7d,0xca,
        0x0c,0x33,0x97,0x06,0xc0
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap32CipherLong[] =
    {
        0x28,0xaa,0xfa,0x90,0x72,0x74,0x86,0xaf,
        0x72,0x73,0x35,0x17,0x70,0x4e,0x7d,0xca,
        0x0c,0x33,0x97,0x06,0xc0,0xbe,0x83,0x87,
        0xdd,0xd3,0xff,0xd8,0xe4,0x6a,0x5b,0x84
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap32_2CipherLong[] =
    {
        0xf5,0x00,0xa2,0x91,0x54,0xa3,0x76,0xa2,
        0xdd,0xad,0x16,0x89,0xe5,0xf0,0x1d,0x40,
        0x84,0xcd,0x74,0x84,0xcb,0x8b,0x9e,0x29,
        0xa9,0x8a,0x12,0x65,0xa0,0x79,0x5e,0xce,
        0xd9,0x50,0x65,0x21,0x86,0xb0,0x85,0x0d,
        0x98,0x2d,0x9a,0x5a,0x11,0xbe,0xa5,0x7f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap64Cipher[] =
    {
        0xfe,0x39,0x27,0x97,0xac,0xe5,0xb8,0x74,
        0xb9,0x8c,0xbf,0x58,0x71,0xa4,0x80,0x33,
        0x3d,0xf7,0xb4,0xfd,0x8c
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap64CipherLong[] =
    {
        0xfe,0x39,0x27,0x97,0xac,0xe5,0xb8,0x74,
        0xb9,0x8c,0xbf,0x58,0x71,0xa4,0x80,0x33,
        0x3d,0xf7,0xb4,0xfd,0x8c,0x55,0x47,0x10,
        0xd5,0x91,0x35,0xbe,0xd8,0x0d,0xa5,0x9e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap64_2CipherLong[] =
    {
        0x59,0xf1,0xed,0x70,0x62,0x42,0xa8,0x06,
        0x07,0x36,0xe1,0xc5,0x04,0x79,0xc3,0x9b,
        0xd1,0x14,0x5c,0xcc,0x6f,0x81,0x5f,0x2f,
        0xa0,0xde,0xcf,0x61,0x55,0x18,0x7a,0xac,
        0xb0,0x59,0x37,0x90,0x53,0xb3,0x00,0x88,
        0xb4,0x49,0x90,0x7b,0x96,0xcd,0xcc,0xc3
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap96Cipher[] =
    {
        0x41,0x21,0x40,0x31,0xfb,0xc8,0xad,0x23,
        0xde,0x00,0x03,0xd8,0x96,0x45,0xc7,0xb5,
        0x47,0xb5,0xf3,0x30,0x3b
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap96CipherLong[] =
    {
        0x41,0x21,0x40,0x31,0xfb,0xc8,0xad,0x23,
        0xde,0x00,0x03,0xd8,0x96,0x45,0xc7,0xb5,
        0x47,0xb5,0xf3,0x30,0x3b,0xef,0xcd,0x80,
        0xe3,0x61,0x34,0xef,0x4e,0x1b,0x79,0xc1
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap96_2CipherLong[] =
    {
        0x3c,0xb2,0xff,0xc0,0x24,0xe1,0xf5,0xc4,
        0x0f,0xd1,0x0a,0x1b,0xbe,0x1f,0x23,0xa1,
        0x8e,0xbf,0x2b,0x96,0xb6,0x37,0xc8,0x25,
        0x06,0x90,0xe2,0xca,0x71,0x24,0x52,0x95,
        0xaa,0x8c,0x80,0xdf,0xb7,0xd7,0x30,0xb0,
        0xcc,0x06,0x4f,0x28,0xa2,0x74,0x27,0xf8
    };
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap128Cipher[] =
    {
        0x5c,0xc3,0x8f,0xab,0x30,0xb6,0xac,0x67,
        0xdc,0xc2,0x1e,0x7b,0x01,0x2e,0xcf,0x98,
        0x8c,0x68,0xa7,0xd9,0x57
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap128CipherLong[] =
    {
        0x5c,0xc3,0x8f,0xab,0x30,0xb6,0xac,0x67,
        0xdc,0xc2,0x1e,0x7b,0x01,0x2e,0xcf,0x98,
        0x8c,0x68,0xa7,0xd9,0x57,0xab,0x09,0x0f,
        0x01,0xc4,0x4e,0x62,0xaf,0xc2,0xdf,0x1a
    };
#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Wrap128_2CipherLong[] =
    {
        0x88,0x0a,0x26,0x4e,0xa8,0x26,0x21,0xe0,
        0xfc,0xbc,0x63,0xdc,0xd9,0x60,0x52,0xb2,
        0x99,0x2f,0xbb,0x1e,0x00,0xf5,0x9f,0x6d,
        0xab,0x48,0x0f,0xc6,0x37,0x12,0x56,0xe3,
        0x12,0x8d,0x31,0xc8,0xea,0xf4,0x41,0x82,
        0x7a,0x88,0xe0,0xd7,0xf0,0x67,0x03,0xa4
    };
#endif
#endif
#ifdef WOLFSSL_AES_256
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Key[] =
    {
        0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
        0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
        0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
        0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Cipher[] =
    {
        0x60,0x1e,0xc3,0x13,0x77,0x57,0x89,0xa5,
        0xb7,0xa7,0xf5,0x04,0xbb,0xf3,0xd2,0x28,
        0xf4,0x43,0xe3,0xca,0x4d,0x62,0xb5,0x9a,
        0xca,0x84,0xe9,0x90,0xca,0xca,0xf5,0xc5,
        0x2b,0x09,0x30,0xda,0xa2,0x3d,0xe9,0x4c,
        0xe8,0x70,0x17,0xba,0x2d,0x84,0x98,0x8d,
        0xdf,0xc9,0xc5,0x8d,0xb6,0x7a,0xad,0xa6,
        0x13,0xc2,0xdd,0x08,0x45,0x79,0x41,0xa6
    };

#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap32Cipher[] =
    {
        0xb0,0xa8,0xc0,0x65,0x85,0x20,0x0d,0x5c,
        0x25,0xcf,0xe7,0x58,0x63,0xc8,0xd4,0xea,
        0xa2,0x13,0x47,0x74,0xda
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap32CipherLong[] =
    {
        0xb0,0xa8,0xc0,0x65,0x85,0x20,0x0d,0x5c,
        0x25,0xcf,0xe7,0x58,0x63,0xc8,0xd4,0xea,
        0xa2,0x13,0x47,0x74,0xda,0x89,0x77,0x40,
        0x28,0x9c,0xe8,0x19,0x26,0x32,0xd8,0x1f
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap32_2CipherLong[] =
    {
        0xf6,0xd9,0x22,0xc6,0x80,0x29,0xaf,0x14,
        0x54,0x6c,0x0a,0xce,0x42,0xea,0x3c,0xa1,
        0x7c,0xeb,0x36,0x0d,0x8e,0xd7,0x8c,0x59,
        0xa8,0x09,0x9f,0x9e,0xba,0x5b,0x95,0xfa,
        0x26,0x8c,0x37,0x59,0xf8,0xae,0x8e,0xaa,
        0x4d,0xe4,0x1c,0xfe,0x51,0xc7,0xb7,0xcc
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap64Cipher[] =
    {
        0x74,0x1a,0x52,0x41,0x76,0xb4,0x11,0x8f,
        0xfd,0x57,0x31,0xfd,0x3d,0x76,0x8f,0x07,
        0xd4,0x94,0x4c,0xcd,0x4d
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap64CipherLong[] =
    {
        0x74,0x1a,0x52,0x41,0x76,0xb4,0x11,0x8f,
        0xfd,0x57,0x31,0xfd,0x3d,0x76,0x8f,0x07,
        0xd4,0x94,0x4c,0xcd,0x4d,0x47,0x5a,0x92,
        0x26,0x49,0x81,0x7a,0xda,0x36,0x27,0x01
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap64_2CipherLong[] =
    {
        0xf7,0x9c,0xbf,0xf6,0xa2,0xaa,0x8a,0x0a,
        0x63,0x8a,0x20,0x2f,0x12,0xf1,0x8e,0x49,
        0x30,0xc0,0x8d,0x5c,0x5f,0x8b,0xbc,0x16,
        0xdd,0x71,0xee,0x13,0x14,0x7b,0xe1,0x25,
        0xcb,0x87,0x8a,0xc6,0xdc,0x1d,0x54,0x7a,
        0xe1,0xe4,0x6f,0x0d,0x95,0x1b,0xd1,0x8b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap96Cipher[] =
    {
        0xb9,0x07,0x02,0xb8,0xbe,0x94,0xbf,0x53,
        0xdf,0x83,0x8e,0x23,0x8c,0x67,0x0c,0x81,
        0xb8,0x69,0xa1,0x48,0x45
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap96CipherLong[] =
    {
        0xb9,0x07,0x02,0xb8,0xbe,0x94,0xbf,0x53,
        0xdf,0x83,0x8e,0x23,0x8c,0x67,0x0c,0x81,
        0xb8,0x69,0xa1,0x48,0x45,0xf1,0xc6,0x27,
        0x36,0xa8,0xb2,0x4b,0x0e,0x62,0x6b,0x72
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap96_2CipherLong[] =
    {
        0xd5,0x56,0x73,0xaa,0xb8,0xe4,0x06,0xf6,
        0x83,0x45,0x3a,0xb4,0xb9,0x63,0xec,0xad,
        0x73,0xc5,0xab,0x78,0xb1,0x21,0xab,0xef,
        0x69,0x15,0xb7,0x0c,0xe9,0xb4,0x3a,0xe7,
        0xbc,0xc4,0x22,0xbd,0x93,0xba,0x52,0xe0,
        0x91,0x2f,0x5e,0x8d,0x6d,0x59,0xf7,0xc2
    };
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap128Cipher[] =
    {
        0x50,0xfd,0x97,0xc3,0xe6,0x1a,0xbb,0x48,
        0x73,0xfb,0x78,0xdf,0x1e,0x8e,0x77,0xe6,
        0x4b,0x45,0x7c,0xd6,0x8a
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap128CipherLong[] =
    {
        0x50,0xfd,0x97,0xc3,0xe6,0x1a,0xbb,0x48,
        0x73,0xfb,0x78,0xdf,0x1e,0x8e,0x77,0xe6,
        0x4b,0x45,0x7c,0xd6,0x8a,0xcc,0xda,0x4a,
        0x89,0xfa,0x23,0x6c,0x06,0xbf,0x26,0x05
    };
#ifdef WOLFSSL_ARMASM
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Wrap128_2CipherLong[] =
    {
        0x24,0x5c,0x09,0xa0,0x3b,0x1a,0x5a,0x94,
        0x2b,0x93,0x56,0x13,0x48,0xa0,0x21,0xce,
        0x95,0x11,0xa3,0x76,0xd6,0x59,0x88,0x42,
        0x04,0x71,0x69,0x62,0x28,0xb2,0xee,0x9d,
        0xd5,0xa0,0xea,0xc7,0x37,0x93,0x92,0xc7,
        0xf2,0xb6,0x8d,0xd9,0x59,0x1a,0xfa,0xbb
    };
#endif
#endif
    int i;
    struct {
        const byte* key;
        int         keySz;
        const byte* iv;
        const byte* plain;
        int         len;
        const byte* cipher;
    } testVec[] = {
#ifdef WOLFSSL_AES_128
        { ctr128Key, (int)sizeof(ctr128Key), ctrIv,
          ctrPlain, (int)sizeof(ctrPlain), ctr128Cipher },
        /* let's try with just 9 bytes, non block size test */
        { ctr128Key, (int)sizeof(ctr128Key), ctrIv,
          ctrPlain, (int)sizeof(oddCipher), ctr128Cipher },
        /* and an additional 9 bytes to reuse tmp left buffer */
        { NULL, 0, NULL, ctrPlain, (int)sizeof(oddCipher), oddCipher },
        /* Counter wrapping */
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr128Wrap128Cipher), ctr128Wrap128Cipher },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr128Wrap128CipherLong),
          ctr128Wrap128CipherLong },
    #ifdef WOLFSSL_ARMASM
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap128_2,
          ctrPlain, (int)sizeof(ctr128Wrap128_2CipherLong),
          ctr128Wrap128_2CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr128Wrap96Cipher), ctr128Wrap96Cipher },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr128Wrap96CipherLong),
          ctr128Wrap96CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap96_2,
          ctrPlain, (int)sizeof(ctr128Wrap96_2CipherLong),
          ctr128Wrap96_2CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr128Wrap64Cipher), ctr128Wrap64Cipher },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr128Wrap64CipherLong),
          ctr128Wrap64CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap64_2,
          ctrPlain, (int)sizeof(ctr128Wrap64_2CipherLong),
          ctr128Wrap64_2CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr128Wrap32Cipher), ctr128Wrap32Cipher },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr128Wrap32CipherLong),
          ctr128Wrap32CipherLong },
        { ctr128Key, (int)sizeof(ctr128Key), ctrIvWrap32_2,
          ctrPlain, (int)sizeof(ctr128Wrap32_2CipherLong),
          ctr128Wrap32_2CipherLong },
    #endif
#endif
#ifdef WOLFSSL_AES_192
        { ctr192Key, (int)sizeof(ctr192Key), ctrIv,
          ctrPlain, (int)sizeof(ctrPlain), ctr192Cipher },
        /* let's try with just 9 bytes, non block size test */
        { ctr192Key, (int)sizeof(ctr192Key), ctrIv,
          ctrPlain, (int)sizeof(oddCipher), ctr192Cipher },
        /* Counter wrapping */
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr192Wrap128Cipher), ctr192Wrap128Cipher },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr192Wrap128CipherLong),
          ctr192Wrap128CipherLong },
    #ifdef WOLFSSL_ARMASM
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap128_2,
          ctrPlain, (int)sizeof(ctr192Wrap128_2CipherLong),
          ctr192Wrap128_2CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr192Wrap96Cipher), ctr192Wrap96Cipher },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr192Wrap96CipherLong),
          ctr192Wrap96CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap96_2,
          ctrPlain, (int)sizeof(ctr192Wrap96_2CipherLong),
          ctr192Wrap96_2CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr192Wrap64Cipher), ctr192Wrap64Cipher },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr192Wrap64CipherLong),
          ctr192Wrap64CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap64_2,
          ctrPlain, (int)sizeof(ctr192Wrap64_2CipherLong),
          ctr192Wrap64_2CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr192Wrap32Cipher), ctr192Wrap32Cipher },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr192Wrap32CipherLong),
          ctr192Wrap32CipherLong },
        { ctr192Key, (int)sizeof(ctr192Key), ctrIvWrap32_2,
          ctrPlain, (int)sizeof(ctr192Wrap32_2CipherLong),
          ctr192Wrap32_2CipherLong },
    #endif
#endif
#ifdef WOLFSSL_AES_256
        { ctr256Key, (int)sizeof(ctr256Key), ctrIv,
          ctrPlain, (int)sizeof(ctrPlain), ctr256Cipher },
        /* let's try with just 9 bytes, non block size test */
        { ctr256Key, (int)sizeof(ctr256Key), ctrIv,
          ctrPlain, (int)sizeof(oddCipher), ctr256Cipher },
        /* Counter wrapping */
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr256Wrap128Cipher), ctr256Wrap128Cipher },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap128,
          ctrPlain, (int)sizeof(ctr256Wrap128CipherLong),
          ctr256Wrap128CipherLong },
    #ifdef WOLFSSL_ARMASM
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap128_2,
          ctrPlain, (int)sizeof(ctr256Wrap128_2CipherLong),
          ctr256Wrap128_2CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr256Wrap96Cipher), ctr256Wrap96Cipher },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap96,
          ctrPlain, (int)sizeof(ctr256Wrap96CipherLong),
          ctr256Wrap96CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap96_2,
          ctrPlain, (int)sizeof(ctr256Wrap96_2CipherLong),
          ctr256Wrap96_2CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr256Wrap64Cipher), ctr256Wrap64Cipher },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap64,
          ctrPlain, (int)sizeof(ctr256Wrap64CipherLong),
          ctr256Wrap64CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap64_2,
          ctrPlain, (int)sizeof(ctr256Wrap64_2CipherLong),
          ctr256Wrap64_2CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr256Wrap32Cipher), ctr256Wrap32Cipher },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap32,
          ctrPlain, (int)sizeof(ctr256Wrap32CipherLong),
          ctr256Wrap32CipherLong },
        { ctr256Key, (int)sizeof(ctr256Key), ctrIvWrap32_2,
          ctrPlain, (int)sizeof(ctr256Wrap32_2CipherLong),
          ctr256Wrap32_2CipherLong },
    #endif
#endif
    };
    #define AES_CTR_TEST_LEN (int)(sizeof(testVec) / sizeof(*testVec))

    for (i = 0; i < AES_CTR_TEST_LEN; i++) {
        if (testVec[i].key != NULL) {
            ret = wc_AesSetKeyDirect(enc, testVec[i].key, testVec[i].keySz,
                testVec[i].iv, AES_ENCRYPTION);
            if (ret != 0) {
                ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
            }
            /* Ctr only uses encrypt, even on key setup */
            ret = wc_AesSetKeyDirect(dec, testVec[i].key, testVec[i].keySz,
                testVec[i].iv, AES_ENCRYPTION);
            if (ret != 0) {
                ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
            }
        }

        ret = wc_AesCtrEncrypt(enc, cipher, testVec[i].plain, testVec[i].len);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
        ret = wc_AesCtrEncrypt(dec, plain, cipher, testVec[i].len);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }

        if (XMEMCMP(plain, ctrPlain, testVec[i].len)) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
#if !(FIPS_VERSION_EQ(2,0) && defined(WOLFSSL_ARMASM))
        if (XMEMCMP(cipher, testVec[i].cipher, testVec[i].len)) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
#endif
    }

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    for (i = 0; i < AES_CTR_TEST_LEN; i++) {
        if (testVec[i].key != NULL) {
            ret = wc_AesSetKeyDirect(enc, testVec[i].key, testVec[i].keySz,
                testVec[i].iv, AES_ENCRYPTION);
            if (ret != 0) {
                ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
            }
            /* Ctr only uses encrypt, even on key setup */
            ret = wc_AesSetKeyDirect(dec, testVec[i].key, testVec[i].keySz,
                testVec[i].iv, AES_ENCRYPTION);
            if (ret != 0) {
                ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
            }
        }

        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        ret = wc_AesCtrEncrypt(enc, cipher, testVec[i].plain, testVec[i].len);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        ret = wc_AesCtrEncrypt(dec, plain, cipher, testVec[i].len);
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }

        if (XMEMCMP(plain, ctrPlain, testVec[i].len)) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
#if !(FIPS_VERSION_EQ(2,0) && defined(WOLFSSL_ARMASM))
        if (XMEMCMP(cipher, testVec[i].cipher, testVec[i].len)) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
#endif
    }

    for (i = 0; i < AES_CTR_TEST_LEN; i++) {
        if (testVec[i].key != NULL) {
            WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
            ret = wc_AesSetKeyDirect(enc, testVec[i].key, testVec[i].keySz,
                testVec[i].iv, AES_ENCRYPTION);
            WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
            if (ret != 0) {
                ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
            }
            /* Ctr only uses encrypt, even on key setup */
            WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
            ret = wc_AesSetKeyDirect(dec, testVec[i].key, testVec[i].keySz,
                testVec[i].iv, AES_ENCRYPTION);
            WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
            if (ret != 0) {
                ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
            }
        }

        ret = wc_AesCtrEncrypt(enc, cipher, testVec[i].plain, testVec[i].len);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
        ret = wc_AesCtrEncrypt(dec, plain, cipher, testVec[i].len);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }

        if (XMEMCMP(plain, ctrPlain, testVec[i].len)) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
#if !(FIPS_VERSION_EQ(2,0) && defined(WOLFSSL_ARMASM))
        if (XMEMCMP(cipher, testVec[i].cipher, testVec[i].len)) {
            ERROR_OUT(WC_TEST_RET_ENC_I(i), out);
        }
#endif
    }

#endif /* DEBUG_VECTOR_REGISTER_ACCESS && WC_AES_C_DYNAMIC_FALLBACK */


out:
    return ret;
}
#endif /* WOLFSSL_AES_COUNTER */


WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aes_test(void)
{
#if defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_COUNTER) || defined(WOLFSSL_AES_DIRECT)
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
    Aes enc[1];
#endif
    int enc_inited = 0;
    byte cipher[AES_BLOCK_SIZE * 4];
#ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
    Aes dec[1];
#endif
    int dec_inited = 0;
    byte plain [AES_BLOCK_SIZE * 4];
#endif /* HAVE_AES_DECRYPT */
#endif /* HAVE_AES_CBC || WOLFSSL_AES_COUNTER || WOLFSSL_AES_DIRECT */
    wc_test_ret_t ret = 0;

#ifdef HAVE_AES_CBC
#ifdef WOLFSSL_AES_128
    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = { /* "Now is the time for all " w/o trailing 0 */
        0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
    {
        0x95,0x94,0x92,0x57,0x5f,0x42,0x81,0x53,
        0x2c,0xcc,0x9d,0x46,0x77,0xa2,0x33,0xcb
    };
    #ifdef HAVE_RENESAS_SYNC
        const byte *key =
                (byte*)guser_PKCbInfo.wrapped_key_aes128;
    #else
        WOLFSSL_SMALL_STACK_STATIC const
            byte key[] = "0123456789abcdef   ";  /* align */
    #endif
    WOLFSSL_SMALL_STACK_STATIC const byte iv[]  = "1234567890abcdef   ";  /* align */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
#if defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_COUNTER) || defined(WOLFSSL_AES_DIRECT)
    if (enc == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif
#if defined(HAVE_AES_DECRYPT) || defined(WOLFSSL_AES_COUNTER) || defined(WOLFSSL_AES_DIRECT)
    if (dec == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif
#endif

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    enc_inited = 1;
#if defined(HAVE_AES_DECRYPT) || defined(WOLFSSL_AES_COUNTER)
    ret = wc_AesInit(dec, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    dec_inited = 1;
#endif

    ret = wc_AesSetKey(enc, key, AES_BLOCK_SIZE, iv, AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#if defined(HAVE_AES_DECRYPT) || defined(WOLFSSL_AES_COUNTER)
    ret = wc_AesSetKey(dec, key, AES_BLOCK_SIZE, iv, AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

#ifdef HAVE_AES_ECB
    {
        WOLFSSL_SMALL_STACK_STATIC const byte verify_ecb[AES_BLOCK_SIZE] = {
            0xd0, 0xc9, 0xd9, 0xc9, 0x40, 0xe8, 0x97, 0xb6,
            0xc8, 0x8c, 0x33, 0x3b, 0xb5, 0x8f, 0x85, 0xd1
        };
        XMEMSET(cipher, 0, AES_BLOCK_SIZE * 4);
        ret = wc_AesEcbEncrypt(enc, cipher, msg, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(cipher, verify_ecb, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #ifdef HAVE_AES_DECRYPT
        XMEMSET(plain, 0, AES_BLOCK_SIZE * 4);
        ret = wc_AesEcbDecrypt(dec, plain, cipher, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif /* HAVE_AES_DECRYPT */
    }
#endif

    XMEMSET(cipher, 0, AES_BLOCK_SIZE * 4);
    ret = wc_AesCbcEncrypt(enc, cipher, msg, AES_BLOCK_SIZE);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    XMEMSET(plain, 0, AES_BLOCK_SIZE * 4);
    ret = wc_AesCbcDecrypt(dec, plain, cipher, AES_BLOCK_SIZE);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */
    /* skipped because wrapped key use in case of renesas sm */
    #ifndef HAVE_RENESAS_SYNC
    if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif
#endif /* WOLFSSL_AES_128 */

#if defined(WOLFSSL_AESNI) && defined(HAVE_AES_DECRYPT)
    {
        WOLFSSL_SMALL_STACK_STATIC const byte bigMsg[] = {
            /* "All work and no play makes Jack a dull boy. " */
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
            0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
            0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
            0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
            0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
            0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
            0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
            0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
            0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
            0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
            0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
            0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
            0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
            0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
            0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
            0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
            0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
            0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
            0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
            0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
            0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
            0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20,
            0x61,0x20,0x64,0x75,0x6c,0x6c,0x20,0x62,
            0x6f,0x79,0x2e,0x20,0x41,0x6c,0x6c,0x20,
            0x77,0x6f,0x72,0x6b,0x20,0x61,0x6e,0x64,
            0x20,0x6e,0x6f,0x20,0x70,0x6c,0x61,0x79,
            0x20,0x6d,0x61,0x6b,0x65,0x73,0x20,0x4a,
            0x61,0x63,0x6b,0x20,0x61,0x20,0x64,0x75,
            0x6c,0x6c,0x20,0x62,0x6f,0x79,0x2e,0x20,
            0x41,0x6c,0x6c,0x20,0x77,0x6f,0x72,0x6b,
            0x20,0x61,0x6e,0x64,0x20,0x6e,0x6f,0x20,
            0x70,0x6c,0x61,0x79,0x20,0x6d,0x61,0x6b,
            0x65,0x73,0x20,0x4a,0x61,0x63,0x6b,0x20
        };
        WOLFSSL_SMALL_STACK_STATIC const byte bigKey[] = "0123456789abcdeffedcba9876543210";
        word32 keySz, msgSz;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        byte *bigCipher = (byte *)XMALLOC(sizeof(bigMsg), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        byte *bigPlain = (byte *)XMALLOC(sizeof(bigMsg), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

        if ((bigCipher == NULL) ||
            (bigPlain == NULL)) {
            if (bigCipher != NULL)
                XFREE(bigCipher, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        }
#else
        byte bigCipher[sizeof(bigMsg)];
        byte bigPlain[sizeof(bigMsg)];
#endif

        /* Iterate from one AES_BLOCK_SIZE of bigMsg through the whole
         * message by AES_BLOCK_SIZE for each size of AES key. */
        for (keySz = 16; keySz <= 32; keySz += 8) {
            for (msgSz = AES_BLOCK_SIZE;
                 msgSz <= sizeof(bigMsg);
                 msgSz += AES_BLOCK_SIZE) {

                XMEMSET(bigCipher, 0, sizeof(bigMsg));
                XMEMSET(bigPlain, 0, sizeof(bigMsg));
                ret = wc_AesSetKey(enc, bigKey, keySz, iv, AES_ENCRYPTION);
                if (ret != 0) {
                    ret = WC_TEST_RET_ENC_EC(ret);
                    break;
                }
                ret = wc_AesSetKey(dec, bigKey, keySz, iv, AES_DECRYPTION);
                if (ret != 0) {
                    ret = WC_TEST_RET_ENC_EC(ret);
                    break;
                }

                ret = wc_AesCbcEncrypt(enc, bigCipher, bigMsg, msgSz);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                if (ret != 0) {
                    ret = WC_TEST_RET_ENC_EC(ret);
                    break;
                }

                ret = wc_AesCbcDecrypt(dec, bigPlain, bigCipher, msgSz);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                if (ret != 0) {
                    ret = WC_TEST_RET_ENC_EC(ret);
                    break;
                }

                if (XMEMCMP(bigPlain, bigMsg, msgSz)) {
                    ret = WC_TEST_RET_ENC_NC;
                    break;
                }
            }
            if (ret != 0)
                break;
        }

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
        /* Iterate from one AES_BLOCK_SIZE of bigMsg through the whole
         * message by AES_BLOCK_SIZE for each size of AES key. */
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
        for (keySz = 16; keySz <= 32; keySz += 8) {
            for (msgSz = AES_BLOCK_SIZE;
                 msgSz <= sizeof(bigMsg);
                 msgSz += AES_BLOCK_SIZE) {

                XMEMSET(bigCipher, 0, sizeof(bigMsg));
                XMEMSET(bigPlain, 0, sizeof(bigMsg));
                ret = wc_AesSetKey(enc, bigKey, keySz, iv, AES_ENCRYPTION);
                if (ret != 0) {
                    ret = WC_TEST_RET_ENC_EC(ret);
                    break;
                }
                ret = wc_AesSetKey(dec, bigKey, keySz, iv, AES_DECRYPTION);
                if (ret != 0) {
                    ret = WC_TEST_RET_ENC_EC(ret);
                    break;
                }

                ret = wc_AesCbcEncrypt(enc, bigCipher, bigMsg, msgSz);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                if (ret != 0) {
                    ret = WC_TEST_RET_ENC_EC(ret);
                    break;
                }

                ret = wc_AesCbcDecrypt(dec, bigPlain, bigCipher, msgSz);
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
            #endif
                if (ret != 0) {
                    ret = WC_TEST_RET_ENC_EC(ret);
                    break;
                }

                if (XMEMCMP(bigPlain, bigMsg, msgSz)) {
                    ret = WC_TEST_RET_ENC_NC;
                    break;
                }
            }
            if (ret != 0)
                break;
        }
        WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
#endif /* DEBUG_VECTOR_REGISTER_ACCESS && WC_AES_C_DYNAMIC_FALLBACK */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        XFREE(bigCipher, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(bigPlain, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

        if (ret != 0)
            goto out;
    }
#endif /* WOLFSSL_AESNI && HAVE_AES_DECRYPT */

    /* Test of AES IV state with encrypt/decrypt */
#if defined(WOLFSSL_AES_128) && !defined(HAVE_RENESAS_SYNC)
    {
        /* Test Vector from "NIST Special Publication 800-38A, 2001 Edition"
         * https://nvlpubs.nist.gov/nistpubs/legacy/sp/nistspecialpublication800-38a.pdf
         */
        WOLFSSL_SMALL_STACK_STATIC const byte msg2[] =
        {
            0x6b, 0xc1, 0xbe, 0xe2, 0x2e, 0x40, 0x9f, 0x96,
            0xe9, 0x3d, 0x7e, 0x11, 0x73, 0x93, 0x17, 0x2a,
            0xae, 0x2d, 0x8a, 0x57, 0x1e, 0x03, 0xac, 0x9c,
            0x9e, 0xb7, 0x6f, 0xac, 0x45, 0xaf, 0x8e, 0x51
        };

        WOLFSSL_SMALL_STACK_STATIC const byte verify2[] =
        {
            0x76, 0x49, 0xab, 0xac, 0x81, 0x19, 0xb2, 0x46,
            0xce, 0xe9, 0x8e, 0x9b, 0x12, 0xe9, 0x19, 0x7d,
            0x50, 0x86, 0xcb, 0x9b, 0x50, 0x72, 0x19, 0xee,
            0x95, 0xdb, 0x11, 0x3a, 0x91, 0x76, 0x78, 0xb2
        };
        WOLFSSL_SMALL_STACK_STATIC const byte key2[] = {
            0x2b, 0x7e, 0x15, 0x16, 0x28, 0xae, 0xd2, 0xa6,
            0xab, 0xf7, 0x15, 0x88, 0x09, 0xcf, 0x4f, 0x3c
        };
        WOLFSSL_SMALL_STACK_STATIC const byte iv2[]  = {
            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
            0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f
        };


        ret = wc_AesSetKey(enc, key2, sizeof(key2), iv2, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        XMEMSET(cipher, 0, AES_BLOCK_SIZE * 2);
        ret = wc_AesCbcEncrypt(enc, cipher, msg2, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    #ifndef HAVE_RENESAS_SYNC
        if (XMEMCMP(cipher, verify2, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    #endif
        ret = wc_AesCbcEncrypt(enc, cipher + AES_BLOCK_SIZE,
                msg2 + AES_BLOCK_SIZE, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(cipher + AES_BLOCK_SIZE, verify2 + AES_BLOCK_SIZE,
                    AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        #if defined(HAVE_AES_DECRYPT)
        ret = wc_AesSetKey(dec, key2, sizeof(key2), iv2, AES_DECRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        XMEMSET(plain, 0, AES_BLOCK_SIZE * 2);
        ret = wc_AesCbcDecrypt(dec, plain, verify2, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(plain, msg2, AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesCbcDecrypt(dec, plain + AES_BLOCK_SIZE,
                verify2 + AES_BLOCK_SIZE, AES_BLOCK_SIZE);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(plain + AES_BLOCK_SIZE, msg2 + AES_BLOCK_SIZE,
                    AES_BLOCK_SIZE))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        #endif /* HAVE_AES_DECRYPT */
    }
#endif /* WOLFSSL_AES_128 && !HAVE_RENESAS_SYNC */
#endif /* HAVE_AES_CBC */

#ifdef WOLFSSL_AES_COUNTER
    ret = aesctr_test(enc, dec, cipher, plain);
    if (ret != 0)
        return ret;
#endif

#if defined(WOLFSSL_AES_DIRECT) && defined(WOLFSSL_AES_256)
    {
        WOLFSSL_SMALL_STACK_STATIC const byte niPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte niCipher[] =
        {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte niKey[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        XMEMSET(cipher, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(enc, niKey, sizeof(niKey), cipher, AES_ENCRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#if !defined(HAVE_SELFTEST) && \
    (defined(WOLFSSL_LINUXKM) || \
     !defined(HAVE_FIPS) || \
     (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3)))
        ret = wc_AesEncryptDirect(enc, cipher, niPlain);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#else
        wc_AesEncryptDirect(enc, cipher, niPlain);
#endif
        if (XMEMCMP(cipher, niCipher, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        XMEMSET(plain, 0, AES_BLOCK_SIZE);
        ret = wc_AesSetKey(dec, niKey, sizeof(niKey), plain, AES_DECRYPTION);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#if !defined(HAVE_SELFTEST) && \
    (defined(WOLFSSL_LINUXKM) || \
     !defined(HAVE_FIPS) || \
     (defined(FIPS_VERSION_GE) && FIPS_VERSION_GE(5,3)))
        ret = wc_AesDecryptDirect(dec, plain, niCipher);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#else
        wc_AesDecryptDirect(dec, plain, niCipher);
#endif
        if (XMEMCMP(plain, niPlain, AES_BLOCK_SIZE) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
#endif /* WOLFSSL_AES_DIRECT && WOLFSSL_AES_256 */

#ifndef HAVE_RENESAS_SYNC
    ret = aes_key_size_test();
    if (ret != 0)
        goto out;
#endif

#if defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128) && \
    !defined(HAVE_RENESAS_SYNC)
    ret = aes_cbc_test();
    if (ret != 0)
        goto out;
#endif

#if defined(WOLFSSL_AES_XTS) && (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5,3))
    #ifdef WOLFSSL_AES_128
    ret = aes_xts_128_test();
    if (ret != 0)
        goto out;
    #endif
    #ifdef WOLFSSL_AES_256
    ret = aes_xts_256_test();
    if (ret != 0)
        goto out;
    #endif
    #if defined(WOLFSSL_AES_128) && defined(WOLFSSL_AES_256)
    ret = aes_xts_sector_test();
    if (ret != 0)
        goto out;
    #endif
    #ifdef WOLFSSL_AES_128
    ret = aes_xts_args_test();
    if (ret != 0)
        goto out;
    #endif
#endif

#if defined(WOLFSSL_AES_CFB)
    ret = aescfb_test();
    if (ret != 0)
        goto out;
#if !defined(HAVE_SELFTEST) && !defined(HAVE_FIPS)
    ret = aescfb1_test();
    if (ret != 0)
        goto out;

    ret = aescfb8_test();
    if (ret != 0)
        goto out;
#endif
#endif

#if defined(HAVE_AES_ECB) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    ret = aesecb_test();
    if (ret != 0)
        goto out;
#endif

  out:

#if defined(HAVE_AES_CBC) || defined(WOLFSSL_AES_COUNTER) || defined(WOLFSSL_AES_DIRECT)
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc) {
        if (enc_inited)
            wc_AesFree(enc);
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
    }
#else
    if (enc_inited)
        wc_AesFree(enc);
#endif
    (void)cipher;
#ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (dec) {
        if (dec_inited)
            wc_AesFree(dec);
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
    }
#else
    if (dec_inited)
        wc_AesFree(dec);
#endif
    (void)plain;
#endif /* HAVE_AES_DECRYPT */
#endif /* HAVE_AES_CBC || WOLFSSL_AES_COUNTER || WOLFSSL_AES_DIRECT */

    return ret;
}

#ifdef WOLFSSL_AES_192
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aes192_test(void)
{
#ifdef HAVE_AES_CBC
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *enc = NULL;
#else
    Aes enc[1];
#endif
    byte cipher[AES_BLOCK_SIZE];
#ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *dec = NULL;
#else
    Aes dec[1];
#endif
    byte plain[AES_BLOCK_SIZE];
#endif
#endif /* HAVE_AES_CBC */
    wc_test_ret_t ret = 0;

#ifdef HAVE_AES_CBC
    /* Test vectors from NIST Special Publication 800-38A, 2001 Edition
     * Appendix F.2.3  */

    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
    {
        0x4f,0x02,0x1d,0xb2,0x43,0xbc,0x63,0x3d,
        0x71,0x78,0x18,0x3a,0x9f,0xa0,0x71,0xe8
    };

    WOLFSSL_SMALL_STACK_STATIC byte key[] = {
        0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
        0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
        0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
    };
    WOLFSSL_SMALL_STACK_STATIC byte iv[]  = {
        0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
        0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif
#endif

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesInit(dec, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

    ret = wc_AesSetKey(enc, key, (int) sizeof(key), iv, AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesSetKey(dec, key, (int) sizeof(key), iv, AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

    XMEMSET(cipher, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcEncrypt(enc, cipher, msg, (int) sizeof(msg));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(cipher, verify, (int) sizeof(cipher)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#ifdef HAVE_AES_DECRYPT
    XMEMSET(plain, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcDecrypt(dec, plain, cipher, (int) sizeof(cipher));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(plain, msg, (int) sizeof(plain))) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
#endif

    wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    wc_AesFree(dec);
#endif

  out:
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif
#endif /* HAVE_AES_CBC */

    return ret;
}
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aes256_test(void)
{
#ifdef HAVE_AES_CBC
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *enc = NULL;
#else
    Aes enc[1];
#endif
    byte cipher[AES_BLOCK_SIZE];
#ifdef HAVE_AES_DECRYPT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *dec = NULL;
#else
    Aes dec[1];
#endif
    byte plain[AES_BLOCK_SIZE];
#endif
#endif /* HAVE_AES_CBC */
    wc_test_ret_t ret = 0;

#ifdef HAVE_AES_CBC
    /* Test vectors from NIST Special Publication 800-38A, 2001 Edition,
     * Appendix F.2.5  */
    WOLFSSL_SMALL_STACK_STATIC const byte msg[] = {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
    {
        0xf5,0x8c,0x4c,0x04,0xd6,0xe5,0xf1,0xba,
        0x77,0x9e,0xab,0xfb,0x5f,0x7b,0xfb,0xd6
    };
#ifdef HAVE_RENESAS_SYNC
    byte *key =
                (byte*)guser_PKCbInfo.wrapped_key_aes256;
    int keySz = (256/8);
#else
    WOLFSSL_SMALL_STACK_STATIC byte key[] = {
        0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
        0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
        0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
        0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
    };
    int keySz = (int)sizeof(key);
#endif
    WOLFSSL_SMALL_STACK_STATIC byte iv[]  = {
        0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
        0x08,0x09,0x0A,0x0B,0x0C,0x0D,0x0E,0x0F
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#ifdef HAVE_AES_DECRYPT
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif
#endif

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesInit(dec, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

    ret = wc_AesSetKey(enc, key, keySz, iv, AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesSetKey(dec, key, keySz, iv, AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

    XMEMSET(cipher, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcEncrypt(enc, cipher, msg, (int) sizeof(msg));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    XMEMSET(plain, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcDecrypt(dec, plain, cipher, (int) sizeof(cipher));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(plain, msg, (int) sizeof(plain))) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
#endif
#ifndef HAVE_RENESAS_SYNC
    if (XMEMCMP(cipher, verify, (int) sizeof(cipher)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif
    wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    wc_AesFree(dec);
#endif

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    ret = wc_AesSetKey(enc, key, keySz, iv, AES_ENCRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesSetKey(dec, key, keySz, iv, AES_DECRYPTION);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

    XMEMSET(cipher, 0, AES_BLOCK_SIZE);
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    ret = wc_AesCbcEncrypt(enc, cipher, msg, (int) sizeof(msg));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    XMEMSET(plain, 0, AES_BLOCK_SIZE);
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    ret = wc_AesCbcDecrypt(dec, plain, cipher, (int) sizeof(cipher));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(plain, msg, (int) sizeof(plain))) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
#endif
#ifndef HAVE_RENESAS_SYNC
    if (XMEMCMP(cipher, verify, (int) sizeof(cipher)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

    wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    wc_AesFree(dec);
#endif

    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    ret = wc_AesSetKey(enc, key, keySz, iv, AES_ENCRYPTION);
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    ret = wc_AesSetKey(dec, key, keySz, iv, AES_DECRYPTION);
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif

    XMEMSET(cipher, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcEncrypt(enc, cipher, msg, (int) sizeof(msg));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    XMEMSET(plain, 0, AES_BLOCK_SIZE);
    ret = wc_AesCbcDecrypt(dec, plain, cipher, (int) sizeof(cipher));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(plain, msg, (int) sizeof(plain))) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
#endif
#ifndef HAVE_RENESAS_SYNC
    if (XMEMCMP(cipher, verify, (int) sizeof(cipher)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

    wc_AesFree(enc);
#ifdef HAVE_AES_DECRYPT
    wc_AesFree(dec);
#endif

#endif

  out:
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#ifdef HAVE_AES_DECRYPT
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
#endif
#endif /* HAVE_AES_CBC */

    return ret;
}
#endif /* WOLFSSL_AES_256 */


#ifdef HAVE_AESGCM

#ifdef WOLFSSL_AES_128
static wc_test_ret_t aesgcm_default_test_helper(byte* key, int keySz, byte* iv, int ivSz,
                byte* plain, int plainSz, byte* cipher, int cipherSz,
                byte* aad, int aadSz, byte* tag, int tagSz)
{
    wc_test_ret_t ret;
    int enc_inited = 0, dec_inited = 0;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *enc = NULL;
    Aes *dec = NULL;
#else
    Aes enc[1];
    Aes dec[1];
#endif

    byte resultT[AES_BLOCK_SIZE];
    byte resultP[AES_BLOCK_SIZE * 3];
    byte resultC[AES_BLOCK_SIZE * 3];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif

    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        enc_inited = 1;
    ret = wc_AesInit(dec, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    else
        dec_inited = 1;

    ret = wc_AesGcmSetKey(enc, key, keySz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesGcmEncrypt(enc, resultC, plain, plainSz, iv, ivSz,
                                     resultT, tagSz, aad, aadSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (cipher != NULL) {
        if (XMEMCMP(cipher, resultC, cipherSz))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
    if (XMEMCMP(tag, resultT, tagSz))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    ret = wc_AesGcmEncrypt(enc, resultC, plain, plainSz, iv, ivSz,
                                     resultT, tagSz, aad, aadSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (cipher != NULL) {
        if (XMEMCMP(cipher, resultC, cipherSz))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
    if (XMEMCMP(tag, resultT, tagSz))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmSetKey(dec, key, keySz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesGcmDecrypt(dec, resultP, resultC, cipherSz,
                   iv, ivSz, resultT, tagSz, aad, aadSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (plain != NULL) {
        if (XMEMCMP(plain, resultP, plainSz))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }

#if defined(DEBUG_VECTOR_REGISTER_ACCESS) && defined(WC_AES_C_DYNAMIC_FALLBACK)
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(SYSLIB_FAILED_E);
    ret = wc_AesGcmDecrypt(dec, resultP, resultC, cipherSz,
                   iv, ivSz, resultT, tagSz, aad, aadSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    WC_DEBUG_SET_VECTOR_REGISTERS_RETVAL(0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (plain != NULL) {
        if (XMEMCMP(plain, resultP, plainSz))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
#endif

#endif /* HAVE_AES_DECRYPT */

    ret = 0;

  out:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc) {
        if (enc_inited)
            wc_AesFree(enc);
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
    }
    if (dec) {
        if (dec_inited)
            wc_AesFree(dec);
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
    }
#else
    if (enc_inited)
        wc_AesFree(enc);
    if (dec_inited)
        wc_AesFree(dec);
#endif

    return ret;
}
#endif


/* tests that only use 12 byte IV and 16 or less byte AAD
 * test vectors are from NIST SP 800-38D
 * https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/CAVP-TESTING-BLOCK-CIPHER-MODES*/
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aesgcm_default_test(void)
{
#ifdef WOLFSSL_AES_128
    byte key1[] = {
        0x29, 0x8e, 0xfa, 0x1c, 0xcf, 0x29, 0xcf, 0x62,
        0xae, 0x68, 0x24, 0xbf, 0xc1, 0x95, 0x57, 0xfc
    };

    byte iv1[] = {
        0x6f, 0x58, 0xa9, 0x3f, 0xe1, 0xd2, 0x07, 0xfa,
        0xe4, 0xed, 0x2f, 0x6d
    };

    ALIGN64 byte plain1[] = {
        0xcc, 0x38, 0xbc, 0xcd, 0x6b, 0xc5, 0x36, 0xad,
        0x91, 0x9b, 0x13, 0x95, 0xf5, 0xd6, 0x38, 0x01,
        0xf9, 0x9f, 0x80, 0x68, 0xd6, 0x5c, 0xa5, 0xac,
        0x63, 0x87, 0x2d, 0xaf, 0x16, 0xb9, 0x39, 0x01
    };

    byte aad1[] = {
        0x02, 0x1f, 0xaf, 0xd2, 0x38, 0x46, 0x39, 0x73,
        0xff, 0xe8, 0x02, 0x56, 0xe5, 0xb1, 0xc6, 0xb1
    };

    ALIGN64 byte cipher1[] = {
        0xdf, 0xce, 0x4e, 0x9c, 0xd2, 0x91, 0x10, 0x3d,
        0x7f, 0xe4, 0xe6, 0x33, 0x51, 0xd9, 0xe7, 0x9d,
        0x3d, 0xfd, 0x39, 0x1e, 0x32, 0x67, 0x10, 0x46,
        0x58, 0x21, 0x2d, 0xa9, 0x65, 0x21, 0xb7, 0xdb
    };

    byte tag1[] = {
        0x54, 0x24, 0x65, 0xef, 0x59, 0x93, 0x16, 0xf7,
        0x3a, 0x7a, 0x56, 0x05, 0x09, 0xa2, 0xd9, 0xf2
    };

    byte key2[] = {
        0x01, 0x6d, 0xbb, 0x38, 0xda, 0xa7, 0x6d, 0xfe,
        0x7d, 0xa3, 0x84, 0xeb, 0xf1, 0x24, 0x03, 0x64
    };

    byte iv2[] = {
        0x07, 0x93, 0xef, 0x3a, 0xda, 0x78, 0x2f, 0x78,
        0xc9, 0x8a, 0xff, 0xe3
    };

    ALIGN64 byte plain2[] = {
        0x4b, 0x34, 0xa9, 0xec, 0x57, 0x63, 0x52, 0x4b,
        0x19, 0x1d, 0x56, 0x16, 0xc5, 0x47, 0xf6, 0xb7
    };

    ALIGN64 byte cipher2[] = {
        0x60, 0x9a, 0xa3, 0xf4, 0x54, 0x1b, 0xc0, 0xfe,
        0x99, 0x31, 0xda, 0xad, 0x2e, 0xe1, 0x5d, 0x0c
    };

    byte tag2[] = {
        0x33, 0xaf, 0xec, 0x59, 0xc4, 0x5b, 0xaf, 0x68,
        0x9a, 0x5e, 0x1b, 0x13, 0xae, 0x42, 0x36, 0x19
    };

    byte key3[] = {
        0xb0, 0x1e, 0x45, 0xcc, 0x30, 0x88, 0xaa, 0xba,
        0x9f, 0xa4, 0x3d, 0x81, 0xd4, 0x81, 0x82, 0x3f
    };

    byte iv3[] = {
        0x5a, 0x2c, 0x4a, 0x66, 0x46, 0x87, 0x13, 0x45,
        0x6a, 0x4b, 0xd5, 0xe1
    };

    byte tag3[] = {
        0x01, 0x42, 0x80, 0xf9, 0x44, 0xf5, 0x3c, 0x68,
        0x11, 0x64, 0xb2, 0xff
    };

    wc_test_ret_t ret;
    ret = aesgcm_default_test_helper(key1, sizeof(key1), iv1, sizeof(iv1),
            plain1, sizeof(plain1), cipher1, sizeof(cipher1),
            aad1, sizeof(aad1), tag1, sizeof(tag1));
    if (ret != 0) {
        return ret;
    }
    ret = aesgcm_default_test_helper(key2, sizeof(key2), iv2, sizeof(iv2),
            plain2, sizeof(plain2), cipher2, sizeof(cipher2),
            NULL, 0, tag2, sizeof(tag2));
    if (ret != 0) {
        return ret;
    }
    ret = aesgcm_default_test_helper(key3, sizeof(key3), iv3, sizeof(iv3),
            NULL, 0, NULL, 0,
            NULL, 0, tag3, sizeof(tag3));
    if (ret != 0) {
        return ret;
    }
#endif

    return 0;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aesgcm_test(void)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *enc = NULL;
    Aes *dec = NULL;
#else
    Aes enc[1];
    Aes dec[1];
#endif

    /*
     * This is Test Case 16 from the document Galois/
     * Counter Mode of Operation (GCM) by McGrew and
     * Viega.
     */
    WOLFSSL_SMALL_STACK_STATIC const byte p[] =
    {
        0xd9, 0x31, 0x32, 0x25, 0xf8, 0x84, 0x06, 0xe5,
        0xa5, 0x59, 0x09, 0xc5, 0xaf, 0xf5, 0x26, 0x9a,
        0x86, 0xa7, 0xa9, 0x53, 0x15, 0x34, 0xf7, 0xda,
        0x2e, 0x4c, 0x30, 0x3d, 0x8a, 0x31, 0x8a, 0x72,
        0x1c, 0x3c, 0x0c, 0x95, 0x95, 0x68, 0x09, 0x53,
        0x2f, 0xcf, 0x0e, 0x24, 0x49, 0xa6, 0xb5, 0x25,
        0xb1, 0x6a, 0xed, 0xf5, 0xaa, 0x0d, 0xe6, 0x57,
        0xba, 0x63, 0x7b, 0x39
    };

#if defined(WOLFSSL_AES_256) || defined(WOLFSSL_AES_192)
    WOLFSSL_SMALL_STACK_STATIC const byte a[] =
    {
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xfe, 0xed, 0xfa, 0xce, 0xde, 0xad, 0xbe, 0xef,
        0xab, 0xad, 0xda, 0xd2
    };
#endif

#ifdef WOLFSSL_AES_256
#ifdef HAVE_RENESAS_SYNC
    const byte *k1 = (byte*)guser_PKCbInfo.wrapped_key_aes256;
    int k1Sz = (int)(256/8);
#else
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] =
    {
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
        0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
        0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
    };
    int k1Sz = (int)sizeof(k1);
#endif
    WOLFSSL_SMALL_STACK_STATIC const byte iv1[] =
    {
        0xca, 0xfe, 0xba, 0xbe, 0xfa, 0xce, 0xdb, 0xad,
        0xde, 0xca, 0xf8, 0x88
    };
#endif /* WOLFSSL_AES_256 */

#if defined(WOLFSSL_AES_256) || defined(WOLFSSL_AES_192)
    WOLFSSL_SMALL_STACK_STATIC const byte c1[] =
    {
        0x52, 0x2d, 0xc1, 0xf0, 0x99, 0x56, 0x7d, 0x07,
        0xf4, 0x7f, 0x37, 0xa3, 0x2a, 0x84, 0x42, 0x7d,
        0x64, 0x3a, 0x8c, 0xdc, 0xbf, 0xe5, 0xc0, 0xc9,
        0x75, 0x98, 0xa2, 0xbd, 0x25, 0x55, 0xd1, 0xaa,
        0x8c, 0xb0, 0x8e, 0x48, 0x59, 0x0d, 0xbb, 0x3d,
        0xa7, 0xb0, 0x8b, 0x10, 0x56, 0x82, 0x88, 0x38,
        0xc5, 0xf6, 0x1e, 0x63, 0x93, 0xba, 0x7a, 0x0a,
        0xbc, 0xc9, 0xf6, 0x62
    };
#endif /* WOLFSSL_AES_256 || WOLFSSL_AES_192 */

    WOLFSSL_SMALL_STACK_STATIC const byte t1[] =
    {
        0x76, 0xfc, 0x6e, 0xce, 0x0f, 0x4e, 0x17, 0x68,
        0xcd, 0xdf, 0x88, 0x53, 0xbb, 0x2d, 0x55, 0x1b
    };

    /* FIPS, QAT and PIC32MZ HW Crypto only support 12-byte IV */
#if !defined(HAVE_FIPS) && \
        !defined(WOLFSSL_PIC32MZ_CRYPT) && \
        !defined(FREESCALE_LTC) && !defined(FREESCALE_MMCAU) && \
        !defined(WOLFSSL_XILINX_CRYPT) && !defined(WOLFSSL_AFALG_XILINX_AES) && \
        !defined(WOLFSSL_SILABS_SE_ACCEL) && !defined(WOLFSSL_KCAPI_AES) && \
        !(defined(WOLF_CRYPTO_CB) && \
            (defined(HAVE_INTEL_QA_SYNC) || defined(HAVE_CAVIUM_OCTEON_SYNC)))

    #define ENABLE_NON_12BYTE_IV_TEST
#ifdef WOLFSSL_AES_192
    /* Test Case 12, uses same plaintext and AAD data. */
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] =
    {
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
        0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08,
        0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte iv2[] =
    {
        0x93, 0x13, 0x22, 0x5d, 0xf8, 0x84, 0x06, 0xe5,
        0x55, 0x90, 0x9c, 0x5a, 0xff, 0x52, 0x69, 0xaa,
        0x6a, 0x7a, 0x95, 0x38, 0x53, 0x4f, 0x7d, 0xa1,
        0xe4, 0xc3, 0x03, 0xd2, 0xa3, 0x18, 0xa7, 0x28,
        0xc3, 0xc0, 0xc9, 0x51, 0x56, 0x80, 0x95, 0x39,
        0xfc, 0xf0, 0xe2, 0x42, 0x9a, 0x6b, 0x52, 0x54,
        0x16, 0xae, 0xdb, 0xf5, 0xa0, 0xde, 0x6a, 0x57,
        0xa6, 0x37, 0xb3, 0x9b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte c2[] =
    {
        0xd2, 0x7e, 0x88, 0x68, 0x1c, 0xe3, 0x24, 0x3c,
        0x48, 0x30, 0x16, 0x5a, 0x8f, 0xdc, 0xf9, 0xff,
        0x1d, 0xe9, 0xa1, 0xd8, 0xe6, 0xb4, 0x47, 0xef,
        0x6e, 0xf7, 0xb7, 0x98, 0x28, 0x66, 0x6e, 0x45,
        0x81, 0xe7, 0x90, 0x12, 0xaf, 0x34, 0xdd, 0xd9,
        0xe2, 0xf0, 0x37, 0x58, 0x9b, 0x29, 0x2d, 0xb3,
        0xe6, 0x7c, 0x03, 0x67, 0x45, 0xfa, 0x22, 0xe7,
        0xe9, 0xb7, 0x37, 0x3b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte t2[] =
    {
        0xdc, 0xf5, 0x66, 0xff, 0x29, 0x1c, 0x25, 0xbb,
        0xb8, 0x56, 0x8f, 0xc3, 0xd3, 0x76, 0xa6, 0xd9
    };
#endif /* WOLFSSL_AES_192 */
#ifdef WOLFSSL_AES_128
    /* The following is an interesting test case from the example
     * FIPS test vectors for AES-GCM. IVlen = 1 byte */
    WOLFSSL_SMALL_STACK_STATIC const byte p3[] =
    {
        0x57, 0xce, 0x45, 0x1f, 0xa5, 0xe2, 0x35, 0xa5,
        0x8e, 0x1a, 0xa2, 0x3b, 0x77, 0xcb, 0xaf, 0xe2
    };

#ifdef HAVE_RENESAS_SYNC
    const byte *k3 =
        (byte*)guser_PKCbInfo.wrapped_key_aes128;
    int k3Sz = (int)(128/8);
#else
    WOLFSSL_SMALL_STACK_STATIC const byte k3[] =
    {
        0xbb, 0x01, 0xd7, 0x03, 0x81, 0x1c, 0x10, 0x1a,
        0x35, 0xe0, 0xff, 0xd2, 0x91, 0xba, 0xf2, 0x4b
    };
    int k3Sz = (int)sizeof(k3);
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte iv3[] =
    {
        0xca
    };

    WOLFSSL_SMALL_STACK_STATIC const byte c3[] =
    {
        0x6b, 0x5f, 0xb3, 0x9d, 0xc1, 0xc5, 0x7a, 0x4f,
        0xf3, 0x51, 0x4d, 0xc2, 0xd5, 0xf0, 0xd0, 0x07
    };

    WOLFSSL_SMALL_STACK_STATIC const byte a3[] =
    {
        0x40, 0xfc, 0xdc, 0xd7, 0x4a, 0xd7, 0x8b, 0xf1,
        0x3e, 0x7c, 0x60, 0x55, 0x50, 0x51, 0xdd, 0x54
    };

    WOLFSSL_SMALL_STACK_STATIC const byte t3[] =
    {
        0x06, 0x90, 0xed, 0x01, 0x34, 0xdd, 0xc6, 0x95,
        0x31, 0x2e, 0x2a, 0xf9, 0x57, 0x7a, 0x1e, 0xa6
    };
#endif /* WOLFSSL_AES_128 */
#ifdef WOLFSSL_AES_256
    int ivlen;
#endif
#endif

    byte resultT[sizeof(t1) + AES_BLOCK_SIZE];
    byte resultP[sizeof(p) + AES_BLOCK_SIZE];
    byte resultC[sizeof(p) + AES_BLOCK_SIZE];
    wc_test_ret_t ret = 0;
#ifdef WOLFSSL_AES_256
    #if !(defined(WOLF_CRYPTO_CB) && defined(HAVE_INTEL_QA_SYNC))
    int  alen;
    #endif
    #if !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT)
    int  plen;
    #endif
#endif
#if defined(WOLFSSL_XILINX_CRYPT_VERSAL)
    byte buf[sizeof(p) + AES_BLOCK_SIZE];
    byte bufA[sizeof(a) + 1];
    byte *large_aad = (byte*)XMALLOC((size_t)1024 + 16, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

#if !defined(BENCH_EMBEDDED) && !defined(HAVE_CAVIUM)
    #if !defined(BENCH_AESGCM_LARGE)
        #define BENCH_AESGCM_LARGE 1024
    #endif

#ifndef WOLFSSL_NO_MALLOC
    byte *large_input = (byte *)XMALLOC(BENCH_AESGCM_LARGE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *large_output = (byte *)XMALLOC(BENCH_AESGCM_LARGE + AES_BLOCK_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *large_outdec = (byte *)XMALLOC(BENCH_AESGCM_LARGE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((! large_input) || (! large_output) || (! large_outdec))
        ERROR_OUT(MEMORY_E, out);

#else
    byte large_input[BENCH_AESGCM_LARGE];
    byte large_output[BENCH_AESGCM_LARGE];
    byte large_outdec[BENCH_AESGCM_LARGE];
#endif

    XMEMSET(large_input, 0, BENCH_AESGCM_LARGE);
    XMEMSET(large_output, 0, BENCH_AESGCM_LARGE + AES_BLOCK_SIZE);
    XMEMSET(large_outdec, 0, BENCH_AESGCM_LARGE);
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
    if ((dec = (Aes *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
#endif

    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesInit(dec, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

#ifdef WOLFSSL_AES_256
    ret = wc_AesGcmSetKey(enc, k1, k1Sz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv1, sizeof(iv1),
                                        resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifndef HAVE_RENESAS_SYNC
    if (XMEMCMP(c1, resultC, sizeof(c1)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(t1, resultT, sizeof(t1)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmSetKey(dec, k1, k1Sz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesGcmDecrypt(dec, resultP, resultC, sizeof(c1),
                      iv1, sizeof(iv1), resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p, resultP, sizeof(p)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */

    /* Large buffer test */
#ifdef BENCH_AESGCM_LARGE
    /* setup test buffer */
    for (alen=0; alen<BENCH_AESGCM_LARGE; alen++)
        large_input[alen] = (byte)alen;

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesGcmEncrypt(enc, large_output, large_input,
                              BENCH_AESGCM_LARGE, iv1, sizeof(iv1),
                              resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmDecrypt(dec, large_outdec, large_output,
                              BENCH_AESGCM_LARGE, iv1, sizeof(iv1), resultT,
                              sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (XMEMCMP(large_input, large_outdec, BENCH_AESGCM_LARGE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif /* HAVE_AES_DECRYPT */
#endif /* BENCH_AESGCM_LARGE */
#if defined(ENABLE_NON_12BYTE_IV_TEST) && defined(WOLFSSL_AES_256)
    /* Variable IV length test */
    for (ivlen=1; ivlen<k1Sz; ivlen++) {
         /* AES-GCM encrypt and decrypt both use AES encrypt internally */
         ret = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), k1,
                         (word32)ivlen, resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
        ret = wc_AesGcmDecrypt(dec, resultP, resultC, sizeof(c1), k1,
                         (word32)ivlen, resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif /* HAVE_AES_DECRYPT */
    }
#endif

#if !(defined(WOLF_CRYPTO_CB) && defined(HAVE_INTEL_QA_SYNC))
    /* Variable authenticated data length test */
    for (alen=0; alen<(int)sizeof(p); alen++) {
         /* AES-GCM encrypt and decrypt both use AES encrypt internally */
         ret = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv1,
                        sizeof(iv1), resultT, sizeof(t1), p, (word32)alen);
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
        ret = wc_AesGcmDecrypt(dec, resultP, resultC, sizeof(c1), iv1,
                        sizeof(iv1), resultT, sizeof(t1), p, (word32)alen);
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif /* HAVE_AES_DECRYPT */
    }
#if defined(WOLFSSL_XILINX_CRYPT_VERSAL)
    if (! large_aad)
        ERROR_OUT(MEMORY_E, out);
    XMEMSET(large_aad, 0, 1024+16);
    /* Variable authenticated data length test */
    for (alen=0; alen<=1024; alen+=16) {
         /* AES-GCM encrypt and decrypt both use AES encrypt internally */
         ret = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv1,
                        sizeof(iv1), resultT, sizeof(t1), large_aad, (word32)alen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
        ret = wc_AesGcmDecrypt(dec, resultP, resultC, sizeof(c1), iv1,
                        sizeof(iv1), resultT, sizeof(t1), large_aad, (word32)alen);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif /* HAVE_AES_DECRYPT */
    }

    /* Test unaligned memory of all potential arguments */
    ret = wc_AesGcmSetKey(enc, k1, k1Sz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    XMEMCPY(&buf[1], p, sizeof(p));
    XMEMCPY(&bufA[1], a, sizeof(a));
    ret = wc_AesGcmEncrypt(enc, &resultC[1], &buf[1], sizeof(p), iv1, sizeof(iv1),
                                        &resultT[1], sizeof(t1), &bufA[1], sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c1, &resultC[1], sizeof(c1)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(t1, &resultT[1], sizeof(t1)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmSetKey(dec, k1, k1Sz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesGcmDecrypt(dec, &resultP[1], &resultC[1], sizeof(c1),
                      iv1, sizeof(iv1), &resultT[1], sizeof(t1), &bufA[1], sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p, &resultP[1], sizeof(p)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */

#endif /* Xilinx Versal */
#endif

#if !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT)
#ifdef BENCH_AESGCM_LARGE
    /* Variable plain text length test */
    for (plen=1; plen<BENCH_AESGCM_LARGE; plen++) {
        /* AES-GCM encrypt and decrypt both use AES encrypt internally */
        ret = wc_AesGcmEncrypt(enc, large_output, large_input,
                                  plen, iv1, sizeof(iv1), resultT,
                                  sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

#ifdef HAVE_AES_DECRYPT
        ret = wc_AesGcmDecrypt(dec, large_outdec, large_output,
                                  plen, iv1, sizeof(iv1), resultT,
                                  sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif /* HAVE_AES_DECRYPT */
    }
#else /* BENCH_AESGCM_LARGE */
    /* Variable plain text length test */
    for (plen=1; plen<(int)sizeof(p); plen++) {
         /* AES-GCM encrypt and decrypt both use AES encrypt internally */
         ret = wc_AesGcmEncrypt(enc, resultC, p, (word32)plen, iv1,
                           sizeof(iv1), resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
        ret = wc_AesGcmDecrypt(dec, resultP, resultC, (word32)plen, iv1,
                           sizeof(iv1), resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#endif /* HAVE_AES_DECRYPT */
    }
#endif /* BENCH_AESGCM_LARGE */
#endif
#endif /* WOLFSSL_AES_256 */

    /* test with IV != 12 bytes */
#ifdef ENABLE_NON_12BYTE_IV_TEST
    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));

#ifdef WOLFSSL_AES_192
    wc_AesGcmSetKey(enc, k2, sizeof(k2));
    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv2, sizeof(iv2),
                                        resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c2, resultC, sizeof(c2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(t2, resultT, sizeof(t1)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmDecrypt(enc, resultP, resultC, sizeof(c1),
                      iv2, sizeof(iv2), resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p, resultP, sizeof(p)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */

    /* Large buffer test */
#ifdef BENCH_AESGCM_LARGE
    wc_AesGcmSetKey(enc, k2, k3Sz);
    wc_AesGcmSetKey(dec, k2, k3Sz);
    /* setup test buffer */
    for (alen=0; alen<BENCH_AESGCM_LARGE; alen++)
        large_input[alen] = (byte)alen;

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesGcmEncrypt(enc, large_output, large_input,
                              BENCH_AESGCM_LARGE, iv1, sizeof(iv1),
                              resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmDecrypt(dec, large_outdec, large_output,
                              BENCH_AESGCM_LARGE, iv1, sizeof(iv1), resultT,
                              sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(large_input, large_outdec, BENCH_AESGCM_LARGE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */
#endif /* BENCH_AESGCM_LARGE */

    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));
#endif /* WOLFSSL_AES_192 */
#ifdef WOLFSSL_AES_128
    wc_AesGcmSetKey(enc, k3, k3Sz);
    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesGcmEncrypt(enc, resultC, p3, sizeof(p3), iv3, sizeof(iv3),
                                        resultT, sizeof(t3), a3, sizeof(a3));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifndef HAVE_RENESAS_SYNC
    if (XMEMCMP(c3, resultC, sizeof(c3)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(t3, resultT, sizeof(t3)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmDecrypt(enc, resultP, resultC, sizeof(c3),
                      iv3, sizeof(iv3), resultT, sizeof(t3), a3, sizeof(a3));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p3, resultP, sizeof(p3)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */

    /* Large buffer test */
#ifdef BENCH_AESGCM_LARGE
    wc_AesGcmSetKey(enc, k3, k3Sz);
    wc_AesGcmSetKey(dec, k3, k3Sz);
    /* setup test buffer */
    for (alen=0; alen<BENCH_AESGCM_LARGE; alen++)
        large_input[alen] = (byte)alen;

    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesGcmEncrypt(enc, large_output, large_input,
                              BENCH_AESGCM_LARGE, iv1, sizeof(iv1),
                              resultT, sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmDecrypt(dec, large_outdec, large_output,
                              BENCH_AESGCM_LARGE, iv1, sizeof(iv1), resultT,
                              sizeof(t1), a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(large_input, large_outdec, BENCH_AESGCM_LARGE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */
#endif /* BENCH_AESGCM_LARGE */
#endif /* WOLFSSL_AES_128 */
#endif /* ENABLE_NON_12BYTE_IV_TEST */

#if defined(WOLFSSL_AES_256) && !defined(WOLFSSL_AFALG_XILINX_AES) && \
    !defined(WOLFSSL_XILINX_CRYPT) && \
    !(defined(WOLF_CRYPTO_CB) && \
        defined(HAVE_INTEL_QA_SYNC) || defined(HAVE_CAVIUM_OCTEON_SYNC))
    XMEMSET(resultT, 0, sizeof(resultT));
    XMEMSET(resultC, 0, sizeof(resultC));
    XMEMSET(resultP, 0, sizeof(resultP));

    wc_AesGcmSetKey(enc, k1, k1Sz);
    /* AES-GCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesGcmEncrypt(enc, resultC, p, sizeof(p), iv1, sizeof(iv1),
                                resultT + 1, sizeof(t1) - 1, a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifndef HAVE_RENESAS_SYNC
    if (XMEMCMP(c1, resultC, sizeof(c1)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(t1, resultT + 1, sizeof(t1) - 1))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmDecrypt(enc, resultP, resultC, sizeof(p),
              iv1, sizeof(iv1), resultT + 1, sizeof(t1) - 1, a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p, resultP, sizeof(p)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */
#endif /* WOLFSSL_AES_256 */

#if !defined(HAVE_FIPS) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))
    /* Test encrypt with internally generated IV */
#if defined(WOLFSSL_AES_256) && !(defined(WC_NO_RNG) || defined(HAVE_SELFTEST)) \
    && !(defined(WOLF_CRYPTO_CB) && defined(HAVE_CAVIUM_OCTEON_SYNC))
    {
        WC_RNG rng;
        byte randIV[12];

        ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(randIV, 0, sizeof(randIV));
        XMEMSET(resultT, 0, sizeof(resultT));
        XMEMSET(resultC, 0, sizeof(resultC));
        XMEMSET(resultP, 0, sizeof(resultP));

        wc_AesGcmSetKey(enc, k1, k1Sz);
        ret = wc_AesGcmSetIV(enc, sizeof(randIV), NULL, 0, &rng);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        ret = wc_AesGcmEncrypt_ex(enc,
                        resultC, p, sizeof(p),
                        randIV, sizeof(randIV),
                        resultT, sizeof(t1),
                        a, sizeof(a));
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &enc->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        /* Check the IV has been set. */
        {
            word32 i, ivSum = 0;

            for (i = 0; i < sizeof(randIV); i++)
                ivSum += randIV[i];
            if (ivSum == 0)
                ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        }

#ifdef HAVE_AES_DECRYPT
        wc_AesGcmSetKey(dec, k1, k1Sz);
        ret = wc_AesGcmSetIV(dec, sizeof(randIV), NULL, 0, &rng);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        ret = wc_AesGcmDecrypt(dec,
                          resultP, resultC, sizeof(c1),
                          randIV, sizeof(randIV),
                          resultT, sizeof(t1),
                          a, sizeof(a));
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &dec->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(p, resultP, sizeof(p)))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */

        wc_FreeRng(&rng);
    }
#endif /* WOLFSSL_AES_256 && !(WC_NO_RNG || HAVE_SELFTEST) */
#endif /* HAVE_FIPS_VERSION >= 2 */

#if !defined(WOLFSSL_AFALG_XILINX_AES) && !defined(WOLFSSL_XILINX_CRYPT)
#ifdef WOLFSSL_AES_256
#ifdef WOLFSSL_AESGCM_STREAM
    ret = wc_AesGcmEncryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGcmEncryptUpdate(enc, resultC, p, sizeof(p), a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGcmEncryptFinal(enc, resultT, sizeof(t1));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(resultC, c1, sizeof(c1)) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(resultT, t1, sizeof(t1)) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmDecryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGcmDecryptUpdate(enc, resultP, c1, sizeof(c1), a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGcmDecryptFinal(enc, t1, sizeof(t1));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(resultP, p, sizeof(p)) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

    /* alen is the size to pass in with each update. */
    for (alen = 1; alen < AES_BLOCK_SIZE + 1; alen++) {
        ret = wc_AesGcmEncryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        /* plen is the offset into AAD to update with. */
        for (plen = 0; plen < (int)sizeof(a); plen += alen)  {
            int len = sizeof(a) - plen;
            if (len > alen) len = alen;
            ret = wc_AesGcmEncryptUpdate(enc, NULL, NULL, 0, a + plen, len);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        }
        /* plen is the offset into plaintext to update with. */
        for (plen = 0; plen < (int)sizeof(p); plen += alen)  {
            int len = sizeof(p) - plen;
            if (len > alen) len = alen;
            ret = wc_AesGcmEncryptUpdate(enc, resultC + plen, p + plen, len,
                NULL, 0);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        }
        ret = wc_AesGcmEncryptFinal(enc, resultT, sizeof(t1));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(resultC, c1, sizeof(c1)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
        if (XMEMCMP(resultT, t1, sizeof(t1)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }

#ifdef HAVE_AES_DECRYPT
    for (alen = 1; alen < AES_BLOCK_SIZE + 1; alen++) {
        ret = wc_AesGcmDecryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        /* plen is the offset into AAD to update with. */
        for (plen = 0; plen < (int)sizeof(a); plen += alen)  {
            int len = sizeof(a) - plen;
            if (len > alen) len = alen;
            ret = wc_AesGcmDecryptUpdate(enc, NULL, NULL, 0, a + plen, len);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        }
        /* plen is the offset into cipher text to update with. */
        for (plen = 0; plen < (int)sizeof(c1); plen += alen)  {
            int len = sizeof(c1) - plen;
            if (len > alen) len = alen;
            ret = wc_AesGcmDecryptUpdate(enc, resultP + plen, c1 + plen, len,
                NULL, 0);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        }
        ret = wc_AesGcmDecryptFinal(enc, t1, sizeof(t1));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(resultP, p, sizeof(p)) != 0)
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
#endif /* HAVE_AES_DECRYPT */
#ifdef BENCH_AESGCM_LARGE
    /* setup test buffer */
    ret = wc_AesGcmEncryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGcmEncryptUpdate(enc, large_output, large_input,
                                    BENCH_AESGCM_LARGE, a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGcmEncryptFinal(enc, resultT, sizeof(t1));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
#ifdef HAVE_AES_DECRYPT
    ret = wc_AesGcmDecryptInit(enc, k1, sizeof(k1), iv1, sizeof(iv1));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGcmDecryptUpdate(enc, large_outdec, large_output,
                                    BENCH_AESGCM_LARGE, a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesGcmDecryptFinal(enc, resultT, sizeof(t1));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(large_input, large_outdec, BENCH_AESGCM_LARGE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif /* HAVE_AES_DECRYPT */
#endif /* BENCH_AESGCM_LARGE */
#endif /* WOLFSSL_AESGCM_STREAM */
#endif /* WOLFSSL_AES_256 */
#endif /* !WOLFSSL_AFALG_XILINX_AES && !WOLFSSL_XILINX_CRYPT */

    wc_AesFree(enc);
    wc_AesFree(dec);

    ret = 0;

  out:

#if !defined(BENCH_EMBEDDED) && !defined(HAVE_CAVIUM) && \
    !defined(WOLFSSL_NO_MALLOC)
    if (large_input)
        XFREE(large_input, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (large_output)
        XFREE(large_output, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (large_outdec)
        XFREE(large_outdec, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif


#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (enc)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
    if (dec)
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}

#ifdef WOLFSSL_AES_128
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t gmac_test(void)
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Gmac *gmac;
#else
    Gmac gmac[1];
#endif

    WOLFSSL_SMALL_STACK_STATIC const byte k1[] =
    {
        0x89, 0xc9, 0x49, 0xe9, 0xc8, 0x04, 0xaf, 0x01,
        0x4d, 0x56, 0x04, 0xb3, 0x94, 0x59, 0xf2, 0xc8
    };
    WOLFSSL_SMALL_STACK_STATIC const byte iv1[] =
    {
        0xd1, 0xb1, 0x04, 0xc8, 0x15, 0xbf, 0x1e, 0x94,
        0xe2, 0x8c, 0x8f, 0x16
    };
    WOLFSSL_SMALL_STACK_STATIC const byte a1[] =
    {
       0x82, 0xad, 0xcd, 0x63, 0x8d, 0x3f, 0xa9, 0xd9,
       0xf3, 0xe8, 0x41, 0x00, 0xd6, 0x1e, 0x07, 0x77
    };
    WOLFSSL_SMALL_STACK_STATIC const byte t1[] =
    {
        0x88, 0xdb, 0x9d, 0x62, 0x17, 0x2e, 0xd0, 0x43,
        0xaa, 0x10, 0xf1, 0x6d, 0x22, 0x7d, 0xc4, 0x1b
    };

#if (!defined(HAVE_FIPS) ||                                             \
     (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2)))

    /* FIPS builds only allow 16-byte auth tags. */
    /* This sample uses a 15-byte auth tag. */
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] =
    {
        0x40, 0xf7, 0xec, 0xb2, 0x52, 0x6d, 0xaa, 0xd4,
        0x74, 0x25, 0x1d, 0xf4, 0x88, 0x9e, 0xf6, 0x5b
    };
    WOLFSSL_SMALL_STACK_STATIC const byte iv2[] =
    {
        0xee, 0x9c, 0x6e, 0x06, 0x15, 0x45, 0x45, 0x03,
        0x1a, 0x60, 0x24, 0xa7
    };
    WOLFSSL_SMALL_STACK_STATIC const byte a2[] =
    {
        0x94, 0x81, 0x2c, 0x87, 0x07, 0x4e, 0x15, 0x18,
        0x34, 0xb8, 0x35, 0xaf, 0x1c, 0xa5, 0x7e, 0x56
    };
    WOLFSSL_SMALL_STACK_STATIC const byte t2[] =
    {
        0xc6, 0x81, 0x79, 0x8e, 0x3d, 0xda, 0xb0, 0x9f,
        0x8d, 0x83, 0xb0, 0xbb, 0x14, 0xb6, 0x91
    };
#endif

    byte tag[16];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((gmac = (Gmac *)XMALLOC(sizeof *gmac, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        return WC_TEST_RET_ENC_ERRNO;
#endif

    XMEMSET(gmac, 0, sizeof *gmac); /* clear context */
    (void)wc_AesInit(&gmac->aes, HEAP_HINT, INVALID_DEVID); /* Make sure devId updated */
    XMEMSET(tag, 0, sizeof(tag));
    wc_GmacSetKey(gmac, k1, sizeof(k1));
    wc_GmacUpdate(gmac, iv1, sizeof(iv1), a1, sizeof(a1), tag, sizeof(t1));
    if (XMEMCMP(t1, tag, sizeof(t1)) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if (!defined(HAVE_FIPS) ||                                  \
     (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2)) )

    XMEMSET(tag, 0, sizeof(tag));
    wc_GmacSetKey(gmac, k2, sizeof(k2));
    wc_GmacUpdate(gmac, iv2, sizeof(iv2), a2, sizeof(a2), tag, sizeof(t2));
    if (XMEMCMP(t2, tag, sizeof(t2)) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

#if !defined(WC_NO_RNG) && !defined(HAVE_SELFTEST) && !defined(NO_AES_DECRYPT)
    {
        WOLFSSL_SMALL_STACK_STATIC const byte badT[] =
        {
            0xde, 0xad, 0xbe, 0xef, 0x17, 0x2e, 0xd0, 0x43,
            0xaa, 0x10, 0xf1, 0x6d, 0x22, 0x7d, 0xc4, 0x1b
        };

        WC_RNG rng;
        byte iv[12];

        #ifndef HAVE_FIPS
        ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        #else
        ret = wc_InitRng(&rng);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        #endif

        ret = wc_GmacVerify(k1, sizeof(k1), iv1, sizeof(iv1), a1, sizeof(a1),
                    t1, sizeof(t1));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        ret = wc_GmacVerify(k1, sizeof(k1), iv1, sizeof(iv1), a1, sizeof(a1),
                    badT, sizeof(badT));
        if (ret != AES_GCM_AUTH_E)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        ret = wc_GmacVerify(k2, sizeof(k2), iv2, sizeof(iv2), a2, sizeof(a2),
                    t2, sizeof(t2));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

        XMEMSET(tag, 0, sizeof(tag));
        XMEMSET(iv, 0, sizeof(iv));
        ret = wc_Gmac(k1, sizeof(k1), iv, sizeof(iv), a1, sizeof(a1),
                    tag, sizeof(tag), &rng);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        ret = wc_GmacVerify(k1, sizeof(k1), iv, sizeof(iv), a1, sizeof(a1),
                    tag, sizeof(tag));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        wc_FreeRng(&rng);
    }
#endif /* !WC_NO_RNG && !HAVE_SELFTEST && !NO_AES_DECRYPT */
#endif /* HAVE_FIPS */

    ret = 0;

  out:
    wc_AesFree(&gmac->aes);
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XFREE(gmac, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 */
#endif /* HAVE_AESGCM */

#if defined(HAVE_AESCCM)

#if defined(WOLFSSL_AES_256)

static wc_test_ret_t aesccm_256_test(void)
{
    wc_test_ret_t ret;
    /* Test vectors from NIST AES CCM 256-bit CAST Example #1 */
    WOLFSSL_SMALL_STACK_STATIC const byte in_key[32] = {
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4A, 0x4B, 0x4C, 0x4D, 0x4E, 0x4F,
        0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
        0x58, 0x59, 0x5A, 0x5B, 0x5C, 0x5D, 0x5E, 0x5F
    };
    WOLFSSL_SMALL_STACK_STATIC const byte in_nonce[7] = {
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16};
    WOLFSSL_SMALL_STACK_STATIC const byte in_auth[8] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07};
    WOLFSSL_SMALL_STACK_STATIC const byte in_plaintext[4] = {
        0x20, 0x21, 0x22, 0x23};
    WOLFSSL_SMALL_STACK_STATIC const byte exp_ciphertext[4] = {
        0x8A, 0xB1, 0xA8, 0x74};
    WOLFSSL_SMALL_STACK_STATIC const byte exp_tag[4] = {
        0x95, 0xFC, 0x08, 0x20};
    byte output[sizeof(in_plaintext)];
    byte atag[sizeof(exp_tag)];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes* aes = (Aes*)XMALLOC(sizeof(Aes), HEAP_HINT, DYNAMIC_TYPE_AES);
    if (aes == NULL) {
        return MEMORY_E;
    }
#else
    Aes aes[1];
#endif

    ret = wc_AesInit(aes, HEAP_HINT, devId);
    if (ret == 0) {
        ret = wc_AesCcmSetKey(aes, in_key, sizeof(in_key));
    }
    if (ret == 0) {
        ret = wc_AesCcmEncrypt(aes, output, in_plaintext, sizeof(in_plaintext),
            in_nonce, sizeof(in_nonce),
            atag, sizeof(atag),
            in_auth, sizeof(in_auth));
    }
    /* Verify we produce the proper ciphertext and tag */
    if (ret == 0 &&
        (XMEMCMP(output, exp_ciphertext, sizeof(output)) ||
         XMEMCMP(atag, exp_tag, sizeof(atag)))) {
        ret = WC_TEST_RET_ENC_NC;
    }

    if (ret == 0) {
        /* decrypt inline */
        ret = wc_AesCcmDecrypt(aes, output, output, sizeof(output),
            in_nonce, sizeof(in_nonce),
            atag, sizeof(atag),
            in_auth, sizeof(in_auth));
    }

    /* Verify decryption was successful */
    if (ret == 0 &&
        XMEMCMP(output, in_plaintext, sizeof(output))) {
        ret = WC_TEST_RET_ENC_NC;
    }

    wc_AesFree(aes);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XFREE(aes, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}

#endif /* WOLFSSL_AES_256 */

#if defined(WOLFSSL_AES_128)

static wc_test_ret_t aesccm_128_test(void)
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Aes *enc;
#else
    Aes enc[1];
#endif

    /* key */
    WOLFSSL_SMALL_STACK_STATIC const byte k[] =
    {
        0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
        0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf
    };

    /* nonce */
    WOLFSSL_SMALL_STACK_STATIC const byte iv[] =
    {
        0x00, 0x00, 0x00, 0x03, 0x02, 0x01, 0x00, 0xa0,
        0xa1, 0xa2, 0xa3, 0xa4, 0xa5
    };

    /* plaintext */
    WOLFSSL_SMALL_STACK_STATIC const byte p[] =
    {
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e
    };
    /* plaintext - long */
    WOLFSSL_SMALL_STACK_STATIC const byte pl[] =
    {
        0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
        0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
        0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
        0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
        0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
        0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
        0x50
    };


    WOLFSSL_SMALL_STACK_STATIC const byte a[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };

    /* ciphertext */
    WOLFSSL_SMALL_STACK_STATIC const byte c[] =
    {
        0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
        0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
        0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84
    };
    /* tag - authentication */
    WOLFSSL_SMALL_STACK_STATIC const byte t[] =
    {
        0x17, 0xe8, 0xd1, 0x2c, 0xfd, 0xf9, 0x26, 0xe0
    };
    /* ciphertext - long */
    WOLFSSL_SMALL_STACK_STATIC const byte cl[] =
    {
        0x58, 0x8c, 0x97, 0x9a, 0x61, 0xc6, 0x63, 0xd2,
        0xf0, 0x66, 0xd0, 0xc2, 0xc0, 0xf9, 0x89, 0x80,
        0x6d, 0x5f, 0x6b, 0x61, 0xda, 0xc3, 0x84, 0xe0,
        0x44, 0x2d, 0xbe, 0x25, 0xfa, 0x48, 0x2b, 0xa8,
        0x36, 0x0b, 0xbf, 0x01, 0xc0, 0x12, 0x45, 0xa4,
        0x82, 0x9f, 0x20, 0x6c, 0xc3, 0xd6, 0xae, 0x5b,
        0x54, 0x8d, 0xd0, 0xb1, 0x69, 0x2c, 0xec, 0x5e,
        0x95, 0xa5, 0x6b, 0x48, 0xc3, 0xc6, 0xc8, 0x9e,
        0xc7, 0x92, 0x98, 0x9d, 0x26, 0x7d, 0x2a, 0x10,
        0x0b
    };
    /* tag - authentication - long */
    WOLFSSL_SMALL_STACK_STATIC const byte tl[] =
    {
        0x89, 0xd8, 0xd2, 0x02, 0xc5, 0xcf, 0xae, 0xf4
    };

    /* tag - authentication - empty plaintext */
    WOLFSSL_SMALL_STACK_STATIC const byte t_empty[] =
    {
        0xe4, 0x28, 0x8a, 0xc3, 0x78, 0x00, 0x0f, 0xf5
    };

    byte t2[sizeof(t)];
    byte p2[sizeof(p)];
    byte c2[sizeof(c)];
    byte iv2[sizeof(iv)];
    byte pl2[sizeof(pl)];
    byte cl2[sizeof(cl)];
    byte tl2[sizeof(tl)];
    byte t_empty2[sizeof(t_empty)];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((enc = (Aes *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES)) == NULL)
        return WC_TEST_RET_ENC_ERRNO;
#endif

    XMEMSET(enc, 0, sizeof *enc); /* clear context */
    XMEMSET(t2, 0, sizeof(t2));
    XMEMSET(c2, 0, sizeof(c2));
    XMEMSET(p2, 0, sizeof(p2));

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesCcmSetKey(enc, k, sizeof(k));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    /* AES-CCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesCcmEncrypt(enc, c2, p, sizeof(c2), iv, sizeof(iv),
                                                 t2, sizeof(t2), a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(c, c2, sizeof(c2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(t, t2, sizeof(t2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = wc_AesCcmDecrypt(enc, p2, c2, sizeof(p2), iv, sizeof(iv),
                                                 t2, sizeof(t2), a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(p, p2, sizeof(p2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* Test the authentication failure */
    t2[0]++; /* Corrupt the authentication tag. */
    ret = wc_AesCcmDecrypt(enc, p2, c, sizeof(p2), iv, sizeof(iv),
                                                 t2, sizeof(t2), a, sizeof(a));
    if (ret == 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* Clear c2 to compare against p2. p2 should be set to zero in case of
     * authentication fail. */
    XMEMSET(c2, 0, sizeof(c2));
    if (XMEMCMP(p2, c2, sizeof(p2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    wc_AesFree(enc);

    XMEMSET(enc, 0, sizeof(Aes)); /* clear context */
    XMEMSET(t2, 0, sizeof(t2));
    XMEMSET(c2, 0, sizeof(c2));
    XMEMSET(p2, 0, sizeof(p2));
    XMEMSET(iv2, 0, sizeof(iv2));

    ret = wc_AesInit(enc, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

#ifndef HAVE_SELFTEST
    /* selftest build does not have wc_AesCcmSetNonce() or
     * wc_AesCcmEncrypt_ex() */
    ret = wc_AesCcmSetKey(enc, k, sizeof(k));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AesCcmSetNonce(enc, iv, sizeof(iv));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesCcmEncrypt_ex(enc, c2, p, sizeof(c2), iv2, sizeof(iv2),
                            t2, sizeof(t2), a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(iv, iv2, sizeof(iv2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(c, c2, sizeof(c2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(t, t2, sizeof(t2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#endif

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    /* test fail on invalid IV sizes */
    ret = wc_AesCcmSetKey(enc, k, sizeof(k));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    /* AES-CCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesCcmEncrypt(enc, c2, p, sizeof(c2), iv, sizeof(iv),
            t2, 1, a, sizeof(a));
    if (ret == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
#endif

    /* AES-CCM encrypt and decrypt both use AES encrypt internally */
    ret = wc_AesCcmEncrypt(enc, cl2, pl, sizeof(cl2), iv, sizeof(iv),
                                                tl2, sizeof(tl2), a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(cl, cl2, sizeof(cl2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    if (XMEMCMP(tl, tl2, sizeof(tl2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = wc_AesCcmDecrypt(enc, pl2, cl2, sizeof(pl2), iv, sizeof(iv),
                                                tl2, sizeof(tl2), a, sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(pl, pl2, sizeof(pl2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    /* test empty message as null input or output with nonzero inSz. */
    ret = wc_AesCcmEncrypt(enc, pl2 /* out */, NULL /* in */, 1 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2),
                              a, sizeof(a));
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesCcmEncrypt(enc, NULL /* out */, (const byte *)"" /* in */, 1 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2),
                              a, sizeof(a));
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesCcmDecrypt(enc, pl2, NULL /* in */, 1 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2), a,
                              sizeof(a));
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    ret = wc_AesCcmDecrypt(enc, NULL /* out */, (const byte *)"" /* in */, 1 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2), a,
                              sizeof(a));
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    /* test empty message as null input and output with zero inSz --
     * must either succeed, or fail early with BAD_FUNC_ARG.
     */
    ret = wc_AesCcmEncrypt(enc, NULL /* out */, NULL /* in */, 0 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2),
                              a, sizeof(a));
    if (ret != BAD_FUNC_ARG) {
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
        if (XMEMCMP(t_empty, t_empty2, sizeof(t_empty2)))
            ERROR_OUT(WC_TEST_RET_ENC_NC, out);

        ret = wc_AesCcmDecrypt(enc, NULL /* out */, NULL /* in */,
                                  0 /* inSz */, iv, sizeof(iv), t_empty2,
                                  sizeof(t_empty2), a, sizeof(a));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    }

    /* test empty message as zero-length string -- must work. */
    ret = wc_AesCcmEncrypt(enc, pl2, (const byte *)"", 0 /* inSz */, iv,
                              sizeof(iv), t_empty2, sizeof(t_empty2), a,
                              sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (XMEMCMP(t_empty, t_empty2, sizeof(t_empty2)))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = wc_AesCcmDecrypt(enc, pl2, (const byte *)"", 0 /* inSz */,
                              iv, sizeof(iv), t_empty2, sizeof(t_empty2), a,
                              sizeof(a));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    wc_AesFree(enc);

    ret = 0;

  out:
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif

    return ret;
}
#endif /* WOLFSSL_AES_128 */

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aesccm_test(void)
{
    wc_test_ret_t ret = 0;
#ifdef WOLFSSL_AES_128
    if (ret == 0)
        ret = aesccm_128_test();
#endif
#ifdef WOLFSSL_AES_256
    if (ret == 0)
        ret = aesccm_256_test();
#endif
    return ret;
}
#endif /* HAVE_AESCCM */


#if defined(WOLFSSL_AES_EAX) && \
    (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aes_eax_test(void)
{
    typedef struct {
        byte key[AES_256_KEY_SIZE];
        int key_length;
        byte iv[AES_BLOCK_SIZE];
        int iv_length;
        byte aad[AES_BLOCK_SIZE * 2];
        int aad_length;
        byte msg[AES_BLOCK_SIZE * 2];
        int msg_length;
        byte ct[AES_BLOCK_SIZE * 2];
        int ct_length;
        byte tag[AES_BLOCK_SIZE];
        int tag_length;
        int valid;
    } AadVector;

    /*  A small selection of Google wycheproof vectors that use vectors
     *  from the original paper: eprint.iacr.org/2003/069
     *  https://github.com/google/wycheproof/blob/master/testvectors/aes_eax_test.json
     */
    WOLFSSL_SMALL_STACK_STATIC const AadVector vectors[] = {
        /* Vector from paper - empty message with auth data */
        {
            /* key, key length  */
            {0x23, 0x39, 0x52, 0xde, 0xe4, 0xd5, 0xed, 0x5f,
             0x9b, 0x9c, 0x6d, 0x6f, 0xf8, 0x0f, 0xf4, 0x78}, 16,
            /* iv, iv length  */
            {0x62, 0xec, 0x67, 0xf9, 0xc3, 0xa4, 0xa4, 0x07,
             0xfc, 0xb2, 0xa8, 0xc4, 0x90, 0x31, 0xa8, 0xb3}, 16,
            /* aad, aad length */
            {0x6b, 0xfb, 0x91, 0x4f, 0xd0, 0x7e, 0xae, 0x6b}, 8,
            /* msg, msg length */
            {0}, 0,
            /* ct, ct length */
            {0}, 0,
            /* tag, tag length */
            {0xe0, 0x37, 0x83, 0x0e, 0x83, 0x89, 0xf2,
             0x7b, 0x02, 0x5a, 0x2d, 0x65, 0x27, 0xe7, 0x9d, 0x01}, 16,
            /* valid */
            1,
        },
        /*  Vector from paper - no auth data, valid auth tag */
        {
            /* key, key length */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv , iv length */
            {0x3c, 0x8c, 0xc2, 0x97, 0x0a, 0x00, 0x8f, 0x75,
             0xcc, 0x5b, 0xea, 0xe2, 0x84, 0x72, 0x58, 0xc2}, 16,
            /* aad, aad length */
            {0}, 0,
            /* msg, msg length */
            {0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11,
             0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11, 0x11}, 32,
            /* ct, ct length */
            {0x3c, 0x44, 0x1f, 0x32, 0xce, 0x07, 0x82, 0x23,
             0x64, 0xd7, 0xa2, 0x99, 0x0e, 0x50, 0xbb, 0x13,
             0xd7, 0xb0, 0x2a, 0x26, 0x96, 0x9e, 0x4a, 0x93,
             0x7e, 0x5e, 0x90, 0x73, 0xb0, 0xd9, 0xc9, 0x68}, 32,
            /* tag, tag length */
            {0xdb, 0x90, 0xbd, 0xb3, 0xda, 0x3d, 0x00, 0xaf,
             0xd0, 0xfc, 0x6a, 0x83, 0x55, 0x1d, 0xa9, 0x5e}, 16,
            /* valid */
            1,
        },
        /* Vector from paper - no auth data with invalid auth tag */
        {
            /* key, key length */
            {0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
             0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f}, 16,
            /* iv, iv length */
            {0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57,
             0x58, 0x59, 0x5a, 0x5b, 0x5c, 0x5d, 0x5e, 0x5f}, 16,
            /* aad, aad length */
            {0}, 0,
            /* msg, msg length */
            {0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
             0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f}, 16,
            /* ct , ct length */
            {0x29, 0xa0, 0x91, 0x4f, 0xec, 0x4b, 0xef, 0x54,
             0xba, 0xbf, 0x66, 0x13, 0xa9, 0xf9, 0xcd, 0x70}, 16,
            /* tag, tag length */
            {0xe7, 0x0e, 0x7c, 0x50, 0x13, 0xa6, 0xdb, 0xf2,
             0x52, 0x98, 0xb1, 0x92, 0x9b, 0xc3, 0x56, 0xa7}, 16,
            /* valid */
            0,
        },
    };

    WOLFSSL_SMALL_STACK_STATIC byte ciphertext[sizeof(vectors[0].ct)];
    WOLFSSL_SMALL_STACK_STATIC byte authtag[sizeof(vectors[0].tag)];
    wc_test_ret_t ret;
    int i;
    int len;

    for (i = 0; i < (int)(sizeof(vectors)/sizeof(vectors[0])); i++) {

        XMEMSET(ciphertext, 0, sizeof(ciphertext));

        len = sizeof(authtag);
        ret = wc_AesEaxEncryptAuth(vectors[i].key, vectors[i].key_length,
                                   ciphertext,
                                   vectors[i].msg, vectors[i].msg_length,
                                   vectors[i].iv, vectors[i].iv_length,
                                   authtag, len,
                                   vectors[i].aad, vectors[i].aad_length);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        /* check ciphertext matches vector */
        if (XMEMCMP(ciphertext, vectors[i].ct, vectors[i].ct_length)) {
            return WC_TEST_RET_ENC_NC;
        }

        /* check that tag matches vector only for vectors marked as valid */
        ret = XMEMCMP(authtag, vectors[i].tag, len);
        if (vectors[i].valid == 1 && ret != 0 ) {
            return WC_TEST_RET_ENC_NC;
        }
        else if (vectors[i].valid == 0 && ret == 0) {
            return WC_TEST_RET_ENC_NC;
        }

        XMEMSET(ciphertext, 0, sizeof(ciphertext));

        ret = wc_AesEaxDecryptAuth(vectors[i].key, vectors[i].key_length,
                                   ciphertext,
                                   vectors[i].ct, vectors[i].ct_length,
                                   vectors[i].iv, vectors[i].iv_length,
                                   authtag, len,
                                   vectors[i].aad, vectors[i].aad_length);
        if (ret != 0) {
            return WC_TEST_RET_ENC_EC(ret);
        }

        /* check decrypted ciphertext matches vector plaintext */
        if (XMEMCMP(ciphertext, vectors[i].msg, vectors[i].msg_length)) {
            return WC_TEST_RET_ENC_NC;
        }

    }
    return 0;
}

#endif  /* WOLFSSL_AES_EAX */



#ifdef HAVE_AES_KEYWRAP

#define MAX_KEYWRAP_TEST_OUTLEN 40
#define MAX_KEYWRAP_TEST_PLAINLEN 32

typedef struct keywrapVector {
    const byte* kek;
    const byte* data;
    const byte* verify;
    word32 kekLen;
    word32 dataLen;
    word32 verifyLen;
} keywrapVector;

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t aeskeywrap_test(void)
{
    int wrapSz, plainSz, testSz, i;

    /* test vectors from RFC 3394 (kek, data, verify) */

#ifdef WOLFSSL_AES_128
    /* Wrap 128 bits of Key Data with a 128-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d1[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v1[] = {
        0x1F, 0xA6, 0x8B, 0x0A, 0x81, 0x12, 0xB4, 0x47,
        0xAE, 0xF3, 0x4B, 0xD8, 0xFB, 0x5A, 0x7B, 0x82,
        0x9D, 0x3E, 0x86, 0x23, 0x71, 0xD2, 0xCF, 0xE5
    };
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
    /* Wrap 128 bits of Key Data with a 192-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d2[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v2[] = {
        0x96, 0x77, 0x8B, 0x25, 0xAE, 0x6C, 0xA4, 0x35,
        0xF9, 0x2B, 0x5B, 0x97, 0xC0, 0x50, 0xAE, 0xD2,
        0x46, 0x8A, 0xB8, 0xA1, 0x7A, 0xD8, 0x4E, 0x5D
    };
#endif

#ifdef WOLFSSL_AES_256
    /* Wrap 128 bits of Key Data with a 256-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k3[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d3[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v3[] = {
        0x64, 0xE8, 0xC3, 0xF9, 0xCE, 0x0F, 0x5B, 0xA2,
        0x63, 0xE9, 0x77, 0x79, 0x05, 0x81, 0x8A, 0x2A,
        0x93, 0xC8, 0x19, 0x1E, 0x7D, 0x6E, 0x8A, 0xE7
    };
#endif

#ifdef WOLFSSL_AES_192
    /* Wrap 192 bits of Key Data with a 192-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k4[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d4[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v4[] = {
        0x03, 0x1D, 0x33, 0x26, 0x4E, 0x15, 0xD3, 0x32,
        0x68, 0xF2, 0x4E, 0xC2, 0x60, 0x74, 0x3E, 0xDC,
        0xE1, 0xC6, 0xC7, 0xDD, 0xEE, 0x72, 0x5A, 0x93,
        0x6B, 0xA8, 0x14, 0x91, 0x5C, 0x67, 0x62, 0xD2
    };
#endif

#ifdef WOLFSSL_AES_256
    /* Wrap 192 bits of Key Data with a 256-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k5[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d5[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v5[] = {
        0xA8, 0xF9, 0xBC, 0x16, 0x12, 0xC6, 0x8B, 0x3F,
        0xF6, 0xE6, 0xF4, 0xFB, 0xE3, 0x0E, 0x71, 0xE4,
        0x76, 0x9C, 0x8B, 0x80, 0xA3, 0x2C, 0xB8, 0x95,
        0x8C, 0xD5, 0xD1, 0x7D, 0x6B, 0x25, 0x4D, 0xA1
    };

    /* Wrap 256 bits of Key Data with a 256-bit KEK */
    WOLFSSL_SMALL_STACK_STATIC const byte k6[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F,
        0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
        0x18, 0x19, 0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte d6[] = {
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xAA, 0xBB, 0xCC, 0xDD, 0xEE, 0xFF,
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };

    WOLFSSL_SMALL_STACK_STATIC const byte v6[] = {
        0x28, 0xC9, 0xF4, 0x04, 0xC4, 0xB8, 0x10, 0xF4,
        0xCB, 0xCC, 0xB3, 0x5C, 0xFB, 0x87, 0xF8, 0x26,
        0x3F, 0x57, 0x86, 0xE2, 0xD8, 0x0E, 0xD3, 0x26,
        0xCB, 0xC7, 0xF0, 0xE7, 0x1A, 0x99, 0xF4, 0x3B,
        0xFB, 0x98, 0x8B, 0x9B, 0x7A, 0x02, 0xDD, 0x21
    };
#endif /* WOLFSSL_AES_256 */

    byte output[MAX_KEYWRAP_TEST_OUTLEN];
    byte plain [MAX_KEYWRAP_TEST_PLAINLEN];

    const keywrapVector test_wrap[] =
    {
    #ifdef WOLFSSL_AES_128
        {k1, d1, v1, sizeof(k1), sizeof(d1), sizeof(v1)},
    #endif
    #ifdef WOLFSSL_AES_192
        {k2, d2, v2, sizeof(k2), sizeof(d2), sizeof(v2)},
    #endif
    #ifdef WOLFSSL_AES_256
        {k3, d3, v3, sizeof(k3), sizeof(d3), sizeof(v3)},
    #endif
    #ifdef WOLFSSL_AES_192
        {k4, d4, v4, sizeof(k4), sizeof(d4), sizeof(v4)},
    #endif
    #ifdef WOLFSSL_AES_256
        {k5, d5, v5, sizeof(k5), sizeof(d5), sizeof(v5)},
        {k6, d6, v6, sizeof(k6), sizeof(d6), sizeof(v6)}
    #endif
    };
    testSz = sizeof(test_wrap) / sizeof(keywrapVector);

    XMEMSET(output, 0, sizeof(output));
    XMEMSET(plain,  0, sizeof(plain));

    for (i = 0; i < testSz; i++) {

        wrapSz = wc_AesKeyWrap(test_wrap[i].kek, test_wrap[i].kekLen,
                               test_wrap[i].data, test_wrap[i].dataLen,
                               output, sizeof(output), NULL);

        if ( (wrapSz < 0) || (wrapSz != (int)test_wrap[i].verifyLen) )
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(output, test_wrap[i].verify, test_wrap[i].verifyLen) != 0)
            return WC_TEST_RET_ENC_NC;

        plainSz = wc_AesKeyUnWrap((byte*)test_wrap[i].kek, test_wrap[i].kekLen,
                                  output, wrapSz,
                                  plain, sizeof(plain), NULL);

        if ( (plainSz < 0) || (plainSz != (int)test_wrap[i].dataLen) )
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plain, test_wrap[i].data, test_wrap[i].dataLen) != 0)
            return WC_TEST_RET_ENC_I(i);
    }

    return 0;
}
#endif /* HAVE_AES_KEYWRAP */


#endif /* NO_AES */

#ifdef HAVE_ARIA
void printOutput(const char *strName, unsigned char *data, unsigned int dataSz)
{
    #ifndef DEBUG_WOLFSSL
    (void)strName;
    (void)data;
    (void)dataSz;
    #else
    WOLFSSL_MSG_EX("%s (%d):", strName,dataSz);
    WOLFSSL_BUFFER(data,dataSz);
    #endif
}

WOLFSSL_TEST_SUBROUTINE int ariagcm_test(MC_ALGID algo)
{
    int ret = 0;
    byte data[] = TEST_STRING;
    word32 dataSz = TEST_STRING_SZ;

    /* Arbitrarily random long key that we will truncate to the right size */
    byte key[] = { 0x1E, 0xCC, 0x95, 0xCB, 0xD3, 0x74, 0x58, 0x4F,
                       0x6F, 0x8A, 0x70, 0x26, 0xF7, 0x3C, 0x8D, 0xB6,
                       0xDC, 0x32, 0x76, 0x20, 0xCF, 0x05, 0x4A, 0xCF,
                       0x11, 0x86, 0xCD, 0x23, 0x5E, 0xC1, 0x6E, 0x2B };
    byte cipher[2*TEST_STRING_SZ], plain[TEST_STRING_SZ], ad[256], authTag[AES_BLOCK_SIZE];
    word32 keySz, adSz = 256, authTagSz = sizeof(authTag);

    wc_Aria aria;
    XMEMSET((void *)&aria, 0, sizeof(aria));
    ret = wc_AriaInitCrypt(&aria, algo);
    if (ret != 0) { ERROR_OUT(WC_TEST_RET_ENC_EC(ret),out); }

    ret = wc_AriaSetKey(&aria, key);
    if (ret != 0) { ERROR_OUT(WC_TEST_RET_ENC_EC(ret),out); }

    MC_GetObjectValue(aria.hSession, aria.hKey, key, &keySz);
    printOutput("Key", key, keySz);

    WC_RNG rng;

    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_AriaGcmSetIV(&aria, GCM_NONCE_MID_SZ, NULL, 0, &rng);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    wc_FreeRng(&rng);

    printOutput("Plaintext", data, sizeof(data));
    XMEMSET(cipher, 0, sizeof(cipher));

    ret = wc_AriaEncrypt(&aria, cipher, data, dataSz,
                         (byte *)aria.nonce, aria.nonceSz, ad, adSz,
                         authTag, authTagSz);
    if (ret != 0) { ERROR_OUT(WC_TEST_RET_ENC_EC(ret),out); }

    printOutput("Ciphertext", cipher, sizeof(cipher));
    printOutput("AuthTag", authTag, sizeof(authTag));

    XMEMSET(plain, 0, sizeof(plain));

    ret = wc_AriaDecrypt(&aria, plain, cipher, dataSz,
                         (byte *)aria.nonce, aria.nonceSz, ad, adSz,
                         authTag, authTagSz);
    if (ret != 0) { ERROR_OUT(WC_TEST_RET_ENC_EC(ret),out); }

    printOutput("Plaintext", plain, sizeof(plain));

    if (XMEMCMP(plain, data, dataSz) != 0)
        ERROR_OUT(WC_TEST_RET_ENC_NC,out);
out:
    if (ret != 0) { wc_AriaFreeCrypt(&aria); }
    else { ret = wc_AriaFreeCrypt(&aria); }

    return ret;
}
#endif /* HAVE_ARIA */


#ifdef HAVE_CAMELLIA

enum {
    CAM_ECB_ENC, CAM_ECB_DEC, CAM_CBC_ENC, CAM_CBC_DEC
};

typedef struct {
    int type;
    const byte* plaintext;
    const byte* iv;
    const byte* ciphertext;
    const byte* key;
    word32 keySz;
    int errorCode;
} test_vector_t;

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t camellia_test(void)
{
    /* Camellia ECB Test Plaintext */
    WOLFSSL_SMALL_STACK_STATIC const byte pte[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10
    };

    /* Camellia ECB Test Initialization Vector */
    WOLFSSL_SMALL_STACK_STATIC const byte ive[] = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

    /* Test 1: Camellia ECB 128-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c1[] =
    {
        0x67, 0x67, 0x31, 0x38, 0x54, 0x96, 0x69, 0x73,
        0x08, 0x57, 0x06, 0x56, 0x48, 0xea, 0xbe, 0x43
    };

    /* Test 2: Camellia ECB 192-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c2[] =
    {
        0xb4, 0x99, 0x34, 0x01, 0xb3, 0xe9, 0x96, 0xf8,
        0x4e, 0xe5, 0xce, 0xe7, 0xd7, 0x9b, 0x09, 0xb9
    };

    /* Test 3: Camellia ECB 256-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k3[] =
    {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xab, 0xcd, 0xef,
        0xfe, 0xdc, 0xba, 0x98, 0x76, 0x54, 0x32, 0x10,
        0x00, 0x11, 0x22, 0x33, 0x44, 0x55, 0x66, 0x77,
        0x88, 0x99, 0xaa, 0xbb, 0xcc, 0xdd, 0xee, 0xff
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c3[] =
    {
        0x9a, 0xcc, 0x23, 0x7d, 0xff, 0x16, 0xd7, 0x6c,
        0x20, 0xef, 0x7c, 0x91, 0x9e, 0x3a, 0x75, 0x09
    };

    /* Camellia CBC Test Plaintext */
    WOLFSSL_SMALL_STACK_STATIC const byte ptc[] =
    {
        0x6B, 0xC1, 0xBE, 0xE2, 0x2E, 0x40, 0x9F, 0x96,
        0xE9, 0x3D, 0x7E, 0x11, 0x73, 0x93, 0x17, 0x2A
    };

    /* Camellia CBC Test Initialization Vector */
    WOLFSSL_SMALL_STACK_STATIC const byte ivc[] =
    {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };

    /* Test 4: Camellia-CBC 128-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k4[] =
    {
        0x2B, 0x7E, 0x15, 0x16, 0x28, 0xAE, 0xD2, 0xA6,
        0xAB, 0xF7, 0x15, 0x88, 0x09, 0xCF, 0x4F, 0x3C
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c4[] =
    {
        0x16, 0x07, 0xCF, 0x49, 0x4B, 0x36, 0xBB, 0xF0,
        0x0D, 0xAE, 0xB0, 0xB5, 0x03, 0xC8, 0x31, 0xAB
    };

    /* Test 5: Camellia-CBC 192-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k5[] =
    {
        0x8E, 0x73, 0xB0, 0xF7, 0xDA, 0x0E, 0x64, 0x52,
        0xC8, 0x10, 0xF3, 0x2B, 0x80, 0x90, 0x79, 0xE5,
        0x62, 0xF8, 0xEA, 0xD2, 0x52, 0x2C, 0x6B, 0x7B
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c5[] =
    {
        0x2A, 0x48, 0x30, 0xAB, 0x5A, 0xC4, 0xA1, 0xA2,
        0x40, 0x59, 0x55, 0xFD, 0x21, 0x95, 0xCF, 0x93
    };

    /* Test 6: CBC 256-bit key */
    WOLFSSL_SMALL_STACK_STATIC const byte k6[] =
    {
        0x60, 0x3D, 0xEB, 0x10, 0x15, 0xCA, 0x71, 0xBE,
        0x2B, 0x73, 0xAE, 0xF0, 0x85, 0x7D, 0x77, 0x81,
        0x1F, 0x35, 0x2C, 0x07, 0x3B, 0x61, 0x08, 0xD7,
        0x2D, 0x98, 0x10, 0xA3, 0x09, 0x14, 0xDF, 0xF4
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c6[] =
    {
        0xE6, 0xCF, 0xA3, 0x5F, 0xC0, 0x2B, 0x13, 0x4A,
        0x4D, 0x2C, 0x0B, 0x67, 0x37, 0xAC, 0x3E, 0xDA
    };

    byte out[CAMELLIA_BLOCK_SIZE];
    Camellia cam;
    int i, testsSz, ret;
    WOLFSSL_SMALL_STACK_STATIC const test_vector_t testVectors[] =
    {
        {CAM_ECB_ENC, pte, ive, c1, k1, sizeof(k1), -114},
        {CAM_ECB_ENC, pte, ive, c2, k2, sizeof(k2), -115},
        {CAM_ECB_ENC, pte, ive, c3, k3, sizeof(k3), -116},
        {CAM_ECB_DEC, pte, ive, c1, k1, sizeof(k1), -117},
        {CAM_ECB_DEC, pte, ive, c2, k2, sizeof(k2), -118},
        {CAM_ECB_DEC, pte, ive, c3, k3, sizeof(k3), -119},
        {CAM_CBC_ENC, ptc, ivc, c4, k4, sizeof(k4), -120},
        {CAM_CBC_ENC, ptc, ivc, c5, k5, sizeof(k5), -121},
        {CAM_CBC_ENC, ptc, ivc, c6, k6, sizeof(k6), -122},
        {CAM_CBC_DEC, ptc, ivc, c4, k4, sizeof(k4), -123},
        {CAM_CBC_DEC, ptc, ivc, c5, k5, sizeof(k5), -124},
        {CAM_CBC_DEC, ptc, ivc, c6, k6, sizeof(k6), -125}
    };

    testsSz = sizeof(testVectors)/sizeof(test_vector_t);
    for (i = 0; i < testsSz; i++) {
        if (wc_CamelliaSetKey(&cam, testVectors[i].key, testVectors[i].keySz,
                                                        testVectors[i].iv) != 0)
            return testVectors[i].errorCode;

        switch (testVectors[i].type) {
            case CAM_ECB_ENC:
                ret = wc_CamelliaEncryptDirect(&cam, out,
                                                testVectors[i].plaintext);
                if (ret != 0 || XMEMCMP(out, testVectors[i].ciphertext,
                                                        CAMELLIA_BLOCK_SIZE))
                    return testVectors[i].errorCode;
                break;
            case CAM_ECB_DEC:
                ret = wc_CamelliaDecryptDirect(&cam, out,
                                                    testVectors[i].ciphertext);
                if (ret != 0 || XMEMCMP(out, testVectors[i].plaintext,
                                                        CAMELLIA_BLOCK_SIZE))
                    return testVectors[i].errorCode;
                break;
            case CAM_CBC_ENC:
                ret = wc_CamelliaCbcEncrypt(&cam, out, testVectors[i].plaintext,
                                                           CAMELLIA_BLOCK_SIZE);
                if (ret != 0 || XMEMCMP(out, testVectors[i].ciphertext,
                                                        CAMELLIA_BLOCK_SIZE))
                    return testVectors[i].errorCode;
                break;
            case CAM_CBC_DEC:
                ret = wc_CamelliaCbcDecrypt(&cam, out,
                                testVectors[i].ciphertext, CAMELLIA_BLOCK_SIZE);
                if (ret != 0 || XMEMCMP(out, testVectors[i].plaintext,
                                                           CAMELLIA_BLOCK_SIZE))
                    return testVectors[i].errorCode;
                break;
            default:
                break;
        }
    }

    /* Setting the IV and checking it was actually set. */
    ret = wc_CamelliaSetIV(&cam, ivc);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(cam.reg, ivc, CAMELLIA_BLOCK_SIZE) != 0)
        return WC_TEST_RET_ENC_NC;

    /* Setting the IV to NULL should be same as all zeros IV */
    ret = wc_CamelliaSetIV(&cam, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(cam.reg, ive, CAMELLIA_BLOCK_SIZE) != 0)
        return WC_TEST_RET_ENC_NC;

    /* First parameter should never be null */
    if (wc_CamelliaSetIV(NULL, NULL) == 0)
        return WC_TEST_RET_ENC_NC;

    /* First parameter should never be null, check it fails */
    if (wc_CamelliaSetKey(NULL, k1, sizeof(k1), NULL) == 0)
        return WC_TEST_RET_ENC_NC;

    /* Key should have a size of 16, 24, or 32 */
    if (wc_CamelliaSetKey(&cam, k1, 0, NULL) == 0)
        return WC_TEST_RET_ENC_NC;

    return 0;
}
#endif /* HAVE_CAMELLIA */

#ifdef WOLFSSL_SM4
#ifdef WOLFSSL_SM4_ECB
static int sm4_ecb_test(void)
{
    /* draft-ribose-cfrg-sm4-10 A.2.1.1 */
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte p1[] = {
        0xAA, 0xAA, 0xAA, 0xAA, 0xBB, 0xBB, 0xBB, 0xBB,
        0xCC, 0xCC, 0xCC, 0xCC, 0xDD, 0xDD, 0xDD, 0xDD,
        0xEE, 0xEE, 0xEE, 0xEE, 0xFF, 0xFF, 0xFF, 0xFF,
        0xAA, 0xAA, 0xAA, 0xAA, 0xBB, 0xBB, 0xBB, 0xBB
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c1_ecb[] = {
        0x5E, 0xC8, 0x14, 0x3D, 0xE5, 0x09, 0xCF, 0xF7,
        0xB5, 0x17, 0x9F, 0x8F, 0x47, 0x4B, 0x86, 0x19,
        0x2F, 0x1D, 0x30, 0x5A, 0x7F, 0xB1, 0x7D, 0xF9,
        0x85, 0xF8, 0x1C, 0x84, 0x82, 0x19, 0x23, 0x04
    };

    wc_Sm4 sm4;
    byte enc[SM4_BLOCK_SIZE * 4];
    byte dec[SM4_BLOCK_SIZE * 4];
    int ret;

    ret = wc_Sm4Init(&sm4, NULL, INVALID_DEVID);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Encrypt and decrypt with ECB. */
    ret = wc_Sm4SetKey(&sm4, k1, sizeof(k1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Sm4EcbEncrypt(&sm4, enc, p1, sizeof(p1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, c1_ecb, sizeof(c1_ecb)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_Sm4EcbDecrypt(&sm4, dec, enc, sizeof(c1_ecb));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(dec, p1, sizeof(p1)) != 0)
        return WC_TEST_RET_ENC_NC;

    wc_Sm4Free(&sm4);

    return 0;
}
#endif

#ifdef WOLFSSL_SM4_CBC
static int sm4_cbc_test(void)
{
    /* draft-ribose-cfrg-sm4-10 A.2.2.1 */
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte p1[] = {
        0xAA, 0xAA, 0xAA, 0xAA, 0xBB, 0xBB, 0xBB, 0xBB,
        0xCC, 0xCC, 0xCC, 0xCC, 0xDD, 0xDD, 0xDD, 0xDD,
        0xEE, 0xEE, 0xEE, 0xEE, 0xFF, 0xFF, 0xFF, 0xFF,
        0xAA, 0xAA, 0xAA, 0xAA, 0xBB, 0xBB, 0xBB, 0xBB
    };
    WOLFSSL_SMALL_STACK_STATIC const byte i1[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c1_cbc[] = {
        0x78, 0xEB, 0xB1, 0x1C, 0xC4, 0x0B, 0x0A, 0x48,
        0x31, 0x2A, 0xAE, 0xB2, 0x04, 0x02, 0x44, 0xCB,
        0x4C, 0xB7, 0x01, 0x69, 0x51, 0x90, 0x92, 0x26,
        0x97, 0x9B, 0x0D, 0x15, 0xDC, 0x6A, 0x8F, 0x6D
    };

    wc_Sm4 sm4;
    byte enc[SM4_BLOCK_SIZE * 4];
    byte dec[SM4_BLOCK_SIZE * 4];
    int ret;

    ret = wc_Sm4Init(&sm4, NULL, INVALID_DEVID);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Encrypt and decrypt with CBC. */
    ret = wc_Sm4SetKey(&sm4, k1, sizeof(k1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Sm4SetIV(&sm4, i1);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Sm4CbcEncrypt(&sm4, enc, p1, sizeof(p1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, c1_cbc, sizeof(c1_cbc)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_Sm4SetIV(&sm4, i1);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Sm4CbcDecrypt(&sm4, dec, enc, sizeof(c1_cbc));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(dec, p1, sizeof(p1)) != 0)
        return WC_TEST_RET_ENC_NC;

    /* Encrypt and decrypt in-place with CBC. */
    ret = wc_Sm4SetKey(&sm4, k1, sizeof(k1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Sm4SetIV(&sm4, i1);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    XMEMCPY(enc, p1, sizeof(p1));
    ret = wc_Sm4CbcEncrypt(&sm4, enc, enc, sizeof(p1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, c1_cbc, sizeof(c1_cbc)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_Sm4SetIV(&sm4, i1);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Sm4CbcDecrypt(&sm4, enc, enc, sizeof(c1_cbc));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, p1, sizeof(p1)) != 0)
        return WC_TEST_RET_ENC_NC;

    wc_Sm4Free(&sm4);

    return 0;
}
#endif

#ifdef WOLFSSL_SM4_CTR
static int sm4_ctr_test(void)
{
    /* draft-ribose-cfrg-sm4-10 A.2.5.1 */
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte i1[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F
    };
    WOLFSSL_SMALL_STACK_STATIC const byte p2[] = {
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
        0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
        0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
        0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c2_ctr[] = {
        0xAC, 0x32, 0x36, 0xCB, 0x97, 0x0C, 0xC2, 0x07,
        0x91, 0x36, 0x4C, 0x39, 0x5A, 0x13, 0x42, 0xD1,
        0xA3, 0xCB, 0xC1, 0x87, 0x8C, 0x6F, 0x30, 0xCD,
        0x07, 0x4C, 0xCE, 0x38, 0x5C, 0xDD, 0x70, 0xC7,
        0xF2, 0x34, 0xBC, 0x0E, 0x24, 0xC1, 0x19, 0x80,
        0xFD, 0x12, 0x86, 0x31, 0x0C, 0xE3, 0x7B, 0x92,
        0x6E, 0x02, 0xFC, 0xD0, 0xFA, 0xA0, 0xBA, 0xF3,
        0x8B, 0x29, 0x33, 0x85, 0x1D, 0x82, 0x45, 0x14
    };

    wc_Sm4 sm4;
    byte enc[SM4_BLOCK_SIZE * 4];
    byte dec[SM4_BLOCK_SIZE * 4];
    int chunk;
    int i;
    int ret;

    ret = wc_Sm4Init(&sm4, NULL, INVALID_DEVID);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Encrypt and decrypt using encrypt with CTR. */
    ret = wc_Sm4SetKey(&sm4, k1, sizeof(k1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Sm4SetIV(&sm4, i1);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Sm4CtrEncrypt(&sm4, enc, p2, sizeof(p2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, c2_ctr, sizeof(c2_ctr)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_Sm4SetIV(&sm4, i1);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_Sm4CtrEncrypt(&sm4, dec, enc, sizeof(c2_ctr));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(dec, p2, sizeof(p2)) != 0)
        return WC_TEST_RET_ENC_NC;

    for (chunk = 1; chunk <= SM4_BLOCK_SIZE + 1; chunk++) {
        ret = wc_Sm4SetIV(&sm4, i1);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(chunk);

        XMEMSET(enc, 0, sizeof(enc));
        for (i = 0; i + chunk <= (int)sizeof(p2); i += chunk) {
            ret = wc_Sm4CtrEncrypt(&sm4, enc + i, p2 + i, chunk);
            if (ret != 0)
                return WC_TEST_RET_ENC_I(i);
        }
        if (i < (int)sizeof(p2)) {
            ret = wc_Sm4CtrEncrypt(&sm4, enc + i, p2 + i, sizeof(p2) - i);
            if (ret != 0)
                return WC_TEST_RET_ENC_I(chunk);
        }
        if (XMEMCMP(enc, c2_ctr, sizeof(c2_ctr)) != 0)
            return WC_TEST_RET_ENC_I(chunk);
    }

    wc_Sm4Free(&sm4);

    return 0;
}
#endif

#ifdef WOLFSSL_SM4_GCM
static int sm4_gcm_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte p1[] = {
        0xAA, 0xAA, 0xAA, 0xAA, 0xBB, 0xBB, 0xBB, 0xBB,
        0xCC, 0xCC, 0xCC, 0xCC, 0xDD, 0xDD, 0xDD, 0xDD,
        0xEE, 0xEE, 0xEE, 0xEE, 0xFF, 0xFF, 0xFF, 0xFF,
        0xAA, 0xAA, 0xAA, 0xAA, 0xBB, 0xBB, 0xBB, 0xBB
    };
    WOLFSSL_SMALL_STACK_STATIC const byte i1[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B
    };
    WOLFSSL_SMALL_STACK_STATIC const byte a1[] = {
        0xFF, 0xEE, 0xDD
    };
    WOLFSSL_SMALL_STACK_STATIC const byte tag1[] = {
        0x83, 0xb2, 0x91, 0xcf, 0x22, 0xc9, 0x5f, 0x89,
        0xde, 0x3d, 0x52, 0x8d, 0xd7, 0x13, 0x50, 0x89
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c1[] = {
        0xff, 0x8b, 0xb2, 0x3b, 0x0a, 0x0a, 0x12, 0xa4,
        0xa8, 0x4c, 0x4f, 0x67, 0x06, 0x81, 0xbb, 0x88,
        0x66, 0x17, 0xc7, 0x43, 0xbf, 0xae, 0x41, 0x40,
        0xec, 0x1e, 0x03, 0x85, 0x2b, 0x56, 0xa8, 0xc0
    };
    /* RFC8998 A.1. */
    WOLFSSL_SMALL_STACK_STATIC const byte i2[] = {
        0x00, 0x00, 0x12, 0x34, 0x56, 0x78, 0x00, 0x00,
        0x00, 0x00, 0xAB, 0xCD
    };
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte p2[] = {
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
        0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
        0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
        0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA
    };
    WOLFSSL_SMALL_STACK_STATIC const byte a2[] = {
        0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE, 0xEF,
        0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE, 0xEF,
        0xAB, 0xAD, 0xDA, 0xD2
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c2[] = {
        0x17, 0xF3, 0x99, 0xF0, 0x8C, 0x67, 0xD5, 0xEE,
        0x19, 0xD0, 0xDC, 0x99, 0x69, 0xC4, 0xBB, 0x7D,
        0x5F, 0xD4, 0x6F, 0xD3, 0x75, 0x64, 0x89, 0x06,
        0x91, 0x57, 0xB2, 0x82, 0xBB, 0x20, 0x07, 0x35,
        0xD8, 0x27, 0x10, 0xCA, 0x5C, 0x22, 0xF0, 0xCC,
        0xFA, 0x7C, 0xBF, 0x93, 0xD4, 0x96, 0xAC, 0x15,
        0xA5, 0x68, 0x34, 0xCB, 0xCF, 0x98, 0xC3, 0x97,
        0xB4, 0x02, 0x4A, 0x26, 0x91, 0x23, 0x3B, 0x8D
    };
    WOLFSSL_SMALL_STACK_STATIC const byte tag2[] = {
        0x83, 0xDE, 0x35, 0x41, 0xE4, 0xC2, 0xB5, 0x81,
        0x77, 0xE0, 0x65, 0xA9, 0xBF, 0x7B, 0x62, 0xEC
    };

    wc_Sm4 sm4;
    byte enc[SM4_BLOCK_SIZE * 4];
    byte dec[SM4_BLOCK_SIZE * 4];
    byte tag[SM4_BLOCK_SIZE];
    int ret;

    ret = wc_Sm4Init(&sm4, NULL, INVALID_DEVID);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Encrypt and decrypt using encrypt with GCM. */
    ret = wc_Sm4GcmSetKey(&sm4, k1, sizeof(k1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Sm4GcmEncrypt(&sm4, enc, p1, sizeof(p1), i1, sizeof(i1), tag,
        sizeof(tag), a1, sizeof(a1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, c1, sizeof(c1)) != 0)
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(tag, tag1, sizeof(tag1)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_Sm4GcmDecrypt(&sm4, dec, enc, sizeof(c1), i1, sizeof(i1), tag,
        sizeof(tag), a1, sizeof(a1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(dec, p1, sizeof(p1)) != 0)
        return WC_TEST_RET_ENC_NC;

    /* RFC8998 test vector. */
    ret = wc_Sm4GcmSetKey(&sm4, k2, sizeof(k2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Sm4GcmEncrypt(&sm4, enc, p2, sizeof(p2), i2, sizeof(i2), tag,
        sizeof(tag), a2, sizeof(a2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, c2, sizeof(c2)) != 0)
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(tag, tag2, sizeof(tag2)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_Sm4GcmDecrypt(&sm4, dec, enc, sizeof(c2), i2, sizeof(i2), tag,
        sizeof(tag), a2, sizeof(a2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(dec, p2, sizeof(p2)) != 0)
        return WC_TEST_RET_ENC_NC;

    wc_Sm4Free(&sm4);

    return 0;
}
#endif

#ifdef WOLFSSL_SM4_CCM
static int sm4_ccm_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte k1[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte p1[] = {
        0xAA, 0xAA, 0xAA, 0xAA, 0xBB, 0xBB, 0xBB, 0xBB,
        0xCC, 0xCC, 0xCC, 0xCC, 0xDD, 0xDD, 0xDD, 0xDD,
        0xEE, 0xEE, 0xEE, 0xEE, 0xFF, 0xFF, 0xFF, 0xFF,
        0xAA, 0xAA, 0xAA, 0xAA, 0xBB, 0xBB, 0xBB, 0xBB
    };
    WOLFSSL_SMALL_STACK_STATIC const byte i1[] = {
        0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
        0x08, 0x09, 0x0A, 0x0B
    };
    WOLFSSL_SMALL_STACK_STATIC const byte a1[] = {
        0xFF, 0xEE, 0xDD
    };
    WOLFSSL_SMALL_STACK_STATIC const byte tag1[] = {
        0x9a, 0x98, 0x04, 0xb6, 0x0f, 0x19, 0x4a, 0x46,
        0xba, 0xed, 0xe6, 0x89, 0x69, 0x34, 0xad, 0x61
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c1[] = {
        0xbd, 0xc0, 0x72, 0x60, 0xda, 0x2d, 0x11, 0xdc,
        0x66, 0x33, 0xcc, 0xec, 0xb2, 0xf4, 0x53, 0x59,
        0x9e, 0xb1, 0xb3, 0x6b, 0x1f, 0x1c, 0xfb, 0x29,
        0xf5, 0x37, 0xfc, 0x00, 0xf2, 0x4e, 0x70, 0x6f
    };
    /* RFC8998 A.1. */
    WOLFSSL_SMALL_STACK_STATIC const byte i2[] = {
        0x00, 0x00, 0x12, 0x34, 0x56, 0x78, 0x00, 0x00,
        0x00, 0x00, 0xAB, 0xCD
    };
    WOLFSSL_SMALL_STACK_STATIC const byte k2[] = {
        0x01, 0x23, 0x45, 0x67, 0x89, 0xAB, 0xCD, 0xEF,
        0xFE, 0xDC, 0xBA, 0x98, 0x76, 0x54, 0x32, 0x10
    };
    WOLFSSL_SMALL_STACK_STATIC const byte p2[] = {
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA,
        0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB, 0xBB,
        0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC, 0xCC,
        0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD, 0xDD,
        0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
        0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
        0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE, 0xEE,
        0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA, 0xAA
    };
    WOLFSSL_SMALL_STACK_STATIC const byte a2[] = {
        0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE, 0xEF,
        0xFE, 0xED, 0xFA, 0xCE, 0xDE, 0xAD, 0xBE, 0xEF,
        0xAB, 0xAD, 0xDA, 0xD2
    };
    WOLFSSL_SMALL_STACK_STATIC const byte c2[] = {
        0x48, 0xAF, 0x93, 0x50, 0x1F, 0xA6, 0x2A, 0xDB,
        0xCD, 0x41, 0x4C, 0xCE, 0x60, 0x34, 0xD8, 0x95,
        0xDD, 0xA1, 0xBF, 0x8F, 0x13, 0x2F, 0x04, 0x20,
        0x98, 0x66, 0x15, 0x72, 0xE7, 0x48, 0x30, 0x94,
        0xFD, 0x12, 0xE5, 0x18, 0xCE, 0x06, 0x2C, 0x98,
        0xAC, 0xEE, 0x28, 0xD9, 0x5D, 0xF4, 0x41, 0x6B,
        0xED, 0x31, 0xA2, 0xF0, 0x44, 0x76, 0xC1, 0x8B,
        0xB4, 0x0C, 0x84, 0xA7, 0x4B, 0x97, 0xDC, 0x5B
    };
    WOLFSSL_SMALL_STACK_STATIC const byte tag2[] = {
        0x16, 0x84, 0x2D, 0x4F, 0xA1, 0x86, 0xF5, 0x6A,
        0xB3, 0x32, 0x56, 0x97, 0x1F, 0xA1, 0x10, 0xF4
    };

    wc_Sm4 sm4;
    byte enc[SM4_BLOCK_SIZE * 4];
    byte dec[SM4_BLOCK_SIZE * 4];
    byte tag[SM4_BLOCK_SIZE];
    int ret;

    ret = wc_Sm4Init(&sm4, NULL, INVALID_DEVID);
    if (ret != 0)
        return -6720;

    /* Encrypt and decrypt using encrypt with CCM. */
    ret = wc_Sm4SetKey(&sm4, k1, sizeof(k1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Sm4CcmEncrypt(&sm4, enc, p1, sizeof(p1), i1, sizeof(i1), tag,
        sizeof(tag), a1, sizeof(a1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, c1, sizeof(c1)) != 0)
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(tag, tag1, sizeof(tag1)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_Sm4CcmDecrypt(&sm4, dec, enc, sizeof(c1), i1, sizeof(i1), tag,
        sizeof(tag), a1, sizeof(a1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(dec, p1, sizeof(p1)) != 0)
        return WC_TEST_RET_ENC_NC;

    /* RFC8998 test vector. */
    ret = wc_Sm4SetKey(&sm4, k2, sizeof(k2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_Sm4CcmEncrypt(&sm4, enc, p2, sizeof(p2), i2, sizeof(i2), tag,
        sizeof(tag), a2, sizeof(a2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(enc, c2, sizeof(c2)) != 0)
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(tag, tag2, sizeof(tag2)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_Sm4CcmDecrypt(&sm4, dec, enc, sizeof(c2), i2, sizeof(i2), tag,
        sizeof(tag), a2, sizeof(a2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(dec, p2, sizeof(p2)) != 0)
        return WC_TEST_RET_ENC_NC;

    wc_Sm4Free(&sm4);

    return 0;
}
#endif

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sm4_test(void)
{
    wc_test_ret_t ret;

#ifdef WOLFSSL_SM4_ECB
    ret = sm4_ecb_test();
    if (ret != 0)
        return ret;
#endif
#ifdef WOLFSSL_SM4_CBC
    ret = sm4_cbc_test();
    if (ret != 0)
        return ret;
#endif
#ifdef WOLFSSL_SM4_CTR
    ret = sm4_ctr_test();
    if (ret != 0)
        return ret;
#endif
#ifdef WOLFSSL_SM4_GCM
    ret = sm4_gcm_test();
    if (ret != 0)
        return ret;
#endif
#ifdef WOLFSSL_SM4_CCM
    ret = sm4_ccm_test();
    if (ret != 0)
        return ret;
#endif

    return 0;
}
#endif

#ifdef HAVE_XCHACHA
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t XChaCha_test(void) {
    wc_test_ret_t ret;

    WOLFSSL_SMALL_STACK_STATIC const byte Plaintext[] = {
        0x54, 0x68, 0x65, 0x20, 0x64, 0x68, 0x6f, 0x6c, 0x65, 0x20, 0x28, 0x70, 0x72, 0x6f, 0x6e, 0x6f, /* The dhole (prono */
        0x75, 0x6e, 0x63, 0x65, 0x64, 0x20, 0x22, 0x64, 0x6f, 0x6c, 0x65, 0x22, 0x29, 0x20, 0x69, 0x73, /* unced "dole") is */
        0x20, 0x61, 0x6c, 0x73, 0x6f, 0x20, 0x6b, 0x6e, 0x6f, 0x77, 0x6e, 0x20, 0x61, 0x73, 0x20, 0x74, /*  also known as t */
        0x68, 0x65, 0x20, 0x41, 0x73, 0x69, 0x61, 0x74, 0x69, 0x63, 0x20, 0x77, 0x69, 0x6c, 0x64, 0x20, /* he Asiatic wild  */
        0x64, 0x6f, 0x67, 0x2c, 0x20, 0x72, 0x65, 0x64, 0x20, 0x64, 0x6f, 0x67, 0x2c, 0x20, 0x61, 0x6e, /* dog, red dog, an */
        0x64, 0x20, 0x77, 0x68, 0x69, 0x73, 0x74, 0x6c, 0x69, 0x6e, 0x67, 0x20, 0x64, 0x6f, 0x67, 0x2e, /* d whistling dog. */
        0x20, 0x49, 0x74, 0x20, 0x69, 0x73, 0x20, 0x61, 0x62, 0x6f, 0x75, 0x74, 0x20, 0x74, 0x68, 0x65, /*  It is about the */
        0x20, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x6f, 0x66, 0x20, 0x61, 0x20, 0x47, 0x65, 0x72, 0x6d, 0x61, /*  size of a Germa */
        0x6e, 0x20, 0x73, 0x68, 0x65, 0x70, 0x68, 0x65, 0x72, 0x64, 0x20, 0x62, 0x75, 0x74, 0x20, 0x6c, /* n shepherd but l */
        0x6f, 0x6f, 0x6b, 0x73, 0x20, 0x6d, 0x6f, 0x72, 0x65, 0x20, 0x6c, 0x69, 0x6b, 0x65, 0x20, 0x61, /* ooks more like a */
        0x20, 0x6c, 0x6f, 0x6e, 0x67, 0x2d, 0x6c, 0x65, 0x67, 0x67, 0x65, 0x64, 0x20, 0x66, 0x6f, 0x78, /*  long-legged fox */
        0x2e, 0x20, 0x54, 0x68, 0x69, 0x73, 0x20, 0x68, 0x69, 0x67, 0x68, 0x6c, 0x79, 0x20, 0x65, 0x6c, /* . This highly el */
        0x75, 0x73, 0x69, 0x76, 0x65, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x73, 0x6b, 0x69, 0x6c, 0x6c, 0x65, /* usive and skille */
        0x64, 0x20, 0x6a, 0x75, 0x6d, 0x70, 0x65, 0x72, 0x20, 0x69, 0x73, 0x20, 0x63, 0x6c, 0x61, 0x73, /* d jumper is clas */
        0x73, 0x69, 0x66, 0x69, 0x65, 0x64, 0x20, 0x77, 0x69, 0x74, 0x68, 0x20, 0x77, 0x6f, 0x6c, 0x76, /* sified with wolv */
        0x65, 0x73, 0x2c, 0x20, 0x63, 0x6f, 0x79, 0x6f, 0x74, 0x65, 0x73, 0x2c, 0x20, 0x6a, 0x61, 0x63, /* es, coyotes, jac */
        0x6b, 0x61, 0x6c, 0x73, 0x2c, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x66, 0x6f, 0x78, 0x65, 0x73, 0x20, /* kals, and foxes  */
        0x69, 0x6e, 0x20, 0x74, 0x68, 0x65, 0x20, 0x74, 0x61, 0x78, 0x6f, 0x6e, 0x6f, 0x6d, 0x69, 0x63, /* in the taxonomic */
        0x20, 0x66, 0x61, 0x6d, 0x69, 0x6c, 0x79, 0x20, 0x43, 0x61, 0x6e, 0x69, 0x64, 0x61, 0x65, 0x2e  /*  family Canidae. */
    };

    WOLFSSL_SMALL_STACK_STATIC const byte Key[] = {
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte IV[] = {
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, /* @ABCDEFGHIJKLMNO */
        0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x58 };                                               /* PQRSTUVW */

    WOLFSSL_SMALL_STACK_STATIC const byte Ciphertext[] = {
        0x45, 0x59, 0xab, 0xba, 0x4e, 0x48, 0xc1, 0x61, 0x02, 0xe8, 0xbb, 0x2c, 0x05, 0xe6, 0x94, 0x7f,
        0x50, 0xa7, 0x86, 0xde, 0x16, 0x2f, 0x9b, 0x0b, 0x7e, 0x59, 0x2a, 0x9b, 0x53, 0xd0, 0xd4, 0xe9,
        0x8d, 0x8d, 0x64, 0x10, 0xd5, 0x40, 0xa1, 0xa6, 0x37, 0x5b, 0x26, 0xd8, 0x0d, 0xac, 0xe4, 0xfa,
        0xb5, 0x23, 0x84, 0xc7, 0x31, 0xac, 0xbf, 0x16, 0xa5, 0x92, 0x3c, 0x0c, 0x48, 0xd3, 0x57, 0x5d,
        0x4d, 0x0d, 0x2c, 0x67, 0x3b, 0x66, 0x6f, 0xaa, 0x73, 0x10, 0x61, 0x27, 0x77, 0x01, 0x09, 0x3a,
        0x6b, 0xf7, 0xa1, 0x58, 0xa8, 0x86, 0x42, 0x92, 0xa4, 0x1c, 0x48, 0xe3, 0xa9, 0xb4, 0xc0, 0xda,
        0xec, 0xe0, 0xf8, 0xd9, 0x8d, 0x0d, 0x7e, 0x05, 0xb3, 0x7a, 0x30, 0x7b, 0xbb, 0x66, 0x33, 0x31,
        0x64, 0xec, 0x9e, 0x1b, 0x24, 0xea, 0x0d, 0x6c, 0x3f, 0xfd, 0xdc, 0xec, 0x4f, 0x68, 0xe7, 0x44,
        0x30, 0x56, 0x19, 0x3a, 0x03, 0xc8, 0x10, 0xe1, 0x13, 0x44, 0xca, 0x06, 0xd8, 0xed, 0x8a, 0x2b,
        0xfb, 0x1e, 0x8d, 0x48, 0xcf, 0xa6, 0xbc, 0x0e, 0xb4, 0xe2, 0x46, 0x4b, 0x74, 0x81, 0x42, 0x40,
        0x7c, 0x9f, 0x43, 0x1a, 0xee, 0x76, 0x99, 0x60, 0xe1, 0x5b, 0xa8, 0xb9, 0x68, 0x90, 0x46, 0x6e,
        0xf2, 0x45, 0x75, 0x99, 0x85, 0x23, 0x85, 0xc6, 0x61, 0xf7, 0x52, 0xce, 0x20, 0xf9, 0xda, 0x0c,
        0x09, 0xab, 0x6b, 0x19, 0xdf, 0x74, 0xe7, 0x6a, 0x95, 0x96, 0x74, 0x46, 0xf8, 0xd0, 0xfd, 0x41,
        0x5e, 0x7b, 0xee, 0x2a, 0x12, 0xa1, 0x14, 0xc2, 0x0e, 0xb5, 0x29, 0x2a, 0xe7, 0xa3, 0x49, 0xae,
        0x57, 0x78, 0x20, 0xd5, 0x52, 0x0a, 0x1f, 0x3f, 0xb6, 0x2a, 0x17, 0xce, 0x6a, 0x7e, 0x68, 0xfa,
        0x7c, 0x79, 0x11, 0x1d, 0x88, 0x60, 0x92, 0x0b, 0xc0, 0x48, 0xef, 0x43, 0xfe, 0x84, 0x48, 0x6c,
        0xcb, 0x87, 0xc2, 0x5f, 0x0a, 0xe0, 0x45, 0xf0, 0xcc, 0xe1, 0xe7, 0x98, 0x9a, 0x9a, 0xa2, 0x20,
        0xa2, 0x8b, 0xdd, 0x48, 0x27, 0xe7, 0x51, 0xa2, 0x4a, 0x6d, 0x5c, 0x62, 0xd7, 0x90, 0xa6, 0x63,
        0x93, 0xb9, 0x31, 0x11, 0xc1, 0xa5, 0x5d, 0xd7, 0x42, 0x1a, 0x10, 0x18, 0x49, 0x74, 0xc7, 0xc5
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    struct ChaCha *chacha = (struct ChaCha *)XMALLOC(sizeof *chacha, HEAP_HINT, DYNAMIC_TYPE_CIPHER);
    byte *buf1 = (byte *)XMALLOC(sizeof Plaintext, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *buf2 = (byte *)XMALLOC(sizeof Plaintext, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((chacha == NULL) || (buf1 == NULL) || (buf2 == NULL))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#else
    struct ChaCha chacha[1];
    byte buf1[sizeof Plaintext];
    byte buf2[sizeof Plaintext];
#endif

    ret = wc_XChacha_SetKey(chacha, Key, sizeof Key, IV, sizeof IV, 0);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_Chacha_Process(chacha, buf1, Plaintext, sizeof Plaintext);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    if (XMEMCMP(buf1, Ciphertext, sizeof Plaintext))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = wc_XChacha_SetKey(chacha, Key, sizeof Key, IV, sizeof IV, 0);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_Chacha_Process(chacha, buf2, buf1, sizeof Plaintext);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    if (XMEMCMP(buf2, Plaintext, sizeof Plaintext))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

  out:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (chacha)
        XFREE(chacha, HEAP_HINT, DYNAMIC_TYPE_CIPHER);
    if (buf1)
        XFREE(buf1, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (buf2)
        XFREE(buf2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif /* HAVE_XCHACHA */

#if defined(HAVE_XCHACHA) && defined(HAVE_POLY1305)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t XChaCha20Poly1305_test(void) {
    wc_test_ret_t ret;

    WOLFSSL_SMALL_STACK_STATIC const byte Plaintext[] = {
        0x4c, 0x61, 0x64, 0x69, 0x65, 0x73, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x47, 0x65, 0x6e, 0x74, 0x6c, /* Ladies and Gentl */
        0x65, 0x6d, 0x65, 0x6e, 0x20, 0x6f, 0x66, 0x20, 0x74, 0x68, 0x65, 0x20, 0x63, 0x6c, 0x61, 0x73, /* emen of the clas */
        0x73, 0x20, 0x6f, 0x66, 0x20, 0x27, 0x39, 0x39, 0x3a, 0x20, 0x49, 0x66, 0x20, 0x49, 0x20, 0x63, /* s of '99: If I c */
        0x6f, 0x75, 0x6c, 0x64, 0x20, 0x6f, 0x66, 0x66, 0x65, 0x72, 0x20, 0x79, 0x6f, 0x75, 0x20, 0x6f, /* ould offer you o */
        0x6e, 0x6c, 0x79, 0x20, 0x6f, 0x6e, 0x65, 0x20, 0x74, 0x69, 0x70, 0x20, 0x66, 0x6f, 0x72, 0x20, /* nly one tip for */
        0x74, 0x68, 0x65, 0x20, 0x66, 0x75, 0x74, 0x75, 0x72, 0x65, 0x2c, 0x20, 0x73, 0x75, 0x6e, 0x73, /* the future, suns */
        0x63, 0x72, 0x65, 0x65, 0x6e, 0x20, 0x77, 0x6f, 0x75, 0x6c, 0x64, 0x20, 0x62, 0x65, 0x20, 0x69, /* creen would be i */
        0x74, 0x2e };                                                                                   /* t. */

    WOLFSSL_SMALL_STACK_STATIC const byte AAD[] = { 0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7 }; /* PQRS........ */

    WOLFSSL_SMALL_STACK_STATIC const byte Key[] = {
        0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
        0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f
    };

    WOLFSSL_SMALL_STACK_STATIC const byte IV[] = {
        0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47, 0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f, /* @ABCDEFGHIJKLMNO */
        0x50, 0x51, 0x52, 0x53, 0x54, 0x55, 0x56, 0x57 };                                               /* PQRSTUVW */

    WOLFSSL_SMALL_STACK_STATIC const byte Ciphertext[] = {
      0xbd, 0x6d, 0x17, 0x9d, 0x3e, 0x83, 0xd4, 0x3b, 0x95, 0x76, 0x57, 0x94, 0x93, 0xc0, 0xe9, 0x39,
      0x57, 0x2a, 0x17, 0x00, 0x25, 0x2b, 0xfa, 0xcc, 0xbe, 0xd2, 0x90, 0x2c, 0x21, 0x39, 0x6c, 0xbb,
      0x73, 0x1c, 0x7f, 0x1b, 0x0b, 0x4a, 0xa6, 0x44, 0x0b, 0xf3, 0xa8, 0x2f, 0x4e, 0xda, 0x7e, 0x39,
      0xae, 0x64, 0xc6, 0x70, 0x8c, 0x54, 0xc2, 0x16, 0xcb, 0x96, 0xb7, 0x2e, 0x12, 0x13, 0xb4, 0x52,
      0x2f, 0x8c, 0x9b, 0xa4, 0x0d, 0xb5, 0xd9, 0x45, 0xb1, 0x1b, 0x69, 0xb9, 0x82, 0xc1, 0xbb, 0x9e,
      0x3f, 0x3f, 0xac, 0x2b, 0xc3, 0x69, 0x48, 0x8f, 0x76, 0xb2, 0x38, 0x35, 0x65, 0xd3, 0xff, 0xf9,
      0x21, 0xf9, 0x66, 0x4c, 0x97, 0x63, 0x7d, 0xa9, 0x76, 0x88, 0x12, 0xf6, 0x15, 0xc6, 0x8b, 0x13,
      0xb5, 0x2e };

    WOLFSSL_SMALL_STACK_STATIC const byte Tag[] = {
      0xc0, 0x87, 0x59, 0x24, 0xc1, 0xc7, 0x98, 0x79, 0x47, 0xde, 0xaf, 0xd8, 0x78, 0x0a, 0xcf, 0x49
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte *buf1 = (byte *)XMALLOC(sizeof Ciphertext + sizeof Tag, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *buf2 = (byte *)XMALLOC(sizeof Plaintext, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((buf1 == NULL) || (buf2 == NULL))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
#else
    byte buf1[sizeof Ciphertext + sizeof Tag];
    byte buf2[sizeof Plaintext];
#endif

    ret = wc_XChaCha20Poly1305_Encrypt(buf1, sizeof Ciphertext + sizeof Tag,
                                       Plaintext, sizeof Plaintext,
                                       AAD, sizeof AAD,
                                       IV, sizeof IV,
                                       Key, sizeof Key);

    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    if (XMEMCMP(buf1, Ciphertext, sizeof Ciphertext))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    if (XMEMCMP(buf1 + sizeof Ciphertext, Tag, CHACHA20_POLY1305_AEAD_AUTHTAG_SIZE))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    ret = wc_XChaCha20Poly1305_Decrypt(buf2, sizeof Plaintext,
                                 buf1, sizeof Ciphertext + sizeof Tag,
                                       AAD, sizeof AAD,
                                       IV, sizeof IV,
                                       Key, sizeof Key);

    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    if (XMEMCMP(buf2, Plaintext, sizeof Plaintext))
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

  out:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (buf1 != NULL)
        XFREE(buf1, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (buf2 != NULL)
        XFREE(buf2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif /* defined(HAVE_XCHACHA) && defined(HAVE_POLY1305) */

#ifndef WC_NO_RNG
static wc_test_ret_t _rng_test(WC_RNG* rng, int errorOffset)
{
    byte block[32];
    wc_test_ret_t ret;
    int i;

    XMEMSET(block, 0, sizeof(block));

    ret = wc_RNG_GenerateBlock(rng, block, sizeof(block));
    if (ret != 0) {
        ret = 1;
        goto exit;
    }

    /* Check for 0's */
    for (i=0; i<(int)sizeof(block); i++) {
        if (block[i] == 0) {
            ret++;
        }
    }
    /* All zeros count check */
    if (ret >= (int)sizeof(block)) {
        ret = 2;
        goto exit;
    }

    ret = wc_RNG_GenerateByte(rng, block);
    if (ret != 0) {
        ret = 3;
        goto exit;
    }

    /* Parameter validation testing. */
    ret = wc_RNG_GenerateBlock(NULL, block, sizeof(block));
    if (ret != BAD_FUNC_ARG) {
        ret = 4;
        goto exit;
    }
    ret = wc_RNG_GenerateBlock(rng, NULL, sizeof(block));
    if (ret != BAD_FUNC_ARG) {
        ret = 5;
        goto exit;
    }

    ret = wc_RNG_GenerateByte(NULL, block);
    if (ret != BAD_FUNC_ARG) {
        ret = 6;
        goto exit;
    }
    ret = wc_RNG_GenerateByte(rng, NULL);
    if (ret != BAD_FUNC_ARG) {
        ret = 7;
        goto exit;
    }

    ret = 0;

exit:
    if (ret != 0)
        ret = errorOffset - (ret * 1000000);

    return ret;
}


static wc_test_ret_t random_rng_test(void)
{
    WC_RNG localRng;
    WC_RNG* rng;
    wc_test_ret_t ret;

    rng = &localRng;
    /* Test stack based RNG. */
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(rng);
#endif
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = _rng_test(rng, WC_TEST_RET_ENC_NC);

    /* Make sure and free RNG */
    wc_FreeRng(rng);

    if (ret != 0)
        return ret;

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && !defined(WOLFSSL_NO_MALLOC)
    {
        byte nonce[8] = { 0 };
        /* Test dynamic RNG. */
        rng = wc_rng_new(nonce, (word32)sizeof(nonce), HEAP_HINT);
        if (rng == NULL)
            return WC_TEST_RET_ENC_ERRNO;

        ret = _rng_test(rng, WC_TEST_RET_ENC_NC);

        wc_rng_free(rng);
    }
#endif

    return ret;
}

#if defined(HAVE_HASHDRBG) && !defined(CUSTOM_RAND_GENERATE_BLOCK)

#ifdef WC_RNG_SEED_CB
static int seed_cb(OS_Seed* os, byte* output, word32 sz)
{
    word32 i;
    (void)os;
    /* Known answer test. Set the seed to the same value every time. */
    for (i = 0; i < sz; i++)
        output[i] = (byte)i;
    return 0;
}

static wc_test_ret_t rng_seed_test(void)
{
#ifndef HAVE_FIPS
    WOLFSSL_SMALL_STACK_STATIC const byte check[] =
    {
        0x83, 0x46, 0x65, 0x2f, 0x5c, 0x44, 0x16, 0x5f,
        0xb3, 0x89, 0x26, 0xde, 0x0b, 0x6b, 0xa2, 0x06,
        0x7e, 0xa7, 0x9a, 0x55, 0x22, 0x01, 0xb0, 0x22,
        0xf4, 0x7e, 0xa2, 0x66, 0xc4, 0x08, 0x6f, 0xba
    };
#else
    /* FIPS uses a longer seed, so different check value. */
    WOLFSSL_SMALL_STACK_STATIC const byte check[] =
    {
        0xaf, 0x31, 0xcc, 0xef, 0xa9, 0x29, 0x4c, 0x24,
        0xbd, 0xa5, 0xa3, 0x52, 0x69, 0xf3, 0xb9, 0xb2,
        0x1e, 0xd4, 0x52, 0x3b, 0x9a, 0x96, 0x06, 0x20,
        0xc0, 0x5f, 0x44, 0x06, 0x1f, 0x80, 0xdf, 0xe0
    };
#endif
    byte output[WC_SHA256_DIGEST_SIZE];
    WC_RNG rng;
    wc_test_ret_t ret;

    ret = wc_SetSeed_Cb(seed_cb);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    }
    ret = wc_InitRng(&rng);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    }
    ret = wc_RNG_GenerateBlock(&rng, output, sizeof(output));
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    }
    ret = XMEMCMP(output, check, sizeof(output));
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);
    }
    ret = wc_FreeRng(&rng);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    }
    ret = wc_SetSeed_Cb(wc_GenerateSeed);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    }

out:
    return ret;
}
#endif


WOLFSSL_TEST_SUBROUTINE wc_test_ret_t random_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte test1Entropy[] =
    {
        0xa6, 0x5a, 0xd0, 0xf3, 0x45, 0xdb, 0x4e, 0x0e, 0xff, 0xe8, 0x75, 0xc3,
        0xa2, 0xe7, 0x1f, 0x42, 0xc7, 0x12, 0x9d, 0x62, 0x0f, 0xf5, 0xc1, 0x19,
        0xa9, 0xef, 0x55, 0xf0, 0x51, 0x85, 0xe0, 0xfb, 0x85, 0x81, 0xf9, 0x31,
        0x75, 0x17, 0x27, 0x6e, 0x06, 0xe9, 0x60, 0x7d, 0xdb, 0xcb, 0xcc, 0x2e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte test1Output[] =
    {
        0xd3, 0xe1, 0x60, 0xc3, 0x5b, 0x99, 0xf3, 0x40, 0xb2, 0x62, 0x82, 0x64,
        0xd1, 0x75, 0x10, 0x60, 0xe0, 0x04, 0x5d, 0xa3, 0x83, 0xff, 0x57, 0xa5,
        0x7d, 0x73, 0xa6, 0x73, 0xd2, 0xb8, 0xd8, 0x0d, 0xaa, 0xf6, 0xa6, 0xc3,
        0x5a, 0x91, 0xbb, 0x45, 0x79, 0xd7, 0x3f, 0xd0, 0xc8, 0xfe, 0xd1, 0x11,
        0xb0, 0x39, 0x13, 0x06, 0x82, 0x8a, 0xdf, 0xed, 0x52, 0x8f, 0x01, 0x81,
        0x21, 0xb3, 0xfe, 0xbd, 0xc3, 0x43, 0xe7, 0x97, 0xb8, 0x7d, 0xbb, 0x63,
        0xdb, 0x13, 0x33, 0xde, 0xd9, 0xd1, 0xec, 0xe1, 0x77, 0xcf, 0xa6, 0xb7,
        0x1f, 0xe8, 0xab, 0x1d, 0xa4, 0x66, 0x24, 0xed, 0x64, 0x15, 0xe5, 0x1c,
        0xcd, 0xe2, 0xc7, 0xca, 0x86, 0xe2, 0x83, 0x99, 0x0e, 0xea, 0xeb, 0x91,
        0x12, 0x04, 0x15, 0x52, 0x8b, 0x22, 0x95, 0x91, 0x02, 0x81, 0xb0, 0x2d,
        0xd4, 0x31, 0xf4, 0xc9, 0xf7, 0x04, 0x27, 0xdf
    };
    WOLFSSL_SMALL_STACK_STATIC const byte test2EntropyA[] =
    {
        0x63, 0x36, 0x33, 0x77, 0xe4, 0x1e, 0x86, 0x46, 0x8d, 0xeb, 0x0a, 0xb4,
        0xa8, 0xed, 0x68, 0x3f, 0x6a, 0x13, 0x4e, 0x47, 0xe0, 0x14, 0xc7, 0x00,
        0x45, 0x4e, 0x81, 0xe9, 0x53, 0x58, 0xa5, 0x69, 0x80, 0x8a, 0xa3, 0x8f,
        0x2a, 0x72, 0xa6, 0x23, 0x59, 0x91, 0x5a, 0x9f, 0x8a, 0x04, 0xca, 0x68
    };
    WOLFSSL_SMALL_STACK_STATIC const byte test2EntropyB[] =
    {
        0xe6, 0x2b, 0x8a, 0x8e, 0xe8, 0xf1, 0x41, 0xb6, 0x98, 0x05, 0x66, 0xe3,
        0xbf, 0xe3, 0xc0, 0x49, 0x03, 0xda, 0xd4, 0xac, 0x2c, 0xdf, 0x9f, 0x22,
        0x80, 0x01, 0x0a, 0x67, 0x39, 0xbc, 0x83, 0xd3
    };
    WOLFSSL_SMALL_STACK_STATIC const byte test2Output[] =
    {
        0x04, 0xee, 0xc6, 0x3b, 0xb2, 0x31, 0xdf, 0x2c, 0x63, 0x0a, 0x1a, 0xfb,
        0xe7, 0x24, 0x94, 0x9d, 0x00, 0x5a, 0x58, 0x78, 0x51, 0xe1, 0xaa, 0x79,
        0x5e, 0x47, 0x73, 0x47, 0xc8, 0xb0, 0x56, 0x62, 0x1c, 0x18, 0xbd, 0xdc,
        0xdd, 0x8d, 0x99, 0xfc, 0x5f, 0xc2, 0xb9, 0x20, 0x53, 0xd8, 0xcf, 0xac,
        0xfb, 0x0b, 0xb8, 0x83, 0x12, 0x05, 0xfa, 0xd1, 0xdd, 0xd6, 0xc0, 0x71,
        0x31, 0x8a, 0x60, 0x18, 0xf0, 0x3b, 0x73, 0xf5, 0xed, 0xe4, 0xd4, 0xd0,
        0x71, 0xf9, 0xde, 0x03, 0xfd, 0x7a, 0xea, 0x10, 0x5d, 0x92, 0x99, 0xb8,
        0xaf, 0x99, 0xaa, 0x07, 0x5b, 0xdb, 0x4d, 0xb9, 0xaa, 0x28, 0xc1, 0x8d,
        0x17, 0x4b, 0x56, 0xee, 0x2a, 0x01, 0x4d, 0x09, 0x88, 0x96, 0xff, 0x22,
        0x82, 0xc9, 0x55, 0xa8, 0x19, 0x69, 0xe0, 0x69, 0xfa, 0x8c, 0xe0, 0x07,
        0xa1, 0x80, 0x18, 0x3a, 0x07, 0xdf, 0xae, 0x17
    };

    byte output[WC_SHA256_DIGEST_SIZE * 4];
    wc_test_ret_t ret;

    ret = wc_RNG_HealthTest(0, test1Entropy, sizeof(test1Entropy), NULL, 0,
                            output, sizeof(output));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(test1Output, output, sizeof(output)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_RNG_HealthTest(1, test2EntropyA, sizeof(test2EntropyA),
                            test2EntropyB, sizeof(test2EntropyB),
                            output, sizeof(output));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(test2Output, output, sizeof(output)) != 0)
        return WC_TEST_RET_ENC_NC;

    /* Basic RNG generate block test */
    if ((ret = random_rng_test()) != 0)
        return ret;

    /* Test the seed check function. */
#if !(defined(HAVE_FIPS) || defined(HAVE_SELFTEST)) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))
    {
        word32 i, outputSz;

        /* Repeat the same byte over and over. Should fail. */
        outputSz = sizeof(output);
        XMEMSET(output, 1, outputSz);
        ret = wc_RNG_TestSeed(output, outputSz);
        if (ret == 0)
            return WC_TEST_RET_ENC_NC;

        /* Every byte of the entropy scratch is different,
         * entropy is a single byte that shouldn't match. */
        outputSz = (sizeof(output) / 2) + 1;
        for (i = 0; i < outputSz; i++)
            output[i] = (byte)i;
        ret = wc_RNG_TestSeed(output, outputSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);

        outputSz = sizeof(output);
        for (i = 0; i < outputSz; i++)
            output[i] = (byte)i;
        ret = wc_RNG_TestSeed(output, outputSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
    }
#endif

    /* Test the seed callback. */
#ifdef WC_RNG_SEED_CB
    if ((ret = rng_seed_test()) != 0)
        return ret;
#endif

    return 0;
}

#else

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t random_test(void)
{
    /* Basic RNG generate block test */
    return random_rng_test();
}

#endif /* HAVE_HASHDRBG && !CUSTOM_RAND_GENERATE_BLOCK */
#endif /* WC_NO_RNG */

#ifndef MEM_TEST_SZ
    #define MEM_TEST_SZ 1024
#endif

#if defined(WOLFSSL_STATIC_MEMORY) || !defined(WOLFSSL_NO_MALLOC)
static int simple_mem_test(int sz)
{
    int ret = 0;
    byte* b;
    int i;

    b = (byte*)XMALLOC(sz, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (b == NULL) {
        return WC_TEST_RET_ENC_NC;
    }
    /* utilize memory */
    for (i = 0; i < sz; i++) {
        b[i] = (byte)i;
    }
    /* read back and verify */
    for (i = 0; i < sz; i++) {
        if (b[i] != (byte)i) {
            ret = WC_TEST_RET_ENC_NC;
            break;
        }
    }
    XFREE(b, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    return ret;
}
#endif

/* If successful, returns the first letter of the byte array `in`.
**
** This is a deceptively simple test of a read-only embedded Linux file system.
** (e.g CFLAGS `-mfdpic` and `-mforce-l32`) for Xtensa Linux ESP32. When the
** `-mforce-l32` is missing, access to `in` will fail with Illegal Instruction.
** Const is on read-only memory-mapped file system, *not* loaded in app memory.
**
** Edit with caution. See PR #6523. */
static wc_test_ret_t const_byte_ptr_test(const byte* in, word32 *outJ)
{
    wc_test_ret_t ret = 0;
    volatile word32 j = -1; /* must be volatile to properly detect error */

    ret = (wc_test_ret_t)*in; /* accessed *in value. */
    (void)ret;
    j = *outJ; /* Found index to use in const array. */

    if (j == 0) {
#ifdef WOLFSSL_DEBUG
        printf("Testing const byte ptr reference...\n");
#endif
        /* although j is zero, in[0] does not detect the Illegal instruction */
        ret = in[j]; /* The big test: can we actually access the `in` data? */
    }
    else {
        ret = -1;
    }

    return ret;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t memory_test(void)
{
    wc_test_ret_t ret = 0;
    word32 j = 0; /* used in embedded const pointer test */

#if defined(COMPLEX_MEM_TEST) || defined(WOLFSSL_STATIC_MEMORY)
    int i;
#endif
#ifdef WOLFSSL_STATIC_MEMORY
    word32 size[] = { WOLFMEM_BUCKETS };
    word32 dist[] = { WOLFMEM_DIST };
    byte buffer[30000]; /* make large enough to involve many bucket sizes */
    int pad = -(int)((wc_ptr_t)buffer) & (WOLFSSL_STATIC_ALIGN - 1);
              /* pad to account for if head of buffer is not at set memory
               * alignment when tests are ran */
#endif

#ifdef WOLFSSL_STATIC_MEMORY
    /* check macro settings */
    if (sizeof(size)/sizeof(word32) != WOLFMEM_MAX_BUCKETS) {
        return WC_TEST_RET_ENC_NC;
    }

    if (sizeof(dist)/sizeof(word32) != WOLFMEM_MAX_BUCKETS) {
        return WC_TEST_RET_ENC_NC;
    }

    for (i = 0; i < WOLFMEM_MAX_BUCKETS; i++) {
        if ((size[i] % WOLFSSL_STATIC_ALIGN) != 0) {
            /* each element in array should be divisible by alignment size */
            return WC_TEST_RET_ENC_NC;
        }
    }

    for (i = 1; i < WOLFMEM_MAX_BUCKETS; i++) {
        if (size[i - 1] >= size[i]) {
            return WC_TEST_RET_ENC_NC; /* sizes should be in increasing order */
        }
    }

    /* check that padding size returned is possible */
    if (wolfSSL_MemoryPaddingSz() < WOLFSSL_STATIC_ALIGN) {
        return WC_TEST_RET_ENC_NC; /* no room for wc_Memory struct */
    }

    ret = wolfSSL_MemoryPaddingSz();
    if (ret < 0) {
        return WC_TEST_RET_ENC_EC(ret);
    }

    if (wolfSSL_MemoryPaddingSz() % WOLFSSL_STATIC_ALIGN != 0) {
        return WC_TEST_RET_ENC_NC; /* not aligned! */
    }

    /* check function to return optimum buffer size (rounded down) */
    ret = wolfSSL_StaticBufferSz(buffer, sizeof(buffer), WOLFMEM_GENERAL);
    if (ret < 0) {
        return WC_TEST_RET_ENC_EC(ret);
    }
    if ((ret - pad) % WOLFSSL_STATIC_ALIGN != 0) {
        return WC_TEST_RET_ENC_NC; /* not aligned! */
    }

    if ((unsigned int)ret > sizeof(buffer)) {
        return WC_TEST_RET_ENC_NC; /* did not round down as expected */
    }

    if (ret != wolfSSL_StaticBufferSz(buffer, ret, WOLFMEM_GENERAL)) {
        return WC_TEST_RET_ENC_NC; /* return value changed when using suggested
                                    * value
                                    */
    }

    ret = wolfSSL_MemoryPaddingSz();
    ret += pad; /* add space that is going to be needed if buffer not aligned */
    if (wolfSSL_StaticBufferSz(buffer, size[0] + ret + 1, WOLFMEM_GENERAL) !=
            (ret + (int)size[0])) {
        return WC_TEST_RET_ENC_NC; /* did not round down to nearest bucket
                                    * value
                                    */
    }

    ret = wolfSSL_StaticBufferSz(buffer, sizeof(buffer), WOLFMEM_IO_POOL);
    if ((ret - pad) < 0) {
        return WC_TEST_RET_ENC_NC;
    }

    if (((ret - pad) % (WOLFMEM_IO_SZ + wolfSSL_MemoryPaddingSz())) != 0) {
        return WC_TEST_RET_ENC_NC; /* not even chunks of memory for IO size */
    }

    if (((ret - pad) % WOLFSSL_STATIC_ALIGN) != 0) {
        return WC_TEST_RET_ENC_NC; /* memory not aligned */
    }

    /* check for passing bad or unknown arguments to functions */
    if (wolfSSL_StaticBufferSz(NULL, 1, WOLFMEM_GENERAL) > 0) {
        return WC_TEST_RET_ENC_NC;
    }

    if (wolfSSL_StaticBufferSz(buffer, 1, WOLFMEM_GENERAL) != 0) {
        return WC_TEST_RET_ENC_NC; /* should round to 0
                             since struct + bucket will not fit */
    }

    (void)dist; /* avoid static analysis warning of variable not used */
#endif

#if defined(WOLFSSL_STATIC_MEMORY) || !defined(WOLFSSL_NO_MALLOC)
    /* simple test */
    ret = simple_mem_test(MEM_TEST_SZ);
    if (ret != 0)
        return ret;
#endif

#ifdef COMPLEX_MEM_TEST
    /* test various size blocks */
    for (i = 1; i < MEM_TEST_SZ; i*=2) {
        ret = simple_mem_test(i);
        if (ret != 0)
            return ret;
    }
#endif

#if !defined(USE_FAST_MATH) && !defined(WOLFSSL_NO_MALLOC) && defined(XREALLOC)
    /* realloc test */
    {
        byte *b = (byte*)XMALLOC(MEM_TEST_SZ, HEAP_HINT,
                                 DYNAMIC_TYPE_TMP_BUFFER);
        #ifndef WOLFSSL_NO_REALLOC
        byte *c = NULL;
        if (b) {
            c = (byte*)XREALLOC(b, MEM_TEST_SZ+sizeof(word32), HEAP_HINT,
                                DYNAMIC_TYPE_TMP_BUFFER);
            if (c)
                b = c;
        }
        #endif
        if (b)
            XFREE(b, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if ((b == NULL)
        #ifndef WOLFSSL_NO_REALLOC
                || (c == NULL)
        #endif
        ) {
            return WC_TEST_RET_ENC_ERRNO;
        }
    }
#endif

    if (ret == 0) {
        /* This test is only interesting on embedded R/O Flash systems */
        if (const_byte_ptr_test(const_byte_array, &j) != CBPTR_EXPECTED) {
            ret = 1;
        }
    }

    return ret;
}

#ifndef NO_FILESYSTEM

/* Cert Paths */
#ifdef FREESCALE_MQX
    #define CERT_PREFIX "a:\\"
    #define CERT_PATH_SEP "\\"
#elif defined(WOLFSSL_uTKERNEL2)
    #define CERT_PREFIX "/uda/"
    #define CERT_PATH_SEP "/"
#elif defined(_WIN32_WCE)
    #define CERT_PREFIX "\\windows\\"
    #define CERT_PATH_SEP "\\"
#endif

#ifndef CERT_PREFIX
    #define CERT_PREFIX "./"
#endif
#ifndef CERT_PATH_SEP
    #define CERT_PATH_SEP "/"
#endif
#ifndef CERT_WRITE_TEMP_DIR
    #define CERT_WRITE_TEMP_DIR CERT_PREFIX
#endif

#define CERT_ROOT CERT_PREFIX "certs" CERT_PATH_SEP

/* Generated Test Certs */
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
    #if !defined(NO_RSA) && !defined(NO_ASN)
        static const char* clientKey  = CERT_ROOT "client-key.der";
        static const char* clientCert = CERT_ROOT "client-cert.der";
        #ifdef WOLFSSL_CERT_EXT
            static const char* clientKeyPub  = CERT_ROOT "client-keyPub.der";
        #endif
    #endif /* !NO_RSA && !NO_ASN */
#endif
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    #if !defined(NO_RSA) && !defined(NO_ASN)
        #if defined(WOLFSSL_CERT_GEN) || defined(HAVE_PKCS7)
            static const char* rsaCaKeyFile  = CERT_ROOT "ca-key.der";
            #ifdef WOLFSSL_CERT_GEN
            static const char* rsaCaCertFile = CERT_ROOT "ca-cert.pem";
            #endif
            #if defined(WOLFSSL_ALT_NAMES) || defined(HAVE_PKCS7)
            static const char* rsaCaCertDerFile = CERT_ROOT "ca-cert.der";
            #endif
            #ifdef HAVE_PKCS7
                static const char* rsaServerCertDerFile =
                                               CERT_ROOT "server-cert.der";
                static const char* rsaServerKeyDerFile =
                                               CERT_ROOT "server-key.der";
            #endif
        #endif
    #endif /* !NO_RSA && !NO_ASN */
#endif /* !USE_CERT_BUFFER_* */
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
    #if !defined(NO_ASN) && !defined(NO_DH)
        static const char* dhParamsFile = CERT_ROOT "dh2048.der";
    #endif
#endif
#if !defined(NO_ASN) && !defined(NO_DH)
    #if defined(WOLFSSL_DH_EXTRA) && (!defined(HAVE_FIPS) ||                \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
        #if !defined(USE_CERT_BUFFERS_2048)
            static const char* dhKeyFile = CERT_ROOT "statickeys/dh-ffdhe2048.der";
            static const char* dhKeyPubFile = CERT_ROOT "statickeys/dh-ffdhe2048-pub.der";
        #endif
    #endif
#endif
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    #ifndef NO_DSA
        static const char* dsaKey = CERT_ROOT "dsa2048.der";
    #endif
#endif /* !USE_CERT_BUFFER_* */
#if !defined(USE_CERT_BUFFERS_256) && !defined(NO_ECC256)
    #ifdef HAVE_ECC
        /* cert files to be used in rsa cert gen test, check if RSA enabled */
        #ifdef HAVE_ECC_KEY_IMPORT
            static const char* eccKeyDerFile = CERT_ROOT "ecc-key.der";
        #endif
#endif
#if !defined(USE_CERT_BUFFERS_256) && !defined(NO_ASN)
        #if defined(HAVE_ECC) && defined(WOLFSSL_CERT_GEN) && \
                !defined(NO_ECC_SECP)
            #ifndef NO_RSA
                static const char* eccKeyPubFileDer = CERT_ROOT "ecc-keyPub.der";
            #endif
            #ifndef NO_ASN_TIME
                static const char* eccCaKeyFile  = CERT_ROOT "ca-ecc-key.der";
                static const char* eccCaCertFile = CERT_ROOT "ca-ecc-cert.pem";
                #ifdef ENABLE_ECC384_CERT_GEN_TEST
                    static const char* eccCaKey384File =
                                                CERT_ROOT "ca-ecc384-key.der";
                    static const char* eccCaCert384File =
                                                CERT_ROOT "ca-ecc384-cert.pem";
                #endif
            #endif
        #endif
        #if defined(HAVE_PKCS7) && defined(HAVE_ECC)
            static const char* eccClientKey  = CERT_ROOT "ecc-client-key.der";
            static const char* eccClientCert = CERT_ROOT "client-ecc-cert.der";
        #endif
    #endif /* HAVE_ECC */
    #ifdef HAVE_ED25519
        #ifdef WOLFSSL_TEST_CERT
            static const char* serverEd25519Cert =
                                         CERT_ROOT "ed25519/server-ed25519.der";
            static const char* caEd25519Cert     =
                                             CERT_ROOT "ed25519/ca-ed25519.der";
        #endif
    #endif
    #ifdef HAVE_ED448
        #ifdef WOLFSSL_TEST_CERT
            static const char* serverEd448Cert =
                                             CERT_ROOT "ed448/server-ed448.der";
            static const char* caEd448Cert     = CERT_ROOT "ed448/ca-ed448.der";
        #endif
    #endif
#endif /* !USE_CERT_BUFFER_* */

#if !defined(NO_ASN_TIME) && !defined(NO_RSA) && defined(WOLFSSL_TEST_CERT) && \
    !defined(NO_FILESYSTEM)
    static const char* certExtNc  =
            CERT_ROOT "test" CERT_PATH_SEP "cert-ext-nc.der";
    static const char* certExtIa  =
            CERT_ROOT "test" CERT_PATH_SEP "cert-ext-ia.der";
    static const char* certExtNct =
            CERT_ROOT "test" CERT_PATH_SEP "cert-ext-nct.der";
#endif

#ifndef NO_WRITE_TEMP_FILES
#ifdef HAVE_ECC
    #ifndef NO_ECC_SECP
        #if defined(WOLFSSL_CERT_GEN) && !defined(NO_ASN_TIME)
        static const char* certEccPemFile = CERT_WRITE_TEMP_DIR "certecc.pem";
        static const char* certEccDerFile = CERT_WRITE_TEMP_DIR "certecc.der";
        #endif
        #if defined(WOLFSSL_CERT_GEN) && !defined(NO_RSA)
        static const char* certEccRsaPemFile = CERT_WRITE_TEMP_DIR "certeccrsa.pem";
        static const char* certEccRsaDerFile = CERT_WRITE_TEMP_DIR "certeccrsa.der";
        #endif
    #endif
    #if defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG) && \
        !defined(WOLF_CRYPTO_CB_ONLY_ECC) && !defined(NO_ASN_CRYPT)
        static const char* eccCaKeyPemFile  = CERT_WRITE_TEMP_DIR "ecc-key.pem";
        static const char* eccPubKeyDerFile = CERT_WRITE_TEMP_DIR "ecc-public-key.der";
        static const char* eccCaKeyTempFile = CERT_WRITE_TEMP_DIR "ecc-key.der";
    #if defined(HAVE_PKCS8) && !defined(WC_NO_RNG) && \
       !defined(WOLF_CRYPTO_CB_ONLY_ECC)
        static const char* eccPkcs8KeyDerFile = CERT_WRITE_TEMP_DIR "ecc-key-pkcs8.der";
    #endif
    #endif /* HAVE_ECC_KEY_EXPORT */
#endif /* HAVE_ECC */

#ifndef NO_RSA
    #if defined(WOLFSSL_CERT_GEN) && !defined(NO_ASN_TIME)
        static const char* otherCertDerFile = CERT_WRITE_TEMP_DIR "othercert.der";
        static const char* certDerFile = CERT_WRITE_TEMP_DIR "cert.der";
        static const char* otherCertPemFile = CERT_WRITE_TEMP_DIR "othercert.pem";
        static const char* certPemFile = CERT_WRITE_TEMP_DIR "cert.pem";
        #if defined(WOLFSSL_CERT_REQ) && !defined(WOLFSSL_NO_MALLOC)
            static const char* certReqDerFile = CERT_WRITE_TEMP_DIR "certreq.der";
            static const char* certReqPemFile = CERT_WRITE_TEMP_DIR "certreq.pem";
        #endif
    #endif
#endif /* !NO_RSA */

#if !defined(NO_RSA) || !defined(NO_DSA)
    #ifdef WOLFSSL_KEY_GEN
        static const char* keyDerFile = CERT_WRITE_TEMP_DIR "key.der";
        static const char* keyPemFile = CERT_WRITE_TEMP_DIR "key.pem";
    #endif
#endif

#endif /* !NO_WRITE_TEMP_FILES */
#endif /* !NO_FILESYSTEM */


#if defined(WOLFSSL_CERT_GEN) && (!defined(NO_RSA) || defined(HAVE_ECC)) || \
  (defined(WOLFSSL_TEST_CERT) && (defined(HAVE_ED25519) || defined(HAVE_ED448)))
static CertName certDefaultName;
static void initDefaultName(void)
{
#if defined(WOLFSSL_MULTI_ATTRIB) && defined(WOLFSSL_TEST_CERT)
    NameAttrib* n;
#endif

    XMEMCPY(certDefaultName.country, "US", sizeof("US"));
    certDefaultName.countryEnc = CTC_PRINTABLE;
    XMEMCPY(certDefaultName.state, "Oregon", sizeof("Oregon"));
    certDefaultName.stateEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.street, "Main St", sizeof("Main St"));
    certDefaultName.streetEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.locality, "Portland", sizeof("Portland"));
    certDefaultName.localityEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.sur, "Test", sizeof("Test"));
    certDefaultName.surEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.org, "wolfSSL", sizeof("wolfSSL"));
    certDefaultName.orgEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.unit, "Development", sizeof("Development"));
    certDefaultName.unitEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.commonName, "www.wolfssl.com", sizeof("www.wolfssl.com"));
    certDefaultName.commonNameEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.serialDev, "wolfSSL12345", sizeof("wolfSSL12345"));
    certDefaultName.serialDevEnc = CTC_PRINTABLE;
    XMEMCPY(certDefaultName.postalCode, "12-456", sizeof("12-456"));
    certDefaultName.postalCodeEnc = CTC_PRINTABLE;
#ifdef WOLFSSL_CERT_EXT
    XMEMCPY(certDefaultName.busCat, "Private Organization", sizeof("Private Organization"));
    certDefaultName.busCatEnc = CTC_UTF8;
    XMEMCPY(certDefaultName.joiSt, "US", sizeof("US"));
    certDefaultName.joiStEnc = CTC_PRINTABLE;
    XMEMCPY(certDefaultName.joiC, "Oregon", sizeof("Oregon"));
    certDefaultName.joiCEnc = CTC_PRINTABLE;
#endif
    XMEMCPY(certDefaultName.email, "info@wolfssl.com", sizeof("info@wolfssl.com"));
    XMEMCPY(certDefaultName.userId, "TestUserID", sizeof("TestUserID"));
    certDefaultName.userIdEnc = CTC_PRINTABLE;

#if defined(WOLFSSL_MULTI_ATTRIB) && defined(WOLFSSL_TEST_CERT)
    /* test having additional OUs and setting DC */
    n = &certDefaultName.name[0];
    n->id   = ASN_ORGUNIT_NAME;
    n->type = CTC_UTF8;
    n->sz   = XSTRLEN("Development-2");
    XMEMCPY(n->value, "Development-2", sizeof("Development-2"));

    #if CTC_MAX_ATTRIB > 3
    n = &certDefaultName.name[1];
    n->id   = ASN_DOMAIN_COMPONENT;
    n->type = CTC_UTF8;
    n->sz   = XSTRLEN("com");
    XMEMCPY(n->value, "com", sizeof("com"));

    n = &certDefaultName.name[2];
    n->id   = ASN_DOMAIN_COMPONENT;
    n->type = CTC_UTF8;
    n->sz   = XSTRLEN("wolfssl");
    XMEMCPY(n->value, "wolfssl", sizeof("wolfssl"));
    #endif
#endif /* WOLFSSL_MULTI_ATTRIB && WOLFSSL_TEST_CERT */

#ifdef WOLFSSL_CUSTOM_OID
    /* TODO: Add test case for custom OID's */
#endif
}

#ifdef WOLFSSL_CERT_EXT
    #if ((defined(HAVE_ED25519) || defined(HAVE_ED448)) && \
                                defined(WOLFSSL_TEST_CERT)) || defined(HAVE_ECC)
    WOLFSSL_SMALL_STACK_STATIC const char certKeyUsage[] =
        "digitalSignature,nonRepudiation";
    #endif
    #if defined(WOLFSSL_CERT_REQ) && !defined(NO_RSA) && !defined(NO_ASN_TIME)
        WOLFSSL_SMALL_STACK_STATIC const char certKeyUsage2[] =
        "digitalSignature,nonRepudiation,keyEncipherment,keyAgreement";
    #endif
#endif /* WOLFSSL_CERT_EXT */
#endif /* WOLFSSL_CERT_GEN */

#ifndef NO_RSA

#if !defined(NO_ASN_TIME) && !defined(NO_RSA) && defined(WOLFSSL_TEST_CERT) && \
    !defined(NO_FILESYSTEM)
static byte minSerial[] = { 0x02, 0x01, 0x01 };
static byte minName[] = { 0x30, 0x00 };
static byte nameBad[] = {
    0x30, 0x08,
          0x31, 0x06,
                0x30, 0x04,
                      0x06, 0x02,
                            0x55, 0x04,
};
static byte minDates[] = {
    0x30, 0x1e,
          0x17, 0x0d,
                0x31, 0x38, 0x30, 0x34, 0x31, 0x33, 0x31, 0x35,
                0x32, 0x33, 0x31, 0x30, 0x5a,
          0x17, 0x0d,
                0x32, 0x31, 0x30, 0x31, 0x30, 0x37, 0x31, 0x35,
                0x32, 0x33, 0x31, 0x30, 0x5a
};
static byte minPubKey[] = {
    0x30, 0x1c,
          0x30, 0x0d,
                0x06, 0x09,
                      0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
                      0x01,
                0x05, 0x00,
          0x03, 0x0b,
                0x00, 0x30, 0x08,
                            0x02, 0x01,
                                  0x03,
                            0x02, 0x03,
                                  0x01, 0x00, 0x01
};
static byte minSigAlg[] = {
    0x30, 0x0d,
          0x06, 0x09,
                0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01,
                0x0b,
          0x05, 0x00
};
static byte minSig[] = {
    0x03, 0x01,
          0x00
};

static int add_seq(byte* certData, int offset, byte* data, byte length)
{
    XMEMMOVE(certData + offset + 2, data, length);
    certData[offset++] = 0x30;
    certData[offset++] = length;
    return offset + length;
}
static int add_data(byte* certData, int offset, byte* data, byte length)
{
    XMEMCPY(certData + offset, data, length);
    return offset + length;
}

static wc_test_ret_t cert_asn1_test(void)
{
    wc_test_ret_t ret;
    int len[3];
    DecodedCert cert;
    byte certData[114];
    byte* badCert = NULL;

    len[2] = add_data(certData, 0, minSerial, (byte)sizeof(minSerial));
    len[2] = add_data(certData, len[2], minSigAlg, (byte)sizeof(minSigAlg));
    len[2] = add_data(certData, len[2], minName, (byte)sizeof(minName));
    len[2] = add_data(certData, len[2], minDates, (byte)sizeof(minDates));
    len[2] = add_data(certData, len[2], minName, (byte)sizeof(minName));
    len[2] = add_data(certData, len[2], minPubKey, (byte)sizeof(minPubKey));
    len[1] = add_seq(certData, 0, certData, len[2]);
    len[1] = add_data(certData, len[1], minSigAlg, (byte)sizeof(minSigAlg));
    len[1] = add_data(certData, len[1], minSig, (byte)sizeof(minSig));
    len[0] = add_seq(certData, 0, certData, len[1]);

    /* Minimal good certificate */
    InitDecodedCert(&cert, certData, len[0], 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
    FreeDecodedCert(&cert);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    /* Bad issuer name */
    len[2] = add_data(certData, 0, minSerial, (byte)sizeof(minSerial));
    len[2] = add_data(certData, len[2], minSigAlg, (byte)sizeof(minSigAlg));
    len[2] = add_data(certData, len[2], nameBad, (byte)sizeof(nameBad));
    len[2] = add_data(certData, len[2], minDates, (byte)sizeof(minDates));
    len[2] = add_data(certData, len[2], minName, (byte)sizeof(minName));
    len[2] = add_data(certData, len[2], minPubKey, (byte)sizeof(minPubKey));
    len[1] = add_seq(certData, 0, certData, len[2]);
    len[1] = add_data(certData, len[1], minSigAlg, (byte)sizeof(minSigAlg));
    len[1] = add_data(certData, len[1], minSig, (byte)sizeof(minSig));
    len[0] = add_seq(certData, 0, certData, len[1]);
    /* Put data into allocated buffer to allow access error checking. */
    badCert = (byte*)XMALLOC(len[0], HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XMEMCPY(badCert, certData, len[0]);
    InitDecodedCert(&cert, badCert, len[0], 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
    FreeDecodedCert(&cert);
    if (ret != ASN_PARSE_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    }
    XFREE(badCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    badCert = NULL;
    ret = 0;

done:
    if (badCert != NULL)
        XFREE(badCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    return ret;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t cert_test(void)
{
#if !defined(NO_FILESYSTEM)
    DecodedCert cert;
    byte*       tmp;
    size_t      bytes;
    XFILE       file;
    wc_test_ret_t ret;

    tmp = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL)
        return WC_TEST_RET_ENC_ERRNO;

    /* Certificate with Name Constraints extension. */
    file = XFOPEN(certExtNc, "rb");
    if (!file) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    InitDecodedCert(&cert, tmp, (word32)bytes, 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    FreeDecodedCert(&cert);

    /* Certificate with Inhibit Any Policy extension. */
    file = XFOPEN(certExtIa, "rb");
    if (!file) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    InitDecodedCert(&cert, tmp, (word32)bytes, 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    FreeDecodedCert(&cert);

    /* Certificate with Netscape Certificate Type extension. */
    file = XFOPEN(certExtNct, "rb");
    if (!file) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    }
    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    InitDecodedCert(&cert, tmp, (word32)bytes, 0);
    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, NULL);
#ifndef IGNORE_NETSCAPE_CERT_TYPE
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#else
    if (ret != ASN_CRIT_EXT_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    }
    ret = 0;
#endif

done:
    FreeDecodedCert(&cert);
    XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* !NO_FILESYSTEM */

    if (ret == 0)
        ret = cert_asn1_test();

    return ret;
}
#endif /* WOLFSSL_TEST_CERT */

#if defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_TEST_CERT) && \
   !defined(NO_FILESYSTEM) && defined(WOLFSSL_CERT_GEN)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t certext_test(void)
{
    DecodedCert cert;
    byte*       tmp;
    size_t      bytes;
    XFILE       file;
    wc_test_ret_t ret;

    /* created from rsa_test : othercert.der */
    byte skid_rsa[]   = "\x33\xD8\x45\x66\xD7\x68\x87\x18\x7E\x54"
                        "\x0D\x70\x27\x91\xC7\x26\xD7\x85\x65\xC0";

    /* created from rsa_test : othercert.der */
    byte akid_rsa[] = "\x27\x8E\x67\x11\x74\xC3\x26\x1D\x3F\xED"
                      "\x33\x63\xB3\xA4\xD8\x1D\x30\xE5\xE8\xD5";

#ifdef HAVE_ECC
   /* created from ecc_test_cert_gen : certecc.der */
#ifdef ENABLE_ECC384_CERT_GEN_TEST
    /* Authority key id from ./certs/ca-ecc384-cert.pem */
    byte akid_ecc[] = "\xAB\xE0\xC3\x26\x4C\x18\xD4\x72\xBB\xD2"
                      "\x84\x8C\x9C\x0A\x05\x92\x80\x12\x53\x52";
#else
    /* Authority key id from ./certs/ca-ecc-cert.pem */
    byte akid_ecc[] = "\x56\x8E\x9A\xC3\xF0\x42\xDE\x18\xB9\x45"
                      "\x55\x6E\xF9\x93\xCF\xEA\xC3\xF3\xA5\x21";
#endif
#endif /* HAVE_ECC */

    /* created from rsa_test : cert.der */
    byte kid_ca[] = "\x33\xD8\x45\x66\xD7\x68\x87\x18\x7E\x54"
                    "\x0D\x70\x27\x91\xC7\x26\xD7\x85\x65\xC0";

    tmp = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL)
        return WC_TEST_RET_ENC_ERRNO;

    /* load othercert.der (Cert signed by an authority) */
    file = XFOPEN(otherCertDerFile, "rb");
    if (!file) {
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_ERRNO;
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        return WC_TEST_RET_ENC_ERRNO;

    InitDecodedCert(&cert, tmp, (word32)bytes, 0);

    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* check the SKID from a RSA certificate */
    if (XMEMCMP(skid_rsa, cert.extSubjKeyId, sizeof(cert.extSubjKeyId)))
        return WC_TEST_RET_ENC_NC;

    /* check the AKID from an RSA certificate */
    if (XMEMCMP(akid_rsa, cert.extAuthKeyId, sizeof(cert.extAuthKeyId)))
        return WC_TEST_RET_ENC_NC;

    /* check the Key Usage from an RSA certificate */
    if (!cert.extKeyUsageSet)
        return WC_TEST_RET_ENC_NC;

    if (cert.extKeyUsage != (KEYUSE_KEY_ENCIPHER|KEYUSE_KEY_AGREE))
        return WC_TEST_RET_ENC_NC;

    /* check the CA Basic Constraints from an RSA certificate */
    if (cert.isCA)
        return WC_TEST_RET_ENC_NC;

#ifndef WOLFSSL_SEP /* test only if not using SEP policies */
    /* check the Certificate Policies Id */
    if (cert.extCertPoliciesNb != 1)
        return WC_TEST_RET_ENC_NC;

    if (strncmp(cert.extCertPolicies[0], "2.16.840.1.101.3.4.1.42", 23))
        return WC_TEST_RET_ENC_NC;
#endif

    FreeDecodedCert(&cert);

#ifdef HAVE_ECC
    /* load certecc.der (Cert signed by our ECC CA test in ecc_test_cert_gen) */
    file = XFOPEN(certEccDerFile, "rb");
    if (!file) {
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_ERRNO;
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        return WC_TEST_RET_ENC_ERRNO;

    InitDecodedCert(&cert, tmp, (word32)bytes, 0);

    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* check the SKID from a ECC certificate - generated dynamically */

    /* check the AKID from an ECC certificate */
    if (XMEMCMP(akid_ecc, cert.extAuthKeyId, sizeof(cert.extAuthKeyId)))
        return WC_TEST_RET_ENC_NC;

    /* check the Key Usage from an ECC certificate */
    if (!cert.extKeyUsageSet)
        return WC_TEST_RET_ENC_NC;

    if (cert.extKeyUsage != (KEYUSE_DIGITAL_SIG|KEYUSE_CONTENT_COMMIT))
        return WC_TEST_RET_ENC_NC;

    /* check the CA Basic Constraints from an ECC certificate */
    if (cert.isCA)
        return WC_TEST_RET_ENC_NC;

#ifndef WOLFSSL_SEP /* test only if not using SEP policies */
    /* check the Certificate Policies Id */
    if (cert.extCertPoliciesNb != 2)
        return WC_TEST_RET_ENC_NC;

    if (strncmp(cert.extCertPolicies[0], "2.4.589440.587.101.2.1.9632587.1", 32))
        return WC_TEST_RET_ENC_NC;

    if (strncmp(cert.extCertPolicies[1], "1.2.13025.489.1.113549", 22))
        return WC_TEST_RET_ENC_NC;
#endif

    FreeDecodedCert(&cert);
#endif /* HAVE_ECC */

    /* load cert.der (self signed certificate) */
    file = XFOPEN(certDerFile, "rb");
    if (!file) {
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_ERRNO;
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        return WC_TEST_RET_ENC_ERRNO;

    InitDecodedCert(&cert, tmp, (word32)bytes, 0);

    ret = ParseCert(&cert, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* check the SKID from a CA certificate */
    if (XMEMCMP(kid_ca, cert.extSubjKeyId, sizeof(cert.extSubjKeyId)))
        return WC_TEST_RET_ENC_NC;

    /* check the AKID from an CA certificate */
    if (XMEMCMP(kid_ca, cert.extAuthKeyId, sizeof(cert.extAuthKeyId)))
        return WC_TEST_RET_ENC_NC;

    /* check the Key Usage from CA certificate */
    if (!cert.extKeyUsageSet)
        return WC_TEST_RET_ENC_NC;

    if (cert.extKeyUsage != (KEYUSE_KEY_CERT_SIGN|KEYUSE_CRL_SIGN))
        return WC_TEST_RET_ENC_NC;

    /* check the CA Basic Constraints CA certificate */
    if (!cert.isCA)
        return WC_TEST_RET_ENC_NC;

#ifndef WOLFSSL_SEP /* test only if not using SEP policies */
    /* check the Certificate Policies Id */
    if (cert.extCertPoliciesNb != 2)
        return WC_TEST_RET_ENC_NC;

    if (strncmp(cert.extCertPolicies[0], "2.16.840.1.101.3.4.1.42", 23))
        return WC_TEST_RET_ENC_NC;

    if (strncmp(cert.extCertPolicies[1], "1.2.840.113549.1.9.16.6.5", 25))
        return WC_TEST_RET_ENC_NC;
#endif

    FreeDecodedCert(&cert);
    XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    return 0;
}
#endif /* WOLFSSL_CERT_EXT && WOLFSSL_TEST_CERT &&
          !NO_FILESYSTEM && WOLFSSL_CERT_GEN */

#if defined(WOLFSSL_CERT_GEN_CACHE) && defined(WOLFSSL_TEST_CERT) && \
    defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t decodedCertCache_test(void)
{
    wc_test_ret_t ret = 0;
    Cert cert;
    FILE* file;
    byte* der;
    word32 derSz;

    derSz = FOURK_BUF;
    der = (byte *)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL)
        ret = WC_TEST_RET_ENC_NC;

    if (ret == 0) {
        /* load cert.der */
        file = XFOPEN(certDerFile, "rb");
        if (file != NULL) {
            derSz = (word32)XFREAD(der, 1, FOURK_BUF, file);
            XFCLOSE(file);
            if (derSz == 0)
                ret = WC_TEST_RET_ENC_ERRNO;

        }
        else
            ret = WC_TEST_RET_ENC_ERRNO;
    }

    if (ret == 0) {
        ret = wc_InitCert_ex(&cert, HEAP_HINT, devId);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetSubjectBuffer(&cert, der, derSz);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetSubjectBuffer(NULL, der, derSz);
        if (ret == BAD_FUNC_ARG)
            ret = 0;
        else
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetSubjectRaw(&cert, der, derSz);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetSubjectRaw(NULL, der, derSz);
        if (ret == BAD_FUNC_ARG)
            ret = 0;
        else
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetIssuerBuffer(&cert, der, derSz);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetIssuerBuffer(NULL, der, derSz);
        if (ret == BAD_FUNC_ARG)
            ret = 0;
        else
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetIssuerRaw(&cert, der, derSz);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetIssuerRaw(NULL, der, derSz);
        if (ret == BAD_FUNC_ARG)
            ret = 0;
        else
            ret = WC_TEST_RET_ENC_EC(ret);
    }

#ifdef WOLFSSL_ALT_NAMES
    if (ret == 0) {
        ret = wc_SetAltNamesBuffer(&cert, der, derSz);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetAltNamesBuffer(NULL, der, derSz);
        if (ret == BAD_FUNC_ARG)
            ret = 0;
        else
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetDatesBuffer(&cert, der, derSz);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetDatesBuffer(NULL, der, derSz);
        if (ret == BAD_FUNC_ARG)
            ret = 0;
        else
            ret = WC_TEST_RET_ENC_EC(ret);
    }
#endif

    if (ret == 0) {
        ret = wc_SetAuthKeyIdFromCert(&cert, der, derSz);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_SetAuthKeyIdFromCert(NULL, der, derSz);
        if (ret == BAD_FUNC_ARG)
            ret = 0;
        else
            ret = WC_TEST_RET_ENC_NC;
    }

    wc_SetCert_Free(&cert);
    if (ret == 0) {
        if(cert.decodedCert != NULL)
            ret = WC_TEST_RET_ENC_NC;
    }

    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    return ret;
}
#endif /* defined(WOLFSSL_CERT_GEN_CACHE) && defined(WOLFSSL_TEST_CERT) &&
          defined(WOLFSSL_CERT_EXT) && defined(WOLFSSL_CERT_GEN) */

#define RSA_TEST_BYTES 512 /* up to 4096-bit key */

#if !defined(NO_ASN) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
                                               !defined(WOLFSSL_RSA_VERIFY_ONLY)
static wc_test_ret_t rsa_flatten_test(RsaKey* key)
{
    wc_test_ret_t ret;
    byte   e[RSA_TEST_BYTES];
    byte   n[RSA_TEST_BYTES];
    word32 eSz = sizeof(e);
    word32 nSz = sizeof(n);

    /* Parameter Validation testing. */
    ret = wc_RsaFlattenPublicKey(NULL, e, &eSz, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaFlattenPublicKey(key, NULL, &eSz, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaFlattenPublicKey(key, e, NULL, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, NULL, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, n, NULL);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != BAD_FUNC_ARG)
#endif
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, n, &nSz);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    eSz = 0;
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != RSA_BUFFER_E)
#endif
        return WC_TEST_RET_ENC_EC(ret);
    eSz = sizeof(e);
    nSz = 0;
    ret = wc_RsaFlattenPublicKey(key, e, &eSz, n, &nSz);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#else
    if (ret != RSA_BUFFER_E)
#endif
        return WC_TEST_RET_ENC_EC(ret);

    return 0;
}
#endif /* NO_ASN */

#if !defined(HAVE_FIPS) && !defined(HAVE_USER_RSA) && !defined(NO_ASN) \
    && !defined(WOLFSSL_RSA_VERIFY_ONLY)
static wc_test_ret_t rsa_export_key_test(RsaKey* key)
{
    wc_test_ret_t ret;
    byte e[3];
    word32 eSz = sizeof(e);
    byte n[RSA_TEST_BYTES];
    word32 nSz = sizeof(n);
    byte d[RSA_TEST_BYTES];
    word32 dSz = sizeof(d);
    byte p[RSA_TEST_BYTES/2];
    word32 pSz = sizeof(p);
    byte q[RSA_TEST_BYTES/2];
    word32 qSz = sizeof(q);
    word32 zero = 0;

    ret = wc_RsaExportKey(NULL, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, NULL, &eSz, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, NULL, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, NULL, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, NULL, d, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, NULL, &dSz, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, NULL, p, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, NULL, &pSz, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, NULL, q, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, NULL, &qSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q, NULL);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_RsaExportKey(key, e, &zero, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != RSA_BUFFER_E)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &zero, d, &dSz, p, &pSz, q, &qSz);
    if (ret != RSA_BUFFER_E)
        return WC_TEST_RET_ENC_EC(ret);
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &zero, p, &pSz, q, &qSz);
    if (ret != RSA_BUFFER_E)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &zero, q, &qSz);
    if (ret != RSA_BUFFER_E)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q, &zero);
    if (ret != RSA_BUFFER_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

    ret = wc_RsaExportKey(key, e, &eSz, n, &nSz, d, &dSz, p, &pSz, q, &qSz);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    return 0;
}
#endif /* !HAVE_FIPS && !USER_RSA && !NO_ASN */

#ifndef NO_SIG_WRAPPER
static wc_test_ret_t rsa_sig_test(RsaKey* key, word32 keyLen, int modLen, WC_RNG* rng)
{
    wc_test_ret_t ret;
    word32 sigSz;
    WOLFSSL_SMALL_STACK_STATIC const byte in[] = TEST_STRING;
    WOLFSSL_SMALL_STACK_STATIC const byte hash[] = {
        0xf2, 0x02, 0x95, 0x65, 0xcb, 0xf6, 0x2a, 0x59,
        0x39, 0x2c, 0x05, 0xff, 0x0e, 0x29, 0xaf, 0xfe,
        0x47, 0x33, 0x8c, 0x99, 0x8d, 0x58, 0x64, 0x83,
        0xa6, 0x58, 0x0a, 0x33, 0x0b, 0x84, 0x5f, 0x5f
    };
    WOLFSSL_SMALL_STACK_STATIC const byte hashEnc[] = {
        0x30, 0x31, 0x30, 0x0d, 0x06, 0x09, 0x60, 0x86,
        0x48, 0x01, 0x65, 0x03, 0x04, 0x02, 0x01, 0x05,
        0x00, 0x04, 0x20,

        0xf2, 0x02, 0x95, 0x65, 0xcb, 0xf6, 0x2a, 0x59,
        0x39, 0x2c, 0x05, 0xff, 0x0e, 0x29, 0xaf, 0xfe,
        0x47, 0x33, 0x8c, 0x99, 0x8d, 0x58, 0x64, 0x83,
        0xa6, 0x58, 0x0a, 0x33, 0x0b, 0x84, 0x5f, 0x5f
    };
    word32 inLen = (word32)XSTRLEN((char*)in);
    byte   out[RSA_TEST_BYTES];

    /* Parameter Validation testing. */
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_NONE, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_RSA, key, 0);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    sigSz = (word32)modLen;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, NULL,
                               inLen, out, &sigSz, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               0, out, &sigSz, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, NULL, &sigSz, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, NULL, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, NULL, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, key, 0, rng);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, key, keyLen, NULL);
#ifdef HAVE_USER_RSA
    /* Implementation using IPP Libraries returns:
     *     -101 = USER_CRYPTO_ERROR
     */
    if (ret == 0)
#elif defined(WOLFSSL_AFALG_XILINX_RSA) || defined(WOLFSSL_XILINX_CRYPT)
    /* blinding / rng handled with hardware acceleration */
    if (ret != 0)
#elif defined(WOLFSSL_ASYNC_CRYPT) || defined(WOLF_CRYPTO_CB)
    /* async may not require RNG */
    #if defined(WOLF_CRYPTO_CB_ONLY_RSA)
    if (ret != NO_VALID_DEVID)
    #else
    if (ret != 0 && ret != MISSING_RNG_E)
    #endif
#elif defined(HAVE_FIPS) || !defined(WC_RSA_BLINDING)
    /* FIPS140 implementation does not do blinding */
    if (ret != 0)
#elif defined(WOLFSSL_RSA_PUBLIC_ONLY) || defined(WOLFSSL_RSA_VERIFY_ONLY)
    if (ret != SIG_TYPE_E)
#elif defined(WOLFSSL_CRYPTOCELL) || defined(WOLFSSL_SE050)
    /* RNG is handled by hardware */
    if (ret != 0)
#else
    if (ret != MISSING_RNG_E)
#endif
        return WC_TEST_RET_ENC_EC(ret);
    sigSz = 0;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, key, keyLen, rng);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, NULL,
                             inLen, out, (word32)modLen, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             0, out, (word32)modLen, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, NULL, (word32)modLen, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, 0, key, keyLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, (word32)modLen, NULL, keyLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, (word32)modLen, key, 0);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

#ifndef HAVE_ECC
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_ECC, key, keyLen);
    if (ret != SIG_TYPE_E)
        return WC_TEST_RET_ENC_EC(ret);
#endif
#if defined(WOLF_CRYPTO_CB_ONLY_RSA)
    return 0;
#endif
    /* Use APIs. */
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_RSA, key, keyLen);
    if (ret != modLen)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_RSA_W_ENC, key, keyLen);
    if (ret != modLen)
        return WC_TEST_RET_ENC_EC(ret);

    sigSz = (word32)ret;
#if !defined(WOLFSSL_RSA_PUBLIC_ONLY) && !defined(WOLFSSL_RSA_VERIFY_ONLY)
    XMEMSET(out, 0, sizeof(out));
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                               inLen, out, &sigSz, key, keyLen, rng);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, (word32)modLen, key, keyLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    sigSz = (word32)sizeof(out);
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA_W_ENC,
                               in, inLen, out, &sigSz, key, keyLen, rng);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA_W_ENC,
                             in, inLen, out, (word32)modLen, key, keyLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Wrong signature type. */
    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA, in,
                             inLen, out, (word32)modLen, key, keyLen);
    if (ret == 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* check hash functions */
    sigSz = (word32)sizeof(out);
    ret = wc_SignatureGenerateHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA,
        hash, (int)sizeof(hash), out, &sigSz, key, keyLen, rng);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SignatureVerifyHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA,
        hash, (int)sizeof(hash), out, (word32)modLen, key, keyLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    sigSz = (word32)sizeof(out);
    ret = wc_SignatureGenerateHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA_W_ENC,
        hashEnc, (int)sizeof(hashEnc), out, &sigSz, key, keyLen, rng);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SignatureVerifyHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_RSA_W_ENC,
        hashEnc, (int)sizeof(hashEnc), out, (word32)modLen, key, keyLen);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
#else
    (void)hash;
    (void)hashEnc;
#endif /* !WOLFSSL_RSA_PUBLIC_ONLY && !WOLFSSL_RSA_VERIFY_ONLY */

    return 0;
}
#endif /* !NO_SIG_WRAPPER */

#ifdef WC_RSA_NONBLOCK
static wc_test_ret_t rsa_nb_test(RsaKey* key, const byte* in, word32 inLen, byte* out,
    word32 outSz, byte* plain, word32 plainSz, WC_RNG* rng)
{
    wc_test_ret_t ret = 0;
    int count;
    int signSz = 0;
    RsaNb nb;
    byte* inlinePlain = NULL;

    /* Enable non-blocking RSA mode - provide context */
    ret = wc_RsaSetNonBlock(key, &nb);
    if (ret != 0)
        return ret;

#ifdef WC_RSA_NONBLOCK_TIME
    /* Enable time based RSA blocking. 8 microseconds max (3.1GHz) */
    ret = wc_RsaSetNonBlockTime(key, 8, 3100);
    if (ret != 0)
        return ret;
#endif

    count = 0;
    do {
        ret = wc_RsaSSL_Sign(in, inLen, out, outSz, key, rng);
        count++; /* track number of would blocks */
        if (ret == FP_WOULDBLOCK) {
            /* do "other" work here */
        }
    } while (ret == FP_WOULDBLOCK);
    if (ret < 0) {
        return ret;
    }
#if defined(DEBUG_WOLFSSL) || defined(WOLFSSL_DEBUG_NONBLOCK)
    printf("RSA non-block sign: %d times\n", count);
#endif
    signSz = ret;

    /* Test non-blocking verify */
    XMEMSET(plain, 0, plainSz);
    count = 0;
    do {
        ret = wc_RsaSSL_Verify(out, (word32)signSz, plain, plainSz, key);
        count++; /* track number of would blocks */
        if (ret == FP_WOULDBLOCK) {
            /* do "other" work here */
        }
    } while (ret == FP_WOULDBLOCK);
    if (ret < 0) {
        return ret;
    }
#if defined(DEBUG_WOLFSSL) || defined(WOLFSSL_DEBUG_NONBLOCK)
    printf("RSA non-block verify: %d times\n", count);
#endif

    if (signSz == ret && XMEMCMP(plain, in, (size_t)ret)) {
        return SIG_VERIFY_E;
    }

    /* Test inline non-blocking verify */
    count = 0;
    do {
        ret = wc_RsaSSL_VerifyInline(out, (word32)signSz, &inlinePlain, key);
        count++; /* track number of would blocks */
        if (ret == FP_WOULDBLOCK) {
            /* do "other" work here */
        }
    } while (ret == FP_WOULDBLOCK);
    if (ret < 0) {
        return ret;
    }
#if defined(DEBUG_WOLFSSL) || defined(WOLFSSL_DEBUG_NONBLOCK)
    printf("RSA non-block inline verify: %d times\n", count);
#endif

    if (signSz == ret && XMEMCMP(inlinePlain, in, (size_t)ret)) {
        return SIG_VERIFY_E;
    }

    /* Disabling non-block RSA mode */
    ret = wc_RsaSetNonBlock(key, NULL);

    (void)count;

    return 0;
}
#endif

#if !defined(HAVE_USER_RSA) && !defined(NO_ASN)
static wc_test_ret_t rsa_decode_test(RsaKey* keyPub)
{
    wc_test_ret_t ret;
    word32     inSz;
    word32     inOutIdx;
    WOLFSSL_SMALL_STACK_STATIC const byte n[2] = { 0x00, 0x23 };
    WOLFSSL_SMALL_STACK_STATIC const byte e[2] = { 0x00, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte good[] = { 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1,
           0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte goodAlgId[] = {
            0x30, 0x18, 0x30, 0x16,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x03, 0x09, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte goodAlgIdNull[] = {
            0x30, 0x1a, 0x30, 0x18,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x05, 0x00, 0x03, 0x09, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23,
            0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badAlgIdNull[] = {
            0x30, 0x1b, 0x30, 0x19,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x05, 0x01, 0x00, 0x03, 0x09, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23,
            0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNotBitString[] = {
            0x30, 0x18, 0x30, 0x16,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x04, 0x09, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badBitStringLen[] = {
            0x30, 0x18, 0x30, 0x16,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x03, 0x0a, 0x00, 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNoSeq[] = {
            0x30, 0x16, 0x30, 0x14,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x07, 0x00, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNoObj[] = {
            0x30, 0x0f, 0x30, 0x0d, 0x05, 0x00, 0x03, 0x09, 0x00, 0x30, 0x06,
            0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badIntN[] = {
            0x30, 0x06, 0x02, 0x05, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNotIntE[] = {
            0x30, 0x06, 0x02, 0x01, 0x23, 0x04, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badLength[] = {
            0x30, 0x04, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };
    WOLFSSL_SMALL_STACK_STATIC const byte badBitStrNoZero[] = {
            0x30, 0x17, 0x30, 0x15,
            0x06, 0x09, 0x2a, 0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01,
            0x03, 0x08, 0x30, 0x06, 0x02, 0x01, 0x23, 0x02, 0x1, 0x03 };

    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Parameter Validation testing. */
    ret = wc_RsaPublicKeyDecodeRaw(NULL, sizeof(n), e, sizeof(e), keyPub);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_RsaPublicKeyDecodeRaw(n, sizeof(n), NULL, sizeof(e), keyPub);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_RsaPublicKeyDecodeRaw(n, sizeof(n), e, sizeof(e), NULL);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_RsaPublicKeyDecodeRaw(n, (word32)-1, e, sizeof(e), keyPub);
#if defined(USE_INTEGER_HEAP_MATH)
    if (ret != 0)
#else
    if (ret != ASN_GETINT_E)
#endif
    {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_RsaPublicKeyDecodeRaw(n, sizeof(n), e, (word32)-1, keyPub);
#if defined(USE_INTEGER_HEAP_MATH)
    if (ret != 0)
#else
    if (ret != ASN_GETINT_E)
#endif
    {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Use API. */
    ret = wc_RsaPublicKeyDecodeRaw(n, sizeof(n), e, sizeof(e), keyPub);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Parameter Validation testing. */
    inSz = sizeof(good);
    ret = wc_RsaPublicKeyDecode(NULL, &inOutIdx, keyPub, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_RsaPublicKeyDecode(good, NULL, keyPub, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_RsaPublicKeyDecode(good, &inOutIdx, NULL, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    /* Use good data and offset to bad data. */
    inOutIdx = 2;
    inSz = sizeof(good) - inOutIdx;
    ret = wc_RsaPublicKeyDecode(good, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inOutIdx = 2;
    inSz = sizeof(goodAlgId) - inOutIdx;
    ret = wc_RsaPublicKeyDecode(goodAlgId, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inOutIdx = 2;
    inSz = sizeof(goodAlgId);
    ret = wc_RsaPublicKeyDecode(goodAlgId, &inOutIdx, keyPub, inSz);
#ifndef WOLFSSL_NO_DECODE_EXTRA
    if (ret != ASN_PARSE_E)
#else
    if (ret != ASN_RSA_KEY_E)
#endif
    {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    /* Try different bad data. */
    inSz = sizeof(badAlgIdNull);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badAlgIdNull, &inOutIdx, keyPub, inSz);
    if (ret != ASN_EXPECT_0_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badNotBitString);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badNotBitString, &inOutIdx, keyPub, inSz);
    if (ret != ASN_BITSTR_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badBitStringLen);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badBitStringLen, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badNoSeq);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badNoSeq, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badNoObj);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badNoObj, &inOutIdx, keyPub, inSz);
    if (ret != ASN_PARSE_E && ret != ASN_OBJECT_ID_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badIntN);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badIntN, &inOutIdx, keyPub, inSz);
    if (ret != ASN_RSA_KEY_E && ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badNotIntE);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badNotIntE, &inOutIdx, keyPub, inSz);
    if (ret != ASN_RSA_KEY_E && ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    /* TODO: Shouldn't pass as the sequence length is too small. */
    inSz = sizeof(badLength);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badLength, &inOutIdx, keyPub, inSz);
#ifndef WOLFSSL_ASN_TEMPLATE
    if (ret != 0)
#else
    if (ret != ASN_PARSE_E)
#endif
    {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    /* TODO: Shouldn't ignore object id's data. */
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    inSz = sizeof(badBitStrNoZero);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(badBitStrNoZero, &inOutIdx, keyPub, inSz);
    if (ret != ASN_EXPECT_0_E && ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    /* Valid data cases. */
    inSz = sizeof(good);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(good, &inOutIdx, keyPub, inSz);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    if (inOutIdx != inSz) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    inSz = sizeof(goodAlgId);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(goodAlgId, &inOutIdx, keyPub, inSz);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    if (inOutIdx != inSz) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }
    wc_FreeRsaKey(keyPub);
    ret = wc_InitRsaKey(keyPub, NULL);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    inSz = sizeof(goodAlgIdNull);
    inOutIdx = 0;
    ret = wc_RsaPublicKeyDecode(goodAlgIdNull, &inOutIdx, keyPub, inSz);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    if (inOutIdx != inSz) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }

done:
    wc_FreeRsaKey(keyPub);
    return ret;
}
#endif

#if defined(WC_RSA_PSS) && !defined(HAVE_FIPS_VERSION) /* not supported with FIPSv1 */
/* Need to create known good signatures to test with this. */
#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
!defined(WOLF_CRYPTO_CB_ONLY_RSA)
static wc_test_ret_t rsa_pss_test(WC_RNG* rng, RsaKey* key)
{
    byte             digest[WC_MAX_DIGEST_SIZE];
    wc_test_ret_t ret = 0;
    const char       inStr[] = TEST_STRING;
    word32           inLen   = (word32)TEST_STRING_SZ;
    word32           outSz;
    word32           plainSz;
    word32           digestSz;
    int              i, j;
#ifdef RSA_PSS_TEST_WRONG_PARAMS
    int              k, l;
#endif
#ifndef WOLFSSL_SE050
    int              len;
#endif
    byte*            plain;
    int              mgf[]   = {
#ifndef NO_SHA
                                 WC_MGF1SHA1,
#endif
#ifdef WOLFSSL_SHA224
                                 WC_MGF1SHA224,
#endif
                                 WC_MGF1SHA256,
#ifdef WOLFSSL_SHA384
                                 WC_MGF1SHA384,
#endif
#ifdef WOLFSSL_SHA512
                                 WC_MGF1SHA512
#endif
                               };
    enum wc_HashType hash[]  = {
#ifndef NO_SHA
                                 WC_HASH_TYPE_SHA,
#endif
#ifdef WOLFSSL_SHA224
                                 WC_HASH_TYPE_SHA224,
#endif
                                 WC_HASH_TYPE_SHA256,
#ifdef WOLFSSL_SHA384
                                 WC_HASH_TYPE_SHA384,
#endif
#ifdef WOLFSSL_SHA512
                                 WC_HASH_TYPE_SHA512,
#endif
                               };

    WC_DECLARE_VAR(in, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(sig, byte, RSA_TEST_BYTES, HEAP_HINT);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || out == NULL || sig == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa_pss);
#endif
    XMEMCPY(in, inStr, inLen);

    /* Test all combinations of hash and MGF. */
    for (j = 0; j < (int)(sizeof(hash)/sizeof(*hash)); j++) {
        /* Calculate hash of message. */
        ret = wc_Hash(hash[j], in, inLen, digest, sizeof(digest));
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
        digestSz = wc_HashGetDigestSize(hash[j]);

#ifdef WOLFSSL_SE050
        /* SE050 only supports MGF matched to same hash type */
        i = j;
#else
        for (i = 0; i < (int)(sizeof(mgf)/sizeof(*mgf)); i++) {
#endif
            outSz = RSA_TEST_BYTES;
            do {
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &key->asyncDev,
                    WC_ASYNC_FLAG_CALL_AGAIN);
            #endif
                if (ret >= 0) {
                    ret = wc_RsaPSS_Sign_ex(digest, digestSz, out, outSz,
                        hash[j], mgf[i], -1, key, rng);
                }
            } while (ret == WC_PENDING_E);
            if (ret <= 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
            outSz = (word32)ret;

            XMEMCPY(sig, out, outSz);
            plain = NULL;
            TEST_SLEEP();

            do {
            #if defined(WOLFSSL_ASYNC_CRYPT)
                ret = wc_AsyncWait(ret, &key->asyncDev,
                    WC_ASYNC_FLAG_CALL_AGAIN);
            #endif
                if (ret >= 0) {
                    ret = wc_RsaPSS_VerifyInline_ex(sig, outSz, &plain, hash[j],
                        mgf[i], -1, key);
                }
            } while (ret == WC_PENDING_E);
            if (ret <= 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
            plainSz = (word32)ret;
            TEST_SLEEP();

#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[j], -1);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[j], -1, 0);
#else
            ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, plain, plainSz,
                              hash[j], -1, wc_RsaEncryptSize(key)*8, HEAP_HINT);
#endif
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);

#ifdef RSA_PSS_TEST_WRONG_PARAMS
            for (k = 0; k < (int)(sizeof(mgf)/sizeof(*mgf)); k++) {
                for (l = 0; l < (int)(sizeof(hash)/sizeof(*hash)); l++) {
                    if (i == k && j == l)
                        continue;

                    XMEMCPY(sig, out, outSz);

                    do {
                    #if defined(WOLFSSL_ASYNC_CRYPT)
                        ret = wc_AsyncWait(ret, &key->asyncDev,
                            WC_ASYNC_FLAG_CALL_AGAIN);
                    #endif
                        if (ret >= 0) {
                            ret = wc_RsaPSS_VerifyInline_ex(sig, outSz,
                                (byte**)&plain, hash[l], mgf[k], -1, key);
                        }
                    } while (ret == WC_PENDING_E);
                    if (ret >= 0)
                        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
                }
            }
#endif
#ifndef WOLFSSL_SE050
        } /* end mgf for loop */
#endif
    }

/* SE050 generates salts internally only of hash length */
#ifndef WOLFSSL_SE050
    /* Test that a salt length of zero works. */
    digestSz = wc_HashGetDigestSize(hash[0]);
    outSz = RSA_TEST_BYTES;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_Sign_ex(digest, digestSz, out, outSz, hash[0],
                mgf[0], 0, key, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
    outSz = (word32)ret;
    TEST_SLEEP();

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_Verify_ex(out, outSz, sig, outSz, hash[0], mgf[0],
                0, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
    plainSz = (word32)ret;
    TEST_SLEEP();

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, sig, plainSz,
                                         hash[0], 0);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, sig, plainSz,
                                         hash[0], 0, 0);
#else
            ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, sig, plainSz,
                hash[0], 0, 0, HEAP_HINT);
#endif
        }
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);

    XMEMCPY(sig, out, outSz);
    plain = NULL;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_VerifyInline_ex(sig, outSz, &plain, hash[0], mgf[0],
                0, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
    plainSz = (word32)ret;
    TEST_SLEEP();

#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], 0);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], 0, 0);
#else
    ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, plain, plainSz, hash[0],
                                    0, 0, HEAP_HINT);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);

    /* Test bad salt lengths in various APIs. */
    digestSz = wc_HashGetDigestSize(hash[0]);
    outSz = RSA_TEST_BYTES;
#ifndef WOLFSSL_PSS_SALT_LEN_DISCOVER
    len = -2;
#else
    len = -3;
#endif
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_Sign_ex(digest, digestSz, out, outSz, hash[0],
                mgf[0], len, key, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_Sign_ex(digest, digestSz, out, outSz, hash[0],
                mgf[0], digestSz + 1, key, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
    TEST_SLEEP();

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_VerifyInline_ex(sig, outSz, &plain, hash[0],
                mgf[0], -2, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
    TEST_SLEEP();

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev,
            WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPSS_VerifyInline_ex(sig, outSz, &plain, hash[0], mgf[0],
                digestSz + 1, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
    TEST_SLEEP();

#ifndef WOLFSSL_PSS_SALT_LEN_DISCOVER
    len = -2;
#else
    len = -3;
#endif
#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], len);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], len, 0);
#else
    ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, plain, plainSz, hash[0],
                                    len, 0, HEAP_HINT);
#endif
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
#ifndef WOLFSSL_PSS_LONG_SALT
    len = digestSz + 1;
#else
    len = plainSz - digestSz - 1;
#endif
#if defined(HAVE_SELFTEST) && \
    (!defined(HAVE_SELFTEST_VERSION) || (HAVE_SELFTEST_VERSION < 2))
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], len);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
#elif defined(HAVE_SELFTEST) && (HAVE_SELFTEST_VERSION == 2)
            ret = wc_RsaPSS_CheckPadding_ex(digest, digestSz, plain, plainSz,
                                         hash[0], len, 0);
    if (ret != BAD_PADDING_E)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
#else
    ret = wc_RsaPSS_CheckPadding_ex2(digest, digestSz, plain, plainSz, hash[0],
                                    len, 0, HEAP_HINT);
    if (ret != PSS_SALTLEN_E)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_pss);
#endif

    ret = 0;
#endif /* WOLFSSL_SE050 */
exit_rsa_pss:
    WC_FREE_VAR(sig, HEAP_HINT);
    WC_FREE_VAR(in, HEAP_HINT);
    WC_FREE_VAR(out, HEAP_HINT);

    return ret;
}
#endif /* !WOLFSSL_RSA_VERIFY_ONLY && !WOLFSSL_RSA_PUBLIC_ONLY */
#endif

#ifdef WC_RSA_NO_PADDING
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t rsa_no_pad_test(void)
{
    WC_RNG rng;
    byte*  tmp = NULL;
    size_t bytes;
    wc_test_ret_t ret;
    word32 inLen   = 0;
    word32 idx     = 0;
    word32 outSz   = RSA_TEST_BYTES;
    word32 plainSz = RSA_TEST_BYTES;
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096) && \
    !defined(NO_FILESYSTEM)
    XFILE  file;
#endif
    WC_DECLARE_VAR(key, RsaKey, 1, HEAP_HINT);
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(plain, byte, RSA_TEST_BYTES, HEAP_HINT);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (key == NULL || out == NULL || plain == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa_nopadding);
#endif

    /* initialize stack structures */
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(key, 0, sizeof(RsaKey));
#ifdef USE_CERT_BUFFERS_1024
    bytes = (size_t)sizeof_client_key_der_1024;
    if (bytes < (size_t)sizeof_client_cert_der_1024)
        bytes = (size_t)sizeof_client_cert_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    bytes = (size_t)sizeof_client_key_der_2048;
    if (bytes < (size_t)sizeof_client_cert_der_2048)
        bytes = (size_t)sizeof_client_cert_der_2048;
#else
    bytes = FOURK_BUF;
#endif

    tmp = (byte*)XMALLOC(bytes, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL
    #ifdef WOLFSSL_ASYNC_CRYPT
        || out == NULL || plain == NULL
    #endif
    ) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa_nopadding);
    }

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, client_key_der_1024, (size_t)sizeof_client_key_der_1024);
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_key_der_2048, (size_t)sizeof_client_key_der_2048);
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_key_der_3072, (size_t)sizeof_client_key_der_3072);
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_key_der_4096, (size_t)sizeof_client_key_der_4096);
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(clientKey, "rb");
    if (!file) {
        err_sys("can't open clientKey, Please run from wolfSSL home dir",
                WC_TEST_RET_ENC_ERRNO);
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa_nopadding);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa_nopadding);
#else
    /* No key to use. */
    ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa_nopadding);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }
    ret = wc_RsaPrivateKeyDecode(tmp, &idx, key, (word32)bytes);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    /* after loading in key use tmp as the test buffer */

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

#ifndef WOLFSSL_RSA_VERIFY_ONLY
    inLen = wc_RsaEncryptSize(key);
    outSz   = inLen;
    plainSz = inLen;
    XMEMSET(tmp, 7, inLen);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaDirect(tmp, inLen, out, &outSz, key,
                    RSA_PRIVATE_ENCRYPT, &rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    /* encrypted result should not be the same as input */
    if (XMEMCMP(out, tmp, inLen) == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa_nopadding);
    }
    TEST_SLEEP();

    /* decrypt with public key and compare result */
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaDirect(out, outSz, plain, &plainSz, key,
                    RSA_PUBLIC_DECRYPT, &rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret <= 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    if (XMEMCMP(plain, tmp, inLen) != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa_nopadding);
    }
    TEST_SLEEP();
#endif

#ifdef WC_RSA_BLINDING
    ret = wc_RsaSetRNG(NULL, &rng);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    ret = wc_RsaSetRNG(key, &rng);
    if (ret < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }
#endif

    /* test encrypt and decrypt using WC_RSA_NO_PAD */
#ifndef WOLFSSL_RSA_VERIFY_ONLY
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(tmp, inLen, out, (int)outSz, key, &rng,
                WC_RSA_NO_PAD, WC_HASH_TYPE_NONE, WC_MGF1NONE, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_VERIFY_ONLY */

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, outSz, plain, (int)plainSz, key,
                WC_RSA_NO_PAD, WC_HASH_TYPE_NONE, WC_MGF1NONE, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    if (XMEMCMP(plain, tmp, inLen) != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa_nopadding);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

    /* test some bad arguments */
    ret = wc_RsaDirect(out, outSz, plain, &plainSz, key, -1,
            &rng);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    ret = wc_RsaDirect(out, outSz, plain, &plainSz, NULL, RSA_PUBLIC_DECRYPT,
            &rng);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    ret = wc_RsaDirect(out, outSz, NULL, &plainSz, key, RSA_PUBLIC_DECRYPT,
            &rng);
    if (ret != LENGTH_ONLY_E || plainSz != inLen) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    ret = wc_RsaDirect(out, outSz - 10, plain, &plainSz, key,
            RSA_PUBLIC_DECRYPT, &rng);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_nopadding);
    }

    /* if making it to this point of code without hitting an ERROR_OUT then
     * all tests have passed */
    ret = 0;

exit_rsa_nopadding:
    wc_FreeRsaKey(key);
    XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR(key, HEAP_HINT);
    WC_FREE_VAR(out, HEAP_HINT);
    WC_FREE_VAR(plain, HEAP_HINT);
    wc_FreeRng(&rng);

    return ret;
}
#endif /* WC_RSA_NO_PADDING */

#if defined(WOLFSSL_HAVE_SP_RSA) && defined(USE_FAST_MATH)
static wc_test_ret_t rsa_even_mod_test(WC_RNG* rng, RsaKey* key)
{
    byte*  tmp = NULL;
    size_t bytes;
    wc_test_ret_t ret;
    word32 inLen   = 0;
#ifndef NO_ASN
    word32 idx     = 0;
#endif
    word32 outSz   = RSA_TEST_BYTES;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    word32 plainSz = RSA_TEST_BYTES;
#endif
#if !defined(USE_CERT_BUFFERS_2048) && !defined(USE_CERT_BUFFERS_3072) && \
    !defined(USE_CERT_BUFFERS_4096) && !defined(NO_FILESYSTEM)
    XFILE  file;
#endif
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    WC_DECLARE_VAR(plain, byte, RSA_TEST_BYTES, HEAP_HINT);
#endif
#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (out == NULL
    #ifndef WOLFSSL_RSA_PUBLIC_ONLY
        || plain == NULL
    #endif
    ) {
        ERROR_OUT(MEMORY_E, exit_rsa_even_mod);
    }
#endif

#if defined(USE_CERT_BUFFERS_2048)
    bytes = (size_t)sizeof_client_key_der_2048;
    if (bytes < (size_t)sizeof_client_cert_der_2048)
        bytes = (size_t)sizeof_client_cert_der_2048;
#else
    bytes = FOURK_BUF;
#endif

    tmp = (byte*)XMALLOC(bytes, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL
    #ifdef WOLFSSL_ASYNC_CRYPT
        || out == NULL || plain == NULL
    #endif
    ) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa_even_mod);
    }

#if defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_key_der_2048, (size_t)sizeof_client_key_der_2048);
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_key_der_3072, (size_t)sizeof_client_key_der_3072);
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_key_der_4096, (size_t)sizeof_client_key_der_4096);
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(clientKey, "rb");
    if (!file) {
        err_sys("can't open ./certs/client-key.der, "
                "Please run from wolfSSL home dir", WC_TEST_RET_ENC_ERRNO);
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa_even_mod);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa_even_mod);
#else
    /* No key to use. */
    ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa_even_mod);
#endif /* USE_CERT_BUFFERS */

#ifndef NO_ASN
    ret = wc_RsaPrivateKeyDecode(tmp, &idx, key, (word32)bytes);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
    }
#else
    #ifdef USE_CERT_BUFFERS_2048
        ret = mp_read_unsigned_bin(&key->n, &tmp[12], 256);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
        }
        ret = mp_set_int(&key->e, WC_RSA_EXPONENT);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
        }
#ifndef NO_SIG_WRAPPER
        modLen = 2048;
#endif
    #else
        #error Not supported yet!
    #endif
#endif

    key->n.dp[0] &= (mp_digit)-2;
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
    }

    /* after loading in key use tmp as the test buffer */
#if !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION == 2) && \
    (defined(WOLFSSL_SP_ARM64_ASM) || defined(WOLFSSL_SP_ARM32_ASM))) && \
    !defined(WOLFSSL_XILINX_CRYPT)
    /* The ARM64_ASM code that was FIPS validated did not return these expected
     * failure codes. These tests cases were added after the assembly was
     * in-lined in the module and validated, these tests will be available in
     * the 140-3 module */
#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
    inLen = 32;
    outSz   = wc_RsaEncryptSize(key);
    XMEMSET(tmp, 7, plainSz);
    ret = wc_RsaSSL_Sign(tmp, inLen, out, outSz, key, rng);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E && ret != MP_INVMOD_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
    }

    ret = wc_RsaSSL_Verify(out, outSz, tmp, inLen, key);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
    }
#endif

#ifdef WC_RSA_BLINDING
    ret = wc_RsaSetRNG(key, rng);
    if (ret < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
    }
#endif

    /* test encrypt and decrypt using WC_RSA_NO_PAD */
#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)
    ret = wc_RsaPublicEncrypt(tmp, inLen, out, (int)outSz, key, rng);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
    }
#endif /* WOLFSSL_RSA_VERIFY_ONLY */

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    ret = wc_RsaPrivateDecrypt(out, outSz, plain, (int)plainSz, key);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E && ret != MP_INVMOD_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa_even_mod);
    }
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */
#endif /* !(HAVE_FIPS_VERSION == 2 && WOLFSSL_SP_ARMxx_ASM) */
    /* if making it to this point of code without hitting an ERROR_OUT then
     * all tests have passed */
    ret = 0;

exit_rsa_even_mod:
    XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    WC_FREE_VAR(out, HEAP_HINT);
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    WC_FREE_VAR(plain, HEAP_HINT);
#endif

    (void)out;
    (void)outSz;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    (void)plain;
    (void)plainSz;
#endif
    (void)inLen;
    (void)rng;

    return ret;
}
#endif /* WOLFSSL_HAVE_SP_RSA */

#if defined(WOLFSSL_CERT_GEN) && !defined(NO_ASN_TIME)
static wc_test_ret_t rsa_certgen_test(RsaKey* key, RsaKey* keypub, WC_RNG* rng, byte* tmp)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    RsaKey      *caKey = (RsaKey *)XMALLOC(sizeof *caKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef WOLFSSL_TEST_CERT
    DecodedCert *decode = (DecodedCert *)XMALLOC(sizeof *decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#else
    RsaKey      caKey[1];
#ifdef WOLFSSL_TEST_CERT
    DecodedCert decode[1];
#endif
#endif
    byte*       der = NULL;
    wc_test_ret_t ret;
    Cert*       myCert = NULL;
    int         certSz;
    size_t      bytes3;
    word32      idx3 = 0;
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    XFILE       file3;
#endif
#if defined(WOLFSSL_ALT_NAMES) && !defined(NO_ASN_TIME)
    struct tm beforeTime;
    struct tm afterTime;
#endif
    const byte  mySerial[8] = {1,2,3,4,5,6,7,8};

    (void)keypub;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (caKey == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#ifdef WOLFSSL_TEST_CERT
    if (decode == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

#endif

    XMEMSET(caKey, 0, sizeof *caKey);

    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }
    myCert = (Cert*)XMALLOC(sizeof(Cert), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (myCert == NULL) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    /* self signed */
    ret = wc_InitCert_ex(myCert, HEAP_HINT, devId);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }

    XMEMCPY(&myCert->subject, &certDefaultName, sizeof(CertName));
    XMEMCPY(myCert->serial, mySerial, sizeof(mySerial));
    myCert->serialSz = (int)sizeof(mySerial);
    myCert->isCA    = 1;
#ifndef NO_SHA256
    myCert->sigType = CTC_SHA256wRSA;
#else
    myCert->sigType = CTC_SHAwRSA;
#endif

#ifdef WOLFSSL_CERT_EXT
    /* add Policies */
    XSTRNCPY(myCert->certPolicies[0], "2.16.840.1.101.3.4.1.42",
            CTC_MAX_CERTPOL_SZ);
    XSTRNCPY(myCert->certPolicies[1], "1.2.840.113549.1.9.16.6.5",
            CTC_MAX_CERTPOL_SZ);
    myCert->certPoliciesNb = 2;

    /* add SKID from the Public Key */
    ret = wc_SetSubjectKeyIdFromPublicKey(myCert, keypub, NULL);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }

     /* add AKID from the Public Key */
    ret = wc_SetAuthKeyIdFromPublicKey(myCert, keypub, NULL);
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }

    /* add Key Usage */
    ret = wc_SetKeyUsage(myCert,"cRLSign,keyCertSign");
    if (ret != 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }
#ifdef WOLFSSL_EKU_OID
    {
        const char unique[] = "2.16.840.1.111111.100.1.10.1";
        ret = wc_SetExtKeyUsageOID(myCert, unique, sizeof(unique), 0,
                    HEAP_HINT);
        if (ret != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
        }
    }
#endif /* WOLFSSL_EKU_OID */
#endif /* WOLFSSL_CERT_EXT */

    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_MakeSelfCert(myCert, der, FOURK_BUF, key, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }
    certSz = (word32)ret;

#ifdef WOLFSSL_TEST_CERT
    InitDecodedCert(decode, der, certSz, HEAP_HINT);
    ret = ParseCert(decode, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0) {
        FreeDecodedCert(decode);
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }
    FreeDecodedCert(decode);
#endif

    ret = SaveDerAndPem(der, certSz, certDerFile, certPemFile,
                        CERT_TYPE);
    if (ret != 0) {
        goto exit_rsa;
    }

    /* Setup Certificate */
    ret = wc_InitCert_ex(myCert, HEAP_HINT, devId);
    if (ret < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }

#ifdef WOLFSSL_ALT_NAMES
        /* Get CA Cert for testing */
    #ifdef USE_CERT_BUFFERS_1024
        XMEMCPY(tmp, ca_cert_der_1024, sizeof_ca_cert_der_1024);
        bytes3 = sizeof_ca_cert_der_1024;
    #elif defined(USE_CERT_BUFFERS_2048)
        XMEMCPY(tmp, ca_cert_der_2048, sizeof_ca_cert_der_2048);
        bytes3 = sizeof_ca_cert_der_2048;
    #else
        file3 = XFOPEN(rsaCaCertDerFile, "rb");
        if (!file3) {
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
        }
        bytes3 = XFREAD(tmp, 1, FOURK_BUF, file3);
        XFCLOSE(file3);
        if (bytes3 == 0)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    #endif /* USE_CERT_BUFFERS */

    #if !defined(NO_FILESYSTEM) && !defined(USE_CERT_BUFFERS_1024) && \
        !defined(USE_CERT_BUFFERS_2048) && !defined(NO_ASN)
        ret = wc_SetAltNames(myCert, rsaCaCertFile);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    #endif
        /* get alt names from der */
        ret = wc_SetAltNamesBuffer(myCert, tmp, (int)bytes3);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

        /* get dates from der */
        ret = wc_SetDatesBuffer(myCert, tmp, (int)bytes3);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    #ifndef NO_ASN_TIME
        ret = wc_GetCertDates(myCert, &beforeTime, &afterTime);
        if (ret < 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    #endif
#endif /* WOLFSSL_ALT_NAMES */

    /* Get CA Key */
#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, ca_key_der_1024, sizeof_ca_key_der_1024);
    bytes3 = sizeof_ca_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, ca_key_der_2048, sizeof_ca_key_der_2048);
    bytes3 = sizeof_ca_key_der_2048;
#else
    file3 = XFOPEN(rsaCaKeyFile, "rb");
    if (!file3) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    bytes3 = XFREAD(tmp, 1, FOURK_BUF, file3);
    XFCLOSE(file3);
    if (bytes3 == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey(caKey, HEAP_HINT);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    ret = wc_RsaPrivateKeyDecode(tmp, &idx3, caKey, (word32)bytes3);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

#ifndef NO_SHA256
    myCert->sigType = CTC_SHA256wRSA;
#else
    myCert->sigType = CTC_SHAwRSA;
#endif

    XMEMCPY(&myCert->subject, &certDefaultName, sizeof(CertName));

#ifdef WOLFSSL_CERT_EXT
    /* add Policies */
    XSTRNCPY(myCert->certPolicies[0], "2.16.840.1.101.3.4.1.42",
            CTC_MAX_CERTPOL_SZ);
    myCert->certPoliciesNb =1;

    /* add SKID from the Public Key */
    ret = wc_SetSubjectKeyIdFromPublicKey(myCert, key, NULL);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    /* add AKID from the CA certificate */
#if defined(USE_CERT_BUFFERS_2048)
    ret = wc_SetAuthKeyIdFromCert(myCert, ca_cert_der_2048,
                                        sizeof_ca_cert_der_2048);
#elif defined(USE_CERT_BUFFERS_1024)
    ret = wc_SetAuthKeyIdFromCert(myCert, ca_cert_der_1024,
                                        sizeof_ca_cert_der_1024);
#else
    ret = wc_SetAuthKeyId(myCert, rsaCaCertFile);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    /* add Key Usage */
    ret = wc_SetKeyUsage(myCert,"keyEncipherment,keyAgreement");
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#endif /* WOLFSSL_CERT_EXT */

#if defined(USE_CERT_BUFFERS_2048)
    ret = wc_SetIssuerBuffer(myCert, ca_cert_der_2048,
                                      sizeof_ca_cert_der_2048);
#elif defined(USE_CERT_BUFFERS_1024)
    ret = wc_SetIssuerBuffer(myCert, ca_cert_der_1024,
                                      sizeof_ca_cert_der_1024);
#else
    ret = wc_SetIssuer(myCert, rsaCaCertFile);
#endif
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    certSz = wc_MakeCert(myCert, der, FOURK_BUF, key, NULL, rng);
    if (certSz < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(certSz), exit_rsa);
    }

    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &caKey->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_SignCert(myCert->bodySz, myCert->sigType, der, FOURK_BUF,
                      caKey, NULL, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    certSz = (word32)ret;

#ifdef WOLFSSL_TEST_CERT
    InitDecodedCert(decode, der, certSz, HEAP_HINT);
    ret = ParseCert(decode, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0) {
        FreeDecodedCert(decode);
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }
    FreeDecodedCert(decode);
#endif

    ret = SaveDerAndPem(der, certSz, otherCertDerFile, otherCertPemFile,
                        CERT_TYPE);
    if (ret != 0) {
        goto exit_rsa;
    }

exit_rsa:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (caKey != NULL) {
        wc_FreeRsaKey(caKey);
        XFREE(caKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#ifdef WOLFSSL_TEST_CERT
    if (decode != NULL)
        XFREE(decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#else
    wc_FreeRsaKey(caKey);
#endif

    XFREE(myCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    return ret;
}
#endif

#if !defined(NO_RSA) && defined(HAVE_ECC) && !defined(NO_ECC_SECP) && \
    defined(WOLFSSL_CERT_GEN)
/* Make Cert / Sign example for ECC cert and RSA CA */
static wc_test_ret_t rsa_ecc_certgen_test(WC_RNG* rng, byte* tmp)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    RsaKey      *caKey = (RsaKey *)XMALLOC(sizeof *caKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key     *caEccKey = (ecc_key *)XMALLOC(sizeof *caEccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key     *caEccKeyPub = (ecc_key *)XMALLOC(sizeof *caEccKeyPub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef WOLFSSL_TEST_CERT
    DecodedCert *decode = (DecodedCert *)XMALLOC(sizeof *decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#else
    RsaKey      caKey[1];
    ecc_key     caEccKey[1];
    ecc_key     caEccKeyPub[1];
#ifdef WOLFSSL_TEST_CERT
    DecodedCert decode[1];
#endif
#endif
    byte*       der = NULL;
    Cert*       myCert = NULL;
    int         certSz;
    size_t      bytes3;
    word32      idx3 = 0;
#if (!defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)) \
    || !defined(USE_CERT_BUFFERS_256)
    XFILE       file3;
#endif
    wc_test_ret_t ret;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((caKey == NULL) || (caEccKey == NULL) || (caEccKeyPub == NULL)
#ifdef WOLFSSL_TEST_CERT
        || (decode == NULL)
#endif
        )
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

    XMEMSET(caKey, 0, sizeof *caKey);
    XMEMSET(caEccKey, 0, sizeof *caEccKey);
    XMEMSET(caEccKeyPub, 0, sizeof *caEccKeyPub);

    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }
    myCert = (Cert*)XMALLOC(sizeof(Cert), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (myCert == NULL) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    /* Get CA Key */
#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, ca_key_der_1024, sizeof_ca_key_der_1024);
    bytes3 = sizeof_ca_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, ca_key_der_2048, sizeof_ca_key_der_2048);
    bytes3 = sizeof_ca_key_der_2048;
#else
    file3 = XFOPEN(rsaCaKeyFile, "rb");
    if (!file3) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    bytes3 = XFREAD(tmp, 1, FOURK_BUF, file3);
    XFCLOSE(file3);
    if (bytes3 == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey(caKey, HEAP_HINT);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    ret = wc_RsaPrivateKeyDecode(tmp, &idx3, caKey, (word32)bytes3);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    /* Get Cert Key */
#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(tmp, ecc_key_pub_der_256, sizeof_ecc_key_pub_der_256);
    bytes3 = sizeof_ecc_key_pub_der_256;
#else
    file3 = XFOPEN(eccKeyPubFileDer, "rb");
    if (!file3) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    bytes3 = XFREAD(tmp, 1, FOURK_BUF, file3);
    XFCLOSE(file3);
    if (bytes3 == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
#endif

    ret = wc_ecc_init_ex(caEccKeyPub, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    idx3 = 0;
    ret = wc_EccPublicKeyDecode(tmp, &idx3, caEccKeyPub, (word32)bytes3);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    /* Setup Certificate */
    ret = wc_InitCert_ex(myCert, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

#ifndef NO_SHA256
    myCert->sigType = CTC_SHA256wRSA;
#else
    myCert->sigType = CTC_SHAwRSA;
#endif

    XMEMCPY(&myCert->subject, &certDefaultName, sizeof(CertName));

#ifdef WOLFSSL_CERT_EXT
    /* add Policies */
    XSTRNCPY(myCert->certPolicies[0], "2.4.589440.587.101.2.1.9632587.1",
            CTC_MAX_CERTPOL_SZ);
    XSTRNCPY(myCert->certPolicies[1], "1.2.13025.489.1.113549",
            CTC_MAX_CERTPOL_SZ);
    myCert->certPoliciesNb = 2;

    /* add SKID from the Public Key */
    ret = wc_SetSubjectKeyIdFromPublicKey(myCert, NULL, caEccKeyPub);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    /* add AKID from the CA certificate */
#if defined(USE_CERT_BUFFERS_2048)
    ret = wc_SetAuthKeyIdFromCert(myCert, ca_cert_der_2048,
                                           sizeof_ca_cert_der_2048);
#elif defined(USE_CERT_BUFFERS_1024)
    ret = wc_SetAuthKeyIdFromCert(myCert, ca_cert_der_1024,
                                           sizeof_ca_cert_der_1024);
#else
    ret = wc_SetAuthKeyId(myCert, rsaCaCertFile);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    /* add Key Usage */
    ret = wc_SetKeyUsage(myCert, certKeyUsage);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#endif /* WOLFSSL_CERT_EXT */

#if defined(USE_CERT_BUFFERS_2048)
    ret = wc_SetIssuerBuffer(myCert, ca_cert_der_2048,
                                      sizeof_ca_cert_der_2048);
#elif defined(USE_CERT_BUFFERS_1024)
    ret = wc_SetIssuerBuffer(myCert, ca_cert_der_1024,
                                      sizeof_ca_cert_der_1024);
#else
    ret = wc_SetIssuer(myCert, rsaCaCertFile);
#endif
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    certSz = wc_MakeCert(myCert, der, FOURK_BUF, NULL, caEccKeyPub, rng);
    if (certSz < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(certSz), exit_rsa);
    }

    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &caEccKey->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret >= 0) {
            ret = wc_SignCert(myCert->bodySz, myCert->sigType, der,
                              FOURK_BUF, caKey, NULL, rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    certSz = (word32)ret;

#ifdef WOLFSSL_TEST_CERT
    InitDecodedCert(decode, der, certSz, 0);
    ret = ParseCert(decode, CERT_TYPE, NO_VERIFY, 0);
    if (ret != 0) {
        FreeDecodedCert(decode);
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }
    FreeDecodedCert(decode);
#endif

    ret = SaveDerAndPem(der, certSz, certEccRsaDerFile, certEccRsaPemFile,
                        CERT_TYPE);
    if (ret != 0) {
        goto exit_rsa;
    }

exit_rsa:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (caKey != NULL) {
        wc_FreeRsaKey(caKey);
        XFREE(caKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (caEccKey != NULL) {
        wc_ecc_free(caEccKey);
        XFREE(caEccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (caEccKeyPub != NULL) {
        wc_ecc_free(caEccKeyPub);
        XFREE(caEccKeyPub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#ifdef WOLFSSL_TEST_CERT
    if (decode != NULL)
        XFREE(decode, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#else
    wc_FreeRsaKey(caKey);
    wc_ecc_free(caEccKey);
    wc_ecc_free(caEccKeyPub);
#endif

    XFREE(myCert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    myCert = NULL;
    XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    der = NULL;

    if (ret >= 0)
        ret = 0;
    return ret;
}
#endif /* !NO_RSA && HAVE_ECC && WOLFSSL_CERT_GEN */

#ifdef WOLFSSL_KEY_GEN
static wc_test_ret_t rsa_keygen_test(WC_RNG* rng)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    RsaKey *genKey = (RsaKey *)XMALLOC(sizeof *genKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    RsaKey genKey[1];
#endif
    wc_test_ret_t ret;
    byte*  der = NULL;
#ifndef WOLFSSL_CRYPTOCELL
    word32 idx = 0;
#endif
    int    derSz = 0;
#if !defined(WOLFSSL_SP_MATH) && !defined(HAVE_FIPS)
    int    keySz = 1024;
#else
    int    keySz = 2048;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (! genKey)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

    XMEMSET(genKey, 0, sizeof *genKey);

    ret = wc_InitRsaKey_ex(genKey, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

#ifdef HAVE_FIPS
    for (;;) {
#endif
        ret = wc_MakeRsaKey(genKey, keySz, WC_RSA_EXPONENT, rng);
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &genKey->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
#ifdef HAVE_FIPS
        if (ret == PRIME_GEN_E)
            continue;
        break;
    }
#endif

    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    TEST_SLEEP();

#ifdef WOLFSSL_RSA_KEY_CHECK
    ret = wc_CheckRsaKey(genKey);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#endif
    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    derSz = wc_RsaKeyToDer(genKey, der, FOURK_BUF);
    if (derSz < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(derSz), exit_rsa);
    }

    ret = SaveDerAndPem(der, derSz, keyDerFile, keyPemFile,
                        PRIVATEKEY_TYPE);
    if (ret != 0) {
        goto exit_rsa;
    }

    wc_FreeRsaKey(genKey);
    ret = wc_InitRsaKey(genKey, HEAP_HINT);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

#ifndef WOLFSSL_CRYPTOCELL
    idx = 0;
    /* The private key part of the key gen pairs from cryptocell can't be exported */
    ret = wc_RsaPrivateKeyDecode(der, &idx, genKey, derSz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#endif /* WOLFSSL_CRYPTOCELL */

exit_rsa:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (genKey) {
        wc_FreeRsaKey(genKey);
        XFREE(genKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_FreeRsaKey(genKey);
#endif

    if (der != NULL) {
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        der = NULL;
    }

    return ret;
}
#endif

#ifndef WOLFSSL_RSA_VERIFY_ONLY
#if !defined(WC_NO_RSA_OAEP) && !defined(WC_NO_RNG) && \
    !defined(HAVE_FAST_RSA) && !defined(HAVE_USER_RSA) && \
     (!defined(HAVE_FIPS) || \
      (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))) \
      && !defined(WOLF_CRYPTO_CB_ONLY_RSA)
static wc_test_ret_t rsa_oaep_padding_test(RsaKey* key, WC_RNG* rng)
{
    wc_test_ret_t ret = 0;
    word32 idx = 0;
    const char inStr[] = TEST_STRING;
    const word32 inLen = (word32)TEST_STRING_SZ;
    const word32 outSz   = RSA_TEST_BYTES;
    const word32 plainSz = RSA_TEST_BYTES;
    byte*  res = NULL;

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, HEAP_HINT);
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(plain, byte, RSA_TEST_BYTES, HEAP_HINT);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || out == NULL || plain == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

    XMEMCPY(in, inStr, inLen);

#ifndef NO_SHA
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif

        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                       WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA, WC_MGF1SHA1, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    TEST_SLEEP();

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    idx = (word32)ret;
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                       WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA, WC_MGF1SHA1, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    TEST_SLEEP();
    #endif /* NO_SHA */
#endif

    #ifndef NO_SHA256
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    TEST_SLEEP();

    idx = (word32)ret;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecryptInline_ex(out, idx, &res, key,
                WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    if (ret != (int)inLen) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    if (XMEMCMP(res, in, inLen)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

    /* check fails if not using the same optional label */
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    TEST_SLEEP();

/* TODO: investigate why Cavium Nitrox doesn't detect decrypt error here */
#if !defined(HAVE_CAVIUM) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
    !defined(WOLFSSL_CRYPTOCELL) && !defined(WOLFSSL_SE050)
    /* label is unused in cryptocell and SE050 so it won't detect decrypt error
     * due to label */
    idx = (word32)ret;
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
               WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, in, inLen);
        }
    } while (ret == WC_PENDING_E);
    if (ret > 0) { /* in this case decrypt should fail */
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    ret = 0;
    TEST_SLEEP();
#endif /* !HAVE_CAVIUM */

    /* check using optional label with encrypt/decrypt */
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
               WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, in, inLen);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    TEST_SLEEP();

    idx = (word32)ret;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
               WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256, in, inLen);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

    #ifndef NO_SHA
        /* check fail using mismatch hash algorithms */
        XMEMSET(plain, 0, plainSz);
        do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
            if (ret >= 0) {
                ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                    WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA, WC_MGF1SHA1, in, inLen);
            }
        } while (ret == WC_PENDING_E);
        if (ret < 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
        TEST_SLEEP();

/* TODO: investigate why Cavium Nitrox doesn't detect decrypt error here */
#if !defined(HAVE_CAVIUM) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
    !defined(WOLFSSL_CRYPTOCELL) && !defined(WOLFSSL_SE050)
        idx = (word32)ret;
        do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
            if (ret >= 0) {
                ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                    WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA256, WC_MGF1SHA256,
                    in, inLen);
            }
        } while (ret == WC_PENDING_E);
        if (ret > 0) { /* should fail */
            ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
        }
        ret = 0;
        TEST_SLEEP();
    #endif /* !HAVE_CAVIUM */
    #endif /* NO_SHA */
#endif /* NO_SHA256 */

#ifdef WOLFSSL_SHA512
    /* Check valid RSA key size is used while using hash length of SHA512
       If key size is less than (hash length * 2) + 2 then is invalid use
       and test, since OAEP padding requires this.
       BAD_FUNC_ARG is returned when this case is not met */
    if (wc_RsaEncryptSize(key) > ((int)WC_SHA512_DIGEST_SIZE * 2) + 2) {
        XMEMSET(plain, 0, plainSz);
        do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
            if (ret >= 0) {
                ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA512, WC_MGF1SHA512, NULL, 0);
            }
        } while (ret == WC_PENDING_E);
        if (ret < 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
        TEST_SLEEP();

        idx = (word32)ret;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
        do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
            if (ret >= 0) {
                ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                  WC_RSA_OAEP_PAD, WC_HASH_TYPE_SHA512, WC_MGF1SHA512, NULL, 0);
            }
        } while (ret == WC_PENDING_E);
        if (ret < 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

        if (XMEMCMP(plain, in, inLen)) {
            ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
        }
        TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */
    }
#endif /* WOLFSSL_SHA512 */

    /* check using pkcsv15 padding with _ex API */
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt_ex(in, inLen, out, outSz, key, rng,
                  WC_RSA_PKCSV15_PAD, WC_HASH_TYPE_NONE, 0, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    TEST_SLEEP();

    idx = (word32)ret;
#ifndef WOLFSSL_RSA_PUBLIC_ONLY
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt_ex(out, idx, plain, plainSz, key,
                  WC_RSA_PKCSV15_PAD, WC_HASH_TYPE_NONE, 0, NULL, 0);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    TEST_SLEEP();
#endif /* WOLFSSL_RSA_PUBLIC_ONLY */

exit_rsa:
    WC_FREE_VAR(in, HEAP_HINT);
    WC_FREE_VAR(out, HEAP_HINT);
    WC_FREE_VAR(plain, HEAP_HINT);

    (void)idx;
    (void)inStr;
    (void)res;

    if (ret >= 0)
        ret = 0;

    return ret;

}
#endif
#endif

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t rsa_test(void)
{
    wc_test_ret_t ret;
    size_t bytes;
    WC_RNG rng;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte*  tmp = NULL;
    byte*  der = NULL;
    RsaKey *key = (RsaKey *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    RsaKey key[1];
    byte tmp[FOURK_BUF];
#endif
#if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    RsaKey *keypub = (RsaKey *)XMALLOC(sizeof *keypub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    RsaKey keypub[1];
#endif
#endif
    word32 idx = 0;
    const char inStr[] = TEST_STRING;
    const word32 inLen = (word32)TEST_STRING_SZ;
    const word32 outSz   = RSA_TEST_BYTES;
    const word32 plainSz = RSA_TEST_BYTES;
    byte*  res = NULL;
#ifndef NO_SIG_WRAPPER
    int modLen;
#endif
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096) && \
    !defined(NO_FILESYSTEM)
    XFILE   file;
#ifdef WOLFSSL_TEST_CERT
    XFILE   file2;
#endif
#endif
#ifdef WOLFSSL_TEST_CERT
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    DecodedCert *cert = (DecodedCert *)XMALLOC(sizeof *cert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    DecodedCert cert[1];
#endif
#ifndef NO_ASN_TIME
    struct tm   timearg;
    const byte* date;
    byte        dateFormat;
    int         dateLength;
#endif
#endif

    WC_DECLARE_VAR(in, byte, TEST_STRING_SZ, HEAP_HINT);
    WC_DECLARE_VAR(out, byte, RSA_TEST_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(plain, byte, RSA_TEST_BYTES, HEAP_HINT);

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (in == NULL || out == NULL || plain == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif

    XMEMCPY(in, inStr, inLen);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
    if (keypub == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif
#ifdef WOLFSSL_TEST_CERT
    if (cert == NULL)
        ERROR_OUT(MEMORY_E, exit_rsa);
#endif
#endif /* WOLFSSL_SMALL_STACK && !WOLFSSL_NO_MALLOC */

    /* initialize stack structures */
    XMEMSET(&rng, 0, sizeof(rng));
    XMEMSET(key, 0, sizeof *key);
#if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
    XMEMSET(keypub, 0, sizeof *keypub);
#endif

#if !defined(HAVE_USER_RSA) && !defined(NO_ASN)
    ret = rsa_decode_test(key);
    if (ret != 0)
        ERROR_OUT(ret, exit_rsa);
#endif

#ifdef USE_CERT_BUFFERS_1024
    bytes = (size_t)sizeof_client_key_der_1024;
    if (bytes < (size_t)sizeof_client_cert_der_1024)
        bytes = (size_t)sizeof_client_cert_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    bytes = (size_t)sizeof_client_key_der_2048;
    if (bytes < (size_t)sizeof_client_cert_der_2048)
        bytes = (size_t)sizeof_client_cert_der_2048;
#elif defined(USE_CERT_BUFFERS_3072)
    bytes = (size_t)sizeof_client_key_der_3072;
    if (bytes < (size_t)sizeof_client_cert_der_3072)
        bytes = (size_t)sizeof_client_cert_der_3072;
#elif defined(USE_CERT_BUFFERS_4096)
    bytes = (size_t)sizeof_client_key_der_4096;
    if (bytes < (size_t)sizeof_client_cert_der_4096)
        bytes = (size_t)sizeof_client_cert_der_4096;
#else
    bytes = FOURK_BUF;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    tmp = (byte*)XMALLOC(bytes, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
#endif

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, client_key_der_1024, (size_t)sizeof_client_key_der_1024);
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_key_der_2048, (size_t)sizeof_client_key_der_2048);
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_key_der_3072, (size_t)sizeof_client_key_der_3072);
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_key_der_4096, (size_t)sizeof_client_key_der_4096);
#elif !defined(NO_FILESYSTEM)
    file = XFOPEN(clientKey, "rb");
    if (!file) {
        err_sys("can't open ./certs/client-key.der, "
                "Please run from wolfSSL home dir", WC_TEST_RET_ENC_ERRNO);
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
#else
    /* No key to use. */
    ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#ifndef NO_ASN
    ret = wc_RsaPrivateKeyDecode(tmp, &idx, key, (word32)bytes);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#ifndef NO_SIG_WRAPPER
    modLen = wc_RsaEncryptSize(key);
#endif
#else
    #ifdef USE_CERT_BUFFERS_2048
        ret = mp_read_unsigned_bin(&key->n, &tmp[12], 256);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
        ret = mp_set_int(&key->e, WC_RSA_EXPONENT);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#ifndef NO_SIG_WRAPPER
        modLen = 2048;
#endif
    #else
        #error Not supported yet!
    #endif
#endif

#ifndef WC_NO_RNG
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#endif

#ifndef NO_SIG_WRAPPER
    ret = rsa_sig_test(key, sizeof *key, modLen, &rng);
    if (ret != 0)
        goto exit_rsa;
#endif

#ifdef WC_RSA_NONBLOCK
    ret = rsa_nb_test(key, in, inLen, out, outSz, plain, plainSz, &rng);
    if (ret != 0)
        goto exit_rsa;
#endif

#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
    !defined(WC_NO_RNG) && !defined(WOLF_CRYPTO_CB_ONLY_RSA)
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPublicEncrypt(in, inLen, out, outSz, key, &rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    TEST_SLEEP();

#ifdef WC_RSA_BLINDING
    {
        wc_test_ret_t tmpret = ret;
        ret = wc_RsaSetRNG(key, &rng);
        if (ret < 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
        ret = tmpret;
    }
#endif

    idx = (word32)ret; /* save off encrypted length */
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecrypt(out, idx, plain, plainSz, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    if (XMEMCMP(plain, in, inLen)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    TEST_SLEEP();

    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaPrivateDecryptInline(out, idx, &res, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    if (ret != (int)inLen) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    if (XMEMCMP(res, in, inLen)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    TEST_SLEEP();

    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
            ret = wc_RsaSSL_Sign(in, inLen, out, outSz, key, &rng);
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    TEST_SLEEP();

#elif defined(WOLFSSL_PUBLIC_MP)
    {
        static byte signature_2048[] = {
            0x07, 0x6f, 0xc9, 0x85, 0x73, 0x9e, 0x21, 0x79,
            0x47, 0xf1, 0xa3, 0xd7, 0xf4, 0x27, 0x29, 0xbe,
            0x99, 0x5d, 0xac, 0xb2, 0x10, 0x3f, 0x95, 0xda,
            0x89, 0x23, 0xb8, 0x96, 0x13, 0x57, 0x72, 0x30,
            0xa1, 0xfe, 0x5a, 0x68, 0x9c, 0x99, 0x9d, 0x1e,
            0x05, 0xa4, 0x80, 0xb0, 0xbb, 0xd9, 0xd9, 0xa1,
            0x69, 0x97, 0x74, 0xb3, 0x41, 0x21, 0x3b, 0x47,
            0xf5, 0x51, 0xb1, 0xfb, 0xc7, 0xaa, 0xcc, 0xdc,
            0xcd, 0x76, 0xa0, 0x28, 0x4d, 0x27, 0x14, 0xa4,
            0xb9, 0x41, 0x68, 0x7c, 0xb3, 0x66, 0xe6, 0x6f,
            0x40, 0x76, 0xe4, 0x12, 0xfd, 0xae, 0x29, 0xb5,
            0x63, 0x60, 0x87, 0xce, 0x49, 0x6b, 0xf3, 0x05,
            0x9a, 0x14, 0xb5, 0xcc, 0xcd, 0xf7, 0x30, 0x95,
            0xd2, 0x72, 0x52, 0x1d, 0x5b, 0x7e, 0xef, 0x4a,
            0x02, 0x96, 0x21, 0x6c, 0x55, 0xa5, 0x15, 0xb1,
            0x57, 0x63, 0x2c, 0xa3, 0x8e, 0x9d, 0x3d, 0x45,
            0xcc, 0xb8, 0xe6, 0xa1, 0xc8, 0x59, 0xcd, 0xf5,
            0xdc, 0x0a, 0x51, 0xb6, 0x9d, 0xfb, 0xf4, 0x6b,
            0xfd, 0x32, 0x71, 0x6e, 0xcf, 0xcb, 0xb3, 0xd9,
            0xe0, 0x4a, 0x77, 0x34, 0xd6, 0x61, 0xf5, 0x7c,
            0xf9, 0xa9, 0xa4, 0xb0, 0x8e, 0x3b, 0xd6, 0x04,
            0xe0, 0xde, 0x2b, 0x5b, 0x5a, 0xbf, 0xd9, 0xef,
            0x8d, 0xa3, 0xf5, 0xb1, 0x67, 0xf3, 0xb9, 0x72,
            0x0a, 0x37, 0x12, 0x35, 0x6c, 0x8e, 0x10, 0x8b,
            0x38, 0x06, 0x16, 0x4b, 0x20, 0x20, 0x13, 0x00,
            0x2e, 0x6d, 0xc2, 0x59, 0x23, 0x67, 0x4a, 0x6d,
            0xa1, 0x46, 0x8b, 0xee, 0xcf, 0x44, 0xb4, 0x3e,
            0x56, 0x75, 0x00, 0x68, 0xb5, 0x7d, 0x0f, 0x20,
            0x79, 0x5d, 0x7f, 0x12, 0x15, 0x32, 0x89, 0x61,
            0x6b, 0x29, 0xb7, 0x52, 0xf5, 0x25, 0xd8, 0x98,
            0xe8, 0x6f, 0xf9, 0x22, 0xb4, 0xbb, 0xe5, 0xff,
            0xd0, 0x92, 0x86, 0x9a, 0x88, 0xa2, 0xaf, 0x6b
        };
        ret = sizeof(signature_2048);
        XMEMCPY(out, signature_2048, ret);
    }
#endif

#if !defined(WC_NO_RNG) && !defined(WC_NO_RSA_OAEP) && \
    ((!defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY)) || \
    defined(WOLFSSL_PUBLIC_MP))  && !defined(WOLF_CRYPTO_CB_ONLY_RSA)
    idx = (word32)ret;
    XMEMSET(plain, 0, plainSz);
    do {
#if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
#endif
        if (ret >= 0) {
#ifndef WOLFSSL_RSA_VERIFY_INLINE

#if defined(WOLFSSL_CRYPTOCELL)
        /*
        Cryptocell requires the input data and signature byte array to verify.

        first argument must be the input data
        second argument must be the length of input data
        third argument must be the signature byte array or the output from
                                                                wc_RsaSSL_Sign()
        fourth argument must be the length of the signature byte array
        */

        ret = wc_RsaSSL_Verify(in, inLen, out, outSz, key);
#else
        ret = wc_RsaSSL_Verify(out, idx, plain, plainSz, key);
#endif /* WOLFSSL_CRYPTOCELL */
#else
            byte* dec = NULL;
            ret = wc_RsaSSL_VerifyInline(out, idx, &dec, key);
            if (ret > 0) {
                XMEMCPY(plain, dec, ret);
            }
#endif
        }
    } while (ret == WC_PENDING_E);
    if (ret < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

    if (XMEMCMP(plain, in, (size_t)ret)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
    }
    TEST_SLEEP();
#endif

#ifndef WOLFSSL_RSA_VERIFY_ONLY
    #if !defined(WC_NO_RSA_OAEP) && !defined(WC_NO_RNG)
    #if !defined(HAVE_FAST_RSA) && !defined(HAVE_USER_RSA) && \
        (!defined(HAVE_FIPS) || \
         (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION >= 2))) \
         && !defined(WOLF_CRYPTO_CB_ONLY_RSA)
    ret = rsa_oaep_padding_test(key, &rng);
    if (ret != 0)
        return ret;

    #endif /* !HAVE_FAST_RSA && !HAVE_FIPS */
    #endif /* WC_NO_RSA_OAEP && !WC_NO_RNG */
#endif /* WOLFSSL_RSA_VERIFY_ONLY */

#if !defined(HAVE_FIPS) && !defined(HAVE_USER_RSA) && !defined(NO_ASN) \
    && !defined(WOLFSSL_RSA_VERIFY_ONLY)
    ret = rsa_export_key_test(key);
    if (ret != 0)
        return ret;
#endif

#if !defined(NO_ASN) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
                                               !defined(WOLFSSL_RSA_VERIFY_ONLY)
    ret = rsa_flatten_test(key);
    if (ret != 0)
        return ret;
#endif

#if !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(NO_ASN) && \
    !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048) && \
    !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
    (void)clientCert;
#endif

#ifdef WOLFSSL_TEST_CERT
#if defined(WOLFSSL_MDK_ARM)
    #define sizeof(s) XSTRLEN((char *)(s))
#endif

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, client_cert_der_1024, (size_t)sizeof_client_cert_der_1024);
    bytes = (size_t)sizeof_client_cert_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_cert_der_2048, (size_t)sizeof_client_cert_der_2048);
    bytes = (size_t)sizeof_client_cert_der_2048;
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_cert_der_3072, (size_t)sizeof_client_cert_der_3072);
    bytes = (size_t)sizeof_client_cert_der_3072;
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_cert_der_4096, (size_t)sizeof_client_cert_der_4096);
    bytes = (size_t)sizeof_client_cert_der_4096;
#elif !defined(NO_FILESYSTEM)
    file2 = XFOPEN(clientCert, "rb");
    if (!file2) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file2);
    XFCLOSE(file2);
    if (bytes == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
#else
    /* No certificate to use. */
    ERROR_OUT(WC_TEST_RET_ENC_NC, exit_rsa);
#endif

#ifdef sizeof
    #undef sizeof
#endif
    InitDecodedCert(cert, tmp, (word32)bytes, NULL);

    ret = ParseCert(cert, CERT_TYPE, NO_VERIFY, NULL);
    if (ret != 0) {
        FreeDecodedCert(cert);
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }

#ifndef NO_ASN_TIME
    ret = wc_GetDateInfo(cert->afterDate, cert->afterDateLen, &date,
                         &dateFormat, &dateLength);
    if (ret != 0) {
        FreeDecodedCert(cert);
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }
    ret = wc_GetDateAsCalendarTime(date, dateLength, dateFormat, &timearg);
    if (ret != 0) {
        FreeDecodedCert(cert);
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    }
#endif

    FreeDecodedCert(cert);
#endif /* WOLFSSL_TEST_CERT */

#ifdef WOLFSSL_CERT_EXT

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, client_keypub_der_1024, sizeof_client_keypub_der_1024);
    bytes = sizeof_client_keypub_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_keypub_der_2048, sizeof_client_keypub_der_2048);
    bytes = sizeof_client_keypub_der_2048;
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_keypub_der_3072, sizeof_client_keypub_der_3072);
    bytes = sizeof_client_keypub_der_3072;
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_keypub_der_4096, sizeof_client_keypub_der_4096);
    bytes = sizeof_client_keypub_der_4096;
#else
    file = XFOPEN(clientKeyPub, "rb");
    if (!file) {
        err_sys("can't open ./certs/client-keyPub.der, "
                "Please run from wolfSSL home dir", WC_TEST_RET_ENC_ERRNO);
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
    }

    bytes = XFREAD(tmp, 1, FOURK_BUF, file);
    XFCLOSE(file);
    if (bytes == 0)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitRsaKey(keypub, HEAP_HINT);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    idx = 0;

    ret = wc_RsaPublicKeyDecode(tmp, &idx, keypub, (word32)bytes);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
#endif /* WOLFSSL_CERT_EXT */

#ifdef WOLFSSL_KEY_GEN
    ret = rsa_keygen_test(&rng);
    if (ret != 0)
        goto exit_rsa;
#endif

#if defined(WOLFSSL_CERT_GEN) && !defined(NO_ASN_TIME)
    /* Make Cert / Sign example for RSA cert and RSA CA */
    ret = rsa_certgen_test(key, keypub, &rng, tmp);
    if (ret != 0)
        goto exit_rsa;

#if !defined(NO_RSA) && defined(HAVE_ECC) && !defined(NO_ECC_SECP)
    ret = rsa_ecc_certgen_test(&rng, tmp);
    if (ret != 0)
        goto exit_rsa;
#endif

#if defined(WOLFSSL_CERT_REQ) && !defined(WOLFSSL_NO_MALLOC)
    {
        Cert        *req;
        int         derSz;
#ifndef WOLFSSL_SMALL_STACK
        byte*  der = NULL;
#endif

        req = (Cert *)XMALLOC(sizeof *req, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if (! req)
            ERROR_OUT(MEMORY_E, exit_rsa);
        der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT,DYNAMIC_TYPE_TMP_BUFFER);
        if (der == NULL) {
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, exit_rsa);
        }

        ret = wc_InitCert_ex(req, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

        req->version = 0;
        req->isCA    = 1;
        XSTRNCPY(req->challengePw, "wolf123", CTC_NAME_SIZE);
        XMEMCPY(&req->subject, &certDefaultName, sizeof(CertName));

    #ifndef NO_SHA256
        req->sigType = CTC_SHA256wRSA;
    #else
        req->sigType = CTC_SHAwRSA;
    #endif

    #ifdef WOLFSSL_CERT_EXT
        /* add SKID from the Public Key */
        ret = wc_SetSubjectKeyIdFromPublicKey(req, keypub, NULL);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

        /* add Key Usage */
        ret = wc_SetKeyUsage(req, certKeyUsage2);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);

        /* add Extended Key Usage */
        ret = wc_SetExtKeyUsage(req,
                                "serverAuth,clientAuth,codeSigning,"
                                "emailProtection,timeStamping,OCSPSigning");
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
    #ifdef WOLFSSL_EKU_OID
        {
            WOLFSSL_SMALL_STACK_STATIC const char unique[] =
                "2.16.840.1.111111.100.1.10.1";
            ret = wc_SetExtKeyUsageOID(req, unique, sizeof(unique), 0,
                                       HEAP_HINT);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
        }
    #endif /* WOLFSSL_EKU_OID */
    #endif /* WOLFSSL_CERT_EXT */

        derSz = wc_MakeCertReq(req, der, FOURK_BUF, key, NULL);
        if (derSz < 0) {
            ERROR_OUT(WC_TEST_RET_ENC_EC(derSz), exit_rsa);
        }

    #ifdef WOLFSSL_CERT_EXT
        /* Try again with "any" flag set, will override all others */
        ret = wc_SetExtKeyUsage(req, "any");
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
        derSz = wc_MakeCertReq(req, der, FOURK_BUF, key, NULL);
        if (derSz < 0) {
            ERROR_OUT(WC_TEST_RET_ENC_EC(derSz), exit_rsa);
        }
    #endif /* WOLFSSL_CERT_EXT */

        ret = 0;
        do {
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
        #endif
            if (ret >= 0) {
                ret = wc_SignCert(req->bodySz, req->sigType, der, FOURK_BUF,
                          key, NULL, &rng);
            }
        } while (ret == WC_PENDING_E);
        if (ret < 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_rsa);
        derSz = (word32)ret;

        ret = SaveDerAndPem(der, derSz, certReqDerFile, certReqPemFile,
                            CERTREQ_TYPE);
        if (ret != 0) {
            goto exit_rsa;
        }

        derSz = wc_MakeCertReq_ex(req, der, FOURK_BUF, RSA_TYPE, key);
        if (derSz < 0) {
            ERROR_OUT(WC_TEST_RET_ENC_EC(derSz), exit_rsa);
        }

        /* Test getting the size of the buffer without providing the buffer.
         * derSz is set to the "largest buffer" we are willing to allocate. */
        derSz = wc_MakeCertReq(req, NULL, 10000, key, NULL);
        if (derSz < 0) {
            ERROR_OUT(WC_TEST_RET_ENC_EC(derSz), exit_rsa);
        }

        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(req, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        der = NULL;
    }
#endif /* WOLFSSL_CERT_REQ */
#endif /* WOLFSSL_CERT_GEN */

#if defined(WC_RSA_PSS) && !defined(HAVE_FIPS_VERSION) /* not supported with FIPSv1 */
/* Need to create known good signatures to test with this. */
#if !defined(WOLFSSL_RSA_VERIFY_ONLY) && !defined(WOLFSSL_RSA_PUBLIC_ONLY) && \
    !defined(WOLF_CRYPTO_CB_ONLY_RSA)
    ret = rsa_pss_test(&rng, key);
    if (ret != 0)
        goto exit_rsa;
#endif
#endif

#if defined(WOLFSSL_HAVE_SP_RSA) && defined(USE_FAST_MATH)
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    /* New key to be loaded in rsa_even_mod_test(). */
    if (key != NULL)
#endif
        wc_FreeRsaKey(key);
    /* New key to be loaded in rsa_even_mod_test(). */
    ret = rsa_even_mod_test(&rng, key);
#endif

exit_rsa:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key != NULL) {
        wc_FreeRsaKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    #if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
    if (keypub != NULL) {
        wc_FreeRsaKey(keypub);
        XFREE(keypub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    #endif
    #ifdef WOLFSSL_TEST_CERT
    if (cert != NULL)
        XFREE(cert, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
     XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
     XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    wc_FreeRsaKey(key);
    #if defined(WOLFSSL_CERT_EXT) || defined(WOLFSSL_CERT_GEN)
    wc_FreeRsaKey(keypub);
    #endif
#endif /* WOLFSSL_SMALL_STACK && !WOLFSSL_NO_MALLOC */

    wc_FreeRng(&rng);

    WC_FREE_VAR(in, HEAP_HINT);
    WC_FREE_VAR(out, HEAP_HINT);
    WC_FREE_VAR(plain, HEAP_HINT);

    (void)res;
    (void)bytes;
    (void)idx;
    (void)in;
    (void)out;
    (void)plain;
    (void)idx;
    (void)inStr;
    (void)inLen;
    (void)outSz;
    (void)plainSz;

    /* ret can be greater then 0 with certgen but all negative values should
     * be returned and treated as an error */
    if (ret >= 0) {
        return 0;
    }
    else {
        return ret;
    }
}

#endif /* !NO_RSA */


#ifndef NO_DH

static wc_test_ret_t dh_fips_generate_test(WC_RNG *rng)
{
    wc_test_ret_t ret = 0;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    DhKey  *key = (DhKey *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    DhKey  key[1];
#endif
    WOLFSSL_SMALL_STACK_STATIC const byte p[] = {
        0xc5, 0x7c, 0xa2, 0x4f, 0x4b, 0xd6, 0x8c, 0x3c,
        0xda, 0xc7, 0xba, 0xaa, 0xea, 0x2e, 0x5c, 0x1e,
        0x18, 0xb2, 0x7b, 0x8c, 0x55, 0x65, 0x9f, 0xea,
        0xe0, 0xa1, 0x36, 0x53, 0x2b, 0x36, 0xe0, 0x4e,
        0x3e, 0x64, 0xa9, 0xe4, 0xfc, 0x8f, 0x32, 0x62,
        0x97, 0xe4, 0xbe, 0xf7, 0xc1, 0xde, 0x07, 0x5a,
        0x89, 0x28, 0xf3, 0xfe, 0x4f, 0xfe, 0x68, 0xbc,
        0xfb, 0x0a, 0x7c, 0xa4, 0xb3, 0x14, 0x48, 0x89,
        0x9f, 0xaf, 0xb8, 0x43, 0xe2, 0xa0, 0x62, 0x5c,
        0xb4, 0x88, 0x3f, 0x06, 0x50, 0x11, 0xfe, 0x65,
        0x8d, 0x49, 0xd2, 0xf5, 0x4b, 0x74, 0x79, 0xdb,
        0x06, 0x62, 0x92, 0x89, 0xed, 0xda, 0xcb, 0x87,
        0x37, 0x16, 0xd2, 0xa1, 0x7a, 0xe8, 0xde, 0x92,
        0xee, 0x3e, 0x41, 0x4a, 0x91, 0x5e, 0xed, 0xf3,
        0x6c, 0x6b, 0x7e, 0xfd, 0x15, 0x92, 0x18, 0xfc,
        0xa7, 0xac, 0x42, 0x85, 0x57, 0xe9, 0xdc, 0xda,
        0x55, 0xc9, 0x8b, 0x28, 0x9e, 0xc1, 0xc4, 0x46,
        0x4d, 0x88, 0xed, 0x62, 0x8e, 0xdb, 0x3f, 0xb9,
        0xd7, 0xc8, 0xe3, 0xcf, 0xb8, 0x34, 0x2c, 0xd2,
        0x6f, 0x28, 0x06, 0x41, 0xe3, 0x66, 0x8c, 0xfc,
        0x72, 0xff, 0x26, 0x3b, 0x6b, 0x6c, 0x6f, 0x73,
        0xde, 0xf2, 0x90, 0x29, 0xe0, 0x61, 0x32, 0xc4,
        0x12, 0x74, 0x09, 0x52, 0xec, 0xf3, 0x1b, 0xa6,
        0x45, 0x98, 0xac, 0xf9, 0x1c, 0x65, 0x8e, 0x3a,
        0x91, 0x84, 0x4b, 0x23, 0x8a, 0xb2, 0x3c, 0xc9,
        0xfa, 0xea, 0xf1, 0x38, 0xce, 0xd8, 0x05, 0xe0,
        0xfa, 0x44, 0x68, 0x1f, 0xeb, 0xd9, 0x57, 0xb8,
        0x4a, 0x97, 0x5b, 0x88, 0xc5, 0xf1, 0xbb, 0xb0,
        0x49, 0xc3, 0x91, 0x7c, 0xd3, 0x13, 0xb9, 0x47,
        0xbb, 0x91, 0x8f, 0xe5, 0x26, 0x07, 0xab, 0xa9,
        0xc5, 0xd0, 0x3d, 0x95, 0x41, 0x26, 0x92, 0x9d,
        0x13, 0x67, 0xf2, 0x7e, 0x11, 0x88, 0xdc, 0x2d
    };
    WOLFSSL_SMALL_STACK_STATIC const byte g[] = {
        0x4a, 0x1a, 0xf3, 0xa4, 0x92, 0xe9, 0xee, 0x74,
        0x6e, 0x57, 0xd5, 0x8c, 0x2c, 0x5b, 0x41, 0x41,
        0x5e, 0xd4, 0x55, 0x19, 0xdc, 0xd9, 0x32, 0x91,
        0xf7, 0xfd, 0xc2, 0x57, 0xff, 0x03, 0x14, 0xdb,
        0xf1, 0xb7, 0x60, 0x0c, 0x43, 0x59, 0x3f, 0xff,
        0xac, 0xf1, 0x80, 0x9a, 0x15, 0x6f, 0xd8, 0x6e,
        0xb7, 0x85, 0x18, 0xc8, 0xec, 0x4e, 0x59, 0x4a,
        0xe2, 0x91, 0x43, 0x4c, 0xeb, 0x95, 0xb6, 0x2e,
        0x9a, 0xea, 0x53, 0x68, 0x80, 0x64, 0x69, 0x40,
        0xf9, 0xec, 0xbd, 0x85, 0x89, 0x26, 0x97, 0x67,
        0xaf, 0xb0, 0xad, 0x00, 0x1b, 0xd4, 0xfd, 0x94,
        0xd3, 0xe9, 0x92, 0xb1, 0xb4, 0xbc, 0x5a, 0xaa,
        0x92, 0x80, 0x89, 0x3b, 0x39, 0x05, 0x6c, 0x22,
        0x26, 0xfe, 0x5a, 0x28, 0x6c, 0x37, 0x50, 0x5a,
        0x38, 0x99, 0xcf, 0xf3, 0xc1, 0x96, 0x45, 0xdc,
        0x01, 0xcb, 0x20, 0x87, 0xa5, 0x00, 0x8c, 0xf5,
        0x4d, 0xc2, 0xef, 0xb8, 0x9b, 0xd1, 0x87, 0xbe,
        0xed, 0xd5, 0x0a, 0x29, 0x15, 0x34, 0x59, 0x4c,
        0x3a, 0x05, 0x22, 0x05, 0x44, 0x4f, 0x9f, 0xc8,
        0x47, 0x12, 0x24, 0x8e, 0xa8, 0x79, 0xe4, 0x67,
        0xba, 0x4d, 0x5b, 0x75, 0x56, 0x95, 0xeb, 0xe8,
        0x8a, 0xfa, 0x8e, 0x01, 0x8c, 0x1b, 0x74, 0x63,
        0xd9, 0x2f, 0xf7, 0xd3, 0x44, 0x8f, 0xa8, 0xf5,
        0xaf, 0x6c, 0x4f, 0xdb, 0xe7, 0xc9, 0x6c, 0x71,
        0x22, 0xa3, 0x1d, 0xf1, 0x40, 0xb2, 0xe0, 0x9a,
        0xb6, 0x72, 0xc9, 0xc0, 0x13, 0x16, 0xa2, 0x4a,
        0xe1, 0x92, 0xc7, 0x54, 0x23, 0xab, 0x9d, 0xa1,
        0xa1, 0xe5, 0x0b, 0xed, 0xba, 0xe8, 0x84, 0x37,
        0xb2, 0xe7, 0xfe, 0x32, 0x8d, 0xfa, 0x1c, 0x53,
        0x77, 0x97, 0xc7, 0xf3, 0x48, 0xc9, 0xdb, 0x2d,
        0x75, 0x52, 0x9d, 0x42, 0x51, 0x78, 0x62, 0x68,
        0x05, 0x45, 0x15, 0xf8, 0xa2, 0x4e, 0xf3, 0x0b
    };
    WOLFSSL_SMALL_STACK_STATIC const byte q[] = {
        0xe0, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
        0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
        0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
        0x40, 0x52, 0xed, 0x41
    };
    WOLFSSL_SMALL_STACK_STATIC const byte q0[] = {
                                                  0x00,
        0xe0, 0x35, 0x37, 0xaf, 0xb2, 0x50, 0x91, 0x8e,
        0xf2, 0x62, 0x2b, 0xd9, 0x9f, 0x6c, 0x11, 0x75,
        0xec, 0x24, 0x5d, 0x78, 0x59, 0xe7, 0x8d, 0xb5,
        0x40, 0x52, 0xed, 0x41
    };
    byte   priv[256];
    byte   pub[256];
    word32 privSz = sizeof(priv);
    word32 pubSz = sizeof(pub);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key == NULL)
        ERROR_OUT(MEMORY_E, exit_gen_test);
#endif

    /* Parameter Validation testing. */
    ret = wc_DhGenerateKeyPair(NULL, rng, priv, &privSz, pub, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, NULL, priv, &privSz, pub, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, rng, NULL, &privSz, pub, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, rng, priv, NULL, pub, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, NULL, &pubSz);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, NULL);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    ret = wc_DhSetKey_ex(key, p, sizeof(p), g, sizeof(g), q0, sizeof(q0));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    wc_FreeDhKey(key);

    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    ret = wc_DhSetKey_ex(key, p, sizeof(p), g, sizeof(g), q, sizeof(q));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    /* Use API. */
    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    ret = wc_DhCheckPubKey_ex(key, pub, pubSz, q0, sizeof(q0));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    wc_FreeDhKey(key);
    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    ret = wc_DhSetKey(key, p, sizeof(p), g, sizeof(g));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    ret = wc_DhCheckPubKey_ex(key, pub, pubSz, q, sizeof(q));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

#ifndef HAVE_SELFTEST
    ret = wc_DhCheckKeyPair(key, pub, pubSz, priv, privSz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    /* Taint the public key so the check fails. */
    pub[0]++;
    ret = wc_DhCheckKeyPair(key, pub, pubSz, priv, privSz);
    if (ret != MP_CMP_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    }

#ifdef WOLFSSL_KEY_GEN
    wc_FreeDhKey(key);
    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    ret = wc_DhGenerateParams(rng, 2048, key);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

    privSz = sizeof(priv);
    pubSz = sizeof(pub);

    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
#endif /* WOLFSSL_KEY_GEN */
#endif /* HAVE_SELFTEST */

    ret = 0;

exit_gen_test:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key) {
        wc_FreeDhKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_FreeDhKey(key);
#endif

    return ret;
}

static wc_test_ret_t dh_generate_test(WC_RNG *rng)
{
    wc_test_ret_t ret = 0;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    DhKey  *smallKey = NULL;
#else
    DhKey  smallKey[1];
#endif
    byte   p[2] = { 1, 7 }; /* 263 in decimal */
    byte   g[2] = { 0, 2 };
#if !defined(WOLFSSL_SP_MATH) && !defined(HAVE_FFDHE)
#ifdef WOLFSSL_DH_CONST
    /* the table for constant DH lookup will round to the lowest byte size 21 */
    byte   priv[21];
    byte   pub[21];
#else
    byte   priv[2];
    byte   pub[2];
#endif
    word32 privSz = sizeof(priv);
    word32 pubSz = sizeof(pub);
#endif
    int smallKey_inited = 0;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((smallKey = (DhKey *)XMALLOC(sizeof(*smallKey), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER)) == NULL)
        return WC_TEST_RET_ENC_ERRNO;
#endif

    ret = wc_InitDhKey_ex(smallKey, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    smallKey_inited = 1;

    /* Parameter Validation testing. */
    ret = wc_InitDhKey_ex(NULL, HEAP_HINT, devId);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    wc_FreeDhKey(NULL);

    ret = wc_DhSetKey(NULL, p, sizeof(p), g, sizeof(g));
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, NULL, sizeof(p), g, sizeof(g));
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, p, 0, g, sizeof(g));
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, p, sizeof(p), NULL, sizeof(g));
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, p, sizeof(p), g, 0);
    if (ret != BAD_FUNC_ARG) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    }
    ret = wc_DhSetKey(smallKey, p, sizeof(p), g, sizeof(g));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

#if !defined(WOLFSSL_SP_MATH) && !defined(HAVE_FFDHE)
    /* Use API. */
    ret = wc_DhGenerateKeyPair(smallKey, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &smallKey->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
    }
#else
    (void)rng;
    #if defined(HAVE_FIPS) || !defined(WOLFSSL_NO_DH186)
    ret = 0;
    #endif
#endif

#if !defined(HAVE_FIPS) && defined(WOLFSSL_NO_DH186)
    {
        byte   priv[260];
        byte   pub[260];
        word32 privSz = sizeof(priv);
        word32 pubSz = sizeof(pub);

        /* test odd ball param generation with DH */
        wc_FreeDhKey(smallKey);
        ret = wc_InitDhKey_ex(smallKey, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

        ret = wc_DhGenerateParams(rng, 2056, smallKey);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);

        privSz = sizeof(priv);
        pubSz = sizeof(pub);

        ret = wc_DhGenerateKeyPair(smallKey, rng, priv, &privSz, pub, &pubSz);
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &smallKey->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), exit_gen_test);
    }
#endif /* !HAVE_FIPS and WOLFSSL_NO_DH186 */

exit_gen_test:
    if (smallKey_inited)
        wc_FreeDhKey(smallKey);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (smallKey != NULL)
        XFREE(smallKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
typedef struct dh_pubvalue_test {
    const byte* data;
    word32 len;
} dh_pubvalue_test;

static wc_test_ret_t dh_test_check_pubvalue(void)
{
    wc_test_ret_t ret;
    word32 i;
    WOLFSSL_SMALL_STACK_STATIC const byte prime[] = {0x01, 0x00, 0x01};
    WOLFSSL_SMALL_STACK_STATIC const byte pubValZero[] = { 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValZeroLong[] = { 0x00, 0x00, 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValOne[] = { 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValOneLong[] = { 0x00, 0x00, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValPrimeMinusOne[] = { 0x01, 0x00, 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValPrimeLong[] = {0x00, 0x01, 0x00, 0x01};
    WOLFSSL_SMALL_STACK_STATIC const byte pubValPrimePlusOne[] = { 0x01, 0x00, 0x02 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTooBig0[] = { 0x02, 0x00, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTooBig1[] = { 0x01, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTooLong[] = { 0x01, 0x00, 0x00, 0x01 };
    const dh_pubvalue_test dh_pubval_fail[] = {
        { prime, sizeof(prime) },
        { pubValZero, sizeof(pubValZero) },
        { pubValZeroLong, sizeof(pubValZeroLong) },
        { pubValOne, sizeof(pubValOne) },
        { pubValOneLong, sizeof(pubValOneLong) },
        { pubValPrimeMinusOne, sizeof(pubValPrimeMinusOne) },
        { pubValPrimeLong, sizeof(pubValPrimeLong) },
        { pubValPrimePlusOne, sizeof(pubValPrimePlusOne) },
        { pubValTooBig0, sizeof(pubValTooBig0) },
        { pubValTooBig1, sizeof(pubValTooBig1) },
        { pubValTooLong, sizeof(pubValTooLong) },
    };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTwo[] = { 0x02 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValTwoLong[] = { 0x00, 0x00, 0x02 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValGood[] = { 0x12, 0x34 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValGoodLen[] = { 0x00, 0x12, 0x34 };
    WOLFSSL_SMALL_STACK_STATIC const byte pubValGoodLong[] = { 0x00, 0x00, 0x12, 0x34 };
    const dh_pubvalue_test dh_pubval_pass[] = {
        { pubValTwo, sizeof(pubValTwo) },
        { pubValTwoLong, sizeof(pubValTwoLong) },
        { pubValGood, sizeof(pubValGood) },
        { pubValGoodLen, sizeof(pubValGoodLen) },
        { pubValGoodLong, sizeof(pubValGoodLong) },
    };

    for (i = 0; i < sizeof(dh_pubval_fail) / sizeof(*dh_pubval_fail); i++) {
        ret = wc_DhCheckPubValue(prime, sizeof(prime), dh_pubval_fail[i].data,
                                                         dh_pubval_fail[i].len);
        if (ret != MP_VAL)
            return WC_TEST_RET_ENC_I(i);
    }

    for (i = 0; i < sizeof(dh_pubval_pass) / sizeof(*dh_pubval_pass); i++) {
        ret = wc_DhCheckPubValue(prime, sizeof(prime), dh_pubval_pass[i].data,
                                                         dh_pubval_pass[i].len);
        if (ret != 0)
            return WC_TEST_RET_ENC_I(i);
    }

    return 0;
}
#endif

#if defined(HAVE_FFDHE)

#if defined(HAVE_FFDHE_4096)
    #define MAX_DH_PRIV_SZ 39
    #define MAX_DH_KEY_SZ  512
#elif defined(HAVE_FFDHE_3072)
    #define MAX_DH_PRIV_SZ 34
    #define MAX_DH_KEY_SZ  384
#else
    #define MAX_DH_PRIV_SZ 29
    #define MAX_DH_KEY_SZ  256
#endif

#ifndef WC_NO_RNG

#if !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION == 2) && \
      (defined(WOLFSSL_SP_ARM64_ASM) || defined(WOLFSSL_SP_ARM32_ASM)))

#ifdef HAVE_PUBLIC_FFDHE
static wc_test_ret_t dh_ffdhe_test(WC_RNG *rng, const DhParams* params)
#else
static wc_test_ret_t dh_ffdhe_test(WC_RNG *rng, int name)
#endif
{
    wc_test_ret_t ret;
    word32 privSz, pubSz, privSz2, pubSz2;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte   *priv = (byte*)XMALLOC(MAX_DH_PRIV_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *pub = (byte*)XMALLOC(MAX_DH_KEY_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *priv2 = (byte*)XMALLOC(MAX_DH_PRIV_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *pub2 = (byte*)XMALLOC(MAX_DH_KEY_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *agree = (byte*)XMALLOC(MAX_DH_KEY_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte   *agree2 = (byte*)XMALLOC(MAX_DH_KEY_SZ, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DhKey  *key = (DhKey*)XMALLOC(sizeof(*key), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DhKey  *key2 = (DhKey*)XMALLOC(sizeof(*key2), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    byte   priv[MAX_DH_PRIV_SZ];
    byte   pub[MAX_DH_KEY_SZ];
    byte   priv2[MAX_DH_PRIV_SZ];
    byte   pub2[MAX_DH_KEY_SZ];
    byte   agree[MAX_DH_KEY_SZ];
    byte   agree2[MAX_DH_KEY_SZ];
    DhKey  key[1];
    DhKey  key2[1];
#endif
    word32 agreeSz = MAX_DH_KEY_SZ;
    word32 agreeSz2 = MAX_DH_KEY_SZ;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((priv == NULL) ||
        (pub == NULL) ||
        (priv2 == NULL) ||
        (pub2 == NULL) ||
        (agree == NULL) ||
        (agree2 == NULL) ||
        (key == NULL) ||
        (key2 == NULL))
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
#endif

    pubSz = MAX_DH_KEY_SZ;
    pubSz2 = MAX_DH_KEY_SZ;
    #ifdef HAVE_PUBLIC_FFDHE
    privSz = MAX_DH_PRIV_SZ;
    privSz2 = MAX_DH_PRIV_SZ;
    #else
    privSz = wc_DhGetNamedKeyMinSize(name);
    privSz2 = privSz;
    #endif

    XMEMSET(key, 0, sizeof(*key));
    XMEMSET(key2, 0, sizeof(*key2));

    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    ret = wc_InitDhKey_ex(key2, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#ifdef HAVE_PUBLIC_FFDHE
    ret = wc_DhSetKey(key, params->p, params->p_len, params->g, params->g_len);
#else
    ret = wc_DhSetNamedKey(key, name);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#ifdef HAVE_PUBLIC_FFDHE
    ret = wc_DhSetKey(key2, params->p, params->p_len, params->g,
                                                                 params->g_len);
#else
    ret = wc_DhSetNamedKey(key2, name);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhGenerateKeyPair(key2, rng, priv2, &privSz2, pub2, &pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhAgree(key, agree, &agreeSz, priv, privSz, pub2, pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhAgree(key2, agree2, &agreeSz2, priv2, privSz2, pub, pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    if (agreeSz != agreeSz2 || XMEMCMP(agree, agree2, agreeSz)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }

#if defined(WOLFSSL_HAVE_SP_DH) || defined(USE_FAST_MATH)
    /* Make p even */
    key->p.dp[0] &= (mp_digit)-2;
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhGenerateKeyPair(key, rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != MP_VAL && ret != MP_EXPTMOD_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    }

    ret = wc_DhAgree(key, agree, &agreeSz, priv, privSz, pub2, pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != MP_VAL && ret != MP_EXPTMOD_E && ret != ASYNC_OP_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    }

#ifndef HAVE_SELFTEST
    ret = wc_DhCheckKeyPair(key, pub, pubSz, priv, privSz);
    if (ret != MP_VAL && ret != MP_EXPTMOD_E && ret != MP_CMP_E &&
            ret != ASYNC_OP_E) {
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    }
#endif

    /* Getting here means success - set ret to 0. */
    ret = 0;
#endif

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC) && \
    !defined(WC_NO_RNG)
    if (priv)
        XFREE(priv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pub)
        XFREE(pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (priv2)
        XFREE(priv2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pub2)
        XFREE(pub2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (agree)
        XFREE(agree, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (agree2)
        XFREE(agree2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (key) {
        wc_FreeDhKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (key2) {
        wc_FreeDhKey(key2);
        XFREE(key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_FreeDhKey(key);
    wc_FreeDhKey(key2);
#endif

    return ret;
}
#endif /* !(HAVE_FIPS_VERSION == 2 && WOLFSSL_SP_ARMxx_ASM) */
#endif /* !WC_NO_RNG */
#endif /* HAVE_FFDHE */

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t dh_test(void)
{
    wc_test_ret_t ret;
    word32 bytes;
    word32 idx = 0, privSz, pubSz, privSz2, pubSz2;
#ifndef WC_NO_RNG
    WC_RNG rng;
    int rngInit = 0;
#endif
    int keyInit = 0, key2Init = 0;

#define DH_TEST_TMP_SIZE 1024
#if !defined(USE_CERT_BUFFERS_3072) && !defined(USE_CERT_BUFFERS_4096)
    #define DH_TEST_BUF_SIZE 256
#else
    #define DH_TEST_BUF_SIZE 512
#endif
#ifndef WC_NO_RNG
    word32 agreeSz = DH_TEST_BUF_SIZE;
    word32 agreeSz2 = DH_TEST_BUF_SIZE;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    DhKey *key = (DhKey *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DhKey *key2 = (DhKey *)XMALLOC(sizeof *key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte  *tmp = (byte *)XMALLOC(DH_TEST_TMP_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    DhKey key[1];
    DhKey key2[1];
    byte  tmp[DH_TEST_TMP_SIZE];
#endif

#ifndef WC_NO_RNG
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte *priv = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *pub = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *priv2 = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *pub2 = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *agree = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *agree2 = (byte *)XMALLOC(DH_TEST_BUF_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (priv == NULL || pub == NULL || priv2 == NULL || pub2 == NULL ||
            agree == NULL || agree2 == NULL) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    }
#else
    byte priv[DH_TEST_BUF_SIZE];
    byte pub[DH_TEST_BUF_SIZE];
    byte priv2[DH_TEST_BUF_SIZE];
    byte pub2[DH_TEST_BUF_SIZE];
    byte agree[DH_TEST_BUF_SIZE];
    byte agree2[DH_TEST_BUF_SIZE];
#endif
#endif /* !WC_NO_RNG */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key == NULL || key2 == NULL || tmp == NULL) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    }
#endif

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, dh_key_der_1024, (size_t)sizeof_dh_key_der_1024);
    bytes = (size_t)sizeof_dh_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, dh_key_der_2048, (size_t)sizeof_dh_key_der_2048);
    bytes = (size_t)sizeof_dh_key_der_2048;
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, dh_key_der_3072, (size_t)sizeof_dh_key_der_3072);
    bytes = (size_t)sizeof_dh_key_der_3072;
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, dh_key_der_4096, (size_t)sizeof_dh_key_der_4096);
    bytes = (size_t)sizeof_dh_key_der_4096;
#elif defined(NO_ASN)
    /* don't use file, no DER parsing */
#elif !defined(NO_FILESYSTEM)
    {
        XFILE file = XFOPEN(dhParamsFile, "rb");
        if (! file)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);

        bytes = (word32) XFREAD(tmp, 1, DH_TEST_TMP_SIZE, file);
        XFCLOSE(file);
        if (bytes == 0)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    }
#else
    /* No DH key to use. */
    ERROR_OUT(WC_TEST_RET_ENC_NC, done);
#endif /* USE_CERT_BUFFERS */

    (void)idx;
    (void)tmp;
    (void)bytes;

    pubSz = DH_TEST_BUF_SIZE;
    pubSz2 = DH_TEST_BUF_SIZE;
    privSz = DH_TEST_BUF_SIZE;
    privSz2 = DH_TEST_BUF_SIZE;

#ifndef WC_NO_RNG
    XMEMSET(&rng, 0, sizeof(rng));
#endif

    /* Use API for coverage. */
    ret = wc_InitDhKey(key);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    wc_FreeDhKey(key);

    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    keyInit = 1;
    ret = wc_InitDhKey_ex(key2, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    key2Init = 1;

#ifdef NO_ASN
#ifndef WOLFSSL_SP_MATH
    ret = wc_DhSetKey(key, dh_p, sizeof(dh_p), dh_g, sizeof(dh_g));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhSetKey(key2, dh_p, sizeof(dh_p), dh_g, sizeof(dh_g));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#else
    ret = wc_DhSetKey(key, dh2048_p, sizeof(dh2048_p), dh2048_g,
        sizeof(dh2048_g));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhSetKey(key2, dh2048_p, sizeof(dh2048_p), dh2048_g,
        sizeof(dh2048_g));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#endif
#else
    ret = wc_DhKeyDecode(tmp, &idx, key, bytes);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    idx = 0;
    ret = wc_DhKeyDecode(tmp, &idx, key2, bytes);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#endif

#ifndef WC_NO_RNG
#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    rngInit = 1;

    ret = wc_DhGenerateKeyPair(key, &rng, priv, &privSz, pub, &pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhGenerateKeyPair(key2, &rng, priv2, &privSz2, pub2, &pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhAgree(key, agree, &agreeSz, priv, privSz, pub2, pubSz2);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhAgree(key2, agree2, &agreeSz2, priv2, privSz2, pub, pubSz);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key2->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    if (agreeSz != agreeSz2 || XMEMCMP(agree, agree2, agreeSz)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }
#endif /* !WC_NO_RNG */

#if defined(WOLFSSL_KEY_GEN) && !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    ret = wc_DhCheckPrivKey(NULL, NULL, 0);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhCheckPrivKey(key, priv, privSz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_DhExportParamsRaw(NULL, NULL, NULL, NULL, NULL, NULL, NULL);
    if (ret != BAD_FUNC_ARG)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    {
        word32 pSz, qSz, gSz;
        ret = wc_DhExportParamsRaw(key, NULL, &pSz, NULL, &qSz, NULL, &gSz);
        if (ret != LENGTH_ONLY_E)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    }
#endif

    /* Test DH key import / export */
#if defined(WOLFSSL_DH_EXTRA) && !defined(NO_FILESYSTEM) && \
        (!defined(HAVE_FIPS) || \
        (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    wc_FreeDhKey(key);
    ret = wc_InitDhKey_ex(key, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#ifndef NO_ASN
    {
        /* DH Private - Key Export / Import */
    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        byte *tmp2;
    #else
        byte  tmp2[DH_TEST_TMP_SIZE];
    #endif

    #if defined(USE_CERT_BUFFERS_2048)
        XMEMCPY(tmp, dh_ffdhe_statickey_der_2048, sizeof_dh_ffdhe_statickey_der_2048);
        bytes = sizeof_dh_ffdhe_statickey_der_2048;
    #else
        XFILE file = XFOPEN(dhKeyFile, "rb");
        if (!file)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
        bytes = (word32)XFREAD(tmp, 1, DH_TEST_TMP_SIZE, file);
        XFCLOSE(file);
        if (bytes == 0)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    #endif
    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        tmp2 = (byte*)XMALLOC(DH_TEST_TMP_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        if (tmp2 == NULL)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    #endif
        idx = 0;
        XMEMSET(tmp2, 0, DH_TEST_TMP_SIZE);

        /* Import DH Private key as DER */
        ret = wc_DhKeyDecode(tmp, &idx, key, bytes);
        if (ret == 0) {
            /* Export as DER */
            idx = DH_TEST_TMP_SIZE;
            ret = wc_DhPrivKeyToDer(key, tmp2, &idx);
        }

        /* Verify export matches original */
        if (ret <= 0 || bytes != idx || XMEMCMP(tmp, tmp2, bytes) != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_NC, done);
        }


        /* DH Public Key - Export / Import */
    #if defined(USE_CERT_BUFFERS_2048)
        XMEMCPY(tmp, dh_ffdhe_pub_statickey_der_2048, sizeof_dh_ffdhe_pub_statickey_der_2048);
        bytes = sizeof_dh_ffdhe_pub_statickey_der_2048;
    #else
        file = XFOPEN(dhKeyPubFile, "rb");
        if (!file)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
        bytes = (word32)XFREAD(tmp, 1, DH_TEST_TMP_SIZE, file);
        XFCLOSE(file);
        if (bytes == 0)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    #endif

        /* for HAVE_WOLF_BIGINT prevent leak */
        wc_FreeDhKey(key);
        (void)wc_InitDhKey_ex(key, HEAP_HINT, devId);

        idx = 0;
        XMEMSET(tmp2, 0, DH_TEST_TMP_SIZE);

        /* Import DH Public key as DER */
        ret = wc_DhKeyDecode(tmp, &idx, key, bytes);
        if (ret == 0) {
            /* Export as DER */
            idx = DH_TEST_TMP_SIZE;
            ret = wc_DhPubKeyToDer(key, tmp2, &idx);
        }

        /* Verify export matches original */
        if (ret <= 0 || bytes != idx || XMEMCMP(tmp, tmp2, bytes) != 0) {
            ERROR_OUT(WC_TEST_RET_ENC_NC, done);
        }

    #if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        XFREE(tmp2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    #endif
    }
#else
    ret = wc_DhSetKey(key, dh_p, sizeof(dh_p), dh_g, sizeof(dh_g));
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#endif /* !NO_ASN */

    privSz = DH_TEST_BUF_SIZE;
    pubSz = DH_TEST_BUF_SIZE;
    ret = wc_DhExportKeyPair(key, priv, &privSz, pub, &pubSz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    ret = wc_DhImportKeyPair(key2, priv, privSz, pub, pubSz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#endif /* WOLFSSL_DH_EXTRA && !NO_FILESYSTEM && !FIPS <= 2 */

#ifndef WC_NO_RNG
    ret = dh_generate_test(&rng);
    if (ret != 0)
        ERROR_OUT(ret, done);

    ret = dh_fips_generate_test(&rng);
    if (ret != 0)
        ERROR_OUT(ret, done);
#endif /* !WC_NO_RNG */

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST)
    ret = dh_test_check_pubvalue();
    if (ret != 0)
        ERROR_OUT(ret, done);
#endif

#if !(defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION == 2) && \
      (defined(WOLFSSL_SP_ARM64_ASM) || defined(WOLFSSL_SP_ARM32_ASM)))
/* RNG with DH and SP_ASM code not supported in the in-lined FIPS ASM code,
 * this will be available for testing in the 140-3 module */
#ifndef WC_NO_RNG
    /* Specialized code for key gen when using FFDHE-2048, FFDHE-3072 and FFDHE-4096 */
    #ifdef HAVE_FFDHE_2048
    #ifdef HAVE_PUBLIC_FFDHE
    ret = dh_ffdhe_test(&rng, wc_Dh_ffdhe2048_Get());
    #else
    ret = dh_ffdhe_test(&rng, WC_FFDHE_2048);
    #endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    #endif
    #ifdef HAVE_FFDHE_3072
    #ifdef HAVE_PUBLIC_FFDHE
    ret = dh_ffdhe_test(&rng, wc_Dh_ffdhe3072_Get());
    #else
    ret = dh_ffdhe_test(&rng, WC_FFDHE_3072);
    #endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    #endif
    #ifdef HAVE_FFDHE_4096
    #ifdef HAVE_PUBLIC_FFDHE
    ret = dh_ffdhe_test(&rng, wc_Dh_ffdhe4096_Get());
    #else
    ret = dh_ffdhe_test(&rng, WC_FFDHE_4096);
    #endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    #endif
#endif /* !WC_NO_RNG */
#endif /* HAVE_FIPS_VERSION == 2 && !WOLFSSL_SP_ARM64_ASM */

    wc_FreeDhKey(key);
    keyInit = 0;

#if !defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) && \
    !defined(WOLFSSL_OLD_PRIME_CHECK) && !defined(WC_NO_RNG)
    /* Test Check Key */
    ret = wc_DhSetCheckKey(key, dh_p, sizeof(dh_p), dh_g, sizeof(dh_g),
                           NULL, 0, 0, &rng);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    keyInit = 1; /* DhSetCheckKey also initializes the key, free it */
#endif

done:

#ifndef WC_NO_RNG
    if (rngInit)
        wc_FreeRng(&rng);
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key) {
        if (keyInit)
            wc_FreeDhKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (key2) {
        if (key2Init)
            wc_FreeDhKey(key2);
        XFREE(key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (tmp)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (priv)
        XFREE(priv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pub)
        XFREE(pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (priv2)
        XFREE(priv2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pub2)
        XFREE(pub2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (agree)
        XFREE(agree, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (agree2)
        XFREE(agree2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    if (keyInit)
        wc_FreeDhKey(key);
    if (key2Init)
        wc_FreeDhKey(key2);
#endif

    (void)privSz;
    (void)pubSz;
    (void)pubSz2;
    (void)privSz2;

    return ret;
#undef DH_TEST_BUF_SIZE
#undef DH_TEST_TMP_SIZE
}

#endif /* NO_DH */


#ifndef NO_DSA

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t dsa_test(void)
{
    wc_test_ret_t ret = 0;
    int answer;
    word32 bytes;
    word32 idx = 0;
    WC_RNG rng;
    int rng_inited = 0;
    wc_Sha sha;
    byte   hash[WC_SHA_DIGEST_SIZE];
    byte   signature[40];
    int key_inited = 0;
#ifdef WOLFSSL_KEY_GEN
    byte*  der = 0;
    int derIn_inited = 0;
    int genKey_inited = 0;
#endif
#define DSA_TEST_TMP_SIZE 1024
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte   *tmp = (byte *)XMALLOC(DSA_TEST_TMP_SIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DsaKey *key = (DsaKey *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#ifdef WOLFSSL_KEY_GEN
    DsaKey *derIn = (DsaKey *)XMALLOC(sizeof *derIn, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    DsaKey *genKey = (DsaKey *)XMALLOC(sizeof *genKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    if ((tmp == NULL) ||
        (key == NULL)
#ifdef WOLFSSL_KEY_GEN
        || (derIn == NULL)
        || (genKey == NULL)
#endif
        ) {
        ret = WC_TEST_RET_ENC_NC;
        goto out;
    }
#else
    byte   tmp[1024];
    DsaKey key[1];
#ifdef WOLFSSL_KEY_GEN
    DsaKey derIn[1];
    DsaKey genKey[1];
#endif
#endif

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(tmp, dsa_key_der_1024, sizeof_dsa_key_der_1024);
    bytes = sizeof_dsa_key_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, dsa_key_der_2048, sizeof_dsa_key_der_2048);
    bytes = sizeof_dsa_key_der_2048;
#else
    {
        XFILE  file = XFOPEN(dsaKey, "rb");
        if (!file)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);

        bytes = (word32) XFREAD(tmp, 1, DSA_TEST_TMP_SIZE, file);
        XFCLOSE(file);
        if (bytes == 0)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, out);
    }
#endif /* USE_CERT_BUFFERS */

    ret = wc_InitSha_ex(&sha, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    wc_ShaUpdate(&sha, tmp, bytes);
    wc_ShaFinal(&sha, hash);
    wc_ShaFree(&sha);

    ret = wc_InitDsaKey(key);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    key_inited = 1;

    ret = wc_DsaPrivateKeyDecode(tmp, &idx, key, bytes);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

#ifndef HAVE_FIPS
    ret = wc_InitRng_ex(&rng, HEAP_HINT, devId);
#else
    ret = wc_InitRng(&rng);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    rng_inited = 1;

    ret = wc_DsaSign(hash, signature, key, &rng);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_DsaVerify(hash, signature, key, &answer);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    if (answer != 1)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    wc_FreeDsaKey(key);
    key_inited = 0;

    ret = wc_InitDsaKey_h(key, NULL);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    key_inited = 1;

#ifdef WOLFSSL_KEY_GEN
    {
    int    derSz = 0;

    ret = wc_InitDsaKey(genKey);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    genKey_inited = 1;

    ret = wc_MakeDsaParameters(&rng, 1024, genKey);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    ret = wc_MakeDsaKey(&rng, genKey);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);

    der = (byte*)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (der == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_NC, out);

    derSz = wc_DsaKeyToDer(genKey, der, FOURK_BUF);
    if (derSz < 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(derSz), out);

    ret = SaveDerAndPem(der, derSz, keyDerFile, keyPemFile,
                        DSA_PRIVATEKEY_TYPE);
    if (ret != 0)
        goto out;

    ret = wc_InitDsaKey(derIn);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    derIn_inited = 1;

    idx = 0;
    ret = wc_DsaPrivateKeyDecode(der, &idx, derIn, derSz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), out);
    }
#endif /* WOLFSSL_KEY_GEN */

  out:

#ifdef WOLFSSL_KEY_GEN
    if (der)
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (tmp)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (key) {
        if (key_inited)
            wc_FreeDsaKey(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#ifdef WOLFSSL_KEY_GEN
    if (derIn) {
        if (derIn_inited)
            wc_FreeDsaKey(derIn);
        XFREE(derIn, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (genKey) {
        if (genKey_inited)
            wc_FreeDsaKey(genKey);
        XFREE(genKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

#else /* !WOLFSSL_SMALL_STACK || WOLFSSL_NO_MALLOC */

    if (key_inited)
        wc_FreeDsaKey(key);
#ifdef WOLFSSL_KEY_GEN
    if (derIn_inited)
        wc_FreeDsaKey(derIn);
    if (genKey_inited)
        wc_FreeDsaKey(genKey);
#endif

#endif

    if (rng_inited)
        wc_FreeRng(&rng);

    return ret;
}

#endif /* NO_DSA */

#ifdef WOLFCRYPT_HAVE_SRP

static wc_test_ret_t generate_random_salt(byte *buf, word32 size)
{
    wc_test_ret_t ret = WC_TEST_RET_ENC_NC;
    WC_RNG rng;

    if(NULL == buf || !size)
        return WC_TEST_RET_ENC_NC;

    if (buf && size && wc_InitRng_ex(&rng, HEAP_HINT, devId) == 0) {
        ret = wc_RNG_GenerateBlock(&rng, (byte *)buf, size);

        wc_FreeRng(&rng);
    }

    return ret;
}

static wc_test_ret_t srp_test_digest(SrpType dgstType)
{
    wc_test_ret_t r;

    byte clientPubKey[192]; /* A */
    byte serverPubKey[192]; /* B */
    word32 clientPubKeySz = 192;
    word32 serverPubKeySz = 192;

    byte username[] = "user";
    word32 usernameSz = 4;

    byte password[] = "password";
    word32 passwordSz = 8;

    WOLFSSL_SMALL_STACK_STATIC const byte N[] = {
        0xfc, 0x58, 0x7a, 0x8a, 0x70, 0xfb, 0x5a, 0x9a,
        0x5d, 0x39, 0x48, 0xbf, 0x1c, 0x46, 0xd8, 0x3b,
        0x7a, 0xe9, 0x1f, 0x85, 0x36, 0x18, 0xc4, 0x35,
        0x3f, 0xf8, 0x8a, 0x8f, 0x8c, 0x10, 0x2e, 0x01,
        0x58, 0x1d, 0x41, 0xcb, 0xc4, 0x47, 0xa8, 0xaf,
        0x9a, 0x6f, 0x58, 0x14, 0xa4, 0x68, 0xf0, 0x9c,
        0xa6, 0xe7, 0xbf, 0x0d, 0xe9, 0x62, 0x0b, 0xd7,
        0x26, 0x46, 0x5b, 0x27, 0xcb, 0x4c, 0xf9, 0x7e,
        0x1e, 0x8b, 0xe6, 0xdd, 0x29, 0xb7, 0xb7, 0x15,
        0x2e, 0xcf, 0x23, 0xa6, 0x4b, 0x97, 0x9f, 0x89,
        0xd4, 0x86, 0xc4, 0x90, 0x63, 0x92, 0xf4, 0x30,
        0x26, 0x69, 0x48, 0x9d, 0x7a, 0x4f, 0xad, 0xb5,
        0x6a, 0x51, 0xad, 0xeb, 0xf9, 0x90, 0x31, 0x77,
        0x53, 0x30, 0x2a, 0x85, 0xf7, 0x11, 0x21, 0x0c,
        0xb8, 0x4b, 0x56, 0x03, 0x5e, 0xbb, 0x25, 0x33,
        0x7c, 0xd9, 0x5a, 0xd1, 0x5c, 0xb2, 0xd4, 0x53,
        0xc5, 0x16, 0x68, 0xf0, 0xdf, 0x48, 0x55, 0x3e,
        0xd4, 0x59, 0x87, 0x64, 0x59, 0xaa, 0x39, 0x01,
        0x45, 0x89, 0x9c, 0x72, 0xff, 0xdd, 0x8f, 0x6d,
        0xa0, 0x42, 0xbc, 0x6f, 0x6e, 0x62, 0x18, 0x2d,
        0x50, 0xe8, 0x18, 0x97, 0x87, 0xfc, 0xef, 0x1f,
        0xf5, 0x53, 0x68, 0xe8, 0x49, 0xd1, 0xa2, 0xe8,
        0xb9, 0x26, 0x03, 0xba, 0xb5, 0x58, 0x6f, 0x6c,
        0x8b, 0x08, 0xa1, 0x7b, 0x6f, 0x42, 0xc9, 0x53
    };

    WOLFSSL_SMALL_STACK_STATIC const byte g[] = {
        0x02
    };

    byte salt[10];

    byte verifier[192];
    word32 v_size = sizeof(verifier);

    word32 clientProofSz = SRP_MAX_DIGEST_SIZE;
    word32 serverProofSz = SRP_MAX_DIGEST_SIZE;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    Srp *cli = (Srp *)XMALLOC(sizeof *cli, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    Srp *srv = (Srp *)XMALLOC(sizeof *srv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *clientProof = (byte *)XMALLOC(SRP_MAX_DIGEST_SIZE, HEAP_HINT,
                                        DYNAMIC_TYPE_TMP_BUFFER); /* M1 */
    byte *serverProof = (byte *)XMALLOC(SRP_MAX_DIGEST_SIZE, HEAP_HINT,
                                        DYNAMIC_TYPE_TMP_BUFFER); /* M2 */

    if ((cli == NULL) ||
        (srv == NULL) ||
        (clientProof == NULL) ||
        (serverProof == NULL)) {
        r = WC_TEST_RET_ENC_NC;
        goto out;
    }
#else
    Srp cli[1], srv[1];
    byte clientProof[SRP_MAX_DIGEST_SIZE]; /* M1 */
    byte serverProof[SRP_MAX_DIGEST_SIZE]; /* M2 */
#endif


    /* set as 0's so if second init on srv not called SrpTerm is not on
     * garbage values */
    XMEMSET(srv, 0, sizeof *srv);
    XMEMSET(cli, 0, sizeof *cli);

    /* generating random salt */

    r = generate_random_salt(salt, sizeof(salt));

    /* client knows username and password.   */
    /* server knows N, g, salt and verifier. */

    if (!r) r = wc_SrpInit_ex(cli, dgstType, SRP_CLIENT_SIDE, HEAP_HINT, devId);
    if (!r) r = wc_SrpSetUsername(cli, username, usernameSz);

    /* loading N, g and salt in advance to generate the verifier. */

    if (!r) r = wc_SrpSetParams(cli, N,    sizeof(N),
                                      g,    sizeof(g),
                                      salt, sizeof(salt));
    if (!r) r = wc_SrpSetPassword(cli, password, passwordSz);
    if (!r) r = wc_SrpGetVerifier(cli, verifier, &v_size);

    /* client sends username to server */

    if (!r) r = wc_SrpInit_ex(srv, dgstType, SRP_SERVER_SIDE, HEAP_HINT, devId);
    if (!r) r = wc_SrpSetUsername(srv, username, usernameSz);
    if (!r) r = wc_SrpSetParams(srv, N,    sizeof(N),
                                      g,    sizeof(g),
                                      salt, sizeof(salt));
    if (!r) r = wc_SrpSetVerifier(srv, verifier, v_size);
    if (!r) r = wc_SrpGetPublic(srv, serverPubKey, &serverPubKeySz);

    /* server sends N, g, salt and B to client */

    if (!r) r = wc_SrpGetPublic(cli, clientPubKey, &clientPubKeySz);
    if (!r) r = wc_SrpComputeKey(cli, clientPubKey, clientPubKeySz,
                                       serverPubKey, serverPubKeySz);
    if (!r) r = wc_SrpGetProof(cli, clientProof, &clientProofSz);

    /* client sends A and M1 to server */

    if (!r) r = wc_SrpComputeKey(srv, clientPubKey, clientPubKeySz,
                                       serverPubKey, serverPubKeySz);
    if (!r) r = wc_SrpVerifyPeersProof(srv, clientProof, clientProofSz);
    if (!r) r = wc_SrpGetProof(srv, serverProof, &serverProofSz);

    /* server sends M2 to client */

    if (!r) r = wc_SrpVerifyPeersProof(cli, serverProof, serverProofSz);

    wc_SrpTerm(cli);
    wc_SrpTerm(srv);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
  out:

    if (cli)
        XFREE(cli, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (srv)
        XFREE(srv, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (clientProof)
        XFREE(clientProof, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (serverProof)
        XFREE(serverProof, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return r;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t srp_test(void)
{
    wc_test_ret_t ret;

#ifndef NO_SHA
    ret = srp_test_digest(SRP_TYPE_SHA);
    if (ret != 0)
        return ret;
#endif
#ifndef NO_SHA256
    ret = srp_test_digest(SRP_TYPE_SHA256);
    if (ret != 0)
        return ret;
#endif
#ifdef WOLFSSL_SHA384
    ret = srp_test_digest(SRP_TYPE_SHA384);
    if (ret != 0)
        return ret;
#endif
#ifdef WOLFSSL_SHA512
    ret = srp_test_digest(SRP_TYPE_SHA512);
    if (ret != 0)
        return ret;
#endif

    return ret;
}

#endif /* WOLFCRYPT_HAVE_SRP */

#if defined(OPENSSL_EXTRA) && !defined(WOLFCRYPT_ONLY)

#if !defined(NO_AES) && !defined(WOLFCRYPT_ONLY)
static wc_test_ret_t openssl_aes_test(void)
{
#ifdef HAVE_AES_CBC
#ifdef WOLFSSL_AES_128
    {
        /* EVP_CipherUpdate test */
        WOLFSSL_SMALL_STACK_STATIC const byte cbcPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
            "0123456789abcdef   ";  /* align */
        WOLFSSL_SMALL_STACK_STATIC const byte iv[]  =
            "1234567890abcdef   ";  /* align */

        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
        EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX en[1];
        EVP_CIPHER_CTX de[1];
#endif
        int outlen ;
        int total = 0;
        int i;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if ((en == NULL) || (de == NULL))
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                    (byte*)cbcPlain, 9) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherUpdate(en, (byte*)&cipher[total], &outlen,
                    (byte*)&cbcPlain[9]  , 9) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;
        if (total != 32)
            return 3408;

        EVP_CIPHER_CTX_cleanup(en);

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher, 6) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                    (byte*)&cipher[6], 12) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 0)
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                    (byte*)&cipher[6+12], 14) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 2)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (total != 18)
            return 3427;

        if (XMEMCMP(plain, cbcPlain, 18))
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_cleanup(de);

        /* test with encrypting/decrypting more than 16 bytes at once */
        total = 0;
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                    (byte*)cbcPlain, 17) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherUpdate(en, (byte*)&cipher[total], &outlen,
                    (byte*)&cbcPlain[17]  , 1) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;
        if (total != 32)
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_cleanup(en);

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher, 17) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        /* final call on non block size should fail */
        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) != 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                    (byte*)&cipher[17], 1) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 0)
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                    (byte*)&cipher[17+1], 14) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 2)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (total != 18)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plain, cbcPlain, 18))
            return WC_TEST_RET_ENC_NC;

        /* test byte by byte decrypt */
        for (i = 0; i < AES_BLOCK_SIZE * 3; i++) {
            plain[i] = i;
        }

        EVP_CIPHER_CTX_cleanup(de);

        total = 0;
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                    (byte*)plain, AES_BLOCK_SIZE * 3) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != AES_BLOCK_SIZE * 3)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != AES_BLOCK_SIZE)
            return WC_TEST_RET_ENC_NC;
        total += outlen;
        if (total != sizeof(plain))
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_cleanup(en);

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        for (i = 0; i < AES_BLOCK_SIZE * 4; i++) {
            if (EVP_CipherUpdate(de, (byte*)plain + total, &outlen,
                        (byte*)cipher + i, 1) == 0)
                return WC_TEST_RET_ENC_NC;

            if (outlen > 0) {
                int j;

                total += outlen;
                for (j = 0; j < total; j++) {
                    if (plain[j] != j) {
                        return WC_TEST_RET_ENC_NC;
                    }
                }
            }
        }

        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;
        if (total != AES_BLOCK_SIZE * 3) {
            return WC_TEST_RET_ENC_NC;
        }
        for (i = 0; i < AES_BLOCK_SIZE * 3; i++) {
            if (plain[i] != i) {
                return WC_TEST_RET_ENC_NC;
            }
        }

        EVP_CIPHER_CTX_cleanup(de);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        wolfSSL_EVP_CIPHER_CTX_free(en);
        wolfSSL_EVP_CIPHER_CTX_free(de);
#endif
    }

    /* set buffers to be exact size to catch potential over read/write */
    {
        /* EVP_CipherUpdate test */
        WOLFSSL_SMALL_STACK_STATIC const byte cbcPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
            "0123456789abcdef   ";  /* align */
        WOLFSSL_SMALL_STACK_STATIC const byte iv[]  =
            "1234567890abcdef   ";  /* align */

        #define EVP_TEST_BUF_SZ 18
        #define EVP_TEST_BUF_PAD 32
        byte cipher[EVP_TEST_BUF_SZ];
        byte plain [EVP_TEST_BUF_SZ];
        byte padded[EVP_TEST_BUF_PAD];
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
        EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX en[1];
        EVP_CIPHER_CTX de[1];
#endif
        int outlen ;
        int total = 0;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if ((en == NULL) || (de == NULL))
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CIPHER_CTX_set_padding(en, 0) != 1)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                    (byte*)cbcPlain, EVP_TEST_BUF_SZ) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        /* should fail here */
        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) != 0)
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_cleanup(en);

        /* turn padding back on and do successful encrypt */
        total = 0;
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CIPHER_CTX_set_padding(en, 1) != 1)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(en, (byte*)padded, &outlen,
                    (byte*)cbcPlain, EVP_TEST_BUF_SZ) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&padded[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;
        if (total != 32)
            return WC_TEST_RET_ENC_NC;
        XMEMCPY(cipher, padded, EVP_TEST_BUF_SZ);

        EVP_CIPHER_CTX_cleanup(en);

        /* test out of bounds read on buffers w/o padding during decryption */
        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_CTX_set_padding(de, 0) != 1)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher,
                    EVP_TEST_BUF_SZ) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        /* should fail since not using padding */
        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) != 0)
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_cleanup(de);

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CIPHER_CTX_set_padding(de, 1) != 1)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(de, (byte*)padded, &outlen, (byte*)padded,
                    EVP_TEST_BUF_PAD) == 0)
            return WC_TEST_RET_ENC_NC;
        if (outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(de, (byte*)&padded[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(padded, cbcPlain, EVP_TEST_BUF_SZ))
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_cleanup(de);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        wolfSSL_EVP_CIPHER_CTX_free(en);
        wolfSSL_EVP_CIPHER_CTX_free(de);
#endif
    }

    {  /* evp_cipher test: EVP_aes_128_cbc */
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *ctx = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX ctx[1];
#endif

        WOLFSSL_SMALL_STACK_STATIC const byte msg[] = { /* "Now is the time for all " w/o trailing 0 */
            0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
            0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
            0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
        };

        WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
        {
            0x95,0x94,0x92,0x57,0x5f,0x42,0x81,0x53,
            0x2c,0xcc,0x9d,0x46,0x77,0xa2,0x33,0xcb
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
            "0123456789abcdef   ";  /* align */
        WOLFSSL_SMALL_STACK_STATIC const byte iv[]  =
            "1234567890abcdef   ";  /* align */

        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if (ctx == NULL)
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(ctx);
        if (EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 1) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_Cipher(ctx, cipher, (byte*)msg, 16) != 16)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_cleanup(ctx);

        EVP_CIPHER_CTX_init(ctx);
        if (EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_Cipher(ctx, plain, cipher, 16) != 16)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_cleanup(ctx);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif
    }  /* end evp_cipher test: EVP_aes_128_cbc*/
#endif /* WOLFSSL_AES_128 */
#endif /* HAVE_AES_CBC */

#ifndef WOLFSSL_NO_OPENSSL_AES_LOW_LEVEL_API

#if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_256)
    {  /* evp_cipher test: EVP_aes_256_ecb*/
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *ctx = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX ctx[1];
#endif
        WOLFSSL_SMALL_STACK_STATIC const byte msg[] =
        {
          0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
          0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
        {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
        {
          0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
          0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
          0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
          0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };


        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if (ctx == NULL)
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(ctx);
        if (EVP_CipherInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key, NULL, 1) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_Cipher(ctx, cipher, (byte*)msg, 16) != 16)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_init(ctx);
        if (EVP_CipherInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key, NULL, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_Cipher(ctx, plain, cipher, 16) != 16)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif
    }  /* end evp_cipher test */
#endif /* HAVE_AES_ECB && WOLFSSL_AES_256 */

#if defined(WOLFSSL_AES_DIRECT) && defined(WOLFSSL_AES_256)
    /* enable HAVE_AES_DECRYPT for AES_encrypt/decrypt */
    {
        /* Test: AES_encrypt/decrypt/set Key */
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        AES_KEY *enc = (AES_KEY *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
        #ifdef HAVE_AES_DECRYPT
        AES_KEY *dec = (AES_KEY *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
        #endif
#else
        AES_KEY enc[1];
        #ifdef HAVE_AES_DECRYPT
        AES_KEY dec[1];
        #endif
#endif

        WOLFSSL_SMALL_STACK_STATIC const byte msg[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
        {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        byte plain[sizeof(msg)];
        byte cipher[sizeof(msg)];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if (enc == NULL)
            return MEMORY_E;
        #ifdef HAVE_AES_DECRYPT
        if (dec == NULL)
            return MEMORY_E;
        #endif
#endif

        AES_set_encrypt_key(key, sizeof(key)*8, enc);
        AES_set_decrypt_key(key,  sizeof(key)*8, dec);

        AES_encrypt(msg, cipher, enc);

        #ifdef HAVE_AES_DECRYPT
        AES_decrypt(cipher, plain, dec);
        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            return WC_TEST_RET_ENC_NC;
        #endif /* HAVE_AES_DECRYPT */

        if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
            return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
        #ifdef HAVE_AES_DECRYPT
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
        #endif
#endif
    }
#endif /* WOLFSSL_AES_DIRECT && WOLFSSL_AES_256 */

#endif /* !WOLFSSL_NO_OPENSSL_AES_LOW_LEVEL_API */

/* EVP_Cipher with EVP_aes_xxx_ctr() */
#ifdef WOLFSSL_AES_COUNTER
    {
        byte plainBuff [64];
        byte cipherBuff[64];

#ifdef WOLFSSL_AES_128
        WOLFSSL_SMALL_STACK_STATIC const byte ctrKey[] =
        {
            0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
            0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctrIv[] =
        {
            0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
            0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
        };


        WOLFSSL_SMALL_STACK_STATIC const byte ctrPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctrCipher[] =
        {
            0x87,0x4d,0x61,0x91,0xb6,0x20,0xe3,0x26,
            0x1b,0xef,0x68,0x64,0x99,0x0d,0xb6,0xce,
            0x98,0x06,0xf6,0x6b,0x79,0x70,0xfd,0xff,
            0x86,0x17,0x18,0x7b,0xb9,0xff,0xfd,0xff,
            0x5a,0xe4,0xdf,0x3e,0xdb,0xd5,0xd3,0x5e,
            0x5b,0x4f,0x09,0x02,0x0d,0xb0,0x3e,0xab,
            0x1e,0x03,0x1d,0xda,0x2f,0xbe,0x03,0xd1,
            0x79,0x21,0x70,0xa0,0xf3,0x00,0x9c,0xee
        };

        WOLFSSL_SMALL_STACK_STATIC const byte oddCipher[] =
        {
            0xb9,0xd7,0xcb,0x08,0xb0,0xe1,0x7b,0xa0,
            0xc2
        };
#endif

        /* test vector from "Recommendation for Block Cipher Modes of Operation"
         * NIST Special Publication 800-38A */
#ifdef WOLFSSL_AES_192
        WOLFSSL_SMALL_STACK_STATIC const byte ctr192Key[] =
        {
            0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
            0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
            0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctr192Iv[] =
        {
            0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
            0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
        };


        WOLFSSL_SMALL_STACK_STATIC const byte ctr192Plain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctr192Cipher[] =
        {
            0x1a,0xbc,0x93,0x24,0x17,0x52,0x1c,0xa2,
            0x4f,0x2b,0x04,0x59,0xfe,0x7e,0x6e,0x0b
        };
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        /* test vector from "Recommendation for Block Cipher Modes of Operation"
         * NIST Special Publication 800-38A */
        WOLFSSL_SMALL_STACK_STATIC const byte ctr256Key[] =
        {
            0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
            0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
            0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
            0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctr256Iv[] =
        {
            0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
            0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
        };


        WOLFSSL_SMALL_STACK_STATIC const byte ctr256Plain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };

        WOLFSSL_SMALL_STACK_STATIC const byte ctr256Cipher[] =
        {
            0x60,0x1e,0xc3,0x13,0x77,0x57,0x89,0xa5,
            0xb7,0xa7,0xf5,0x04,0xbb,0xf3,0xd2,0x28
        };
#endif /* WOLFSSL_AES_256 */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
        EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX en[1];
        EVP_CIPHER_CTX de[1];
#endif
#ifdef WOLFSSL_AES_128

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *p_en;
        EVP_CIPHER_CTX *p_de;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if ((en == NULL) || (de == NULL))
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_ctr(),
                (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain,
                AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_ctr(),
                (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
                AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(cipherBuff, ctrCipher, AES_BLOCK_SIZE*4))
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(plainBuff, ctrPlain, AES_BLOCK_SIZE*4))
            return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        p_en = wolfSSL_EVP_CIPHER_CTX_new();
        if (p_en == NULL)
            return WC_TEST_RET_ENC_ERRNO;
        p_de = wolfSSL_EVP_CIPHER_CTX_new();
        if (p_de == NULL)
            return WC_TEST_RET_ENC_ERRNO;

        if (EVP_CipherInit(p_en, EVP_aes_128_ctr(),
                (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_Cipher(p_en, (byte*)cipherBuff, (byte*)ctrPlain,
                AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherInit(p_de, EVP_aes_128_ctr(),
                (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_Cipher(p_de, (byte*)plainBuff, (byte*)cipherBuff,
                AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;

        wolfSSL_EVP_CIPHER_CTX_free(p_en);
        wolfSSL_EVP_CIPHER_CTX_free(p_de);
#endif /* WOLFSSL_SMALL_STACK && !WOLFSSL_NO_MALLOC */

        if (XMEMCMP(cipherBuff, ctrCipher, AES_BLOCK_SIZE*4))
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(plainBuff, ctrPlain, AES_BLOCK_SIZE*4))
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain, 9) != 9)
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff, 9) != 9)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plainBuff, ctrPlain, 9))
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(cipherBuff, ctrCipher, 9))
            return WC_TEST_RET_ENC_NC;

        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain, 9) != 9)
            return WC_TEST_RET_ENC_NC;
        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff, 9) != 9)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plainBuff, ctrPlain, 9))
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(cipherBuff, oddCipher, 9))
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_192_ctr(),
                (unsigned char*)ctr192Key, (unsigned char*)ctr192Iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctr192Plain,
                AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_192_ctr(),
            (unsigned char*)ctr192Key, (unsigned char*)ctr192Iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        XMEMSET(plainBuff, 0, sizeof(plainBuff));
        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
                AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plainBuff, ctr192Plain, sizeof(ctr192Plain)))
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(ctr192Cipher, cipherBuff, sizeof(ctr192Cipher)))
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_256_ctr(),
            (unsigned char*)ctr256Key, (unsigned char*)ctr256Iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctr256Plain,
                AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_256_ctr(),
            (unsigned char*)ctr256Key, (unsigned char*)ctr256Iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        XMEMSET(plainBuff, 0, sizeof(plainBuff));
        if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
                AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plainBuff, ctr256Plain, sizeof(ctr256Plain)))
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(ctr256Cipher, cipherBuff, sizeof(ctr256Cipher)))
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        wolfSSL_EVP_CIPHER_CTX_free(en);
        wolfSSL_EVP_CIPHER_CTX_free(de);
#endif

#endif /* WOLFSSL_AES_256 */
    }
#endif /* HAVE_AES_COUNTER */

#ifndef WOLFSSL_NO_OPENSSL_AES_LOW_LEVEL_API

#if defined(WOLFSSL_AES_CFB) && defined(WOLFSSL_AES_128)
    {
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        AES_KEY *enc = (AES_KEY *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
        AES_KEY *dec = (AES_KEY *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#else
        AES_KEY enc[1];
        AES_KEY dec[1];
#endif

        WOLFSSL_SMALL_STACK_STATIC const byte setIv[] = {
            0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,
            0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f
        };

        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
        {
            0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
            0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
        };

        WOLFSSL_SMALL_STACK_STATIC const byte cipher1[] =
        {
            0x3b,0x3f,0xd9,0x2e,0xb7,0x2d,0xad,0x20,
            0x33,0x34,0x49,0xf8,0xe8,0x3c,0xfb,0x4a,
            0xc8,0xa6,0x45,0x37,0xa0,0xb3,0xa9,0x3f,
            0xcd,0xe3,0xcd,0xad,0x9f,0x1c,0xe5,0x8b
        };

        WOLFSSL_SMALL_STACK_STATIC const byte msg[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51
        };

        byte cipher[AES_BLOCK_SIZE * 2];
        byte iv[AES_BLOCK_SIZE]; /* iv buffer is updeated by API */
        int  num = 0;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if ((enc == NULL) || (dec == NULL))
            return MEMORY_E;
#endif

        XMEMCPY(iv, setIv, sizeof(setIv));
        wolfSSL_AES_set_encrypt_key(key, sizeof(key) * 8, enc);
        wolfSSL_AES_set_encrypt_key(key, sizeof(key) * 8, dec);

        wolfSSL_AES_cfb128_encrypt(msg, cipher, AES_BLOCK_SIZE - 1, enc, iv,
                &num, AES_ENCRYPT);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE - 1))
            return WC_TEST_RET_ENC_NC;

        if (num != 15) /* should have used 15 of the 16 bytes */
            return WC_TEST_RET_ENC_NC;

        wolfSSL_AES_cfb128_encrypt(msg + AES_BLOCK_SIZE - 1,
                cipher + AES_BLOCK_SIZE - 1, AES_BLOCK_SIZE + 1, enc, iv,
                &num, AES_ENCRYPT);

        if (XMEMCMP(cipher, cipher1, AES_BLOCK_SIZE * 2))
            return WC_TEST_RET_ENC_NC;

        if (num != 0)
            return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
        XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
#endif
    }
#endif /* WOLFSSL_AES_CFB && WOLFSSL_AES_128 */

#endif /* !WOLFSSL_NO_OPENSSL_AES_LOW_LEVEL_API */

    return 0;
}


#endif /* !defined(NO_AES) && !defined(WOLFCRYPT_ONLY) */

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t openssl_test(void)
{
    wc_test_ret_t ret;
    EVP_MD_CTX md_ctx;
    testVector a, b, c, d, e, f;
    byte       hash[WC_SHA256_DIGEST_SIZE*2];  /* max size */

    a.inLen = 0;
    b.inLen = c.inLen = d.inLen = e.inLen = f.inLen = a.inLen;

    (void)a;
    (void)b;
    (void)c;
    (void)d;
    (void)e;
    (void)f;

    /* test malloc / free , 10 is an arbitrary amount of memory chosen */
    {
        byte* p;

        p = (byte*)CRYPTO_malloc(10, "", 0);

        if (p == NULL) {
            return WC_TEST_RET_ENC_NC;
        }
        XMEMSET(p, 0, 10);
        CRYPTO_free(p, "", 0);
    }

#ifndef NO_MD5
    a.input  = "1234567890123456789012345678901234567890123456789012345678"
               "9012345678901234567890";
    a.output = "\x57\xed\xf4\xa2\x2b\xe3\xc9\x55\xac\x49\xda\x2e\x21\x07\xb6"
               "\x7a";
    a.inLen  = XSTRLEN(a.input);
    a.outLen = WC_MD5_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_md5());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, a.input, (unsigned long)a.inLen);
    }
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS)
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(hash, a.output, WC_MD5_DIGEST_SIZE) != 0)
        return WC_TEST_RET_ENC_NC;
#endif /* NO_MD5 */

#ifndef NO_SHA
    b.input  = "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa"
               "aaaaaaaaaa";
    b.output = "\xAD\x5B\x3F\xDB\xCB\x52\x67\x78\xC2\x83\x9D\x2F\x15\x1E\xA7"
               "\x53\x99\x5E\x26\xA0";
    b.inLen  = XSTRLEN(b.input);
    b.outLen = WC_SHA_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha1());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, b.input, (unsigned long)b.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS)
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(hash, b.output, b.outLen) != 0)
        return WC_TEST_RET_ENC_NC;
#endif /* NO_SHA */

#ifdef WOLFSSL_SHA224
    e.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    e.output = "\xc9\x7c\xa9\xa5\x59\x85\x0c\xe9\x7a\x04\xa9\x6d\xef\x6d\x99"
               "\xa9\xe0\xe0\xe2\xab\x14\xe6\xb8\xdf\x26\x5f\xc0\xb3";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA224_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha224());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, e.input, (unsigned long)e.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, e.output, e.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* WOLFSSL_SHA224 */

#ifndef NO_SHA256
    d.input  = "abcdbcdecdefdefgefghfghighijhijkijkljklmklmnlmnomnopnopq";
    d.output = "\x24\x8D\x6A\x61\xD2\x06\x38\xB8\xE5\xC0\x26\x93\x0C\x3E\x60"
               "\x39\xA3\x3C\xE4\x59\x64\xFF\x21\x67\xF6\xEC\xED\xD4\x19\xDB"
               "\x06\xC1";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA256_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha256());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, d.input, (unsigned long)d.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, d.output, d.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* !NO_SHA256 */

#ifdef WOLFSSL_SHA384
    e.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    e.output = "\x09\x33\x0c\x33\xf7\x11\x47\xe8\x3d\x19\x2f\xc7\x82\xcd\x1b"
               "\x47\x53\x11\x1b\x17\x3b\x3b\x05\xd2\x2f\xa0\x80\x86\xe3\xb0"
               "\xf7\x12\xfc\xc7\xc7\x1a\x55\x7e\x2d\xb9\x66\xc3\xe9\xfa\x91"
               "\x74\x60\x39";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA384_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha384());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, e.input, (unsigned long)e.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, e.output, e.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* WOLFSSL_SHA384 */

#ifdef WOLFSSL_SHA512
    f.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    f.output = "\x8e\x95\x9b\x75\xda\xe3\x13\xda\x8c\xf4\xf7\x28\x14\xfc\x14"
               "\x3f\x8f\x77\x79\xc6\xeb\x9f\x7f\xa1\x72\x99\xae\xad\xb6\x88"
               "\x90\x18\x50\x1d\x28\x9e\x49\x00\xf7\xe4\x33\x1b\x99\xde\xc4"
               "\xb5\x43\x3a\xc7\xd3\x29\xee\xb6\xdd\x26\x54\x5e\x96\xe5\x5b"
               "\x87\x4b\xe9\x09";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_SHA512_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha512());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, f.input, (unsigned long)f.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, f.output, f.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }

#if !defined(WOLFSSL_NOSHA512_224) && \
   (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)

    f.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    f.output = "\x23\xfe\xc5\xbb\x94\xd6\x0b\x23\x30\x81\x92\x64\x0b\x0c\x45"
               "\x33\x35\xd6\x64\x73\x4f\xe4\x0e\x72\x68\x67\x4a\xf9";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_SHA512_224_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha512_224());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, f.input, (unsigned long)f.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, f.output, f.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* !WOLFSSL_NOSHA512_224 && !FIPS ... */

#if !defined(WOLFSSL_NOSHA512_256) && \
   (!defined(HAVE_FIPS) || FIPS_VERSION_GE(5, 3)) && !defined(HAVE_SELFTEST)
    f.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    f.output = "\x39\x28\xe1\x84\xfb\x86\x90\xf8\x40\xda\x39\x88\x12\x1d\x31"
               "\xbe\x65\xcb\x9d\x3e\xf8\x3e\xe6\x14\x6f\xea\xc8\x61\xe1\x9b"
               "\x56\x3a";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_SHA512_256_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha512_256());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, f.input, (unsigned long)f.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, f.output, f.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* !WOLFSSL_NOSHA512_224 && !FIPS ... */
#endif /* WOLFSSL_SHA512 */

#ifdef WOLFSSL_SHA3
#ifndef WOLFSSL_NOSHA3_224
    e.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    e.output = "\x54\x3e\x68\x68\xe1\x66\x6c\x1a\x64\x36\x30\xdf\x77\x36\x7a"
               "\xe5\xa6\x2a\x85\x07\x0a\x51\xc1\x4c\xbf\x66\x5c\xbc";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA3_224_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha3_224());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, e.input, (unsigned long)e.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, e.output, e.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* WOLFSSL_NOSHA3_224 */

#ifndef WOLFSSL_NOSHA3_256
    d.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    d.output = "\x91\x6f\x60\x61\xfe\x87\x97\x41\xca\x64\x69\xb4\x39\x71\xdf"
               "\xdb\x28\xb1\xa3\x2d\xc3\x6c\xb3\x25\x4e\x81\x2b\xe2\x7a\xad"
               "\x1d\x18";
    d.inLen  = XSTRLEN(d.input);
    d.outLen = WC_SHA3_256_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha3_256());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, d.input, (unsigned long)d.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, d.output, d.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* WOLFSSL_NOSHA3_256 */

    e.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    e.output = "\x79\x40\x7d\x3b\x59\x16\xb5\x9c\x3e\x30\xb0\x98\x22\x97\x47"
               "\x91\xc3\x13\xfb\x9e\xcc\x84\x9e\x40\x6f\x23\x59\x2d\x04\xf6"
               "\x25\xdc\x8c\x70\x9b\x98\xb4\x3b\x38\x52\xb3\x37\x21\x61\x79"
               "\xaa\x7f\xc7";
    e.inLen  = XSTRLEN(e.input);
    e.outLen = WC_SHA3_384_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha3_384());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, e.input, (unsigned long)e.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS || XMEMCMP(hash, e.output, e.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }

#ifndef WOLFSSL_NOSHA3_512
    f.input  = "abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhi"
               "jklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu";
    f.output = "\xaf\xeb\xb2\xef\x54\x2e\x65\x79\xc5\x0c\xad\x06\xd2\xe5\x78"
               "\xf9\xf8\xdd\x68\x81\xd7\xdc\x82\x4d\x26\x36\x0f\xee\xbf\x18"
               "\xa4\xfa\x73\xe3\x26\x11\x22\x94\x8e\xfc\xfd\x49\x2e\x74\xe8"
               "\x2e\x21\x89\xed\x0f\xb4\x40\xd1\x87\xf3\x82\x27\x0c\xb4\x55"
               "\xf2\x1d\xd1\x85";
    f.inLen  = XSTRLEN(f.input);
    f.outLen = WC_SHA3_512_DIGEST_SIZE;

    EVP_MD_CTX_init(&md_ctx);
    ret = EVP_DigestInit(&md_ctx, EVP_sha3_512());
    if (ret == WOLFSSL_SUCCESS) {
        ret = EVP_DigestUpdate(&md_ctx, f.input, (unsigned long)f.inLen);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_DigestFinal(&md_ctx, hash, 0);
    }
    EVP_MD_CTX_cleanup(&md_ctx);
    if (ret != WOLFSSL_SUCCESS ||
            XMEMCMP(hash, f.output, f.outLen) != 0) {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* WOLFSSL_NOSHA3_512 */
#endif /* WOLFSSL_SHA3 */

#ifndef WC_NO_RNG
    if (RAND_bytes(hash, sizeof(hash)) != WOLFSSL_SUCCESS)
        return WC_TEST_RET_ENC_NC;
#endif

#ifndef NO_MD5
    c.input  = "what do ya want for nothing?";
    c.output = "\x55\x78\xe8\x48\x4b\xcc\x93\x80\x93\xec\x53\xaf\x22\xd6\x14"
               "\x76";
    c.inLen  = XSTRLEN(c.input);
    c.outLen = WC_MD5_DIGEST_SIZE;

#if defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)
    /* Expect failure with MD5 + HMAC when using FIPS 140-3. */
    if (HMAC(EVP_md5(), "JefeJefeJefeJefe", 16, (byte*)c.input, (int)c.inLen,
            hash, 0) != NULL)
#else
    if (HMAC(EVP_md5(), "JefeJefeJefeJefe", 16, (byte*)c.input, (int)c.inLen,
            hash, 0) == NULL ||
        XMEMCMP(hash, c.output, c.outLen) != 0)
#endif
    {
        return WC_TEST_RET_ENC_NC;
    }
#endif /* NO_MD5 */

#ifndef NO_DES3
    { /* des test */
    WOLFSSL_SMALL_STACK_STATIC const byte vector[] = { /* "now is the time for all " w/o trailing 0 */
        0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
        0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
        0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
    };
    byte plain[24];
    byte cipher[24];
    const_DES_cblock key = {
        0x01,0x23,0x45,0x67,0x89,0xab,0xcd,0xef
    };
    DES_cblock iv = {
        0x12,0x34,0x56,0x78,0x90,0xab,0xcd,0xef
    };
    DES_key_schedule sched;
    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0x8b,0x7c,0x52,0xb0,0x01,0x2b,0x6c,0xb8,
        0x4f,0x0f,0xeb,0xf3,0xfb,0x5f,0x86,0x73,
        0x15,0x85,0xb3,0x22,0x4b,0x86,0x2b,0x4b
    };

    DES_key_sched(&key, &sched);

    DES_cbc_encrypt(vector, cipher, sizeof(vector), &sched, &iv, DES_ENCRYPT);
    DES_cbc_encrypt(cipher, plain, sizeof(vector), &sched, &iv, DES_DECRYPT);

    if (XMEMCMP(plain, vector, sizeof(vector)) != 0)
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(cipher, verify, sizeof(verify)) != 0)
        return WC_TEST_RET_ENC_NC;

        /* test changing iv */
    DES_ncbc_encrypt(vector, cipher, 8, &sched, &iv, DES_ENCRYPT);
    DES_ncbc_encrypt(vector + 8, cipher + 8, 16, &sched, &iv, DES_ENCRYPT);

    if (XMEMCMP(cipher, verify, sizeof(verify)) != 0)
        return WC_TEST_RET_ENC_NC;

    }  /* end des test */
#endif /* NO_DES3 */

#if !defined(NO_AES) && !defined(WOLFCRYPT_ONLY)
    if ((ret = openssl_aes_test()) != 0) {
        return ret;
    }
#if defined(WOLFSSL_AES_128) && defined(HAVE_AES_CBC)
    {   /* evp_cipher test: EVP_aes_128_cbc */
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *ctx = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX ctx[1];
#endif
        int idx, cipherSz, plainSz;
        WOLFSSL_SMALL_STACK_STATIC const byte msg[] = { /* "Now is the time for all " w/o trailing 0 */
            0x6e,0x6f,0x77,0x20,0x69,0x73,0x20,0x74,
            0x68,0x65,0x20,0x74,0x69,0x6d,0x65,0x20,
            0x66,0x6f,0x72,0x20,0x61,0x6c,0x6c,0x20
        };
        WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
            0x95,0x94,0x92,0x57,0x5f,0x42,0x81,0x53,
            0x2c,0xcc,0x9d,0x46,0x77,0xa2,0x33,0xcb,
            0x3b,0x5d,0x41,0x97,0x94,0x25,0xa4,0xb4,
            0xae,0x7b,0x34,0xd0,0x3f,0x0c,0xbc,0x06
        };
        WOLFSSL_SMALL_STACK_STATIC const byte verify2[] = {
            0x95,0x94,0x92,0x57,0x5f,0x42,0x81,0x53,
            0x2c,0xcc,0x9d,0x46,0x77,0xa2,0x33,0xcb,
            0x7d,0x37,0x7b,0x0b,0x44,0xaa,0xb5,0xf0,
            0x5f,0x34,0xb4,0xde,0xb5,0xbd,0x2a,0xbb
        };
        WOLFSSL_SMALL_STACK_STATIC const byte key[] =
            "0123456789abcdef   ";  /* align */
        WOLFSSL_SMALL_STACK_STATIC const byte iv[]  =
            "1234567890abcdef   ";  /* align */
        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if (ctx == NULL)
            return MEMORY_E;
#endif

        cipherSz = 0;
        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 1);
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherUpdate(ctx, cipher, &idx, (byte*)msg, sizeof(msg));
            if (ret == WOLFSSL_SUCCESS)
                cipherSz += idx;
        }
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherFinal(ctx, cipher + cipherSz, &idx);
            if (ret == WOLFSSL_SUCCESS)
                cipherSz += idx;
        }
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (cipherSz != (int)sizeof(verify) || XMEMCMP(cipher, verify, cipherSz))
            return WC_TEST_RET_ENC_NC;

        /* check partial decrypt (not enough padding for full block) */
        plainSz = 0;
        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 0);
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherUpdate(ctx, plain, &idx, cipher, 1);
            if (ret == WOLFSSL_SUCCESS)
                plainSz += idx;
        }
        if (ret == WOLFSSL_SUCCESS) {
            /* this test should fail... not enough padding for full block */
            ret = EVP_CipherFinal(ctx, plain + plainSz, &idx);
            if (plainSz == 0 && ret != WOLFSSL_SUCCESS)
                ret = WOLFSSL_SUCCESS;
            else
                ret = WC_TEST_RET_ENC_NC;
        }
        else
            ret = WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != WOLFSSL_SUCCESS)
            return ret;

        plainSz = 0;
        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 0);
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherUpdate(ctx, plain, &idx, cipher, cipherSz);
            if (ret == WOLFSSL_SUCCESS)
                plainSz += idx;
        }
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherFinal(ctx, plain + plainSz, &idx);
            if (ret == WOLFSSL_SUCCESS)
                plainSz += idx;
        }
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (plainSz != (int)sizeof(msg) || XMEMCMP(plain, msg, sizeof(msg)))
            return WC_TEST_RET_ENC_NC;

        cipherSz = 0;
        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_128_cbc(), key, iv, 1);
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherUpdate(ctx, cipher, &idx, msg, AES_BLOCK_SIZE);
            if (ret == WOLFSSL_SUCCESS)
                cipherSz += idx;
        }
        if (ret == WOLFSSL_SUCCESS) {
            ret = EVP_CipherFinal(ctx, cipher + cipherSz, &idx);
            if (ret == WOLFSSL_SUCCESS)
                cipherSz += idx;
        }
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (cipherSz != (int)sizeof(verify2) || XMEMCMP(cipher, verify2, cipherSz))
            return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif
    }  /* end evp_cipher test: EVP_aes_128_cbc*/
#endif /* WOLFSSL_AES_128 && HAVE_AES_CBC */

#if defined(HAVE_AES_ECB) && defined(WOLFSSL_AES_256)
    {   /* evp_cipher test: EVP_aes_256_ecb*/
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *ctx = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX ctx[1];
#endif
        WOLFSSL_SMALL_STACK_STATIC const byte msg[] = {
          0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
          0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
        };
        WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
            0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
            0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
        };
        WOLFSSL_SMALL_STACK_STATIC const byte key[] = {
          0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
          0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
          0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
          0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
        };
        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if (ctx == NULL)
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key, NULL, 1);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_Cipher(ctx, cipher, (byte*)msg, 16);
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != 16)
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_init(ctx);
        ret = EVP_CipherInit(ctx, EVP_aes_256_ecb(), (unsigned char*)key, NULL, 0);
        if (ret == WOLFSSL_SUCCESS)
            ret = EVP_Cipher(ctx, plain, cipher, 16);
        EVP_CIPHER_CTX_cleanup(ctx);
        if (ret != 16)
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
            return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        wolfSSL_EVP_CIPHER_CTX_free(ctx);
#endif
    }  /* end evp_cipher test */
#endif /* HAVE_AES_ECB && WOLFSSL_AES_128 */

#ifndef WOLFSSL_NO_OPENSSL_AES_LOW_LEVEL_API

#if defined(WOLFSSL_AES_DIRECT) && defined(WOLFSSL_AES_256)
  /* enable HAVE_AES_DECRYPT for AES_encrypt/decrypt */
{

  /* Test: AES_encrypt/decrypt/set Key */
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
  AES_KEY *enc = (AES_KEY *)XMALLOC(sizeof *enc, HEAP_HINT, DYNAMIC_TYPE_AES);
  #ifdef HAVE_AES_DECRYPT
  AES_KEY *dec = (AES_KEY *)XMALLOC(sizeof *dec, HEAP_HINT, DYNAMIC_TYPE_AES);
  #endif
#else
  AES_KEY enc[1];
  #ifdef HAVE_AES_DECRYPT
  AES_KEY dec[1];
  #endif
#endif

  WOLFSSL_SMALL_STACK_STATIC const byte msg[] =
  {
      0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
      0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
  };

  WOLFSSL_SMALL_STACK_STATIC const byte verify[] =
  {
      0xf3,0xee,0xd1,0xbd,0xb5,0xd2,0xa0,0x3c,
      0x06,0x4b,0x5a,0x7e,0x3d,0xb1,0x81,0xf8
  };

  WOLFSSL_SMALL_STACK_STATIC const byte key[] =
  {
      0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
      0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
      0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
      0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
  };

  byte plain[sizeof(msg)];
  byte cipher[sizeof(msg)];

  printf("openSSL extra test\n") ;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
  if (enc == NULL)
      return MEMORY_E;
  #ifdef HAVE_AES_DECRYPT
  if (dec == NULL)
      return MEMORY_E;
  #endif
#endif

  AES_set_encrypt_key(key, sizeof(key)*8, enc);
  AES_set_decrypt_key(key,  sizeof(key)*8, dec);

  AES_encrypt(msg, cipher, enc);

#ifdef HAVE_AES_DECRYPT
  AES_decrypt(cipher, plain, dec);
  if (XMEMCMP(plain, msg, AES_BLOCK_SIZE))
      return WC_TEST_RET_ENC_NC;
#endif /* HAVE_AES_DECRYPT */

  if (XMEMCMP(cipher, verify, AES_BLOCK_SIZE))
      return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
  XFREE(enc, HEAP_HINT, DYNAMIC_TYPE_AES);
  #ifdef HAVE_AES_DECRYPT
  XFREE(dec, HEAP_HINT, DYNAMIC_TYPE_AES);
  #endif
#endif
}

#endif /* WOLFSSL_AES_DIRECT && WOLFSSL_AES_256 */

#endif /* !WOLFSSL_NO_OPENSSL_AES_LOW_LEVEL_API */

/* EVP_Cipher with EVP_aes_xxx_ctr() */
#ifdef WOLFSSL_AES_COUNTER
{
    byte plainBuff [64];
    byte cipherBuff[64];

#ifdef WOLFSSL_AES_128
    WOLFSSL_SMALL_STACK_STATIC const byte ctrKey[] =
    {
        0x2b,0x7e,0x15,0x16,0x28,0xae,0xd2,0xa6,
        0xab,0xf7,0x15,0x88,0x09,0xcf,0x4f,0x3c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctrIv[] =
    {
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctrPlain[] =
    {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
        0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
        0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
        0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
        0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
        0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
        0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctrCipher[] =
    {
        0x87,0x4d,0x61,0x91,0xb6,0x20,0xe3,0x26,
        0x1b,0xef,0x68,0x64,0x99,0x0d,0xb6,0xce,
        0x98,0x06,0xf6,0x6b,0x79,0x70,0xfd,0xff,
        0x86,0x17,0x18,0x7b,0xb9,0xff,0xfd,0xff,
        0x5a,0xe4,0xdf,0x3e,0xdb,0xd5,0xd3,0x5e,
        0x5b,0x4f,0x09,0x02,0x0d,0xb0,0x3e,0xab,
        0x1e,0x03,0x1d,0xda,0x2f,0xbe,0x03,0xd1,
        0x79,0x21,0x70,0xa0,0xf3,0x00,0x9c,0xee
    };

    WOLFSSL_SMALL_STACK_STATIC const byte oddCipher[] =
    {
        0xb9,0xd7,0xcb,0x08,0xb0,0xe1,0x7b,0xa0,
        0xc2
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    EVP_CIPHER_CTX *p_en;
    EVP_CIPHER_CTX *p_de;
#endif

#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
    /* test vector from "Recommendation for Block Cipher Modes of Operation"
     * NIST Special Publication 800-38A */
    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Key[] =
    {
        0x8e,0x73,0xb0,0xf7,0xda,0x0e,0x64,0x52,
        0xc8,0x10,0xf3,0x2b,0x80,0x90,0x79,0xe5,
        0x62,0xf8,0xea,0xd2,0x52,0x2c,0x6b,0x7b
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Iv[] =
    {
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    };


    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Plain[] =
    {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr192Cipher[] =
    {
        0x1a,0xbc,0x93,0x24,0x17,0x52,0x1c,0xa2,
        0x4f,0x2b,0x04,0x59,0xfe,0x7e,0x6e,0x0b
    };
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
    /* test vector from "Recommendation for Block Cipher Modes of Operation"
     * NIST Special Publication 800-38A */
    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Key[] =
    {
        0x60,0x3d,0xeb,0x10,0x15,0xca,0x71,0xbe,
        0x2b,0x73,0xae,0xf0,0x85,0x7d,0x77,0x81,
        0x1f,0x35,0x2c,0x07,0x3b,0x61,0x08,0xd7,
        0x2d,0x98,0x10,0xa3,0x09,0x14,0xdf,0xf4
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Iv[] =
    {
        0xf0,0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,
        0xf8,0xf9,0xfa,0xfb,0xfc,0xfd,0xfe,0xff
    };


    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Plain[] =
    {
        0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
        0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a
    };

    WOLFSSL_SMALL_STACK_STATIC const byte ctr256Cipher[] =
    {
        0x60,0x1e,0xc3,0x13,0x77,0x57,0x89,0xa5,
        0xb7,0xa7,0xf5,0x04,0xbb,0xf3,0xd2,0x28
    };
#endif /* WOLFSSL_AES_256 */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
    EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
    if ((en == NULL) || (de == NULL))
        return MEMORY_E;
#else
    EVP_CIPHER_CTX en[1];
    EVP_CIPHER_CTX de[1];
#endif

#ifdef WOLFSSL_AES_128

    EVP_CIPHER_CTX_init(en);
    if (EVP_CipherInit(en, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return WC_TEST_RET_ENC_NC;
    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain,
            AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
        return WC_TEST_RET_ENC_NC;
    EVP_CIPHER_CTX_init(de);
    if (EVP_CipherInit(de, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return WC_TEST_RET_ENC_NC;

    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
            AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(cipherBuff, ctrCipher, AES_BLOCK_SIZE*4))
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(plainBuff, ctrPlain, AES_BLOCK_SIZE*4))
        return WC_TEST_RET_ENC_NC;

    EVP_CIPHER_CTX_cleanup(en);
    EVP_CIPHER_CTX_cleanup(de);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    p_en = wolfSSL_EVP_CIPHER_CTX_new();
    if (p_en == NULL)
        return WC_TEST_RET_ENC_ERRNO;
    p_de = wolfSSL_EVP_CIPHER_CTX_new();
    if (p_de == NULL)
        return WC_TEST_RET_ENC_ERRNO;

    if (EVP_CipherInit(p_en, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return WC_TEST_RET_ENC_NC;
    if (EVP_Cipher(p_en, (byte*)cipherBuff, (byte*)ctrPlain,
            AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
        return WC_TEST_RET_ENC_NC;
    if (EVP_CipherInit(p_de, EVP_aes_128_ctr(),
            (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return WC_TEST_RET_ENC_NC;

    if (EVP_Cipher(p_de, (byte*)plainBuff, (byte*)cipherBuff,
            AES_BLOCK_SIZE*4) != AES_BLOCK_SIZE*4)
        return WC_TEST_RET_ENC_NC;

    EVP_CIPHER_CTX_cleanup(p_en);
    EVP_CIPHER_CTX_cleanup(p_de);

    wolfSSL_EVP_CIPHER_CTX_free(p_en);
    wolfSSL_EVP_CIPHER_CTX_free(p_de);
#endif /* WOLFSSL_SMALL_STACK && !WOLFSSL_NO_MALLOC */

    if (XMEMCMP(cipherBuff, ctrCipher, AES_BLOCK_SIZE*4))
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(plainBuff, ctrPlain, AES_BLOCK_SIZE*4))
        return WC_TEST_RET_ENC_NC;

    EVP_CIPHER_CTX_init(en);
    if (EVP_CipherInit(en, EVP_aes_128_ctr(),
        (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return WC_TEST_RET_ENC_NC;
    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain, 9) != 9)
        return WC_TEST_RET_ENC_NC;

    EVP_CIPHER_CTX_init(de);
    if (EVP_CipherInit(de, EVP_aes_128_ctr(),
        (unsigned char*)ctrKey, (unsigned char*)ctrIv, 0) == 0)
        return WC_TEST_RET_ENC_NC;

    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff, 9) != 9)
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(plainBuff, ctrPlain, 9))
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(cipherBuff, ctrCipher, 9))
        return WC_TEST_RET_ENC_NC;

    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctrPlain, 9) != 9)
        return WC_TEST_RET_ENC_NC;
    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff, 9) != 9)
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(plainBuff, ctrPlain, 9))
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(cipherBuff, oddCipher, 9))
        return WC_TEST_RET_ENC_NC;

    EVP_CIPHER_CTX_cleanup(en);
    EVP_CIPHER_CTX_cleanup(de);
#endif /* WOLFSSL_AES_128 */

#ifdef WOLFSSL_AES_192
    EVP_CIPHER_CTX_init(en);
    if (EVP_CipherInit(en, EVP_aes_192_ctr(),
            (unsigned char*)ctr192Key, (unsigned char*)ctr192Iv, 0) == 0)
        return WC_TEST_RET_ENC_NC;
    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctr192Plain,
            AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
        return WC_TEST_RET_ENC_NC;
    EVP_CIPHER_CTX_init(de);
    if (EVP_CipherInit(de, EVP_aes_192_ctr(),
        (unsigned char*)ctr192Key, (unsigned char*)ctr192Iv, 0) == 0)
        return WC_TEST_RET_ENC_NC;

    XMEMSET(plainBuff, 0, sizeof(plainBuff));
    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
            AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(plainBuff, ctr192Plain, sizeof(ctr192Plain)))
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(ctr192Cipher, cipherBuff, sizeof(ctr192Cipher)))
        return WC_TEST_RET_ENC_NC;

    EVP_CIPHER_CTX_cleanup(en);
    EVP_CIPHER_CTX_cleanup(de);
#endif /* WOLFSSL_AES_192 */

#ifdef WOLFSSL_AES_256
    EVP_CIPHER_CTX_init(en);
    if (EVP_CipherInit(en, EVP_aes_256_ctr(),
        (unsigned char*)ctr256Key, (unsigned char*)ctr256Iv, 0) == 0)
        return WC_TEST_RET_ENC_NC;
    if (EVP_Cipher(en, (byte*)cipherBuff, (byte*)ctr256Plain,
            AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
        return WC_TEST_RET_ENC_NC;
    EVP_CIPHER_CTX_init(de);
    if (EVP_CipherInit(de, EVP_aes_256_ctr(),
        (unsigned char*)ctr256Key, (unsigned char*)ctr256Iv, 0) == 0)
        return WC_TEST_RET_ENC_NC;

    XMEMSET(plainBuff, 0, sizeof(plainBuff));
    if (EVP_Cipher(de, (byte*)plainBuff, (byte*)cipherBuff,
            AES_BLOCK_SIZE) != AES_BLOCK_SIZE)
        return WC_TEST_RET_ENC_NC;

    if (XMEMCMP(plainBuff, ctr256Plain, sizeof(ctr256Plain)))
        return WC_TEST_RET_ENC_NC;
    if (XMEMCMP(ctr256Cipher, cipherBuff, sizeof(ctr256Cipher)))
        return WC_TEST_RET_ENC_NC;

    EVP_CIPHER_CTX_cleanup(en);
    EVP_CIPHER_CTX_cleanup(de);
#endif /* WOLFSSL_AES_256 */

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    wolfSSL_EVP_CIPHER_CTX_free(en);
    wolfSSL_EVP_CIPHER_CTX_free(de);
#endif
}
#endif /* HAVE_AES_COUNTER */

#if defined(HAVE_AES_CBC) && defined(WOLFSSL_AES_128)
{
      /* EVP_CipherUpdate test */


        WOLFSSL_SMALL_STACK_STATIC const byte cbcPlain[] =
        {
            0x6b,0xc1,0xbe,0xe2,0x2e,0x40,0x9f,0x96,
            0xe9,0x3d,0x7e,0x11,0x73,0x93,0x17,0x2a,
            0xae,0x2d,0x8a,0x57,0x1e,0x03,0xac,0x9c,
            0x9e,0xb7,0x6f,0xac,0x45,0xaf,0x8e,0x51,
            0x30,0xc8,0x1c,0x46,0xa3,0x5c,0xe4,0x11,
            0xe5,0xfb,0xc1,0x19,0x1a,0x0a,0x52,0xef,
            0xf6,0x9f,0x24,0x45,0xdf,0x4f,0x9b,0x17,
            0xad,0x2b,0x41,0x7b,0xe6,0x6c,0x37,0x10
        };

        byte key[] = "0123456789abcdef   ";  /* align */
        byte iv[]  = "1234567890abcdef   ";  /* align */

        byte cipher[AES_BLOCK_SIZE * 4];
        byte plain [AES_BLOCK_SIZE * 4];
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        EVP_CIPHER_CTX *en = wolfSSL_EVP_CIPHER_CTX_new();
        EVP_CIPHER_CTX *de = wolfSSL_EVP_CIPHER_CTX_new();
#else
        EVP_CIPHER_CTX en[1];
        EVP_CIPHER_CTX de[1];
#endif
        int outlen ;
        int total = 0;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        if ((en == NULL) || (de == NULL))
            return MEMORY_E;
#endif

        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return WC_TEST_RET_ENC_NC;
        /* openSSL compatibility, if(inlen == 0)return 1; */
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                                                    (byte*)cbcPlain, 0) != 1)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 1) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen,
                                                    (byte*)cbcPlain, 9) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherUpdate(en, (byte*)&cipher[total], &outlen,
                                                (byte*)&cbcPlain[9]  , 9) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;
        if(total != 32)
            return WC_TEST_RET_ENC_NC;

        total = 0;
        EVP_CIPHER_CTX_init(de);
        if (EVP_CipherInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher, 6) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                                                    (byte*)&cipher[6], 12) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 0)
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen,
                                                (byte*)&cipher[6+12], 14) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherFinal(de, (byte*)&plain[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 2)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if(total != 18)
            return WC_TEST_RET_ENC_NC;

        if (XMEMCMP(plain, cbcPlain, 18))
            return WC_TEST_RET_ENC_NC;

        total = 0;
        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(en);
        if (EVP_EncryptInit(en, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv) == 0)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CipherUpdate(en, (byte*)cipher, &outlen, (byte*)cbcPlain, 9) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherUpdate(en, (byte*)&cipher[total], &outlen, (byte*)&cbcPlain[9]  , 9) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_EncryptFinal(en, (byte*)&cipher[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;
        if(total != 32)
            return 3438;

        total = 0;
        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(de);
        if (EVP_DecryptInit(de, EVP_aes_128_cbc(),
            (unsigned char*)key, (unsigned char*)iv) == 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CipherUpdate(de, (byte*)plain, &outlen, (byte*)cipher, 6) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 0)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen, (byte*)&cipher[6], 12) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 0)
        total += outlen;

        if (EVP_CipherUpdate(de, (byte*)&plain[total], &outlen, (byte*)&cipher[6+12], 14) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 16)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if (EVP_DecryptFinal(de, (byte*)&plain[total], &outlen) == 0)
            return WC_TEST_RET_ENC_NC;
        if(outlen != 2)
            return WC_TEST_RET_ENC_NC;
        total += outlen;

        if(total != 18)
            return 3447;

        if (XMEMCMP(plain, cbcPlain, 18))
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_key_length(NULL) != 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_key_length(EVP_aes_128_cbc()) != 16)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_CTX_mode(NULL) != 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_CTX_mode(en) != (en->flags & WOLFSSL_EVP_CIPH_MODE))
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(en);
        if (EVP_CipherInit_ex(en, EVP_aes_128_cbc(), NULL,
            (unsigned char*)key, (unsigned char*)iv, 0) == 0)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(en);
        if (EVP_EncryptInit_ex(en, EVP_aes_128_cbc(), NULL,
                (unsigned char*)key, (unsigned char*)iv) == 0)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_EncryptFinal_ex(NULL, NULL, NULL) != WOLFSSL_FAILURE)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_EncryptFinal(NULL, NULL, NULL) != WOLFSSL_FAILURE)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(de);
        if (EVP_DecryptInit_ex(de, EVP_aes_128_cbc(), NULL,
                (unsigned char*)key, (unsigned char*)iv) == 0)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_DecryptFinal(NULL, NULL, NULL) != WOLFSSL_FAILURE)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_DecryptFinal_ex(NULL, NULL, NULL) != WOLFSSL_FAILURE)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_CTX_block_size(NULL) != BAD_FUNC_ARG)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        EVP_CIPHER_CTX_init(en);
        EVP_EncryptInit_ex(en, EVP_aes_128_cbc(), NULL,
                (unsigned char*)key, (unsigned char*)iv);
        if (EVP_CIPHER_CTX_block_size(en) != en->block_size)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_block_size(NULL) != BAD_FUNC_ARG)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_block_size(EVP_aes_128_cbc()) != AES_BLOCK_SIZE)
            return WC_TEST_RET_ENC_NC;

        if (WOLFSSL_EVP_CIPHER_mode(NULL) != 0)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_flags(EVP_aes_128_cbc()) != WOLFSSL_EVP_CIPH_CBC_MODE)
            return WC_TEST_RET_ENC_NC;

        EVP_CIPHER_CTX_clear_flags(en, 0xFFFFFFFF);
        EVP_CIPHER_CTX_set_flags(en, 42);
        if (en->flags != 42)
            return WC_TEST_RET_ENC_NC;

        if (EVP_CIPHER_CTX_set_padding(NULL, 0) != BAD_FUNC_ARG)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CIPHER_CTX_set_padding(en, 0) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (EVP_CIPHER_CTX_set_padding(en, 1) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;

        if (wolfSSL_EVP_CIPHER_CTX_cleanup(en) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;
        if (wolfSSL_EVP_CIPHER_CTX_cleanup(de) != WOLFSSL_SUCCESS)
            return WC_TEST_RET_ENC_NC;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
        wolfSSL_EVP_CIPHER_CTX_free(en);
        wolfSSL_EVP_CIPHER_CTX_free(de);
#endif
    }
#endif /* WOLFSSL_AES_128 && HAVE_AES_CBC */
#endif /* ifndef NO_AES */
    return 0;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t openSSL_evpMD_test(void)
{
    wc_test_ret_t ret = 0;
#if !defined(NO_SHA256) && !defined(NO_SHA)
    WOLFSSL_EVP_MD_CTX* ctx;
    WOLFSSL_EVP_MD_CTX* ctx2;

    ctx = EVP_MD_CTX_create();
    ctx2 = EVP_MD_CTX_create();

    ret = EVP_DigestInit(ctx, EVP_sha256());
    if (ret != SSL_SUCCESS) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    ret = EVP_MD_CTX_copy(ctx2, ctx);
    if (ret != SSL_SUCCESS) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_MD_type(EVP_sha256()) != EVP_MD_CTX_type(ctx2)) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    ret = EVP_DigestInit(ctx, EVP_sha1());
    if (ret != SSL_SUCCESS) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_MD_type(EVP_sha256()) != EVP_MD_CTX_type(ctx2)) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    ret = EVP_MD_CTX_copy_ex(ctx2, ctx);
    if (ret != SSL_SUCCESS) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_MD_type(EVP_sha256()) == EVP_MD_CTX_type(ctx2)) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_MD_type(EVP_sha1()) != EVP_MD_CTX_type(ctx2)) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_DigestInit_ex(ctx, EVP_sha1(), NULL) != SSL_SUCCESS) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    if (EVP_add_digest(NULL) != 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    if (wolfSSL_EVP_add_cipher(NULL) != 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto openSSL_evpMD_test_done;
    }

    ret = 0; /* got to success state without jumping to end with a fail */

openSSL_evpMD_test_done:
    EVP_MD_CTX_destroy(ctx);
    EVP_MD_CTX_destroy(ctx2);
#endif /* NO_SHA256 */

    return ret;
}

#ifdef DEBUG_SIGN
static void show(const char *title, const char *p, unsigned int s) {
    char* i;
    printf("%s: ", title);
    for (i = p;
            i < p + s;
            printf("%c", *i), i++);
    printf("\n");
}
#else
#define show(a,b,c) WC_DO_NOTHING
#endif

#define FOURK_BUFF 4096

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t openssl_pkey0_test(void)
{
    wc_test_ret_t ret = 0;
#if !defined(NO_RSA) && !defined(HAVE_USER_RSA) && !defined(NO_SHA)
    byte*   prvTmp;
    byte*   pubTmp;
    int prvBytes;
    int pubBytes;
    RSA *prvRsa = NULL;
    RSA *pubRsa = NULL;
    EVP_PKEY *prvPkey = NULL;
    EVP_PKEY *pubPkey = NULL;
    EVP_PKEY_CTX *enc = NULL;
    EVP_PKEY_CTX *dec = NULL;

    byte   in[] = TEST_STRING;
    byte   out[256];
    size_t outlen;
    size_t keySz;
    byte   plain[256];
#if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    XFILE  keyFile;
    XFILE  keypubFile;
    char cliKey[]    = "./certs/client-key.der";
    char cliKeypub[] = "./certs/client-keyPub.der";

#endif

    prvTmp = (byte*)XMALLOC(FOURK_BUFF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (prvTmp == NULL)
        return WC_TEST_RET_ENC_ERRNO;
    pubTmp = (byte*)XMALLOC(FOURK_BUFF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pubTmp == NULL) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_NC;
    }

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(prvTmp, client_key_der_1024, sizeof_client_key_der_1024);
    prvBytes = sizeof_client_key_der_1024;
    XMEMCPY(pubTmp, client_keypub_der_1024, sizeof_client_keypub_der_1024);
    pubBytes = sizeof_client_keypub_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(prvTmp, client_key_der_2048, sizeof_client_key_der_2048);
    prvBytes = sizeof_client_key_der_2048;
    XMEMCPY(pubTmp, client_keypub_der_2048, sizeof_client_keypub_der_2048);
    pubBytes = sizeof_client_keypub_der_2048;
#else
    keyFile = XFOPEN(cliKey, "rb");
    if (!keyFile) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("can't open ./certs/client-key.der, "
                "Please run from wolfSSL home dir", WC_TEST_RET_ENC_ERRNO);
        return WC_TEST_RET_ENC_ERRNO;
    }
    prvBytes = (int)XFREAD(prvTmp, 1, (int)FOURK_BUFF, keyFile);
    XFCLOSE(keyFile);
    if (prvBytes == 0) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_ERRNO;
    }
    keypubFile = XFOPEN(cliKeypub, "rb");
    if (!keypubFile) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("can't open ./certs/client-cert.der, "
                "Please run from wolfSSL home dir", WC_TEST_RET_ENC_ERRNO);
        return WC_TEST_RET_ENC_ERRNO;
    }
    pubBytes = (int)XFREAD(pubTmp, 1, (int)FOURK_BUFF, keypubFile);
    XFCLOSE(keypubFile);
    if (pubBytes == 0) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_ERRNO;
    }
#endif /* USE_CERT_BUFFERS */

        prvRsa = wolfSSL_RSA_new();
        pubRsa = wolfSSL_RSA_new();
        if((prvRsa == NULL) || (pubRsa == NULL)){
            printf("error with RSA_new\n");
            ret = WC_TEST_RET_ENC_ERRNO;
            goto openssl_pkey0_test_done;
        }

        ret = wolfSSL_RSA_LoadDer_ex(prvRsa, prvTmp, prvBytes, WOLFSSL_RSA_LOAD_PRIVATE);
        if(ret != SSL_SUCCESS){
            printf("error with RSA_LoadDer_ex\n");
            ret = WC_TEST_RET_ENC_EC(ret);
            goto openssl_pkey0_test_done;
        }

        ret = wolfSSL_RSA_LoadDer_ex(pubRsa, pubTmp, pubBytes, WOLFSSL_RSA_LOAD_PUBLIC);
        if(ret != SSL_SUCCESS){
            printf("error with RSA_LoadDer_ex\n");
            ret = WC_TEST_RET_ENC_EC(ret);
            goto openssl_pkey0_test_done;
        }
        keySz = (size_t)RSA_size(pubRsa);

        prvPkey = wolfSSL_EVP_PKEY_new();
        pubPkey = wolfSSL_EVP_PKEY_new();
        if((prvPkey == NULL) || (pubPkey == NULL)){
            printf("error with PKEY_new\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }
        ret  = wolfSSL_EVP_PKEY_set1_RSA(prvPkey, prvRsa);
        ret += wolfSSL_EVP_PKEY_set1_RSA(pubPkey, pubRsa);
        if(ret != 2){
            printf("error with PKEY_set1_RSA\n");
            ret = WC_TEST_RET_ENC_I(ret);
            goto openssl_pkey0_test_done;
        }

        dec = EVP_PKEY_CTX_new(prvPkey, NULL);
        enc = EVP_PKEY_CTX_new(pubPkey, NULL);
        if((dec == NULL)||(enc==NULL)){
            printf("error with EVP_PKEY_CTX_new\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }

        ret = EVP_PKEY_decrypt_init(dec);
        if (ret != 1) {
            printf("error with decrypt init\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }
        ret = EVP_PKEY_encrypt_init(enc);
        if (ret != 1) {
            printf("error with encrypt init\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }
        XMEMSET(out, 0, sizeof(out));
        ret = EVP_PKEY_encrypt(enc, out, &outlen, in, sizeof(in));
        if (ret != 1) {
            printf("error encrypting msg\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }

        show("encrypted msg", out, outlen);

        XMEMSET(plain, 0, sizeof(plain));
        ret = EVP_PKEY_decrypt(dec, plain, &outlen, out, keySz);
        if (ret != 1) {
            printf("error decrypting msg\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }
        show("decrypted msg", plain, outlen);

        /* RSA_PKCS1_OAEP_PADDING test */
        ret = EVP_PKEY_decrypt_init(dec);
        if (ret != 1) {
            printf("error with decrypt init\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }
        ret = EVP_PKEY_encrypt_init(enc);
        if (ret != 1) {
            printf("error with encrypt init\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }

        if (EVP_PKEY_CTX_set_rsa_padding(dec, RSA_PKCS1_PADDING) <= 0) {
            printf("first set rsa padding error\n");
            ret = WC_TEST_RET_ENC_EC(ret);
            goto openssl_pkey0_test_done;
        }

#ifndef HAVE_FIPS
        if (EVP_PKEY_CTX_set_rsa_padding(dec, RSA_PKCS1_OAEP_PADDING) <= 0){
            printf("second set rsa padding error\n");
            ret = WC_TEST_RET_ENC_EC(ret);
            goto openssl_pkey0_test_done;
        }

        if (EVP_PKEY_CTX_set_rsa_padding(enc, RSA_PKCS1_OAEP_PADDING) <= 0) {
            printf("third set rsa padding error\n");
            ret = WC_TEST_RET_ENC_EC(ret);
            goto openssl_pkey0_test_done;
        }
#endif

        XMEMSET(out, 0, sizeof(out));
        ret = EVP_PKEY_encrypt(enc, out, &outlen, in, sizeof(in));
        if (ret != 1) {
            printf("error encrypting msg\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }

        show("encrypted msg", out, outlen);

        XMEMSET(plain, 0, sizeof(plain));
        ret = EVP_PKEY_decrypt(dec, plain, &outlen, out, keySz);
        if (ret != 1) {
            printf("error decrypting msg\n");
            ret = WC_TEST_RET_ENC_NC;
            goto openssl_pkey0_test_done;
        }

        show("decrypted msg", plain, outlen);

        ret = 0; /* made it to this point without error then set success */
openssl_pkey0_test_done:

        wolfSSL_RSA_free(prvRsa);
        wolfSSL_RSA_free(pubRsa);
        EVP_PKEY_free(pubPkey);
        EVP_PKEY_free(prvPkey);
        EVP_PKEY_CTX_free(dec);
        EVP_PKEY_CTX_free(enc);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif /* NO_RSA */

        return ret;
}


WOLFSSL_TEST_SUBROUTINE wc_test_ret_t openssl_pkey1_test(void)
{
    wc_test_ret_t ret = 0;
#if !defined(NO_FILESYSTEM) && !defined(NO_RSA) && !defined(HAVE_USER_RSA) && \
    !defined(NO_SHA)
    EVP_PKEY_CTX* dec = NULL;
    EVP_PKEY_CTX* enc = NULL;
    EVP_PKEY* pubKey  = NULL;
    EVP_PKEY* prvKey  = NULL;
    X509* x509 = NULL;

    WOLFSSL_SMALL_STACK_STATIC const unsigned char msg[] = "sugar slapped";
    const unsigned char* clikey;
    long cliKeySz;
    size_t outlen;
    int keyLenBits = 2048;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    unsigned char *tmp = (unsigned char *)XMALLOC(FOURK_BUF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    unsigned char *cipher = (unsigned char *)XMALLOC(RSA_TEST_BYTES, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    unsigned char *plain = (unsigned char *)XMALLOC(RSA_TEST_BYTES, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((tmp == NULL) ||
        (cipher == NULL) ||
        (plain == NULL)) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }
#else
    unsigned char tmp[FOURK_BUF];
    unsigned char cipher[RSA_TEST_BYTES];
    unsigned char plain[RSA_TEST_BYTES];
#endif

#if defined(USE_CERT_BUFFERS_1024)
    XMEMCPY(tmp, client_key_der_1024, sizeof_client_key_der_1024);
    cliKeySz = (long)sizeof_client_key_der_1024;

    x509 = wolfSSL_X509_load_certificate_buffer(client_cert_der_1024,
            sizeof_client_cert_der_1024, SSL_FILETYPE_ASN1);
    keyLenBits = 1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(tmp, client_key_der_2048, sizeof_client_key_der_2048);
    cliKeySz = (long)sizeof_client_key_der_2048;

    x509 = wolfSSL_X509_load_certificate_buffer(client_cert_der_2048,
            sizeof_client_cert_der_2048, SSL_FILETYPE_ASN1);
#elif defined(USE_CERT_BUFFERS_3072)
    XMEMCPY(tmp, client_key_der_3072, sizeof_client_key_der_3072);
    cliKeySz = (long)sizeof_client_key_der_3072;

    x509 = wolfSSL_X509_load_certificate_buffer(client_cert_der_3072,
            sizeof_client_cert_der_3072, SSL_FILETYPE_ASN1);
    keyLenBits = 3072;
#elif defined(USE_CERT_BUFFERS_4096)
    XMEMCPY(tmp, client_key_der_4096, sizeof_client_key_der_4096);
    cliKeySz = (long)sizeof_client_key_der_4096;

    x509 = wolfSSL_X509_load_certificate_buffer(client_cert_der_4096,
            sizeof_client_cert_der_4096, SSL_FILETYPE_ASN1);
    keyLenBits = 4096;
#else
    {
        XFILE f;

        f = XFOPEN(clientKey, "rb");

        if (!f) {
            err_sys("can't open ./certs/client-key.der, "
                    "Please run from wolfSSL home dir", WC_TEST_RET_ENC_ERRNO);
            ret = WC_TEST_RET_ENC_ERRNO;
            goto openssl_pkey1_test_done;
        }

        cliKeySz = (long)XFREAD(tmp, 1, FOURK_BUF, f);
        XFCLOSE(f);
        if (cliKeySz == 0)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, openssl_pkey1_test_done);
    }

    /* using existing wolfSSL api to get public and private key */
    x509 = wolfSSL_X509_load_certificate_file(clientCert, SSL_FILETYPE_ASN1);
#endif /* USE_CERT_BUFFERS */
    clikey = tmp;

    if ((prvKey = EVP_PKEY_new()) == NULL) {
        ret = WC_TEST_RET_ENC_ERRNO;
        goto openssl_pkey1_test_done;
    }
    EVP_PKEY_free(prvKey);
    prvKey = NULL;

    if (x509 == NULL) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

    pubKey = X509_get_pubkey(x509);
    if (pubKey == NULL) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

    prvKey = d2i_PrivateKey(EVP_PKEY_RSA, NULL, &clikey, cliKeySz);
    if (prvKey == NULL) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

    /* phase 2 API to create EVP_PKEY_CTX and encrypt/decrypt */
    if (EVP_PKEY_bits(prvKey) != keyLenBits) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_size(prvKey) != keyLenBits/8) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

    dec = EVP_PKEY_CTX_new(prvKey, NULL);
    enc = EVP_PKEY_CTX_new(pubKey, NULL);
    if (dec == NULL || enc == NULL) {
        ret = WC_TEST_RET_ENC_ERRNO;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_decrypt_init(dec) != 1) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_encrypt_init(enc) != 1) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_CTX_set_rsa_padding(dec, RSA_PKCS1_PADDING) <= 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

#ifndef HAVE_FIPS
    if (EVP_PKEY_CTX_set_rsa_padding(dec, RSA_PKCS1_OAEP_PADDING) <= 0){
        ret = WC_TEST_RET_ENC_EC(ret);
        goto openssl_pkey1_test_done;
    }

    if (EVP_PKEY_CTX_set_rsa_padding(enc, RSA_PKCS1_OAEP_PADDING) <= 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto openssl_pkey1_test_done;
    }
#endif

    XMEMSET(cipher, 0, RSA_TEST_BYTES);
    outlen = keyLenBits/8;
    if (EVP_PKEY_encrypt(enc, cipher, &outlen, msg, sizeof(msg)) < 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto openssl_pkey1_test_done;
    }

    XMEMSET(plain, 0, RSA_TEST_BYTES);
    if (EVP_PKEY_decrypt(dec, plain, &outlen, cipher, outlen) != 1) {
        ret = WC_TEST_RET_ENC_NC;
        goto openssl_pkey1_test_done;
    }

openssl_pkey1_test_done:
    if (pubKey != NULL) {
        EVP_PKEY_free(pubKey);
    }
    if (prvKey != NULL) {
        EVP_PKEY_free(prvKey);
    }
    if (dec != NULL) {
        EVP_PKEY_CTX_free(dec);
    }
    if (enc != NULL) {
        EVP_PKEY_CTX_free(enc);
    }
    if (x509 != NULL) {
        X509_free(x509);
    }

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (tmp != NULL)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (cipher != NULL)
        XFREE(cipher, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (plain != NULL)
        XFREE(plain, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
#endif

    return ret;
}


WOLFSSL_TEST_SUBROUTINE wc_test_ret_t openssl_evpSig_test(void)
{
#if !defined(NO_RSA) && !defined(NO_SHA) && !defined(HAVE_USER_RSA)
    byte*   prvTmp;
    byte*   pubTmp;
    int prvBytes;
    int pubBytes;
    RSA *prvRsa;
    RSA *pubRsa;
    EVP_PKEY *prvPkey;
    EVP_PKEY *pubPkey;

    EVP_MD_CTX* sign;
    EVP_MD_CTX* verf;
    char msg[] = "see spot run";
    unsigned char sig[256];
    unsigned int sigSz;
    const void* pt;
    unsigned int count;
    wc_test_ret_t ret, ret1, ret2;

    #if !defined(USE_CERT_BUFFERS_1024) && !defined(USE_CERT_BUFFERS_2048)
    XFILE   keyFile;
    XFILE   keypubFile;
    char cliKey[]    = "./certs/client-key.der";
    char cliKeypub[] = "./certs/client-keyPub.der";
    #endif

    prvTmp = (byte*)XMALLOC(FOURK_BUFF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (prvTmp == NULL)
        return WC_TEST_RET_ENC_ERRNO;
    pubTmp = (byte*)XMALLOC(FOURK_BUFF, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (pubTmp == NULL) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_NC;
    }

#ifdef USE_CERT_BUFFERS_1024
    XMEMCPY(prvTmp, client_key_der_1024, sizeof_client_key_der_1024);
    prvBytes = sizeof_client_key_der_1024;
    XMEMCPY(pubTmp, client_keypub_der_1024, sizeof_client_keypub_der_1024);
    pubBytes = sizeof_client_keypub_der_1024;
#elif defined(USE_CERT_BUFFERS_2048)
    XMEMCPY(prvTmp, client_key_der_2048, sizeof_client_key_der_2048);
    prvBytes = sizeof_client_key_der_2048;
    XMEMCPY(pubTmp, client_keypub_der_2048, sizeof_client_keypub_der_2048);
    pubBytes = sizeof_client_keypub_der_2048;
#else
    keyFile = XFOPEN(cliKey, "rb");
    if (!keyFile) {
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("can't open ./certs/client-key.der, "
                "Please run from wolfSSL home dir", WC_TEST_RET_ENC_ERRNO);
        return WC_TEST_RET_ENC_ERRNO;
    }
    prvBytes = (int)XFREAD(prvTmp, 1, (int)FOURK_BUFF, keyFile);
    XFCLOSE(keyFile);
    if (prvBytes == 0) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_ERRNO;
    }
    keypubFile = XFOPEN(cliKeypub, "rb");
    if (!keypubFile) {
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("can't open ./certs/client-cert.der, "
                "Please run from wolfSSL home dir", WC_TEST_RET_ENC_ERRNO);
        return WC_TEST_RET_ENC_ERRNO;
    }
    pubBytes = (int)XFREAD(pubTmp, 1, (int)FOURK_BUFF, keypubFile);
    XFCLOSE(keypubFile);
    if (pubBytes == 0) {
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        return WC_TEST_RET_ENC_ERRNO;
    }
    #endif /* USE_CERT_BUFFERS */

    prvRsa = wolfSSL_RSA_new();
    pubRsa = wolfSSL_RSA_new();
    if((prvRsa == NULL) || (pubRsa == NULL)){
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        err_sys("ERROR with RSA_new", WC_TEST_RET_ENC_NC);
        return WC_TEST_RET_ENC_NC;
    }

    ret1 = wolfSSL_RSA_LoadDer_ex(prvRsa, prvTmp, prvBytes, WOLFSSL_RSA_LOAD_PRIVATE);
    ret2 = wolfSSL_RSA_LoadDer_ex(pubRsa, pubTmp, pubBytes, WOLFSSL_RSA_LOAD_PUBLIC);
    if((ret1 != SSL_SUCCESS) || (ret2 != SSL_SUCCESS)){
      printf("error with RSA_LoadDer_ex\n");
      return WC_TEST_RET_ENC_NC;
    }

    prvPkey = wolfSSL_EVP_PKEY_new();
    pubPkey = wolfSSL_EVP_PKEY_new();
    if((prvPkey == NULL) || (pubPkey == NULL)){
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        printf("error with KEY_new\n");
        return WC_TEST_RET_ENC_NC;
    }
    ret1  = wolfSSL_EVP_PKEY_set1_RSA(prvPkey, prvRsa);
    ret2  = wolfSSL_EVP_PKEY_set1_RSA(pubPkey, pubRsa);
    if((ret1 != 1) || (ret2 != 1)){
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        printf("error with EVP_PKEY_set1_RSA\n");
        return WC_TEST_RET_ENC_NC;
    }

    /****************** sign and verify *******************/
    sign = EVP_MD_CTX_create();
    verf = EVP_MD_CTX_create();
    if((sign == NULL)||(verf == NULL)){
        printf("error with EVP_MD_CTX_create\n");
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        return WC_TEST_RET_ENC_NC;
    }

    ret = EVP_SignInit(sign, EVP_sha1());
    if (ret != SSL_SUCCESS){
        printf("error with EVP_SignInit\n");
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        return WC_TEST_RET_ENC_NC;
    }

    count = sizeof(msg);
    show("message = ", (char *)msg, count);

    /* sign */
    XMEMSET(sig, 0, sizeof(sig));
    pt = (const void*)msg;
    ret1 = EVP_SignUpdate(sign, pt, count);
    ret2 = EVP_SignFinal(sign, sig, &sigSz, prvPkey);
    if((ret1 != SSL_SUCCESS) || (ret2 != SSL_SUCCESS)){
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        printf("error with EVP_MD_CTX_create\n");
        return WC_TEST_RET_ENC_NC;
    }
    show("signature = ", (char *)sig, sigSz);

    /* verify */
    pt = (const void*)msg;
    ret1 = EVP_VerifyInit(verf, EVP_sha1());
    ret2 = EVP_VerifyUpdate(verf, pt, count);
    if((ret1 != SSL_SUCCESS) || (ret2 != SSL_SUCCESS)){
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        printf("error with EVP_Verify\n");
        return WC_TEST_RET_ENC_NC;
    }
    if (EVP_VerifyFinal(verf, sig, sigSz, pubPkey) != 1) {
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        printf("error with EVP_VerifyFinal\n");
        return WC_TEST_RET_ENC_NC;
    }

    /* expect fail without update */
    EVP_VerifyInit(verf, EVP_sha1());
    if (EVP_VerifyFinal(verf, sig, sigSz, pubPkey) == 1) {
        XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
        EVP_MD_CTX_destroy(sign);
        EVP_MD_CTX_destroy(verf);
        printf("EVP_VerifyInit without update not detected\n");
        return WC_TEST_RET_ENC_NC;
    }

    XFREE(pubTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    XFREE(prvTmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    EVP_MD_CTX_destroy(sign);
    EVP_MD_CTX_destroy(verf);

    wolfSSL_RSA_free(prvRsa);
    wolfSSL_RSA_free(pubRsa);
    EVP_PKEY_free(pubPkey);
    EVP_PKEY_free(prvPkey);

#endif /* NO_RSA */
    return 0;
}
#endif /* OPENSSL_EXTRA */


#ifndef NO_PWDBASED
#ifdef HAVE_SCRYPT
/* Test vectors taken from RFC 7914: scrypt PBKDF - Section 12. */
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t scrypt_test(void)
{
#ifdef HAVE_FIPS
    /* RFC 7914 test vector keys are too short for FIPS. */
#else
    wc_test_ret_t ret;
    byte  derived[64];

    WOLFSSL_SMALL_STACK_STATIC const byte verify1[] = {
        0x77, 0xd6, 0x57, 0x62, 0x38, 0x65, 0x7b, 0x20,
        0x3b, 0x19, 0xca, 0x42, 0xc1, 0x8a, 0x04, 0x97,
        0xf1, 0x6b, 0x48, 0x44, 0xe3, 0x07, 0x4a, 0xe8,
        0xdf, 0xdf, 0xfa, 0x3f, 0xed, 0xe2, 0x14, 0x42,
        0xfc, 0xd0, 0x06, 0x9d, 0xed, 0x09, 0x48, 0xf8,
        0x32, 0x6a, 0x75, 0x3a, 0x0f, 0xc8, 0x1f, 0x17,
        0xe8, 0xd3, 0xe0, 0xfb, 0x2e, 0x0d, 0x36, 0x28,
        0xcf, 0x35, 0xe2, 0x0c, 0x38, 0xd1, 0x89, 0x06
    };
#if !defined(BENCH_EMBEDDED)
    WOLFSSL_SMALL_STACK_STATIC const byte verify2[] = {
        0xfd, 0xba, 0xbe, 0x1c, 0x9d, 0x34, 0x72, 0x00,
        0x78, 0x56, 0xe7, 0x19, 0x0d, 0x01, 0xe9, 0xfe,
        0x7c, 0x6a, 0xd7, 0xcb, 0xc8, 0x23, 0x78, 0x30,
        0xe7, 0x73, 0x76, 0x63, 0x4b, 0x37, 0x31, 0x62,
        0x2e, 0xaf, 0x30, 0xd9, 0x2e, 0x22, 0xa3, 0x88,
        0x6f, 0xf1, 0x09, 0x27, 0x9d, 0x98, 0x30, 0xda,
        0xc7, 0x27, 0xaf, 0xb9, 0x4a, 0x83, 0xee, 0x6d,
        0x83, 0x60, 0xcb, 0xdf, 0xa2, 0xcc, 0x06, 0x40
    };
#endif
#if !defined(BENCH_EMBEDDED) && !defined(WOLFSSL_LINUXKM) && !defined(HAVE_INTEL_QA)
    WOLFSSL_SMALL_STACK_STATIC const byte verify3[] = {
        0x70, 0x23, 0xbd, 0xcb, 0x3a, 0xfd, 0x73, 0x48,
        0x46, 0x1c, 0x06, 0xcd, 0x81, 0xfd, 0x38, 0xeb,
        0xfd, 0xa8, 0xfb, 0xba, 0x90, 0x4f, 0x8e, 0x3e,
        0xa9, 0xb5, 0x43, 0xf6, 0x54, 0x5d, 0xa1, 0xf2,
        0xd5, 0x43, 0x29, 0x55, 0x61, 0x3f, 0x0f, 0xcf,
        0x62, 0xd4, 0x97, 0x05, 0x24, 0x2a, 0x9a, 0xf9,
        0xe6, 0x1e, 0x85, 0xdc, 0x0d, 0x65, 0x1e, 0x40,
        0xdf, 0xcf, 0x01, 0x7b, 0x45, 0x57, 0x58, 0x87
    };
#endif
#ifdef SCRYPT_TEST_ALL
    /* Test case is very slow.
     * Use for confirmation after code change or new platform.
     */
    WOLFSSL_SMALL_STACK_STATIC const byte verify4[] = {
        0x21, 0x01, 0xcb, 0x9b, 0x6a, 0x51, 0x1a, 0xae,
        0xad, 0xdb, 0xbe, 0x09, 0xcf, 0x70, 0xf8, 0x81,
        0xec, 0x56, 0x8d, 0x57, 0x4a, 0x2f, 0xfd, 0x4d,
        0xab, 0xe5, 0xee, 0x98, 0x20, 0xad, 0xaa, 0x47,
        0x8e, 0x56, 0xfd, 0x8f, 0x4b, 0xa5, 0xd0, 0x9f,
        0xfa, 0x1c, 0x6d, 0x92, 0x7c, 0x40, 0xf4, 0xc3,
        0x37, 0x30, 0x40, 0x49, 0xe8, 0xa9, 0x52, 0xfb,
        0xcb, 0xf4, 0x5c, 0x6f, 0xa7, 0x7a, 0x41, 0xa4
    };
#endif

    ret = wc_scrypt(derived, NULL, 0, NULL, 0, 4, 1, 1, sizeof(verify1));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(derived, verify1, sizeof(verify1)) != 0)
        return WC_TEST_RET_ENC_NC;

#if !defined(BENCH_EMBEDDED)
    ret = wc_scrypt(derived, (byte*)"password", 8, (byte*)"NaCl", 4, 10, 8, 16,
                    sizeof(verify2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(derived, verify2, sizeof(verify2)) != 0)
        return WC_TEST_RET_ENC_NC;
#endif

    /* Test case with parallel overflowing */
    ret = wc_scrypt(derived, (byte*)"password", 16, (byte*)"NaCl", 16, 2, 4, 8388608,
                    sizeof(verify1));
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    /* Don't run these test on embedded, since they use large mallocs */
#if !defined(BENCH_EMBEDDED) && !defined(WOLFSSL_LINUXKM) && !defined(HAVE_INTEL_QA)
    ret = wc_scrypt(derived, (byte*)"pleaseletmein", 13,
                    (byte*)"SodiumChloride", 14, 14, 8, 1, sizeof(verify3));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(derived, verify3, sizeof(verify3)) != 0)
        return WC_TEST_RET_ENC_NC;

#ifdef SCRYPT_TEST_ALL
    ret = wc_scrypt(derived, (byte*)"pleaseletmein", 13,
                    (byte*)"SodiumChloride", 14, 20, 8, 1, sizeof(verify4));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(derived, verify4, sizeof(verify4)) != 0)
        return WC_TEST_RET_ENC_NC;
#endif
#endif /* !BENCH_EMBEDDED && !defined(WOLFSSL_LINUXKM) && !HAVE_INTEL_QA */

#if !defined(BENCH_EMBEDDED)
    ret = wc_scrypt_ex(derived, (byte*)"password", 8, (byte*)"NaCl", 4, 1<<10,
                       8, 16, sizeof(verify2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    if (XMEMCMP(derived, verify2, sizeof(verify2)) != 0)
        return WC_TEST_RET_ENC_NC;
#endif

#endif /* !HAVE_FIPS */

    return 0;
}
#endif

#ifdef HAVE_PKCS12
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pkcs12_test(void)
{
    WOLFSSL_SMALL_STACK_STATIC const byte passwd[] = { 0x00, 0x73, 0x00, 0x6d, 0x00, 0x65, 0x00, 0x67,
                            0x00, 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte salt[] =   { 0x0a, 0x58, 0xCF, 0x64, 0x53, 0x0d, 0x82, 0x3f };

    WOLFSSL_SMALL_STACK_STATIC const byte passwd2[] = { 0x00, 0x71, 0x00, 0x75, 0x00, 0x65, 0x00, 0x65,
                             0x00, 0x67, 0x00, 0x00 };
    WOLFSSL_SMALL_STACK_STATIC const byte salt2[] =   { 0x16, 0x82, 0xC0, 0xfC, 0x5b, 0x3f, 0x7e, 0xc5 };
    byte  derived[64];

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0x27, 0xE9, 0x0D, 0x7E, 0xD5, 0xA1, 0xC4, 0x11,
        0xBA, 0x87, 0x8B, 0xC0, 0x90, 0xF5, 0xCE, 0xBE,
        0x5E, 0x9D, 0x5F, 0xE3, 0xD6, 0x2B, 0x73, 0xAA
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify2[] = {
        0x90, 0x1B, 0x49, 0x70, 0xF0, 0x94, 0xF0, 0xF8,
        0x45, 0xC0, 0xF3, 0xF3, 0x13, 0x59, 0x18, 0x6A,
        0x35, 0xE3, 0x67, 0xFE, 0xD3, 0x21, 0xFD, 0x7C
    };

    int id         =  1;
    int kLen       = 24;
    int iterations =  1;
    wc_test_ret_t ret = wc_PKCS12_PBKDF(derived, passwd, sizeof(passwd), salt, 8,
                                                  iterations, kLen, WC_SHA256, id);

    if (ret < 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(derived, verify, kLen) != 0)
        return WC_TEST_RET_ENC_NC;

    iterations = 1000;
    ret = wc_PKCS12_PBKDF(derived, passwd2, sizeof(passwd2), salt2, 8,
                                                  iterations, kLen, WC_SHA256, id);
    if (ret < 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_PKCS12_PBKDF_ex(derived, passwd2, sizeof(passwd2), salt2, 8,
                                       iterations, kLen, WC_SHA256, id, HEAP_HINT);
    if (ret < 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(derived, verify2, 24) != 0)
        return WC_TEST_RET_ENC_NC;

    return 0;
}
#endif /* HAVE_PKCS12 */

#if defined(HAVE_PBKDF2) && !defined(NO_SHA256) && !defined(NO_HMAC)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pbkdf2_test(void)
{
    char passwd[] = "passwordpassword";
    WOLFSSL_SMALL_STACK_STATIC const byte salt[] = { 0x78, 0x57, 0x8E, 0x5a, 0x5d, 0x63, 0xcb, 0x06 };
    int   iterations = 2048;
    int   kLen = 24;
    byte  derived[64];

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0x43, 0x6d, 0xb5, 0xe8, 0xd0, 0xfb, 0x3f, 0x35, 0x42, 0x48, 0x39, 0xbc,
        0x2d, 0xd4, 0xf9, 0x37, 0xd4, 0x95, 0x16, 0xa7, 0x2a, 0x9a, 0x21, 0xd1
    };

    wc_test_ret_t ret = wc_PBKDF2_ex(derived, (byte*)passwd, (int)XSTRLEN(passwd), salt,
              (int)sizeof(salt), iterations, kLen, WC_SHA256, HEAP_HINT, devId);
    if (ret != 0)
        return ret;

    if (XMEMCMP(derived, verify, sizeof(verify)) != 0)
        return WC_TEST_RET_ENC_NC;

    return 0;

}
#endif /* HAVE_PBKDF2 && !NO_SHA256 && !NO_HMAC */

#if defined(HAVE_PBKDF1) && !defined(NO_SHA)
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pbkdf1_test(void)
{
    char passwd[] = "password";
    WOLFSSL_SMALL_STACK_STATIC const byte salt[] = { 0x78, 0x57, 0x8E, 0x5a, 0x5d, 0x63, 0xcb, 0x06 };
    int   iterations = 1000;
    int   kLen = 16;
    byte  derived[16];

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0xDC, 0x19, 0x84, 0x7E, 0x05, 0xC6, 0x4D, 0x2F,
        0xAF, 0x10, 0xEB, 0xFB, 0x4A, 0x3D, 0x2A, 0x20
    };

    wc_test_ret_t ret = wc_PBKDF1_ex(derived, kLen, NULL, 0, (byte*)passwd,
        (int)XSTRLEN(passwd), salt, (int)sizeof(salt), iterations, WC_SHA,
        HEAP_HINT);
    if (ret != 0)
        return ret;

    if (XMEMCMP(derived, verify, sizeof(verify)) != 0)
        return WC_TEST_RET_ENC_NC;

    return 0;
}
#endif /* HAVE_PBKDF2 && !NO_SHA */

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t pwdbased_test(void)
{
   wc_test_ret_t ret = 0;

#if defined(HAVE_PBKDF1) && !defined(NO_SHA)
   ret = pbkdf1_test();
   if (ret != 0)
      return ret;
#endif
#if defined(HAVE_PBKDF2) && !defined(NO_SHA256) && !defined(NO_HMAC)
   ret = pbkdf2_test();
   if (ret != 0)
      return ret;
#endif
#ifdef HAVE_PKCS12
   ret = pkcs12_test();
   if (ret != 0)
      return ret;
#endif
#ifdef HAVE_SCRYPT
   ret = scrypt_test();
   if (ret != 0)
      return ret;
#endif
   return ret;
}

#endif /* NO_PWDBASED */

#if defined(HAVE_HKDF) && !defined(NO_HMAC)

#if defined(WOLFSSL_AFALG_XILINX) || defined(WOLFSSL_AFALG_XILINX_AES) ||     \
    defined(WOLFSSL_AFALG_XILINX_SHA3) || defined(WOLFSSL_AFALG_HASH_KEEP) || \
    defined(WOLFSSL_AFALG_XILINX_RSA)
/* hkdf_test has issue with WOLFSSL_TEST_SUBROUTINE set on Xilinx with afalg */
static wc_test_ret_t hkdf_test(void)
#else
WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hkdf_test(void)
#endif
{
    wc_test_ret_t ret = 0;

#if !defined(NO_SHA) || !defined(NO_SHA256)
    int L;
    byte okm1[42];
    byte ikm1[22] = { 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                      0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
                      0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b };
#ifndef HAVE_FIPS
    byte salt1[13] ={ 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                      0x08, 0x09, 0x0a, 0x0b, 0x0c };
    byte info1[10] ={ 0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
                      0xf8, 0xf9 };
#endif
#ifndef NO_SHA
    byte res1[42] = { 0x0a, 0xc1, 0xaf, 0x70, 0x02, 0xb3, 0xd7, 0x61,
                      0xd1, 0xe5, 0x52, 0x98, 0xda, 0x9d, 0x05, 0x06,
                      0xb9, 0xae, 0x52, 0x05, 0x72, 0x20, 0xa3, 0x06,
                      0xe0, 0x7b, 0x6b, 0x87, 0xe8, 0xdf, 0x21, 0xd0,
                      0xea, 0x00, 0x03, 0x3d, 0xe0, 0x39, 0x84, 0xd3,
                      0x49, 0x18 };
#ifndef HAVE_FIPS
    byte res2[42] = { 0x08, 0x5a, 0x01, 0xea, 0x1b, 0x10, 0xf3, 0x69,
                      0x33, 0x06, 0x8b, 0x56, 0xef, 0xa5, 0xad, 0x81,
                      0xa4, 0xf1, 0x4b, 0x82, 0x2f, 0x5b, 0x09, 0x15,
                      0x68, 0xa9, 0xcd, 0xd4, 0xf1, 0x55, 0xfd, 0xa2,
                      0xc2, 0x2e, 0x42, 0x24, 0x78, 0xd3, 0x05, 0xf3,
                      0xf8, 0x96 };
#endif
#endif /* !NO_SHA */
#ifndef NO_SHA256
    byte res3[42] = { 0x8d, 0xa4, 0xe7, 0x75, 0xa5, 0x63, 0xc1, 0x8f,
                      0x71, 0x5f, 0x80, 0x2a, 0x06, 0x3c, 0x5a, 0x31,
                      0xb8, 0xa1, 0x1f, 0x5c, 0x5e, 0xe1, 0x87, 0x9e,
                      0xc3, 0x45, 0x4e, 0x5f, 0x3c, 0x73, 0x8d, 0x2d,
                      0x9d, 0x20, 0x13, 0x95, 0xfa, 0xa4, 0xb6, 0x1a,
                      0x96, 0xc8 };
#ifndef HAVE_FIPS
    byte res4[42] = { 0x3c, 0xb2, 0x5f, 0x25, 0xfa, 0xac, 0xd5, 0x7a,
                      0x90, 0x43, 0x4f, 0x64, 0xd0, 0x36, 0x2f, 0x2a,
                      0x2d, 0x2d, 0x0a, 0x90, 0xcf, 0x1a, 0x5a, 0x4c,
                      0x5d, 0xb0, 0x2d, 0x56, 0xec, 0xc4, 0xc5, 0xbf,
                      0x34, 0x00, 0x72, 0x08, 0xd5, 0xb8, 0x87, 0x18,
                      0x58, 0x65 };
#endif
#endif /* !NO_SHA256 */

    XMEMSET(okm1, 0, sizeof(okm1));
    L = (int)sizeof(okm1);

#ifndef NO_SHA
    ret = wc_HKDF(WC_SHA, ikm1, (word32)sizeof(ikm1), NULL, 0, NULL, 0,
        okm1, L);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(okm1, res1, L) != 0)
        return WC_TEST_RET_ENC_NC;

#ifndef HAVE_FIPS
    /* fips can't have key size under 14 bytes, salt is key too */
    L = (int)sizeof(okm1);
    ret = wc_HKDF(WC_SHA, ikm1, 11, salt1, (word32)sizeof(salt1),
        info1, (word32)sizeof(info1), okm1, L);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(okm1, res2, L) != 0)
        return WC_TEST_RET_ENC_NC;
#endif /* HAVE_FIPS */
#endif /* !NO_SHA */

#ifndef NO_SHA256
    ret = wc_HKDF(WC_SHA256, ikm1, (word32)sizeof(ikm1), NULL, 0, NULL, 0,
        okm1, L);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(okm1, res3, L) != 0)
        return WC_TEST_RET_ENC_NC;

#ifndef HAVE_FIPS
    /* fips can't have key size under 14 bytes, salt is key too */
    ret = wc_HKDF(WC_SHA256, ikm1, (word32)sizeof(ikm1),
        salt1, (word32)sizeof(salt1), info1, (word32)sizeof(info1), okm1, L);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(okm1, res4, L) != 0)
        return WC_TEST_RET_ENC_NC;
#endif /* HAVE_FIPS */
#endif /* !NO_SHA256 */
#endif /* !NO_SHA || !NO_SHA256 */

    return ret;
}

#endif /* HAVE_HKDF */


#ifdef WOLFSSL_WOLFSSH

typedef struct {
    byte hashId;
    byte keyId;
    const byte* k;
    word32 kSz;
    const byte* h;
    word32 hSz;
    const byte* sessionId;
    word32 sessionIdSz;
    const byte* expectedKey;
    word32 expectedKeySz;
} SshKdfTestVector;


/** Test Vector Set #3: SHA-256 **/
static const byte sshKdfTvSet3k[] = {
    0x6A, 0xC3, 0x82, 0xEA, 0xAC, 0xA0, 0x93, 0xE1,
    0x25, 0xE2, 0x5C, 0x24, 0xBE, 0xBC, 0x84, 0x64,
    0x0C, 0x11, 0x98, 0x75, 0x07, 0x34, 0x4B, 0x5C,
    0x73, 0x9C, 0xEB, 0x84, 0xA9, 0xE0, 0xB2, 0x22,
    0xB9, 0xA8, 0xB5, 0x1C, 0x83, 0x9E, 0x5E, 0xBE,
    0x49, 0xCF, 0xAD, 0xBF, 0xB3, 0x95, 0x99, 0x76,
    0x4E, 0xD5, 0x22, 0x09, 0x9D, 0xC9, 0x12, 0x75,
    0x19, 0x50, 0xDC, 0x7D, 0xC9, 0x7F, 0xBD, 0xC0,
    0x63, 0x28, 0xB6, 0x8F, 0x22, 0x78, 0x1F, 0xD3,
    0x15, 0xAF, 0x56, 0x80, 0x09, 0xA5, 0x50, 0x9E,
    0x5B, 0x87, 0xA1, 0x1B, 0xF5, 0x27, 0xC0, 0x56,
    0xDA, 0xFF, 0xD8, 0x2A, 0xB6, 0xCB, 0xC2, 0x5C,
    0xCA, 0x37, 0x14, 0x34, 0x59, 0xE7, 0xBC, 0x63,
    0xBC, 0xDE, 0x52, 0x75, 0x7A, 0xDE, 0xB7, 0xDF,
    0x01, 0xCF, 0x12, 0x17, 0x3F, 0x1F, 0xEF, 0x81,
    0x02, 0xEC, 0x5A, 0xB1, 0x42, 0xC2, 0x13, 0xDD,
    0x9D, 0x30, 0x69, 0x62, 0x78, 0xA8, 0xD8, 0xBC,
    0x32, 0xDD, 0xE9, 0x59, 0x2D, 0x28, 0xC0, 0x78,
    0xC6, 0xD9, 0x2B, 0x94, 0x7D, 0x82, 0x5A, 0xCA,
    0xAB, 0x64, 0x94, 0x84, 0x6A, 0x49, 0xDE, 0x24,
    0xB9, 0x62, 0x3F, 0x48, 0x89, 0xE8, 0xAD, 0xC3,
    0x8E, 0x8C, 0x66, 0x9E, 0xFF, 0xEF, 0x17, 0x60,
    0x40, 0xAD, 0x94, 0x5E, 0x90, 0xA7, 0xD3, 0xEE,
    0xC1, 0x5E, 0xFE, 0xEE, 0x78, 0xAE, 0x71, 0x04,
    0x3C, 0x96, 0x51, 0x11, 0x03, 0xA1, 0x6B, 0xA7,
    0xCA, 0xF0, 0xAC, 0xD0, 0x64, 0x2E, 0xFD, 0xBE,
    0x80, 0x99, 0x34, 0xFA, 0xA1, 0xA5, 0xF1, 0xBD,
    0x11, 0x04, 0x36, 0x49, 0xB2, 0x5C, 0xCD, 0x1F,
    0xEE, 0x2E, 0x38, 0x81, 0x5D, 0x4D, 0x5F, 0x5F,
    0xC6, 0xB4, 0x10, 0x29, 0x69, 0xF2, 0x1C, 0x22,
    0xAE, 0x1B, 0x0E, 0x7D, 0x36, 0x03, 0xA5, 0x56,
    0xA1, 0x32, 0x62, 0xFF, 0x62, 0x8D, 0xE2, 0x22
};
static const byte sshKdfTvSet3h[] = {
    0x7B, 0x70, 0x01, 0x18, 0x5E, 0x25, 0x6D, 0x44,
    0x93, 0x44, 0x5F, 0x39, 0xA5, 0x5F, 0xB9, 0x05,
    0xE6, 0x32, 0x1F, 0x4B, 0x5D, 0xD8, 0xBB, 0xF3,
    0x10, 0x0D, 0x51, 0xBA, 0x0B, 0xDA, 0x3D, 0x2D
};
static const byte sshKdfTvSet3sid[] = {
    0x7B, 0x70, 0x01, 0x18, 0x5E, 0x25, 0x6D, 0x44,
    0x93, 0x44, 0x5F, 0x39, 0xA5, 0x5F, 0xB9, 0x05,
    0xE6, 0x32, 0x1F, 0x4B, 0x5D, 0xD8, 0xBB, 0xF3,
    0x10, 0x0D, 0x51, 0xBA, 0x0B, 0xDA, 0x3D, 0x2D
};
static const byte sshKdfTvSet3a[] = {
    0x81, 0xF0, 0x33, 0x0E, 0xF6, 0xF0, 0x53, 0x61,
    0xB3, 0x82, 0x3B, 0xFD, 0xED, 0x6E, 0x1D, 0xE9
};
static const byte sshKdfTvSet3b[] = {
    0x3F, 0x6F, 0xD2, 0x06, 0x5E, 0xEB, 0x2B, 0x0B,
    0x1D, 0x93, 0x19, 0x5A, 0x1F, 0xED, 0x48, 0xA5
};
static const byte sshKdfTvSet3c[] = {
    0xC3, 0x54, 0x71, 0x03, 0x4E, 0x6F, 0xD6, 0x54,
    0x76, 0x13, 0x17, 0x8E, 0x23, 0x43, 0x5F, 0x21
};
static const byte sshKdfTvSet3d[] = {
    0x7E, 0x9D, 0x79, 0x03, 0x20, 0x90, 0xD9, 0x9F,
    0x98, 0xB0, 0x15, 0x63, 0x4D, 0xD9, 0xF4, 0x62
};
static const byte sshKdfTvSet3e[] = {
    0x24, 0xEE, 0x55, 0x9A, 0xD7, 0xCE, 0x71, 0x2B,
    0x68, 0x5D, 0x0B, 0x22, 0x71, 0xE4, 0x43, 0xC1,
    0x7A, 0xB1, 0xD1, 0xDC, 0xEB, 0x5A, 0x36, 0x05,
    0x69, 0xD2, 0x5D, 0x5D, 0xC2, 0x43, 0x00, 0x2F
};
static const byte sshKdfTvSet3f[] = {
    0xC3, 0x41, 0x9C, 0x2B, 0x96, 0x62, 0x35, 0x86,
    0x9D, 0x71, 0x4B, 0xA5, 0xAC, 0x48, 0xDD, 0xB7,
    0xD9, 0xE3, 0x5C, 0x8C, 0x19, 0xAA, 0xC7, 0x34,
    0x22, 0x33, 0x7A, 0x37, 0x34, 0x53, 0x60, 0x7E
};

static const SshKdfTestVector sshKdfTestVectors[] = {
    {WC_HASH_TYPE_SHA256, 'A',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3a, sizeof(sshKdfTvSet3a)},
    {WC_HASH_TYPE_SHA256, 'B',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3b, sizeof(sshKdfTvSet3b)},
    {WC_HASH_TYPE_SHA256, 'C',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3c, sizeof(sshKdfTvSet3c)},
    {WC_HASH_TYPE_SHA256, 'D',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3d, sizeof(sshKdfTvSet3d)},
    {WC_HASH_TYPE_SHA256, 'E',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3e, sizeof(sshKdfTvSet3e)},
    {WC_HASH_TYPE_SHA256, 'F',
        sshKdfTvSet3k, sizeof(sshKdfTvSet3k),
        sshKdfTvSet3h, sizeof(sshKdfTvSet3h),
        sshKdfTvSet3sid, sizeof(sshKdfTvSet3sid),
        sshKdfTvSet3f, sizeof(sshKdfTvSet3f)},
};


WOLFSSL_TEST_SUBROUTINE wc_test_ret_t sshkdf_test(void)
{
    wc_test_ret_t result = 0;
    word32 i;
    word32 tc = sizeof(sshKdfTestVectors)/sizeof(SshKdfTestVector);
    const SshKdfTestVector* tv = NULL;
    byte cKey[32]; /* Greater of SHA256_DIGEST_SIZE and AES_BLOCK_SIZE */
    /* sId - Session ID, eKey - Expected Key, cKey - Calculated Key */

    for (i = 0, tv = sshKdfTestVectors; i < tc; i++, tv++) {
        result = wc_SSH_KDF(tv->hashId, tv->keyId,
                cKey, tv->expectedKeySz,
                tv->k, tv->kSz, tv->h, tv->hSz,
                tv->sessionId, tv->sessionIdSz);

        if (result != 0) {
            printf("KDF: Could not derive key.\n");
            result = WC_TEST_RET_ENC_EC(result);
        }
        else {
            if (XMEMCMP(cKey, tv->expectedKey, tv->expectedKeySz) != 0) {
                printf("KDF: Calculated Key does not match Expected Key.\n");
                result = WC_TEST_RET_ENC_EC(result);
            }
        }

        if (result != 0) break;
    }

    return result;
}

#endif /* WOLFSSL_WOLFSSH */


#ifdef WOLFSSL_TLS13

#define TLSV13_PSK_DHE_SZ 40
typedef struct {
    enum wc_HashType hashAlg;
    word32 pskSz;
    word32 dheSz;
    byte psk[TLSV13_PSK_DHE_SZ];
    byte dhe[TLSV13_PSK_DHE_SZ];
    byte hashHello1[WC_MAX_DIGEST_SIZE];
    byte hashHello2[WC_MAX_DIGEST_SIZE];
    byte hashFinished1[WC_MAX_DIGEST_SIZE];
    byte hashFinished2[WC_MAX_DIGEST_SIZE];
    /* Expected */
    byte clientEarlyTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte earlyExporterMasterSecret[WC_MAX_DIGEST_SIZE];
    byte clientHandshakeTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte serverHandshakeTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte clientApplicationTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte serverApplicationTrafficSecret[WC_MAX_DIGEST_SIZE];
    byte exporterMasterSecret[WC_MAX_DIGEST_SIZE];
    byte resumptionMasterSecret[WC_MAX_DIGEST_SIZE];
} Tls13KdfTestVector;

/* The following tests come from the CAVP test vectors we used for
 * our FIPS validation. The hash values used are the components from
 * the test hashed together. hashHello1 is the hash of the
 * clientHelloRandom value of the test vector. hashHello2 is the hash
 * of the clientHelloRandom and serverHelloRandom values from the test
 * vector. hashFinished1 is clientHelloRandom, serverHelloRandom, and
 * serverFinishedRandom. hashFinished2 is clientHelloRandom,
 * serverHelloRandom, serverFinishedRandom, and clietnFinishedRandom
 * hashed together. */
static const Tls13KdfTestVector tls13KdfTestVectors[] = {
{ /* 1 */
    WC_HASH_TYPE_SHA256, 35, 35,
    { /* PSK */
        0x7b, 0xf1, 0x05, 0x31, 0x36, 0xfa, 0x03, 0xdc,
        0x31, 0x97, 0x88, 0x04, 0x9c, 0xbc, 0xee, 0xf7,
        0x8d, 0x84, 0x95, 0x26, 0xaf, 0x1d, 0x68, 0xb0,
        0x60, 0x7a, 0xcc, 0x4f, 0xc1, 0xd3, 0xa1, 0x68,
        0x7f, 0x6d, 0xbe
    },
    { /* DHE */
        0x6e, 0xa1, 0x77, 0xab, 0x2f, 0x43, 0xd2, 0x4b,
        0xe5, 0xa1, 0x09, 0xe0, 0x7a, 0xd0, 0x01, 0x35,
        0x8d, 0xf8, 0xf2, 0x5c, 0x91, 0x02, 0xb0, 0x6c,
        0x3f, 0xeb, 0xee, 0xa4, 0x42, 0x19, 0xce, 0xdc,
        0x81, 0x26, 0x40
    },
    { /* Hello 1 */
        0xd9, 0x4b, 0xe4, 0x17, 0xef, 0x58, 0x73, 0x7d,
        0x28, 0x3d, 0xf0, 0xcc, 0x05, 0x03, 0xaf, 0xac,
        0x3d, 0x92, 0x79, 0x48, 0xe8, 0x8c, 0xdb, 0xce,
        0x95, 0x82, 0x21, 0x31, 0x7b, 0x61, 0xd7, 0xc6
    },
    { /* Hello 2 */
        0xb7, 0x7f, 0x29, 0x91, 0xa4, 0x8b, 0x34, 0xdb,
        0xbd, 0xc7, 0x54, 0x1c, 0x3b, 0x86, 0xa3, 0x69,
        0xfe, 0x26, 0xe4, 0x7b, 0xac, 0x57, 0x71, 0xb3,
        0x32, 0x97, 0xed, 0xd2, 0x0e, 0x95, 0xb8, 0x63
    },
    { /* Finished 1 */
        0x65, 0xdb, 0x6d, 0x71, 0x71, 0xd0, 0xd8, 0x49,
        0xd0, 0x3c, 0x8e, 0x2b, 0x24, 0xdf, 0xc2, 0xe9,
        0xd6, 0xfd, 0xea, 0x04, 0x95, 0x7c, 0xf0, 0x7e,
        0x57, 0x74, 0x7c, 0xdd, 0xa3, 0x0b, 0x2b, 0x36
    },
    { /* Finished 2 */
        0x28, 0xf2, 0xf2, 0x79, 0xcf, 0x20, 0x52, 0x90,
        0x1d, 0x91, 0x05, 0xad, 0x44, 0x26, 0x23, 0x96,
        0x32, 0xce, 0xec, 0x61, 0xd1, 0xbf, 0x00, 0x48,
        0x4a, 0xa5, 0x60, 0xcc, 0x28, 0xb5, 0x8d, 0x98
    },
    { /* Client Early Traffic Secret */
        0x07, 0x14, 0x6a, 0x26, 0x5b, 0x6c, 0x7f, 0x4d, 0x6b, 0x47, 0x3f, 0xd5,
        0x03, 0x1d, 0xd2, 0x23, 0x3d, 0x89, 0x3e, 0xc6, 0x51, 0xd1, 0xac, 0xf8,
        0x28, 0xae, 0x4b, 0x76, 0xc8, 0x10, 0x7e, 0xdd
    },
    { /* Early Exporter Master Secret */
        0xb8, 0xd3, 0x25, 0x7e, 0x2d, 0x41, 0x7b, 0xcb, 0x5e, 0x82, 0x49, 0xf5,
        0x51, 0x3d, 0xb7, 0x59, 0x32, 0xb3, 0xdf, 0x99, 0x4e, 0x04, 0x69, 0xc6,
        0x96, 0x8e, 0xe6, 0x3d, 0x91, 0xe4, 0x81, 0x11
    },
    { /* Client Handshake Traffic Secret */
        0xd9, 0x3b, 0x54, 0xe2, 0xb0, 0xd1, 0x85, 0xf0, 0xfd, 0xf3, 0x48, 0x4a,
        0xf8, 0x0b, 0xa5, 0xdc, 0x4c, 0x37, 0xcb, 0xd4, 0x20, 0xaf, 0x60, 0xc7,
        0xd5, 0x50, 0x5d, 0x0c, 0x77, 0x3b, 0x6f, 0xd2
    },
    { /* Server Handshake Traffic Secret */
        0x4d, 0x40, 0x2b, 0xd2, 0x8c, 0x33, 0x90, 0x39, 0x67, 0x67, 0x05, 0xf7,
        0x5d, 0x37, 0x1e, 0xdc, 0x4a, 0x70, 0x6b, 0x9e, 0xf8, 0x06, 0x61, 0x89,
        0x70, 0xe1, 0x3d, 0x36, 0xad, 0x88, 0x7e, 0x5b
    },
    { /* Client Application Traffic Secret */
        0x74, 0x6e, 0xa0, 0x13, 0x18, 0x34, 0x48, 0x4d, 0x23, 0x31, 0xf1, 0xf9,
        0xee, 0x44, 0x6d, 0xad, 0xc1, 0xad, 0x92, 0x73, 0xca, 0x27, 0x16, 0x91,
        0xa2, 0x50, 0x9a, 0xfc, 0xec, 0xf0, 0x6b, 0x24
    },
    { /* Server Application Traffic Secret */
        0x89, 0x18, 0x7e, 0x34, 0x8d, 0xfc, 0x14, 0xb1, 0x4f, 0x21, 0xd8, 0x29,
        0xdb, 0x9b, 0xfb, 0x55, 0xcf, 0xa1, 0x4f, 0x95, 0xf8, 0xe0, 0xb0, 0x83,
        0xd5, 0x34, 0x9e, 0x0b, 0x83, 0x37, 0x42, 0x93
    },
    { /* Exporter Master Secret */
        0x7d, 0xc8, 0x88, 0x46, 0xd5, 0x57, 0x15, 0xb6, 0x24, 0x25, 0x92, 0x61,
        0xb1, 0x18, 0x86, 0x2a, 0x6d, 0xa5, 0x84, 0xeb, 0x59, 0xdf, 0x13, 0xbd,
        0x73, 0xaa, 0x5d, 0x65, 0xab, 0xd9, 0xb4, 0x56
    },
    { /* Resumption Master Secret */
        0x20, 0xb7, 0xd0, 0xe3, 0x82, 0x01, 0xa1, 0x04, 0xb8, 0x13, 0x29, 0xed,
        0x35, 0xe4, 0x2f, 0xbf, 0x58, 0x23, 0x7f, 0x21, 0xdb, 0x9f, 0xf8, 0xe0,
        0xe8, 0xe4, 0xab, 0xc4, 0xa1, 0x61, 0xb9, 0xbb
    }
},
{ /* 6 */
    WC_HASH_TYPE_SHA256, 0, 33,
    { 0 }, /* PSK */
    { /* DHE */
        0x7a, 0x46, 0x8c, 0x5a, 0xd1, 0x8e, 0x95, 0xba,
        0x61, 0xe6, 0x6f, 0xe6, 0x76, 0x0c, 0x20, 0x43,
        0x16, 0x82, 0x15, 0xfe, 0x54, 0xa3, 0xc7, 0xfd,
        0x3b, 0x2c, 0x88, 0xb4, 0xd3, 0x42, 0x70, 0x12,
        0x18
    },
    { /* Hello 1 */
        0x63, 0x83, 0x58, 0xab, 0x36, 0xcd, 0x0c, 0xf3,
        0x26, 0x07, 0xb5, 0x5f, 0x0b, 0x8b, 0x45, 0xd6,
        0x7d, 0x5b, 0x42, 0xdc, 0xa8, 0xaa, 0x06, 0xfb,
        0x20, 0xa5, 0xbb, 0x85, 0xdb, 0x54, 0xd8, 0x8b
    },
    { /* Hello 2 */
        0xea, 0xfe, 0x9e, 0x8e, 0xff, 0x1f, 0x6f, 0x43,
        0xf9, 0x5d, 0xfd, 0xbf, 0xe2, 0x5f, 0x02, 0x2f,
        0x6d, 0x47, 0x60, 0x9a, 0x48, 0x9a, 0x75, 0xfb,
        0xb5, 0x4a, 0xbf, 0x9c, 0x4e, 0xff, 0xbf, 0x0b
    },
    { /* Finished 1 */
        0xca, 0x25, 0xb3, 0x53, 0x8e, 0x6d, 0xc3, 0x36,
        0x17, 0x30, 0x07, 0xdf, 0x0d, 0xd7, 0x79, 0xb0,
        0x7f, 0xcb, 0xbe, 0x7a, 0xbc, 0x2d, 0x9f, 0x2d,
        0x94, 0x44, 0x94, 0xe6, 0xa4, 0xf3, 0xe8, 0x53
    },
    { /* Finished 2 */
        0x2e, 0xa6, 0x5a, 0xaf, 0xb5, 0xba, 0x9f, 0x2f,
        0x74, 0x83, 0x5d, 0xbf, 0x86, 0xa4, 0xa6, 0xf6,
        0xb9, 0x89, 0xdf, 0x17, 0xe1, 0xa8, 0x14, 0xc0,
        0xe1, 0x50, 0xfa, 0xec, 0xfa, 0xae, 0x8b, 0x7b
    },
    {
        0x20, 0x18, 0x72, 0x7c, 0xde, 0x3a, 0x85, 0x17, 0x72, 0xdc, 0xd7, 0x72,
        0xb0, 0xfc, 0x45, 0xd0, 0x62, 0xb9, 0xbb, 0x38, 0x69, 0x05, 0x7b, 0xb4,
        0x5e, 0x58, 0x5d, 0xed, 0xcd, 0x0b, 0x96, 0xd3
    },
    {
        0x68, 0x10, 0x20, 0xd1, 0x5e, 0xfc, 0x0c, 0x53, 0x85, 0xbb, 0xdb, 0x18,
        0xa8, 0x78, 0xf1, 0x2b, 0x13, 0xba, 0x64, 0x1d, 0xe7, 0x09, 0xbe, 0x13,
        0x49, 0x26, 0xf9, 0x98, 0x56, 0xf1, 0x43, 0xfb
    },
    {
        0x24, 0x35, 0x3e, 0x10, 0x6f, 0x39, 0x50, 0xd6, 0xa2, 0x12, 0x99, 0xf2,
        0xd5, 0xf5, 0x19, 0xf5, 0x84, 0xed, 0xee, 0x78, 0x2a, 0xa6, 0xfa, 0x3d,
        0x06, 0xa8, 0xa7, 0x5d, 0x97, 0x78, 0xd6, 0x58
    },
    {
        0xf4, 0x57, 0xac, 0x24, 0x7a, 0xfb, 0x7c, 0x3b, 0xb6, 0x39, 0x17, 0x14,
        0xd9, 0xd4, 0x58, 0x4d, 0x46, 0xd5, 0x1b, 0xde, 0xf7, 0x9d, 0x06, 0xee,
        0x8d, 0x1a, 0x2c, 0x25, 0x6d, 0x64, 0xde, 0x89
    },
    {
        0xb6, 0x00, 0xce, 0x63, 0xed, 0x65, 0x8b, 0x66, 0x66, 0x42, 0xc6, 0xbd,
        0x89, 0xc4, 0x71, 0x6f, 0xce, 0x28, 0xb2, 0xac, 0x97, 0x07, 0x5b, 0xea,
        0xb8, 0x1d, 0x4c, 0xeb, 0x9e, 0x71, 0x07, 0x8f
    },
    {
        0xf8, 0x92, 0xc8, 0xba, 0xe7, 0x83, 0xfe, 0x68, 0xe4, 0xd6, 0x5e, 0xcb,
        0xb3, 0xef, 0x49, 0xd0, 0xe7, 0xb1, 0xac, 0xcb, 0x39, 0x19, 0xfd, 0xa7,
        0xf7, 0xca, 0xab, 0x1e, 0x42, 0x14, 0xd8, 0xe7
    },
    {
        0x32, 0x4a, 0x1a, 0xad, 0xe2, 0xbb, 0x55, 0x8a, 0xdd, 0xe9, 0xa5, 0x2a,
        0x46, 0x5e, 0x6c, 0x83, 0x66, 0x27, 0x27, 0x94, 0xdd, 0x68, 0x59, 0xa0,
        0xbb, 0xe8, 0x31, 0x7c, 0x39, 0xd7, 0xfd, 0x6d
    },
    {
        0x58, 0xbc, 0x6c, 0x5b, 0x24, 0xad, 0x82, 0xb3, 0xcc, 0xc7, 0xd1, 0xa1,
        0xaa, 0x2b, 0x98, 0x9f, 0x2f, 0x7e, 0xa9, 0x63, 0xc2, 0x8e, 0xb6, 0x06,
        0xc2, 0x2b, 0x74, 0x4b, 0x79, 0x19, 0x7e, 0x2e
    }
},
{ /* 11 */
    WC_HASH_TYPE_SHA256, 33, 0,
    { /* PSK */
        0x3d, 0x39, 0x49, 0x36, 0x98, 0xc5, 0xfd, 0xcd,
        0xa0, 0x17, 0xbd, 0x65, 0x0a, 0xdb, 0xd4, 0x07,
        0x56, 0xa2, 0x7b, 0xb8, 0x2a, 0x7e, 0xfb, 0x26,
        0x74, 0xe1, 0xbc, 0x08, 0x4b, 0xf0, 0x30, 0x14,
        0x12
    },
    { 0 }, /* DHE */
    { /* Hello 1 */
        0xb7, 0x44, 0x74, 0x6c, 0x57, 0x1f, 0xf3, 0x84,
        0x8f, 0x63, 0xfb, 0x8c, 0x94, 0x6c, 0x16, 0x68,
        0x4b, 0xe1, 0xb5, 0xb5, 0x2a, 0x4e, 0x5f, 0xdf,
        0x4b, 0x53, 0xb2, 0x35, 0xfc, 0x30, 0xf1, 0x36
    },
    { /* Hello 2 */
        0xe6, 0x4f, 0x3a, 0x4f, 0xd7, 0xe0, 0x64, 0xd4,
        0x69, 0x50, 0xe4, 0x8b, 0xba, 0xbc, 0x47, 0x74,
        0xa7, 0x9b, 0x40, 0x91, 0x8f, 0xa8, 0x72, 0x22,
        0x97, 0xad, 0x43, 0xa7, 0x11, 0x86, 0xb5, 0x72
    },
    { /* Finished 1 */
        0x5f, 0xa6, 0x10, 0xe2, 0xa3, 0x99, 0x0b, 0x5e,
        0x57, 0xee, 0xc3, 0x3a, 0x8e, 0x04, 0xf3, 0x0e,
        0x58, 0x02, 0x09, 0xb2, 0x7e, 0x2d, 0xc6, 0xd2,
        0x08, 0xae, 0x68, 0x0a, 0x55, 0xa5, 0xda, 0x51
    },
    { /* Finished 2 */
        0xfc, 0x5b, 0xc0, 0x7e, 0x1b, 0xaa, 0xc0, 0xb4,
        0x34, 0x85, 0x49, 0x8e, 0x16, 0x31, 0x98, 0xdf,
        0x10, 0x54, 0x22, 0xda, 0x1e, 0x6b, 0x51, 0xf6,
        0x97, 0x57, 0xa0, 0x7a, 0x92, 0xe7, 0x47, 0x52
    },
    {
        0x80, 0xfa, 0x36, 0x30, 0xb8, 0x65, 0xb3, 0x2a, 0x1d, 0x68, 0x91, 0x06,
        0x98, 0xa0, 0x17, 0x8f, 0xee, 0xb7, 0x9e, 0x3d, 0xd8, 0x84, 0x99, 0x30,
        0xb9, 0xd6, 0x09, 0x25, 0x5e, 0xfb, 0x8f, 0xd3 },
    {
        0xa9, 0x89, 0x29, 0x70, 0xe4, 0x55, 0xec, 0x97, 0xfb, 0x24, 0x5b, 0xf9,
        0xf1, 0xa3, 0x19, 0x3d, 0xf1, 0x31, 0x14, 0xcd, 0x2a, 0xed, 0x21, 0xc8,
        0xb1, 0x53, 0xad, 0x11, 0x0b, 0x9e, 0x5a, 0xee },
    {
        0x72, 0xad, 0x8d, 0x7f, 0xfc, 0xb7, 0x68, 0xda, 0x27, 0x60, 0x37, 0xa3,
        0x4a, 0x63, 0xe8, 0xa5, 0xc8, 0xcd, 0x36, 0x6a, 0x77, 0x99, 0x0d, 0xa9,
        0xb1, 0x5b, 0x2f, 0x47, 0x2e, 0x22, 0xa7, 0x5e },
    {
        0x95, 0x6e, 0x85, 0x09, 0xe5, 0x04, 0x88, 0x14, 0x28, 0x8d, 0xdf, 0xe6,
        0x0d, 0x0f, 0x0d, 0x6b, 0x4e, 0x66, 0x1c, 0x03, 0xb9, 0xaa, 0x2d, 0x45,
        0x56, 0x67, 0x5c, 0x55, 0x29, 0xd6, 0x89, 0xd0 },
    {
        0xe8, 0xf2, 0x14, 0xf9, 0x9b, 0x2b, 0x9f, 0x24, 0x2b, 0x37, 0xbe, 0x86,
        0xdb, 0x23, 0x4b, 0xbe, 0x39, 0x57, 0xe8, 0xa9, 0xa5, 0xee, 0x08, 0xf2,
        0x75, 0x58, 0xdb, 0xd9, 0x51, 0xc1, 0x46, 0x02 },
    {
        0x3d, 0x19, 0xaf, 0xa3, 0x0b, 0x21, 0xf7, 0x3d, 0xe7, 0x37, 0x6e, 0x32,
        0x13, 0x48, 0x9d, 0xea, 0xe0, 0x90, 0xbf, 0x64, 0x48, 0xf7, 0x1e, 0xcc,
        0xf0, 0xbc, 0x92, 0xd7, 0x8a, 0x4a, 0xa8, 0xc1 },
    {
        0x16, 0x35, 0xb1, 0x66, 0x28, 0xa3, 0x3e, 0x19, 0xf5, 0x2d, 0x92, 0x22,
        0x95, 0x48, 0xe8, 0x34, 0x7b, 0x30, 0x50, 0xa2, 0xa0, 0xd9, 0xc2, 0x59,
        0x39, 0xf9, 0x8c, 0x69, 0xf2, 0x2a, 0xb9, 0xff },
    {
        0x32, 0x71, 0xa6, 0x87, 0x0c, 0x97, 0x42, 0x07, 0xdd, 0x5f, 0xc9, 0x44,
        0xa5, 0x7c, 0x50, 0x14, 0xfd, 0xe7, 0x5f, 0x8b, 0xd3, 0x2f, 0xdc, 0x9b,
        0xa9, 0x93, 0x22, 0x19, 0xe6, 0xf2, 0x0c, 0xd8 }
},
#ifdef WOLFSSL_SHA384
{ /* 26 */
    WC_HASH_TYPE_SHA384, 35, 35,
    { /* PSK */
        0x62, 0x83, 0x25, 0xc7, 0xcc, 0x08, 0x5e, 0x63,
        0x64, 0x56, 0xf0, 0xc6, 0x88, 0x27, 0x5a, 0x5b,
        0x68, 0x59, 0x0b, 0x14, 0x55, 0x13, 0x2e, 0xfd,
        0x8f, 0x28, 0x5b, 0x3d, 0xe3, 0xad, 0x67, 0xe4,
        0x68, 0xba, 0xf9
    },
    { /* DHE */
        0xa8, 0xb1, 0xab, 0xd8, 0xc8, 0x5b, 0x52, 0xdf,
        0x7f, 0x49, 0x10, 0xf4, 0xa1, 0x31, 0xd1, 0x91,
        0x36, 0xc1, 0x87, 0x5d, 0x42, 0x2a, 0xe7, 0x1d,
        0x2c, 0x29, 0x3d, 0x40, 0x64, 0x61, 0x63, 0x76,
        0xd8, 0x66, 0xac
    },
    { /* Hello 1 */
        0x6f, 0xc6, 0x4c, 0xe1, 0xc6, 0x68, 0x34, 0x8c,
        0x0a, 0xe1, 0xf8, 0xb8, 0x3e, 0xd4, 0xf8, 0x0b,
        0x54, 0x50, 0xe4, 0xc5, 0x4a, 0x33, 0x7d, 0xbd,
        0x90, 0xd2, 0xa2, 0xb9, 0xb7, 0x92, 0xed, 0xab,
        0x14, 0xf1, 0xe4, 0x86, 0x22, 0x67, 0xd7, 0x44,
        0x03, 0x21, 0xdc, 0x51, 0x52, 0x7f, 0x35, 0x80
    },
    { /* Hello 2 */
        0x3e, 0xcf, 0x2f, 0xc3, 0x87, 0xba, 0xc5, 0xbd,
        0x7c, 0xe8, 0x35, 0x5b, 0x95, 0x51, 0x30, 0x3b,
        0x08, 0xcc, 0x2a, 0x7d, 0xb5, 0x74, 0x7c, 0x16,
        0xb3, 0x0b, 0xe7, 0x61, 0xa3, 0x7c, 0x6c, 0xbd,
        0x39, 0x74, 0xfd, 0x1e, 0x4c, 0xff, 0xc8, 0xcc,
        0xa0, 0xef, 0x29, 0x4d, 0x94, 0xaa, 0x55, 0x6f,
    },
    { /* Finished 1 */
        0x06, 0xc1, 0x47, 0x78, 0x66, 0x53, 0x6f, 0x24,
        0x94, 0x61, 0x69, 0xec, 0xd8, 0x60, 0x31, 0x2f,
        0xbf, 0xd6, 0x8a, 0x29, 0x17, 0xff, 0xa3, 0x88,
        0x13, 0x09, 0x8c, 0x9d, 0x6c, 0x64, 0x84, 0x48,
        0x44, 0xdd, 0x2d, 0x29, 0x4d, 0xe6, 0x98, 0x2b,
        0x45, 0x3b, 0x84, 0x33, 0x79, 0xb2, 0x75, 0x68
    },
    { /* Finished 2 */
        0x28, 0x1e, 0x18, 0xf7, 0x9c, 0x32, 0xa9, 0xbf,
        0x0c, 0x24, 0x58, 0x21, 0xce, 0xbc, 0xf2, 0x44,
        0xb1, 0x18, 0xaf, 0x9d, 0xd9, 0x20, 0xf9, 0xf4,
        0xed, 0xcc, 0x53, 0x82, 0x66, 0x5c, 0x46, 0x94,
        0x8c, 0x36, 0x5e, 0xca, 0x9f, 0xd8, 0x9a, 0xd3,
        0xf0, 0xe1, 0x53, 0x71, 0xdd, 0x19, 0x1e, 0x59
    },
    {
        0xd0, 0xef, 0xa8, 0xcb, 0x5b, 0x14, 0x0f, 0x0a, 0x62, 0xba, 0x5a, 0xb1,
        0xc5, 0xb5, 0x3f, 0x11, 0xda, 0xa1, 0x0c, 0x9c, 0xb4, 0x32, 0x48, 0x4e,
        0xfa, 0x84, 0x4f, 0xe4, 0xe7, 0x91, 0x8f, 0x42, 0x3f, 0xc7, 0x4e, 0xd3,
        0x83, 0x3d, 0x7f, 0x70, 0x12, 0xee, 0x9a, 0x37, 0x01, 0xbb, 0x14, 0xd3
    },
    {
        0x48, 0x6f, 0x77, 0x1d, 0x39, 0x1b, 0xa5, 0x9a, 0x76, 0xd9, 0x1d, 0x7d,
        0xb3, 0xd9, 0xb9, 0x78, 0x35, 0x0f, 0xd0, 0xe1, 0x07, 0x1f, 0x8d, 0xe5,
        0x75, 0x00, 0xda, 0xc0, 0x19, 0x01, 0xfb, 0x08, 0x35, 0xe7, 0x18, 0x8f,
        0xf0, 0x19, 0xfb, 0x46, 0xf6, 0xa5, 0x77, 0x0e, 0x90, 0x38, 0x8b, 0x15
    },
    {
        0x80, 0x8c, 0xa7, 0x24, 0x97, 0xf9, 0xd3, 0x52, 0xb0, 0x69, 0x9d, 0x4b,
        0xa4, 0x19, 0x4a, 0xb1, 0x46, 0x53, 0x3a, 0xc8, 0xe4, 0x02, 0x69, 0xf2,
        0xe7, 0xb6, 0x1d, 0x33, 0x51, 0xcc, 0x14, 0x40, 0x4a, 0xb0, 0xe7, 0x58,
        0x84, 0xba, 0xc2, 0x14, 0x58, 0x6b, 0xb9, 0xdc, 0x50, 0x98, 0x67, 0x01
    },
    {
        0xb1, 0xa8, 0xc0, 0x06, 0xb3, 0x2e, 0xa7, 0x8a, 0x6a, 0x12, 0x88, 0x00,
        0x65, 0x88, 0x9c, 0x5d, 0x35, 0xee, 0xe5, 0x51, 0x0b, 0x62, 0xf8, 0x67,
        0xe5, 0xef, 0x15, 0x1f, 0x23, 0x02, 0x74, 0x08, 0x9c, 0xc8, 0xba, 0x27,
        0x5d, 0x32, 0x19, 0x6f, 0x6d, 0x5d, 0x72, 0x5e, 0x15, 0xde, 0x30, 0xc3
    },
    {
        0xfd, 0xce, 0xf5, 0x65, 0x45, 0x84, 0xfb, 0x8c, 0x79, 0xa4, 0x6c, 0x1b,
        0x0e, 0x1b, 0xfd, 0x26, 0xa2, 0x53, 0xf4, 0x4e, 0x00, 0x4d, 0x4b, 0x0b,
        0x24, 0x6d, 0x35, 0x35, 0xd9, 0x97, 0x70, 0xc5, 0xf4, 0xee, 0xe3, 0xba,
        0x31, 0x1e, 0x2a, 0x42, 0xcb, 0xdf, 0x40, 0xb1, 0x14, 0xb8, 0x53, 0xce
    },
    {
        0xbb, 0xb3, 0x26, 0x7c, 0x22, 0x21, 0x9b, 0x72, 0x32, 0xa1, 0x97, 0xfb,
        0x78, 0x8c, 0xbe, 0x3d, 0x71, 0x45, 0xb8, 0xf5, 0x24, 0x8f, 0x0f, 0xac,
        0x42, 0x5b, 0x81, 0xe8, 0xd0, 0x71, 0x4a, 0xcb, 0x32, 0x3f, 0x03, 0xfb,
        0xec, 0x6a, 0x1f, 0x76, 0x80, 0x65, 0x01, 0x7a, 0x3d, 0xce, 0xc4, 0xdf
    },
    {
        0x3f, 0xcf, 0x2f, 0x63, 0x94, 0x94, 0x99, 0xfd, 0x04, 0x3a, 0x89, 0x83,
        0xcf, 0x06, 0x05, 0xec, 0x20, 0x3e, 0x5f, 0x51, 0x9d, 0x6e, 0x4a, 0xc6,
        0xf1, 0x2b, 0x37, 0x17, 0x34, 0x72, 0x6e, 0x1d, 0x2a, 0xfd, 0xc7, 0x73,
        0xb5, 0x07, 0x22, 0x81, 0x32, 0x2e, 0x21, 0x85, 0xaf, 0x10, 0xb2, 0x73
    },
    {
        0x52, 0x0c, 0x3d, 0x2e, 0x2d, 0x4a, 0x11, 0xae, 0x96, 0x78, 0xe9, 0x5b,
        0xd8, 0x0f, 0x6c, 0xf4, 0xbd, 0x96, 0x13, 0x55, 0x88, 0xdd, 0xa3, 0x67,
        0x36, 0x86, 0x1e, 0x0b, 0x36, 0x41, 0xec, 0xf6, 0x04, 0xb2, 0xc4, 0x16,
        0xbc, 0x2c, 0xdb, 0x30, 0x02, 0x94, 0xd4, 0x42, 0xbf, 0x38, 0xee, 0x9d
    }
},
{ /* 36 */
    WC_HASH_TYPE_SHA384, 0, 33,
    { 0 }, /* PSK */
    { /* DHE */
        0xd3, 0x00, 0x72, 0x9a, 0xa8, 0xc5, 0xf3, 0xc4,
        0xf1, 0xa0, 0x26, 0x89, 0x65, 0x70, 0xc7, 0x0b,
        0x77, 0xbb, 0xe1, 0x4b, 0x2b, 0xa8, 0x4f, 0xa6,
        0x09, 0x4b, 0xba, 0x45, 0x36, 0x15, 0xee, 0x68,
        0xfd
    },
    { /* Hello 1 */
        0x10, 0x9d, 0x8b, 0xa2, 0x93, 0xe7, 0xd3, 0xb9,
        0xb4, 0x0f, 0xeb, 0x6a, 0xb9, 0x69, 0xcb, 0x39,
        0x16, 0x29, 0xcc, 0xd3, 0xcc, 0x1a, 0x4c, 0x1b,
        0x53, 0x7c, 0x33, 0x88, 0x06, 0xbc, 0x0a, 0x02,
        0xa0, 0xbe, 0x62, 0xc0, 0xe6, 0x5e, 0x97, 0x5b,
        0x6a, 0xa1, 0x98, 0xf3, 0xd2, 0x1e, 0xcd, 0xc5
    },
    { /* Hello 2 */
        0x74, 0xc0, 0x07, 0x2c, 0xc1, 0x63, 0xcc, 0x11,
        0xad, 0x1a, 0x55, 0x63, 0xbc, 0x20, 0x77, 0x96,
        0x30, 0x1c, 0x68, 0x45, 0x1e, 0x9b, 0xa7, 0xb4,
        0xf3, 0x04, 0x45, 0x16, 0x76, 0x55, 0xf9, 0xdf,
        0x4b, 0x2f, 0x1a, 0xdf, 0x5a, 0xb0, 0x93, 0xc9,
        0xab, 0xf5, 0x32, 0x47, 0x79, 0x9c, 0x01, 0xeb
    },
    { /* Finished 1 */
        0x27, 0x08, 0x8e, 0xa5, 0xf1, 0x30, 0xe1, 0xd6,
        0x4f, 0xa2, 0x9e, 0x3b, 0x03, 0x2d, 0x2e, 0xa3,
        0x84, 0x75, 0x51, 0x3a, 0xc3, 0xf6, 0xee, 0x2e,
        0x37, 0x0c, 0xe3, 0x28, 0x46, 0xa5, 0x2d, 0xc7,
        0xf0, 0x64, 0x78, 0x53, 0x66, 0x43, 0x02, 0xa4,
        0x7a, 0x43, 0x66, 0x4b, 0xa7, 0xcb, 0x97, 0x16
    },
    { /* Finished 2 */
        0x1d, 0x0d, 0xf8, 0xe1, 0x81, 0xa5, 0xbd, 0xa8,
        0x6f, 0x9d, 0x01, 0xa4, 0x9a, 0x92, 0xe2, 0xef,
        0x08, 0xab, 0xef, 0x3e, 0x2d, 0xd4, 0x82, 0xac,
        0x68, 0x9d, 0xe0, 0x54, 0x17, 0xde, 0x1a, 0xed,
        0x57, 0xcb, 0xd9, 0x2d, 0xc8, 0xbc, 0x93, 0xe6,
        0xa3, 0xec, 0xde, 0xee, 0xa1, 0x1c, 0x41, 0x85
    },
    {
        0x7f, 0x1f, 0xe6, 0x7b, 0xd8, 0xf5, 0x2b, 0x37, 0xbe, 0xb7, 0xd0, 0x37,
        0xce, 0x46, 0xad, 0x04, 0x2f, 0xc7, 0xdb, 0xc9, 0x9a, 0xb6, 0x00, 0x3f,
        0xc1, 0x97, 0xe9, 0x5c, 0x5e, 0x14, 0xd1, 0x38, 0x4d, 0x55, 0xe1, 0x07,
        0xb5, 0x85, 0x6d, 0xfa, 0xa7, 0x66, 0xad, 0xfa, 0xb6, 0xad, 0x29, 0x44
    },
    {
        0x4e, 0x6b, 0x20, 0x99, 0x55, 0x1b, 0x21, 0x89, 0xb6, 0x70, 0xdb, 0xe8,
        0xa7, 0x16, 0x55, 0xf2, 0x93, 0x13, 0x90, 0x7d, 0xfa, 0x62, 0x65, 0x53,
        0xa0, 0x97, 0xe9, 0xb4, 0xc0, 0xf1, 0xc9, 0x1a, 0x67, 0xdd, 0xca, 0x57,
        0xbc, 0xca, 0x39, 0xe6, 0x39, 0x6b, 0x63, 0x47, 0x25, 0x08, 0x3a, 0xd7
    },
    {
        0x35, 0x0d, 0xac, 0xd8, 0x10, 0x6a, 0x46, 0x50, 0x66, 0xae, 0x02, 0xc9,
        0xde, 0x13, 0x48, 0xce, 0x53, 0xd4, 0x92, 0x62, 0xc5, 0x65, 0x10, 0x08,
        0xc2, 0xc2, 0x82, 0xed, 0x9d, 0xc9, 0x6f, 0xa8, 0xc3, 0xc1, 0x0b, 0x7c,
        0xe1, 0x97, 0x85, 0xd6, 0x46, 0x29, 0x0e, 0x42, 0x51, 0xc1, 0x35, 0xcf
    },
    {
        0x3d, 0x5d, 0x84, 0xbd, 0x16, 0x46, 0x34, 0xb3, 0xf6, 0x31, 0x49, 0x3e,
        0x8d, 0xdc, 0xcb, 0x8c, 0x6a, 0x42, 0xf4, 0x88, 0xfc, 0x19, 0xfa, 0xa2,
        0x25, 0xc7, 0xa0, 0xa4, 0xca, 0xf0, 0xea, 0x2d, 0xe8, 0xc4, 0x02, 0x14,
        0x63, 0xfb, 0xd3, 0x7b, 0x51, 0x1c, 0xce, 0xca, 0xa3, 0xc3, 0xe4, 0xa5
    },
    {
        0x7c, 0x3a, 0x55, 0x92, 0x2e, 0xdd, 0x75, 0xdd, 0x76, 0x54, 0x4a, 0x9f,
        0xd0, 0xa2, 0x88, 0x83, 0xe9, 0x27, 0xda, 0x30, 0xe9, 0x96, 0x58, 0xc5,
        0xb7, 0x56, 0xfc, 0x4b, 0xb8, 0x5d, 0xee, 0x46, 0x70, 0x4e, 0x1b, 0x06,
        0x86, 0xaf, 0x48, 0x5c, 0x17, 0x35, 0xfa, 0x69, 0xc2, 0x4d, 0xfb, 0x09
    },
    {
        0x00, 0x0e, 0x28, 0x51, 0xc1, 0x7f, 0x41, 0x89, 0x6f, 0x9a, 0xca, 0x15,
        0xee, 0xed, 0x43, 0xca, 0x6d, 0x65, 0x6f, 0x51, 0x18, 0x6c, 0x08, 0x4b,
        0x77, 0xca, 0x75, 0xc4, 0xc3, 0xde, 0x29, 0x41, 0x8b, 0xaf, 0xa7, 0x1c,
        0x28, 0x37, 0xa0, 0xa0, 0x74, 0x8e, 0x09, 0x42, 0x7a, 0x1b, 0x68, 0xdb
    },
    {
        0x14, 0x8f, 0xab, 0x28, 0x64, 0xea, 0x45, 0x88, 0xdb, 0xc1, 0xc6, 0xa0,
        0x48, 0xdf, 0x15, 0xd0, 0x28, 0x07, 0x2d, 0x6c, 0xb8, 0x42, 0xbb, 0x60,
        0x02, 0x08, 0x9e, 0x29, 0x9b, 0x8d, 0xd6, 0x1c, 0xaf, 0xf2, 0x1a, 0xdc,
        0xf0, 0x78, 0x0b, 0x4d, 0x90, 0xa1, 0x0c, 0xb3, 0x13, 0xde, 0xca, 0x5a
    },
    {
        0x4d, 0x80, 0x7d, 0x0b, 0xb9, 0x00, 0x6f, 0x65, 0x51, 0x65, 0x23, 0xde,
        0x72, 0xdc, 0x4f, 0x04, 0xa5, 0xa2, 0x90, 0x45, 0x51, 0x9e, 0xd0, 0x3a,
        0xe4, 0xd7, 0x78, 0xa3, 0x0f, 0x2d, 0x65, 0x12, 0xad, 0xc8, 0x92, 0x30,
        0x79, 0x9d, 0x9d, 0x08, 0x7a, 0x9c, 0x9f, 0x83, 0xb1, 0xca, 0x59, 0x56
    }
},
{ /* 41 */
    WC_HASH_TYPE_SHA384, 33, 0,
    { /* PSK */
        0xa4, 0x8b, 0x1b, 0x5f, 0xd0, 0xea, 0x75, 0x62,
        0x06, 0x4d, 0x68, 0x40, 0x85, 0x20, 0x45, 0x95,
        0x4a, 0x00, 0xca, 0x05, 0xeb, 0xd4, 0x1d, 0x48,
        0x81, 0x89, 0xe8, 0x86, 0x43, 0xfa, 0x28, 0x17,
        0x12
    },
    { 0 }, /* DHE */
    { /* Hello 1 */
        0x03, 0x7c, 0x33, 0x75, 0xdc, 0xc5, 0x46, 0x3a,
        0x0d, 0x56, 0xc6, 0xfb, 0xab, 0x1e, 0x1d, 0xda,
        0x59, 0xc2, 0xb2, 0xb1, 0x7c, 0x48, 0x9b, 0x06,
        0x0a, 0x5a, 0xbb, 0xf8, 0x98, 0x53, 0x78, 0x2d,
        0xd2, 0xcc, 0x87, 0x68, 0x25, 0xdd, 0x88, 0x22,
        0xcd, 0xb7, 0x74, 0x55, 0x21, 0xf9, 0x34, 0x98
    },
    { /* Hello 2 */
        0x03, 0xb4, 0xfb, 0xcc, 0x28, 0x2c, 0xc1, 0x70,
        0x42, 0x73, 0x57, 0xac, 0xdb, 0x47, 0x71, 0xf6,
        0x2e, 0x11, 0x8a, 0x5b, 0x47, 0x2f, 0x02, 0x54,
        0x95, 0x34, 0xed, 0x5f, 0x19, 0xc1, 0x75, 0xe0,
        0x76, 0xad, 0xb0, 0x90, 0x57, 0xcd, 0xfd, 0xd7,
        0x58, 0x1f, 0x0d, 0x6b, 0x9e, 0x51, 0x3c, 0x08
    },
    { /* Finished 1 */
        0x2b, 0x50, 0xd9, 0xa7, 0x43, 0x24, 0xda, 0x2c,
        0x7a, 0xaa, 0x0e, 0x37, 0xd7, 0x6b, 0x2c, 0xab,
        0x8e, 0xb2, 0xfe, 0x31, 0x1b, 0xa8, 0x12, 0x59,
        0x5b, 0x7b, 0xdc, 0x3e, 0xa7, 0x86, 0xa5, 0x48,
        0xe4, 0x46, 0x2b, 0x4c, 0xc1, 0x66, 0x4b, 0xf3,
        0x2a, 0x99, 0x93, 0x08, 0xbc, 0x3d, 0x08, 0x76
    },
    { /* Finished 2 */
        0x7c, 0x34, 0xc8, 0x56, 0x17, 0xf1, 0x62, 0x1c,
        0x9f, 0x0b, 0xeb, 0xfd, 0x69, 0x72, 0x51, 0xc5,
        0xfa, 0x74, 0x87, 0xc9, 0xbd, 0x50, 0xe9, 0x48,
        0xa7, 0x3c, 0x94, 0x3e, 0x06, 0x7d, 0xe8, 0x8e,
        0xc1, 0xd1, 0x08, 0x1f, 0x5d, 0x48, 0x8a, 0x25,
        0xfc, 0xea, 0xe7, 0xd9, 0xd4, 0xd0, 0xf9, 0xad
    },
    {
        0x4b, 0x0b, 0xed, 0xb9, 0xc8, 0xb8, 0xa8, 0x1e, 0xb0, 0x81, 0x76, 0xd5,
        0x33, 0x22, 0x71, 0x33, 0x3a, 0x85, 0x19, 0x67, 0x7e, 0x91, 0x37, 0xf2,
        0xa6, 0x11, 0x22, 0xdf, 0x41, 0x04, 0x3d, 0xa9, 0x13, 0xb9, 0xb2, 0xb1,
        0xbb, 0xd8, 0xef, 0x23, 0x7c, 0xc2, 0xab, 0x70, 0x1b, 0x51, 0x9f, 0xc9
    },
    {
        0xeb, 0x96, 0x10, 0x8c, 0x7d, 0x92, 0xea, 0x80, 0x86, 0xb2, 0xf8, 0x27,
        0xf2, 0x9a, 0x09, 0xc1, 0x7c, 0x09, 0x43, 0xbc, 0xfe, 0xc8, 0x75, 0xe0,
        0x97, 0xe7, 0x6d, 0xd5, 0xb2, 0x3c, 0xed, 0x12, 0xb7, 0x74, 0x0e, 0xe3,
        0xb6, 0xe0, 0xba, 0xe1, 0x8d, 0x89, 0xcf, 0x4f, 0x57, 0xf6, 0x6d, 0x90
    },
    {
        0x22, 0xb0, 0x39, 0x34, 0xb6, 0x6c, 0x2d, 0x7a, 0x97, 0x1c, 0x5d, 0xcc,
        0x78, 0x84, 0x71, 0xbb, 0xc6, 0x7b, 0xb6, 0xbc, 0xcc, 0x0b, 0xf8, 0xac,
        0x8e, 0xd7, 0x20, 0xbd, 0xbe, 0x32, 0xf0, 0xd6, 0xe9, 0x69, 0x13, 0xf2,
        0x9a, 0xce, 0xfe, 0x86, 0xd3, 0xee, 0xba, 0x69, 0x51, 0xb6, 0x77, 0x56
    },
    {
        0x16, 0xfd, 0xda, 0xf3, 0x5e, 0xb9, 0xa6, 0x17, 0x24, 0xb2, 0x16, 0x9f,
        0xb6, 0x59, 0x13, 0x0f, 0x25, 0x5a, 0xf1, 0x5b, 0x5f, 0xe4, 0x54, 0x2a,
        0xa7, 0xbf, 0x29, 0xaf, 0x5a, 0x77, 0xf4, 0x4f, 0x25, 0xba, 0x94, 0xad,
        0x6b, 0x91, 0x3b, 0xe7, 0xd5, 0x73, 0x0d, 0xff, 0xaa, 0xe3, 0x72, 0x2c
    },
    {
        0x22, 0xb4, 0x94, 0xc0, 0x53, 0xd7, 0x82, 0x06, 0x38, 0x9d, 0x4a, 0xa0,
        0x3f, 0xf1, 0x5f, 0x6e, 0x23, 0x8d, 0x09, 0x62, 0xbf, 0x6f, 0x7c, 0x84,
        0xc6, 0x3e, 0x15, 0xad, 0x18, 0x37, 0x76, 0x29, 0xc7, 0xd6, 0x68, 0x0c,
        0x1e, 0xc6, 0x93, 0x31, 0xef, 0x85, 0x69, 0x30, 0x68, 0xf0, 0x1e, 0x37
    },
    {
        0x6d, 0x4d, 0x20, 0xaf, 0x47, 0xe8, 0x1b, 0xfa, 0xd0, 0xb6, 0xc8, 0x97,
        0xd1, 0x03, 0xfc, 0x9d, 0x59, 0xa0, 0x68, 0x9d, 0xe9, 0x17, 0x8b, 0xce,
        0x48, 0x2c, 0x77, 0x8a, 0x22, 0x4b, 0x5c, 0x54, 0x22, 0xa1, 0x15, 0x12,
        0xe1, 0x07, 0x8e, 0x15, 0xd8, 0x7b, 0x16, 0x65, 0x99, 0x6b, 0xcb, 0x71
    },
    {
        0x79, 0x64, 0x79, 0xdd, 0x75, 0x5c, 0x6f, 0x98, 0xac, 0x03, 0xe0, 0xcd,
        0x92, 0xba, 0x0e, 0x2d, 0xb4, 0xd1, 0x8b, 0x97, 0xd0, 0x85, 0xbb, 0x2e,
        0x4f, 0x26, 0x93, 0xf5, 0x1d, 0xf3, 0xd2, 0x43, 0x4f, 0xd2, 0x47, 0xaa,
        0x91, 0x1e, 0xf3, 0x67, 0x10, 0x18, 0x2c, 0xb9, 0x01, 0xba, 0x10, 0x9f
    },
    {
        0x79, 0xb6, 0x9c, 0xbe, 0xf1, 0x6a, 0xb0, 0x92, 0xa0, 0x29, 0x52, 0x61,
        0xf1, 0xcd, 0x3a, 0x67, 0xe1, 0x6b, 0xb8, 0x9d, 0x0d, 0x95, 0xb6, 0x03,
        0x80, 0x1f, 0xd5, 0x75, 0xb6, 0x1d, 0x79, 0x02, 0x93, 0x43, 0x77, 0xa7,
        0x9d, 0x2f, 0xc3, 0x84, 0xc6, 0x83, 0x76, 0x16, 0x06, 0x98, 0x7b, 0x79
    }
},
#endif /* WOLFSSL_SHA384 */
};

static const char protocolLabel[] = "tls13 ";
static const char ceTrafficLabel[] = "c e traffic";
static const char eExpMasterLabel[] = "e exp master";
static const char cHsTrafficLabel[] = "c hs traffic";
static const char sHsTrafficLabel[] = "s hs traffic";
static const char cAppTrafficLabel[] = "c ap traffic";
static const char sAppTrafficLabel[] = "s ap traffic";
static const char expMasterLabel[] = "exp master";
static const char resMasterLabel[] = "res master";
static const char derivedLabel[] = "derived";


WOLFSSL_TEST_SUBROUTINE wc_test_ret_t tls13_kdf_test(void)
{
    wc_test_ret_t ret = 0;
    word32 i;
    word32 tc = sizeof(tls13KdfTestVectors)/sizeof(Tls13KdfTestVector);
    const Tls13KdfTestVector* tv = NULL;

    for (i = 0, tv = tls13KdfTestVectors; i < tc; i++, tv++) {
        byte output[WC_MAX_DIGEST_SIZE];
        byte secret[WC_MAX_DIGEST_SIZE];
        byte salt[WC_MAX_DIGEST_SIZE];
        byte zeroes[WC_MAX_DIGEST_SIZE];
        byte hashZero[WC_MAX_DIGEST_SIZE];
        int hashAlgSz;

        XMEMSET(zeroes, 0, sizeof zeroes);

        hashAlgSz = wc_HashGetDigestSize(tv->hashAlg);
        if (hashAlgSz == BAD_FUNC_ARG) break;
        ret = wc_Hash(tv->hashAlg, NULL, 0, hashZero, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Extract(secret, NULL, 0,
                (tv->pskSz == 0) ? zeroes : (byte*)tv->psk,
                tv->pskSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)ceTrafficLabel, (word32)XSTRLEN(ceTrafficLabel),
                tv->hashHello1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->clientEarlyTrafficSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)eExpMasterLabel, (word32)XSTRLEN(eExpMasterLabel),
                tv->hashHello1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->earlyExporterMasterSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(salt, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)derivedLabel, (word32)XSTRLEN(derivedLabel),
                hashZero, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Extract(secret, salt, hashAlgSz,
                (tv->dheSz == 0) ? zeroes : (byte*)tv->dhe,
                tv->dheSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)cHsTrafficLabel, (word32)XSTRLEN(cHsTrafficLabel),
                tv->hashHello2, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->clientHandshakeTrafficSecret,
                output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)sHsTrafficLabel, (word32)XSTRLEN(sHsTrafficLabel),
                tv->hashHello2, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->serverHandshakeTrafficSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(salt, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)derivedLabel, (word32)XSTRLEN(derivedLabel),
                hashZero, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Extract(secret, salt, hashAlgSz,
                zeroes, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)cAppTrafficLabel, (word32)XSTRLEN(cAppTrafficLabel),
                tv->hashFinished1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->clientApplicationTrafficSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)sAppTrafficLabel, (word32)XSTRLEN(sAppTrafficLabel),
                tv->hashFinished1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->serverApplicationTrafficSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)expMasterLabel, (word32)XSTRLEN(expMasterLabel),
                tv->hashFinished1, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->exporterMasterSecret, output, hashAlgSz);
        if (ret != 0) break;

        ret = wc_Tls13_HKDF_Expand_Label(output, hashAlgSz,
                secret, hashAlgSz,
                (byte*)protocolLabel, (word32)XSTRLEN(protocolLabel),
                (byte*)resMasterLabel, (word32)XSTRLEN(resMasterLabel),
                tv->hashFinished2, hashAlgSz, tv->hashAlg);
        if (ret != 0) break;

        ret = XMEMCMP(tv->resumptionMasterSecret, output, hashAlgSz);
        if (ret != 0) break;
    }

    return ret;
}

#endif /* WOLFSSL_TLS13 */

static const int fiducial2 = WC_TEST_RET_LN; /* source code reference point --
                                              * see print_fiducials() below.
                                              */

#if defined(HAVE_ECC) && defined(HAVE_X963_KDF)

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t x963kdf_test(void)
{
    wc_test_ret_t ret;
    byte kek[128];

#ifndef NO_SHA
    /* SHA-1, COUNT = 0
     * shared secret length: 192
     * SharedInfo length: 0
     * key data length: 128
     */
    WOLFSSL_SMALL_STACK_STATIC const byte Z[] = {
        0x1c, 0x7d, 0x7b, 0x5f, 0x05, 0x97, 0xb0, 0x3d,
        0x06, 0xa0, 0x18, 0x46, 0x6e, 0xd1, 0xa9, 0x3e,
        0x30, 0xed, 0x4b, 0x04, 0xdc, 0x64, 0xcc, 0xdd
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify[] = {
        0xbf, 0x71, 0xdf, 0xfd, 0x8f, 0x4d, 0x99, 0x22,
        0x39, 0x36, 0xbe, 0xb4, 0x6f, 0xee, 0x8c, 0xcc
    };
#endif

#ifndef NO_SHA256
    /* SHA-256, COUNT = 3
     * shared secret length: 192
     * SharedInfo length: 0
     * key data length: 128
     */
    WOLFSSL_SMALL_STACK_STATIC const byte Z2[] = {
        0xd3, 0x8b, 0xdb, 0xe5, 0xc4, 0xfc, 0x16, 0x4c,
        0xdd, 0x96, 0x7f, 0x63, 0xc0, 0x4f, 0xe0, 0x7b,
        0x60, 0xcd, 0xe8, 0x81, 0xc2, 0x46, 0x43, 0x8c
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify2[] = {
        0x5e, 0x67, 0x4d, 0xb9, 0x71, 0xba, 0xc2, 0x0a,
        0x80, 0xba, 0xd0, 0xd4, 0x51, 0x4d, 0xc4, 0x84
    };
#endif

#ifdef WOLFSSL_SHA512
    /* SHA-512, COUNT = 0
     * shared secret length: 192
     * SharedInfo length: 0
     * key data length: 128
     */
    WOLFSSL_SMALL_STACK_STATIC const byte Z3[] = {
        0x87, 0xfc, 0x0d, 0x8c, 0x44, 0x77, 0x48, 0x5b,
        0xb5, 0x74, 0xf5, 0xfc, 0xea, 0x26, 0x4b, 0x30,
        0x88, 0x5d, 0xc8, 0xd9, 0x0a, 0xd8, 0x27, 0x82
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify3[] = {
        0x94, 0x76, 0x65, 0xfb, 0xb9, 0x15, 0x21, 0x53,
        0xef, 0x46, 0x02, 0x38, 0x50, 0x6a, 0x02, 0x45
    };

    /* SHA-512, COUNT = 0
     * shared secret length: 521
     * SharedInfo length: 128
     * key data length: 1024
     */
    WOLFSSL_SMALL_STACK_STATIC const byte Z4[] = {
        0x00, 0xaa, 0x5b, 0xb7, 0x9b, 0x33, 0xe3, 0x89,
        0xfa, 0x58, 0xce, 0xad, 0xc0, 0x47, 0x19, 0x7f,
        0x14, 0xe7, 0x37, 0x12, 0xf4, 0x52, 0xca, 0xa9,
        0xfc, 0x4c, 0x9a, 0xdb, 0x36, 0x93, 0x48, 0xb8,
        0x15, 0x07, 0x39, 0x2f, 0x1a, 0x86, 0xdd, 0xfd,
        0xb7, 0xc4, 0xff, 0x82, 0x31, 0xc4, 0xbd, 0x0f,
        0x44, 0xe4, 0x4a, 0x1b, 0x55, 0xb1, 0x40, 0x47,
        0x47, 0xa9, 0xe2, 0xe7, 0x53, 0xf5, 0x5e, 0xf0,
        0x5a, 0x2d
    };

    WOLFSSL_SMALL_STACK_STATIC const byte info4[] = {
        0xe3, 0xb5, 0xb4, 0xc1, 0xb0, 0xd5, 0xcf, 0x1d,
        0x2b, 0x3a, 0x2f, 0x99, 0x37, 0x89, 0x5d, 0x31
    };

    WOLFSSL_SMALL_STACK_STATIC const byte verify4[] = {
        0x44, 0x63, 0xf8, 0x69, 0xf3, 0xcc, 0x18, 0x76,
        0x9b, 0x52, 0x26, 0x4b, 0x01, 0x12, 0xb5, 0x85,
        0x8f, 0x7a, 0xd3, 0x2a, 0x5a, 0x2d, 0x96, 0xd8,
        0xcf, 0xfa, 0xbf, 0x7f, 0xa7, 0x33, 0x63, 0x3d,
        0x6e, 0x4d, 0xd2, 0xa5, 0x99, 0xac, 0xce, 0xb3,
        0xea, 0x54, 0xa6, 0x21, 0x7c, 0xe0, 0xb5, 0x0e,
        0xef, 0x4f, 0x6b, 0x40, 0xa5, 0xc3, 0x02, 0x50,
        0xa5, 0xa8, 0xee, 0xee, 0x20, 0x80, 0x02, 0x26,
        0x70, 0x89, 0xdb, 0xf3, 0x51, 0xf3, 0xf5, 0x02,
        0x2a, 0xa9, 0x63, 0x8b, 0xf1, 0xee, 0x41, 0x9d,
        0xea, 0x9c, 0x4f, 0xf7, 0x45, 0xa2, 0x5a, 0xc2,
        0x7b, 0xda, 0x33, 0xca, 0x08, 0xbd, 0x56, 0xdd,
        0x1a, 0x59, 0xb4, 0x10, 0x6c, 0xf2, 0xdb, 0xbc,
        0x0a, 0xb2, 0xaa, 0x8e, 0x2e, 0xfa, 0x7b, 0x17,
        0x90, 0x2d, 0x34, 0x27, 0x69, 0x51, 0xce, 0xcc,
        0xab, 0x87, 0xf9, 0x66, 0x1c, 0x3e, 0x88, 0x16
    };
#endif

#ifndef NO_SHA
    ret = wc_X963_KDF(WC_HASH_TYPE_SHA, Z, sizeof(Z), NULL, 0,
                      kek, sizeof(verify));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(verify, kek, sizeof(verify)) != 0)
        return WC_TEST_RET_ENC_NC;
#endif

#ifndef NO_SHA256
    ret = wc_X963_KDF(WC_HASH_TYPE_SHA256, Z2, sizeof(Z2), NULL, 0,
                      kek, sizeof(verify2));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(verify2, kek, sizeof(verify2)) != 0)
        return WC_TEST_RET_ENC_NC;
#endif

#ifdef WOLFSSL_SHA512
    ret = wc_X963_KDF(WC_HASH_TYPE_SHA512, Z3, sizeof(Z3), NULL, 0,
                      kek, sizeof(verify3));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(verify3, kek, sizeof(verify3)) != 0)
        return WC_TEST_RET_ENC_NC;

    ret = wc_X963_KDF(WC_HASH_TYPE_SHA512, Z4, sizeof(Z4), info4,
                      sizeof(info4), kek, sizeof(verify4));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    if (XMEMCMP(verify4, kek, sizeof(verify4)) != 0)
        return WC_TEST_RET_ENC_NC;
#endif

    return 0;
}

#endif /* HAVE_X963_KDF */

#if defined(HAVE_HPKE) && (defined(HAVE_ECC) || defined(HAVE_CURVE25519)) && \
    defined(HAVE_AESGCM)

static wc_test_ret_t hpke_test_single(Hpke* hpke)
{
    wc_test_ret_t ret = 0;
    int rngRet = 0;
    WC_RNG rng[1];
    const char* start_text = "this is a test";
    const char* info_text = "info";
    const char* aad_text = "aad";
    byte ciphertext[MAX_HPKE_LABEL_SZ];
    byte plaintext[MAX_HPKE_LABEL_SZ];
    void* receiverKey = NULL;
    void* ephemeralKey = NULL;
#ifdef WOLFSSL_SMALL_STACK
    byte *pubKey = NULL; /* public key */
    word16 pubKeySz = (word16)HPKE_Npk_MAX;
#else
    byte pubKey[HPKE_Npk_MAX]; /* public key */
    word16 pubKeySz = (word16)sizeof(pubKey);
#endif

    rngRet = ret = wc_InitRng(rng);

    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

#ifdef WOLFSSL_SMALL_STACK
    if (ret == 0) {
        pubKey = (byte *)XMALLOC(pubKeySz, HEAP_HINT,
                                    DYNAMIC_TYPE_TMP_BUFFER);
        if (pubKey == NULL)
            ret = WC_TEST_RET_ENC_EC(MEMORY_E);
    }
#endif

    /* generate the keys */
    if (ret == 0) {
        ret = wc_HpkeGenerateKeyPair(hpke, &ephemeralKey, rng);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = wc_HpkeGenerateKeyPair(hpke, &receiverKey, rng);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    /* seal */
    if (ret == 0) {
        ret = wc_HpkeSealBase(hpke, ephemeralKey, receiverKey,
            (byte*)info_text, (word32)XSTRLEN(info_text),
            (byte*)aad_text, (word32)XSTRLEN(aad_text),
            (byte*)start_text, (word32)XSTRLEN(start_text),
            ciphertext);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    /* export ephemeral key */
    if (ret == 0) {
        ret = wc_HpkeSerializePublicKey(hpke, ephemeralKey, pubKey, &pubKeySz);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    /* open with exported ephemeral key */
    if (ret == 0) {
        ret = wc_HpkeOpenBase(hpke, receiverKey, pubKey, pubKeySz,
            (byte*)info_text, (word32)XSTRLEN(info_text),
            (byte*)aad_text, (word32)XSTRLEN(aad_text),
            ciphertext, (word32)XSTRLEN(start_text),
            plaintext);
        if (ret != 0)
            ret = WC_TEST_RET_ENC_EC(ret);
    }

    if (ret == 0) {
        ret = XMEMCMP(plaintext, start_text, XSTRLEN(start_text));
        if (ret != 0)
            ret = WC_TEST_RET_ENC_NC;
    }

    if (ephemeralKey != NULL)
        wc_HpkeFreeKey(hpke, hpke->kem, ephemeralKey, hpke->heap);

    if (receiverKey != NULL)
        wc_HpkeFreeKey(hpke, hpke->kem, receiverKey, hpke->heap);

#ifdef WOLFSSL_SMALL_STACK
    if (pubKey != NULL)
        XFREE(pubKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    if (rngRet == 0)
        wc_FreeRng(rng);

    return ret;
}

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t hpke_test(void)
{
    wc_test_ret_t ret = 0;
    Hpke hpke[1];

#if defined(HAVE_ECC)
    #if defined(WOLFSSL_SHA224) || !defined(NO_SHA256)
    /* p256 */
    ret = wc_HpkeInit(hpke, DHKEM_P256_HKDF_SHA256, HKDF_SHA256,
        HPKE_AES_128_GCM, NULL);

    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = hpke_test_single(hpke);

    if (ret != 0)
        return ret;
    #endif

    #if defined(WOLFSSL_SHA384) && \
        (defined(HAVE_ECC384) || defined(HAVE_ALL_CURVES))
    /* p384 */
    ret = wc_HpkeInit(hpke, DHKEM_P384_HKDF_SHA384, HKDF_SHA384,
        HPKE_AES_128_GCM, NULL);

    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = hpke_test_single(hpke);

    if (ret != 0)
        return ret;
    #endif

    #if (defined(WOLFSSL_SHA384) || defined(WOLFSSL_SHA512)) && \
        (defined(HAVE_ECC521) || defined(HAVE_ALL_CURVES))
    /* p521 */
    ret = wc_HpkeInit(hpke, DHKEM_P521_HKDF_SHA512, HKDF_SHA512,
        HPKE_AES_128_GCM, NULL);

    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = hpke_test_single(hpke);

    if (ret != 0)
        return ret;
    #endif
#endif

#if defined(HAVE_CURVE25519)
    /* test with curve25519 and aes256 */
    ret = wc_HpkeInit(hpke, DHKEM_X25519_HKDF_SHA256, HKDF_SHA256,
        HPKE_AES_256_GCM, NULL);

    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = hpke_test_single(hpke);

    if (ret != 0)
        return ret;
#endif

    return ret;
/* x448 and chacha20 are unimplemented */
}
#endif /* HAVE_HPKE && HAVE_ECC && HAVE_AESGCM */

#if defined(WC_SRTP_KDF)
typedef struct Srtp_Kdf_Tv {
    const unsigned char* key;
    word32 keySz;
    const unsigned char* salt;
    word32 saltSz;
    int kdfIdx;
    const unsigned char* index;
    const unsigned char* ke;
    const unsigned char* ka;
    const unsigned char* ks;
    const unsigned char* index_c;
    const unsigned char* ke_c;
    const unsigned char* ka_c;
    const unsigned char* ks_c;
    word32 keSz;
    word32 kaSz;
    word32 ksSz;
} Srtp_Kdf_Tv;

WOLFSSL_TEST_SUBROUTINE wc_test_ret_t srtpkdf_test(void)
{
    wc_test_ret_t ret = 0;
    /* 128-bit key, kdrIdx = -1 */
    WOLFSSL_SMALL_STACK_STATIC const byte key_0[] = {
        0xc4, 0x80, 0x9f, 0x6d, 0x36, 0x98, 0x88, 0x72,
        0x8e, 0x26, 0xad, 0xb5, 0x32, 0x12, 0x98, 0x90
    };
    WOLFSSL_SMALL_STACK_STATIC const byte salt_0[] = {
        0x0e, 0x23, 0x00, 0x6c, 0x6c, 0x04, 0x4f, 0x56,
        0x62, 0x40, 0x0e, 0x9d, 0x1b, 0xd6
    };
    WOLFSSL_SMALL_STACK_STATIC const byte index_0[] = {
        0x48, 0x71, 0x65, 0x64, 0x9c, 0xca
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ke_0[] = {
        0xdc, 0x38, 0x21, 0x92, 0xab, 0x65, 0x10, 0x8a,
        0x86, 0xb2, 0x59, 0xb6, 0x1b, 0x3a, 0xf4, 0x6f
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ka_0[] = {
        0xb8, 0x39, 0x37, 0xfb, 0x32, 0x17, 0x92, 0xee,
        0x87, 0xb7, 0x88, 0x19, 0x3b, 0xe5, 0xa4, 0xe3,
        0xbd, 0x32, 0x6e, 0xe4
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ks_0[] = {
        0xf1, 0xc0, 0x35, 0xc0, 0x0b, 0x5a, 0x54, 0xa6,
        0x16, 0x92, 0xc0, 0x16, 0x27, 0x6c
    };
    WOLFSSL_SMALL_STACK_STATIC const byte index_c_0[] = {
        0x56, 0xf3, 0xf1, 0x97
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ke_c_0[] = {
        0xab, 0x5b, 0xe0, 0xb4, 0x56, 0x23, 0x5d, 0xcf,
        0x77, 0xd5, 0x08, 0x69, 0x29, 0xba, 0xfb, 0x38
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ka_c_0[] = {
        0xc5, 0x2f, 0xde, 0x0b, 0x80, 0xb0, 0xf0, 0xba,
        0xd8, 0xd1, 0x56, 0x45, 0xcb, 0x86, 0xe7, 0xc7,
        0xc3, 0xd8, 0x77, 0x0e
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ks_c_0[] = {
        0xde, 0xb5, 0xf8, 0x5f, 0x81, 0x33, 0x6a, 0x96,
        0x5e, 0xd3, 0x2b, 0xb7, 0xed, 0xe8
    };
    /* 192-bit key, kdrIdx = 0 */
    WOLFSSL_SMALL_STACK_STATIC const byte key_1[] = {
        0xbb, 0x04, 0x5b, 0x1f, 0x53, 0xc6, 0x93, 0x2c,
        0x2b, 0xa6, 0x88, 0xf5, 0xe3, 0xf2, 0x24, 0x70,
        0xe1, 0x7d, 0x7d, 0xec, 0x8a, 0x93, 0x4d, 0xf2
    };
    WOLFSSL_SMALL_STACK_STATIC const byte salt_1[] = {
        0xe7, 0x22, 0xab, 0x92, 0xfc, 0x7c, 0x89, 0xb6,
        0x53, 0x8a, 0xf9, 0x3c, 0xb9, 0x52
    };
    WOLFSSL_SMALL_STACK_STATIC const byte index_1[] = {
        0xd7, 0x87, 0x8f, 0x33, 0xb1, 0x76
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ke_1[] = {
        0x2c, 0xc8, 0x3e, 0x54, 0xb2, 0x33, 0x89, 0xb3,
        0x71, 0x65, 0x0f, 0x51, 0x61, 0x65, 0xe4, 0x93,
        0x07, 0x4e, 0xb3, 0x47, 0xba, 0x2d, 0x60, 0x60
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ka_1[] = {
        0x2e, 0x80, 0xe4, 0x82, 0x55, 0xa2, 0xbe, 0x6d,
        0xe0, 0x46, 0xcc, 0xc1, 0x75, 0x78, 0x6e, 0x78,
        0xd1, 0xd1, 0x47, 0x08
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ks_1[] = {
        0xe0, 0xc1, 0xe6, 0xaf, 0x1e, 0x8d, 0x8c, 0xfe,
        0xe5, 0x60, 0x70, 0xb5, 0xe6, 0xea
    };
    WOLFSSL_SMALL_STACK_STATIC const byte index_c_1[] = {
        0x40, 0xbf, 0xd4, 0xa9
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ke_c_1[] = {
        0x94, 0x0f, 0x55, 0xce, 0x58, 0xd8, 0x16, 0x65,
        0xf0, 0xfa, 0x46, 0x40, 0x0c, 0xda, 0xb1, 0x11,
        0x9e, 0x69, 0xa0, 0x93, 0x4e, 0xd7, 0xf2, 0x84
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ka_c_1[] = {
        0xf5, 0x41, 0x6f, 0xc2, 0x65, 0xc5, 0xb3, 0xef,
        0xbb, 0x22, 0xc8, 0xfc, 0x6b, 0x00, 0x14, 0xb2,
        0xf3, 0x3b, 0x8e, 0x29
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ks_c_1[] = {
        0x35, 0xb7, 0x42, 0x43, 0xf0, 0x01, 0x01, 0xb4,
        0x68, 0xa1, 0x28, 0x80, 0x37, 0xf0
    };
    /* 256-bit key, kdrIdx = 1 */
    WOLFSSL_SMALL_STACK_STATIC const byte key_2[] = {
        0x10, 0x38, 0x0a, 0xcd, 0xd6, 0x47, 0xab, 0xee,
        0xc0, 0xd4, 0x44, 0xf4, 0x7e, 0x51, 0x36, 0x02,
        0x79, 0xa8, 0x94, 0x80, 0x35, 0x40, 0xed, 0x50,
        0xf4, 0x45, 0x30, 0x3d, 0xb5, 0xf0, 0x2b, 0xbb
    };
    WOLFSSL_SMALL_STACK_STATIC const byte salt_2[] = {
        0xc7, 0x31, 0xf2, 0xc8, 0x40, 0x43, 0xb8, 0x74,
        0x8a, 0x61, 0x84, 0x7a, 0x25, 0x8a
    };
    WOLFSSL_SMALL_STACK_STATIC const byte index_2[] = {
        0x82, 0xf1, 0x84, 0x8c, 0xac, 0x42
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ke_2[] = {
        0xb2, 0x26, 0x60, 0xaf, 0x08, 0x23, 0x14, 0x98,
        0x91, 0xde, 0x5d, 0x87, 0x95, 0x61, 0xca, 0x8f,
        0x0e, 0xce, 0xfb, 0x68, 0x4d, 0xd6, 0x28, 0xcb,
        0x28, 0xe2, 0x27, 0x20, 0x2d, 0xff, 0x64, 0xbb
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ka_2[] = {
        0x12, 0x6f, 0x52, 0xe8, 0x07, 0x7f, 0x07, 0x84,
        0xa0, 0x61, 0x96, 0xf8, 0xee, 0x4d, 0x05, 0x57,
        0x65, 0xc7, 0x50, 0xc1
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ks_2[] = {
        0x18, 0x5a, 0x59, 0xe5, 0x91, 0x4d, 0xc9, 0x6c,
        0xfa, 0x5b, 0x36, 0x06, 0x8c, 0x9a
    };
    WOLFSSL_SMALL_STACK_STATIC const byte index_c_2[] = {
        0x31, 0x2d, 0x58, 0x15
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ke_c_2[] = {
        0x14, 0xf2, 0xc8, 0x25, 0x02, 0x79, 0x22, 0xa1,
        0x96, 0xb6, 0xf7, 0x07, 0x76, 0xa6, 0xa3, 0xc4,
        0x37, 0xdf, 0xa0, 0xf8, 0x78, 0x93, 0x2c, 0xfa,
        0xea, 0x35, 0xf0, 0xf3, 0x3f, 0x32, 0x6e, 0xfd
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ka_c_2[] = {
        0x6e, 0x3d, 0x4a, 0x99, 0xea, 0x2f, 0x9d, 0x13,
        0x4a, 0x1e, 0x71, 0x2e, 0x15, 0xc0, 0xca, 0xb6,
        0x35, 0x78, 0xdf, 0xa4
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ks_c_2[] = {
        0xae, 0xe4, 0xec, 0x18, 0x31, 0x70, 0x5d, 0x3f,
        0xdc, 0x97, 0x89, 0x88, 0xfd, 0xff
    };
    /* 128-bit key, kdrIdx = 8 */
    WOLFSSL_SMALL_STACK_STATIC const byte key_3[] = {
        0x36, 0xb4, 0xde, 0xcb, 0x2e, 0x51, 0x23, 0x76,
        0xe0, 0x27, 0x7e, 0x3e, 0xc8, 0xf6, 0x54, 0x04
    };
    WOLFSSL_SMALL_STACK_STATIC const byte salt_3[] = {
        0x73, 0x26, 0xf4, 0x3f, 0xc0, 0xd9, 0xc6, 0xe3,
        0x2f, 0x92, 0x7d, 0x46, 0x12, 0x76
    };
    WOLFSSL_SMALL_STACK_STATIC const byte index_3[] = {
        0x44, 0x73, 0xb2, 0x2d, 0xb2, 0x60
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ke_3[] = {
        0x79, 0x91, 0x3d, 0x7b, 0x20, 0x5d, 0xea, 0xe2,
        0xeb, 0x46, 0x89, 0x68, 0x5a, 0x06, 0x73, 0x74
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ka_3[] = {
        0x2d, 0x2e, 0x97, 0x4e, 0x76, 0x8c, 0x62, 0xa6,
        0x57, 0x80, 0x13, 0x42, 0x0b, 0x51, 0xa7, 0x66,
        0xea, 0x31, 0x24, 0xe6
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ks_3[] = {
        0xcc, 0xd7, 0x31, 0xf6, 0x3b, 0xf3, 0x89, 0x8a,
        0x5b, 0x7b, 0xb5, 0x8b, 0x4c, 0x3f
    };
    WOLFSSL_SMALL_STACK_STATIC const byte index_c_3[] = {
        0x4a, 0x7d, 0xaa, 0x85
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ke_c_3[] = {
        0x34, 0x99, 0x71, 0xfe, 0x12, 0x93, 0xae, 0x8c,
        0x4a, 0xe9, 0x84, 0xe4, 0x93, 0x53, 0x63, 0x88
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ka_c_3[] = {
        0xa4, 0x53, 0x5e, 0x0a, 0x9c, 0xf2, 0xce, 0x13,
        0xef, 0x7a, 0x13, 0xee, 0x0a, 0xef, 0xba, 0x17,
        0x05, 0x18, 0xe3, 0xed
    };
    WOLFSSL_SMALL_STACK_STATIC const byte ks_c_3[] = {
        0xe1, 0x29, 0x4f, 0x61, 0x30, 0x3c, 0x4d, 0x46,
        0x5f, 0x5c, 0x81, 0x3c, 0x38, 0xb6
    };
    #define SRTP_TV_CNT     4
    Srtp_Kdf_Tv tv[SRTP_TV_CNT] = {
        { key_0, (word32)sizeof(key_0), salt_0, (word32)sizeof(salt_0), -1,
          index_0, ke_0, ka_0, ks_0, index_c_0, ke_c_0, ka_c_0, ks_c_0,
          16, 20, 14 },
        { key_1, (word32)sizeof(key_1), salt_1, (word32)sizeof(salt_1), 0,
          index_1, ke_1, ka_1, ks_1, index_c_1, ke_c_1, ka_c_1, ks_c_1,
          24, 20, 14 },
        { key_2, (word32)sizeof(key_2), salt_2, (word32)sizeof(salt_2), 1,
          index_2, ke_2, ka_2, ks_2, index_c_2, ke_c_2, ka_c_2, ks_c_2,
          32, 20, 14 },
        { key_3, (word32)sizeof(key_3), salt_3, (word32)sizeof(salt_3), 8,
          index_3, ke_3, ka_3, ks_3, index_c_3, ke_c_3, ka_c_3, ks_c_3,
          16, 20, 14 },
    };
    int i;
    int idx;
    unsigned char keyE[32];
    unsigned char keyA[20];
    unsigned char keyS[14];

    for (i = 0; (ret == 0) && (i < SRTP_TV_CNT); i++) {
    #ifndef WOLFSSL_AES_128
        if (tv[i].keySz == AES_128_KEY_SIZE) {
            continue;
        }
    #endif
    #ifndef WOLFSSL_AES_192
        if (tv[i].keySz == AES_192_KEY_SIZE) {
            continue;
        }
    #endif
    #ifndef WOLFSSL_AES_256
        if (tv[i].keySz == AES_256_KEY_SIZE) {
            continue;
        }
    #endif

        ret = wc_SRTP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
            tv[i].kdfIdx, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
            keyS, tv[i].ksSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        if (XMEMCMP(keyE, tv[i].ke, tv[i].keSz) != 0)
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(keyA, tv[i].ka, tv[i].kaSz) != 0)
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(keyS, tv[i].ks, tv[i].ksSz) != 0)
            return WC_TEST_RET_ENC_NC;

        ret = wc_SRTP_KDF_label(tv[i].key, tv[i].keySz, tv[i].salt,
            tv[i].saltSz, tv[i].kdfIdx, tv[i].index, WC_SRTP_LABEL_ENCRYPTION,
            keyE, tv[i].keSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        if (XMEMCMP(keyE, tv[i].ke, tv[i].keSz) != 0)
            return WC_TEST_RET_ENC_NC;

        ret = wc_SRTP_KDF_label(tv[i].key, tv[i].keySz, tv[i].salt,
            tv[i].saltSz, tv[i].kdfIdx, tv[i].index, WC_SRTP_LABEL_MSG_AUTH,
            keyA, tv[i].kaSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        if (XMEMCMP(keyA, tv[i].ka, tv[i].kaSz) != 0)
            return WC_TEST_RET_ENC_NC;

        ret = wc_SRTP_KDF_label(tv[i].key, tv[i].keySz, tv[i].salt,
            tv[i].saltSz, tv[i].kdfIdx, tv[i].index, WC_SRTP_LABEL_SALT, keyS,
            tv[i].ksSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        if (XMEMCMP(keyS, tv[i].ks, tv[i].ksSz) != 0)
            return WC_TEST_RET_ENC_NC;

        ret = wc_SRTCP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
            tv[i].kdfIdx, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
            keyS, tv[i].ksSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        if (XMEMCMP(keyE, tv[i].ke_c, tv[i].keSz) != 0)
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(keyA, tv[i].ka_c, tv[i].kaSz) != 0)
            return WC_TEST_RET_ENC_NC;
        if (XMEMCMP(keyS, tv[i].ks_c, tv[i].ksSz) != 0)
            return WC_TEST_RET_ENC_NC;

        ret = wc_SRTCP_KDF_label(tv[i].key, tv[i].keySz, tv[i].salt,
            tv[i].saltSz, tv[i].kdfIdx, tv[i].index_c,
            WC_SRTCP_LABEL_ENCRYPTION, keyE, tv[i].keSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        if (XMEMCMP(keyE, tv[i].ke_c, tv[i].keSz) != 0)
            return WC_TEST_RET_ENC_NC;

        ret = wc_SRTCP_KDF_label(tv[i].key, tv[i].keySz, tv[i].salt,
            tv[i].saltSz, tv[i].kdfIdx, tv[i].index_c, WC_SRTCP_LABEL_MSG_AUTH,
            keyA, tv[i].kaSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        if (XMEMCMP(keyA, tv[i].ka_c, tv[i].kaSz) != 0)
            return WC_TEST_RET_ENC_NC;

        ret = wc_SRTCP_KDF_label(tv[i].key, tv[i].keySz, tv[i].salt,
            tv[i].saltSz, tv[i].kdfIdx, tv[i].index_c, WC_SRTCP_LABEL_SALT,
            keyS, tv[i].ksSz);
        if (ret != 0)
            return WC_TEST_RET_ENC_EC(ret);
        if (XMEMCMP(keyS, tv[i].ks_c, tv[i].ksSz) != 0)
            return WC_TEST_RET_ENC_NC;
    }

#ifdef WOLFSSL_AES_128
    i = 0;
#elif defined(WOLFSSL_AES_192)
    i = 1;
#else
    i = 2;
#endif
    ret = wc_SRTP_KDF(tv[i].key, 33, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, 33, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(tv[i].key, 15, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, 15, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, 15,
        tv[i].kdfIdx, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, 15,
        tv[i].kdfIdx, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(NULL, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(NULL, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(tv[i].key, tv[i].keySz, NULL, tv[i].saltSz,
            tv[i].kdfIdx, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
            keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, tv[i].keySz, NULL, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        25, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        25, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        -2, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        -2, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index, NULL, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index_c, NULL, tv[i].keSz, keyA, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index, keyE, tv[i].keSz, NULL, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index_c, keyE, tv[i].keSz, NULL, tv[i].kaSz,
        keyS, tv[i].ksSz);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    ret = wc_SRTP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        NULL, tv[i].ksSz);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_SRTCP_KDF(tv[i].key, tv[i].keySz, tv[i].salt, tv[i].saltSz,
        tv[i].kdfIdx, tv[i].index_c, keyE, tv[i].keSz, keyA, tv[i].kaSz,
        NULL, tv[i].ksSz);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    idx = wc_SRTP_KDF_kdr_to_idx(0);
    if (idx != -1)
        return WC_TEST_RET_ENC_NC;
    for (i = 0; i < 32; i++) {
        word32 kdr = 1U << i;
        idx = wc_SRTP_KDF_kdr_to_idx(kdr);
        if (idx != i)
            return WC_TEST_RET_ENC_NC;
    }

    return 0;
}
#endif

#ifdef HAVE_ECC

/* size to use for ECC key gen tests */
#ifndef ECC_KEYGEN_SIZE
    #if !defined(NO_ECC256) || defined(WOLFSSL_SM2)
        #define ECC_KEYGEN_SIZE 32
    #elif defined(HAVE_ECC384)
        #define ECC_KEYGEN_SIZE 48
    #elif defined(HAVE_ECC224)
        #define ECC_KEYGEN_SIZE 28
    #elif defined(HAVE_ECC521)
        #define ECC_KEYGEN_SIZE 66
    #else
        #error No ECC keygen size defined for test
    #endif
#endif
#ifdef BENCH_EMBEDDED
    #define ECC_SHARED_SIZE 128
#else
    #define ECC_SHARED_SIZE MAX_ECC_BYTES
#endif
#if defined(WOLFSSL_ECDSA_DETERMINISTIC_K) || defined(WOLFSSL_ECDSA_DETERMINISTIC_K_VARIANT)
#define HAVE_ECC_DETERMINISTIC_K
#define ECC_DIGEST_SIZE     WC_SHA256_DIGEST_SIZE
#else
#define ECC_DIGEST_SIZE     MAX_ECC_BYTES
#endif
#define ECC_SIG_SIZE        ECC_MAX_SIG_SIZE

#ifdef NO_ECC_SECP
    #define NO_ECC_VECTOR_TEST
#endif

#ifndef NO_ECC_VECTOR_TEST
    #if (defined(HAVE_ECC192) || defined(HAVE_ECC224) ||\
         !defined(NO_ECC256) || defined(HAVE_ECC384) ||\
         defined(HAVE_ECC521) || defined(HAVE_ALL_CURVES))
        #define HAVE_ECC_VECTOR_TEST
    #endif
#endif

#ifdef HAVE_ECC_VECTOR_TEST
typedef struct eccVector {
    const char* msg; /* SHA-1 Encoded Message */
    const char* Qx;
    const char* Qy;
    const char* d; /* Private Key */
    const char* R;
    const char* S;
    const char* curveName;
    word32 msgLen;
    word32 keySize;
#ifndef NO_ASN
    const byte* r;
    word32      rSz;
    const byte* s;
    word32      sSz;
#endif
} eccVector;

#if !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static wc_test_ret_t ecc_test_vector_item(const eccVector* vector)
{
    wc_test_ret_t ret = 0;
    int verify = 0;
    word32  sigSz;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key *userA = (ecc_key *)XMALLOC(sizeof *userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key userA[1];
#endif
    WC_DECLARE_VAR(sig, byte, ECC_SIG_SIZE, HEAP_HINT);
#if !defined(NO_ASN) && !defined(HAVE_SELFTEST)
    word32  sigRawSz, rSz = MAX_ECC_BYTES, sSz = MAX_ECC_BYTES;
    WC_DECLARE_VAR(sigRaw, byte, ECC_SIG_SIZE, HEAP_HINT);
    WC_DECLARE_VAR(r, byte, MAX_ECC_BYTES, HEAP_HINT);
    WC_DECLARE_VAR(s, byte, MAX_ECC_BYTES, HEAP_HINT);
#endif

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
    if (sig == NULL)
        ERROR_OUT(MEMORY_E, done);
#if !defined(NO_ASN) && !defined(HAVE_SELFTEST)
    if (sigRaw == NULL || r == NULL || s == NULL)
        ERROR_OUT(MEMORY_E, done);
#endif
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (userA == NULL)
        ERROR_OUT(MEMORY_E, done);
#endif

    ret = wc_ecc_init_ex(userA, HEAP_HINT, devId);
    if (ret != 0)
        goto done;

    ret = wc_ecc_import_raw(userA, vector->Qx, vector->Qy,
                                                  vector->d, vector->curveName);
    if (ret != 0)
        goto done;

    XMEMSET(sig, 0, ECC_SIG_SIZE);
    sigSz = ECC_SIG_SIZE;
    ret = wc_ecc_rs_to_sig(vector->R, vector->S, sig, &sigSz);
    if (ret != 0)
        goto done;

#if !defined(NO_ASN) && !defined(HAVE_SELFTEST)
    XMEMSET(sigRaw, 0, ECC_SIG_SIZE);
    sigRawSz = ECC_SIG_SIZE;
    ret = wc_ecc_rs_raw_to_sig(vector->r, vector->rSz, vector->s, vector->sSz,
                                                             sigRaw, &sigRawSz);
    if (ret != 0)
        goto done;
    if (sigSz != sigRawSz || XMEMCMP(sig, sigRaw, sigSz) != 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }

    ret = wc_ecc_sig_to_rs(sig, sigSz, r, &rSz, s, &sSz);
    if (ret != 0)
        goto done;
    if (rSz != vector->rSz || XMEMCMP(r, vector->r, rSz) != 0 ||
        sSz != vector->sSz || XMEMCMP(s, vector->s, sSz) != 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }
#endif

#ifdef HAVE_ECC_VERIFY
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_verify_hash(sig, sigSz, (byte*)vector->msg,
                                               vector->msgLen, &verify, userA);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        goto done;
    TEST_SLEEP();

    if (verify != 1)
        ret = WC_TEST_RET_ENC_NC;
#endif

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (userA != NULL) {
        wc_ecc_free(userA);
        XFREE(userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(userA);
#endif

#if !defined(NO_ASN) && !defined(HAVE_SELFTEST)
    WC_FREE_VAR(sigRaw, HEAP_HINT);
    WC_FREE_VAR(r, HEAP_HINT);
    WC_FREE_VAR(s, HEAP_HINT);
#endif
    WC_FREE_VAR(sig, HEAP_HINT);

    return ret;
}

static wc_test_ret_t ecc_test_vector(int keySize)
{
    wc_test_ret_t ret;
    eccVector vec;

    XMEMSET(&vec, 0, sizeof(vec));
    vec.keySize = (word32)keySize;

    switch(keySize) {

#if defined(HAVE_ECC112) || defined(HAVE_ALL_CURVES)
    case 14:
        return 0;
#endif /* HAVE_ECC112 */
#if defined(HAVE_ECC128) || defined(HAVE_ALL_CURVES)
    case 16:
        return 0;
#endif /* HAVE_ECC128 */
#if defined(HAVE_ECC160) || defined(HAVE_ALL_CURVES)
    case 20:
        return 0;
#endif /* HAVE_ECC160 */

#if defined(HAVE_ECC192) || defined(HAVE_ALL_CURVES)
    case 24:
        /* first [P-192,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\x60\x80\x79\x42\x3f\x12\x42\x1d\xe6\x16\xb7\x49\x3e\xbe\x55\x1c\xf4\xd6\x5b\x92";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\xeb\xf7\x48\xd7\x48\xeb\xbc\xa7\xd2\x9f\xb4\x73\x69\x8a\x6e\x6b"
                "\x4f\xb1\x0c\x86\x5d\x4a\xf0\x24\xcc\x39\xae\x3d\xf3\x46\x4b\xa4"
                "\xf1\xd6\xd4\x0f\x32\xbf\x96\x18\xa9\x1b\xb5\x98\x6f\xa1\xa2\xaf"
                "\x04\x8a\x0e\x14\xdc\x51\xe5\x26\x7e\xb0\x5e\x12\x7d\x68\x9d\x0a"
                "\xc6\xf1\xa7\xf1\x56\xce\x06\x63\x16\xb9\x71\xcc\x7a\x11\xd0\xfd"
                "\x7a\x20\x93\xe2\x7c\xf2\xd0\x87\x27\xa4\xe6\x74\x8c\xc3\x2f\xd5"
                "\x9c\x78\x10\xc5\xb9\x01\x9d\xf2\x1c\xdc\xc0\xbc\xa4\x32\xc0\xa3"
                "\xee\xd0\x78\x53\x87\x50\x88\x77\x11\x43\x59\xce\xe4\xa0\x71\xcf";
            vec.msgLen = 128;
        #endif
        vec.Qx  = "07008ea40b08dbe76432096e80a2494c94982d2d5bcf98e6";
        vec.Qy  = "76fab681d00b414ea636ba215de26d98c41bd7f2e4d65477";
        vec.d   = "e14f37b3d1374ff8b03f41b9b3fdd2f0ebccf275d660d7f3";
        vec.R   = "6994d962bdd0d793ffddf855ec5bf2f91a9698b46258a63e";
        vec.S   = "02ba6465a234903744ab02bc8521405b73cf5fc00e1a9f41";
        vec.curveName = "SECP192R1";
    #ifndef NO_ASN
        vec.r   = (byte*)"\x69\x94\xd9\x62\xbd\xd0\xd7\x93\xff\xdd\xf8\x55"
                         "\xec\x5b\xf2\xf9\x1a\x96\x98\xb4\x62\x58\xa6\x3e";
        vec.rSz = 24;
        vec.s   = (byte*)"\x02\xba\x64\x65\xa2\x34\x90\x37\x44\xab\x02\xbc"
                         "\x85\x21\x40\x5b\x73\xcf\x5f\xc0\x0e\x1a\x9f\x41";
        vec.sSz = 24;
    #endif
        break;
#endif /* HAVE_ECC192 */

#if defined(HAVE_ECC224) || defined(HAVE_ALL_CURVES)
    case 28:
        /* first [P-224,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\xb9\xa3\xb8\x6d\xb0\xba\x99\xfd\xc6\xd2\x94\x6b\xfe\xbe\x9c\xe8\x3f\x10\x74\xfc";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\x36\xc8\xb2\x29\x86\x48\x7f\x67\x7c\x18\xd0\x97\x2a\x9e\x20\x47"
                "\xb3\xaf\xa5\x9e\xc1\x62\x76\x4e\xc3\x0b\x5b\x69\xe0\x63\x0f\x99"
                "\x0d\x4e\x05\xc2\x73\xb0\xe5\xa9\xd4\x28\x27\xb6\x95\xfc\x2d\x64"
                "\xd9\x13\x8b\x1c\xf4\xc1\x21\x55\x89\x4c\x42\x13\x21\xa7\xbb\x97"
                "\x0b\xdc\xe0\xfb\xf0\xd2\xae\x85\x61\xaa\xd8\x71\x7f\x2e\x46\xdf"
                "\xe3\xff\x8d\xea\xb4\xd7\x93\x23\x56\x03\x2c\x15\x13\x0d\x59\x9e"
                "\x26\xc1\x0f\x2f\xec\x96\x30\x31\xac\x69\x38\xa1\x8d\x66\x45\x38"
                "\xb9\x4d\xac\x55\x34\xef\x7b\x59\x94\x24\xd6\x9b\xe1\xf7\x1c\x20";
            vec.msgLen = 128;
        #endif
        vec.Qx  = "8a4dca35136c4b70e588e23554637ae251077d1365a6ba5db9585de7";
        vec.Qy  = "ad3dee06de0be8279d4af435d7245f14f3b4f82eb578e519ee0057b1";
        vec.d   = "97c4b796e1639dd1035b708fc00dc7ba1682cec44a1002a1a820619f";
        vec.R   = "147b33758321e722a0360a4719738af848449e2c1d08defebc1671a7";
        vec.S   = "24fc7ed7f1352ca3872aa0916191289e2e04d454935d50fe6af3ad5b";
        vec.curveName = "SECP224R1";
    #ifndef NO_ASN
        vec.r   = (byte*)"\x14\x7b\x33\x75\x83\x21\xe7\x22\xa0\x36\x0a\x47"
                         "\x19\x73\x8a\xf8\x48\x44\x9e\x2c\x1d\x08\xde\xfe"
                         "\xbc\x16\x71\xa7";
        vec.rSz = 28;
        vec.s   = (byte*)"\x24\xfc\x7e\xd7\xf1\x35\x2c\xa3\x87\x2a\xa0\x91"
                         "\x61\x91\x28\x9e\x2e\x04\xd4\x54\x93\x5d\x50\xfe"
                         "\x6a\xf3\xad\x5b";
        vec.sSz = 28;
    #endif
        break;
#endif /* HAVE_ECC224 */

#if defined(HAVE_ECC239) || defined(HAVE_ALL_CURVES)
    case 30:
        return 0;
#endif /* HAVE_ECC239 */

#if !defined(NO_ECC256) || defined(HAVE_ALL_CURVES)
    case 32:
        /* first [P-256,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\xa3\xf9\x1a\xe2\x1b\xa6\xb3\x03\x98\x64\x47\x2f\x18\x41\x44\xc6\xaf\x62\xcd\x0e";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\xa2\x4b\x21\x76\x2e\x6e\xdb\x15\x3c\xc1\x14\x38\xdb\x0e\x92\xcd"
                "\xf5\x2b\x86\xb0\x6c\xa9\x70\x16\x06\x27\x59\xc7\x0d\x36\xd1\x56"
                "\x2c\xc9\x63\x0d\x7f\xc7\xc7\x74\xb2\x8b\x54\xe3\x1e\xf5\x58\x72"
                "\xb2\xa6\x5d\xf1\xd7\xec\x26\xde\xbb\x33\xe7\xd9\x27\xef\xcc\xf4"
                "\x6b\x63\xde\x52\xa4\xf4\x31\xea\xca\x59\xb0\x5d\x2e\xde\xc4\x84"
                "\x5f\xff\xc0\xee\x15\x03\x94\xd6\x1f\x3d\xfe\xcb\xcd\xbf\x6f\x5a"
                "\x73\x38\xd0\xbe\x3f\x2a\x77\x34\x51\x98\x3e\xba\xeb\x48\xf6\x73"
                "\x8f\xc8\x95\xdf\x35\x7e\x1a\x48\xa6\x53\xbb\x35\x5a\x31\xa1\xb4"
            vec.msgLen = 128;
        #endif
        vec.Qx  = "fa2737fb93488d19caef11ae7faf6b7f4bcd67b286e3fc54e8a65c2b74aeccb0";
        vec.Qy  = "d4ccd6dae698208aa8c3a6f39e45510d03be09b2f124bfc067856c324f9b4d09";
        vec.d   = "be34baa8d040a3b991f9075b56ba292f755b90e4b6dc10dad36715c33cfdac25";
        vec.R   = "2b826f5d44e2d0b6de531ad96b51e8f0c56fdfead3c236892e4d84eacfc3b75c";
        vec.S   = "a2248b62c03db35a7cd63e8a120a3521a89d3d2f61ff99035a2148ae32e3a248";
    #ifndef NO_ASN
        vec.r   = (byte*)"\x2b\x82\x6f\x5d\x44\xe2\xd0\xb6\xde\x53\x1a\xd9"
                         "\x6b\x51\xe8\xf0\xc5\x6f\xdf\xea\xd3\xc2\x36\x89"
                         "\x2e\x4d\x84\xea\xcf\xc3\xb7\x5c";
        vec.rSz = 32;
        vec.s   = (byte*)"\xa2\x24\x8b\x62\xc0\x3d\xb3\x5a\x7c\xd6\x3e\x8a"
                         "\x12\x0a\x35\x21\xa8\x9d\x3d\x2f\x61\xff\x99\x03"
                         "\x5a\x21\x48\xae\x32\xe3\xa2\x48";
        vec.sSz = 32;
    #endif
        vec.curveName = "SECP256R1";
        break;
#endif /* !NO_ECC256 */

#if defined(HAVE_ECC320) || defined(HAVE_ALL_CURVES)
    case 40:
        return 0;
#endif /* HAVE_ECC320 */

#if defined(HAVE_ECC384) || defined(HAVE_ALL_CURVES)
    case 48:
        /* first [P-384,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\x9b\x9f\x8c\x95\x35\xa5\xca\x26\x60\x5d\xb7\xf2\xfa\x57\x3b\xdf\xc3\x2e\xab\x8b";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\xab\xe1\x0a\xce\x13\xe7\xe1\xd9\x18\x6c\x48\xf7\x88\x9d\x51\x47"
                "\x3d\x3a\x09\x61\x98\x4b\xc8\x72\xdf\x70\x8e\xcc\x3e\xd3\xb8\x16"
                "\x9d\x01\xe3\xd9\x6f\xc4\xf1\xd5\xea\x00\xa0\x36\x92\xbc\xc5\xcf"
                "\xfd\x53\x78\x7c\x88\xb9\x34\xaf\x40\x4c\x03\x9d\x32\x89\xb5\xba"
                "\xc5\xae\x7d\xb1\x49\x68\x75\xb5\xdc\x73\xc3\x09\xf9\x25\xc1\x3d"
                "\x1c\x01\xab\xda\xaf\xeb\xcd\xac\x2c\xee\x43\x39\x39\xce\x8d\x4a"
                "\x0a\x5d\x57\xbb\x70\x5f\x3b\xf6\xec\x08\x47\x95\x11\xd4\xb4\xa3"
                "\x21\x1f\x61\x64\x9a\xd6\x27\x43\x14\xbf\x0d\x43\x8a\x81\xe0\x60"
            vec.msgLen = 128;
        #endif
        vec.Qx  = "e55fee6c49d8d523f5ce7bf9c0425ce4ff650708b7de5cfb095901523979a7f042602db30854735369813b5c3f5ef868";
        vec.Qy  = "28f59cc5dc509892a988d38a8e2519de3d0c4fd0fbdb0993e38f18506c17606c5e24249246f1ce94983a5361c5be983e";
        vec.d   = "a492ce8fa90084c227e1a32f7974d39e9ff67a7e8705ec3419b35fb607582bebd461e0b1520ac76ec2dd4e9b63ebae71";
        vec.R   = "6820b8585204648aed63bdff47f6d9acebdea62944774a7d14f0e14aa0b9a5b99545b2daee6b3c74ebf606667a3f39b7";
        vec.S   = "491af1d0cccd56ddd520b233775d0bc6b40a6255cc55207d8e9356741f23c96c14714221078dbd5c17f4fdd89b32a907";
        vec.curveName = "SECP384R1";
    #ifndef NO_ASN
        vec.r   = (byte*)"\x68\x20\xb8\x58\x52\x04\x64\x8a\xed\x63\xbd\xff"
                         "\x47\xf6\xd9\xac\xeb\xde\xa6\x29\x44\x77\x4a\x7d"
                         "\x14\xf0\xe1\x4a\xa0\xb9\xa5\xb9\x95\x45\xb2\xda"
                         "\xee\x6b\x3c\x74\xeb\xf6\x06\x66\x7a\x3f\x39\xb7";
        vec.rSz = 48;
        vec.s   = (byte*)"\x49\x1a\xf1\xd0\xcc\xcd\x56\xdd\xd5\x20\xb2\x33"
                         "\x77\x5d\x0b\xc6\xb4\x0a\x62\x55\xcc\x55\x20\x7d"
                         "\x8e\x93\x56\x74\x1f\x23\xc9\x6c\x14\x71\x42\x21"
                         "\x07\x8d\xbd\x5c\x17\xf4\xfd\xd8\x9b\x32\xa9\x07";
        vec.sSz = 48;
    #endif
        break;
#endif /* HAVE_ECC384 */

#if defined(HAVE_ECC512) || defined(HAVE_ALL_CURVES)
    case 64:
        return 0;
#endif /* HAVE_ECC512 */

#if defined(HAVE_ECC521) || defined(HAVE_ALL_CURVES)
    case 66:
        /* first [P-521,SHA-1] vector from FIPS 186-3 NIST vectors */
        #if 1
            vec.msg = "\x1b\xf7\x03\x9c\xca\x23\x94\x27\x3f\x11\xa1\xd4\x8d\xcc\xb4\x46\x6f\x31\x61\xdf";
            vec.msgLen = 20;
        #else
            /* This is the raw message prior to SHA-1 */
            vec.msg =
                "\x50\x3f\x79\x39\x34\x0a\xc7\x23\xcd\x4a\x2f\x4e\x6c\xcc\x27\x33"
                "\x38\x3a\xca\x2f\xba\x90\x02\x19\x9d\x9e\x1f\x94\x8b\xe0\x41\x21"
                "\x07\xa3\xfd\xd5\x14\xd9\x0c\xd4\xf3\x7c\xc3\xac\x62\xef\x00\x3a"
                "\x2d\xb1\xd9\x65\x7a\xb7\x7f\xe7\x55\xbf\x71\xfa\x59\xe4\xd9\x6e"
                "\xa7\x2a\xe7\xbf\x9d\xe8\x7d\x79\x34\x3b\xc1\xa4\xbb\x14\x4d\x16"
                "\x28\xd1\xe9\xe9\xc8\xed\x80\x8b\x96\x2c\x54\xe5\xf9\x6d\x53\xda"
                "\x14\x7a\x96\x38\xf9\x4a\x91\x75\xd8\xed\x61\x05\x5f\x0b\xa5\x73"
                "\xa8\x2b\xb7\xe0\x18\xee\xda\xc4\xea\x7b\x36\x2e\xc8\x9c\x38\x2b"
            vec.msgLen = 128;
        #endif
        vec.Qx  = "12fbcaeffa6a51f3ee4d3d2b51c5dec6d7c726ca353fc014ea2bf7cfbb9b910d32cbfa6a00fe39b6cdb8946f22775398b2e233c0cf144d78c8a7742b5c7a3bb5d23";
        vec.Qy  = "09cdef823dd7bf9a79e8cceacd2e4527c231d0ae5967af0958e931d7ddccf2805a3e618dc3039fec9febbd33052fe4c0fee98f033106064982d88f4e03549d4a64d";
        vec.d   = "1bd56bd106118eda246155bd43b42b8e13f0a6e25dd3bb376026fab4dc92b6157bc6dfec2d15dd3d0cf2a39aa68494042af48ba9601118da82c6f2108a3a203ad74";
        vec.R   = "0bd117b4807710898f9dd7778056485777668f0e78e6ddf5b000356121eb7a220e9493c7f9a57c077947f89ac45d5acb6661bbcd17abb3faea149ba0aa3bb1521be";
        vec.S   = "019cd2c5c3f9870ecdeb9b323abdf3a98cd5e231d85c6ddc5b71ab190739f7f226e6b134ba1d5889ddeb2751dabd97911dff90c34684cdbe7bb669b6c3d22f2480c";
        vec.curveName = "SECP521R1";
    #ifndef NO_ASN
        vec.r   = (byte*)"\xbd\x11\x7b\x48\x07\x71\x08\x98\xf9\xdd\x77\x78"
                         "\x05\x64\x85\x77\x76\x68\xf0\xe7\x8e\x6d\xdf\x5b"
                         "\x00\x03\x56\x12\x1e\xb7\xa2\x20\xe9\x49\x3c\x7f"
                         "\x9a\x57\xc0\x77\x94\x7f\x89\xac\x45\xd5\xac\xb6"
                         "\x66\x1b\xbc\xd1\x7a\xbb\x3f\xae\xa1\x49\xba\x0a"
                         "\xa3\xbb\x15\x21\xbe";
        vec.rSz = 65;
        vec.s   = (byte*)"\x19\xcd\x2c\x5c\x3f\x98\x70\xec\xde\xb9\xb3\x23"
                         "\xab\xdf\x3a\x98\xcd\x5e\x23\x1d\x85\xc6\xdd\xc5"
                         "\xb7\x1a\xb1\x90\x73\x9f\x7f\x22\x6e\x6b\x13\x4b"
                         "\xa1\xd5\x88\x9d\xde\xb2\x75\x1d\xab\xd9\x79\x11"
                         "\xdf\xf9\x0c\x34\x68\x4c\xdb\xe7\xbb\x66\x9b\x6c"
                         "\x3d\x22\xf2\x48\x0c";
        vec.sSz = 65;
    #endif
        break;
#endif /* HAVE_ECC521 */
    default:
        return NOT_COMPILED_IN; /* Invalid key size / Not supported */
    }; /* Switch */

    ret = ecc_test_vector_item(&vec);
    if (ret < 0) {
        return ret;
    }

    return 0;
}
#endif /* WOLF_CRYPTO_CB_ONLY_ECC */

#if defined(HAVE_ECC_SIGN) && (defined(WOLFSSL_ECDSA_DETERMINISTIC_K) || \
                               defined(WOLFSSL_ECDSA_DETERMINISTIC_K_VARIANT)) \
    && (!defined(FIPS_VERSION_GE) || FIPS_VERSION_GE(5,3))
#if defined(HAVE_ECC256)
static wc_test_ret_t ecc_test_deterministic_k(WC_RNG* rng)
{
    wc_test_ret_t ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *key = NULL;
#else
    ecc_key key[1];
#endif
    int key_inited = 0;
    byte sig[72];
    word32 sigSz;
    WOLFSSL_SMALL_STACK_STATIC const unsigned char msg[] = "sample";
    unsigned char hash[32];
    WOLFSSL_SMALL_STACK_STATIC const char* dIUT =
        "C9AFA9D845BA75166B5C215767B1D6934E50C3DB36E89B127B8A622B120F6721";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTx =
        "60FED4BA255A9D31C961EB74C6356D68C049B8923B61FA6CE669622E60F29FB6";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTy =
        "7903FE1008B8BC99A41AE9E95628BC64F2F1B20C2D7E9F5177A3C294D4462299";
    WOLFSSL_SMALL_STACK_STATIC const byte expSig[] = {
        0x30, 0x46, 0x02, 0x21, 0x00, 0xEF, 0xD4, 0x8B,
        0x2A, 0xAC, 0xB6, 0xA8, 0xFD, 0x11, 0x40, 0xDD,
        0x9C, 0xD4, 0x5E, 0x81, 0xD6, 0x9D, 0x2C, 0x87,
        0x7B, 0x56, 0xAA, 0xF9, 0x91, 0xC3, 0x4D, 0x0E,
        0xA8, 0x4E, 0xAF, 0x37, 0x16, 0x02, 0x21, 0x00,
        0xF7, 0xCB, 0x1C, 0x94, 0x2D, 0x65, 0x7C, 0x41,
        0xD4, 0x36, 0xC7, 0xA1, 0xB6, 0xE2, 0x9F, 0x65,
        0xF3, 0xE9, 0x00, 0xDB, 0xB9, 0xAF, 0xF4, 0x06,
        0x4D, 0xC4, 0xAB, 0x2F, 0x84, 0x3A, 0xCD, 0xA8
    };

#ifdef WOLFSSL_SMALL_STACK
    key = (ecc_key *)XMALLOC(sizeof(*key), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (key == NULL)
        return MEMORY_E;
#endif

    ret = wc_ecc_init_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        goto done;
    }
    key_inited = 1;
    ret = wc_ecc_import_raw(key, QIUTx, QIUTy, dIUT, "SECP256R1");
    if (ret != 0) {
        goto done;
    }

    ret = wc_Hash(WC_HASH_TYPE_SHA256, msg,
            (word32)XSTRLEN((const char*)msg), hash, sizeof(hash));
    if (ret != 0) {
        goto done;
    }

    ret = wc_ecc_set_deterministic(key, 1);
    if (ret != 0) {
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, key.asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    if (sigSz != sizeof(expSig)) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }
    if (XMEMCMP(sig, expSig, sigSz) != 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, key.asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

done:
    if (key_inited)
        wc_ecc_free(key);
 #ifdef WOLFSSL_SMALL_STACK
    XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
   return ret;
}
#endif

#ifdef WOLFSSL_PUBLIC_MP
#if defined(HAVE_ECC384)
/* KAT from RFC6979 */
static wc_test_ret_t ecc384_test_deterministic_k(WC_RNG* rng)
{
    wc_test_ret_t ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *key;
    mp_int *r, *s, *expR, *expS;
#else
    ecc_key key[1];
    mp_int r[1], s[1], expR[1], expS[1];
#endif
    int key_inited = 0;
    WOLFSSL_SMALL_STACK_STATIC const unsigned char msg[] = "sample";
    unsigned char hash[32];
    WOLFSSL_SMALL_STACK_STATIC const char* dIUT =
        "6B9D3DAD2E1B8C1C05B19875B6659F4DE23C3B667BF297BA9AA47740787137D8"
        "96D5724E4C70A825F872C9EA60D2EDF5";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTx =
        "EC3A4E415B4E19A4568618029F427FA5DA9A8BC4AE92E02E06AAE5286B300C64"
        "DEF8F0EA9055866064A254515480BC13";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTy =
        "8015D9B72D7D57244EA8EF9AC0C621896708A59367F9DFB9F54CA84B3F1C9DB1"
        "288B231C3AE0D4FE7344FD2533264720";
    WOLFSSL_SMALL_STACK_STATIC const char* expRstr =
       "21B13D1E013C7FA1392D03C5F99AF8B30C570C6F98D4EA8E354B63A21D3DAA33"
       "BDE1E888E63355D92FA2B3C36D8FB2CD";
    WOLFSSL_SMALL_STACK_STATIC const char* expSstr =
       "F3AA443FB107745BF4BD77CB3891674632068A10CA67E3D45DB2266FA7D1FEEB"
       "EFDC63ECCD1AC42EC0CB8668A4FA0AB0";

#ifdef WOLFSSL_SMALL_STACK
    key = (ecc_key *)XMALLOC(sizeof(*key), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    r = (mp_int *)XMALLOC(sizeof(*r), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    s = (mp_int *)XMALLOC(sizeof(*s), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    expR = (mp_int *)XMALLOC(sizeof(*expR), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    expS = (mp_int *)XMALLOC(sizeof(*expS), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((key == NULL) ||
        (r == NULL) ||
        (s == NULL) ||
        (expR == NULL) ||
        (expS == NULL))
    {
        ret = MEMORY_E;
        goto done;
    }
#endif

    ret = mp_init_multi(r, s, expR, expS, NULL, NULL);
    if (ret != MP_OKAY) {
        goto done;
    }
    ret = wc_ecc_init_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        goto done;
    }
    key_inited = 1;

    ret = wc_ecc_import_raw(key, QIUTx, QIUTy, dIUT, "SECP384R1");
    if (ret != 0) {
        goto done;
    }

    ret = wc_Hash(WC_HASH_TYPE_SHA256, msg,
            (word32)XSTRLEN((const char*)msg), hash, sizeof(hash));
    if (ret != 0) {
        goto done;
    }

    ret = wc_ecc_set_deterministic(key, 1);
    if (ret != 0) {
        goto done;
    }

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, key.asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash_ex(hash, sizeof(hash), rng, key, r, s);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    mp_read_radix(expR, expRstr, MP_RADIX_HEX);
    mp_read_radix(expS, expSstr, MP_RADIX_HEX);
    if (mp_cmp(r, expR) != MP_EQ) {
        ret = WC_TEST_RET_ENC_NC;
    }

done:
    if (key_inited)
        wc_ecc_free(key);
#ifdef WOLFSSL_SMALL_STACK
    if (key != NULL)
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (r != NULL)
        XFREE(r, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (s != NULL)
        XFREE(s, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (expR != NULL)
        XFREE(expR, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (expS != NULL)
        XFREE(expS, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif /* HAVE_ECC384 */

#if defined(HAVE_ECC521)
/* KAT from RFC6979 */
static wc_test_ret_t ecc521_test_deterministic_k(WC_RNG* rng)
{
    wc_test_ret_t ret;
#ifdef WOLFSSL_SMALL_STACK
    ecc_key *key;
    mp_int *r, *s, *expR, *expS;
#else
    ecc_key key[1];
    mp_int r[1], s[1], expR[1], expS[1];
#endif
    int key_inited = 0;
    WOLFSSL_SMALL_STACK_STATIC const unsigned char msg[] = "sample";
    unsigned char hash[32];

    WOLFSSL_SMALL_STACK_STATIC const char* dIUT =
       "0FAD06DAA62BA3B25D2FB40133DA757205DE67F5BB0018FEE8C86E1B68C7E75C"
       "AA896EB32F1F47C70855836A6D16FCC1466F6D8FBEC67DB89EC0C08B0E996B83"
       "538";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTx =
        "1894550D0785932E00EAA23B694F213F8C3121F86DC97A04E5A7167DB4E5BCD3"
        "71123D46E45DB6B5D5370A7F20FB633155D38FFA16D2BD761DCAC474B9A2F502"
        "3A4";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTy =
        "0493101C962CD4D2FDDF782285E64584139C2F91B47F87FF82354D6630F746A2"
        "8A0DB25741B5B34A828008B22ACC23F924FAAFBD4D33F81EA66956DFEAA2BFDF"
        "CF5";
    WOLFSSL_SMALL_STACK_STATIC const char* expRstr =
        "1511BB4D675114FE266FC4372B87682BAECC01D3CC62CF2303C92B3526012659"
        "D16876E25C7C1E57648F23B73564D67F61C6F14D527D54972810421E7D87589E"
        "1A7";
    WOLFSSL_SMALL_STACK_STATIC const char* expSstr =
        "04A171143A83163D6DF460AAF61522695F207A58B95C0644D87E52AA1A347916"
        "E4F7A72930B1BC06DBE22CE3F58264AFD23704CBB63B29B931F7DE6C9D949A7E"
        "CFC";

#ifdef WOLFSSL_SMALL_STACK
    key = (ecc_key *)XMALLOC(sizeof(*key), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    r = (mp_int *)XMALLOC(sizeof(*r), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    s = (mp_int *)XMALLOC(sizeof(*s), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    expR = (mp_int *)XMALLOC(sizeof(*expR), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    expS = (mp_int *)XMALLOC(sizeof(*expS), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);

    if ((key == NULL) ||
        (r == NULL) ||
        (s == NULL) ||
        (expR == NULL) ||
        (expS == NULL))
    {
        ret = MEMORY_E;
        goto done;
    }
#endif

    ret = mp_init_multi(r, s, expR, expS, NULL, NULL);
    if (ret != MP_OKAY) {
        goto done;
    }
    ret = wc_ecc_init_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        return WC_TEST_RET_ENC_EC(ret);
    }
    key_inited = 1;

    ret = wc_ecc_import_raw(key, QIUTx, QIUTy, dIUT, "SECP521R1");
    if (ret != 0) {
        goto done;
    }

    ret = wc_Hash(WC_HASH_TYPE_SHA256, msg,
            (word32)XSTRLEN((const char*)msg), hash, sizeof(hash));
    if (ret != 0) {
        goto done;
    }

    ret = wc_ecc_set_deterministic(key, 1);
    if (ret != 0) {
        goto done;
    }

    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, key.asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash_ex(hash, sizeof(hash), rng, key, r, s);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    mp_read_radix(expR, expRstr, MP_RADIX_HEX);
    mp_read_radix(expS, expSstr, MP_RADIX_HEX);
    if (mp_cmp(r, expR) != MP_EQ) {
        ret = WC_TEST_RET_ENC_NC;
    }

done:
    if (key_inited)
        wc_ecc_free(key);
#ifdef WOLFSSL_SMALL_STACK
    if (key != NULL)
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (r != NULL)
        XFREE(r, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (s != NULL)
        XFREE(s, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (expR != NULL)
        XFREE(expR, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (expS != NULL)
        XFREE(expS, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif /* HAVE_ECC521 */
#endif /* WOLFSSL_PUBLIC_MP */
#endif /* HAVE_ECC_SIGN && (WOLFSSL_ECDSA_DETERMINISTIC_K ||
                            WOLFSSL_ECDSA_DETERMINISTIC_K_VARIANT)
          && (!FIPS_VERSION_GE || FIPS_VERSION_GE(5,3)) */


#if defined(HAVE_ECC_SIGN) && defined(WOLFSSL_ECDSA_SET_K) && \
    !defined(WOLFSSL_KCAPI_ECC)
static wc_test_ret_t ecc_test_sign_vectors(WC_RNG* rng)
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key *key = NULL;
#else
    ecc_key key[1];
#endif
    int key_inited = 0;
    byte sig[72];
    word32 sigSz;
    WOLFSSL_SMALL_STACK_STATIC const unsigned char hash[32] = "test wolfSSL deterministic sign";
    WOLFSSL_SMALL_STACK_STATIC const char* dIUT =   "7d7dc5f71eb29ddaf80d6214632eeae03d9058af1fb6d22ed80badb62bc1a534";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTx =  "ead218590119e8876b29146ff89ca61770c4edbbf97d38ce385ed281d8a6b230";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTy =  "28af61281fd35e2fa7002523acc85a429cb06ee6648325389f59edfce1405141";
    WOLFSSL_SMALL_STACK_STATIC const byte k[1] = { 0x02 };
    WOLFSSL_SMALL_STACK_STATIC const byte expSig[71] = {
        0x30, 0x45, 0x02, 0x20, 0x7c, 0xf2, 0x7b, 0x18,
        0x8d, 0x03, 0x4f, 0x7e, 0x8a, 0x52, 0x38, 0x03,
        0x04, 0xb5, 0x1a, 0xc3, 0xc0, 0x89, 0x69, 0xe2,
        0x77, 0xf2, 0x1b, 0x35, 0xa6, 0x0b, 0x48, 0xfc,
        0x47, 0x66, 0x99, 0x78, 0x02, 0x21, 0x00, 0xa8,
        0x43, 0xa0, 0xce, 0x6c, 0x5e, 0x17, 0x8a, 0x53,
        0x4d, 0xaf, 0xd2, 0x95, 0x78, 0x9f, 0x84, 0x4f,
        0x94, 0xb8, 0x75, 0xa3, 0x19, 0xa5, 0xd4, 0xdf,
        0xe1, 0xd4, 0x5e, 0x9d, 0x97, 0xfe, 0x81
    };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((key = (ecc_key *)XMALLOC(sizeof(*key), HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER)) == NULL)
        return MEMORY_E;
#endif

    ret = wc_ecc_init_ex(key, HEAP_HINT, devId);
    if (ret != 0) {
        goto done;
    }
    key_inited = 1;

    ret = wc_ecc_import_raw(key, QIUTx, QIUTy, dIUT, "SECP256R1");
    if (ret != 0) {
        goto done;
    }
#if (!defined(HAVE_FIPS) || (defined(HAVE_FIPS_VERSION) \
    && (HAVE_FIPS_VERSION > 2)))
    wc_ecc_set_flags(key, WC_ECC_FLAG_DEC_SIGN);
#endif

    ret = wc_ecc_sign_set_k(k, sizeof(k), key);
    if (ret != 0) {
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    if (sigSz != sizeof(expSig)) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }
    if (XMEMCMP(sig, expSig, sigSz) != 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }

    sigSz = sizeof(sig);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(hash, sizeof(hash), sig, &sigSz, rng, key);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

done:
    if (key_inited)
        wc_ecc_free(key);
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif

    return ret;
}
#endif

#if defined(HAVE_ECC_CDH) && defined(HAVE_ECC_DHE)
static wc_test_ret_t ecc_test_cdh_vectors(WC_RNG* rng)
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key *pub_key = (ecc_key *)XMALLOC(sizeof *pub_key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *priv_key = (ecc_key *)XMALLOC(sizeof *priv_key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key pub_key[1], priv_key[1];
#endif
    byte    sharedA[32] = {0}, sharedB[32] = {0};
    word32  x, z;

    WOLFSSL_SMALL_STACK_STATIC const char* QCAVSx = "700c48f77f56584c5cc632ca65640db91b6bacce3a4df6b42ce7cc838833d287";
    WOLFSSL_SMALL_STACK_STATIC const char* QCAVSy = "db71e509e3fd9b060ddb20ba5c51dcc5948d46fbf640dfe0441782cab85fa4ac";
    WOLFSSL_SMALL_STACK_STATIC const char* dIUT =   "7d7dc5f71eb29ddaf80d6214632eeae03d9058af1fb6d22ed80badb62bc1a534";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTx =  "ead218590119e8876b29146ff89ca61770c4edbbf97d38ce385ed281d8a6b230";
    WOLFSSL_SMALL_STACK_STATIC const char* QIUTy =  "28af61281fd35e2fa7002523acc85a429cb06ee6648325389f59edfce1405141";
    WOLFSSL_SMALL_STACK_STATIC const char* ZIUT =   "46fc62106420ff012e54a434fbdd2d25ccc5852060561e68040dd7778997bd7b";

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((pub_key == NULL) ||
        (priv_key == NULL)) {
        ret = MEMORY_E;
        goto done;
    }
#endif

    XMEMSET(pub_key, 0, sizeof *pub_key);
    XMEMSET(priv_key, 0, sizeof *priv_key);

    /* setup private and public keys */
    ret = wc_ecc_init_ex(pub_key, HEAP_HINT, devId);
    if (ret != 0)
        goto done;
    ret = wc_ecc_init_ex(priv_key, HEAP_HINT, devId);
    if (ret != 0)
        goto done;
    wc_ecc_set_flags(pub_key, WC_ECC_FLAG_COFACTOR);
    wc_ecc_set_flags(priv_key, WC_ECC_FLAG_COFACTOR);
    ret = wc_ecc_import_raw(pub_key, QCAVSx, QCAVSy, NULL, "SECP256R1");
    if (ret != 0)
        goto done;
    ret = wc_ecc_import_raw(priv_key, QIUTx, QIUTy, dIUT, "SECP256R1");
    if (ret != 0)
        goto done;

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(priv_key, rng);
    if (ret != 0)
        goto done;
#else
    (void)rng;
#endif

    /* compute ECC Cofactor shared secret */
    x = sizeof(sharedA);
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &priv_key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(priv_key, pub_key, sharedA, &x);
    } while (ret == WC_PENDING_E);
    if (ret != 0) {
        goto done;
    }
    TEST_SLEEP();

    /* read in expected Z */
    z = sizeof(sharedB);
    ret = Base16_Decode((const byte*)ZIUT, (word32)XSTRLEN(ZIUT), sharedB, &z);
    if (ret != 0)
        goto done;

    /* compare results */
    if (x != z || XMEMCMP(sharedA, sharedB, x)) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (priv_key) {
        wc_ecc_free(priv_key);
        XFREE(priv_key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (pub_key) {
        wc_ecc_free(pub_key);
        XFREE(pub_key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(priv_key);
    wc_ecc_free(pub_key);
#endif

    return ret;
}
#endif /* HAVE_ECC_CDH && HAVE_ECC_DHE */
#endif /* HAVE_ECC_VECTOR_TEST */

#ifdef HAVE_ECC_KEY_IMPORT
/* returns 0 on success */
static wc_test_ret_t ecc_test_make_pub(WC_RNG* rng)
{
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key *key = (ecc_key *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT)
    ecc_key *pub = (ecc_key *)XMALLOC(sizeof *pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    byte *exportBuf = (byte *)XMALLOC(ECC_BUFSIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *tmp = (byte *)XMALLOC(ECC_BUFSIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key key[1];
#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG) && \
    !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    ecc_key pub[1];
#endif
    byte exportBuf[ECC_BUFSIZE];
    byte tmp[ECC_BUFSIZE];
#endif
    const byte* msg = (const byte*)"test wolfSSL ECC public gen";
    word32 x;
    word32 tmpSz;
    wc_test_ret_t ret = 0;
    ecc_point* pubPoint = NULL;
#ifdef HAVE_ECC_VERIFY
    int verify = 0;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((key == NULL) ||
#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT)
        (pub == NULL) ||
#endif
        (exportBuf == NULL) ||
        (tmp == NULL))
        ERROR_OUT(MEMORY_E, done);
#endif

    (void)msg;
    (void)verify;
    (void)exportBuf;
    (void)rng;

    wc_ecc_init_ex(key, HEAP_HINT, devId);

#ifndef NO_ECC256
#ifdef USE_CERT_BUFFERS_256
    XMEMCPY(tmp, ecc_key_der_256, (size_t)sizeof_ecc_key_der_256);
    tmpSz = (size_t)sizeof_ecc_key_der_256;
#else
    {
        XFILE file = XFOPEN(eccKeyDerFile, "rb");
        if (!file) {
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
        }

        tmpSz = (word32)XFREAD(tmp, 1, ECC_BUFSIZE, file);
        XFCLOSE(file);
        if (tmpSz == 0)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    }
#endif /* USE_CERT_BUFFERS_256 */

    /* import private only then test with */
    ret = wc_ecc_import_private_key(tmp, tmpSz, NULL, 0, NULL);
    if (ret == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }

    ret = wc_ecc_import_private_key(NULL, tmpSz, NULL, 0, key);
    if (ret == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }

    x = 0;
    ret = wc_EccPrivateKeyDecode(tmp, &x, key, tmpSz);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#ifdef HAVE_ECC_KEY_EXPORT
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_private_only(key, exportBuf, &x);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    /* make private only key */
    wc_ecc_free(key);
    wc_ecc_init_ex(key, HEAP_HINT, devId);
    ret = wc_ecc_import_private_key(exportBuf, x, NULL, 0, key);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    x = ECC_BUFSIZE;
    ret = wc_ecc_export_x963_ex(key, exportBuf, &x, 0);
    if (ret == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }

#endif /* HAVE_ECC_KEY_EXPORT */

    ret = wc_ecc_make_pub(NULL, NULL);
    if (ret == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }
    TEST_SLEEP();

#ifndef WOLFSSL_NO_MALLOC
    pubPoint = wc_ecc_new_point_h(HEAP_HINT);
    if (pubPoint == NULL) {
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
    }
#if !defined(WOLFSSL_CRYPTOCELL)
    ret = wc_ecc_make_pub(key, pubPoint);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
       ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#endif
    TEST_SLEEP();

#ifdef HAVE_ECC_KEY_EXPORT
    /* export should still fail, is private only key */
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_x963_ex(key, exportBuf, &x, 0);
    if (ret == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }
#endif /* HAVE_ECC_KEY_EXPORT */
#endif /* !WOLFSSL_NO_MALLOC */
#endif /* !NO_ECC256 */

    /* create a new key since above test for loading key is not supported */
#if defined(WOLFSSL_CRYPTOCELL) || defined(NO_ECC256) || \
    defined(WOLFSSL_QNX_CAAM) || defined(WOLFSSL_SE050) || \
    defined(WOLFSSL_SECO_CAAM) || defined(WOLFSSL_IMXRT1170_CAAM)
    ret  = wc_ecc_make_key(rng, ECC_KEYGEN_SIZE, key);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#endif

#if defined(HAVE_ECC_SIGN) && (!defined(ECC_TIMING_RESISTANT) || \
    (defined(ECC_TIMING_RESISTANT) && !defined(WC_NO_RNG))) && \
    !defined(WOLF_CRYPTO_CB_ONLY_ECC) && !defined(HAVE_ECC_DETERMINISTIC_K)
    tmpSz = ECC_BUFSIZE;
    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0) {
            ret = wc_ecc_sign_hash(msg, (word32)XSTRLEN((const char* )msg), tmp,
                &tmpSz, rng, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

#ifdef HAVE_ECC_VERIFY
    /* try verify with private only key */
    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0) {
            ret = wc_ecc_verify_hash(tmp, tmpSz, msg,
                (word32)XSTRLEN((const char*)msg), &verify, key);
        }
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    if (verify != 1) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }
    TEST_SLEEP();
#ifdef HAVE_ECC_KEY_EXPORT
    /* exporting the public part should now work */
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_x963_ex(key, exportBuf, &x, 0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#endif /* HAVE_ECC_KEY_EXPORT */
#endif /* HAVE_ECC_VERIFY */

#endif /* HAVE_ECC_SIGN */

#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG)
    /* now test private only key with creating a shared secret */
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_private_only(key, exportBuf, &x);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#if !defined(WOLFSSL_QNX_CAAM) && !defined(WOLFSSL_SE050)
    /* make private only key */
    wc_ecc_free(key);
    wc_ecc_init_ex(key, HEAP_HINT, devId);
    ret = wc_ecc_import_private_key(exportBuf, x, NULL, 0, key);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    /* check that public export fails with private only key */
    x = ECC_BUFSIZE;
    ret = wc_ecc_export_x963_ex(key, exportBuf, &x, 0);
    if (ret == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }
#endif /* WOLFSSL_QNX_CAAM */

#ifndef WOLF_CRYPTO_CB_ONLY_ECC
    /* make public key for shared secret */
    wc_ecc_init_ex(pub, HEAP_HINT, devId);
    ret = wc_ecc_make_key(rng, ECC_KEYGEN_SIZE, pub);
#ifdef HAVE_ECC_CDH
    wc_ecc_set_flags(key, WC_ECC_FLAG_COFACTOR);
#endif
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &pub->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(key, rng);
    if (ret != 0)
        goto done;
#endif

    x = ECC_BUFSIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0) {
            ret = wc_ecc_shared_secret(key, pub, exportBuf, &x);
        }
    } while (ret == WC_PENDING_E);
    wc_ecc_free(pub);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();
#endif /* HAVE_ECC_DHE && HAVE_ECC_KEY_EXPORT && !WC_NO_RNG */
#endif /* WOLF_CRYPTO_CB_ONLY_ECC */
    ret = 0;

done:

    wc_ecc_del_point_h(pubPoint, HEAP_HINT);

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key != NULL) {
        wc_ecc_free(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#if defined(HAVE_ECC_DHE) && defined(HAVE_ECC_KEY_EXPORT)
    if (pub != NULL)
        XFREE(pub, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#endif
    if (exportBuf != NULL)
        XFREE(exportBuf, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (tmp != NULL)
        XFREE(tmp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    wc_ecc_free(key);
#endif

    return ret;
}

#if defined(HAVE_ECC_KEY_EXPORT) && !defined(NO_ASN_CRYPT) && \
   !defined(WC_NO_RNG) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static wc_test_ret_t ecc_test_key_decode(WC_RNG* rng, int keySize)
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key *eccKey = (ecc_key *)XMALLOC(sizeof *eccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    byte *tmpBuf = (byte *)XMALLOC(ECC_BUFSIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key eccKey[1];
    byte tmpBuf[ECC_BUFSIZE];
#endif
    word32 tmpSz;
    word32 idx;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((eccKey == NULL) || (tmpBuf == NULL))
        ERROR_OUT(MEMORY_E, done);
#endif

    ret = wc_ecc_init(eccKey);
    if (ret != 0) {
        goto done;
    }
    ret = wc_ecc_make_key(rng, keySize, eccKey);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &eccKey->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0) {
        goto done;
    }

    tmpSz = ECC_BUFSIZE;
    ret = wc_EccKeyToDer(eccKey, tmpBuf, tmpSz);
    wc_ecc_free(eccKey);
    if (ret < 0) {
        goto done;
    }
    tmpSz = (word32)ret;

    ret = wc_ecc_init(eccKey);
    if (ret != 0) {
        goto done;
    }
    idx = 0;
    ret = wc_EccPrivateKeyDecode(tmpBuf, &idx, eccKey, tmpSz);
    if (ret != 0) {
        goto done;
    }
    wc_ecc_free(eccKey);

    ret = wc_ecc_init(eccKey);
    if (ret != 0) {
        goto done;
    }

    idx = 0;
    ret = wc_EccPublicKeyDecode(tmpBuf, &idx, eccKey, tmpSz);
    if (ret != 0) {
        goto done;
    }

    ret = 0;

  done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (eccKey != NULL) {
        wc_ecc_free(eccKey);
        XFREE(eccKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (tmpBuf != NULL)
        XFREE(tmpBuf, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    wc_ecc_free(eccKey);
#endif

    return ret;
}
#endif /* HAVE_ECC_KEY_EXPORT && !NO_ASN_CRYPT */
#endif /* HAVE_ECC_KEY_IMPORT */

#if defined(HAVE_ECC_KEY_EXPORT) && !defined(NO_ASN_CRYPT) && \
   !defined(WC_NO_RNG) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static wc_test_ret_t ecc_test_key_gen(WC_RNG* rng, int keySize)
{
    wc_test_ret_t ret = 0;
    int    derSz;
#ifdef HAVE_PKCS8
    word32 pkcs8Sz;
#endif
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    byte *der = (byte *)XMALLOC(ECC_BUFSIZE, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *userA = (ecc_key *)XMALLOC(sizeof *userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    byte der[ECC_BUFSIZE];
    ecc_key userA[1];
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((der == NULL) || (userA == NULL))
        ERROR_OUT(WC_TEST_RET_ENC_EC(MEMORY_E), done);
#endif

    ret = wc_ecc_init_ex(userA, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    ret = wc_ecc_make_key(rng, keySize, userA);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

    ret = wc_ecc_check_key(userA);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

    derSz = wc_EccKeyToDer(userA, der, ECC_BUFSIZE);
    if (derSz < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_I(derSz), done);
    }

    ret = SaveDerAndPem(der, derSz, eccCaKeyTempFile, eccCaKeyPemFile,
                        ECC_PRIVATEKEY_TYPE);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    /* test export of public key */
    derSz = wc_EccPublicKeyToDer(userA, der, ECC_BUFSIZE, 1);
    if (derSz < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_I(derSz), done);
    }
    if (derSz == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }

#ifdef HAVE_COMP_KEY
    /* test export of compressed public key */
    derSz = wc_EccPublicKeyToDer_ex(userA, der, ECC_BUFSIZE, 1, 1);
    if (derSz < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_I(derSz), done);
    }
    if (derSz == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }
#endif

    ret = SaveDerAndPem(der, derSz, eccPubKeyDerFile, NULL, 0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#ifdef HAVE_PKCS8
    /* test export of PKCS#8 unencrypted private key */
    pkcs8Sz = FOURK_BUF;
    derSz = wc_EccPrivateKeyToPKCS8(userA, der, &pkcs8Sz);
    if (derSz < 0) {
        ERROR_OUT(WC_TEST_RET_ENC_I(derSz), done);
    }

    if (derSz == 0) {
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }

    ret = SaveDerAndPem(der, derSz, eccPkcs8KeyDerFile, NULL, 0);
    if (ret != 0) {
        goto done;
    }
#endif /* HAVE_PKCS8 */

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (der != NULL)
        XFREE(der, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    if (userA != NULL) {
        wc_ecc_free(userA);
        XFREE(userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(userA);
#endif

    return ret;
}
#endif /* HAVE_ECC_KEY_EXPORT && !NO_ASN_CRYPT */

static wc_test_ret_t ecc_test_curve_size(WC_RNG* rng, int keySize, int testVerifyCount,
    int curve_id, const ecc_set_type* dp)
{
#if defined(HAVE_ECC_DHE) && !defined(WC_NO_RNG) && \
    !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)
    WC_DECLARE_VAR(sharedA, byte, ECC_SHARED_SIZE, HEAP_HINT);
    WC_DECLARE_VAR(sharedB, byte, ECC_SHARED_SIZE, HEAP_HINT);
    word32  y;
#endif
#ifdef HAVE_ECC_KEY_EXPORT
    #define ECC_KEY_EXPORT_BUF_SIZE (MAX_ECC_BYTES * 2 + 32)
    WC_DECLARE_VAR(exportBuf, byte, ECC_KEY_EXPORT_BUF_SIZE, HEAP_HINT);
#endif
    word32  x = 0;
#if !defined(ECC_TIMING_RESISTANT) || (defined(ECC_TIMING_RESISTANT) && \
    !defined(WC_NO_RNG) && !defined(WOLFSSL_KCAPI_ECC)) && \
    defined(HAVE_ECC_SIGN)
    WC_DECLARE_VAR(sig, byte, ECC_SIG_SIZE, HEAP_HINT);
    WC_DECLARE_VAR(digest, byte, ECC_DIGEST_SIZE, HEAP_HINT);
    int     i;
#ifdef HAVE_ECC_VERIFY
    int     verify;
#endif /* HAVE_ECC_VERIFY */
#endif /* HAVE_ECC_SIGN */
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key *userA = (ecc_key *)XMALLOC(sizeof *userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *userB = (ecc_key *)XMALLOC(sizeof *userB, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key *pubKey = (ecc_key *)XMALLOC(sizeof *pubKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key userA[1];
    ecc_key userB[1];
    ecc_key pubKey[1];
#endif
#ifndef WC_NO_RNG
    int     curveSize;
#endif

#ifdef WC_DECLARE_VAR_IS_HEAP_ALLOC
#if (defined(HAVE_ECC_DHE) || defined(HAVE_ECC_CDH)) && !defined(WC_NO_RNG) && \
    !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)
    if (sharedA == NULL || sharedB == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
#endif

#ifdef HAVE_ECC_KEY_EXPORT
    if (exportBuf == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
#endif

#if !defined(ECC_TIMING_RESISTANT) || (defined(ECC_TIMING_RESISTANT) && \
    !defined(WC_NO_RNG) && !defined(WOLFSSL_KCAPI_ECC)) && \
    defined(HAVE_ECC_SIGN)
    if (sig == NULL || digest == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
#endif
#endif /* WOLFSSL_SMALL_STACK && !WOLFSSL_NO_MALLOC */

    (void)testVerifyCount;
    (void)dp;
    (void)x;

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((userA == NULL) ||
        (userB == NULL) ||
        (pubKey == NULL))
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
#endif

    XMEMSET(userA, 0, sizeof *userA);
    XMEMSET(userB, 0, sizeof *userB);
    XMEMSET(pubKey, 0, sizeof *pubKey);

    ret = wc_ecc_init_ex(userA, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    ret = wc_ecc_init_ex(userB, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    ret = wc_ecc_init_ex(pubKey, HEAP_HINT, devId);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#ifdef WOLFSSL_CUSTOM_CURVES
    if (dp != NULL) {
        ret = wc_ecc_set_custom_curve(userA, dp);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
        ret = wc_ecc_set_custom_curve(userB, dp);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    }
#endif

#ifndef WC_NO_RNG
    ret = wc_ecc_make_key_ex(rng, keySize, userA, curve_id);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
#ifdef WOLF_CRYPTO_CB_ONLY_ECC
    if (ret == NO_VALID_DEVID) {
        ret = 0;
        goto done; /* no software case */
    }
#endif
    if (ret == ECC_CURVE_OID_E)
        goto done; /* catch case, where curve is not supported */
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

    if (wc_ecc_get_curve_idx(curve_id) != -1) {
        curveSize = wc_ecc_get_curve_size_from_id(userA->dp->id);
        if (curveSize != userA->dp->size)
            ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    }

    ret = wc_ecc_check_key(userA);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

/* ATECC508/608 configuration may not support more than one ECDH key */
#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)

    ret = wc_ecc_make_key_ex(rng, keySize, userB, curve_id);
#if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_NONE);
#endif
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

    /* only perform the below tests if the key size matches */
    if (dp == NULL && keySize > 0 && wc_ecc_size(userA) != keySize)
        /* Not an error, just not a key size match */
        WARNING_OUT(ECC_CURVE_OID_E, done);

#ifdef HAVE_ECC_DHE
#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(userA, rng);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    ret = wc_ecc_set_rng(userB, rng);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#endif

    x = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userA, userB, sharedA, &x);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

    y = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userB, userA, sharedB, &y);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    if (y != x)
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);

    if (XMEMCMP(sharedA, sharedB, x))
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    TEST_SLEEP();

#ifdef HAVE_ECC_CDH
    /* add cofactor flag */
    wc_ecc_set_flags(userA, WC_ECC_FLAG_COFACTOR);
    wc_ecc_set_flags(userB, WC_ECC_FLAG_COFACTOR);

    x = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userA, userB, sharedA, &x);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

    y = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userB, userA, sharedB, &y);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    if (y != x)
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);

    if (XMEMCMP(sharedA, sharedB, x))
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    TEST_SLEEP();

    /* remove cofactor flag */
    wc_ecc_set_flags(userA, 0);
    wc_ecc_set_flags(userB, 0);
#endif /* HAVE_ECC_CDH */
#endif /* HAVE_ECC_DHE */
#endif /* !WOLFSSL_ATECC508A && WOLFSSL_ATECC608A */

#ifdef HAVE_ECC_KEY_EXPORT
    x = ECC_KEY_EXPORT_BUF_SIZE;
    ret = wc_ecc_export_x963_ex(userA, exportBuf, &x, 0);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#ifdef HAVE_ECC_KEY_IMPORT
    #ifdef WOLFSSL_CUSTOM_CURVES
        if (dp != NULL) {
            ret = wc_ecc_set_custom_curve(pubKey, dp);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
        }
    #endif
    ret = wc_ecc_import_x963_ex(exportBuf, x, pubKey, curve_id);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A)
#ifdef HAVE_ECC_DHE
    y = ECC_SHARED_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret(userB, pubKey, sharedB, &y);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    if (XMEMCMP(sharedA, sharedB, y))
        ERROR_OUT(WC_TEST_RET_ENC_NC, done);
    TEST_SLEEP();
#endif /* HAVE_ECC_DHE */

    #ifdef HAVE_COMP_KEY
        /* try compressed export / import too */
        x = ECC_KEY_EXPORT_BUF_SIZE;
        ret = wc_ecc_export_x963_ex(userA, exportBuf, &x, 1);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
        wc_ecc_free(pubKey);

        ret = wc_ecc_init_ex(pubKey, HEAP_HINT, devId);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    #ifdef WOLFSSL_CUSTOM_CURVES
        if (dp != NULL) {
            ret = wc_ecc_set_custom_curve(pubKey, dp);
            if (ret != 0)
                ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
        }
    #endif
        ret = wc_ecc_import_x963_ex(exportBuf, x, pubKey, curve_id);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

    #ifdef HAVE_ECC_DHE
        y = ECC_SHARED_SIZE;
        do {
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &userB->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
        #endif
            if (ret == 0)
                ret = wc_ecc_shared_secret(userB, pubKey, sharedB, &y);
        } while (ret == WC_PENDING_E);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);

        if (XMEMCMP(sharedA, sharedB, y))
            ERROR_OUT(WC_TEST_RET_ENC_NC, done);
        TEST_SLEEP();
    #endif /* HAVE_ECC_DHE */
    #endif /* HAVE_COMP_KEY */

#endif /* !WOLFSSL_ATECC508A && !WOLFSSL_ATECC608A */
#endif /* !WC_NO_RNG */

#endif /* HAVE_ECC_KEY_IMPORT */
#endif /* HAVE_ECC_KEY_EXPORT */

    /* For KCAPI cannot sign using generated ECDH key */
#if !defined(ECC_TIMING_RESISTANT) || (defined(ECC_TIMING_RESISTANT) && \
    !defined(WC_NO_RNG) && !defined(WOLFSSL_KCAPI_ECC))
#ifdef HAVE_ECC_SIGN
    /* ECC w/out Shamir has issue with all 0 digest */
    /* WC_BIGINT doesn't have 0 len well on hardware */
    /* Cryptocell has issues with all 0 digest */
#if defined(ECC_SHAMIR) && !defined(WOLFSSL_ASYNC_CRYPT) && \
    !defined(WOLFSSL_CRYPTOCELL)
    /* test DSA sign hash with zeros */
    for (i = 0; i < (int)ECC_DIGEST_SIZE; i++) {
        digest[i] = 0;
    }

    x = ECC_SIG_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(digest, ECC_DIGEST_SIZE, sig, &x, rng,
                                                                        userA);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

#ifdef HAVE_ECC_VERIFY
    for (i=0; i<testVerifyCount; i++) {
        verify = 0;
        do {
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
        #endif
            if (ret == 0)
                ret = wc_ecc_verify_hash(sig, x, digest, ECC_DIGEST_SIZE,
                                                               &verify, userA);
        } while (ret == WC_PENDING_E);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
        if (verify != 1)
            ERROR_OUT(WC_TEST_RET_ENC_NC, done);
        TEST_SLEEP();
    }
#endif /* HAVE_ECC_VERIFY */
#endif /* ECC_SHAMIR && !WOLFSSL_ASYNC_CRYPT && !WOLFSSL_CRYPTOCELL */

    /* test DSA sign hash with sequence (0,1,2,3,4,...) */
    for (i = 0; i < (int)ECC_DIGEST_SIZE; i++) {
        digest[i] = (byte)i;
    }

    x = ECC_SIG_SIZE;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_sign_hash(digest, ECC_DIGEST_SIZE, sig, &x, rng, userA);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
    TEST_SLEEP();

#ifdef HAVE_ECC_VERIFY
    for (i=0; i<testVerifyCount; i++) {
        verify = 0;
        do {
        #if defined(WOLFSSL_ASYNC_CRYPT)
            ret = wc_AsyncWait(ret, &userA->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
        #endif
            if (ret == 0)
                ret = wc_ecc_verify_hash(sig, x, digest, ECC_DIGEST_SIZE, &verify, userA);
        } while (ret == WC_PENDING_E);
        if (ret != 0)
            ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
        if (verify != 1)
            ERROR_OUT(WC_TEST_RET_ENC_NC, done);
        TEST_SLEEP();
    }
#endif /* HAVE_ECC_VERIFY */
#endif /* HAVE_ECC_SIGN */
#endif /* !ECC_TIMING_RESISTANT || (ECC_TIMING_RESISTANT &&
        * !WC_NO_RNG && !WOLFSSL_KCAPI_ECC) */

#if defined(HAVE_ECC_KEY_EXPORT) && !defined(WC_NO_RNG) && \
    !defined(WOLFSSL_ATECC508) && !defined(WOLFSSL_ATECC608A) && \
    !defined(WOLFSSL_KCAPI_ECC)
    x = ECC_KEY_EXPORT_BUF_SIZE;
    ret = wc_ecc_export_private_only(userA, exportBuf, &x);
    if (ret != 0)
        ERROR_OUT(WC_TEST_RET_ENC_EC(ret), done);
#elif defined(HAVE_ECC_KEY_EXPORT)
    (void)exportBuf;
#endif /* HAVE_ECC_KEY_EXPORT */

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (userA != NULL) {
        wc_ecc_free(userA);
        XFREE(userA, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (userB != NULL) {
        wc_ecc_free(userB);
        XFREE(userB, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (pubKey != NULL) {
        wc_ecc_free(pubKey);
        XFREE(pubKey, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(pubKey);
    wc_ecc_free(userB);
    wc_ecc_free(userA);
#endif

#if defined(HAVE_ECC_DHE) || defined(HAVE_ECC_CDH)
    WC_FREE_VAR(sharedA, HEAP_HINT);
    WC_FREE_VAR(sharedB, HEAP_HINT);
#endif
#ifdef HAVE_ECC_KEY_EXPORT
    WC_FREE_VAR(exportBuf, HEAP_HINT);
#endif
#ifdef HAVE_ECC_SIGN
    WC_FREE_VAR(sig, HEAP_HINT);
    WC_FREE_VAR(digest, HEAP_HINT);
#endif

    (void)keySize;
    (void)curve_id;
    (void)rng;

    return ret;
}

#undef  ECC_TEST_VERIFY_COUNT
#define ECC_TEST_VERIFY_COUNT 2
static wc_test_ret_t ecc_test_curve(WC_RNG* rng, int keySize, int curve_id)
{
    wc_test_ret_t ret;

    ret = ecc_test_curve_size(rng, keySize, ECC_TEST_VERIFY_COUNT, curve_id,
        NULL);
    if (ret < 0) {
        if (ret == ECC_CURVE_OID_E) {
            /* ignore error for curves not found */
            /* some curve sizes are only available with:
                HAVE_ECC_SECPR2, HAVE_ECC_SECPR3, HAVE_ECC_BRAINPOOL
                and HAVE_ECC_KOBLITZ */
        }
        else {
            printf("ecc_test_curve_size %d failed!\n", keySize);
            return ret;
        }
    }
#ifndef WOLF_CRYPTO_CB_ONLY_ECC

#ifdef HAVE_ECC_VECTOR_TEST
    ret = ecc_test_vector(keySize);
    if (ret < 0) {
        printf("ecc_test_vector %d failed!\n", keySize);
        return ret;
    }
#endif

#if defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT) && \
    !defined(NO_ASN_CRYPT) && !defined(WC_NO_RNG)
    ret = ecc_test_key_decode(rng, keySize);
    if (ret < 0) {
        if (ret == ECC_CURVE_OID_E) {
            /* ignore error for curves not found */
        }
        else {
            printf("ecc_test_key_decode %d failed!\n", keySize);
            return ret;
        }
    }
#endif

#if defined(HAVE_ECC_KEY_EXPORT) && !defined(NO_ASN_CRYPT) && !defined(WC_NO_RNG)
    ret = ecc_test_key_gen(rng, keySize);
    if (ret < 0) {
        if (ret == ECC_CURVE_OID_E) {
            /* ignore error for curves not found */
        }
        else {
            printf("ecc_test_key_gen %d failed!\n", keySize);
            return ret;
        }
    }
#endif
#endif /* WOLF_CRYPTO_CB_ONLY_ECC */
    return 0;
}

#if (!defined(NO_ECC256) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 256
#if !defined(WOLFSSL_ATECC508A) && !defined(WOLFSSL_ATECC608A) && \
    defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT) && \
    !defined(WOLFSSL_NO_MALLOC) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static wc_test_ret_t ecc_point_test(void)
{
    wc_test_ret_t ret;
    ecc_point* point;
    ecc_point* point2;
#ifdef HAVE_COMP_KEY
    ecc_point* point3;
    ecc_point* point4;
#endif
    word32     outLen;
    byte       out[65];
    byte       der[] = { 0x04, /* = Uncompressed */
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                         0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };
#if defined(HAVE_COMP_KEY) && (!defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    byte       derComp0[] = { 0x02, /* = Compressed, y even */
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };
    byte       derComp1[] = { 0x03, /* = Compressed, y odd */
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
                              0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08 };
#endif
    byte       altDer[] = { 0x04, /* = Uncompressed */
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
                            0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07 };
    int curve_idx = wc_ecc_get_curve_idx(ECC_SECP256R1);

    /* if curve P256 is not enabled then test should not fail */
    if (curve_idx == ECC_CURVE_INVALID)
        return 0;

    outLen = sizeof(out);
    point = wc_ecc_new_point();
    if (point == NULL)
        return WC_TEST_RET_ENC_ERRNO;
    point2 = wc_ecc_new_point();
    if (point2 == NULL) {
        wc_ecc_del_point(point);
        return WC_TEST_RET_ENC_NC;
    }
#ifdef HAVE_COMP_KEY
    point3 = wc_ecc_new_point();
    if (point3 == NULL) {
        wc_ecc_del_point(point2);
        wc_ecc_del_point(point);
        return WC_TEST_RET_ENC_NC;
    }
    point4 = wc_ecc_new_point();
    if (point4 == NULL) {
        wc_ecc_del_point(point3);
        wc_ecc_del_point(point2);
        wc_ecc_del_point(point);
        return WC_TEST_RET_ENC_NC;
    }
#endif

    /* Parameter Validation testing. */
    wc_ecc_del_point(NULL);
    ret = wc_ecc_import_point_der(NULL, sizeof(der), curve_idx, point);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_import_point_der(der, sizeof(der), ECC_CURVE_INVALID, point);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_import_point_der(der, sizeof(der), curve_idx, NULL);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_export_point_der(-1, point, out, &outLen);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_export_point_der(curve_idx, NULL, out, &outLen);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_export_point_der(curve_idx, point, NULL, &outLen);
    if (ret != LENGTH_ONLY_E || outLen != sizeof(out)) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_export_point_der(curve_idx, point, out, NULL);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    outLen = 0;
    ret = wc_ecc_export_point_der(curve_idx, point, out, &outLen);
    if (ret != BUFFER_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_copy_point(NULL, NULL);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_copy_point(NULL, point2);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_copy_point(point, NULL);
    if (ret != ECC_BAD_ARG_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_cmp_point(NULL, NULL);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_cmp_point(NULL, point2);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_cmp_point(point, NULL);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    /* Use API. */
    ret = wc_ecc_import_point_der(der, sizeof(der), curve_idx, point);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    outLen = sizeof(out);
    ret = wc_ecc_export_point_der(curve_idx, point, out, &outLen);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    if (outLen != sizeof(der)) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }
    if (XMEMCMP(out, der, outLen) != 0) {
        ret = WC_TEST_RET_ENC_NC;
        goto done;
    }

    ret = wc_ecc_copy_point(point2, point);
    if (ret != MP_OKAY) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_cmp_point(point2, point);
    if (ret != MP_EQ) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    ret = wc_ecc_import_point_der(altDer, sizeof(altDer), curve_idx, point2);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_cmp_point(point2, point);
    if (ret != MP_GT) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

#if defined(HAVE_COMP_KEY) && (!defined(HAVE_FIPS) && !defined(HAVE_SELFTEST) || \
    (defined(HAVE_FIPS_VERSION) && (HAVE_FIPS_VERSION > 2)))
    ret = wc_ecc_import_point_der(derComp0, sizeof(derComp0)*2-1, curve_idx, point3);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    ret = wc_ecc_import_point_der_ex(derComp0, sizeof(derComp0), curve_idx, point4, 0);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    ret = wc_ecc_cmp_point(point3, point4);
    if (ret != MP_EQ) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    ret = wc_ecc_import_point_der(derComp1, sizeof(derComp1)*2-1, curve_idx, point3);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    ret = wc_ecc_import_point_der_ex(derComp1, sizeof(derComp1), curve_idx, point4, 0);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    ret = wc_ecc_cmp_point(point3, point4);
    if (ret != MP_EQ) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
#endif

done:
#ifdef HAVE_COMP_KEY
    wc_ecc_del_point(point4);
    wc_ecc_del_point(point3);
#endif
    wc_ecc_del_point(point2);
    wc_ecc_del_point(point);

    return ret;
}
#endif /* !WOLFSSL_ATECC508A && HAVE_ECC_KEY_IMPORT && HAVE_ECC_KEY_EXPORT */

#if !defined(NO_SIG_WRAPPER) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static wc_test_ret_t ecc_sig_test(WC_RNG* rng, ecc_key* key)
{
    wc_test_ret_t ret;
    word32  sigSz;
    int     size;
    byte    out[ECC_MAX_SIG_SIZE];
    byte    in[] = TEST_STRING;
    WOLFSSL_SMALL_STACK_STATIC const byte hash[] = {
        0xf2, 0x02, 0x95, 0x65, 0xcb, 0xf6, 0x2a, 0x59,
        0x39, 0x2c, 0x05, 0xff, 0x0e, 0x29, 0xaf, 0xfe,
        0x47, 0x33, 0x8c, 0x99, 0x8d, 0x58, 0x64, 0x83,
        0xa6, 0x58, 0x0a, 0x33, 0x0b, 0x84, 0x5f, 0x5f
    };
    word32 inLen = (word32)XSTRLEN((char*)in);

    size = wc_ecc_sig_size(key);

    ret = wc_SignatureGetSize(WC_SIGNATURE_TYPE_ECC, key, sizeof(*key));
    if (ret != size)
        return WC_TEST_RET_ENC_NC;

    sigSz = (word32)ret;
    ret = wc_SignatureGenerate(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_ECC, in,
                               inLen, out, &sigSz, key, sizeof(*key), rng);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    TEST_SLEEP();

    ret = wc_SignatureVerify(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_ECC, in,
                             inLen, out, sigSz, key, sizeof(*key));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    TEST_SLEEP();

    sigSz = (word32)sizeof(out);
    ret = wc_SignatureGenerateHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_ECC,
        hash, (int)sizeof(hash), out, &sigSz, key, sizeof(*key), rng);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    TEST_SLEEP();

    ret = wc_SignatureVerifyHash(WC_HASH_TYPE_SHA256, WC_SIGNATURE_TYPE_ECC,
        hash, (int)sizeof(hash), out, sigSz, key, sizeof(*key));
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
    TEST_SLEEP();

    return 0;
}
#endif

#if defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT) && \
   !defined(WOLF_CRYPTO_CB_ONLY_ECC)

static wc_test_ret_t ecc_exp_imp_test(ecc_key* key)
{
    wc_test_ret_t ret;
    int        curve_id;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key    *keyImp = (ecc_key *)XMALLOC(sizeof *keyImp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key    keyImp[1];
#endif
    byte       priv[32];
    word32     privLen;
    byte       pub[65*2];
    word32     pubLen, pubLenX, pubLenY;
    const char qx[] = "7a4e287890a1a47ad3457e52f2f76a83"
                      "ce46cbc947616d0cbaa82323818a793d";
    const char qy[] = "eec4084f5b29ebf29c44cce3b3059610"
                      "922f8b30ea6e8811742ac7238fe87308";
    const char d[]  = "8c14b793cb19137e323a6d2e2a870bca"
                      "2e7a493ec1153b3a95feb8a4873f8d08";

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (keyImp == NULL)
        ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
#endif

    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    privLen = sizeof(priv);
    ret = wc_ecc_export_private_only(key, priv, &privLen);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    pubLen = sizeof(pub);
    ret = wc_ecc_export_point_der(key->idx, &key->pubkey, pub, &pubLen);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    ret = wc_ecc_import_private_key(priv, privLen, pub, pubLen, keyImp);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    wc_ecc_free(keyImp);
    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    ret = wc_ecc_import_raw_ex(keyImp, qx, qy, d, ECC_SECP256R1);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    wc_ecc_free(keyImp);
    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    curve_id = wc_ecc_get_curve_id(key->idx);
    if (curve_id < 0) {
        ret = WC_TEST_RET_ENC_EC(curve_id);
        goto done;
    }

    /* test import private only */
    ret = wc_ecc_import_private_key_ex(priv, privLen, NULL, 0, keyImp,
                                       curve_id);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    wc_ecc_free(keyImp);
    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    /* test export public raw */
    pubLenX = pubLenY = 32;
    ret = wc_ecc_export_public_raw(key, pub, &pubLenX, &pub[32], &pubLenY);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

#ifndef HAVE_SELFTEST
    /* test import of public */
    ret = wc_ecc_import_unsigned(keyImp, pub, &pub[32], NULL, ECC_SECP256R1);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
#endif

    wc_ecc_free(keyImp);
    wc_ecc_init_ex(keyImp, HEAP_HINT, devId);

    /* test export private and public raw */
    pubLenX = pubLenY = privLen = 32;
    ret = wc_ecc_export_private_raw(key, pub, &pubLenX, &pub[32], &pubLenY,
        priv, &privLen);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

#ifndef HAVE_SELFTEST
    /* test import of private and public */
    ret = wc_ecc_import_unsigned(keyImp, pub, &pub[32], priv, ECC_SECP256R1);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
#endif

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (keyImp != NULL) {
        wc_ecc_free(keyImp);
        XFREE(keyImp, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(keyImp);
#endif

    return ret;
}
#endif /* HAVE_ECC_KEY_IMPORT && HAVE_ECC_KEY_EXPORT */

#if defined(HAVE_ECC_KEY_IMPORT) && !defined(WOLFSSL_VALIDATE_ECC_IMPORT) && \
    !defined(WOLFSSL_CRYPTOCELL) && !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static wc_test_ret_t ecc_mulmod_test(ecc_key* key1)
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key    *key2 = (ecc_key *)XMALLOC(sizeof *key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    ecc_key    *key3 = (ecc_key *)XMALLOC(sizeof *key3, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key    key2[1];
    ecc_key    key3[1];
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if ((key2 == NULL) || (key3 == NULL))
        ERROR_OUT(MEMORY_E, done);
#endif

    wc_ecc_init_ex(key2, HEAP_HINT, devId);
    wc_ecc_init_ex(key3, HEAP_HINT, devId);

    /* TODO: Use test data, test with WOLFSSL_VALIDATE_ECC_IMPORT. */
    /* Need base point (Gx,Gy) and parameter A - load them as the public and
     * private key in key2.
     */
    ret = wc_ecc_import_raw_ex(key2, key1->dp->Gx, key1->dp->Gy, key1->dp->Af,
                               ECC_SECP256R1);
    if (ret != 0)
        goto done;

    /* Need a point (Gx,Gy) and prime - load them as the public and private key
     * in key3.
     */
    ret = wc_ecc_import_raw_ex(key3, key1->dp->Gx, key1->dp->Gy,
                               key1->dp->prime, ECC_SECP256R1);
    if (ret != 0)
        goto done;

    ret = wc_ecc_mulmod(wc_ecc_key_get_priv(key1), &key2->pubkey, &key3->pubkey,
                        wc_ecc_key_get_priv(key2), wc_ecc_key_get_priv(key3),
                        1);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key2 != NULL) {
        wc_ecc_free(key2);
        XFREE(key2, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
    if (key3 != NULL) {
        wc_ecc_free(key3);
        XFREE(key3, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(key3);
    wc_ecc_free(key2);
#endif

    return ret;
}
#endif

#if defined(HAVE_ECC_DHE) && !defined(WC_NO_RNG) && \
    !defined(WOLF_CRYPTO_CB_ONLY_ECC)
static wc_test_ret_t ecc_ssh_test(ecc_key* key, WC_RNG* rng)
{
    wc_test_ret_t ret;
    byte   out[128];
    word32 outLen = sizeof(out);

    /* Parameter Validation testing. */
    ret = wc_ecc_shared_secret_ssh(NULL, &key->pubkey, out, &outLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_ecc_shared_secret_ssh(key, NULL, out, &outLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_ecc_shared_secret_ssh(key, &key->pubkey, NULL, &outLen);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);
    ret = wc_ecc_shared_secret_ssh(key, &key->pubkey, out, NULL);
    if (ret != BAD_FUNC_ARG)
        return WC_TEST_RET_ENC_EC(ret);

#if defined(ECC_TIMING_RESISTANT) && (!defined(HAVE_FIPS) || \
    (!defined(HAVE_FIPS_VERSION) || (HAVE_FIPS_VERSION != 2))) && \
    !defined(HAVE_SELFTEST)
    ret = wc_ecc_set_rng(key, rng);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);
#else
    (void)rng;
#endif

    /* Use API. */
    ret = 0;
    do {
    #if defined(WOLFSSL_ASYNC_CRYPT)
        ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_CALL_AGAIN);
    #endif
        if (ret == 0)
            ret = wc_ecc_shared_secret_ssh(key, &key->pubkey, out, &outLen);
    } while (ret == WC_PENDING_E);
    if (ret != 0)
        return WC_TEST_RET_ENC_EC(ret);

    TEST_SLEEP();
    return 0;
}
#endif /* HAVE_ECC_DHE && !WC_NO_RNG */

static wc_test_ret_t ecc_def_curve_test(WC_RNG *rng)
{
    wc_test_ret_t ret;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key *key = (ecc_key *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key key[1];
#endif
#if !defined(NO_ECC_SECP) && \
    ((defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT)) || \
     (defined(HAVE_ECC_KEY_IMPORT) && !defined(WOLFSSL_VALIDATE_ECC_IMPORT)))
    word32 idx = 0;
#endif

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key == NULL)
        ERROR_OUT(MEMORY_E, done);
#endif

    wc_ecc_init_ex(key, HEAP_HINT, devId);

    /* Use API */
    ret = wc_ecc_set_flags(NULL, 0);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_ecc_set_flags(key, 0);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
#ifndef WOLF_CRYPTO_CB_ONLY_ECC
#ifndef WC_NO_RNG
    ret = wc_ecc_make_key(rng, ECC_KEYGEN_SIZE, key);
    #if defined(WOLFSSL_ASYNC_CRYPT)
    ret = wc_AsyncWait(ret, &key->asyncDev, WC_ASYNC_FLAG_NONE);
    #endif
    if (ret != 0) {
        goto done;
    }

    #ifndef NO_SIG_WRAPPER
    ret = ecc_sig_test(rng, key);
    if (ret < 0)
        goto done;
    #endif
    TEST_SLEEP();

    #if defined(HAVE_ECC_DHE) && !defined(WOLFSSL_CRYPTOCELL) && \
       !defined(WOLF_CRYPTO_CB_ONLY_ECC)
    ret = ecc_ssh_test(key, rng);
    if (ret < 0)
        goto done;
    #endif

    wc_ecc_free(key);
#else
    (void)rng;
#endif /* !WC_NO_RNG */

#if !defined(NO_ECC_SECP) && \
    ((defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT)) || \
     (defined(HAVE_ECC_KEY_IMPORT) && !defined(WOLFSSL_VALIDATE_ECC_IMPORT)))
    /* Use test ECC key - ensure real private "d" exists */
    #ifdef USE_CERT_BUFFERS_256
    ret = wc_EccPrivateKeyDecode(ecc_key_der_256, &idx, key,
        sizeof_ecc_key_der_256);
    #else
    {
        XFILE file = XFOPEN(eccKeyDerFile, "rb");
        byte der[128];
        word32 derSz;
        if (!file) {
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
        }
        derSz = (word32)XFREAD(der, 1, sizeof(der), file);
        XFCLOSE(file);
        if (derSz == 0)
            ERROR_OUT(WC_TEST_RET_ENC_ERRNO, done);
        ret = wc_EccPrivateKeyDecode(der, &idx, key, derSz);
    }
    #endif
    if (ret != 0) {
        goto done;
    }

#if defined(HAVE_ECC_KEY_IMPORT) && defined(HAVE_ECC_KEY_EXPORT)
    ret = ecc_exp_imp_test(key);
    if (ret < 0)
        goto done;
#endif
#if defined(HAVE_ECC_KEY_IMPORT) && !defined(WOLFSSL_VALIDATE_ECC_IMPORT) && \
    !defined(WOLFSSL_CRYPTOCELL)
    ret = ecc_mulmod_test(key);
    if (ret < 0)
        goto done;
#endif
#endif
#else
   (void)rng;
   (void)idx;
#endif /* WOLF_CRYPTO_CB_ONLY_ECC */
done:

    wc_ecc_free(key);
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key != NULL) {
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#endif

    return ret;
}
#endif /* !NO_ECC256 || HAVE_ALL_CURVES */

#if defined(WOLFSSL_CERT_EXT) && \
    (!defined(NO_ECC256) || defined(HAVE_ALL_CURVES)) && ECC_MIN_KEY_SZ <= 256
static wc_test_ret_t ecc_decode_test(void)
{
    wc_test_ret_t ret;
    word32     inSz;
    word32     inOutIdx;
#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    ecc_key    *key = (ecc_key *)XMALLOC(sizeof *key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
#else
    ecc_key    key[1];
#endif

    /* SECP256R1 OID: 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07 */

    /* This is ecc_clikeypub_der_256. */
    WOLFSSL_SMALL_STACK_STATIC const byte good[] = {
            0x30, 0x59, 0x30, 0x13, 0x06, 0x07, 0x2a, 0x86, 0x48, 0xce,
            0x3d, 0x02, 0x01, 0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d,
            0x03, 0x01, 0x07, 0x03, 0x42, 0x00, 0x04, 0x55, 0xbf, 0xf4,
            0x0f, 0x44, 0x50, 0x9a, 0x3d, 0xce, 0x9b, 0xb7, 0xf0, 0xc5,
            0x4d, 0xf5, 0x70, 0x7b, 0xd4, 0xec, 0x24, 0x8e, 0x19, 0x80,
            0xec, 0x5a, 0x4c, 0xa2, 0x24, 0x03, 0x62, 0x2c, 0x9b, 0xda,
            0xef, 0xa2, 0x35, 0x12, 0x43, 0x84, 0x76, 0x16, 0xc6, 0x56,
            0x95, 0x06, 0xcc, 0x01, 0xa9, 0xbd, 0xf6, 0x75, 0x1a, 0x42,
            0xf7, 0xbd, 0xa9, 0xb2, 0x36, 0x22, 0x5f, 0xc7, 0x5d, 0x7f,
            0xb4 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNoObjId[] = { 0x30, 0x08, 0x30, 0x06, 0x03, 0x04,
            0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badOneObjId[] = { 0x30, 0x0a, 0x30, 0x08, 0x06, 0x00,
            0x03, 0x04, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badObjId1Len[] = { 0x30, 0x0c, 0x30, 0x0a, 0x06, 0x09,
            0x06, 0x00, 0x03, 0x04, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badObj2d1Len[] = { 0x30, 0x0c, 0x30, 0x0a, 0x06, 0x00,
            0x06, 0x07, 0x03, 0x04, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNotBitStr[] = { 0x30, 0x14, 0x30, 0x0b, 0x06, 0x00,
            0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
            0x04, 0x04, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badBitStrLen[] = { 0x30, 0x14, 0x30, 0x0b, 0x06, 0x00,
            0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
            0x03, 0x05, 0x00, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badNoBitStrZero[] = { 0x30, 0x13, 0x30, 0x0a, 0x06, 0x00,
            0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
            0x03, 0x03, 0x04, 0x01, 0x01 };
    WOLFSSL_SMALL_STACK_STATIC const byte badPoint[] = { 0x30, 0x12, 0x30, 0x09, 0x06, 0x00,
            0x06, 0x08, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x03, 0x01, 0x07,
            0x03, 0x03, 0x00, 0x04, 0x01 };

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key == NULL)
        ERROR_OUT(MEMORY_E, done);
#endif

    XMEMSET(key, 0, sizeof *key);
    wc_ecc_init_ex(key, HEAP_HINT, devId);

    inSz = sizeof(good);
    ret = wc_EccPublicKeyDecode(NULL, &inOutIdx, key, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_EccPublicKeyDecode(good, NULL, key, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, NULL, inSz);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, key, 0);
    if (ret != BAD_FUNC_ARG) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    /* Change offset to produce bad input data. */
    inOutIdx = 2;
    inSz = sizeof(good) - inOutIdx;
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inOutIdx = 4;
    inSz = sizeof(good) - inOutIdx;
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    /* Bad data. */
    inSz = sizeof(badNoObjId);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badNoObjId, &inOutIdx, key, inSz);
    if (ret != ASN_OBJECT_ID_E && ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badOneObjId);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badOneObjId, &inOutIdx, key, inSz);
    if (ret != ASN_OBJECT_ID_E && ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badObjId1Len);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badObjId1Len, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badObj2d1Len);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badObj2d1Len, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badNotBitStr);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badNotBitStr, &inOutIdx, key, inSz);
    if (ret != ASN_BITSTR_E && ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badBitStrLen);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badBitStrLen, &inOutIdx, key, inSz);
    if (ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badNoBitStrZero);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badNoBitStrZero, &inOutIdx, key, inSz);
    if (ret != ASN_EXPECT_0_E && ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }
    inSz = sizeof(badPoint);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(badPoint, &inOutIdx, key, inSz);
    if (ret != ASN_ECC_KEY_E && ret != ASN_PARSE_E) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

    inSz = sizeof(good);
    inOutIdx = 0;
    ret = wc_EccPublicKeyDecode(good, &inOutIdx, key, inSz);
    if (ret != 0) {
        ret = WC_TEST_RET_ENC_EC(ret);
        goto done;
    }

done:

#if defined(WOLFSSL_SMALL_STACK) && !defined(WOLFSSL_NO_MALLOC)
    if (key != NULL) {
        wc_ecc_free(key);
        XFREE(key, HEAP_HINT, DYNAMIC_TYPE_TMP_BUFFER);
    }
#else
    wc_ecc_free(key);
#endif

    return ret;
}
#endif /* WOLFSSL_CERT_EXT */

#ifdef WOLFSSL_CUSTOM_CURVES
static const byte eccKeyExplicitCurve[] = {
    0x30, 0x81, 0xf5, 0x30, 0x81, 0xae, 0x06, 0x07,
 