/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include "engines/game.h"
#include "common/gui_options.h"
#include "common/language.h"

namespace Glk {
namespace Frotz {

/**
 * Game descriptor for ZCode games
 */
struct FrotzGameDescription {
	const char *const _gameId;
	const char *const _extra;
	const char *const _md5;
	size_t _filesize;
	Common::Language _language;
	const char *const _guiOptions;
};

/**
 * Original games from Infocom
 */
const PlainGameDescriptor INFOCOM_GAME_LIST[] = {
	// Infocom games
	{ "amfv", "A Mind Forever Voyaging" },
	{ "arthur", "Arthur: The Quest for Excalibur" },
	{ "ballyhoo", "Ballyhoo" },
	{ "beyondzork", "Beyond Zork" },
	{ "borderzone", "Border Zone" },
	{ "bureaucracy", "Bureaucracy" },
	{ "cutthroats", "Cutthroats" },
	{ "deadline", "Deadline" },
	{ "enchanter", "Enchanter" },
	{ "hhgttg", "The Hitchhiker's Guide to the Galaxy" },
	{ "hollywoodhijinx", "Hollywood Hijinx" },
	{ "infidel", "Infidel" },
	{ "journey", "Journey" },
	{ "lgop", "Leather Goddesses of Phobos" },
	{ "lurkinghorror", "The Lurking Horror" },
	{ "minizork1", "Mini Zork I: The Great Underground Empire" },
	{ "moonmist", "Moonmist" },
	{ "nordbert", "Nord and Bert Couldn't Make Head or Tail of It" },
	{ "planetfall", "Planetfall" },
	{ "plunderedhearts", "Plundered Hearts" },
	{ "infocomsampler1", "Infocom Sampler 1" },
	{ "infocomsampler2", "Infocom Sampler 2" },
	{ "seastalker", "Seastalker" },
	{ "sherlockriddle", "Sherlock: The Riddle of the Crown Jewels" },
	{ "shogun", "James Clavell's Shogun" },
	{ "sorcerer", "Sorcerer" },
	{ "spellbreaker", "Spellbreaker" },
	{ "starcross", "Starcross" },
	{ "stationfall", "Stationfall" },
	{ "suspect", "Suspect" },
	{ "suspended", "Suspended" },
	{ "trinity", "Trinity" },
	{ "wishbringer", "Wishbringer" },
	{ "thewitness", "The Witness" },
	{ "zork0", "Zork Zero: The Revenge of Megaboz" },
	{ "zork1", "Zork I: The Great Underground Empire" },
	{ "zork2", "Zork II: The Wizard of Frobozz" },
	{ "zork3", "Zork III: The Dungeon Master" },
	{ "ztuu", "Zork: The Undiscovered Underground" },
	{ nullptr, nullptr }
};

/**
 * All the other subsequent non-Infocom games using the format
 */
const PlainGameDescriptor ZCODE_GAME_LIST[] = {
	{ "zcode", "Unknown Z-code game" },

	// English games
	{ "404life",           "404 - Life not found" },
	{ "69105keys",         "69,105 Keys" },
	{ "905",               "9:05" },
	{ "9dancers",          "The Nine Dancers (Larsoft Adventure number 4)" },
	{ "cockandbull",       "A Cock and Bull Story" },
	{ "aasmasters",        "AAS Masters, in which all is revealed" },
	{ "accuse",            "Accuse" },
	{ "acheton",           "Acheton" },
	{ "acorncourt",        "The Acorn Court" },
	{ "acrobat",           "The Mysterious Case of the Acrobat and His Peers" },
	{ "acrossstars",       "Across the Stars" },
	{ "acrossstarsclues",  "Across the Stars: Invisiclues" },
	{ "addendum",          "Flawed Addendum" },
	{ "adv",               "Adventure, Colossal Cave" },
	{ "adv350",            "Adventure, 350 point Colossal Cave" },
	{ "adv440",            "Adventure II, 440 point Colossal Cave" },
	{ "adv550",            "Adventure 3, 550 point Colossal Cave" },
	{ "adv551",            "Adventure 6, 551 point Colossal Cave" },
	{ "adventuretime",     "Adventure Time" },
	{ "adverbum",          "Ad Verbum" },
	{ "affront",           "Annoyotron IV: Affrontotron" },
	{ "aisle",             "Aisle" },
	{ "alice",             "Alice Through the Looking Glass" },
	{ "allroads",          "All Roads" },
	{ "alongtheriver",     "Along the River" },
	{ "alpha",             "Journey to Alpha Centauri (In Real Time)" },
	{ "ambassadorsdaughter", "The Ambassador''s Daughter, a Brief Romance" },
	{ "amish",             "Amishville" },
	{ "amiss",             "Amissville" },
	{ "anchor",            "Anchorhead: an Interactive Tale of Lovecraftian Horror" },
	{ "hipponewyear",      "And A Hippo New Year" },
	{ "animals",           "Animals 1.1" },
	{ "annoy",             "Annoyotron" },
	{ "aotyrz",            "Attack of the Yeti Robot Zombies" },
	{ "aphasiaquest",      "Aphasia Quest" },
	{ "appall",            "Appallatron: Annoyotron 3" },
	{ "aridandpale",       "Arid and Pale" },
	{ "asylum",            "Asylum" },
	{ "atrocitron",        "Atrocitron, An Interactive Puzzlebox" },
	{ "ats",               "A Tight Spot" },
	{ "atwork",            "Danger! Adventurer At Work!" },
	{ "aug4",              "Augmented Fourth, an Interactive Performance" },
	{ "avon",              "Avon" },
	{ "awakening",         "The Awakening" },
	{ "awitl",             "A Week In The Life" },
	{ "ayac",              "Are You A Chef? An interactive ifMUD-saving" },
	{ "b2demo",            "Pick up the Phone Booth and Die, Part 2" },
	{ "b7snare",           "Snare, an interactive Blake's 7 adventure" },
	{ "babytree",          "Baby tree" },
	{ "backtowakeUp",      "Back to WakeUp, a Backpacker Tale" },
	{ "martyquest",        "Back to the Future - Marty Quest: an adventure through time" },
	{ "backup",            "Backup, an Interactive System Failure" },
	{ "balances",          "Balances, An Interactive Short Story" },
	{ "baldersdeath",      "Balder's Death" },
	{ "ballerina102",      "Not Just an Ordinary Ballerina" },
	{ "balt24",            "Baltimore:24, An Exercise in Interactive Fiction" },
	{ "bathtub",           "There's a Snake in the Bathtub, a Test of Patience" },
	{ "bazic",             "baZic version 0.1, Z-Machine BASIC implementation" },
	{ "bear",              "A Bear's Night Out, an Interactive Children's Story" },
	{ "bedlam",            "Bedlam, An Interactive Preview" },
	{ "bedtime",           "Bed Time" },
	{ "andrewplotkin",     "Being Andrew Plotkin" },
	{ "beingsteve",        "Being Steve" },
	{ "beneath",           "Beneath: a Transformation" },
	{ "beyond",            "Beyond" },
	{ "bibleretold",       "The Bible Retold: The Bread and the Fishes" },
	{ "bicon",             "BiCon, an Interactive Infatuation" },
	{ "biscuit",           "Biscuit, An Interactive Funeral" },
	{ "bishoes",           "Buried In Shoes" },
	{ "bj",                "Blow Job Drifter, An Interactive You Kno What" },
	{ "blair",             "Tales From The College Presents A Breath Of Fresh Blair" },
	{ "bluechairs",        "Blue Chairs" },
	{ "blues",             "Tinseltown Blues: A quest for success in Hollyweird" },
	{ "bofh",              "The Bastard Operator from Hell" },
	{ "bomber",            "The Mad Bomber" },
	{ "bookvol",           "Book and Volume" },
	{ "booth",             "Pick up the Phone Booth and Die" },
	{ "boothdye",          "Pick up the Phone Booth and Dye" },
	{ "brainnightguest",   "Brain of the Night Guest" },
	{ "brandx",            "BrandX" },
	{ "breakin",           "Break-In, an Interactive Burglary" },
	{ "bronze",            "Bronze, Inform 7 demonstration game" },
	{ "bryantcollection",  "The Bryant Collection, an Interactive Anthology" },
	{ "bse",               "BSE, An Interactive Epidemic" },
	{ "building",          "Building" },
	{ "nightbunnies",      "The Night of the Vampire Bunnies" },
	{ "bureaucrocy",       "Bureaucrocy, a frustrating tale about paperwork" },
	{ "burglar",           "Burglar! A Learning Experience" },
	{ "burnsnightsupper",  "Burns Night Supper" },
	{ "burnkoran",         "Burn The Koran and Die" },
	{ "busted",            "Busted! A game of high cunning and low humor" },
	{ "byzantine",         "Byzantine Perspective" },
	{ "cabal",             "The Cabal: The Interactive Illuminati" },
	{ "cacophony",         "Cacophony, an interactive stumbling" },
	{ "calendar",          "Calendar, an Inform 7 abuse" },
	{ "calmmutemoving",    "Calm, Mute, Moving" },
	{ "calypso",           "Calypso" },
	{ "candy",             "Candy, An Attempt at Reliving Childhood" },
	{ "cars",              "ASCII Cars!!! A game of racing with the imagination" },
	{ "casting",           "Casting" },
	{ "castleadventure",   "Castle Adventure!" },
	{ "castleredprince",   "Castle of the Red Prince, an interactive land of darkness" },
	{ "siliconcastles",    "Silicon Castles" },
	{ "catcherintherye",   "Catcher in the Rye" },
	{ "catseye",           "Cat's Eye, Miniventure #2" },
	{ "causality",         "Causality: The Search for Eternal Life" },
	{ "caveadventure",     "Cave Adventure" },
	{ "cavernofdoom",      "Zork: The Cavern of Doom" },
	{ "cavernsofchaos",    "Caverns of Chaos" },
	{ "cavetrip",          "The Spelunking Trip" },
	{ "ccake",             "Arthur Yahtzee: The Curse of Hell's Cheesecake" },
	{ "chaos",             "Chaos" },
	{ "chaosgame",         "Chaos" },
	{ "cheater",           "Cheater, An Annoying Adventure" },
	{ "cheesedoff",        "Cheesed Off!" },
	{ "cheeseshop",        "Cheeseshop" },
	{ "cheshirecat",       "Save the Cheshire Cat!" },
	{ "chico",             "Chico and I Ran" },
	{ "childsplay",        "Child's Play, A child, a toy, and a rival" },
	{ "chix",              "Chicks Dig Jerks" },
	{ "cia",               "CIA Adventure" },
	{ "claw",              "Wearing the Claw" },
	{ "codenamesilver",    "Code Name Silver Steel" },
	{ "cointoss",          "Coin toss" },
	{ "coke",              "Coke Is It!" },
	{ "coldiron",          "Cold Iron" },
	{ "colonists",         "Colonists" },
	{ "coloromc",          "Color of Milk Coffee" },
	{ "comp96",            "Yearly IF competitions unofficial 'front-end' game" },
	{ "conankill",         "Conan Kill Everything" },
	{ "dreamcorruptor",    "Corrupter of Dreams" },
	{ "cottage",           "Cottage" },
	{ "cove",              "The Cove" },
	{ "creepydemo",        "A Day in the Creepy Life of Bob Demo" },
	{ "crimescene",        "A Crime Scene, a Short Story" },
	{ "criticalbreach",    "Critical Breach, an Escape Story" },
	{ "crobe",             "Crobe" },
	{ "cryptographer",     "Cryptographer" },
	{ "crystalpalace",     "The Crystal Palace" },
	{ "csbb",              "Crystal and Stone Beetle and Bone" },
	{ "ctdoom",            "Countdown to Doom" },
	{ "curses",            "Curses, An Interactive Diversion" },
	{ "curves",            "Dangerous Curves" },
	{ "cycles",            "Vicious Cycles" },
	{ "cyclops",           "The Land of the Cyclops" },
	{ "dday",              "D-Day" },
	{ "damnatiomemoriae",  "Damnatio Memoriae" },
	{ "darkiss1",          "Darkiss! Wrath of the Vampire - Chapter 1: The Awakening" },
	{ "dashslapney",       "Dash Slapney, Patrol Leader" },
	{ "dayinlife",         "A Day in Life" },
	{ "dd4",               "Dutch Dapper IV: The Final Voyage" },
	{ "deadmansgrave",     "Dead Man's Grave: A Tell Don't Show Mystery" },
	{ "deadmeat",          "Dead Meat in the Pit" },
	{ "deadpavane",        "Dead Pavane for a Princess" },
	{ "deadreckoning",     "Dead Reckoning" },
	{ "deadsville",        "Deadsville" },
	{ "death",             "Death to my Enemies" },
	{ "debate",            "Debate" },
	{ "deephome",          "Deephome" },
	{ "degeneracy",        "Degeneracy" },
	{ "dejavu",            "Deja Vu" },
	{ "deliciousbreakfast", "Delicious Breakfast" },
	{ "delusions",         "Delusions" },
	{ "detective",         "Detective" },
	{ "detention",         "Detention, an attempt to escape from school" },
	{ "devildoit",         "The Devil Made Me Do It" },
	{ "devours",           "All Things Devours" },
	{ "dewdrops",          "Within a Wreath of Dewdrops" },
	{ "djinni",            "The Djinni Chronicles \"Undercurrents of Manipulation\"" },
	{ "dogslife",          "It's a Dog's Life" },
	{ "zunidoll",          "The Zuni Doll" },
	{ "dontgo",            "Don't Go" },
	{ "dontpeeyourself",   "Don't Pee Yourself!" },
	{ "dotd",              "Dawn of the Demon" },
	{ "downthematrix",     "Down The Matrix" },
	{ "dpod",              "Dracula - Prince of Darkness" },
	{ "dracula2",          "Dracula: Part 2, The Arrival" },
	{ "dracula1",          "Dracula: Part 1, The First Night" },
	{ "dragon",            "Dragon Adventure" },
	{ "dragonflies",       "Dragon Flies Like Labradorite" },
	{ "dragontroll",       "The Dragon and the Troll" },
	{ "dreamhold",         "The Dreamhold" },
	{ "dreamtooreal",      "A Dream Too Real" },
	{ "dual",              "Dual Transform" },
	{ "dumont",            "Dr. Dumont's Wild P.A.R.T.I" },
	{ "eas",               "Earth And Sky: Episode 1" },
	{ "cliffedge",         "Edge of the Cliff" },
	{ "edifice",           "The Edifice" },
	{ "egyptianwalking",   "Egyptian Walking Simulator" },
	{ "elephants",         "When I Was Shot By Elephants III" },
	{ "eleven",            "Film at Eleven" },
	{ "eliza",             "Eliza" },
	{ "enemies",           "Enemies" },
	{ "enigma",            "Enimga" },
	{ "enterprise",        "The Enterprise Incidents" },
	{ "entropy",           "Entropy" },
	{ "epyk",              "Eypk" },
	{ "erden",             "Travels in the Land of Erden: In Quest of the Adventure" },
	{ "eric",              "Eric The Power-Mad Dungeon Master" },
	{ "escape",            "Escape!" },
	{ "escaperemember",    "An Escape To Remember" },
	{ "eurydice",          "Eurydice" },
	{ "f209",              "Apartment 209" },
	{ "fable",             "A Fable" },
	{ "faculty",           "The Care and Feeding of Adjuncts" },
	{ "failsafe",          "FailSafe" },
	{ "farm",              "The Farmer's Daughter" },
	{ "fff",               "Fox, Fowl and Feed" },
	{ "figaro",            "Figaro" },
	{ "figueres",          "Figueres in my Basement" },
	{ "findesick",         "Fin de sickleburg" },
	{ "findthebed",        "Find the Bed" },
	{ "finetuned",         "Fine-Tuned" },
	{ "fingertipsfriend",  "Fingertips: I Found a New Friend" },
	{ "fingertipsmilk",    "Fingertips: Please Pass the Milk Please" },
	{ "firstday",          "The First Day of My New Life" },
	{ "fmvpoker",          "Frobozz Magic Video Poker" },
	{ "forestdemo",        "Forest Demo" },
	{ "fork",              "Fork: The Great Underground Dining Room" },
	{ "hiddennazi",        "The Game Formerly Known as Hidden Nazi Mode" },
	{ "forms",             "Of Forms Unknown" },
	{ "fracture",          "Fractured Metamorphoses (Example Version)" },
	{ "fragileshells",     "Fragile Shells" },
	{ "frankie",           "Frankenstein Adventure" },
	{ "freefall",          "Free Fall" },
	{ "frobozzi",          "The Encyclopedia Frobozzica (Abridged Edition)" },
	{ "frozen",            "Frozen: A Night at the Lab" },
	{ "fyleet",            "Fyleet" },
	{ "ga",                "Geocaching Adventure - GC3JJ9C - Cryptic Puzzle #6" },
	{ "galatea",           "Galatea" },
	{ "gamer",             "Gamer: Digital Limbo" },
	{ "gamlet",            "Gamlet" },
	{ "gardening",         "Gardening for Beginners" },
	{ "gatoron",           "Gator-On, Friend to Wetlands!" },
	{ "gaucho",            "Gaucho" },
	{ "gd",                "Goodbye Doggy" },
	{ "geb",               "Goose, Egg, Badger" },
	{ "geist",             "Geist" },
	{ "gerbilriot",        "Gerbil Riot" },
	{ "ghost",             "The Ghost Train" },
	{ "glass",             "Glass, a fractured fairy tale" },
	{ "glik1",             "Glik part 1: Undead Menace" },
	{ "gnuzoo",            "Gnu in the Zoo" },
	{ "godot",             "Looking For Godot" },
	{ "goldilocks",        "Goldilocks is a Fox!" },
	{ "golf",              "Textfire Golf" },
	{ "gostak",            "The Gostak" },
	{ "gourmet",           "Gourmet" },
	{ "gourmetgaffe",      "Gourmet Gaffe" },
	{ "gowest",            "Go West" },
	{ "greatxavio",        "The Great Xavio" },
	{ "greenrain",         "A Green Rain" },
	{ "growingup",         "Growing Up" },
	{ "grue",              "GRUE" },
	{ "guard",             "Guard Duty" },
	{ "guess",             "Guess The Verb!" },
	{ "guestreet",         "Life on Gue Street" },
	{ "gumshoe",           "Gumshoe" },
	{ "gussdeath",         "Guss's Death" },
	{ "halloweve",         "Hallow Eve" },
	{ "hamhouse",          "In the House of Professor Evil: The HAM HOUSE" },
	{ "hamil",             "Hamil" },
	{ "hangar22",          "Hangar 22" },
	{ "happyeverafter",    "Happy Ever After" },
	{ "crabhat",           "Oh No, Mr Crab Stole Your Hat!" },
	{ "hauntedhouse",      "Haunted House" },
	{ "hauntings",         "Hauntings" },
	{ "heist",             "Heist: The Crime of the Century" },
	{ "heliopause",        "Hoist Sail for the Heliopause and Home" },
	{ "welcometohell",     "Welcome to Hell" },
	{ "hellosword",        "Hello Sword - The journey" },
	{ "hellsbasement",     "Hell's Basement" },
	{ "helpcollides",      "When Help Collides: The Wreck of the H.M.S. Snark" },
	{ "heroes",            "Heroes" },
	{ "heroinesmantle",    "Heroine's Mantle" },
	{ "hidepachyderm",     "Hide a pachyderm!" },
	{ "hlainform",         "HLA Inform: A Classic Quest" },
	{ "hobbittruestory",   "The Hobbit - The True Story" },
	{ "robotempire",       "Holy Robot Empire" },
	{ "home",              "Home" },
	{ "homecoming",        "Homecoming" },
	{ "hoosegow",          "Hoosegow, a Wild West Wreck" },
	{ "housekey1",         "Housekey, Part I" },
	{ "housedream",        "House of Dream of Moon" },
	{ "humongouscave",     "Adventure in Humongous Cave" },
	{ "humongouscavehints", "Humongous Cave Hints" },
	{ "hummingbird",       "Flight of the Hummingbird" },
	{ "hunterdark",        "Hunter, in Darkness" },
	{ "hyperrpg",          "Hyper RPG Game!" },
	{ "i0",                "I-0: the \"jailbait on the interstate\" game" },
	{ "ibo",               "Ibo" },
	{ "iceweb",            "Iceweb" },
	{ "identity",          "Identity" },
	{ "ifwhispers5",       "IF Whispers 5" },
	{ "ifaquarium",        "IF Aquarium" },
	{ "ifquake",           "Text adventure Quake Level 1" },
	{ "ill",               "I'll" },
	{ "imiagination",      "Imiagination" },
	{ "cubicle",           "In The Cubicle" },
	{ "inamanor",          "In a Manor of Speaking" },
	{ "inevita",           "Inevitable" },
	{ "informatory",       "Informatory" },
	{ "inhumane",          "Inhumane: An Infralogic Massacre" },
	{ "insight",           "Insight" },
	{ "intangible",        "Intangible" },
	{ "ifplayer",          "Interactive Fiction Player" },
	{ "interviewrockstar", "Interview with a Rock Star" },
	{ "lionskin",          "In the Skin of a Lion Quest: Caravaggio's Journey" },
	{ "intruder",          "Intruder - Interactive Fiction: A Burglary" },
	{ "invisibleadv",      "The Believable Adventures of an Invisible Man" },
	{ "pressedon",         "I pressed on, being chased by a stapler with my name on it" },
	{ "iraqiinvasion",     "Iraqi Invasion: A Text Misadventure" },
	{ "islandsfaraway",    "Islands Far Away" },
	{ "beanstalker",       "The Bean Stalker" },
	{ "jetblue",           "Jet-Blue" },
	{ "jewelofknowledge",  "The Jewel of Knowledge" },
	{ "jigsaw",            "Jigsaw, An Interactive History" },
	{ "jigsawrules",       "Jigsaw: Rules and Footnotes" },
	{ "juicehead",         "Juicehead, an interactive binge" },
	{ "justanotherday",    "Just Another Day" },
	{ "karn",              "Return to Karn" },
	{ "keepingdido",       "Keeping Dido" },
	{ "kidnapsea",         "Kidnapped - On the Sea" },
	{ "kierkegaardsspider", "Kierkegaard's Spider" },
	{ "kiiwii",            "Kii!Wii! A tiny friend to brighten your day" },
	{ "killingthedoctor",  "Killing the Doctor" },
	{ "kirby",             "The X-Child, Kevin Johnson Residence Hall Saga 2" },
	{ "kitten",            "robotfindskitten" },
	{ "kooku",             "Kook U" },
	{ "lambs",             "Silence of the Lambs" },
	{ "lambs2",            "Silence of the Lambs 2" },
	{ "largemachine",      "Large Machine" },
	{ "lash",              "Local Asynchronous Satellite Hookup" },
	{ "ldodoom",           "Last Days of Doom" },
	{ "leaptime",          "Leap Time" },
	{ "lecture",           "Lecture Feature" },
	{ "lex",               "LeX" },
	{ "librarian",         "Life of A Librarian" },
	{ "libraryfront",      "All Quiet on the Library Front" },
	{ "lifeordeath",       "Life or Death" },
	{ "blacklily",         "The Black Lily" },
	{ "limp",              "Limp" },
	{ "nemeanlion",        "The Nemean Lion: another anonymous joke game" },
	{ "lists",             "Lists and Lists" },
	{ "livejoseph",        "LiveJoseph" },
	{ "llr3",              "A Little Like Rogue" },
	{ "lmstvg",            "LMS The Video Game" },
	{ "lookingtothesky",   "Looking to the Sky" },
	{ "motherloose",       "Mother Loose" },
	{ "lostspellmaker",    "The Lost Spellmaker" },
	{ "lostpig",           "Lost Pig And Place Under Ground" },
	{ "lostsheep",         "The Bible Retold: The Lost Sheep" },
	{ "ludicorp",           "The Ludicorp Mystery" },
	{ "magictoyshop",      "The Magic Toyshop" },
	{ "magicmuffin",       "Magic Muffin - The Desert" },
	{ "makeitgood",        "Make it Good" },
	{ "mansion",           "Mansion" },
	{ "revenger",          "Revenger" },
	{ "masquerade",        "Masquerade" },
	{ "medusa",            "Medusa, NPC Conversations Example" },
	{ "mercy",             "Mercy" },
	{ "metamorphoses",     "Metamorphoses" },
	{ "mhpquest",          "Quest for the Magic Healing Plant" },
	{ "midpoints",         "Midpoints" },
	{ "mimesis",           "Sins Against Mimesis" },
	{ "mindelectric",      "The Mind Electric" },
	{ "mines",             "Mines" },
	{ "christminster",     "Christminster" },
	{ "misdirection",      "The Act Of Misdirection" },
	{ "missinggrandpa",    "Missing Grandpa: Lost in Time" },
	{ "mobius",            "Mobius" },
	{ "monstermaker",      "Monster Maker" },
	{ "monzasphantom",     "Monza's Phantom" },
	{ "moonshaped",        "Moon-Shaped" },
	{ "moonglow",          "Moonglow, Miniventure #1" },
	{ "moonwrecked",       "Moonwrecked" },
	{ "more",              "More" },
	{ "mortlakemanor",     "Mortlake Manor" },
	{ "mountain",          "Mountain, an Interactive Expedition" },
	{ "mousequest",        "Mouse Quest Chapter 1 - The Arrival of Winter" },
	{ "mrp",               "The Story of Mr. P." },
	{ "mst3k1",            "Detective, An Interactive MiSTing (Mystery Science Theater 3000)" },
	{ "mst3k2",            "A Fable, interactive MiSTing-up of \"A Fable\"" },
	{ "mulldoon",          "The Mulldoon Legacy" },
	{ "mulldoonmurders",   "The Mulldoon Murders" },
	{ "murdac",            "Murdac" },
	{ "murdererleft",      "What The Murderer Had Left" },
	{ "muse",              "Muse: An Autumn Romance" },
	{ "musician",          "The Musician" },
	{ "myangel",           "My Angel" },
	{ "mylastduchess",     "My Last Duchess" },
	{ "conceptisproven",   "My Little Project Concept is Proven" },
	{ "oceantower",        "Love, Hate and the Mysterious Ocean Tower" },
	{ "nameless",          "Endless, Nameless" },
	{ "nautilisia",        "Nautilisia" },
	{ "newday",            "A New Day" },
	{ "nidus",             "Nidus" },
	{ "nightcomputer",     "Night at the Computer Center" },
	{ "nightchristmas",    "An Abbreviated Night Before Christmas" },
	{ "nihilism",          "The Abyss" },
	{ "ninepoints",        "Nine Points" },
	{ "njag2",             "Not Just A Game" },
	{ "noroom",            "No Room" },
	{ "northnorth",        "The Northnorth Passage" },
	{ "nudistsgonewild",   "Nudists Gone Wild" },
	{ "adventurelobjan",   "Adventure (Lobjan translation)" },
	{ "oad",               "Only After Dark" },
	{ "odieus",            "Odieus's Quest for the Magic Flingshot (Beta)" },
	{ "omniquest",         "Omniquest" },
	{ "ogisoas",           "One Game in Search of a Story" },
	{ "onegirl",           "One Girl" },
	{ "onyourback",        "On Your Back" },
	{ "orevore",           "Orevore Courier" },
	{ "bloodless",         "Bloodless on the Orient Express" },
	{ "orion",             "The Orion Agenda" },
	{ "ottumwa",           "PDFA Ottumwa" },
	{ "outofthepit",       "Out of the Pit" },
	{ "paddlingmania",     "Total Paddling Mania" },
	{ "paint",             "Paint and Corners" },
	{ "palebluelight",     "Pale Blue Light" },
	{ "paperchase",        "Paper Chase" },
	{ "parallel",          "Parallel" },
	{ "paranoia",          "Paranoia" },
	{ "parc",              "Parc" },
	{ "pathway",           "Pathway to Destruction" },
	{ "peacock",           "Not Made With Hands" },
	{ "pegasus",           "Pegasus" },
	{ "pentari",           "Pentari" },
	{ "perilousmagic",     "Perilous Magic" },
	{ "perrysworld",       "Perry's World" },
	{ "phlegm",            "Phlegm" },
	{ "photograph",        "Photograph, a Portrait of Reflection" },
	{ "photopia",          "Photopia" },
	{ "piece",             "Piece of Mind" },
	{ "pigpancake",        "Pigpancake" },
	{ "piracy2",           "Piracy 2.0" },
	{ "piraterailroad",    "Pirate Railroad" },
	{ "praser5",           "Praser 5" },
	{ "primer",            "Primer" },
	{ "primrose",          "The Primrose Path" },
	{ "iceprincess",       "The Ice Princess" },
	{ "probing",           "Offensive Probing" },
	{ "promoted",          "Promoted!" },
	{ "puerto",            "The board game Puerto Rico" },
	{ "punkpoints",        "Punk Points" },
	{ "puppetman",         "The Puppet-Man (Larsoft Adventure number 5)" },
	{ "putpbaa",           "Pick Up the Phone Booth and Aisle" },
	{ "pytho",             "Pytho's Mask" },
	{ "muffinquest",       "The Quest for the Magic Muffin" },
	{ "muffinquest3",      "Quest for the Magic Bagel...Err Muffin 3" },
	{ "simpletheft2",      "A Simple Theft 2: A Simple Theftier" },
	{ "quidditch1954",     "The Quidditch Final of 1954" },
	{ "ralph",             "Ralph, An Interactive Sniffing" },
	{ "rameses",           "Rameses, a Tale of Heroes" },
	{ "rans",              "RANS, an interworld progress" },
	{ "ranshints",         "RANS Hints" },
	{ "rachaelbadday",     "Rachel has a bad day" },
	{ "readinginmay",      "A Reading in May" },
	{ "relief",            "The Hunt For Relief" },
	{ "reorsbushcave",     "Reor's Bush-Cave (The Sprout Pouch pt 4)" },
	{ "reser",             "Rock'Em Sock'Em Robots" },
	{ "resident",          "The Resident" },
	{ "retrofatale",       "Retro Fatale" },
	{ "reverb",            "Reverberations" },
	{ "reverzi",           "Reverzi" },
	{ "ribbons",           "Ribbons" },
	{ "risenecropolis",    "Rise of the Necropolis" },
	{ "risorg",            "Risorgimento Represso" },
	{ "robots",            "Robots - Another Abuse of the Z-Machine" },
	{ "rogue",             "zRogue" },
	{ "roomserial",        "Room Serial, an escape game" },
	{ "rota",              "The Reliques of Tolti-Aph" },
	{ "rpn",               "RPN" },
	{ "rtdoom",            "Return to Doom" },
	{ "samegame",          "SameGame, another episode in the Z-Machine abuse saga" },
	{ "samhain",           "Samhain: Pick Up The Pumpkin and KILL" },
	{ "sangraal",          "Sangraal" },
	{ "santassleighride",  "Santa's Sleigh Ride" },
	{ "saveprinceton",     "Save Princeton" },
	{ "savoirfaire",       "Savoir-Faire" },
	{ "scald",             "Scald" },
	{ "informschool",      "Inform School" },
	{ "schooldays",        "Schooldays" },
	{ "scopa",             "Scopa, a graphical (Z-code V6) card game" },
	{ "finalselection",    "Final Selection" },
	{ "sfiction",          "Speculative Fiction: Beginner's Lessons" },
	{ "shadowgate",        "Shadowgate" },
	{ "sherlock1",         "Sherlock gamebook #1: Murder at the Diogenes Club" },
	{ "sherlock2",         "Sherlock gamebook #2: The Black River Emerald" },
	{ "sherlock3",         "Sherlock gamebook #3: Death at Appledore Towers" },
	{ "sherlock4",         "Sherlock gamebook #4: The Crown vs Dr. Watson" },
	{ "shade",             "Shade" },
	{ "shadowofmemories",  "Shadow of Memories" },
	{ "shadowsoldiers",    "Shadow Soldiers" },
	{ "shallow",           "Shallow" },
	{ "sherbet",           "The Meteor, the Stone and a Long Glass of Sherbet" },
	{ "shrapnel",          "Shrapnel" },
	{ "sixgrayrats",       "Six Gray Rats Crawl Up The Pillow" },
	{ "slackerx",          "Slacker X" },
	{ "sleepcycle",        "Sleep Cycle" },
	{ "snafufun",          "SNAFUFUN" },
	{ "snowquest",         "Snowquest" },
	{ "neverplayed",       "So, You've Never Played a Text Adventure Before, Huh?" },
	{ "sofar",             "So Far, An Interactive Catharsis" },
	{ "softporn",          "Softporn Adventure" },
	{ "solitary",          "Solitary" },
	{ "somewhere",         "Somewhere" },
	{ "soreality",         "So Reality" },
	{ "spacestation",      "Space Station: based on Planetfall example transcript" },
	{ "spaceinvaderz",     "Space InvaderZ" },
	{ "spadventure",       "SpAdventure" },
	{ "spiritwrak",        "SpiritWrak" },
	{ "sporkery1",         "Sporkery 1: There Will Be Sporking" },
	{ "spot",              "The Spot" },
	{ "spring",            "She's Got a Thing For a Spring" },
	{ "spycatcher",        "Spycatcher" },
	{ "starborn",          "Starborn" },
	{ "seeksorrow",        "Starry Seeksorrow" },
	{ "stealingthestolen", "Stealing the Stolen" },
	{ "stiffmst",          "Stiffy Makane: Mystery Science Theater 3000" },
	{ "stinkorswim",       "Stink or Swim" },
	{ "stonecave",         "The Stone Cave" },
	{ "stormcellar",       "Storm Cellar" },
	{ "strangeworld",      "Strange World" },
	{ "suicide",           "Suicide, a self-ordered death sentence" },
	{ "sunburn",           "Sunburn" },	
	{ "sunburst",          "Sunburst: A C64 science fiction adventure game" },
	{ "sundayafternoon",   "Sunday Afternoon" },
	{ "briantimmons",      "The Surprising Case of Brian Timmons" },
	{ "sushi",             "A Day For Fresh Sushi" },
	{ "sutwin",            "The Space Under the Window" },
	{ "suvehnux",          "Suveh Nux" },
	{ "swineback",         "Swineback Ridge" },
	{ "sycamoratree",      "Sycamora Tree" },
	{ "taipan",            "Taipan!" },
	{ "spiderandweb",      "Spider And Web" },
	{ "tatctae",           "Time: All things come to an end" },
	{ "tauntingdonut",     "Taunting Donut" },
	{ "tblw",              "The Blood lust Warrior" },
	{ "tcomremake",        "The Color of Magic remake" },
	{ "tcoty",             "The Citizen of the Year" },
	{ "teacherfeature",    "Teacher Feature" },
	{ "williamtell",       "William Tell" },
	{ "telling",           "Whom The Telling Changed" },
	{ "temple",            "The Temple" },
	{ "terrortabby",       "Attack of the Terror Tabby!" },
	{ "tesseract",         "Beyond The Tesseract" },
	{ "tgm",               "The Great Machine - a fragment" },
	{ "thatdamnelevator",  "That Damn Elevator" },
	{ "thatdamnremote",    "thatdamnremote" },
	{ "cenricfamilycurse", "The Cenric Family Curse" },
	{ "vergingpaths",      "The Garden of Verging Paths" },
	{ "minimalistgame",    "The Minimalist Game" },
	{ "talemorning",       "The Mundane Tale of the Morning After" },
	{ "paperbagprincess",  "The Paper Bag Princess" },
	{ "worldupsidedown",   "The World Turned Upside Down" },
	{ "advhoudini",        "The Adventures of Houdini" },
	{ "penury",            "The Algophilists' Penury" },
	{ "theatre",           "Theatre" },
	{ "dayishothitler",    "The Day I Shot..." },
	{ "doghouse",          "The Dog/House" },
	{ "emptyroom",         "The Empty Room" },
	{ "forgottengirls",    "The Forgotten Girls" },
	{ "henribeauchamp",    "The Gallery of Henri Beauchamp" },
	{ "garliccage",        "The Garlic Cage, Episode I" },	
	{ "grandquest",        "The Grand Quest" },
	{ "thegreat",          "The Great, A Voyage To The Inner Self" },
	{ "greatpancake",      "The Great Pancake Detectives - Case #27" },
	{ "horriblepyramid",   "The Horrible Pyramid" },
	{ "house",             "The House" },
	{ "houseoffear",       "The House of Fear" },
	{ "island",            "The Island" },
	{ "kazooist",          "The Kazooist" },
	{ "priceoffreedom",    "The Price of Freedom" },
	{ "prize",             "The Prize" },
	{ "nascarexperience",  "The Realistic Nascar eXperience" },
	{ "smallroom",         "Trapped in a Small Room" },
	{ "spotlight",         "The Spotlight" },
	{ "stargods",          "The Star Gods" },
	{ "terribleoldmanse",  "The Terrible Old Manse: 8bit fun in 7bit ASCII" },
	{ "townmusicians",     "The Town Musicians" },
	{ "warblersnest",      "The Warbler's Nest" },
	{ "thorn",             "The Thorn" },
	{ "threecows",         "Three Cows and Two Doors" },
	{ "threediopolis",     "Threediopolis" },
	{ "threemore",         "Three More Visitors" },
	{ "timefortea",        "Time For Tea: A Game of Tea, Cakes, and Deadly Secrets" },
	{ "tirehoax",          "My Magic Tire Hoax" },
	{ "tk1",               "Time Killer #1: Claustrophobia" },
	{ "tkatc",             "The King and the Crown, Special Edition" },
	{ "toask",             "Treasures of a Slaver's Kingdom" },
	{ "tok",               "ToK" },
	{ "downtowntokyo",     "Downtown Tokyo Present Day" },
	{ "tower",             "Tower, a surreal trip" },
	{ "toxinx",            "Toxin X" },
	{ "trapped",           "Trapped" },
	{ "troll",             "Zork: A Troll's Eye View" },
	{ "trw",               "Tull Road Warrior" },
	{ "trystoffate",       "Tryst of Fate" },
	{ "tubetrouble",       "Tube Trouble" },
	{ "tutorial",          "Tutorial" },
	{ "tutorialhotel",     "Hotel Tutorial" },
	{ "tuuli",             "Tuuli" },
	{ "typo",              "Typo!" },
	{ "uhohdemo",          "Uh-oh! (demo version)" },
	{ "underdoos",         "The Underoos that ate New York!" },
	{ "underthebed",       "Under the Bed" },
	{ "ungodlyhour",       "Ungodly hour" },
	{ "uninvited",         "Uninvited" },
	{ "unicornpool",       "The Unicorn Pool" },
	{ "vacation",          "Vacation Gone Awry" },
	{ "vagueness",         "What Happens In Vagueness" },
	{ "vampiresun",        "House of the Midnight Sun - A Vampire's Tale" },
	{ "notinvenice",       "Not in Venice" },
	{ "vespers",           "Vespers" },
	{ "varicella",         "Varicella" },
	{ "vigilance",         "Internal Vigilance" },
	{ "vindaloo",          "Vindaloo" },
	{ "violet",            "Violet" },
	{ "virtuality",        "Virtuality" },
	{ "lackofvision",      "Lack of Vision" },
	{ "visitor",           "The Visitor" },
	{ "vosr",              "Voices of Spoon River" },
	{ "wadewar3",          "The WadeWars Book III" },
	{ "walkamongus",       "Walk Among Us" },
	{ "walkinthepark",     "A Walk in the Park" },
	{ "wallpaper",         "Delightful Wallpaper" },
	{ "warp",              "Warp!" },
	{ "wizardscastle",     "Wizard's Castle" },
	{ "weareunfinished",   "We Are Unfinished" },
	{ "weapon",            "The Weapon" },
	{ "weather",           "A Change in the Weather" },
	{ "thewedding",        "The Wedding" },
	{ "weding",            "Wedding" },
	{ "weirdcity",         "Weird City Interloper" },
	{ "weirdness",         "Weirdness: Strange, Different, and Altogether Weird" },
	{ "wwwanderer",        "Werewolves and Wanderer" },
	{ "wernersquest1",     "Werner's Quest Parts 1" },
	{ "wernersquest2",     "Werner's Quest Parts 2" },
	{ "wernersquest3",     "Werner's Quest Parts 3" },
	{ "wernersquest4",     "Werner's Quest Parts 4" },
	{ "whispers",          "The Corn Identity, an \"IF Whispers\" collaborative project" },
	{ "whitehouses",       "White Houses" },
	{ "wildflowers",       "Wildflowers" },
	{ "winchester",        "Winchester's Nightmare" },		
	{ "windhall",          "The Windhall Chronicles, Volume One" },
	{ "winterwonderland",  "Winter Wonderland" },
	{ "wir1",              "When in Rome 1: Accounting for Taste" },
	{ "wir2",              "When in Rome 2: Far from Home" },
	{ "wireless",          "Get Magazine. Open Magazine. Read Article" },
	{ "insidewoman",       "Inside Woman" },
	{ "wossname",          "Spodgeville Murphy and The Jewelled Eye of Wossname" },
	{ "wrenlaw",           "Wrenlaw" },
	{ "wscholars",         "Weishaupt Scholars" },
	{ "wump2ka",           "Wumpus 2000, The Virulent Labyrinth Of Yob-Shuggoth" },
	{ "wumpus",            "Hunt the Wumpus" },
	{ "wurm",              "Wurm" },
	{ "xenophobia",        "Xenophobia" },
	{ "yagwad",            "YAGWAD: Yes, Another Game With A Dragon!" },
	{ "yakshaving",        "Yak Shaving for Kicks and Giggles!" },
	{ "yomomma",           "Raising the Flag on Mount Yo Momma" },
	{ "stewgoing",         "You've Got A Stew Going!" },
	{ "zlife",             "Z-Life" },
	{ "zassball",          "ZassBall, Another Abuse of the Z-Machine" },
	{ "zbefunge",          "ZBefunge 0.7 beta" },
	{ "zcamel",            "Camel" },
	{ "zcatalog",          "The Z-Files, a Z-Code Catalog" },
	{ "zchess",            "Z-Chess" },
	{ "zdungeon",          "ZDungeon" },
	{ "zedfunge",          "ZedFunge 0.7.3 beta" },
	{ "zedit",             "ZEdit, The World's Most Portable Text Editor" },
	{ "zegro",             "Zegrothenus" },
	{ "zenon",             "Escape from the Starship Zenon" },
	{ "zenspeak",          "Zen Speaks!" },
	{ "zokoban",           "Z-Machine Sokoban" },
	{ "zombies",           "Zombies, yet another abuse of the Z-Machine" },
	{ "zorklxix",          "Zork LXIX: The Great Underground Hot Dog" },
	{ "zorkianstories1",   "Zorkian Stories 1: G.U.E" },
	{ "zorknplus9",        "Zork N plus 9" },
	{ "zracer",            "ZRacer" },
	{ "zsnake",            "Z-Snake" },
	{ "ztornado",          "Z-Tornado" },
	{ "ztrek",             "Super Z Trek" },
	{ "zugzwang",          "Zugzwang: The Interactive Life of a Chess Piece" },

	// Painfull Little Stupid Games
	{ "plsg1",             "Dinnertime: Painless Little Stupid Games #1" },
	{ "plsg2",             "To Get To The Other Side: Painless Little Stupid Games #2" },
	{ "plsg3",             "They're After You!: Painless Little Stupid Games #3" },
	{ "plsg4",             "Mazemapper: Painless Little Stupid Games #4" },
	{ "plsg5",             "The Mean Story: Painless Little Stupid Games #5" },
	{ "plsg6",             "Mahadev: Painless Little Stupid Games #6" },
	{ "plsg7",             "Sturdlint: Painless Little Stupid Games #7" },
	{ "plsg8",             "The Last Dark Day: Painless Little Stupid Games #8" },
	{ "plsg9",             "Zork LXIX: Painless Little Stupid Games #9" },
	{ "plsg10",            "The Valley House: Painless Little Stupid Games #10" },

	// Converted Scott Adams games
	{ "adventurelandi5",     "Adventureland" },
	{ "pirateadventurei5",   "Pirate Adventure" },
	{ "missionimpossiblei5", "Mission Impossible" },
	{ "voodoocastlei5",      "Voodoo Castle" },
	{ "thecounti5",          "The Count" },
	{ "strangeodysseyi5",    "Strange Odyssey" },
	{ "mysteryfunhousei5",   "Mystery Fun House" },
	{ "pyramidofdoomi5",     "Pyramid Of Doom" },
	{ "ghosttowni5",         "Ghost Town" },
	{ "savageisland1i5",     "Savage Island, Part 1" },
	{ "savageisland2i5",     "Savage Island, Part 2" },
	{ "goldenvoyagei5",      "The Golden Voyage" },
	{ "adventure13i5",       "Adventure 13" },
	{ "adventure14i5",       "Adventure 14" },
	{ "buckaroobanzaii5",    "Buckaroo Banzai" },
	{ "marveladventurei5",   "Marvel Adventure #1" },
	{ "questprobe2i5",       "Questprobe 2: Spiderman" },
	{ "scottsampleri5",      "Adventure International's Mini-Adventure Sampler" },

	// Mysterious Adventures by Brian Howarth
	{ "goldenbatoni5", "Mysterious Adventures 1: The Golden Baton" },
	{ "timemachinei5", "Mysterious Adventures 2: The Time Machine" },
	{ "arrowofdeath1i5", "Mysterious Adventures 3: Arrow of Death Part 1" },
	{ "arrowofdeath2i5", "Mysterious Adventures 4: Arrow of Death Part 2" },
	{ "pulsar7i5", "Mysterious Adventures 5: Escape from Pulsar 7" },
	{ "circusi5", "Mysterious Adventures 6: Circus" },
	{ "feasibilityi5", "Mysterious Adventures 7: Feasibility Experiment" },
	{ "akyrzi5", "Mysterious Adventures 8: The Wizard of Akyrz" },
	{ "perseusi5", "Mysterious Adventures 9: Perseus and Andromeda" },
	{ "10indiansi5", "Mysterious Adventures 10: Ten Little Indians" },
	{ "waxworksi5", "Mysterious Adventures 11: Waxworks" },

	// 1992 album Apollo 18, by They Might be Giants
	{ "apollo1", "Apollo 18 01: Dig My Grave" },
	{ "apollo2", "Apollo 18 02: I Palindrome I" },
	{ "apollo3", "Apollo 18 03: She's Actual Size" },
	{ "apollo4", "Apollo 18 04: My Evil Twin" },
	{ "apollo5", "Apollo 18 05: Mammal" },
	{ "apollo6", "Apollo 18 06: The Statue Got Me High" },
	{ "apollo7", "Apollo 18 07: Spider" },
	{ "apollo8", "Apollo 18 08: The Guitar(The Lion Sleeps Tonight)" },
	{ "apollo9", "Apollo 18 09: Dinner Bell" },
	{ "apollo10", "Apollo 18 10: Narrow Your Eyes" },
	{ "apollo11", "Apollo 18 11: Hall of Heads" },
	{ "apollo12", "Apollo 18 12: Which Describes How You're Feeling" },
	{ "apollo13", "Apollo 18 13: See the Constellation" },
	{ "apollo14", "Apollo 18 14: If I Wasn't Shy" },
	{ "apollo15", "Apollo 18 15: Turn Around" },
	{ "apollo16", "Apollo 18 16: Hypnotist of Ladies" },
	{ "apollo17", "Apollo 18 17: Fingertips - Everything Is Catching on Fire" },
	{ "apollo18", "Apollo 18 18: Fingertips - Fingertips" },
	{ "apollo19", "Apollo 18 19: Fingertips - I Hear the Wind Blow" },
	{ "apollo20", "Apollo 18 20: Fingertips - Hey Now, Everybody" },
	{ "apollo21", "Apollo 18 21: Fingertips - Who's That Standing Out the Window" },
	{ "apollo22", "Apollo 18 22: Fingertips - I Found a New Friend" },
	{ "apollo23", "Apollo 18 23: Fingertips - Come On and Wreck My Car" },
	{ "apollo24", "Apollo 18 24: Fingertips - Aren't You the Guy Who Hit Me in the Eye" },
	{ "apollo25", "Apollo 18 25: Fingertips - Please Pass the Milk Please" },
	{ "apollo26", "Apollo 18 26: Fingertips - Leave Me Alone" },
	{ "apollo27", "Apollo 18 27: Fingertips - Who's Knockin' on the Wall" },
	{ "apollo28", "Apollo 18 28: Fingertips - All Alone" },
	{ "apollo29", "Apollo 18 29: Fingertips - What's That Blue Thing Doing Here" },
	{ "apollo30", "Apollo 18 30: Fingertips - Something Grabbed Ahold of My Hand" },
	{ "apollo31", "Apollo 18 31: Fingertips - I Don't Understand You" },
	{ "apollo32", "Apollo 18 32: Fingertips - I Heard a Sound" },
	{ "apollo33", "Apollo 18 33: Fingertips - Mysterious Whisper" },
	{ "apollo34", "Apollo 18 34: Fingertips - The Day That Love Came to Play" },
	{ "apollo35", "Apollo 18 35: Fingertips - I'm Having a Heart Attack" },
	{ "apollo36", "Apollo 18 36: Fingertips - Fingertips(Reprise)" },
	{ "apollo37", "Apollo 18 37: Fingertips - I Walk Along Darkened Corridors" },
	{ "apollo38", "Apollo 18 38: Space Suit" },

	// Danish games
	{ "nissen", "Pa loftet sidder nissen" },

	// French games
	{ "basketball", "Champion de basket-ball" },
	{ "dreamlands", "Echappee Belle Dans Les Contrees du Reve" },
	{ "espions", "Les espions ne meurent jamais" },
	{ "filaments", "Filaments" },
	{ "initiation", "Initiation" },
	{ "kheper", "Kheper" },
	{ "verdeterre", "Le butin du Capitaine Verdeterre" },
	{ "lieuxcommuns", "Lieux communs" },
	{ "lmpsd", "La Mort Pour Seul Destin" },
	{ "ombre", "Ombre" },
	{ "princesse", "Ma princesse adoree" },
	{ "sarvegne", "Sarvegne" },
	{ "katana", "Le Scarabee et le Katana" },
	{ "sdlc", "Sortir de la chambre" },
	{ "balcon", "Sorciere au balcon" },
	{ "templedefeu", "Le Temple de Feu" },

	// German games
	{ "bearg", "Ein Bar Geht Aus" },
	{ "bewerbung", "Die Bewerbung" },
	{ "deklinator", "Object declination tool" },
	{ "edendemo", "Der Abentheurliche Informissimus Teutsch demo" },
	{ "halb2", "Halb Zwei" },
	{ "herr", "Die Geschichte des Herrn P. von Hannes Schuller" },
	{ "jazteg", "Jazz auf Tegemis" },
	{ "karisma", "Klub Karisma" },
	{ "knack", "Knack!" },
	{ "o", "O" },
	{ "starrider", "Star Rider" },
	{ "mchatton", "Tutorial Eine Einfuhrung in Textadventures von Cooper McHatton" },
	{ "wasserhasser", "Wasser-Hasser" },
	{ "wichtel", "Wichtel" },

	// Italian games
	{ "armando", "L'Armando" },
	{ "ayon", "Nel Mondo di Ayon" },
	{ "darkiss", "Darkiss! Il bacio del vampiro" },
	{ "darkiss2", "Darkiss! Il bacio del vampiro. Capitolo 2" },
	{ "filamit", "Filaments" },
	{ "flamel", "Flamel" },
	{ "giardino", "Il giardino incantato" },
	{ "kangourou", "Kangourou dell'informatica 2013" },
	{ "koohinoor", "Kooh-I-Noor" },
	{ "luna", "La Pietra della Luna" },
	{ "poesia", "Manca solo un verso a quella poesia" },
	{ "oldwest1", "Pecos Town, Old West Episode I" },
	{ "rovo", "Il rovo" },
	{ "scarafaggio", "Lo Scarafaggio" },
	{ "sognodisangue", "Sogno di Sangue" },
	{ "strega", "La strega di Maughn" },
	{ "tesla", "In Cerca Di Tesla" },
	{ "villamorgana", "Villa Morgana" },
	{ "zazie", "Zazie, una lettura interattiva" },
	{ "zenfactorspa", "ZenFactor Spa" },
	{ "zombie", "Uno Zombie a Deadville" },
	{ "Zorkita", "Zork I: Il Grande Impero Sotterraneo" },

	// Slovenian games
	{ "zenin", "Zenin na begu" },

	// Spanish games
	{ "abalanzate", "Abalanzate" },
	{ "absolutos", "Los sonidos absolutos" },
	{ "afuera", "Afuera" },
	{ "amanda", "Amanda" },
	{ "aque1", "Aquelarre" },
	{ "casi", "Casi Muerto" },
	{ "Celos", "Un Asunto de Celos" },
	{ "Cerillera", "La Pequena Cerillera" },
	{ "churro", "Churro patatero" },
	{ "csa", "Cacahuetes" },
	{ "cueva", "La Oscura Cueva" },
	{ "ddddddcrj", "Cirith Ungol" },
	{ "despert", "El Despertar" },
	{ "draculasp2", "Dracula: Episodio 2, La Llegada" },
	{ "draculasp", "Dracula: Episodio 1, La Primera Noche" },
	{ "ascenso", "El ascenso de Kunelar" },
	{ "regalo", "El regalo de Gorbag" },
	{ "elultimohogar", "Misterio en el Ultimo Hogar" },
	{ "ergotdelima", "Lime Ergot" },
	{ "forrajeo", "Forrajeo" },
	{ "fotopia", "Fotopia" },
	{ "gorbag", "El regalo de Gorbag" },
	{ "gorron", "El gorron del tren" },
	{ "goteras", "Goteras" },
	{ "hhorcus", "Homo Homini Orcus" },
	{ "islas", "El archipielago" },
	{ "kerulen", "Ke rulen los petas, por Grendelkhan" },
	{ "konix", "Konix" },
	{ "kunelar", "El ascenso de Kunelar" },
	{ "lamansion", "La Mansion" },
	{ "lldc", "La Llamada de Cthulhu" },
	{ "macetas", "Macetas" },
	{ "meeva", "La mediana evasion" },
	{ "megara", "Los placeres de Megara" },
	{ "modusvivendi","An ancient Roman tale" },
	{ "mpdroidone", "Operacion MPDroid1" },
	{ "navidad", "Una pequena historia de Navidad" },
	{ "necedad", "Por la Necedad Humana" },
	{ "culpatuya", "No es culpa tuya Maria" },
	{ "ocaso", "Ocaso Mortal I: The Bug" },
	{ "oculta", "La cara oculta de la luna" },
	{ "olvido", "Olvido Mortal" },
	{ "osito", "La Intrepida Noche del Osito" },
	{ "panajo", "Pan de Ajo, by Incanus" },
	{ "pesadillavoraz", "Pesadilla voraz" },
	{ "pincoya", "Ofrenda a La Pincoya" },
	{ "playera", "En la playa" },
	{ "protector", "El Protector" },
	{ "reflejos", "Reflejos blancos" },
	{ "senten", "La Sentencia" },
	{ "sombras", "Sombras de Moria" },
	{ "tokland", "La isla de Tokland" },
	{ "torre", "Misterio en la torre" },

	// Swedish games
	{ "aventyr", "Aventyr" },
	{ "drakmagi", "Drakmagi" },
	{ "hotellet", "Hotel Noir" },
	{ "pangnyheten", "Pangnyheten" },
	{ "storforsen", "Storforsen" },
	{ "stuga", "Stuga" },
	{ "vanyar", "Vanyar" },

	{ nullptr, nullptr }
};

#define NONE GUIO4(GUIO_NOSPEECH, GUIO_NOSFX, GUIO_NOMUSIC, GUIO_NOSUBTITLES)
#define ENTRY0(ID, VERSION, MD5, FILESIZE) { ID, VERSION, MD5, FILESIZE, Common::EN_ANY, NONE }
#define ENTRY1(ID, VERSION, MD5, FILESIZE, LANG) { ID, VERSION, MD5, FILESIZE, LANG, NONE }
#define ENTRYS(ID, VERSION, MD5, FILESIZE) { ID, VERSION, MD5, FILESIZE, Common::EN_ANY, GUIO3(GUIO_NOSPEECH, GUIO_NOMUSIC, GUIO_NOSUBTITLES) }

#define FROTZ_TABLE_END_MARKER { nullptr, nullptr, nullptr, 0, Common::EN_ANY, "" }

const FrotzGameDescription FROTZ_GAMES[] = {
	// Infocom Games - English
	ENTRY0("amfv", "R77-850814", "b7ffaed0ca4a90450f92b34066133377", 262016),
	ENTRY0("amfv", "R79-851122", "1e37dbcf7ccc9244dbfc3229796362f4", 262544),
	ENTRY0("arthur", "R54-890606", "ced2c66d03a49de0e8190b468332f081", 271360),
	ENTRY0("arthur", "R74-890714", "13d13f375f85a874c82a8ac7ad69dc41", 269200),
	ENTRY0("ballyhoo", "R97-851218", "7944e832a7d7b34037c7b6791de43dbd", 128556),
	ENTRY0("beyondzork", "R49-870917", "a5547795def620d0a75a064f9a37ab2d", 261900),
	ENTRY0("beyondzork", "R51-870923", "73948f415596fa4d9afe442b2c19e61f", 261548),
	ENTRY0("beyondzork", "R57-871221", "c56cac07a500e5864a994b19286bc07c", 261388),
	ENTRY0("borderzone", "R9-871008", "189231ed0675f6be3be86856f49211af", 178372),
	ENTRY0("bureaucracy", "R86-870212", "2bb00311d4c201082cfcd278ae5db921", 243144),
	ENTRY0("bureaucracy", "R116-870602", "a8ae194257a989ed3d82648a507466f2", 243340),
	ENTRY0("cutthroats", "R23-840809", "059801d9f90fffeb3645816c37c7eda2", 112558),
	ENTRY0("deadline", "R22-820809", "1610e84ca2505885566e648c1c525976", 111782),
	ENTRY0("deadline", "R26-821108", "e1ae6af1098067b86076c34865ae713c", 108372),
	ENTRY0("deadline", "R27-831006", "166ffb7cabc6b85f210655f371c89c46", 108454),
	ENTRY0("enchanter", "R10-830810", "7b41d915b4c2e31423d99925e9438aa4", 109126),
	ENTRY0("enchanter", "R15-831107", "e70f21aad650dd196fa3601cab5e0fc5", 109230),
	ENTRY0("enchanter", "R16-831118", "46187e0691f6f5ecdd5a336885db6aad", 109234),
	ENTRY0("enchanter", "R29-860820", "f87cdafad3682ead25cfc473656ff713", 111126),
	ENTRY0("hhgttg", "R47-840914", "fdda8f4239819402c62db866bb61a648", 112622),
	ENTRY0("hhgttg", "R56-841221", "a214fcb42bc9f554d07d983a12f6a062", 113444),
	ENTRY0("hhgttg", "R58-851002", "e867d49ad1fb9406ff4e0678a4ee2ac9", 113332),
	ENTRY0("hhgttg", "R59-851108", "34f6abc1f2a42be127ef434fc475f0ee", 113334),
	ENTRY0("hhgttg", "R31-871119", "379022bcd4ec74b90274c6100c33f579", 158412),
	ENTRY0("hollywoodhijinx", "R37-861215", "7b52824057ae24e098c228c41460ef75", 109650),
	ENTRY0("infidel", "R22-830916", "38f713e53af720624434529ea780040c", 93556),
	ENTRY0("journey", "R30-890322", "c9893bc0399080bd3850d4db2120d110", 280472),
	ENTRY0("journey", "R77-890616", "8a4ab56f62e1b7c918b837794182dbcd", 282176),
	ENTRY0("journey", "R83-890706", "c33ea33ab8aec6c617734dcfe1211067", 282312),
	ENTRY0("lgop", "R0", "69b3534570851b90d7f53ebe9d224a6a", 128998),
	ENTRY0("lgop", "R4-880405", "6bdae7434df7c03f3589ece0bed3317d", 159928),
	ENTRY0("lgop", "R59-860730", "e81237e220a612c5a93fbcc1fdf85a0a", 129022),
	ENTRYS("lurkinghorror", "R203", "e2d2505510479fec0405727e3d0abc10", 128986),
	ENTRYS("lurkinghorror", "R219", "83936d75c2cfd71fb64bf63c4696b9ac", 129704),
	ENTRYS("lurkinghorror", "R221", "c60cd0bf3c6eda867241378c7cb5464a", 129944),
	ENTRY0("minizork1", "R34-871124", "0d7700679e5e63dec97f712698610a46", 52216),
	ENTRY0("moonmist", "R4-860918", "284797c3025ffaf76aecfa5c2bbffa86", 129002),
	ENTRY0("moonmist", "R9-861022", "698475de2769c66bc5a1eca600c71561", 128866),
	ENTRY0("nordbert", "R19-870722", "da1e189e19e3b24b2e35bd41fc32d261", 170284),
	ENTRY0("planetfall", "R20-830708", "15815c461a8548b7630d2aee46d07cc7", 107958),
	ENTRY0("planetfall", "R26-831014", "cf6ce61eb2eff9d4f18d7bcba7c12cfb", 108674),
	ENTRY0("planetfall", "R29-840118", "9facd8b974e658520fb762af4c4789dc", 109052),
	ENTRY0("planetfall", "R37-851003", "01844816673414c97d21dc003304989b",109398),
	ENTRY0("planetfall", "R10-880531", "34c69f1d24418fd4d2de195a1d7546c4", 136560),
	ENTRY0("plunderedhearts", "R26-870730", "fe5b9eb91949d41838166364f1753b10", 128962),
	ENTRY0("infocomsampler1", "R26-840731", "5483febc51abd55fb5e04c4c97a0b260", 112610),
	ENTRY0("infocomsampler1", "R53-850407", "47b8b8394e25faec870a798145529688", 126708),
	ENTRY0("infocomsampler1", "R55-850823", "05d9d1a1c3c73fce9e24ab695ece16c8", 126902),
	ENTRY0("infocomsampler2", "R97-870601", "201fa230a942df5aa75bb5b5f609e8ce", 125314),
	ENTRY0("seastalker", "R15-840501", "2f0220b0390deda695e01832a92b5493", 117738),
	ENTRY0("seastalker", "R15-840522", "050961fa7788c309bbf40accbff2ffdf", 117728),
	ENTRY0("seastalker", "R16-850515", "eb39dff7beb3589c8581dd2e3569eb78", 117752),
	ENTRY0("seastalker", "R16-850603", "bccf194b1e823e37db2431b586662773", 117762),
	ENTRY0("seastalker", "R86-840320", "64fb27e7b9fd682ff4f0d0ec6616a468", 116456),
	ENTRYS("sherlockriddle", "R21-871214", "69862f7f07a4e977159ea4da7f2f2ba6", 188444),
	ENTRYS("sherlockriddle", "R26-880127", "2cb2bda2e34eb7f9494cb585720e74cd", 190180),
	ENTRY0("shogun", "R322-890706", "62cca41feb94082442026f44f3e48e19", 344816),
	ENTRY0("sorcerer", "R4-840131", "d4a914fdfe90f5cd055a03b9aa24addd", 109734),
	ENTRY0("sorcerer", "R6-840508", "7ee357c10a9e049fe7c641a4817ee575", 109482),
	ENTRY0("sorcerer", "R13-851021", "7a076459806eaee72015b2b2882a89dc", 108692),
	ENTRY0("sorcerer", "R15-851108", "cde88a011d2ba183ff69b47b0d8881c6", 108682),
	ENTRY0("spellbreaker", "R63-850916", "b7b9eef231dee03fb40a9d98416fa0d5", 128480),
	ENTRY0("spellbreaker", "R87-860904", "852286847f4cdd790075fa824260ff4e", 128916),
	ENTRY0("starcross", "R15-820901", "fb2e6d9a0ad5822f3a8d4aec949e4e3c", 84984),
	ENTRY0("starcross", "R17-821021", "ed1e62e1f0eb9d819be45c076c5729f7", 83792),
	ENTRY0("stationfall", "R107-870430", "cfadfb66afabaa2971ec9b4ae65266ca", 128934),
	ENTRY0("suspect", "R14-841005", "3d759ccb19233f51968fa79d7374b393", 118692),
	ENTRY0("suspended", "v5-830222", "d898430e3cccdee9f9acfffcc9ef709c", 105418),
	ENTRY0("suspended", "R7-830419", "65f0cc760a2500d110242fbf942f1028", 105500),
	ENTRY0("suspended", "R8-830521", "b749d42462dfec21831b69635cd9c5e8", 105492),
	ENTRY0("suspended", "R8_2-840521", "6088ad7cb553626b52875a9b8e801312", 105584),
	ENTRY0("trinity", "R11-860509", "994ea591f8d401e11661c912b92ee05e", 262016),
	ENTRY0("trinity", "R12-860926", "5377dc1ee39f1c8ed572944f89946eb2", 262064),
	ENTRY0("wishbringer", "R23-880706", "bec823084c5622e88eca5a886278d2a5", 164712),
	ENTRY0("wishbringer", "R68-850501", "898b9b157ce8e54a0953366d6317fbd5", 128952),
	ENTRY0("wishbringer", "R69-850920", "e7c0412c4b3bda39de438a02cbae3816", 128904),
	ENTRY0("thewitness", "R13-830524", "d2297ddfe2c1b976c1b0c381ab01e2b3", 102608),
	ENTRY0("thewitness", "R18-830910", "a6e441b0b92a72537c830fed201267af", 103728),
	ENTRY0("thewitness", "R22-840924", "1019b9b1e1aa2c6eda945d7d92c2073a", 104664),
	ENTRY0("zork0", "R296-881019", "fca554c21542729c9d154c319af6307e", 295536),
	ENTRY0("zork0", "R366-demo-890323", "b58c35dc2ba36d48fade99564922c7c3", 296376),
	ENTRY0("zork0", "R366-890323", "e787b2cad2d6f29fd812e737f75646e8", 296376),
	ENTRY0("zork0", "R383-890602", "32e3e7ec438dabe77df2351af6ece324", 299392),
	ENTRY0("zork0", "R393-890714", "29fb0e090bbff7bc8e9661e55da69ae7", 299968),
	ENTRY0("zork1", "R15-UG3AU5", "fa2d22304700898cb8de921f46ca4bc9", 78566),
	ENTRY0("zork1", "R20", "b222bed4a0ab2650135cba7f4b1b1c67", 75734),
	ENTRY0("zork1", "R23-820428", "6ad3d9ab2874beefcbc53630e9261118", 75780),
	ENTRY0("zork1", "R25-820515", "287a1ce17f458fb2e776499a13796719", 75808),
	ENTRY0("zork1", "R26-820803", "285f1d7c5deb1a2f23825f63823d0777", 75964),
	ENTRY0("zork1", "R28-821013", "83bb70d73f3b4b5c4a32d8588b2d0707", 76018),
	ENTRY0("zork1", "R30-830330", "d6d8b3ae49a683a6fce2383a8bab36a5", 76324),
	ENTRY0("zork1", "R5", "dd5ba502b30189d03abfcfb9817dffe0", 82836),
	ENTRY0("zork1", "R52-871125", "e56267fd041c71fc229f7deb6b6537c2", 105264),
	ENTRY0("zork1", "R75-830929", "b35bca8dd18f6312c7e54dcd7958d7e5", 84868),
	ENTRY0("zork1", "R76-840509", "50ebf3c0c959ac2571c23cb7f7907c70", 84874),
	ENTRY0("zork1", "R88-840726", "d708b6751126f3b2b7612c760f080d41", 84876),
	ENTRY0("zork2", "R15-820308", "4b6ecc8e40243ddbd4cc19ef82304c3b", 82110),
	ENTRY0("zork2", "R17-820427", "386f2cd937e0ca316695d6ddca521c78", 82368),
	ENTRY0("zork2", "R18-820512", "a019dd721134b57f5926ee2adf634b55", 82422),
	ENTRY0("zork2", "R18_2-820517", "6cafa0e5239a74aa120bb8e2c33441be", 82422),
	ENTRY0("zork2", "R19-820721", "a5236020509af26b47c737e51ce884aa", 82586),
	ENTRY0("zork2", "R22-830331", "600264d62720731283454592261ec3fe", 82920),
	ENTRY0("zork2", "R23-830411", "6c2e766b553c127bb07f7a0f8fe03ae2", 81876),
	ENTRY0("zork2", "R48-840904", "a5064c9c3ce0bc02f16e01d745f39b67", 89912),
	ENTRY0("zork2", "R7-UG3AU5", "8243ce12e7b3ce24b150f34cc2cb472c", 85260),
	ENTRY0("zork3", "R10-820818", "ba4897f4d82ba08906295dd3aebf501a", 82334),
	ENTRY0("zork3", "R15-830331", "2fb29e6f5eebb643f42714ca9086e145", 82558),
	ENTRY0("zork3", "R15_2-840518", "672b54d8f457bd3be32e41fc9e069d71", 82642),
	ENTRY0("zork3", "R16-830410", "4717f8ec2f08da7d438c05f1351d28bd", 81626),
	ENTRY0("zork3", "R17-840727", "c5dc520f469771c59d193558d405341d", 82714),
	ENTRY0("ztuu", "ztuu-970828", "3a55991be19943a13da58a91cd3615eb", 102524),

	// Infocom Games - Foreign
	ENTRY1("zork1", "R3-880113", "9f336c92c1fd392fc7e81288e5c9b6ab", 116216, Common::DE_DEU),

	// English games
	ENTRY0("404life", "110524", "a4ee7ba2cb611e0ae3e413a6eb9dc4da", 506770),
	ENTRY0("69105keys", "090302", "bc343936e0e9d79a3736b200eadfc6ee", 195162),
	ENTRY0("905", "120724", "885acc1cf4ae18428d8a1998d57f2925", 87040),
	ENTRY0("9dancers", "040718", "8a7faee46b0d6f35cb0cbeb7ac8631b2", 99328),
	ENTRY0("cockandbull", "170401", "6036a188e9def979678697ce7be14171", 895024),
	ENTRY0("aasmasters", "030410", "bb6645f33130c7881da72ff0bfb6fb5b", 68608),
	ENTRY0("accuse", "070321", "3a1098524f8d6964a5d7a2579e9367f1", 129528),
	ENTRY0("acheton", "111115", "5abc5f815b7f4f6031781a3e6126ef03", 250880),
	ENTRY0("acorncourt", "970904", "119dc6466da205261efc1ef8e00d26d1", 55296),
	ENTRY0("acrobat", "090111", "adaf3516d405d7dc65cbff1986f41af2", 2068718),
	ENTRY0("acrossstars", "100329", "ef910f930a01ac4d24051b8fce549180", 524288),
	ENTRY0("acrossstarsclues", "100213", "d23e9bf6019f212ff324f4595f8b6177", 122880),
	ENTRY0("addendum", "080611", "e6ec3704b04d638f53aaf5f082efb424", 199420),
	ENTRY0("adv440", "160307", "af1ab807e8c438d0e548c6b4903c33a1", 193536),
	ENTRY0("adv550", "160307", "631ab32f7370e1c1e5b4e9fe6ad539fb", 231424),
	ENTRY0("adv551", "171110", "24449bf1875c0b75491173ba4243bc99", 334848),
	ENTRY0("adv350", "151001", "88fe704848a77346d196a46f7cfb3b38", 66414),
	ENTRY0("adv350", "060321", "d00c3717a46734c2ae96fb8d2ad0a226", 138240),
	ENTRY0("adv350", "011123", "5d4867b23e904d22453f031c87d6dcf0", 118272),
	ENTRY0("adv350", "160307", "5f800280865d57cbfeb66695e79dc9b9", 147456),
	ENTRY0("adv", "150118", "c776a1a9a8122967160fb51a39485113", 431616),
	ENTRY0("adventureland", "961111", "21f1bd1815a8b3ba5730fe168ff88e59", 49152),
	ENTRY0("adventuretime", "110111", "b6e0e55780b699e37411f0eb176f7496", 359368),
	ENTRY0("adverbum", "060905", "e5c04bcadb953ad0a280b541f7510338", 138240),
	ENTRY0("affront", "040226", "1b10a2dcedebdea772cea1c7b0407331", 50176),
	ENTRY0("aisle", "990528", "a7af83193b4139f65c020ac49ff30447", 122368),
	ENTRY0("alice", "030501", "ce25288e736c1c420a990b609943a990", 86528),
	ENTRY0("allroads", "011119", "84ce94d8c6c77452537511f14739b612", 172032),
	ENTRY0("alongtheriver", "100630", "1b670d231c4b0cc10b51ba60196c6a68", 198144),
	ENTRY0("alpha", "981017", "f8fbbc64455efc259f9b41517d82b4ab", 51200),
	ENTRY0("ambassadorsdaughter", "150525", "d415de88a7a756be04f5203f8b4668ee", 255488),
	ENTRY0("amish", "020603", "d4194b4de41c9663f464bb33f1917a7d", 50688),
	ENTRY0("amiss", "020409", "fa22d754f2b9daedc232bfbc6b28b6d1", 143872),
	ENTRY0("anchor", "990206", "2bc49d98d980bfefa064ae76f4a217e5", 520192),
	ENTRY0("hipponewyear", "130210", "b226fcbf871613684d8d2c7941e02e82", 317440),
	ENTRY0("animals", "150314", "d5e0c7c6641709eeb5c4c0797519962c", 32768),
	ENTRY0("annoy", "990127", "0383c40a28c606ddf8bfba3a66469055", 71168),
	ENTRY0("aotyrz", "060825", "d9264cedc9cb438eedf67f611dbdbb3e", 159232),
	ENTRY0("aphasiaquest", "150426", "ee480b0a0983e636649777a745a8f6f0", 233896),
	ENTRY0("appall", "020422", "3c33d5639fa341829dde7c4b9b1875fd", 52736),
	ENTRY0("aridandpale", "090501", "d18d5d2b8ceef04234f64373958097e5", 199400),
	ENTRY0("asylum", "090721", "1fcf07b963cd664cf11660882de99e47", 135680),
	ENTRY0("atrocitron", "161215", "6798fc939e1c6a92eb8e356e359a0ee4", 91648),
	ENTRY0("ats", "000001", "accd04838197c1e4937bac253349b623", 387072),
	ENTRY0("atwork", "000705", "44e22cd9dd5124dd149b761dfd6e93e8", 78336),
	ENTRY0("aug4", "130702", "0b9a2b7c37f3d56fcd8d0b531582f8df", 364544),
	ENTRY0("avon", "111115", "82a3d640af8cb7707a1b4301192679b1", 104960),
	ENTRY0("awakening", "980726", "66181a19b3316f6644c20a9ee18be40f", 99328),
	ENTRY0("awitl", "980215", "8cf72be55d75137cc435ab25f479c123", 56320),
	ENTRY0("ayac", "010107", "b503a9f2817ce97480160e454e470c58", 68608),
	ENTRY0("b2demo", "961017", "28d166262aa22fb55fdb9685a5a124ef", 65024),
	ENTRY0("b7snare", "150118", "5e1057596f6fe7d0efb100f80ae65f27", 284672),
	ENTRY0("babytree", "120512", "79e230657b07b2e8fa45ffbe4635d03f", 294352),
	ENTRY0("backtowakeUp", "080313", "73ada6a69651bea315206d7994164b95", 640246),
	ENTRY0("martyquest", "120430", "80b274b7feb7c19ee0aeba85dac0d688", 268272),
	ENTRY0("backup", "091204", "52d1d40ff557c21b7d4464e12b6d646c", 429188),
	ENTRY0("balances", "961216", "2b740e4bf08e64580085640b40a0ead8", 75264),
	ENTRY0("baldersdeath", "111107", "cec997b2ba4917a2dab2640f276f1c7c", 254404),
	ENTRY0("ballerina102", "991128", "9f92c9d90a536f62d7a535e7026c28bc", 508928),
	ENTRY0("balt24", "970706", "7c45c1ea2780a5eed643f35fb44dccda", 58368),
	ENTRY0("bathtub", "060409", "580a828689b64b150d29c4247c873acc", 85504),
	ENTRY0("bazic", "010710", "a6c277d223557127c7d0678aa128d501", 55296),
	ENTRY0("bear", "990224", "8e96c4de71e6fd4bdafb10bdea76893b", 109568),
	ENTRY0("bedlam", "970711", "6c816efb1a7fc99345fdb85bb9e01ac6", 59392),
	ENTRY0("bedtime", "120324", "2a39f06bdad42384b026059d690cd779", 260120),
	ENTRY0("andrewplotkin", "080423", "9679960dde743ff1f506c12db29c5214", 366740),
	ENTRY0("beingsteve", "060519", "a446f37fcd5945428797573295acbe03", 124810),
	ENTRY0("beneath", "080107", "8e53f4d65b0b68ef92fb2fbd3592dcc5", 176128),
	ENTRY0("beyond", "100115", "61bd0b7eae9e81c7d01496d623a497e1", 336982),
	ENTRY0("bibleretold", "102283", "158f9c1818f7a80607a3b8281a9bc140", 169472),
	ENTRY0("bicon", "101006", "2f2d993faf161318c4b4c5ee0957e288", 366836),
	ENTRY0("biscuit", "010310", "e4d8262dbeeab8e340e522e5cef740a8", 137728),
	ENTRY0("bishoes", "090513", "05c6dae3ea1896e756ce54bae26564ae", 85504),
	ENTRY0("bj", "981103", "2787b18265a5bceccd0a4d311f94d97a", 158720),
	ENTRY0("blair", "971119", "39a811c3280594ef64942d73ab11f7d4", 83456),
	ENTRY0("bluechairs", "041229", "4ebedbd54fc2eae27fc7d867cadb37c3", 241152),
	ENTRY0("blues", "020618", "dea57d7b1b41b5961143e7faa6125ad9", 261632),
	ENTRY0("bofh", "030202", "c43d7674558043729cf3916f7e1bfe4d", 101376),
	ENTRY0("bomber", "971123", "1c1740d9770d53d7a18cf56006e8d88b", 3584),
	ENTRY0("bookvol", "051225", "c949b9db879b10faea5cfca45ee6a657", 225792),
	ENTRY0("booth", "960409", "953578eeec2b85c92fd54d87f3e9c7fb", 44544),
	ENTRY0("boothdye", "345678", "772134a9ceb8c7dc5f2b10d6d139aa0d", 51712),
	ENTRY0("brainnightguest", "110312", "a53b823f032f6f24cf726cb3807126e0", 232448),
	ENTRY0("brandx", "111115", "2937e62eefe4f059ea9d8d85f4fd55a0", 112128),
	ENTRY0("breakin", "000926", "0cc04d377f04ba75f647c9a21688b9b6", 208896),
	ENTRY0("bronze", "060503", "ef5b57f69601b8f5944699a6b31bc58d", 492472),
	ENTRY0("bryantcollection", "090401", "2c331a7c183ef29ea13b68075a58c73b", 528620),
	ENTRY0("bse", "970110", "d260f722540f8c24a8c5b88778c76261", 89088),
	ENTRY0("building", "030706", "cc5ab40754fde9739113ef2524ef66b5", 253440),
	ENTRY0("nightbunnies", "971205", "6a2cc6996865a4d1520ae467a5a802fd", 78848),
	ENTRY0("bureaucrocy", "141113", "7dcbbc724b7416ad0b80b3159c7e3525", 387606),
	ENTRY0("burglar", "010925", "ca9ae17d983afb2d79aad60486a5a859", 71680),
	ENTRY0("burnsnightsupper", "120124", "a1153aab71b7a3bc4bd1aa056f3145e8", 251342),
	ENTRY0("burnkoran", "101128", "216f61842ba2024b291b4e0ae7769500", 222746),
	ENTRY0("busted", "941223", "e164cfeed308fd25bc102a18b3c8f15c", 82944),
	ENTRY0("byzantine", "091119", "61dcb1e79885f76f9f5f5c62ce2c7e8d", 1727208),
	ENTRY0("cabal", "041127", "b8f7f30bef898f4ed1814b0094c40b1e", 178176),
	ENTRY0("cacophony", "091209", "a612e786ee256ecf14580bd7e922b27a", 364032),
	ENTRY0("calendar", "070805", "8f8f6f45daa40e640805b83279196cfe", 137728),
	ENTRY0("calmmutemoving", "110712", "c4f0e45605825341e566bbd4a1d52522", 173056),
	ENTRY0("calypso", "140421", "5fadae3572711526d7b61c90bab58087", 97792),
	ENTRY0("candy", "970621", "322249ca0d96df20c010d8c2226cf0f2", 59392),
	ENTRY0("cars", "980923", "c857ca87a713ac8d3ae13a49ea05f994", 57344),
	ENTRY0("cars", "981010", "b6d057301718325b393e7e8e99bf09e6", 57344),
	ENTRY0("casting", "050707", "11ec0ca3c23c327e77a83985305c17f4", 247808),
	ENTRY0("castleadventure", "121207", "7119fd8068a8e467e3e34c66cbee005b", 243200),
	ENTRY0("castleredprince", "130227", "1dc1bc1337931b25d57594e566152772", 362880),
	ENTRY0("catcherintherye", "060611", "0bb71366efbf85f9a4700c6f121e7972", 198550),
	ENTRY0("catseye", "041018", "7ff8e2d98b56e7f79508f6e22d6383ae", 10239),
	ENTRY0("causality", "240304", "4ee72c8d430dc4d0e28fadf99b1dcd73", 117760),
	ENTRY0("caveadventure", "070718", "6d2fc2d639efba54382481ac319500b9", 550986),
	ENTRY0("cavernofdoom", "030312", "87769a24b64ec0dc2e261c7111d7662d", 133120),
	ENTRY0("cavernsofchaos", "990813", "05104b58bb5d6e1765b8a3be541381d3", 24576),
	ENTRY0("cavetrip", "041225", "5f0668876d3f8b03e62bd53d3ab276f3", 117760),
	ENTRY0("ccake", "000311", "2f0745fb253ef799472afb4e7e7f13a9", 80384),
	ENTRY0("chaos", "090801", "ede5478e9a87906205d784181b33e8aa", 96768),
	ENTRY0("chaosgame", "090526", "0b902ce73efd76e5c57230a25e27f364", 176004),
	ENTRY0("cheater", "960920", "69753e7cb886a003615b8aa415702135", 48640),
	ENTRY0("cheesedoff", "160724", "f22a5e611479057236e0a8af31b62e70", 448080),
	ENTRY0("cheeseshop", "021230", "88329068474b92abf4b4363c177f6971", 85504),
	ENTRY0("cheshirecat", "141012", "0a8c570fb7499bcc35109fdf7320132b", 79872),
	ENTRY0("chico", "000413", "b46bc30a1495c8cc9b1f9399f0ac5ed5", 162304),
	ENTRY0("childsplay", "080129", "d21a117c5a40afcbdb04a08aa109da57", 535396),
	ENTRY0("chix", "991124", "760b7fb0fcd9c99ae1295c06ff52529d", 206848),
	ENTRY0("cia", "961218", "2cdacedf416e7a575de47412e1d164f8", 54784),
	ENTRY0("claw", "970327", "4d3992e95530a301ca58939a197c6b1b", 130048),
	ENTRY0("codenamesilver", "170619", "8250de9dd95418d6cfa1b5f323254ac4", 357888),
	ENTRY0("cointoss", "150217", "e19c6d27ccdae1fee28136f8efb829ec", 314426),
	ENTRY0("coke", "990331", "8ee8282eceb70c6bb6b9c427d1e01aff", 107008),
	ENTRY0("coldiron", "111119", "82f24d7b6d66e76871c64ea47b4a6f5e", 230400),
	ENTRY0("colonists", "130911", "2f5bfc75a420df67718f542d67e9c118", 534020),
	ENTRY0("coloromc", "120110", "95a3d47250d6cdacfb2c9aa334ae71db", 194560),
	ENTRY0("comp96", "970626", "208b6f721e472f89654e6feb1b54b747", 84992),
	ENTRY0("conankill", "050428", "eff6760b564715f763d6fc25fce8a7eb", 91136),
	ENTRY0("dreamcorruptor", "171029", "1f4f956de3054621f928204c0e6b0d6d", 429160),
	ENTRY0("cottage", "090715", "b2c6f964c0bd4d8a0c12009697ff7307", 616308),
	ENTRY0("cove", "000525", "a84f448606c1b271a41b225244ef43a9", 130048),
	ENTRY0("creepydemo", "100317", "e211ed2eef47d78f72330d98e0f9503f", 150016),
	ENTRY0("crimescene", "130123", "508196f1cd974f789fdc2af69767a1d1", 320916),
	ENTRY0("criticalbreach", "130817", "348fc7ef47cab92d0b7f286da3c3ab90", 268800),
	ENTRY0("crobe", "111115", "d8b138a87f6226b8283c33c50c6f254c", 92160),
	ENTRY0("cryptographer", "071216", "5c306d5f1cee2d4eabf199012a354ff8", 40448),
	ENTRY0("csbb", "090625", "30bd815bc7d5943a0c8725e3b31ec373", 489984),
	ENTRY0("ctdoom", "000920", "24e754f22952fa0dfd88fd1e50e4e4d0", 132608),
	ENTRY0("curses", "951024", "636ca27c82d3af77142ea92e6522b9ba", 259072),
	ENTRY0("curves", "010613", "c80a64ffb0a19c5cb89108fb36485d04", 524288),
	ENTRY0("cycles", "020222", "cfe1bb9275f72a63e6b4d96fb39907b3", 121344),
	ENTRY0("cyclops", "020505", "f4ab4dc7f8f8e0fc1f716b3a43273b58", 78336),
	ENTRY0("crystalpalace", "111125", "70b49b3ed49f022f73796765008e1e6a", 310316),
	ENTRY0("dday", "110720", "897fbdf9cb2468b09c30961524d36ae5", 266364),
	ENTRY0("damnatiomemoriae", "061008", "c69e98feed6b49dcecc040a6babab455", 166474),
	ENTRY0("darkiss1", "160130", "8c8f759915eedbf014b1c8d8ecc282db", 134144),
	ENTRY0("dashslapney", "120510", "d934c25530f689c8c25ce71bf184b82a", 338426),
	ENTRY0("dayinlife", "080406", "ad0b46d14e3a8d577a58f0d93d700f4f", 173558),
	ENTRY0("dd4", "030207", "c5921ad782bc25cbd7e3f8c8b1412a4a", 163328),
	ENTRY0("deadpavane", "131204", "cf0f923ae3b079c29608770aecdefdf1", 269270),
	ENTRY0("deadreckoning", "030730", "1232dc599a00548bcc2d6453a01c5e50", 87040),
	ENTRY0("deadsville", "081105", "58ddf8bfbd4a78aa722127193f6f54f5", 120320),
	ENTRY0("deadmansgrave", "151101", "85750a3498daff7750b65f298858f175", 547808),
	ENTRY0("deadmeat", "170527", "c0f127f032ade1f46267028371e68e5b", 252374),
	ENTRY0("death", "030309", "4e013a77c165338cbf2662ba04465a9b", 120320),
	ENTRY0("debate", "070412", "bc1c6db5b5fe43d6d11982907debeed4", 162816),
	ENTRY0("deephome", "991210", "d30357d2b08ab21e8f2da0e74a3c87d7", 133120),
	ENTRY0("degeneracy", "010331", "d42d960bcfc2a8e1fadc809b31c96f02", 167424),
	ENTRY0("dejavu", "930921", "ece489ece5cffa0b8575f44e094a115e", 22500),
	ENTRY0("deliciousbreakfast", "111015", "393e71995dbb13c23a1b5dd5403341d7", 152852),
	ENTRY0("delusions", "971121", "8e78eef73d07048d99514bab624fb9aa", 193024),
	ENTRY0("detective", "000715", "c09fde6c6777c2c422de18668cf986a2", 108032),
	ENTRY0("detention", "130827", "c2361faf8bbc8996fc0db8e71ef001da", 244684),
	ENTRY0("devildoit", "000724", "c4c97ca9af421dc9a14849355e7f7a1f", 60416),
	ENTRY0("devours", "050325", "f9be89a5a26be53b52c08b6cea0ed3d1", 160768),
	ENTRY0("dewdrops", "050115", "32e07656bbfbc7b4d0b5fe8fd4d1ca7d", 109056),
	ENTRY0("djinni", "001117", "107f0d7432596234db354dbe3cbb4b68", 105472),
	ENTRY0("dogslife", "981015", "e6f0adca898d757c49c9d81a67d3b6cc", 67584),
	ENTRY0("zunidoll", "971031", "128ad329e657c405f85ddbc19bd26538", 76800),
	ENTRY0("dontgo", "120119", "5b153263f946a6013d80d07fa1573dbf", 195584),
	ENTRY0("dontpeeyourself", "150807", "ffbdd3b0860e5a5b5ffbbdbb6ac4fc46", 211968),
	ENTRY0("dotd", "050619", "48b47df16c2d32612fe2f09f234d99dd", 245248),
	ENTRY0("downthematrix", "120521", "291d6e004fe0ace1dbaf79434fbd5403", 319976),
	ENTRY0("dpod", "051012", "b573b31e2df3f90b396a6ddaf27e120e", 303104),
	ENTRY0("dracula2", "080819", "446564750ebe364539c27f89a520bb08", 108544),
	ENTRY0("dracula1", "070424", "42b5559ca5f01f7ef63ef4573d71326b", 87552),
	ENTRY0("dragon", "040211", "744c35647fddfb830594596b4b350b71", 145408),
	ENTRY0("dragonflies", "111130", "ff692bffb014fe109c3bfe2e072ac98b", 565060),
	ENTRY0("dragontroll", "070120", "44e48cb9689e156d322a1de0a14d420e", 3584),
	ENTRY0("dreamhold", "041231", "78eb852b88f9424647a4e18caa4c6707", 386560),
	ENTRY0("dreamtooreal", "061128", "ac2bc1ebcfd2c8b5eed405c4f7fcb10c", 138124),
	ENTRY0("dual", "100225", "57f0e173be4258de1ce224e74172ea11", 430282),
	ENTRY0("dumont", "990223", "c5b6b0c7dcf3e771207bc44a7a3eaa0b", 224768),
	ENTRY0("eas", "011204", "5dffcdadd1f24333aa6cb03fefce22fe", 183296),
	ENTRY0("cliffedge", "111030", "2bcba042a3b56fb010c9ab0e64cecad5", 324608),
	ENTRY0("edifice", "980206", "e2fd79d86f7e77659ef60519d2423856", 181760),
	ENTRY0("egyptianwalking", "151231", "ef80dd50c700becf2b4b254ce3ad3c9f", 334848),
	ENTRY0("elephants", "120409", "6e9cbdf3128079d42abef931964ed835", 236032),
	ENTRY0("eleven", "001231", "a573abd0e05318d72502e2ff3b48f5fa", 214528),
	ENTRY0("eliza", "990207", "293e7e1e2790af326547cd7be6c44d92", 7680),
	ENTRY0("enemies", "990115", "ff9fc17bf66ffbfba125d9a7c20c2c60", 289280),
	ENTRY0("enigma", "160202", "743b7440094810ebe9b9d86d994ef07b", 418468),
	ENTRY0("enterprise", "020428", "791a68da0a0da75307ce60bcf6f07f55", 156672),
	ENTRY0("entropy", "090516", "2acf5da303368989679bbb8ee71da5ea", 125440),
	ENTRY0("epyk", "091111", "2d14b9b9499bc87ecad01215009fa2c7", 236988),
	ENTRY0("erden", "980131", "6125e83116854253240b489a22e83b9f", 348160),
	ENTRY0("eric", "000001", "6ed92a1deef039921701441bc371883a", 116224),
	ENTRY0("escape", "130210", "988b6899f8d3c9bc4bbd3153ce1ab31f", 275358),
	ENTRY0("escaperemember", "060705", "57b062c6f3e42cac8cb0775265c33c42", 284810),
	ENTRY0("eurydice", "121206", "1f4112cdff2e5c924e54bfe06181ff49", 845206),
	ENTRY0("f209", "970806", "98d739b8f6f5df78cbdc338dd50f89c5", 56832),
	ENTRY0("fable", "000715", "603d4573ad89adabb7bf8d0b031705c0", 95744),
	ENTRY0("faculty", "080116", "8998ea6435611921f97bb67fff0dd130", 218612),
	ENTRY0("failsafe", "001218", "05293830576ae81e9a61f38cab731d38", 78336),
	ENTRY0("farm", "020101", "9838d582cdd87814598eff791775a53d", 135168),
	ENTRY0("fff", "080121", "c29786d16b5498656e19f06eb3d58d94", 100864),
	ENTRY0("figaro", "100911", "b696b4578c9d4044edb137e3a87b0f50", 406596),
	ENTRY0("figueres", "1", "c0d9ff83a9ff5eba8938934a339aee6d", 277992),
	ENTRY0("findesick", "180802", "ccb98f2fa05f747e00c883c47e8ba31f", 331776),
	ENTRY0("findthebed", "100708", "02d3f5a2fba50f42c750a902f04f54c8", 289224),
	ENTRY0("finetuned", "020520", "354ff8d5758cd27f054d5f575dd37742", 293888),
	ENTRY0("fingertipsfriend", "140106", "44cad06c5c79a7f99e53974a7d08c273", 423480),
	ENTRY0("fingertipsmilk", "140312", "01d3504181777b30f96de20dfedda359", 278664),
	ENTRY0("fmvpoker", "001227", "0d7bec942838ba2d42af2f836e2a8fc5", 24064),
	ENTRY0("forestdemo", "121216", "f629bfa6755cd36f45dbc3fe12b97936", 243130),
	ENTRY0("fork", "071205", "cc07f97369a7f20da06eabf4fa8f2e4a", 201728),
	ENTRY0("forms", "970206", "7de27dd6539ed6ef1c6176ab030510dd", 91136),
	ENTRY0("fracture", "010306", "d684caf97f01d447b37376ef7c708a85", 111104),
	ENTRY0("fragileshells", "111109", "6232ef15b7508aab6072a85afb86908c", 547610),
	ENTRY0("frankie", "040304", "1ce64a1c5a2d928cfb48b4d760246972", 91648),
	ENTRY0("freefall", "951111", "3e7898a1e767a2de61336ff2ff4bb1f4", 3584),
	ENTRY0("frobozzi", "990129", "3ca33fdafd15fed40ce32de557a84195", 184320),
	ENTRY0("frozen", "960620", "e6185f87a3bfb42baf2f1de528a479a4", 63488),
	ENTRY0("fyleet", "111115", "df1bc4b00ad8bc376a29c74ad7c23a22", 141824),
	ENTRY0("fmvpoker", "000913", "3f4ee585aa91660d5483ef2f132fb1d3", 19456),
	ENTRY0("firstday", "110119", "58c46cbeeaf96c3302515cb3e9aedcec", 454606),
	ENTRY0("hiddennazi", "100911", "93521e3903e1fa63cfb7ababebd4c3e1", 367570),
	ENTRY0("ga", "120510", "ec52eee6bc32f783edc18088b6abf943", 177664),
	ENTRY0("galatea", "040208", "7d2f6474d49625ff3537f594f17d2469", 441992),
	ENTRY0("gamer", "120415", "f5cdd32b3c0c1f09fe7c659103317365", 293356),
	ENTRY0("gamlet", "041223", "c434581d222f89b85e67bea37aa702af", 260608),
	ENTRY0("gardening", "080412", "2d9ff32f10bc569977984c9042afee4f", 394156),
	ENTRY0("gatoron", "091116", "00ddc82b9268ea0968084e215057f21b", 123904),
	ENTRY0("gaucho", "120805", "3ec2942af85d52ecd22c6ee3decf9b8b", 176640),
	ENTRY0("gd", "010806", "5be045c6983f16f2d44e660f9f0192a2", 50176),
	ENTRY0("geb", "051028", "8b731db0410993b8421e06dd7f3e37b2", 165376),
	ENTRY0("geist", "140115", "96f54aaa29ddd0bbfa5db914150633a5", 244736),
	ENTRY0("gerbilriot", "000513", "69d3926fdd11464a3a79bc10c03cd324", 79360),
	ENTRY0("ghost", "290912", "3e737617d1eb7f551d858e1b9ad5dd3f", 210944),
	ENTRY0("glass", "061008", "f687986dab7bd4e2f61d3f282c1ded4d", 373812),
	ENTRY0("glik1", "111211", "1cfc3dd21b01fa968d36b6d58fe80862", 353620),
	ENTRY0("gnuzoo", "160331", "1f9274506529a4c934f09cfe06cda804", 478060),
	ENTRY0("godot", "960599", "953905ec43f6e595d1634adb37ed879e", 49152),
	ENTRY0("goldilocks", "090410", "1e2b32a419bfc9703dda529f40c01782", 217600),
	ENTRY0("golf", "010114", "5dd8638e40d97806c5eb9c05618db775", 195584),
	ENTRY0("gostak", "020305", "06a39a234ada7e57733e1ab442909170", 88064),
	ENTRY0("gourmet", "031116", "6a78f4aa50d962668a2bf7b0e788bafe", 360808),
	ENTRY0("gourmetgaffe", "120624", "0c59f0c03fd36303ec02d370cee9b650", 257156),
	ENTRY0("gowest", "120425", "2c1e329d9cde395785f8323a740a361e", 251988),
	ENTRY0("greatxavio", "041204", "3725be7347d460cc3661b0268db55db5", 177664),
	ENTRY0("greenrain", "100611", "f28e8e500d4fbc94d59f5bfdb9cf7f4d", 275516),
	ENTRY0("growingup", "110825", "9bada495ce70bfcae2566f01b844d20a", 267720),
	ENTRY0("grue", "170810", "578ee64d95f854174a256e75e90f2306", 365056),
	ENTRY0("guard", "991231", "db56b6f0f381ba32f30749530ebdc25f", 410624),
	ENTRY0("guess", "001210", "df6dcaadb7f2d25cbbeb1b357c716a32", 235008),
	ENTRY0("guestreet", "100031", "2637075afb748532ebf851310158017e", 12800),
	ENTRY0("gumshoe", "960331", "935eabe90b3904a0cde9811a463fcc39", 143360),
	ENTRY0("gussdeath", "040131", "1c3d12957d1ecf3ce28b7fe36cb34111", 41472),
	ENTRY0("halloweve", "111028", "0a209a7dc47242f906d8b782e4efb887", 324608),
	ENTRY0("hamil", "111115", "3c6f7131752edfd7b78252112b3a6e66", 81920),
	ENTRY0("hamhouse", "081124", "068b91ec4a227760e08f1ba48acaeb7a", 238080),
	ENTRY0("hangar22", "110617", "a96bff6dae1e76771688f73e6570550e", 152064),
	ENTRY0("happyeverafter", "130114", "921e9eec1c4049b630b0a00379bd7d5b", 121856),
	ENTRY0("crabhat", "130216", "c7ccd1b68fd860cbaec5ee721e0a06e4", 154112),
	ENTRY0("hauntedhouse", "170103", "6d1a6caced27d901c09896ac74de6022", 400078),
	ENTRY0("hauntings", "111109", "931b910ab47044e8792c4d5adc5163e3", 225792),
	ENTRY0("heist", "990217", "d37eab3288f34d5673f72d8697835e8e", 358400),
	ENTRY0("heliopause", "100703", "423b5192b31eb0705a928b3900b6b208", 389388),
	ENTRY0("welcometohell", "980816", "1c7493e0ca533b9cc04097d90b112b24", 51200),
	ENTRY0("hellsbasement", "140826", "749bad5b6169b1cee259b9f21cfee75b", 275890),
	ENTRY0("helpcollides", "030208", "2185bd75e8cc26ac7773effc300ec210", 286720),
	ENTRY0("heroes", "011130", "1bfe7f301e15816e46c60edfae9030a5", 204800),
	ENTRY0("heroinesmantle", "001211", "4beb00dfd7abed53bb529946662fc252", 523776),
	ENTRY0("hidepachyderm", "151011", "304cbcbdbd9338fc60872270905ed6f0", 215040),
	ENTRY0("hlainform", "050715", "e919e8ca4e2358b010d3a758a98609c8", 168960),
	ENTRY0("hobbittruestory", "170425", "5964f6951be8da16fb350b9d0982c89e", 374272),
	ENTRY0("hobbittruestory", "170601", "037bdb4c4518a9335f86b71a11a673e9", 414292),
	ENTRY0("robotempire", "140601", "0be65fa2571465799544009bbf9cc083", 431986),
	ENTRY0("home", "110204", "454bbecf0847b69f17748a04e225fca6", 156672),
	ENTRY0("homecoming", "140903", "b3987678e4d9fcc95fa9718c3f9ce7a8", 1089176),
	ENTRY0("hoosegow", "130320", "5d8d2858bb803b227ac923a654dd367c", 2024678),
	ENTRY0("housekey1", "080112", "ddc18dd2a69887dec6c1d4d994d35a8e", 134144),
	ENTRY0("housedream", "071204", "64709099677209a81c9edd196ef3e266", 312320),
	ENTRY1("hellosword", "060113", "7c5289ab479a5aace75a5518969fedf3", 153600, Common::EN_ANY),
	ENTRY0("humongouscave", "000001", "4d0a5a75192c8343315855f881874515", 332800),
	ENTRY0("humongouscavehints", "000001", "d6581a6fcf31851267e1acc00c466e4b", 124928),
	ENTRY0("hummingbird", "110101", "1b48ec7b38899b5747ad4a16be029889", 775268),
	ENTRY0("hunterdark", "991119", "3c3288e7023f07e525d7301bb84ffba1", 116224),
	ENTRY0("hyperrpg", "090630", "46d18c446800ea7cdaa7f1d370ab7c05", 198628),
	ENTRY0("i0", "140603", "878cd0b3cb05340b49cf18999ba59fab", 219136),
	ENTRY0("ibo", "120109", "b592aa3222e86ee1baaa45107cca6f83", 196608),
	ENTRY0("iceweb", "160229", "b4f294ede4bcf5134a49d3a8d47cd2a0", 506932),
	ENTRY0("identity", "041121", "72fe1d48259be82d586bb4eb48ccaff9", 126976),
	ENTRY0("ifaquarium", "100515", "5f99b211f5bef1c1d1e89762dd451cac", 235088),
	ENTRY0("ifquake", "040812", "4c75b746abbae0e869ede0b06501f823", 133120),
	ENTRY0("ifwhispers5", "120312", "ebdd62f2484dc87955055cb1093d9359", 326144),
	ENTRY0("ill", "000218", "34f194a208f828c4c77d3954082d75a7", 73216),
	ENTRY0("imiagination", "110608", "df766ac578a86f0b9d7231bddac5b09b", 252886),
	ENTRY0("inamanor", "180117", "65b795d35855bb12e8914eeedd522c2c", 506996),
	ENTRY0("inevita", "030428", "6152b19bbf3eb3dd4334004e7094dbe8", 213504),
	ENTRY0("informatory", "981211", "c9eb276f103f83b8e7044c1f3930264b", 141312),
	ENTRY0("inhumane", "950817", "e07f6bde81912a655468010fa13ed9f9", 68096),
	ENTRY0("insight", "030209", "77cd4bd598e9f175faf8f84d6751c140", 151040),
	ENTRY0("intangible", "131229", "b1941de7a9ce4efc9c21e7ba11072954", 304060),
	ENTRY0("ifplayer", "130325", "d471621d98969c2c3050586af788a88f", 251826),
	ENTRY0("interviewrockstar", "120223", "559b8521d944c959739095fc3f0e06b7", 243226),
	ENTRY0("lionskin", "121106", "dfdc63b480603baf62a2ccb34533bf0a", 325570),
	ENTRY0("invisibleadv", "100130", "f42de996030f1caeb1695c6923db2249", 336896),
	ENTRY0("cubicle", "160411", "6a83df1b1ce0d799847c80741d2cc7df", 347684),
	ENTRY0("pressedon", "130323", "26ece4d715035234a5d6d8275bd4de85", 246968),
	ENTRY0("islandsfaraway", "100426", "67a890cf7b6bc8c2b40ff101a3a69784", 262616),
	ENTRY0("intruder", "990210", "c42924c9879af8cb682d490aaf9841b0", 175104),
	ENTRY0("iraqiinvasion", "080131", "427ff378923f675915b95993b5858704", 260780),
	ENTRY0("IraqiInvasion", "080131", "c3ba7d2b115625e962e0614a5c381f4d", 259072),
	ENTRY0("beanstalker", "180105", "6490a77e5f0d5e01b37c0ed29e017a56", 33812),
	ENTRY0("jetblue", "051012", "e1b0f957e0a4967a0f5d66f9e7313944", 209920),
	ENTRY0("jewelofknowledge", "990710", "38c384ce3d1c31791fe98859558dcb68", 225792),
	ENTRY0("jigsaw", "951129", "44b8fdbadfc399f48a3367fade486e07", 304640),
	ENTRY0("jigsawrules", "951128", "a1f0f980850e6b7e4ef79f93d549a227", 64000),
	ENTRY0("jigsaw", "951129", "faefa6d593cebdd177167c24f7cfd373", 262144),
	ENTRY0("juicehead", "150112", "54288e72711653d544da189152eaea0b", 272146),
	ENTRY0("justanotherday", "170129", "33a1880b1813fbc0d7ceac6ca75dfe59", 616844),
	ENTRY0("karn", "961207", "daa05c92a3a1b678c29fe8bbf64a09be", 165888),
	ENTRY0("keepingdido", "120516", "56eb7cbeb898cbd892146d9795a91d4b", 332180),
	ENTRY0("kidnapsea", "110608", "ddcf23cb530746544e0f416337dcf6bc", 191488),
	ENTRY0("kierkegaardsspider", "120413", "19f457515033fd938856e6507b8a6bc1", 254414),
	ENTRY0("kiiwii", "141113", "091ab4e450db053a419fb4a8ffe87a22", 255488),
	ENTRY0("killingthedoctor", "010221", "6370cf6b47fee82e75f1edb6e4b7f7b0", 57344),
	ENTRY0("kirby", "062776", "7750af7ab62cb01dd04e5b665d299d1e", 143360),
	ENTRY0("kitten", "130320", "a55520ba5084b68bfb3d1ba9346dd2d6", 45056),
	ENTRY0("kooku", "971119", "68dc431ab53ff87af1a3bc037122667e", 94208),
	ENTRY0("lambs", "971010", "39b487be161e229f56badbec69466a5c", 47104),
	ENTRY0("lambs2", "021230", "2f6596397a42fdd83407cb2520cffaea", 59904),
	ENTRY0("largemachine", "010905", "abd265d7c2622ff6d9ca9ff5266cb802", 160768),
	ENTRY0("lash", "000806", "1ff1bffe7386b44fc4d34a34fc59ed3e", 370176),
	ENTRY0("ldodoom", "000801", "b4a663fc9adf6b344b7b64eb58ceff1f", 200192),
	ENTRY0("leaptime", "081230", "aa7363833f61f3b5ac21d2d681ab74bb", 233984),
	ENTRY0("lecture", "140813", "b8f9441556f6e785fade1e41d75458d3", 493520),
	ENTRY0("lex", "120830", "29de1a490f09a9c756b28c00bcc95230", 368060),
	ENTRY0("libraryfront", "951204", "e54ca81e93629a6e8ab2dbc84a6712b4", 72704),
	ENTRY0("librarian", "130211", "ff40eb307c9ec5b015531201cb0bcf00", 247754),
	ENTRY0("lifeordeath", "980702", "aa820dffedafe40d7d3f895ec8f30840", 51200),
	ENTRY0("blacklily", "150318", "f74981accbff8eaa36e07bb175640ea2", 199124),
	ENTRY0("limp", "980517", "9e36e35b17cbdebcd9d348de156b51df", 47104),
	ENTRY0("nemeanlion", "081113", "c298fdabefb085ca297f0f0917e2fd4e", 77824),
	ENTRY0("lists", "960823", "c4373396c5c8e499073b175349161359", 116224),
	ENTRY0("livejoseph", "111227", "08e7c0024e7735cc4ef7e8cf0e04c112", 264782),
	ENTRY0("llr3", "070314", "4c2e6391bf678f0563fee5492c7e73c7", 164864),
	ENTRY0("lmstvg", "081227", "971bdd7f99e59ea1afe0629c2adaa9f4", 189508),
	ENTRY0("lookingtothesky", "110313", "c83056d3ad3775af870fab9ef800a00e", 379738),
	ENTRY0("motherloose", "981128", "87de736abcf52f27a00d09b74cfda240", 206336),
	ENTRY0("lostspellmaker", "980116", "f569560e28e57b0cc15abe69292ddb63", 128000),
	ENTRY0("lostpig", "080406", "400ab3efad70e5b454452e89bf0625e5", 285184),
	ENTRY0("lostpig", "080406", "f7ede4fcda5d7ee1fd8831e7f8fd2256", 390256),
	ENTRY0("lostsheep", "130000", "dfed127ff9f1300dca25fb9f7d126716", 190976),
	ENTRY0("ludicorp", "031010", "65250e5c5cfcb403f9fb870ac7e30041", 101888),
	ENTRY0("magictoyshop", "951018", "68d597fc1a7b22419033628967dad19f", 73216),
	ENTRY0("magicmuffin", "120505", "b68fdf687db87217caef5e6c9de65f4a", 208660),
	ENTRY0("makeitgood", "090921", "1f4feb2361e05361ee27bf82f8a3c559", 512606),
	ENTRY0("mansion", "010505", "cb310588d3ebdfe419b16ed29f5bc8df", 107008),
	ENTRY0("revenger", "000403", "1f658ebcb395be17154e732cc6306dc9", 88064),
	ENTRY0("masquerade", "010122", "a88a4f7cc20883249db3aaae1aad871c", 225280),
	ENTRY0("medusa", "030314", "f9e93b13a6f940ef0d25cfef559e36c8", 109056),
	ENTRY0("mercy", "980217", "865d53d9764636ddf1bcaa2b703673a6", 97792),
	ENTRY0("metamorphoses", "020222", "cb23d62a4dce52afd6104683aface9f6", 177664),
	ENTRY0("mhpquest", "960613", "58b0832c491d961426b73a6b989e905a", 52736),
	ENTRY0("midpoints", "010330", "1004502d7dea284be6c6298316c5f9c8", 94720),
	ENTRY0("mimesis", "980110", "abd66d64b98d47ebc88f69d35816fdf8", 90112),
	ENTRY0("mindelectric", "941008", "900aae2e5ef17d957e26e8bc0ef90b86", 70656),
	ENTRY0("mines", "101029", "fb2d0e729e8b50d7d10f71b06317b3b3", 65536),
	ENTRY0("christminster", "961117", "86e6349dfa3ef6153cd1c0ecb9b4fc6a", 228352),
	ENTRY0("misdirection", "060304", "39a12ab5997240cf406f3c67001685fe", 221184),
	ENTRY0("missinggrandpa", "120319", "3e315935045c1cdd15bb37a8e8700cc1", 590198),
	ENTRY0("mobius", "061119", "ffafe18502f8c83789f904dde63f9938", 427364),
	ENTRY0("monstermaker", "150817", "f0565d38e01ae2ec048b5f3b8ceb4de4", 285696),
	ENTRY0("monzasphantom", "070219", "b5f91ecd114f3257ad269f1afb64b64e", 116092),
	ENTRY0("moonshaped", "080710", "1cfa040aace3a4cb249818684fb43b9f", 447840),
	ENTRY0("moonglow", "041011", "3ee29824d9875a4ae21e82b88813990e", 10239),
	ENTRY0("moonwrecked", "110309", "1cc71e3087cb4f92fc244a5c72b8a078", 264668),
	ENTRY0("more", "140604", "6e3ec8554b9c877c088fa586c965d896", 315392),
	ENTRY0("mortlakemanor", "120304", "80d317fd404451e436e9c36de5445de8", 261632),
	ENTRY0("mountain", "030317", "b84bb15597f7aa53ae407e2d996c0c6c", 103424),
	ENTRY0("mousequest", "100509", "2993b235743e6a6a4d69063e80c187d9", 325062),
	ENTRY0("mrp", "140620", "974627f05fc3e1c2260925a78171d73a", 175288),
	ENTRY0("mst3k1", "960831", "362107be49493f6ed1ca7664d7884a66", 105984),
	ENTRY0("mst3k1", "000715", "0a0748937d23bf380cd139de874df32d", 141312),
	ENTRY0("mst3k2", "000715", "cf16a0b763735b06cebef3cd5fdf2c78", 121344),
	ENTRY0("mulldoon", "000724", "84ef2f5f00c06d6cad6b0817844fade5", 468992),
	ENTRY0("mulldoonmurders", "020214", "1dfc1e3605f6eebfca49e8fca17e957b", 231424),
	ENTRY0("murdac", "111115", "790b5e8cf9677615d17ee75e89a99d4a", 78336),
	ENTRY0("murdererleft", "120117", "cb99b94b046bd7b7db8912617f3da9c5", 196608),
	ENTRY0("muse", "990922", "4dd315769b9d4c4782adaac1d42d0283", 175104),
	ENTRY0("musician", "070820", "0b5aabdb180fdf0199f7779baab3e354", 142848),
	ENTRY0("mylastduchess", "140510", "82cd30149bdfec0ae3a149b1f612dfa9", 258998),
	ENTRY0("conceptisproven", "120620", "e5f443e775546084096fe11afb6a313f", 291292),
	ENTRY0("oceantower", "110510", "27e6ac7767ac9dfe5f0082765f6a21b7", 306962),
	ENTRY0("myangel", "001218", "f478be18002189fe31df1c499ceab89f", 257536),
	ENTRY0("nameless", "131206", "197a58d317be0e6060c490bd40baf8a1", 472064),
	ENTRY0("nautilisia", "130422", "aaa73314eca2a1731f5f212c7de11aef", 1796142),
	ENTRY0("newday", "560827", "c43124915cd5677e0c2e89f081586e8b", 125440),
	ENTRY0("nidus", "140416", "8eefda240f46c6b5516a3a1fdfa53497", 117248),
	ENTRY0("nightcomputer", "961118", "c2ecd9885f53897d253ff11cb80f62ad", 74752),
	ENTRY0("nightchristmas", "061224", "e4eea0aa89a8cb9a614c1e8122498425", 124894),
	ENTRY0("nihilism", "151001", "9157c2d79e12924a91ed2fd4b3bb490a", 317952),
	ENTRY0("ninepoints", "961127", "278e64924fed8e10b89819c0433daeb2", 74752),
	ENTRY0("njag2", "000319", "b566c2127fdd479ae4afdb5f2d019403", 147456),
	ENTRY0("noroom", "040117", "d3fd94d4560c15ff6bfb8327b56c97f4", 61952),
	ENTRY0("northnorth", "141114", "447ba8bf4e026bd1c6995a3d9306d207", 347136),
	ENTRY0("nudistsgonewild", "120425", "5f98bc9a14ff9a8c2cabd78238fafcb3", 180224),
	ENTRY0("adventurelobjan", "961209", "e8cdbc003c5bc0965ad2f04a4a7806db", 147968),
	ENTRY0("oad", "000913", "c4d1e88c95803bb9ae0e669c575feb67", 78336),
	ENTRY0("odieus", "941021", "b75120eda34c37c87201405426329892", 50176),
	ENTRY0("omniquest", "040127", "6d246b048e071dca1c2f47e64e76f10d", 95744),
	ENTRY0("onegirl", "020113", "7ff5749ec2881b1343c06fc5950f928e", 173568),
	ENTRY0("ogisoas", "101102", "673a5bb3476d15e3d54ddba3176d9568", 155648),
	ENTRY0("onyourback", "130506", "c942e2c31131f799481e61ebd878b4f2", 173056),
	ENTRY0("orevore", "071116", "fc4d75670412a50ffdc98e483adfaf5f", 415744),
	ENTRY0("bloodless", "111217", "bf633d44d9ae34fb006aa1c135d24fe2", 188928),
	ENTRY0("orion", "041123", "16995ce9a4abe4a4670471beda078611", 204288),
	ENTRY0("ottumwa", "021409", "7bf33289ff4c6fbdfd467b387534b309", 123904),
	ENTRY0("outofthepit", "110515", "dc78b694b2ca682084ac7efd769acd9d", 276932),
	ENTRY0("paddlingmania", "314159", "8e995f6fdfcf7e96ee9cf6a142bc5053", 54784),
	ENTRY0("paint", "980316", "2028550b59c1b2f0892e90545ba15adb", 9216),
	ENTRY0("palebluelight", "110708", "94ada352a565eeae35738b0ecdd3ecfd", 355538),
	ENTRY0("paperchase", "950503", "dd421533f35d62d0da704958e5b33d51", 57344),
	ENTRY0("parallel", "080219", "b50ce87302a473152ae9d556dc5f9419", 158720),
	ENTRY0("parallel", "140414", "2e50a0ce61383d3c28c23f2b142a0f03", 210900),
	ENTRY0("paranoia", "981208", "1309a1b60d62af820d17ee812b0f8a61", 80384),
	ENTRY0("paranoia", "090803", "887280c21df983c85eef98080abd182e", 137090),
	ENTRY0("parc", "111115", "bd8698030864dacbf494226fc6b9dd0c", 98816),
	ENTRY0("pathway", "080511", "8eb9bc735afa30d84fd4f913fc29bbc3", 264192),
	ENTRY0("peacock", "000208", "5a4cdec49d14f905013ad9ae35ba17f8", 78336),
	ENTRY0("pegasus", "130708", "f65a086805d473b3ccbfb9971383ef65", 417200),
	ENTRY0("pentari", "030206", "3879d37f7417f6ce2a0acb96faf00c86", 91136),
	ENTRY0("perilousmagic", "990821", "56901ae08078fc15fb74a50e48101bbf", 66560),
	ENTRY0("perrysworld", "070501", "6466fe74c657f960521237d5a3274b05", 124300),
	ENTRY0("phlegm", "970804", "2f6f7667a3c2782aa0f24eeef7c7fec0", 70656),
	ENTRY0("photograph", "040827", "0949e8e4d3e1e035913fc75112f40459", 264012),
	ENTRY0("photopia", "120416", "3e9ea30956eb3e9494e6e5d9881a1307", 239616),
	ENTRY0("piece", "970819", "5703a209e660a58f456b0ec99707c3fd", 105984),
	ENTRY0("pigpancake", "110410", "7e6e0b1c90a9a01b4fbaea21ad729d55", 236446),
	ENTRY0("piracy2", "100408", "86e2ed40f55413f29fa97bd6e4c31260", 249856),
	ENTRY0("piraterailroad", "110920", "20d019cd1654f5d9d3c1bcffdbded2cc", 333500),
	ENTRY0("iceprincess", "960901", "f00f7e3f96670613ee73ec51ef2b2957", 84480),
	ENTRY0("probing", "990919", "f7cbdd49ae15d380f66ab247c118f17d", 73216),
	ENTRY0("puppetman", "040803", "ad3143a695c3406e7ede1f39dfb6ba63", 110080),
	ENTRY0("punkpoints", "010108", "ffb32fb83c4480a3109755572819623a", 136704),
	ENTRY0("putpbaa", "010618", "458a35f2318afa617fe4679f8c0ec9a5", 132096),
	ENTRY0("pytho", "020223", "a5e3d0ebd1f81ca341cf93a721f6ed3c", 293376),
	ENTRY0("muffinquest", "120326", "58eedaa233021cbd8f90ea503b0c3fc1", 438082),
	ENTRY0("muffinquest3", "120505", "41462801e25ca195c377ad34ee080c9c", 439162),
	ENTRY0("simpletheft2", "110601", "21c13d6583c2ce9714e9bdc31adcab82", 357818),
	ENTRY0("quidditch1954", "121022", "ae9626ce114047e0d1247d226cd9cc4a", 330710),
	ENTRY0("ralph", "040309", "ff6d29e293d82c49b4a5f4b21ce96655", 78848),
	ENTRY0("rameses", "061023", "6ea654d98a64cb50eff35d7a613cdfb1", 167166),
	ENTRY0("rans", "000918", "1577894d9022d221f9db27f7a96d4d4f", 180224),
	ENTRY0("ranshints", "000917", "b509a63314d10abf17272244db3f3bb0", 64000),
	ENTRY0("rachaelbadday", "070328", "3147589b43da574303bffb26c5f317ba", 269312),
	ENTRY0("readinginmay", "120109", "9ae7c7c4f4162262ea3fdfe49c30c0e7", 196096),
	ENTRY0("relief", "000820", "91046311f44a670f51738fcf5dc5a6cd", 94720),
	ENTRY0("reorsbushcave", "170416", "1d010db82d499b06e27c28375f867fcb", 329216),
	ENTRY0("reorsbushcave", "170406", "f84a6d0aaebc242b5b12fdf3f9315a33", 1058880),
	ENTRY0("reser", "050209", "ba7e7d973ce19279d7eacf9c5ef80136", 90112),
	ENTRY0("resident", "970908", "ae5612cc75ac734de51d864c9e8d9976", 136704),
	ENTRY0("retrofatale", "110713", "b7816c11b5dc619c9b1e37ffb6575577", 222208),
	ENTRY0("reverb", "990110", "fb3d80d0f92857286a4fd7e6c2892293", 106496),
	ENTRY0("reverzi", "991218", "31252071da0fb3dde08ae45f74a768f9", 9728),
	ENTRY0("ribbons", "010702", "7c45bffcbbd611a499e0803805fa87ea", 110080),
	ENTRY0("risenecropolis", "171208", "c01ccee1f849be82cccc85a1ab62d52b", 473566),
	ENTRY0("risorg", "171114", "e6e6c0e75711a28aac39afe185bea1f6", 475616),
	ENTRY0("robots", "980115", "0f9ad287635965ae521c62d29329eb2d", 6144),
	ENTRY0("rogue", "980706", "c4e104c4dec6381ee1a8943aa4e008e5", 116736),
	ENTRY0("roomserial", "120627", "9470b6e7a2cdcb43422e75deba20b864", 302080),
	ENTRY0("rota", "060430", "ca0f6e049bf7b17407e28a3f88e19416", 671830),
	ENTRY0("rpn", "090531", "d74a5da655a81e03447b4a241c1b5b21", 142720),
	ENTRY0("rtdoom", "000831", "5a8494839033e8c4dd036b6875e1641e", 179200),
	ENTRY0("samegame", "980731", "ffe1011cf0a3cae333825fcda58153da", 7168),
	ENTRY0("samhain", "001021", "9cb11fc72283939161b456c1592b00d0", 69120),
	ENTRY0("sangraal", "111115", "b663b3dd73ca57b4ddd6fafe7ec4ac0d", 150016),
	ENTRY0("santassleighride", "081227", "a4b5276e5885ed3e93f1ae0319488404", 441730),
	ENTRY0("saveprinceton", "041125", "13f9e45a4cbf39d541dea08b774377d1", 288256),
	ENTRY0("savoirfaire", "040205", "eb122ee416ee8fa1e6f909b6de6ad9c0", 442212),
	ENTRY0("scald", "180206", "07fb2e2748c3835bf4d1aaba70a708c7", 194048),
	ENTRY0("informschool", "991217", "388c8e73dd3611e67ed335b6234f4e2e", 241664),
	ENTRY0("schooldays", "092800", "2c3334c637e37b1b80ea089d6911477a", 192000),
	ENTRY0("scopa", "110321", "621b223a5f02c7e49d18ae0d6b588d19", 101376),
	ENTRY0("finalselection", "061009", "2dd1a94db68abf9626f55e8bd6fa91a7", 151040),
	ENTRY0("sfiction", "120706", "662d83be00d1a832e37bdedc565c1a71", 275968),
	ENTRY0("shadowgate", "040616", "9015104db32c046798870273f0754d3c", 141824),
	ENTRY0("sherlock1", "021024", "97577dc5e6be837277acd5c134620d92", 230912),
	ENTRY0("sherlock2", "021101", "b4e67e63abe681449d5cc727b161e4ea", 285794),
	ENTRY0("sherlock3", "021114", "605303b5dfddc04e590e6f060369463f", 220672),
	ENTRY0("sherlock4", "021231", "7b231c7acafc9a4959c859b68578d528", 217600),
	ENTRY0("shade", "001127", "e9ef7c17a40a4d0a30bb30c37ae1e4c5", 108544),
	ENTRY0("shadowofmemories", "061124", "07b4c14e309c00f8adec901afbc0e58b", 179094),
	ENTRY0("shadowsoldiers", "070625", "985406d500afb5e29a3b174b28b1f21e", 150418),
	ENTRY0("shallow", "121109", "cc2d82b5453ec921e69a80c04e1f1ad7", 245696),
	ENTRY0("sherbet", "961216", "46a865c6f01a800536463ffe93fb3d9d", 174592),
	ENTRY0("shrapnel", "000212", "cce4edfddaa7ce948cacdb0fde52fb2d", 98304),
	ENTRY0("sixgrayrats", "150331", "1a2b10729ebb83a08a38f0c80020a34d", 428544),
	ENTRY0("slackerx", "971009", "c7014f074407ddebc685dcd7bce5bd37", 55296),
	ENTRY0("sleepcycle", "101217", "5c9c1b6c46c057099cdc6ad04f30407d", 253892),
	ENTRY0("snafufun", "990925", "2b04ac88022c9df74b4ae59cb3f75f27", 13312),
	ENTRY0("snowquest", "100127", "2442363dd6f7f2ee6a6ca840ca9d9881", 1034206),
	ENTRY0("sofar", "961218", "026e1c0b7ef555011df2a9f72a2db574", 300032),
	ENTRY0("softporn", "971018", "6624ea7bffdef1e14e28f57a90b621b0", 105984),
	ENTRY0("solitary", "040607", "18011f0b8e3cf6aa6403313bdec2df0e", 97280),
	ENTRY0("somewhere", "080129", "abbd658087b2c7d0a52a5a787c75c0a3", 189952),
	ENTRY0("soreality", "100127", "1a43e51209ae7a2fa51081aff804b3c6", 146944),
	ENTRY0("neverplayed", "140327", "69d214a7a657a8ea7f2cb951b8bc3edd", 980260),
	ENTRY0("spacestation", "040130", "1bcf00508a8054c366d07dd507f81b22", 137216),
	ENTRY0("spaceinvaderz", "980710", "51ebab3a743c783ed284a582346b90ba", 22528),
	ENTRY0("spadventure", "971030", "519630c6c94fd783104061e439d57a8c", 143360),
	ENTRY0("spiritwrak", "960606", "bfa367d09a262d4efc83cb1ba1a99efc", 260096),
	ENTRY0("sporkery1", "080111", "eebd7008071529686b367b6641b9c52b", 147968),
	ENTRY0("spot", "021025", "0f1feec607bd423d9f1a95748dbd1245", 55296),
	ENTRY0("spring", "080207", "e74ccd9b57ee13df8ad2d20c14cb76b4", 330240),
	ENTRY0("spycatcher", "111115", "628c860890e6393f22d81af5c96e9bcb", 126464),
	ENTRY0("starborn", "110116", "0f22fb91edc9e64ad292fd6a7ca39f2e", 418624),
	ENTRY0("seeksorrow", "160602", "6108059cd52dc1aba9112b554d6aee0f", 1365918),
	ENTRY0("stealingthestolen", "161222", "6daa42b7ae7f933fcf2fa441e702ea70", 424914),
	ENTRY0("stiffmst", "690609", "33746a619e4ac30a1b9be82349187062", 82432),
	ENTRY0("stinkorswim", "090105", "7ed52b7400c044b4c3b47fa44154b5bb", 232448),
	ENTRY0("stonecave", "091103", "87858ce7d11ae814029c5b550682c255", 220672),
	ENTRY0("stormcellar", "081022", "c38434a33b97a9616956ce48a7d69404", 384600),
	ENTRY0("strangeworld", "091012", "d7c09eadf30d77858c88dc0f6706ceaa", 247230),
	ENTRY0("suicide", "101103", "a936b12bd8f8c80e5948f6035e1eb86b", 343040),
	ENTRY0("sunburn", "150214", "598cc42e8184a840bab2d38c6d6dece0", 471332),
	ENTRY0("sunburst", "070222", "9f1eee6db8dce7cfb803e5c430fe942a", 65532),
	ENTRY0("sundayafternoon", "121213", "73d9c96d54922bcd58781aff41e449ed", 347136),
	ENTRY0("briantimmons", "130114", "94545a75f7c9f0bc33a171a925f49af6", 544936),
	ENTRY0("sushi", "010416", "a8a04fb57890e1cd1d13bae889a5e2af", 327668),
	ENTRY0("sutwin", "970402", "b4fe3b29aab816470906ce3ae0613ba4", 31744),
	ENTRY0("suvehnux", "071226", "e82d434c1c0a73a8755a9394a7c2e088", 248320),
	ENTRY0("swineback", "060422", "853342a5b088a2998201123b0c5faa73", 111104),
	ENTRY0("sycamoratree", "980107", "335cb583a295cfd8a3de4e9cd2a267ee", 54272),
	ENTRY0("taipan", "020520", "0bd7a62517400c66fc961908688d1671", 37888),
	ENTRY0("spiderandweb", "980226", "2bac499c020fcdd75b9c4b65e4c1d85d", 221184),
	ENTRY0("tatctae", "970521", "21cf40691e0ba92d8e699b0fa3e18728", 378368),
	ENTRY0("tauntingdonut", "080612", "4d8443e896e7f294cc4a250c6090f6cf", 167482),
	ENTRY0("tblw", "121030", "4b6744d2b13fd85db5020788e0888f0d", 295424),
	ENTRY0("tcoty", "060908", "8bff4a19b0c1b9aa661f6e00d86df3b4", 471950),
	ENTRY0("teacherfeature", "071126", "be559856dd11f6b1fa5c5ed10f83f90a", 118784),
	ENTRY0("williamtell", "021025", "4f098036ffdc7463000c4f81b6300123", 68096),
	ENTRY0("telling", "060518", "cfcbfc5d4a7faad9724f7bae961ee71b", 316014),
	ENTRY0("temple", "021118", "1592469a7f1f503789443d2f56f5113f", 107008),
	ENTRY0("terrortabby", "080514", "86d448c209e93f1caa12a319406313fa", 204800),
	ENTRY0("tesseract", "031227", "583899fe706bd1367e31ad5a9f1c8ed2", 93696),
	ENTRY0("tgm", "050330", "e345e2527801880bffc74981571875c9", 44544),
	ENTRY0("thatdamnremote", "091214", "2b10cdcc7a83f6e82d3909838e5b439f", 251490),
	ENTRY0("thatdamnelevator", "160710", "4986a2629b35b9092b5bddf40df5417b", 415184),
	ENTRY0("penury", "120528", "58939b4506dc1e27b36cb7e75a1e2479", 231424),
	ENTRY0("penury", "120528", "5d0cb16db2d9e72ce9f27d3764a375a9", 250436),
	ENTRY0("cenricfamilycurse", "131022", "f0496ff6c8bc01931f6034373fbf1d44", 281730),
	ENTRY0("cenricfamilycurse", "131218", "4fa3db6430e3a54f88962a75857ebb2f", 289096),
	ENTRY0("tcomremake", "121103", "a36310280c2a70393dfa8750ce31c3dd", 1174070),
	ENTRY0("tcomremake", "121103", "2fd621df080f2b8c5fbe78ca99dc00c4", 176640),
	ENTRY0("tcomremake", "121103", "aa9995de3edd044db0f10fee2ba0f3ba", 256500),
	ENTRY0("house", "070714", "4b146b3d5b81a3b3359ed3f3059910ab", 468830),
	ENTRY0("advhoudini", "120511", "51982015d7f804d5bf5a25e2906b281a", 579228),
	ENTRY0("theatre", "951203", "8b3db2f9039696ffa0114d55ac219ab9", 185856),
	ENTRY0("dayishothitler", "081206", "2bb145727b016ab76c35acfae729293e", 175616),
	ENTRY0("doghouse", "101130", "6476e788310c44c52d5249a66d07d037", 712070),
	ENTRY0("emptyroom", "100813", "9b3b2c5ae701ef457baaf3db6e2aa06e", 320960),
	ENTRY0("forgottengirls", "120817", "421e83ead89a7a0f2fd05791f9b87db7", 410034),
	ENTRY0("henribeauchamp", "080621", "d9088b3260fdaf30fa81cde478472eca", 179712),
	ENTRY0("garliccage", "120209", "04a598fea9264a290e2c34f0b63f8042", 176640),
	ENTRY0("grandquest", "091209", "3fd8b15a0282e9eb9ef642776bbd916e", 181760),
	ENTRY0("thegreat", "010902", "285ad766a386886e814e6c22ed990a7f", 91648),
	ENTRY0("greatpancake", "180117", "9f4e359b40bc53631937c2c44ee5082c", 264268),
	ENTRY0("horriblepyramid", "131201", "b3dfa0f543f7ca3343ca62cc5ec7847f", 365004),
	ENTRY0("houseoffear", "141121", "86812b0b8866038dcce67ba265e5bd5e", 572530),
	ENTRY0("island", "130226", "6ee80981b5921ad6f159ed740b54dd1a", 354882),
	ENTRY0("kazooist", "130105", "9ae3841d39c8b6f8dffa9c84408d9744", 204308),
	ENTRY0("priceoffreedom", "120919", "589c7d77bae5aaa502807021e506fb6b", 178072),
	ENTRY0("prize", "120827", "9dc341cb340b6950602c52bd8ef76305", 408786),
	ENTRY0("nascarexperience", "130827", "d45dc3ab1f9898cf85a09854db0a6b17", 152576),
	ENTRY0("smallroom", "100805", "5102ac48e17b26b69ab8737ceb00b3d5", 154112),
	ENTRY0("spotlight", "140531", "ac25066421da9530f5f19c7245d0f6d9", 336384),
	ENTRY0("stargods", "111216", "92a0b939e5d12178f74b3e23ece89ca1", 268288),
	ENTRY0("terribleoldmanse", "100819", "518a610437203e063fa550d5c240e664", 891570),
	ENTRY0("townmusicians", "120125", "35624466eb61b2dcbc408cd6c75a6ab9", 414042),
	ENTRY0("warblersnest", "131221", "147d80b524cf8ee812459c0a7426cb87", 863268),
	ENTRY0("vergingpaths", "151027", "c6df1e824df593e8c4995502e6704571", 1131672),
	ENTRY0("minimalistgame", "101102", "4ea052eed3e86283912bff5e817151fb", 151040),
	ENTRY0("talemorning", "140109", "234a6da218d56ca47410f7e03c2b89a8", 286756),
	ENTRY0("paperbagprincess", "150818", "fdf4a244b41e4a314cfa189ba85453cb", 398848),
	ENTRY0("worldupsidedown", "151224", "1b7311638555848aaf3a50857ed4035b", 457496),
	ENTRY0("thorn", "030701", "63faf28ec7cad962816e9ed3a7310a74", 75776),
	ENTRY0("threecows", "120208", "449fd20d3b0981ba6f8a7d929e56c820", 153088),
	ENTRY0("threediopolis", "140817", "76929b10a1fddcbba4e704b9d7fecdc0", 584032),
	ENTRY0("threemore", "120110", "d89d23f3cacd8541df1b1792362b206b", 155136),
	ENTRY0("timefortea", "100617", "af469380d1ace75480a80f578091ac4f", 313344),
	ENTRY0("tirehoax", "080104", "09b696f73a1d2d37ee376fac97a2c406", 228746),
	ENTRY0("tk1", "950925", "7c93a305295c891ba9de5cad4c190f8c", 52224),
	ENTRY0("tkatc", "151121", "cecca5aa05f7ea35550b473b90c3766e", 408576),
	ENTRY0("tok", "100227", "1b3b24c4616bf7e47e99eacca7308153", 241590),
	ENTRY0("downtowntokyo", "000615", "8d9ef45f2bc4fdb8eccf7d9991a2c76b", 107520),
	ENTRY0("tower", "151228", "04de5c49bb3eecb4e4d1cb0fd5ea4a93", 391156),
	ENTRY0("toxinx", "110519", "2223181742216f1c1bb74bca88dda729", 200704),
	ENTRY0("trapped", "110413", "b05f8909d1bbedb97f6c50eac19d6b2a", 254362),
	ENTRY0("toask", "130614", "1491b55bbcb2a0e7b1bfe99b94a9d387", 524288),
	ENTRY0("troll", "980518", "3e0f69e678dd289cd32bf41b2be58bd6", 64000),
	ENTRY0("trw", "021229", "1fdb2baeefcabb635ddbbb3433b1b125", 126464),
	ENTRY0("trystoffate", "970629", "7f7eb7e7df03025b9046139d042dd363", 165888),
	ENTRY0("tubetrouble", "950901", "478a208e21def77097309c2b152a968e", 50176),
	ENTRY0("tutorial", "111030", "36147ba605eb49902f9d514a08638cd1", 468142),
	ENTRY0("tutorialhotel", "100713", "2d57f52f53fa0845eb19ea97a29005fa", 154624),
	ENTRY0("tuuli", "180501", "0b7e2bb3f6887f30fb98b54230acf039", 611734),
	ENTRY0("typo", "041119", "549875ea83b4230896fa7d3fa18ba0a3", 100864),
	ENTRY0("uhohdemo", "980218", "4e8b90354c96760f36bfbfa2ba4fc04c", 176640),
	ENTRY0("underdoos", "970329", "3ef9b348b4223b901bdfacb854fee16e", 58880),
	ENTRY0("underthebed", "120522", "96659568d70385c022a9ecc41fd6219c", 246272),
	ENTRY0("ungodlyhour", "140927", "d91b088567ff78a3caaa863cfc6db7b0", 333874),
	ENTRY0("uninvited", "121216", "2f364062b6d1f05c8b518a11d58c399b", 193536),
	ENTRY0("uninvited", "121216", "c2b16596310202bf40c512c5de571858", 623410),
	ENTRY0("unicornpool", "100914", "8bf72169c75e347b52aa592b544e1357", 111616),
	ENTRY0("vacation", "060527", "0a0da195fa5c41a59028e69eeae2d9e0", 248320),
	ENTRY0("vagueness", "080825", "6d3641b71d42516e573ff0cff90cd40e", 298402),
	ENTRY0("vampiresun", "020405", "3364b9d9d1a754e1e3bab61a96957ed2", 254464),
	ENTRY0("notinvenice", "120117", "191d800215e21a764394ec46efe9a728", 194560),
	ENTRY0("vespers", "051128", "35377bb25ea1f3038377c8fb51795d6c", 309760),
	ENTRY0("varicella", "990831", "f5791cd7d8ebfd568928eb2b888a5264", 501760),
	ENTRY0("vigilance", "070106", "97364e2e3f4197bf9dcfeef8fa6e6ee9", 183296),
	ENTRY0("vindaloo", "960613", "d4e69f29f435b55dff5057a1d0d5bd45", 53248),
	ENTRY0("violet", "081123", "0941c8db96a7ef82ebf10fe6cdd1859e", 601262),
	ENTRY0("virtuality", "100131", "a68fe31bf3cce9b3dea7da0d733f45b0", 397244),
	ENTRY0("lackofvision", "980412", "25a2830653906a2225742505e1fd7cac", 47616),
	ENTRY0("visitor", "000321", "9786e58864fdd7429c3218b07d023498", 56832),
	ENTRY0("vosr", "060925", "73746a1dd6938ca4b034eb424936ffe7", 267776),
	ENTRY0("wadewar3", "020512", "458b5b368a71329dcfc59223c449cf88", 143872),
	ENTRY0("walkinthepark", "100213", "d116d7a9a5160fc336b36f7e8a077a42", 145408),
	ENTRY0("walkamongus", "181031", "5b6b726c28b897d299d4add5fcd3943e", 2292016),
	ENTRY0("wallpaper", "061117", "b25d80417e7b783bdc377302a3b53138", 344978),
	ENTRY0("warp", "640101", "3554d76096e5d06d9417e20a21e95ea0", 70144),
	ENTRY0("wizardscastle", "000918", "b205946471d687d83b8dfc988734ab43", 34304),
	ENTRY0("weapon", "010706", "b28a2d2ab4a2ba54e5c5d980764c26a4", 142336),
	ENTRY0("weather", "960613", "00f90d5b28604243708ad41cc6a7dcea", 93696),
	ENTRY0("thewedding", "970602", "40b0b13d420f894ebac54106f0e92ff8", 151552),
	ENTRY0("thewedding", "100221", "b3749fb3d3999331bcd4415969cb6602", 255940),
	ENTRY0("weirdcity", "140613", "0a434b5011389e47b391b707ccf6611a", 339786),
	ENTRY0("weirdness", "030922", "e98bc679d94c0c1c6a241737f7c8ae28", 89600),
	ENTRY0("wwwanderer", "080705", "a401b781048229b05b444bdbb68e5b71", 283592),
	ENTRY0("wernersquest1", "020225", "e158f13e9f55eef58a0f7a6affec180e", 52736),
	ENTRY0("wernersquest2", "020225", "6ebf8ffd5687674754473f4851df4c01", 52736),
	ENTRY0("wernersquest3", "020225", "690a16946cacccaf2d5fd3ca2cf9fc4a", 52736),
	ENTRY0("wernersquest4", "020225", "2098db329eff1b0d1cad56f89bd38723", 52736),
	ENTRY0("weareunfinished", "160419", "28d307b40b627af2708c71aaf43a64d5", 413298),
	ENTRY0("whispers", "050926", "360c24ec5c47ced2b03ce38c6e4e2e8d", 177152),
	ENTRY0("whitehouses", "140511", "bfa0206e9903d19551db483f810efa09", 423362),
	ENTRY0("wildflowers", "140825", "11df47a8f9ddebbe3ed903ef8c85597a", 518936),
	ENTRY0("winchester", "020131", "9abc235bc791a3f38ee81aa6496112d0", 293888),
	ENTRY0("windhall", "960828", "eced9e2a38a5b694ceab8ffa4c120b90", 320000),
	ENTRY0("winterwonderland", "030227", "0994f3c477e8e221d9c70df72fb936df", 194560),
	ENTRY0("wir1", "060503", "f9b1425b5c55f1b3f3caa50a7963be17", 652866),
	ENTRY0("wir2", "060503", "c64dfefbc7a320126342f6a522a3b8bd", 514520),
	ENTRY0("wireless", "040909", "2af916f6295f32533eadd97afccb11d8", 140800),
	ENTRY0("insidewoman", "090628", "c9505d9dc5badfd95a6be7b4b4edc2a8", 478720),
	ENTRY0("wossname", "000225", "ddf9652869c0892cc2d2150607dbf0c0", 77312),
	ENTRY0("wrenlaw", "130429", "24e41af3de0fd4052ef537d0e82b067c", 1835620),
	ENTRY0("wscholars", "071107", "03d438cfb4920cc831da090e0c150f2a", 587032),
	ENTRY0("wump2ka", "041119", "64e790c40f04ab7fe2405807f0cecb9f", 174080),
	ENTRY0("wumpus", "991216", "abdd37af526d03538cbb20d91a941489", 12800),
	ENTRY0("wurm", "021126", "0ffbc60fcccaccf1abfa877acf293b2a", 4096),
	ENTRY0("xenophobia", "111115", "d132c3b5defcef212f36e03a7c9a2e74", 124928),
	ENTRY0("yagwad", "001121", "6d74b0e48dc46e589efd071e3d329f45", 167936),
	ENTRY0("yakshaving", "101127", "3c4114a0999f2a68bd08c58fe0494a97", 394958),
	ENTRY0("yomomma", "100228", "325cd63060b5509300c71af58902670c", 1081664),
	ENTRY0("stewgoing", "150410", "93fdaa375c1db772b27918f096f43f53", 1604582),
	ENTRY0("zlife", "960121", "36bc7d2fdd0bbac996466e05af239924", 6656),
	ENTRY0("zassball", "980314", "ab906aa444b7fdd0a92119828616bb58", 12288),
	ENTRY0("zbefunge", "021128", "6ae16ce61922211922edf777bb6113ce", 56320),
	ENTRY0("zcamel", "000918", "96b316f9b7e133eaccbec98a4fb0cf46", 6656),
	ENTRY0("zchess", "040124", "456fe3c2d3d986e8652c4a439e738686", 6656),
	ENTRY0("zdungeon", "040826", "ddcf45ee10cc5f42ac1b273eb9a0d1f8", 188928),
	ENTRY0("zedfunge", "031111", "3198856622768685e17ffba6b8313e49", 108032),
	ENTRY0("zedit", "971103", "40a26af1581ebdbbd867b18229ce8187", 65024),
	ENTRY0("zegro", "041112", "bb7a0ef8d23cfb88afb2dc1e40be9613", 261632),
	ENTRY0("zenon", "100122", "2c82b81ac43fcfbf2abee4d8c4f98cad", 67584),
	ENTRY0("zenspeak", "990217", "53a344bef35a5778307aeb937a31f001", 123392),
	ENTRY0("zcatalog", "980519", "f5ad7533e0d33ced32429a04473ec3d0", 126976),
	ENTRY0("zokoban", "990810", "e599ec4ff1eee51afbd6a3f8e8d8680e", 18944),
	ENTRY0("zombies", "990524", "4535f316650ee6c76b95ced1100b12d0", 7680),
	ENTRY0("zorkianstories1", "121014", "dbf1900ac273a3051cc2bd405aef4620", 380454),
	ENTRY0("zorknplus9", "120517", "3d95fc9e6e60d372413bf10a90570b2e", 402874),
	ENTRY0("zracer", "071203", "04d60a3f69b68955b008cf7a7b08017f", 40960),
	ENTRY0("zsnake", "010201", "f6aca196d9a1de4f05df6af74327f8f8", 11264),
	ENTRY0("ztornado", "030711", "dd3edc31ee39d186566a6386c29aafe1", 20992),
	ENTRY0("ztrek", "000229", "99653ccefa7203a37e8f0ed71c88fa98", 29696),
	ENTRY0("ztuu", "970828", "ee5f33204264a1316c6acdb1036d19a9", 229888),
	ENTRY0("zugzwang", "990710", "e538e90a90a7c280a078b50d2ad2e44d", 58880),

	// Painless Little Stupid Games
	ENTRY0("plsg1", "110205", "1cb5c04a2373bbda0bb6abcc3d49ba84", 168960),
	ENTRY0("plsg2", "110210", "c3868744b56fe4b3b8e8e5c1eac80864", 181760),
	ENTRY0("plsg3", "110310", "a640d77b82894fde6b6b3d3cc89553f7", 180224),
	ENTRY0("plsg4", "110414", "7eda62012f870848603ba669d3a3d305", 198144),
	ENTRY0("plsg5", "110420", "f5c7139c859b8f5209115b79479314e2", 181248),
	ENTRY0("plsg6", "110527", "e972666101acf79534d71686fa68f369", 185856),
	ENTRY0("plsg7", "110630", "dd0a2e670a56cd937671b94394bc0fbb", 182784),
	ENTRY0("plsg8", "110822", "24d0f12ead2634292dc07b634d65059a", 164864),
	ENTRY0("plsg9", "110901", "32cddc8302fb9b566c83d4c8d5d928be", 206336),
	ENTRY0("plsg10", "111118", "30705c933686364f9086a36aac9579df", 260096),

	// Converted Scott Adams games
	ENTRY0("adventureland", "941017", "cde66d37efaff46f18e67b0f39e4d0cd", 43008),
	ENTRY0("scottsampler", "980329", "86c473c81e86637105108afa943c3ced", 23040),
	ENTRY0("buckaroobanzai", "980329", "cdbf748cdcee3cade378cf62cfe01d43", 24064),
	ENTRY0("thecount", "980329", "a0060ef9c9fa5cc3d3dbbc060f6451c2", 25088),
	ENTRY0("mysteryfunhouse", "980329", "4b78c1883356db8184b351c5a269fdce", 24064),
	ENTRY0("ghosttown", "980329", "0240f4119bb9b8e8197f37049c9b4f82", 25600),
	ENTRY0("goldenvoyage", "980329", "d986f2ac673abdce741c90e8b9fc3acf", 26112),
	ENTRY0("marveladventure", "HULK-980329", "964dfa22fcd54d2674123951af79136b", 25600),
	ENTRY0("strangeodyssey", "980329", "8216fc5ca7ed593d6a9c4265064d83a4", 24576),
	ENTRY0("pirateadventure", "980329", "0c3d27eaa6563835bfb1aadd309e7a00", 24064),
	ENTRY0("pyramidofdoom", "980329", "a6fc7fd81b7330bc254afbac17b29058", 26112),
	ENTRY0("adventure14", "980329", "56e5d7c33d5403ed59a62f67744f4d02", 26624),
	ENTRY0("savageisland1", "980329", "6f6cf307a97becb32524fe66a910587e", 24576),
	ENTRY0("savageisland2", "980329", "a43ab1063e6a8d4849ad3b69f1e4cacb", 25600),
	ENTRY0("missionimpossible", "980329", "9c759b65e43e2d9d6aa02122248040ae", 24064),
	ENTRY0("adventure13", "980329", "54cc89bbead7dac21455b9c00f32f604", 25088),
	ENTRY0("questprobe2", "980329", "6b9cb86c332c092b3a93973ba9f4c946", 27136),
	ENTRY0("voodoocastle", "980329", "bff285e6c9291fc6ba77c9743b610c2d", 24064),

	// Mysterious Adventures by Brian Howarth
	ENTRY0("goldenbatoni5", "980406", "408b31a15c429f7ca83b2ac80764ffa8", 20992),
	ENTRY0("timemachinei5", "980406", "e0a0335705aab9642b7625f26c00eca2", 22016),
	ENTRY0("arrowofdeath1i5", "980406", "a3827232bf54c339a5ec5ab906fd1857", 22016),
	ENTRY0("arrowofdeath2i5", "980406", "5a437f2cbc4f99d8cd741e83e2abe4cd", 24064),
	ENTRY0("pulsar7i5", "980406", "d93cc91cda58c75259b2c872921a17a8", 26112),
	ENTRY0("circusi5", "980406", "be7a07e042f5d1b0cde3d3b1cd85dee3", 22528),
	ENTRY0("feasibilityi5", "980406", "e0c46523e043bc75f8e04714396e17ff", 22528),
	ENTRY0("akyrzi5", "980406", "61c29077bee55dce614a729705099282", 24064),
	ENTRY0("perseusi5", "980406", "c1ab2f87658691f773599d9973bf72a1", 23040),
	ENTRY0("10indiansi5", "980406", "b3e4b8376f7c553064ceff8f25936385", 22528),
	ENTRY0("waxworksi5", "980406", "98e52d813cb28f899916ef7129c85a0e", 24064),
	ENTRY0("goldenbatoni5", "041209", "817ca85193d842b9716d4b688d6fe9d1", 27644),
	ENTRY0("timemachinei5", "041209", "460ad097aeb7b800f237692aaec8fda2", 28156),
	ENTRY0("arrowofdeath1i5", "041209", "616f481469279a1184d1d8fcad84ed4e", 28156),
	ENTRY0("arrowofdeath2i5", "041209", "058dbdf618b22e9dd47f42f7e98e6fdd", 30716),
	ENTRY0("pulsar7i5", "041209", "41a2b9048af4600f43c829e2348b5fc6", 32764),
	ENTRY0("circusi5", "041209", "0c5a65e665b773fc39bdcbe194ad99cc", 29180),
	ENTRY0("feasibilityi5", "041209", "2159059c9b506af4f10c7cf9133fdd00", 28672),
	ENTRY0("akyrzi5", "041209", "cfbef40e735057b6fbbde3991f6ee4c9", 31232),
	ENTRY0("perseusi5", "041209", "1c0fef44034daa16ada548caac232337", 29696),
	ENTRY0("10indiansi5", "041209", "63b01fa007b977be144bcd3a6f6e8dcf", 29180),
	ENTRY0("waxworksi5", "041209", "859a006a14bd69b22135688248756ba0", 30720),
	ENTRY0("goldenbatoni5", "110126", "9de3f1a8624e20409c92325b30a3b490", 156156),
	ENTRY0("timemachinei5", "110126", "684e96c6adaccfd5f4138dce069d3fc3", 137728),
	ENTRY0("arrowofdeath1i5", "110126", "3deca9c9fce4fb995e0681ecdfb39cf2", 173056),
	ENTRY0("arrowofdeath2i5", "110126", "4a0c494ff4564e659a29fbd3d67696f6", 172540),
	ENTRY0("pulsar7i5", "110126", "5d6897ee80078c0286a52589ae305633", 135168),
	ENTRY0("circusi5", "110126", "72721053ed49e1d701146332215f63e9", 118780),
	ENTRY0("feasibilityi5", "110126", "8f2f18c6fd76be74e612ee9b271055b1", 172540),
	ENTRY0("akyrzi5", "110126", "92c5abee1a097f1e8e61f744ba9ddb3f", 119296),
	ENTRY0("perseusi5", "110126", "39be8e28753aa0ac87c49b6dedb712a2", 132096),
	ENTRY0("10indiansi5", "110126", "0c5d25323a3b649ea432025001edb638", 170492),
	ENTRY0("waxworksi5", "110126", "c51e911d1228d8adfc07ed138bc90079", 116220),

	// 1992 album Apollo 18, by They Might be Giants
	ENTRY0("apollo1", "120315", "1aec299147675a5c6e10d548a4eeba05", 251100),
	ENTRY0("apollo4", "120322", "83f92f1fc39bd699ac32935d8eefd35a", 426528),
	ENTRY0("apollo5", "120324", "6c889ec9c330169740afafe7694030cf", 199680),
	ENTRY0("apollo6", "120316", "a13d84152dc3b9ec5e6b489d987e8130", 906236),
	ENTRY0("apollo7", "120324", "a08c999b6408a9c17e2f3ab55a82e919", 363458),
	ENTRY0("apollo8", "120325", "1b0e0c32915e81873bc30f900bfdf371", 352706),
	ENTRY0("apollo9", "120323", "e48c6dad0b11bde066c4e29da68f71ef", 448972),
	ENTRY0("apollo11", "120304", "ad4b4d5a6d2c0c251a891cca5d47dfcd", 262406),
	ENTRY0("apollo13", "120323", "b45ad2f48f3da1bbe8650a2df70ba625", 350570),
	ENTRY0("apollo14", "120324", "dedd30cae688dc8a88bd79d8004ee6e0", 278016),
	ENTRY0("apollo16", "120324", "e7cc8f53c82196ef2281e1345c5048df", 155648),
	ENTRY0("apollo17", "120324", "cd6aef7581d5c7221d2b95b55d4d3961", 264822),
	ENTRY0("apollo18", "120324", "604b89f3066da9a62922ff2c45a669ea", 327008),
	ENTRY0("apollo20", "120324", "1cca9cd17cd354131ae0888307ec1ec2", 200192),
	ENTRY0("apollo21", "120324", "6d6fc49e004828df9ca9fa2013a0a231", 199680),
	ENTRY0("apollo22", "120323", "7284829f0999d3cfa9de81a97bc6f2e2", 385536),
	ENTRY0("apollo23", "120324", "f7a7021baaa261e70d284e2c97e54115", 345088),
	ENTRY0("apollo24", "120324", "c9bbd95fdfbe3fda922cfba0f57d201c", 310000),
	ENTRY0("apollo25", "120322", "2ae649402f52358473f1a61bd8f0cfd4", 369664),
	ENTRY0("apollo26", "120325", "1327ebd3016d873961f6ee35408a03fb", 251354),
	ENTRY0("apollo27", "120324", "1cc4273e5417578445a6b528dd3cdff7", 208896),
	ENTRY0("apollo28", "120324", "7c655dde263ec0e75289e4a77d2b53c3", 173056),
	ENTRY0("apollo29", "120304", "f4d87b9a51126d095a2af8de5bb6ad04", 494298),
	ENTRY0("apollo30", "120324", "53f36fb7ff374a59f77679cf3a5d1bd5", 207872),
	ENTRY0("apollo31", "120314", "fb8cad57d9305ffbfc2dd69e6406a0fd", 197120),
	ENTRY0("apollo32", "120323", "8ec4ee91cd1186c4deef126b40b78cfc", 151552),
	ENTRY0("apollo34", "120325", "78f0a5bfce8eb2801350793738a2355f", 281600),
	ENTRY0("apollo35", "120324", "bcce4073de352739be245f21685ef841", 336864),
	ENTRY0("apollo36", "120324", "01f6f3552fff6a8eaaf5ad31c4c57f64", 209408),
	ENTRY0("apollo37", "120323", "55b8607a451ab30d03314f815a30e40b", 327144),
	ENTRY0("apollo38", "120321", "a760169a6643d08ff3db22be84caed37", 353734),

	// Danish games
	ENTRY1("nissen", "171207", "c81784afb569b863098e8374dfdd4f32", 142848, Common::DA_DAN),

	// Dutch games
	ENTRY1("adv", "020822", "dff45a0d89ed78a204d49d18341d15ce", 153600, Common::NL_NLD),

	// French games
	ENTRY1("adv", "000531", "50027e3339900e1ca07f72e96396240e", 129536, Common::FR_FRA),
	ENTRY1("basketball", "180904", "92b21e4bc9163727c3dd586056f26fd8", 64000, Common::FR_FRA),
	ENTRY1("dreamlands", "050908", "79cecc22e3f020a3ccc23912122785d4", 79872, Common::FR_FRA),
	ENTRY1("espions", "070417", "7e59a5199b12376697deb4694cfac744", 124416, Common::FR_FRA),
	ENTRY1("filaments", "030301", "0aa6d27086b546f6bd4dea84717da6ff", 337920, Common::FR_FRA),
	ENTRY1("initiation", "070820", "3b50132dd7007bfe1b7cbc7147df37ee", 91136, Common::FR_FRA),
	ENTRY1("kheper", "070829", "3612cef78d31e80632d6b69b48652306", 94208, Common::FR_FRA),
	ENTRY1("verdeterre", "140228", "6665bda7c172285e281de2e120dac250", 1385176, Common::FR_FRA),
	ENTRY1("lieuxcommuns", "070910", "d1032dc1a9635358e0aec29038fdb34e", 253952, Common::FR_FRA),
	ENTRY1("lmpsd", "040921", "bb90b4548c5160b10a7b6cbfdb5384a9", 215040, Common::FR_FRA),
	ENTRY1("ombre", "150820", "a3df02e2ecda768542f326480709db02", 137216, Common::FR_FRA),
	ENTRY1("princesse", "131028", "16cdf0b604c538e12de37b4c1e806e79", 124928, Common::FR_FRA),
	ENTRY1("sarvegne", "061101", "8c308818d08d924fdf69250fd5fdb210", 224768, Common::FR_FRA),
	ENTRY1("katana", "070906", "133b2ba4e38fc5e4565dd8f22582b5a7", 562502, Common::FR_FRA),
	ENTRY1("sdlc", "070917", "c80e6bf2a43340c87272f16bf3e64397", 67584, Common::FR_FRA),
	ENTRY1("balcon", "070823", "6072d8d30e4e1546e72e851bd724bf78", 173440, Common::FR_FRA),
	ENTRY1("templedefeu", "070731", "05b8c729867644a2b6c417297dcabea3", 131072, Common::FR_FRA),

	// German games
	ENTRY1("adv", "980419", "6a04a6b303f09359155eb23baa201efc", 126464, Common::DE_DEU),
	ENTRY1("bearg", "000326", "b85bc696a58e11f6a0cf72f4cf08ad7c", 134656, Common::DE_DEU),
	ENTRY1("bewerbung", "020429", "5b2a90b66bfcf4564b37dab92afe846a", 114688, Common::DE_DEU),
	ENTRY1("deklinator", "040630", "1ca69e16e8e3e1941eb5c4cd403b2022", 103424, Common::DE_DEU),
	ENTRY1("edendemo", "020401", "73a6cf485aa833ca895b0f860493cfce", 221618, Common::DE_DEU),
	ENTRY1("halb2", "020523", "d424290a37b5efc67482e4fc9224bc0e", 30208, Common::DE_DEU),
	ENTRY1("herr", "140620", "a6493768373d50096344f4319e7f074a", 208028, Common::DE_DEU),
	ENTRY1("jazteg", "040522", "6635a44223e0017418acdeb0c78a9c7a", 192000, Common::DE_DEU),
	ENTRY1("karisma", "130908", "48d6eb07a649a614b037722af9718bf2", 291328, Common::DE_DEU),
	ENTRY1("knack", "081215", "b8fa6a55c469cd92d01985933ac093a3", 64388, Common::DE_DEU),
	ENTRY1("o", "120412", "850ed6092daa2dcc87862404a2e88338", 240424, Common::DE_DEU),
	ENTRY1("starrider", "040110", "6c42a3e46e29419d9d0e3786117ab94b", 222208, Common::DE_DEU),
	ENTRY1("mchatton", "120413", "1485309be7aed1ed0ada396fe3eeb1ec", 290768, Common::DE_DEU),
	ENTRY1("wasserhasser", "140105", "1d332f660aec117a4460d0555a2b30f6", 165336, Common::DE_DEU),
	ENTRY1("wichtel", "021006", "f52166e02c6bd5e0311145683f415ef5", 88576, Common::DE_DEU),

	// Italian games
	ENTRY1("armando", "050429", "b74fb87510b6baa50307c2b63ba108de", 71168, Common::IT_ITA),
	ENTRY1("ayon", "130730", "da1a2dab91d8ccec4b69d955487e9539", 232960, Common::IT_ITA),
	ENTRY1("darkiss", "151120", "347db588b5663ab9be83048430d9f4a0", 156160, Common::IT_ITA),
	ENTRY1("darkiss2", "150616", "899316750483830be4f38caf37ba7d7f", 219136, Common::IT_ITA),
	ENTRY1("enigma", "111115", "6e9390f62c07038d44ab17167093c935", 111616, Common::IT_ITA),
	ENTRY1("filamit", "030812", "edde5a37d24b112fb1cf3ff7fb133579", 333312, Common::IT_ITA),
	ENTRY1("flamel", "010503", "02bb0beba5cf37e7a9b6fe3558e20cab", 198656, Common::IT_ITA),
	ENTRY1("giardino", "150614", "f3244aa61ce6b3a4dd860c315bd24aa7", 124416, Common::IT_ITA),
	ENTRY1("hellosword", "060113", "a432a8286f9f76dae891699175332237", 166400, Common::IT_ITA),
	ENTRY1("kangourou", "130503", "55449579a0ecd73fb788120dd0707bc0", 95744, Common::IT_ITA),
	ENTRY1("koohinoor", "060717", "867d2f12ae29cd97aadc1a3c6b2e2a2a", 88576, Common::IT_ITA),
	ENTRY1("luna", "110106", "28f1d106a70a0f50aee1d2628bc42055", 522076, Common::IT_ITA),
	ENTRY1("poesia", "398874", "feabf26f662e70f4d15ac290b8884e3a", 197632, Common::IT_ITA),
	ENTRY1("poesia", "398874", "c383d29e1237cfc4760b46a618f211d2", 196096, Common::IT_ITA),
	ENTRY1("oldwest1", "020514", "0182ca4b8153fc90e7be355d1ef51c7e", 157184, Common::IT_ITA),
	ENTRY1("rovo", "080424", "b2f570c667e32ea44c121b44cc6baeda", 113664, Common::IT_ITA),
	ENTRY1("scarafaggio", "070906", "6e980b7108fd4af148308214d6efc6bf", 131584, Common::IT_ITA),
	ENTRY1("sognodisangue", "150702", "d00baeb8f4cca59ba88b24b6e8b71246", 151552, Common::IT_ITA),
	ENTRY1("strega", "081007", "71f9cbdeb334cc1de6fce6e1a53cecf3", 153088, Common::IT_ITA),
	ENTRY1("tesla", "160613", "e2650593a2aadaef8fb9a5f3b8e10c27", 94720, Common::IT_ITA),
	ENTRY1("villamorgana", "081014", "6be5dee7a5d1608362083850c9bfc679", 170256, Common::IT_ITA),
	ENTRY1("zazie", "030113", "906b9a3e02b2080ce7f06595c8bdcbb3", 89600, Common::IT_ITA),
	ENTRY1("zenfactorspa", "100524", "22373bcd74d843ce647c4bd7b6a4404b", 288256, Common::IT_ITA),
	ENTRY1("zombie", "180820", "19a1369039b5226f86323ab19e7aea96", 135168, Common::IT_ITA),
	ENTRY1("Zorkita", "000031", "3d85a97ddfc1fb0f6bfbf1cb00b4df7b", 192512, Common::IT_ITA),

	// Russian games
	ENTRY0("allroads", "070701", "427693ae57580bbc57f204c50437d3cc", 198656),
	ENTRY0("bluechairs", "111222", "9d6af2460c034fa9078846180ec31f05", 338944),
	ENTRY0("metamorphoses", "080518", "bc16740cc10500b0d1e1e8e768da6417", 203776),
	ENTRY0("spiderandweb", "091122", "3ea174936bc6d56c3f4d0bd5fa1fe720", 321536),

	// Slovenian games
	ENTRY1("zenin", "070628", "bda2d35eb0614374d02bae623d3a22ec", 90112, Common::SK_SVK),

	// Spanish games
	ENTRY1("abalanzate", "001116", "d235b2e983f74f6176aea5b1d1418a78", 79872, Common::ES_ESP),
	ENTRY1("afuera", "010101", "9ed80d0b530f38cce7a7b2c0f1b6ccd9", 116224, Common::ES_ESP),
	ENTRY1("amanda", "091110", "11b63cb4c4ca11b86e835c1b00f9c5ae", 132096, Common::ES_ESP),
	ENTRY1("amanda", "091110", "c373f508436b06081cd76039dc17582e", 342504, Common::ES_ESP),
	ENTRY1("aque1", "000428", "5d16ddd8030635e10065b7e36ba5f59e", 86528, Common::ES_ESP),
	ENTRY1("adv", "971209", "2c38b40ffbc8c29fff29acbbefa317e8", 126976, Common::ES_ESP),
	ENTRY1("casi", "000311", "d9351b2b336b96f481bc42e222adc3e4", 97792, Common::ES_ESP),
	ENTRY1("Celos", "010403", "6f4dc34a02fe5eb872ffe99faa06fb79", 69632, Common::ES_ESP),
	ENTRY1("Cerillera", "101213", "6bd57d7dd178cc4d3ef895c147e65087", 1154350, Common::ES_ESP),
	ENTRY1("churro", "020116", "885c3ffa9a9aeb8518746d69211bff4b", 225792, Common::ES_ESP),
	ENTRY1("csa", "150201", "2efa23d92a10a64196ea6f01dea556c2", 72704, Common::ES_ESP),
	ENTRY1("cueva", "150301", "e0a6f6e6949944b7793f2822af687f2f", 74752, Common::ES_ESP),
	ENTRY1("ddddddcrj", "050616", "1b89e39bfdcf2ddd4675d8a8013746da", 66048, Common::ES_ESP),
	ENTRY1("despert", "980909", "f6c469e0931c9f18f149e1b6da484436", 129536, Common::ES_ESP),
	ENTRY1("draculasp2", "080819", "4de47380bf5d802f295a487eaf1499c6", 123392, Common::ES_ESP),
	ENTRY1("draculasp", "071227", "53865e944daea77afeaf9cb909cfe85a", 101888, Common::ES_ESP),
	ENTRY1("ascenso", "090409", "352b9bb39f2fff76b409025670169a98", 380434, Common::ES_ESP),
	ENTRY1("regalo", "100104", "8d7ea3a09f39d1d2de103e5117ad3224", 336064, Common::ES_ESP),
	ENTRY1("elultimohogar", "021225", "45edda9ec6eb400f409681d3f2b052d4", 122368, Common::ES_ESP),
	ENTRY1("ergotdelima", "170616", "0bcc4c1c8cc24165cb8f02f17f00682d", 606834, Common::ES_ESP),
	ENTRY1("forrajeo", "010101", "d86123253ae4b35570013dd87e48036a", 108032, Common::ES_ESP),
	ENTRY1("fotopia", "991220", "bb067cca7cd769c20e7bb5dc9ed09c65", 214016, Common::ES_ESP),
	ENTRY1("gorbag", "100104", "581e67f731d6b1d0d40bfc38cb531bf9", 199168, Common::ES_ESP),
	ENTRY1("gorron", "020726", "67a7a86523a72c85b9cc0a0cf730ee75", 80896, Common::ES_ESP),
	ENTRY1("goteras", "010102", "892f263e65c00fd92f6e384b2729acbe", 128000, Common::ES_ESP),
	ENTRY1("hhorcus", "100818", "092849be8d49f1ef509a88a1bc5bbea8", 118262, Common::ES_ESP),
	ENTRY1("i0", "000630", "d43fcdb06a748ea24f2329aba8c5761e", 218624, Common::ES_ESP),
	ENTRY1("islas", "050531", "3d7cee978d4f69e41e8af1a8ccda2b9d", 369152, Common::ES_ESP),
	ENTRY1("kerulen", "090720", "b1b024e24c78e37d424163c5b2a6f1ad", 105984, Common::ES_ESP),
	ENTRY1("konix", "080505", "c9a4128b4874ec735f77295f590a3108", 74240, Common::ES_ESP),
	ENTRY1("kunelar", "090409", "108dd4e7623634e6ff7ca976118dfa29", 220672, Common::ES_ESP),
	ENTRY1("lldc", "666777", "024e9465504e1ab7bda9399602102876", 164864, Common::ES_ESP),
	ENTRY1("megara", "091204", "6798f8acb0c0c60d1026a1e9a6469b55", 472716, Common::ES_ESP),
	ENTRY1("absolutos", "080516", "9505fa8fc4f4c2d06730ce6c33f0fd43", 270222, Common::ES_ESP),
	ENTRY1("macetas", "010102", "3e987e21093af65c8fce2e458ee3dafb", 234496, Common::ES_ESP),
	ENTRY1("lamansion", "010103", "0acd4655161f834b562b5560353877bd", 138752, Common::ES_ESP),
	ENTRY1("meeva", "021130", "4d988f4963d14f43cf61c44417be22ae", 135168, Common::ES_ESP),
	ENTRY1("megara", "091204", "94b7019ed62d257344ef39727011c250", 214016, Common::ES_ESP),
	ENTRY1("modusvivendi", "010101", "20795fdfe47bc6f6c519f3fe11e34c17", 234496, Common::ES_ESP),
	ENTRY1("mpdroidone", "170820", "ea6bf4230bf6f267f76e191d84fb9804", 353792, Common::ES_ESP),
	ENTRY1("navidad", "050110", "6fa3ccdcce7b0c45e608f2948a63646d", 102912, Common::ES_ESP),
	ENTRY1("necedad", "010925", "fc25f03b71e39ddca21e2c93607ac660", 101888, Common::ES_ESP),
	ENTRY1("culpatuya", "080723", "77cd0cbf40cd6badfa9edb9306108008", 198656, Common::ES_ESP),
	ENTRY1("culpatuya", "080723", "01ef491a2099414817f2a20d0ae78dd2", 536956, Common::ES_ESP),
	ENTRY1("ocaso", "020215", "017f1ee4813271b886347af89b4c917d", 164352, Common::ES_ESP),
	ENTRY1("oculta", "050214", "06340073888cd6850b9d176ed3b1a251", 190464, Common::ES_ESP),
	ENTRY1("olvido", "001230", "c7673107bf9909890480935b4d10ea24", 91648, Common::ES_ESP),
	ENTRY1("osito", "021002", "7ab4fdbf2a13da9742350cf9f89121e6", 128512, Common::ES_ESP),
	ENTRY1("panajo", "010102", "3969bf7b2dd00b342e6c0b4ec797919a", 124416, Common::ES_ESP),
	ENTRY1("pesadillavoraz", "080430", "79f8cb30a31146e08465acd9038ba7e6", 604046, Common::ES_ESP),
	ENTRY1("pincoya", "010102", "bafb201726dd331308cdf34ec1e478cb", 168960, Common::ES_ESP),
	ENTRY1("playera", "010101", "9866408f548c8606e521075907b33ca7", 80896, Common::ES_ESP),
	ENTRY1("protector", "010103", "3189852634dcc62258c4e3af727bac46", 154624, Common::ES_ESP),
	ENTRY1("reflejos", "010101", "7edc3b30022e97978ea93ef5c22edccd", 144384, Common::ES_ESP),
	ENTRY1("senten", "991225", "e578cb2626d969bba50d2ccd6d863ade", 90624, Common::ES_ESP),
	ENTRY1("sombras", "021127", "72b83812567f8a4c9cd523b6a09a9c65", 121856, Common::ES_ESP),
	ENTRY1("tokland", "001130", "7f5c796474c250f418a47fa9285e3116", 139776, Common::ES_ESP),
	ENTRY1("torre", "000208", "2a5bca50855883c01ce4e7e30c7bd444", 126464, Common::ES_ESP),
	ENTRY1("tuuli", "180501", "9382d5a2886dd7681203128a183ebae3", 671650, Common::ES_ESP),

	// Swedish games
	ENTRY1("aventyr", "071029", "fff0eb351b418ada4010d56c4298d6ac", 133632, Common::SE_SWE),
	ENTRY1("drakmagi", "080419", "12739044930fc939b0adf0efd5432713", 98304, Common::SE_SWE),
	ENTRY1("hotellet", "041212", "efb166d12edc19b19dd1d6e99d67800e", 109056, Common::SE_SWE),
	ENTRY1("pangnyheten", "040916", "2049135115dff02b2baf1b7bfb59606c", 185856, Common::SE_SWE),
	ENTRY1("storforsen", "041212", "5e3c06b6b0650f938d2831b2aac98153", 103936, Common::SE_SWE),
	ENTRY1("stuga", "090712", "58eef72aada351e7e059202ab00342c4", 588800, Common::SE_SWE),
	ENTRY1("vanyar", "030613", "abbc2b2fa0f5e922b45a3f8698ba9ad1", 110080, Common::SE_SWE),

	FROTZ_TABLE_END_MARKER
};

} // End of namespace Frotz
} // End of namespace Glk
