/* Copyright (c) 2017, Google Inc.
 *
 * Permission to use, copy, modify, and/or distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
 * SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION
 * OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE. */

#include <openssl/crypto.h>

#include <stdio.h>
#include <stdlib.h>

#include <openssl/aead.h>
#include <openssl/aes.h>
#include <openssl/bn.h>
#include <openssl/ctrdrbg.h>
#include <openssl/curve25519.h>
#include <openssl/dh.h>
#include <openssl/digest.h>
#include <openssl/ec.h>
#include <openssl/ecdsa.h>
#include <openssl/ec_key.h>
#include <openssl/evp.h>
#include <openssl/hkdf.h>
#include <openssl/hmac.h>
#include <openssl/kdf.h>
#include <openssl/nid.h>
#include <openssl/rsa.h>
#include <openssl/sha.h>

#include "../../internal.h"
#include "../curve25519/internal.h"
#include "../dh/internal.h"
#include "../ec/internal.h"
#include "../ecdsa/internal.h"
#include "../ml_kem/ml_kem.h"
#include "../rand/internal.h"
#include "../rsa/internal.h"

static void hexdump(const uint8_t *in, size_t len) {
  for (size_t i = 0; i < len; i++) {
    fprintf(stderr, "%02x", in[i]);
  }
}

static int check_test(const void *expected, const void *actual,
                      size_t expected_len, const char *name) {
  if (OPENSSL_memcmp(actual, expected, expected_len) != 0) {
    fprintf(stderr, "%s failed.\nExpected:   ", name);
    hexdump(expected, expected_len);
    fprintf(stderr, "\nCalculated: ");
    hexdump(actual, expected_len);
    fprintf(stderr, "\n");
    fflush(stderr);
    return 0;
  }
  return 1;
}

static int set_bignum(BIGNUM **out, const uint8_t *in, size_t len) {
  *out = BN_bin2bn(in, len, NULL);
  return *out != NULL;
}

static int serialize_ecdsa_sig(uint8_t *out, size_t out_len,
                               const ECDSA_SIG *sig) {
  if ((out_len & 1) ||  //
      !BN_bn2bin_padded(out, out_len / 2, sig->r) ||
      !BN_bn2bin_padded(out + out_len / 2, out_len / 2, sig->s)) {
    return 0;
  }
  return 1;
}

static ECDSA_SIG *parse_ecdsa_sig(const uint8_t *in, size_t in_len) {
  ECDSA_SIG *ret = ECDSA_SIG_new();
  if (!ret || //
      (in_len & 1) ||
      BN_bin2bn(in, in_len/2, ret->r) == NULL ||
      BN_bin2bn(in + in_len/2, in_len/2, ret->s) == NULL) {
    ECDSA_SIG_free(ret);
    ret = NULL;
  }
  return ret;
}

static RSA *self_test_rsa_key(void) {
  static const uint8_t kN[] = {
      0xd3, 0x3a, 0x62, 0x9f, 0x07, 0x77, 0xb0, 0x18, 0xf3, 0xff, 0xfe, 0xcc,
      0xc9, 0xa2, 0xc2, 0x3a, 0xa6, 0x1d, 0xd8, 0xf0, 0x26, 0x5b, 0x38, 0x90,
      0x17, 0x48, 0x15, 0xce, 0x21, 0xcd, 0xd6, 0x62, 0x99, 0xe2, 0xd7, 0xda,
      0x40, 0x80, 0x3c, 0xad, 0x18, 0xb7, 0x26, 0xe9, 0x30, 0x8a, 0x23, 0x3f,
      0x68, 0x9a, 0x9c, 0x31, 0x34, 0x91, 0x99, 0x06, 0x11, 0x36, 0xb2, 0x9e,
      0x3a, 0xd0, 0xbc, 0xb9, 0x93, 0x4e, 0xb8, 0x72, 0xa1, 0x9f, 0xb6, 0x8c,
      0xd5, 0x17, 0x1f, 0x7e, 0xaa, 0x75, 0xbb, 0xdf, 0xa1, 0x70, 0x48, 0xc4,
      0xec, 0x9a, 0x51, 0xed, 0x41, 0xc9, 0x74, 0xc0, 0x3e, 0x1e, 0x85, 0x2f,
      0xbe, 0x34, 0xc7, 0x65, 0x34, 0x8b, 0x4d, 0x55, 0x4b, 0xe1, 0x45, 0x54,
      0x0d, 0x75, 0x7e, 0x89, 0x4d, 0x0c, 0xf6, 0x33, 0xe5, 0xfc, 0xfb, 0x56,
      0x1b, 0xf2, 0x39, 0x9d, 0xe0, 0xff, 0x55, 0xcf, 0x02, 0x05, 0xb9, 0x74,
      0xd2, 0x91, 0xfc, 0x87, 0xe1, 0xbb, 0x97, 0x2a, 0xe4, 0xdd, 0x20, 0xc0,
      0x38, 0x47, 0xc0, 0x76, 0x3f, 0xa1, 0x9b, 0x5c, 0x20, 0xff, 0xff, 0xc7,
      0x49, 0x3b, 0x4c, 0xaf, 0x99, 0xa6, 0x3e, 0x82, 0x5c, 0x58, 0x27, 0xce,
      0x01, 0x03, 0xc3, 0x16, 0x35, 0x20, 0xe9, 0xf0, 0x15, 0x7a, 0x41, 0xd5,
      0x1f, 0x52, 0xea, 0xdf, 0xad, 0x4c, 0xbb, 0x0d, 0xcb, 0x04, 0x91, 0xb0,
      0x95, 0xa8, 0xce, 0x25, 0xfd, 0xd2, 0x62, 0x47, 0x77, 0xee, 0x13, 0xf1,
      0x48, 0x72, 0x9e, 0xd9, 0x2d, 0xe6, 0x5f, 0xa4, 0xc6, 0x9e, 0x5a, 0xb2,
      0xc6, 0xa2, 0xf7, 0x0a, 0x16, 0x17, 0xae, 0x6b, 0x1c, 0x30, 0x7c, 0x63,
      0x08, 0x83, 0xe7, 0x43, 0xec, 0x54, 0x5e, 0x2c, 0x08, 0x0b, 0x5e, 0x46,
      0xa7, 0x10, 0x93, 0x43, 0x53, 0x4e, 0xe3, 0x16, 0x73, 0x55, 0xce, 0xf2,
      0x94, 0xc0, 0xbe, 0xb3,
  };
  static const uint8_t kE[] = {0x01, 0x00, 0x01};  // 65537
  static const uint8_t kD[] = {
      0x2f, 0x2c, 0x1e, 0xd2, 0x3d, 0x2c, 0xb1, 0x9b, 0x21, 0x02, 0xce, 0xb8,
      0x95, 0x5f, 0x4f, 0xd9, 0x21, 0x38, 0x11, 0x36, 0xb0, 0x9a, 0x36, 0xab,
      0x97, 0x47, 0x75, 0xf7, 0x2e, 0xfd, 0x75, 0x1f, 0x58, 0x16, 0x9c, 0xf6,
      0x14, 0xe9, 0x8e, 0xa3, 0x69, 0x9d, 0x9d, 0x86, 0xfe, 0x5c, 0x1b, 0x3b,
      0x11, 0xf5, 0x55, 0x64, 0x77, 0xc4, 0xfc, 0x53, 0xaa, 0x8c, 0x78, 0x9f,
      0x75, 0xab, 0x20, 0x3a, 0xa1, 0x77, 0x37, 0x22, 0x02, 0x8e, 0x54, 0x8a,
      0x67, 0x1c, 0x5e, 0xe0, 0x3e, 0xd9, 0x44, 0x37, 0xd1, 0x29, 0xee, 0x56,
      0x6c, 0x30, 0x9a, 0x93, 0x4d, 0xd9, 0xdb, 0xc5, 0x03, 0x1a, 0x75, 0xcc,
      0x0f, 0xc2, 0x61, 0xb5, 0x6c, 0x62, 0x9f, 0xc6, 0xa8, 0xc7, 0x8a, 0x60,
      0x17, 0x11, 0x62, 0x4c, 0xef, 0x74, 0x31, 0x97, 0xad, 0x89, 0x2d, 0xe8,
      0x31, 0x1d, 0x8b, 0x58, 0x82, 0xe3, 0x03, 0x1a, 0x6b, 0xdf, 0x3f, 0x3e,
      0xa4, 0x27, 0x19, 0xef, 0x46, 0x7a, 0x90, 0xdf, 0xa7, 0xe7, 0xc9, 0x66,
      0xab, 0x41, 0x1d, 0x65, 0x78, 0x1c, 0x18, 0x40, 0x5c, 0xd6, 0x87, 0xb5,
      0xea, 0x29, 0x44, 0xb3, 0xf5, 0xb3, 0xd2, 0x4f, 0xce, 0x88, 0x78, 0x49,
      0x27, 0x4e, 0x0b, 0x30, 0x85, 0xfb, 0x73, 0xfd, 0x8b, 0x32, 0x15, 0xee,
      0x1f, 0xc9, 0x0e, 0x89, 0xb9, 0x43, 0x2f, 0xe9, 0x60, 0x8d, 0xda, 0xae,
      0x2b, 0x30, 0x99, 0xee, 0x88, 0x81, 0x20, 0x7b, 0x4a, 0xc3, 0x18, 0xf2,
      0x94, 0x02, 0x79, 0x94, 0xaa, 0x65, 0xd9, 0x1b, 0x45, 0x2a, 0xac, 0x6e,
      0x30, 0x48, 0x57, 0xea, 0xbe, 0x79, 0x7d, 0xfc, 0x67, 0xaa, 0x47, 0xc0,
      0xf7, 0x52, 0xfd, 0x0b, 0x63, 0x4e, 0x3d, 0x2e, 0xcc, 0x36, 0xa0, 0xdb,
      0x92, 0x0b, 0xa9, 0x1b, 0xeb, 0xc2, 0xd5, 0x08, 0xd3, 0x85, 0x87, 0xf8,
      0x5d, 0x1a, 0xf6, 0xc1,
  };
  static const uint8_t kP[] = {
      0xf7, 0x06, 0xa3, 0x98, 0x8a, 0x52, 0xf8, 0x63, 0x68, 0x27, 0x4f, 0x68,
      0x7f, 0x34, 0xec, 0x8e, 0x5d, 0xf8, 0x30, 0x92, 0xb3, 0x62, 0x4c, 0xeb,
      0xdb, 0x19, 0x6b, 0x09, 0xc5, 0xa3, 0xf0, 0xbb, 0xff, 0x0f, 0xc2, 0xd4,
      0x9b, 0xc9, 0x54, 0x4f, 0xb9, 0xf9, 0xe1, 0x4c, 0xf0, 0xe3, 0x4c, 0x90,
      0xda, 0x7a, 0x01, 0xc2, 0x9f, 0xc4, 0xc8, 0x8e, 0xb1, 0x1e, 0x93, 0x75,
      0x75, 0xc6, 0x13, 0x25, 0xc3, 0xee, 0x3b, 0xcc, 0xb8, 0x72, 0x6c, 0x49,
      0xb0, 0x09, 0xfb, 0xab, 0x44, 0xeb, 0x4d, 0x40, 0xf0, 0x61, 0x6b, 0xe5,
      0xe6, 0xfe, 0x3e, 0x0a, 0x77, 0x26, 0x39, 0x76, 0x3d, 0x4c, 0x3e, 0x9b,
      0x5b, 0xc0, 0xaf, 0xa2, 0x58, 0x76, 0xb0, 0xe9, 0xda, 0x7f, 0x0e, 0x78,
      0xc9, 0x76, 0x49, 0x5c, 0xfa, 0xb3, 0xb0, 0x15, 0x4b, 0x41, 0xc7, 0x27,
      0xa4, 0x75, 0x28, 0x5c, 0x30, 0x69, 0x50, 0x29,
  };
  static const uint8_t kQ[] = {
      0xda, 0xe6, 0xd2, 0xbb, 0x44, 0xff, 0x4f, 0xdf, 0x57, 0xc1, 0x11, 0xa3,
      0x51, 0xba, 0x17, 0x89, 0x4c, 0x01, 0xc0, 0x0c, 0x97, 0x34, 0x50, 0xcf,
      0x32, 0x1e, 0xc0, 0xbd, 0x7b, 0x35, 0xb5, 0x6a, 0x26, 0xcc, 0xea, 0x4c,
      0x8e, 0x87, 0x4a, 0x67, 0x8b, 0xd3, 0xe5, 0x4f, 0x3a, 0x60, 0x48, 0x59,
      0x04, 0x93, 0x39, 0xd7, 0x7c, 0xfb, 0x19, 0x1a, 0x34, 0xd5, 0xe8, 0xaf,
      0xe7, 0x22, 0x2c, 0x0d, 0xc2, 0x91, 0x69, 0xb6, 0xe9, 0x2a, 0xe9, 0x1c,
      0x4c, 0x6e, 0x8f, 0x40, 0xf5, 0xa8, 0x3e, 0x82, 0x69, 0x69, 0xbe, 0x9f,
      0x7d, 0x5c, 0x7f, 0x92, 0x78, 0x17, 0xa3, 0x6d, 0x41, 0x2d, 0x72, 0xed,
      0x3f, 0x71, 0xfa, 0x97, 0xb4, 0x63, 0xe4, 0x4f, 0xd9, 0x46, 0x03, 0xfb,
      0x00, 0xeb, 0x30, 0x70, 0xb9, 0x51, 0xd9, 0x0a, 0xd2, 0xf8, 0x50, 0xd4,
      0xfb, 0x43, 0x84, 0xf8, 0xac, 0x58, 0xc3, 0x7b,
  };
  static const uint8_t kDModPMinusOne[] = {
      0xf5, 0x50, 0x8f, 0x88, 0x7d, 0xdd, 0xb5, 0xb4, 0x2a, 0x8b, 0xd7, 0x4d,
      0x23, 0xfe, 0xaf, 0xe9, 0x16, 0x22, 0xd2, 0x41, 0xed, 0x88, 0xf2, 0x70,
      0xcb, 0x4d, 0xeb, 0xc1, 0x71, 0x97, 0xc4, 0x0b, 0x3e, 0x5a, 0x2d, 0x96,
      0xab, 0xfa, 0xfd, 0x12, 0x8b, 0xd3, 0x3e, 0x4e, 0x05, 0x6f, 0x04, 0xeb,
      0x59, 0x3c, 0x0e, 0xa1, 0x73, 0xbe, 0x9d, 0x99, 0x2f, 0x05, 0xf9, 0x54,
      0x8d, 0x98, 0x1e, 0x0d, 0xc4, 0x0c, 0xc3, 0x30, 0x23, 0xff, 0xe5, 0xd0,
      0x2b, 0xd5, 0x4e, 0x2b, 0xa0, 0xae, 0xb8, 0x32, 0x84, 0x45, 0x8b, 0x3c,
      0x6d, 0xf0, 0x10, 0x36, 0x9e, 0x6a, 0xc4, 0x67, 0xca, 0xa9, 0xfc, 0x06,
      0x96, 0xd0, 0xbc, 0xda, 0xd1, 0x55, 0x55, 0x8d, 0x77, 0x21, 0xf4, 0x82,
      0x39, 0x37, 0x91, 0xd5, 0x97, 0x56, 0x78, 0xc8, 0x3c, 0xcb, 0x5e, 0xf6,
      0xdc, 0x58, 0x48, 0xb3, 0x7c, 0x94, 0x29, 0x39,
  };
  static const uint8_t kDModQMinusOne[] = {
      0x64, 0x65, 0xbd, 0x7d, 0x1a, 0x96, 0x26, 0xa1, 0xfe, 0xf3, 0x94, 0x0d,
      0x5d, 0xec, 0x85, 0xe2, 0xf8, 0xb3, 0x4c, 0xcb, 0xf9, 0x85, 0x8b, 0x12,
      0x9c, 0xa0, 0x32, 0x32, 0x35, 0x92, 0x5a, 0x94, 0x47, 0x1b, 0x70, 0xd2,
      0x90, 0x04, 0x49, 0x01, 0xd8, 0xc5, 0xe4, 0xc4, 0x43, 0xb7, 0xe9, 0x36,
      0xba, 0xbc, 0x73, 0xa8, 0xfb, 0xaf, 0x86, 0xc1, 0xd8, 0x3d, 0xcb, 0xac,
      0xf1, 0xcb, 0x60, 0x7d, 0x27, 0x21, 0xde, 0x64, 0x7f, 0xe8, 0xa8, 0x65,
      0xcc, 0x40, 0x60, 0xff, 0xa0, 0x2b, 0xfc, 0x0f, 0x80, 0x1d, 0x79, 0xca,
      0x58, 0x8a, 0xd6, 0x0f, 0xed, 0x78, 0x9a, 0x02, 0x00, 0x04, 0xc2, 0x53,
      0x41, 0xe8, 0x1a, 0xd0, 0xfd, 0x71, 0x5b, 0x43, 0xac, 0x19, 0x4a, 0xb6,
      0x12, 0xa3, 0xcb, 0xe1, 0xc7, 0x7d, 0x5c, 0x98, 0x74, 0x4e, 0x63, 0x74,
      0x6b, 0x91, 0x7a, 0x29, 0x3b, 0x92, 0xb2, 0x85,
  };
  static const uint8_t kQInverseModP[] = {
      0xd0, 0xde, 0x19, 0xda, 0x1e, 0xa2, 0xd8, 0x8f, 0x1c, 0x92, 0x73, 0xb0,
      0xc9, 0x90, 0xc7, 0xf5, 0xec, 0xc5, 0x89, 0x01, 0x05, 0x78, 0x11, 0x2d,
      0x74, 0x34, 0x44, 0xad, 0xd5, 0xf7, 0xa4, 0xfe, 0x9f, 0x25, 0x4d, 0x0b,
      0x92, 0xe3, 0xb8, 0x7d, 0xd3, 0xfd, 0xa5, 0xca, 0x95, 0x60, 0xa3, 0xf9,
      0x55, 0x42, 0x14, 0xb2, 0x45, 0x51, 0x9f, 0x73, 0x88, 0x43, 0x8a, 0xd1,
      0x65, 0x9e, 0xd1, 0xf7, 0x82, 0x2a, 0x2a, 0x8d, 0x70, 0x56, 0xe3, 0xef,
      0xc9, 0x0e, 0x2a, 0x2c, 0x15, 0xaf, 0x7f, 0x97, 0x81, 0x66, 0xf3, 0xb5,
      0x00, 0xa9, 0x26, 0xcc, 0x1e, 0xc2, 0x98, 0xdd, 0xd3, 0x37, 0x06, 0x79,
      0xb3, 0x60, 0x58, 0x79, 0x99, 0x3f, 0xa3, 0x15, 0x1f, 0x31, 0xe3, 0x11,
      0x88, 0x4c, 0x35, 0x57, 0xfa, 0x79, 0xd7, 0xd8, 0x72, 0xee, 0x73, 0x95,
      0x89, 0x29, 0xc7, 0x05, 0x27, 0x68, 0x90, 0x15,
  };

  RSA *rsa = RSA_new();
  if (rsa == NULL ||
      !set_bignum(&rsa->n, kN, sizeof(kN)) ||
      !set_bignum(&rsa->e, kE, sizeof(kE)) ||
      !set_bignum(&rsa->d, kD, sizeof(kD)) ||
      !set_bignum(&rsa->p, kP, sizeof(kP)) ||
      !set_bignum(&rsa->q, kQ, sizeof(kQ)) ||
      !set_bignum(&rsa->dmp1, kDModPMinusOne, sizeof(kDModPMinusOne)) ||
      !set_bignum(&rsa->dmq1, kDModQMinusOne, sizeof(kDModQMinusOne)) ||
      !set_bignum(&rsa->iqmp, kQInverseModP, sizeof(kQInverseModP))) {
    RSA_free(rsa);
    return NULL;
  }

  return rsa;
}

static DH *self_test_ffc_dh_key(const uint8_t *p, size_t p_len,
                                const uint8_t *q, size_t q_len,
                                const uint8_t *g, size_t g_len,
                                const uint8_t *priv_key, size_t priv_key_len,
                                const uint8_t *pub_key, size_t pub_key_len) {
  DH *dh = DH_new();
  if (dh == NULL ||
      !set_bignum(&dh->p, p, p_len) ||
      !set_bignum(&dh->q, q, q_len) ||
      !set_bignum(&dh->g, g, g_len) ||
      !set_bignum(&dh->priv_key, priv_key, priv_key_len) ||
      !set_bignum(&dh->pub_key, pub_key, pub_key_len)) {
    DH_free(dh);
    return NULL;
  }
  return dh;
}

// domainParameterGenerationMode is FB
static DH *self_test_ffc_dh_fb_key(void) {
  // The data is fetched from ACVP data.
  // Details are available in CryptoAlg-851?selectedConversation=c0120d22-f2bd-4eae-8ae6-b04fcca86743
  // File name: 197488/683891/testvector-request.json
  // Other details: "tgId": 2, "tcId": 6
  static const uint8_t kDH_p[256] = {
      0xd9, 0xc4, 0x37, 0xc1, 0xa3, 0xe3, 0x5c, 0xb8, 0xbe, 0xc8, 0x0a, 0x83,
      0xa3, 0xd3, 0x04, 0x3f, 0xd1, 0x4c, 0x40, 0xbd, 0x3c, 0x3c, 0x29, 0x26,
      0xe5, 0x16, 0xde, 0xde, 0xd7, 0xdf, 0xe6, 0x68, 0x4f, 0x85, 0xa2, 0x46,
      0x65, 0xc1, 0xaf, 0xa8, 0x61, 0x99, 0xf8, 0xc3, 0xd9, 0xd3, 0xa9, 0xc9,
      0x15, 0xa9, 0x13, 0x5c, 0x0b, 0xac, 0xf6, 0x7c, 0xbc, 0xe9, 0x35, 0x6c,
      0xac, 0xf1, 0x28, 0xa2, 0x43, 0xef, 0xfd, 0x2a, 0x59, 0xd8, 0x5d, 0x89,
      0x7e, 0x28, 0x90, 0x46, 0xf5, 0x44, 0xa8, 0x07, 0x8e, 0x70, 0xef, 0x27,
      0x36, 0x78, 0x24, 0x3a, 0x22, 0xd4, 0x16, 0xa8, 0xb4, 0xea, 0x6c, 0x13,
      0x6c, 0xd9, 0x98, 0xc2, 0x59, 0xfe, 0xdb, 0x99, 0xf9, 0xeb, 0x07, 0x7d,
      0x43, 0xdd, 0x5e, 0xbf, 0x39, 0x59, 0xb4, 0xb8, 0x64, 0x88, 0xfe, 0xdb,
      0x6c, 0x8a, 0x43, 0xac, 0x67, 0x13, 0xa9, 0x78, 0x79, 0xa7, 0xff, 0x4e,
      0xc6, 0x1a, 0x3e, 0x13, 0x3e, 0xe8, 0x15, 0x57, 0xab, 0x29, 0xe7, 0x86,
      0x5e, 0x8c, 0xb3, 0x24, 0xb0, 0x3c, 0x6a, 0x3c, 0x7d, 0x4c, 0xc9, 0xb5,
      0x4d, 0x31, 0xb5, 0x0a, 0x28, 0xbb, 0x7d, 0x02, 0x75, 0x12, 0xa6, 0xca,
      0x9d, 0x1f, 0x28, 0x45, 0xb9, 0xaf, 0x63, 0x66, 0xbb, 0x49, 0x3f, 0x28,
      0x66, 0x8c, 0xb5, 0x1d, 0x81, 0x08, 0xbd, 0x1e, 0x6b, 0x43, 0x31, 0x4e,
      0x88, 0x48, 0x25, 0xe8, 0xc4, 0xd7, 0x83, 0x5f, 0xa3, 0x65, 0x1a, 0xc7,
      0x2d, 0x3b, 0xeb, 0xb0, 0xb2, 0xa9, 0x83, 0x96, 0xaa, 0x7c, 0x23, 0x15,
      0x0e, 0x2e, 0x6e, 0x46, 0x5d, 0x1b, 0x34, 0xc0, 0x70, 0xfd, 0xa8, 0x8d,
      0x82, 0x97, 0x10, 0xd4, 0x3e, 0x45, 0x16, 0x3e, 0x54, 0x42, 0x4a, 0x2a,
      0x76, 0x5c, 0x6f, 0x30, 0x44, 0x1d, 0xf8, 0xc7, 0x07, 0xe2, 0xb8, 0xd9,
      0xac, 0x74, 0x73, 0x09
  };
  static const uint8_t kDH_q[28] = {
      0xcc, 0x9c, 0x34, 0x91, 0x8e, 0x8b, 0xa0, 0x86, 0x23, 0xa2, 0x76, 0x82,
      0xeb, 0xd1, 0x98, 0x5f, 0xab, 0x27, 0x56, 0x44, 0x66, 0x70, 0x50, 0xc9,
      0x35, 0xdc, 0x14, 0xc7
  };
  static const uint8_t kDH_g[256] = {
      0x31, 0xee, 0x30, 0xb2, 0x17, 0x6f, 0x2a, 0xd1, 0x67, 0x09, 0xdf, 0x01,
      0x38, 0x0b, 0x81, 0xf6, 0x7a, 0x5d, 0xee, 0x06, 0xd0, 0xed, 0x20, 0x0e,
      0x80, 0xda, 0xe0, 0x97, 0x7d, 0xda, 0x02, 0xff, 0x86, 0xf2, 0x9f, 0x6a,
      0x30, 0x20, 0xe4, 0x3f, 0x71, 0x1a, 0x69, 0x04, 0x16, 0x60, 0xa6, 0xae,
      0x04, 0xb0, 0x33, 0xe5, 0xbd, 0xdb, 0x7a, 0x39, 0xaf, 0x70, 0xf7, 0x5c,
      0x55, 0xfb, 0x31, 0x56, 0xe7, 0xe4, 0xee, 0xa1, 0x15, 0x87, 0xf2, 0xdd,
      0x58, 0x65, 0x75, 0x09, 0xc1, 0x4d, 0xe9, 0xf4, 0x84, 0xb6, 0x8b, 0x66,
      0xf8, 0xde, 0xf0, 0x10, 0x0f, 0x8d, 0x6d, 0xc3, 0x7d, 0x82, 0x65, 0x3e,
      0x34, 0x99, 0xb6, 0x62, 0xe9, 0x71, 0x94, 0x5e, 0x6a, 0x55, 0x0f, 0x9e,
      0xc7, 0x7f, 0x53, 0xb3, 0x5c, 0x3b, 0x26, 0x83, 0xc1, 0xbc, 0x50, 0xc8,
      0xac, 0xbc, 0xd0, 0x40, 0xf9, 0x3c, 0x41, 0x28, 0xa0, 0xeb, 0xef, 0xb3,
      0x58, 0xe4, 0x57, 0xaa, 0xf8, 0x5d, 0x97, 0x1f, 0x29, 0xd5, 0x69, 0xcc,
      0x2d, 0xd3, 0xcf, 0xf9, 0x72, 0x8f, 0xae, 0xe3, 0x41, 0x93, 0x3f, 0x34,
      0x3e, 0x36, 0x8c, 0xdc, 0xd4, 0x37, 0x55, 0x8b, 0x81, 0xaa, 0xff, 0xa7,
      0x0b, 0xaf, 0xb3, 0x76, 0xe3, 0x12, 0xb4, 0x24, 0xcf, 0xfb, 0x42, 0x43,
      0x2e, 0xb7, 0x48, 0x01, 0xcf, 0x24, 0x03, 0x84, 0x74, 0x0a, 0x7a, 0xa2,
      0xaa, 0x3f, 0x38, 0xc0, 0x77, 0xa2, 0xb3, 0x60, 0x30, 0x00, 0x77, 0xdc,
      0x10, 0x5a, 0xd8, 0xc1, 0x82, 0xde, 0xb7, 0x48, 0x79, 0xa5, 0x0c, 0xd0,
      0x6c, 0x52, 0xde, 0x5d, 0xf0, 0x4a, 0x58, 0x3b, 0x8e, 0xde, 0xf1, 0xc0,
      0x42, 0x63, 0x1e, 0x4d, 0xcf, 0x26, 0x44, 0x9e, 0x50, 0x98, 0x03, 0xbc,
      0x5b, 0xfc, 0xef, 0x07, 0x3d, 0xae, 0xf7, 0xda, 0x9d, 0x76, 0x8a, 0x8d,
      0xa8, 0xb4, 0xe9, 0x79
  };
  const uint8_t kDH_private_key[28] = {
      0x75, 0x89, 0x8a, 0xbe, 0xc3, 0xc9, 0xc8, 0x7b, 0x04, 0x49, 0x47, 0xf6,
      0xc5, 0x1f, 0x9f, 0x71, 0x7f, 0x4a, 0x1d, 0x7c, 0xc3, 0x9a, 0xae, 0xcd,
      0x83, 0x53, 0xba, 0x25
  };
  const uint8_t kDH_public_key[256] = {
      0x80, 0xbc, 0xbe, 0xf0, 0xad, 0x46, 0xfe, 0x97, 0x79, 0x4b, 0xd1, 0x49,
      0x00, 0x04, 0xf0, 0x7f, 0x32, 0xac, 0x56, 0x17, 0x6b, 0xea, 0x84, 0xb5,
      0xdc, 0xc7, 0x2f, 0xec, 0x5e, 0x87, 0xd2, 0xd6, 0xa6, 0x9a, 0xbd, 0x73,
      0x12, 0x8b, 0x6a, 0x8a, 0x42, 0xe9, 0x21, 0x74, 0x28, 0xda, 0x64, 0xb2,
      0x8b, 0x05, 0x04, 0x80, 0x02, 0xcf, 0x8a, 0xab, 0xa0, 0xfe, 0x9c, 0xfb,
      0xfd, 0x3e, 0xb9, 0xa5, 0xfe, 0x08, 0xa5, 0xf3, 0x97, 0xdd, 0x28, 0x38,
      0x33, 0x41, 0x0d, 0x84, 0x1f, 0x46, 0xd3, 0x03, 0xfb, 0x6e, 0xac, 0x2f,
      0x17, 0x97, 0x5f, 0xad, 0x73, 0x23, 0xd2, 0xb3, 0x71, 0x26, 0xdb, 0x0e,
      0x97, 0xa4, 0x15, 0x36, 0x2c, 0x61, 0xc1, 0x21, 0xa6, 0xd1, 0x4d, 0xc6,
      0x54, 0xb4, 0xad, 0x5e, 0x9c, 0x0f, 0xe8, 0x00, 0xd5, 0x4a, 0x62, 0x32,
      0xfe, 0x62, 0x9e, 0x0b, 0x7c, 0xc8, 0x54, 0x8d, 0x83, 0xca, 0x36, 0x4d,
      0x0d, 0x5a, 0xbf, 0xe1, 0x5d, 0x5f, 0xfc, 0xc8, 0x3c, 0xc0, 0xec, 0xf4,
      0x35, 0x62, 0xd4, 0x16, 0xc3, 0x18, 0x0b, 0xf0, 0x41, 0x52, 0x9e, 0x57,
      0xa7, 0xf9, 0xfd, 0x14, 0xfc, 0x1b, 0x5c, 0xa5, 0x85, 0xf3, 0x05, 0xb6,
      0x6c, 0xde, 0xa0, 0x58, 0x49, 0xe0, 0xcc, 0x53, 0x25, 0x9f, 0xbd, 0xfb,
      0xab, 0x75, 0x83, 0x18, 0xc4, 0x9d, 0x2a, 0x95, 0xb0, 0x53, 0xc4, 0x7c,
      0xdc, 0x91, 0x55, 0x01, 0x93, 0xe1, 0x44, 0xd9, 0x64, 0x6c, 0xd9, 0xa5,
      0x70, 0xe9, 0x5b, 0x31, 0x83, 0x93, 0x1d, 0x79, 0x15, 0xfa, 0x8e, 0x84,
      0x4f, 0x04, 0x17, 0x8a, 0x12, 0x69, 0x83, 0x9c, 0xd6, 0x8b, 0x78, 0x58,
      0xfa, 0x2c, 0x6b, 0xeb, 0xe8, 0x47, 0xf8, 0x14, 0x0e, 0x33, 0x7a, 0x95,
      0xce, 0x34, 0x0f, 0x68, 0x32, 0x44, 0x76, 0xf6, 0xe8, 0x2e, 0x89, 0x72,
      0x11, 0x49, 0x04, 0x12
  };
  return self_test_ffc_dh_key(kDH_p, sizeof(kDH_p),
                              kDH_q, sizeof(kDH_q),
                              kDH_g, sizeof(kDH_g),
                              kDH_private_key, sizeof(kDH_private_key),
                              kDH_public_key, sizeof(kDH_public_key));
}

static EC_KEY *self_test_ecdsa_key(void) {
  static const uint8_t kQx[] = {
      0xc8, 0x15, 0x61, 0xec, 0xf2, 0xe5, 0x4e, 0xde, 0xfe, 0x66, 0x17,
      0xdb, 0x1c, 0x7a, 0x34, 0xa7, 0x07, 0x44, 0xdd, 0xb2, 0x61, 0xf2,
      0x69, 0xb8, 0x3d, 0xac, 0xfc, 0xd2, 0xad, 0xe5, 0xa6, 0x81,
  };
  static const uint8_t kQy[] = {
      0xe0, 0xe2, 0xaf, 0xa3, 0xf9, 0xb6, 0xab, 0xe4, 0xc6, 0x98, 0xef,
      0x64, 0x95, 0xf1, 0xbe, 0x49, 0xa3, 0x19, 0x6c, 0x50, 0x56, 0xac,
      0xb3, 0x76, 0x3f, 0xe4, 0x50, 0x7e, 0xec, 0x59, 0x6e, 0x88,
  };
  static const uint8_t kD[] = {
      0xc6, 0xc1, 0xaa, 0xda, 0x15, 0xb0, 0x76, 0x61, 0xf8, 0x14, 0x2c,
      0x6c, 0xaf, 0x0f, 0xdb, 0x24, 0x1a, 0xff, 0x2e, 0xfe, 0x46, 0xc0,
      0x93, 0x8b, 0x74, 0xf2, 0xbc, 0xc5, 0x30, 0x52, 0xb0, 0x77,
  };

  EC_KEY *ec_key = EC_KEY_new();
  BIGNUM *qx = BN_bin2bn(kQx, sizeof(kQx), NULL);
  BIGNUM *qy = BN_bin2bn(kQy, sizeof(kQy), NULL);
  BIGNUM *d = BN_bin2bn(kD, sizeof(kD), NULL);
  if (ec_key == NULL || qx == NULL || qy == NULL || d == NULL ||
      !EC_KEY_set_group(ec_key, EC_group_p256()) ||
      !EC_KEY_set_public_key_affine_coordinates(ec_key, qx, qy) ||
      !EC_KEY_set_private_key(ec_key, d)) {
    EC_KEY_free(ec_key);
    ec_key = NULL;
  }

  BN_free(qx);
  BN_free(qy);
  BN_free(d);
  return ec_key;
}

static DH *self_test_dh(void) {
  DH *dh = DH_get_rfc7919_2048();
  if (!dh) {
    return NULL;
  }

  BIGNUM *priv = BN_new();
  if (!priv) {
    goto err;
  }

  // kFFDHE2048PrivateKeyData is a 225-bit value. (225 because that's the
  // minimum private key size in
  // https://tools.ietf.org/html/rfc7919#appendix-A.1.)
  static const BN_ULONG kFFDHE2048PrivateKeyData[] = {
      TOBN(0x187be36b, 0xd38a4fa1),
      TOBN(0x0a152f39, 0x6458f3b8),
      TOBN(0x0570187e, 0xc422eeb7),
      TOBN(0x00000001, 0x91173f2a),
  };

  bn_set_static_words(priv, kFFDHE2048PrivateKeyData,
                      OPENSSL_ARRAY_SIZE(kFFDHE2048PrivateKeyData));

  if (!DH_set0_key(dh, NULL, priv)) {
    goto err;
  }
  return dh;

err:
  BN_free(priv);
  DH_free(dh);
  return NULL;
}


// Lazy self-tests
//
// Self tests that are slow are deferred until the corresponding algorithm is
// actually exercised, in FIPS mode. (In non-FIPS mode these tests are only run
// when requested by |BORINGSSL_self_test|.)

static int boringssl_self_test_rsa(void) {
  int ret = 0;
  uint8_t output[256];

  RSA *const rsa_key = self_test_rsa_key();
  if (rsa_key == NULL) {
    fprintf(stderr, "RSA key construction failed\n");
    goto err;
  }
  // Disable blinding for the power-on tests because it's not needed and
  // triggers an entropy draw.
  rsa_key->flags |= RSA_FLAG_NO_BLINDING;

  // RSA Sign KAT

  static const uint8_t kRSASignPlaintext[32] = {
      0xd2, 0xb5, 0x6e, 0x53, 0x30, 0x6f, 0x72, 0x0d, 0x79, 0x29, 0xd8,
      0x70, 0x8b, 0xf4, 0x6f, 0x1c, 0x22, 0x30, 0x03, 0x05, 0x58, 0x2b,
      0x11, 0x5b, 0xed, 0xca, 0xc7, 0x22, 0xd8, 0xaa, 0x5a, 0xb2,
  };
  static const uint8_t kRSASignSignature[256] = {
      0xb7, 0xee, 0x25, 0x11, 0x66, 0xd4, 0xfd, 0x87, 0x10, 0x8f, 0x7c, 0x85,
      0x9a, 0x1d, 0x35, 0x42, 0x2b, 0x02, 0x30, 0xaa, 0x8a, 0x81, 0xf7, 0x50,
      0xb2, 0x64, 0xc0, 0x15, 0xe6, 0xbd, 0x5d, 0xe1, 0x03, 0x8b, 0xf4, 0xd3,
      0x5b, 0x4a, 0x8c, 0xf1, 0x7f, 0x3f, 0xcd, 0x1f, 0xe0, 0x90, 0x78, 0x3a,
      0xde, 0xef, 0xd2, 0x69, 0xe7, 0xe5, 0x59, 0xba, 0xc9, 0xde, 0x53, 0x74,
      0x84, 0xaa, 0x33, 0x06, 0x29, 0x47, 0xd6, 0x0f, 0x00, 0xca, 0xe2, 0xde,
      0xab, 0x6c, 0x86, 0x9d, 0xb2, 0xc6, 0x4f, 0xb6, 0x8e, 0xde, 0xbe, 0x1c,
      0x26, 0x46, 0x2f, 0xbb, 0x05, 0xc1, 0x2b, 0x79, 0xb0, 0x82, 0x4e, 0xc9,
      0x03, 0xb3, 0x1a, 0x27, 0xd0, 0xc8, 0x52, 0x2e, 0x22, 0x04, 0x70, 0x50,
      0x1d, 0xcf, 0xd6, 0x6d, 0x7a, 0xa0, 0x88, 0x90, 0x3a, 0x49, 0x64, 0x65,
      0x72, 0x88, 0x51, 0xe6, 0xdb, 0x8e, 0x8c, 0xbe, 0x1c, 0x50, 0xc7, 0x1a,
      0xc0, 0x8b, 0x44, 0x3d, 0x84, 0x46, 0xb0, 0xa2, 0x7b, 0x37, 0xc8, 0x03,
      0xf0, 0x01, 0x2e, 0x76, 0xc1, 0x0b, 0x7b, 0x0a, 0x5e, 0x62, 0xd9, 0x93,
      0x4b, 0x86, 0xaf, 0xa8, 0x44, 0x20, 0x3f, 0xca, 0xe7, 0x6b, 0x8d, 0x27,
      0x56, 0xf6, 0x03, 0xc1, 0xd5, 0x5e, 0xaa, 0x64, 0x34, 0xa4, 0xf6, 0xf2,
      0x0b, 0x86, 0xc1, 0xd4, 0xa4, 0xa9, 0x6e, 0x58, 0x8c, 0x0a, 0x09, 0xca,
      0xad, 0x4b, 0xd6, 0x6b, 0x3b, 0xfe, 0xfe, 0xfb, 0x09, 0x3c, 0x76, 0xdb,
      0x33, 0x2d, 0x62, 0x28, 0x44, 0xc4, 0xd5, 0x43, 0xe4, 0x86, 0x2e, 0xa8,
      0xa0, 0x26, 0xa8, 0x8c, 0xe9, 0x40, 0x7c, 0xf8, 0x59, 0x84, 0x07, 0xbd,
      0x90, 0xe0, 0xc7, 0xcb, 0x76, 0x8b, 0x41, 0x0f, 0xba, 0xed, 0xc9, 0xbf,
      0x14, 0x77, 0xc5, 0xdb, 0x16, 0xd1, 0xdb, 0x33, 0xbc, 0x20, 0xd0, 0xec,
      0x04, 0xbf, 0xf8, 0xe3,
  };

  unsigned sig_len;
  if (!rsa_digestsign_no_self_test(EVP_sha256(), kRSASignPlaintext,
                         sizeof(kRSASignPlaintext),output, &sig_len, rsa_key) ||
      !check_test(kRSASignSignature, output, sizeof(kRSASignSignature),
                  "RSA-sign KAT")) {
    fprintf(stderr, "RSA signing test failed.\n");
    goto err;
  }

  // RSA Verify KAT

  static const uint8_t kRSAVerifyPlaintext[32] = {
      0x09, 0x65, 0x2f, 0xd8, 0xed, 0x9d, 0xc2, 0x6d, 0xbc, 0xbf, 0xf2,
      0xa7, 0xa5, 0xed, 0xe1, 0x37, 0x13, 0x78, 0x21, 0x36, 0xcf, 0x8d,
      0x22, 0x3d, 0xab, 0x93, 0xb4, 0x12, 0xa8, 0xb5, 0x15, 0x53,
  };
  static const uint8_t kRSAVerifySignature[256] = {
      0xc5, 0xef, 0x03, 0x0d, 0x00, 0xa1, 0x3e, 0x3a, 0x70, 0x5b, 0x23, 0xe1,
      0xe3, 0xde, 0x3f, 0x2c, 0x8e, 0x84, 0xb2, 0xe8, 0x2d, 0x1b, 0xec, 0x14,
      0x11, 0x6f, 0x82, 0x45, 0xe5, 0xb6, 0xfa, 0x4b, 0x20, 0x7f, 0x12, 0xaf,
      0xe7, 0x2c, 0x8d, 0x36, 0x67, 0x5a, 0xcb, 0x7d, 0x67, 0x0f, 0x6a, 0x5c,
      0x59, 0x0e, 0x44, 0x71, 0x6c, 0x3d, 0xf3, 0x11, 0x04, 0xbe, 0xa8, 0x9f,
      0x61, 0xbe, 0xcd, 0x6c, 0xc1, 0x88, 0x81, 0x48, 0x01, 0xd3, 0x08, 0xce,
      0xec, 0x2a, 0x84, 0x3e, 0xc7, 0xf2, 0x5e, 0xbc, 0xdd, 0xe5, 0x88, 0xdd,
      0x89, 0x80, 0x32, 0x69, 0x28, 0xb1, 0x08, 0x43, 0xc4, 0xb3, 0x19, 0x03,
      0x38, 0xb0, 0xa0, 0x7c, 0x5a, 0x94, 0xa5, 0x3f, 0x6d, 0x84, 0xde, 0x79,
      0x47, 0xf3, 0xdb, 0x3d, 0x9f, 0x73, 0x06, 0x10, 0xbf, 0x46, 0x3c, 0x29,
      0x1f, 0xd9, 0x01, 0xab, 0x8a, 0x54, 0xe4, 0x7d, 0xbb, 0x19, 0x6d, 0x8a,
      0xf5, 0x3e, 0xf1, 0x5f, 0xd0, 0x6b, 0x7a, 0xd0, 0xdd, 0xb6, 0x5c, 0x83,
      0xc8, 0xfe, 0xe8, 0xd0, 0xa7, 0x08, 0x33, 0x47, 0x33, 0xcb, 0xe7, 0xb0,
      0x33, 0x22, 0x69, 0x5d, 0x9a, 0x40, 0x6a, 0x68, 0x78, 0x67, 0xb3, 0x82,
      0x94, 0xd8, 0x63, 0x48, 0x05, 0xd4, 0xcb, 0x68, 0x73, 0x73, 0x55, 0xd7,
      0x76, 0x48, 0x4d, 0xf3, 0xcd, 0x8b, 0x8f, 0xeb, 0x51, 0xfd, 0x94, 0x60,
      0x82, 0x18, 0xd8, 0x8e, 0xb2, 0xf2, 0xd0, 0x40, 0x64, 0x31, 0x2a, 0xd0,
      0x41, 0x2e, 0x7a, 0x96, 0xd4, 0x9a, 0x9d, 0x71, 0xd7, 0x66, 0x06, 0xab,
      0x7a, 0x5b, 0xd9, 0x9b, 0xc3, 0x1c, 0x37, 0x59, 0x3c, 0x83, 0x7f, 0x15,
      0xba, 0x86, 0x01, 0x92, 0x6a, 0x1f, 0x69, 0x19, 0xd1, 0x11, 0x0b, 0x55,
      0x90, 0x6e, 0x18, 0xe2, 0x9e, 0x2a, 0x94, 0x04, 0x5a, 0xe9, 0x21, 0x8b,
      0xc6, 0xc8, 0xda, 0x74,
  };
  if (!rsa_digestverify_no_self_test(EVP_sha256(), kRSAVerifyPlaintext,
                               sizeof(kRSAVerifyPlaintext), kRSAVerifySignature,
                               sizeof(kRSAVerifySignature), rsa_key)) {
    fprintf(stderr, "RSA-verify KAT failed.\n");
    goto err;
  }

  ret = 1;

err:
  RSA_free(rsa_key);

  return ret;
}

static int boringssl_self_test_ecc(void) {
  int ret = 0;
  EC_KEY *ec_key = NULL;
  EC_POINT *ec_point_in = NULL;
  EC_POINT *ec_point_out = NULL;
  BIGNUM *ec_scalar = NULL;
  ECDSA_SIG *sig = NULL;

  ec_key = self_test_ecdsa_key();
  if (ec_key == NULL) {
    fprintf(stderr, "ECDSA KeyGen failed\n");
    goto err;
  }

  // ECDSA Sign/Verify KAT

  static const uint8_t kECDSASignPlaintext[32] = {
      0x1e, 0x35, 0x93, 0x0b, 0xe8, 0x60, 0xd0, 0x94, 0x2c, 0xa7, 0xbb,
      0xd6, 0xf6, 0xde, 0xd8, 0x7f, 0x15, 0x7e, 0x4d, 0xe2, 0x4f, 0x81,
      0xed, 0x4b, 0x87, 0x5c, 0x0e, 0x01, 0x8e, 0x89, 0xa8, 0x1f,
  };
  static const uint8_t kECDSASignSig[64] = {
      0x67, 0x80, 0xc5, 0xfc, 0x70, 0x27, 0x5e, 0x2c, 0x70, 0x61, 0xa0,
      0xe7, 0x87, 0x7b, 0xb1, 0x74, 0xde, 0xad, 0xeb, 0x98, 0x87, 0x02,
      0x7f, 0x3f, 0xa8, 0x36, 0x54, 0x15, 0x8b, 0xa7, 0xf5, 0x0c, 0x3a,
      0x82, 0xd9, 0x65, 0xb2, 0xa7, 0x2a, 0xc5, 0xa6, 0x07, 0xbc, 0x56,
      0xab, 0x37, 0x22, 0xc8, 0x42, 0xd9, 0x21, 0xc0, 0x49, 0x36, 0xe9,
      0xe2, 0x5f, 0xd7, 0xb6, 0x00, 0xe7, 0xd8, 0xdc, 0x80,
  };

  // The 'k' value for ECDSA is fixed to avoid an entropy draw.
  uint8_t ecdsa_k[32] = {0};
  ecdsa_k[31] = 42;

  sig = ecdsa_digestsign_no_self_test(EVP_sha256(),kECDSASignPlaintext, sizeof
                      (kECDSASignPlaintext), ec_key, ecdsa_k, sizeof(ecdsa_k));

  uint8_t ecdsa_sign_output[64];
  if (sig == NULL ||
      !serialize_ecdsa_sig(ecdsa_sign_output, sizeof(ecdsa_sign_output), sig) ||
      !check_test(kECDSASignSig, ecdsa_sign_output, sizeof(ecdsa_sign_output),
                  "ECDSA-sign signature")) {
    fprintf(stderr, "ECDSA-sign KAT failed.\n");
    goto err;
  }

  static const uint8_t kECDSAVerifyPlaintext[32] = {
      0x78, 0x7c, 0x50, 0x5c, 0x60, 0xc9, 0xe4, 0x13, 0x6c, 0xe4, 0x48,
      0xba, 0x93, 0xff, 0x71, 0xfa, 0x9c, 0x18, 0xf4, 0x17, 0x09, 0x4f,
      0xdf, 0x5a, 0xe2, 0x75, 0xc0, 0xcc, 0xd2, 0x67, 0x97, 0xad,
  };
  static const uint8_t kECDSAVerifySig[64] = {
      0x67, 0x80, 0xc5, 0xfc, 0x70, 0x27, 0x5e, 0x2c, 0x70, 0x61, 0xa0,
      0xe7, 0x87, 0x7b, 0xb1, 0x74, 0xde, 0xad, 0xeb, 0x98, 0x87, 0x02,
      0x7f, 0x3f, 0xa8, 0x36, 0x54, 0x15, 0x8b, 0xa7, 0xf5, 0x0c, 0x3c,
      0x77, 0xd1, 0xb6, 0xe0, 0x9e, 0x74, 0x7b, 0xc5, 0xab, 0x55, 0x01,
      0xd7, 0x5e, 0x61, 0x8d, 0x8e, 0x5b, 0x27, 0x2e, 0x15, 0x9f, 0xf3,
      0x41, 0x3c, 0xb7, 0x1a, 0x81, 0x40, 0x8d, 0x56, 0x05,
  };

  ECDSA_SIG_free(sig);
  sig = parse_ecdsa_sig(kECDSAVerifySig, sizeof(kECDSAVerifySig));
  if (!sig ||
      !ecdsa_digestverify_no_self_test(EVP_sha256(), kECDSAVerifyPlaintext,
                                  sizeof(kECDSAVerifyPlaintext), sig, ec_key)) {
    fprintf(stderr, "ECDSA-verify KAT failed.\n");
    goto err;
  }

  // Primitive Z Computation KAT (IG 9.6).

  // kP256Point is SHA256("Primitive Z Computation KAT")×G within P-256.
  static const uint8_t kP256Point[65] = {
      0x04, 0x4e, 0xc1, 0x94, 0x8c, 0x5c, 0xf4, 0x37, 0x35, 0x0d, 0xa3,
      0xf9, 0x55, 0xf9, 0x8b, 0x26, 0x23, 0x5c, 0x43, 0xe0, 0x83, 0x51,
      0x2b, 0x0d, 0x4b, 0x56, 0x24, 0xc3, 0xe4, 0xa5, 0xa8, 0xe2, 0xe9,
      0x95, 0xf2, 0xc4, 0xb9, 0xb7, 0x48, 0x7d, 0x2a, 0xae, 0xc5, 0xc0,
      0x0a, 0xcc, 0x1b, 0xd0, 0xec, 0xb8, 0xdc, 0xbe, 0x0c, 0xbe, 0x52,
      0x79, 0x93, 0x7c, 0x0b, 0x92, 0x2b, 0x7f, 0x17, 0xa5, 0x80,
  };
  // kP256Scalar is SHA256("Primitive Z Computation KAT scalar").
  static const uint8_t kP256Scalar[32] = {
      0xe7, 0x60, 0x44, 0x91, 0x26, 0x9a, 0xfb, 0x5b, 0x10, 0x2d, 0x6e,
      0xa5, 0x2c, 0xb5, 0x9f, 0xeb, 0x70, 0xae, 0xde, 0x6c, 0xe3, 0xbf,
      0xb3, 0xe0, 0x10, 0x54, 0x85, 0xab, 0xd8, 0x61, 0xd7, 0x7b,
  };
  // kP256PointResult is |kP256Scalar|×|kP256Point|.
  static const uint8_t kP256PointResult[65] = {
      0x04, 0xf1, 0x63, 0x00, 0x88, 0xc5, 0xd5, 0xe9, 0x05, 0x52, 0xac,
      0xb6, 0xec, 0x68, 0x76, 0xb8, 0x73, 0x7f, 0x0f, 0x72, 0x34, 0xe6,
      0xbb, 0x30, 0x32, 0x22, 0x37, 0xb6, 0x2a, 0x80, 0xe8, 0x9e, 0x6e,
      0x6f, 0x36, 0x02, 0xe7, 0x21, 0xd2, 0x31, 0xdb, 0x94, 0x63, 0xb7,
      0xd8, 0x19, 0x0e, 0xc2, 0xc0, 0xa7, 0x2f, 0x15, 0x49, 0x1a, 0xa2,
      0x7c, 0x41, 0x8f, 0xaf, 0x9c, 0x40, 0xaf, 0x2e, 0x4a, 0x0c,
  };

  const EC_GROUP *ec_group = EC_group_p256();
  ec_point_in = EC_POINT_new(ec_group);
  ec_point_out = EC_POINT_new(ec_group);
  ec_scalar = BN_new();
  uint8_t z_comp_result[65];
  if (ec_point_in == NULL || ec_point_out == NULL || ec_scalar == NULL ||
      !EC_POINT_oct2point(ec_group, ec_point_in, kP256Point, sizeof(kP256Point),
                          NULL) ||
      !BN_bin2bn(kP256Scalar, sizeof(kP256Scalar), ec_scalar) ||
      !ec_point_mul_no_self_test(ec_group, ec_point_out, NULL, ec_point_in,
                                 ec_scalar, NULL) ||
      !EC_POINT_point2oct(ec_group, ec_point_out, POINT_CONVERSION_UNCOMPRESSED,
                          z_comp_result, sizeof(z_comp_result), NULL) ||
      !check_test(kP256PointResult, z_comp_result, sizeof(z_comp_result),
                  "Z Computation Result")) {
    fprintf(stderr, "Z-computation KAT failed.\n");
    goto err;
  }

  ret = 1;

err:
  EC_KEY_free(ec_key);
  EC_POINT_free(ec_point_in);
  EC_POINT_free(ec_point_out);
  BN_free(ec_scalar);
  ECDSA_SIG_free(sig);

  return ret;
}

static int boringssl_self_test_ffdh(void) {
  int ret = 0;
  DH *dh = NULL;
  DH *fb_dh = NULL;
  BIGNUM *ffdhe2048_value = NULL;
  BIGNUM *fb_peers_key = NULL;

  // FFC Diffie-Hellman KAT

  // kFFDHE2048PublicValueData is an arbitrary public value, mod
  // kFFDHE2048Data. (The private key happens to be 4096.)
  static const BN_ULONG kFFDHE2048PublicValueData[] = {
      TOBN(0x187be36b, 0xd38a4fa1), TOBN(0x0a152f39, 0x6458f3b8),
      TOBN(0x0570187e, 0xc422eeb7), TOBN(0x18af7482, 0x91173f2a),
      TOBN(0xe9fdac6a, 0xcff4eaaa), TOBN(0xf6afebb7, 0x6e589d6c),
      TOBN(0xf92f8e9a, 0xb7e33fb0), TOBN(0x70acf2aa, 0x4cf36ddd),
      TOBN(0x561ab426, 0xd07137fd), TOBN(0x5f57d037, 0x430ee91e),
      TOBN(0xe3e768c8, 0x60d10b8a), TOBN(0xb14884d8, 0xa18af8ce),
      TOBN(0xf8a98014, 0xa12b74e4), TOBN(0x748d407c, 0x3437b7a8),
      TOBN(0x627588c4, 0x9875d5a7), TOBN(0xdd24a127, 0x53c8f09d),
      TOBN(0x85a997d5, 0x0cd51aec), TOBN(0x44f0c619, 0xce348458),
      TOBN(0x9b894b24, 0x5f6b69a1), TOBN(0xae1302f2, 0xf6d4777e),
      TOBN(0xe6678eeb, 0x375db18e), TOBN(0x2674e1d6, 0x4fbcbdc8),
      TOBN(0xb297a823, 0x6fa93d28), TOBN(0x6a12fb70, 0x7c8c0510),
      TOBN(0x5c6d1aeb, 0xdb06f65b), TOBN(0xe8c2954e, 0x4c1804ca),
      TOBN(0x06bdeac1, 0xf5500fa7), TOBN(0x6a315604, 0x189cd76b),
      TOBN(0xbae7b0b3, 0x6e362dc0), TOBN(0xa57c73bd, 0xdc70fb82),
      TOBN(0xfaff50d2, 0x9d573457), TOBN(0x352bd399, 0xbe84058e),
  };
  static const uint8_t kDHOutput[2048 / 8] = {
      0x2a, 0xe6, 0xd3, 0xa6, 0x13, 0x58, 0x8e, 0xce, 0x53, 0xaa, 0xf6, 0x5d,
      0x9a, 0xae, 0x02, 0x12, 0xf5, 0x80, 0x3d, 0x06, 0x09, 0x76, 0xac, 0x57,
      0x37, 0x9e, 0xab, 0x38, 0x62, 0x25, 0x05, 0x1d, 0xf3, 0xa9, 0x39, 0x60,
      0xf6, 0xae, 0x90, 0xed, 0x1e, 0xad, 0x6e, 0xe9, 0xe3, 0xba, 0x27, 0xf6,
      0xdb, 0x54, 0xdf, 0xe2, 0xbd, 0xbb, 0x7f, 0xf1, 0x81, 0xac, 0x1a, 0xfa,
      0xdb, 0x87, 0x07, 0x98, 0x76, 0x90, 0x21, 0xf2, 0xae, 0xda, 0x0d, 0x84,
      0x97, 0x64, 0x0b, 0xbf, 0xb8, 0x8d, 0x10, 0x46, 0xe2, 0xd5, 0xca, 0x1b,
      0xbb, 0xe5, 0x37, 0xb2, 0x3b, 0x35, 0xd3, 0x1b, 0x65, 0xea, 0xae, 0xf2,
      0x03, 0xe2, 0xb6, 0xde, 0x22, 0xb7, 0x86, 0x49, 0x79, 0xfe, 0xd7, 0x16,
      0xf7, 0xdc, 0x9c, 0x59, 0xf5, 0xb7, 0x70, 0xc0, 0x53, 0x42, 0x6f, 0xb1,
      0xd2, 0x4e, 0x00, 0x25, 0x4b, 0x2d, 0x5a, 0x9b, 0xd0, 0xe9, 0x27, 0x43,
      0xcc, 0x00, 0x66, 0xea, 0x94, 0x7a, 0x0b, 0xb9, 0x89, 0x0c, 0x5e, 0x94,
      0xb8, 0x3a, 0x78, 0x9c, 0x4d, 0x84, 0xe6, 0x32, 0x2c, 0x38, 0x7c, 0xf7,
      0x43, 0x9c, 0xd8, 0xb8, 0x1c, 0xce, 0x24, 0x91, 0x20, 0x67, 0x7a, 0x54,
      0x1f, 0x7e, 0x86, 0x7f, 0xa1, 0xc1, 0x03, 0x4e, 0x2c, 0x26, 0x71, 0xb2,
      0x06, 0x30, 0xb3, 0x6c, 0x15, 0xcc, 0xac, 0x25, 0xe5, 0x37, 0x3f, 0x24,
      0x8f, 0x2a, 0x89, 0x5e, 0x3d, 0x43, 0x94, 0xc9, 0x36, 0xae, 0x40, 0x00,
      0x6a, 0x0d, 0xb0, 0x6e, 0x8b, 0x2e, 0x70, 0x57, 0xe1, 0x88, 0x53, 0xd6,
      0x06, 0x80, 0x2a, 0x4e, 0x5a, 0xf0, 0x1e, 0xaa, 0xcb, 0xab, 0x06, 0x0e,
      0x27, 0x0f, 0xd9, 0x88, 0xd9, 0x01, 0xe3, 0x07, 0xeb, 0xdf, 0xc3, 0x12,
      0xe3, 0x40, 0x88, 0x7b, 0x5f, 0x59, 0x78, 0x6e, 0x26, 0x20, 0xc3, 0xdf,
      0xc8, 0xe4, 0x5e, 0xb8,
  };

  // |kDH_fb_peer_public| and |kDH_fb_z| is fetched from ACVP data.
  // domainParameterGenerationMode is FB.
  // Details are available in
  // CryptoAlg-851?selectedConversation=c0120d22-f2bd-4eae-8ae6-b04fcca86743
  // File name: 197488/683891/testvector-request.json
  // Other details: "tgId": 2, "tcId": 6
  const uint8_t kDH_fb_peer_public[256] = {
    0x8f, 0xbc, 0x50, 0x66, 0x4b, 0x2c, 0x9e, 0x2e, 0x7d, 0x4c, 0x64, 0x1a,
    0xe2, 0xd4, 0xd2, 0xcc, 0x6a, 0xcf, 0xe6, 0xbd, 0xf3, 0x3d, 0x39, 0xf2,
    0x1d, 0xe4, 0xc3, 0x45, 0xb4, 0x51, 0x7a, 0xbd, 0x9e, 0x7d, 0x49, 0xf2,
    0xbd, 0x03, 0x4d, 0x54, 0xf3, 0x97, 0x84, 0xfe, 0x07, 0x31, 0x98, 0x0e,
    0x78, 0x5f, 0xe8, 0x5d, 0xf4, 0x6a, 0xf4, 0xf9, 0xef, 0x25, 0x6b, 0x3e,
    0x1a, 0xb2, 0x0a, 0x42, 0xec, 0x19, 0xad, 0xe9, 0x68, 0xa9, 0x8f, 0xfd,
    0x51, 0xd4, 0x95, 0x88, 0x09, 0x83, 0x28, 0xc8, 0xd6, 0x54, 0x05, 0xd1,
    0xc3, 0x75, 0xb2, 0xbf, 0x03, 0xdd, 0x5f, 0x01, 0x18, 0x6b, 0xbb, 0x8d,
    0x49, 0x75, 0x2d, 0x0d, 0xdf, 0x62, 0x0b, 0xbf, 0x70, 0xbc, 0x58, 0x25,
    0xdb, 0x37, 0xde, 0xb2, 0xea, 0xd5, 0x11, 0x57, 0xc1, 0x83, 0x26, 0x53,
    0x5d, 0x61, 0x42, 0xf9, 0xbf, 0x51, 0xf8, 0x38, 0x93, 0x7f, 0x2d, 0xdd,
    0x5f, 0x57, 0xab, 0x41, 0xf2, 0xda, 0x88, 0xe8, 0x9d, 0x0d, 0xca, 0x5d,
    0x54, 0xe6, 0x79, 0xdf, 0xe2, 0x63, 0x8a, 0x62, 0x9f, 0x48, 0x1c, 0xc3,
    0x09, 0x80, 0x32, 0x46, 0x9c, 0x76, 0xe1, 0xf3, 0xa5, 0xa8, 0x4d, 0xac,
    0xb0, 0x2e, 0x42, 0x3c, 0x1d, 0x68, 0xf2, 0x88, 0xad, 0xd7, 0x3e, 0xa7,
    0xac, 0x4c, 0x13, 0x91, 0xc1, 0x43, 0xce, 0xa5, 0x20, 0x38, 0x7d, 0x8c,
    0x05, 0x2c, 0x96, 0xd2, 0xd6, 0x2a, 0x75, 0xc1, 0xf0, 0x15, 0xa1, 0x5c,
    0xed, 0x80, 0xf9, 0x2e, 0x47, 0x11, 0x2c, 0x15, 0x6d, 0x97, 0x6f, 0x7a,
    0x2e, 0x73, 0xf7, 0x1f, 0xc8, 0x89, 0xd9, 0x34, 0x62, 0x8a, 0xdc, 0xae,
    0xe2, 0xdf, 0xda, 0x03, 0x6d, 0xce, 0x8a, 0xe2, 0x02, 0xf3, 0xd9, 0xb7,
    0x86, 0xf5, 0x3d, 0x5f, 0x28, 0xeb, 0x74, 0x81, 0xd0, 0x40, 0x6e, 0xa6,
    0x90, 0x1f, 0x97,
#if !defined(BORINGSSL_FIPS_BREAK_FFC_DH)
    0xbe
#else
    0x00,
#endif
  };
  const uint8_t kDH_fb_z[256] = {
    0x8a, 0x03, 0x41, 0x31, 0x59, 0xda, 0x27, 0xff, 0x91, 0x0b, 0xd9, 0x46,
    0x08, 0x8f, 0x08, 0x67, 0x89, 0xa2, 0x0a, 0xac, 0x32, 0x9e, 0x8f, 0x05,
    0xde, 0x0d, 0x4f, 0xb8, 0x35, 0xf8, 0x5f, 0x56, 0x6f, 0x51, 0x75, 0xfd,
    0xa2, 0x50, 0x5e, 0x5e, 0x92, 0x42, 0x89, 0xd1, 0x6c, 0xe2, 0xab, 0x13,
    0x15, 0xd9, 0x72, 0x22, 0x66, 0x2e, 0x64, 0x83, 0x7d, 0x21, 0x51, 0x2b,
    0x8d, 0x79, 0xe5, 0x53, 0x4b, 0x7d, 0xf6, 0x6c, 0x8d, 0x13, 0x8a, 0xcc,
    0x9b, 0xed, 0x8d, 0xe9, 0x25, 0xd7, 0x31, 0x9b, 0x49, 0x0b, 0xc2, 0x5e,
    0x5e, 0xa7, 0x48, 0xb8, 0xf9, 0x66, 0xbc, 0x4e, 0x1e, 0x5b, 0xfe, 0x08,
    0x1f, 0x5f, 0x29, 0xdf, 0xfa, 0x27, 0x08, 0xad, 0x40, 0xff, 0x07, 0xd8,
    0xb6, 0xe8, 0x7e, 0x03, 0xc2, 0xe2, 0xdd, 0x29, 0xb1, 0x8d, 0x4b, 0x68,
    0x51, 0x94, 0xb9, 0x72, 0xb2, 0x49, 0x20, 0xa1, 0x80, 0x16, 0x09, 0x20,
    0x83, 0xa6, 0x13, 0x70, 0x0a, 0x42, 0x62, 0x8c, 0xd6, 0x1e, 0x9f, 0x64,
    0x18, 0x41, 0x48, 0x49, 0xe5, 0xcb, 0x3b, 0xd2, 0x86, 0x48, 0x92, 0x83,
    0x69, 0xc9, 0xa9, 0x99, 0xe6, 0xc7, 0xdc, 0x08, 0xee, 0xdc, 0x64, 0x43,
    0x42, 0xb7, 0x49, 0x39, 0x4b, 0x0d, 0x3a, 0xfc, 0x73, 0x63, 0xa7, 0x65,
    0x61, 0x9e, 0x45, 0xfd, 0x72, 0x0f, 0x6c, 0xef, 0x1a, 0x1d, 0xa7, 0xdc,
    0x81, 0xfd, 0x03, 0x62, 0x2a, 0x55, 0xbf, 0x88, 0x09, 0xf7, 0x1e, 0xd9,
    0xbc, 0xdd, 0x62, 0x33, 0xe7, 0xa0, 0xd5, 0xfa, 0x55, 0xca, 0xa0, 0xb8,
    0x47, 0xc7, 0xf4, 0xbc, 0x15, 0x98, 0x7d, 0x63, 0xf4, 0x71, 0xc0, 0x25,
    0x34, 0x96, 0x0f, 0xb5, 0xeb, 0xa9, 0x2e, 0x0c, 0xbf, 0x12, 0x99, 0xc0,
    0xbd, 0x0e, 0x65, 0xa3, 0xad, 0x77, 0x75, 0xc5, 0x99, 0xeb, 0x30, 0xe9,
    0x65, 0x90, 0xbc, 0x7e,
  };

  ffdhe2048_value = BN_new();
  if (ffdhe2048_value) {
    bn_set_static_words(ffdhe2048_value, kFFDHE2048PublicValueData,
                        OPENSSL_ARRAY_SIZE(kFFDHE2048PublicValueData));
  }

  dh = self_test_dh();
  uint8_t dh_out[sizeof(kDHOutput)];
  if (dh == NULL || ffdhe2048_value == NULL || sizeof(dh_out) != DH_size(dh) ||
      dh_compute_key_padded_no_self_test(dh_out, ffdhe2048_value, dh) !=
          sizeof(dh_out) ||
      !check_test(kDHOutput, dh_out, sizeof(dh_out), "FFC DH")) {
    fprintf(stderr, "FFDH failed.\n");
    goto err;
  }

  // FFC Diffie-Hellman KAT with FB domain parameters.
  set_bignum(&fb_peers_key, kDH_fb_peer_public, sizeof(kDH_fb_peer_public));
  fb_dh = self_test_ffc_dh_fb_key();
  uint8_t fb_dh_out[sizeof(kDH_fb_z)];
  if (fb_dh == NULL || fb_peers_key == NULL ||
      sizeof(fb_dh_out) != DH_size(fb_dh) ||
      dh_compute_key_padded_no_self_test(fb_dh_out, fb_peers_key, fb_dh) !=
          sizeof(fb_dh_out) ||
      !check_test(kDH_fb_z, fb_dh_out, sizeof(fb_dh_out), "FFC DH FB")) {
    fprintf(stderr, "FFDH FB failed.\n");
    goto err;
  }

  ret = 1;

err:
  DH_free(dh);
  BN_free(ffdhe2048_value);
  DH_free(fb_dh);
  BN_free(fb_peers_key);

  return ret;
}

static int boringssl_self_test_ml_kem(void) {
  int ret = 0;

  const uint8_t kKeyGenSeed[MLKEM512_KEYGEN_SEED_LEN] = {
      0xec, 0xdb, 0x97, 0x62, 0xf4, 0x78, 0xb2, 0xfa, 0x26, 0x3d, 0xf4,
      0x6d, 0xe4, 0x47, 0xf3, 0xd1, 0x52, 0xa1, 0xbc, 0x0e, 0x02, 0xee,
      0x95, 0x36, 0x77, 0x30, 0x11, 0x64, 0xd1, 0x5d, 0x20, 0xd7, 0x1b,
      0x07, 0x4b, 0xff, 0x80, 0x44, 0x44, 0x5e, 0x11, 0x66, 0x0b, 0x1b,
      0x6b, 0x26, 0xdf, 0x24, 0x2b, 0x8f, 0xc0, 0x2b, 0x9e, 0x8d, 0xf5,
      0x38, 0xdb, 0x17, 0xa6, 0x39, 0xd7, 0xc4, 0x61, 0x32
  };
  const uint8_t kKeyGenEK[MLKEM512_PUBLIC_KEY_BYTES] = {
      0xe9, 0xe0, 0x22, 0x0a, 0x69, 0x45, 0x56, 0xbc, 0x0a, 0x4b, 0x5a, 0xad,
      0xe0, 0x32, 0xb2, 0xaa, 0x87, 0x7c, 0x0d, 0xa8, 0x16, 0xef, 0x32, 0xa8,
      0x4d, 0x77, 0xbb, 0xff, 0x46, 0x56, 0xe6, 0x80, 0x85, 0x4e, 0x69, 0x61,
      0xa4, 0x17, 0x08, 0x2b, 0xd4, 0x6f, 0x50, 0x23, 0xc4, 0x7c, 0xea, 0x09,
      0xbb, 0xa2, 0x14, 0x5d, 0x41, 0x51, 0x88, 0xe1, 0x38, 0x03, 0xbb, 0x5e,
      0x7f, 0xc1, 0x9d, 0x39, 0x81, 0x9e, 0x59, 0xca, 0x7f, 0xdc, 0x38, 0x05,
      0x1d, 0x96, 0x94, 0x01, 0xd4, 0xc4, 0x11, 0xda, 0xc0, 0x0a, 0x47, 0xab,
      0x57, 0x52, 0x7d, 0xaa, 0x89, 0x7a, 0x37, 0x75, 0x93, 0x51, 0xb4, 0xb1,
      0x18, 0x5b, 0x14, 0x96, 0xd7, 0x9a, 0xa3, 0x32, 0x29, 0xbe, 0xa3, 0xc0,
      0xe8, 0xdb, 0xcf, 0x1d, 0x7a, 0x64, 0x73, 0x09, 0x28, 0x77, 0x5c, 0x74,
      0xc9, 0x46, 0x18, 0x1b, 0xb8, 0x8e, 0x0a, 0x38, 0xa1, 0x65, 0xeb, 0xbd,
      0xc7, 0x86, 0x0b, 0x01, 0x4c, 0x59, 0x89, 0x4c, 0x72, 0x11, 0xbc, 0x23,
      0xd3, 0x90, 0x77, 0xe6, 0x29, 0x44, 0xd3, 0xc4, 0x8c, 0x43, 0x4b, 0x73,
      0x7b, 0x1a, 0x64, 0xb5, 0xf9, 0xa1, 0x64, 0x6a, 0x18, 0xa5, 0x19, 0x5a,
      0x54, 0x60, 0x3f, 0xfc, 0x28, 0x0b, 0xf5, 0xea, 0x1b, 0xf3, 0xd8, 0xcb,
      0x41, 0x12, 0xaa, 0x82, 0xfa, 0xb8, 0x9e, 0x26, 0x33, 0x66, 0x55, 0x97,
      0xbe, 0xd3, 0x0d, 0xe6, 0xc8, 0x89, 0x7a, 0x96, 0x13, 0xea, 0x8c, 0x11,
      0x73, 0x65, 0x24, 0xa7, 0x08, 0x62, 0x66, 0x52, 0x3f, 0xba, 0xb8, 0x87,
      0x46, 0x54, 0x94, 0xeb, 0xfc, 0x96, 0xfb, 0xe4, 0x46, 0x67, 0x75, 0xbc,
      0x0d, 0x70, 0x7e, 0x62, 0x4b, 0xb3, 0xde, 0x70, 0x0e, 0xaa, 0xc9, 0xb3,
      0x7c, 0x2b, 0xc1, 0x2b, 0x52, 0xaa, 0x2d, 0xfb, 0x73, 0xb7, 0x00, 0x6b,
      0xd9, 0x56, 0xc0, 0xb9, 0x32, 0x4c, 0x47, 0x29, 0x5e, 0x4e, 0x4c, 0x4c,
      0xc4, 0x7b, 0xa9, 0xb5, 0xa8, 0x99, 0x69, 0x4b, 0x38, 0xa6, 0x4a, 0x29,
      0x12, 0x06, 0x84, 0xbf, 0x64, 0xc6, 0x48, 0x21, 0x1b, 0x18, 0xd0, 0x17,
      0xce, 0x34, 0x84, 0x9c, 0x4a, 0x09, 0x2b, 0x24, 0xbc, 0xb8, 0x96, 0x0d,
      0x87, 0x04, 0x22, 0xeb, 0xd9, 0x54, 0xbc, 0x72, 0x64, 0xba, 0x95, 0xa7,
      0x27, 0xa3, 0x19, 0x26, 0xf3, 0x1c, 0x12, 0x38, 0x1d, 0x0d, 0x05, 0x1e,
      0x1e, 0x04, 0x19, 0x0b, 0x94, 0x93, 0x22, 0x44, 0x28, 0x45, 0xf3, 0xaa,
      0x6f, 0xb1, 0x79, 0x40, 0x44, 0x2a, 0x9c, 0x1a, 0x18, 0x3b, 0x23, 0x97,
      0xdc, 0xc0, 0x77, 0x0c, 0xb2, 0xb4, 0xa0, 0x7a, 0xaf, 0x0a, 0xf5, 0x26,
      0x36, 0x77, 0x8a, 0xc9, 0x94, 0x9d, 0x2d, 0xb4, 0x2f, 0x3d, 0x03, 0x6b,
      0xc8, 0x42, 0xb2, 0xf3, 0xb0, 0xb1, 0x1d, 0x27, 0x8e, 0xa0, 0x25, 0xcc,
      0x16, 0xac, 0x8b, 0x69, 0x86, 0xa9, 0xb1, 0x8b, 0x33, 0x56, 0x79, 0x0d,
      0xbc, 0x82, 0x1d, 0x0a, 0xc7, 0xb5, 0xea, 0xe9, 0x65, 0xfc, 0xf1, 0x4b,
      0x0e, 0x20, 0x2b, 0x00, 0xec, 0x5d, 0x70, 0x7c, 0x45, 0xaf, 0x52, 0xbe,
      0x2d, 0x97, 0x71, 0xbc, 0xc5, 0x3b, 0x31, 0xf2, 0x52, 0x5a, 0xb5, 0x3f,
      0x95, 0xa3, 0xb0, 0x02, 0x42, 0xc1, 0x1c, 0x49, 0x4e, 0x25, 0x54, 0x43,
      0x8b, 0x0a, 0x4f, 0x68, 0x19, 0x40, 0x19, 0x18, 0x1b, 0xbd, 0xb1, 0x94,
      0x91, 0x03, 0x07, 0xbc, 0x19, 0xaf, 0xae, 0x55, 0x3d, 0xd6, 0x66, 0x56,
      0x43, 0x32, 0xc0, 0x78, 0x60, 0x4f, 0xbd, 0x8b, 0xc4, 0xfd, 0xc9, 0x2f,
      0x92, 0x01, 0x80, 0x9a, 0xb4, 0x95, 0xc1, 0xb5, 0x35, 0x53, 0x0b, 0xad,
      0x2e, 0xb3, 0x57, 0x53, 0xf9, 0x19, 0x56, 0xb5, 0x8b, 0xfb, 0xf6, 0x2d,
      0x1e, 0x33, 0xc8, 0x61, 0xf7, 0x38, 0x46, 0x33, 0x4c, 0x4a, 0x2a, 0xc5,
      0x57, 0x69, 0x84, 0xef, 0x11, 0x63, 0xa5, 0x04, 0x54, 0xa3, 0xe4, 0xce,
      0x83, 0x88, 0x98, 0xf2, 0x25, 0x86, 0x81, 0x89, 0xbd, 0xae, 0x5a, 0x03,
      0xef, 0xd8, 0x00, 0x1d, 0x2a, 0x9f, 0x55, 0x7c, 0xb6, 0xcd, 0x93, 0xa4,
      0x6f, 0xbb, 0x13, 0x79, 0xd8, 0x16, 0x45, 0xb8, 0x53, 0xcb, 0x98, 0xb4,
      0x7b, 0xa8, 0x7a, 0x66, 0xfc, 0xbc, 0x0a, 0xda, 0x76, 0x50, 0x6b, 0x8d,
      0xcd, 0x65, 0x90, 0x42, 0x64, 0x0e, 0xf6, 0x16, 0x82, 0x0f, 0xf2, 0x0b,
      0x26, 0x43, 0x33, 0x99, 0x37, 0x1d, 0xc4, 0xc5, 0x02, 0x87, 0x34, 0x9c,
      0xa2, 0x76, 0x3e, 0x30, 0x73, 0x86, 0x5c, 0x5b, 0xc9, 0xaa, 0xa1, 0x43,
      0x7b, 0x86, 0xb2, 0x0b, 0x25, 0xb6, 0xf0, 0x88, 0xbd, 0x6a, 0x4c, 0x94,
      0x83, 0x84, 0x2d, 0x7c, 0x65, 0x25, 0x7c, 0xc4, 0x2a, 0xd7, 0xc7, 0xb2,
      0x6f, 0x58, 0xc8, 0xed, 0x19, 0x9b, 0x84, 0x8b, 0x8c, 0x6f, 0x40, 0x8d,
      0x48, 0x76, 0x37, 0x75, 0x18, 0x88, 0xa9, 0x5a, 0x44, 0x9c, 0xab, 0x47,
      0xb7, 0xa2, 0xac, 0xe6, 0xa8, 0x2c, 0x7d, 0x46, 0x91, 0x05, 0x26, 0x41,
      0x4a, 0xb1, 0x49, 0xb1, 0x13, 0x43, 0xb0, 0xea, 0x32, 0xd9, 0x69, 0xa5,
      0x08, 0xa8, 0x1c, 0xf8, 0x96, 0xb0, 0x81, 0x08, 0x7f, 0x4d, 0x52, 0xa2,
      0xa9, 0xe0, 0x77, 0x87, 0x8a, 0x43, 0xe5, 0x97, 0x1d, 0x74, 0x86, 0x3a,
      0x7b, 0xf2, 0x00, 0x37, 0xb2, 0x97, 0x47, 0x03, 0x14, 0x0e, 0xa8, 0x16,
      0x0a, 0x5b, 0xa7, 0x22, 0x27, 0xc1, 0x5b, 0xec, 0x5d, 0xf1, 0x71, 0x14,
      0x01, 0x98, 0x89, 0x6d, 0x31, 0x73, 0x2a, 0x24, 0x1f, 0x5b, 0x93, 0x8f,
      0xf2, 0x33, 0xcb, 0xa2, 0xdc, 0x82, 0xde, 0x91, 0xc6, 0xb4, 0x93, 0x78,
      0x2b, 0x61, 0x63, 0xbc, 0x12, 0x82, 0xdf, 0x85, 0x5e, 0xe6, 0xa6, 0xa6,
      0x59, 0x75, 0xb3, 0x3c, 0xb6, 0xa2, 0x7d, 0x25, 0x8b, 0xd3, 0x17, 0xd0,
      0x4c, 0xef, 0x8e, 0xb5, 0x57, 0xba, 0x02, 0xd4, 0x94, 0x71, 0x92, 0x3c,
      0xd6, 0x0e, 0x99, 0x17, 0x96, 0x6b, 0xe9, 0x0f};
  const uint8_t kKeyGenDK[MLKEM512_SECRET_KEY_BYTES] = {
      0x88, 0xc1, 0x2c, 0xea, 0xa6, 0xcb, 0x91, 0xf5, 0x89, 0xac, 0xb8, 0x6d,
      0x91, 0x3c, 0x7a, 0x60, 0xf7, 0xcd, 0xab, 0xe3, 0xb7, 0xb5, 0x90, 0x09,
      0x1d, 0x00, 0x84, 0xe2, 0x9a, 0x04, 0x9b, 0x43, 0x68, 0x41, 0xf2, 0x47,
      0x3b, 0x03, 0x16, 0x5a, 0xe9, 0xc6, 0xa9, 0x82, 0x6d, 0x6c, 0x65, 0x0d,
      0x04, 0xb3, 0x88, 0xef, 0xf5, 0x94, 0x50, 0x5b, 0x7e, 0x54, 0x70, 0x95,
      0x30, 0x54, 0x68, 0x25, 0xa0, 0x70, 0xa6, 0x25, 0xb0, 0xe5, 0xfa, 0x86,
      0x6e, 0x6a, 0xaf, 0x40, 0xc2, 0x41, 0x42, 0x46, 0x24, 0x09, 0x73, 0xc7,
      0x59, 0x8a, 0xae, 0x7c, 0x36, 0x3e, 0x43, 0x03, 0xab, 0xb7, 0xa1, 0x11,
      0x31, 0xb4, 0x64, 0xa9, 0x43, 0x99, 0x6d, 0xe7, 0x59, 0x2c, 0xa0, 0x49,
      0x22, 0xea, 0x8a, 0x4d, 0x73, 0xb4, 0x43, 0xea, 0x04, 0x8c, 0x06, 0xac,
      0xc4, 0xe5, 0x5a, 0x8f, 0x25, 0x4b, 0xf6, 0xd2, 0x71, 0xfd, 0x82, 0x71,
      0x19, 0xec, 0x5b, 0x55, 0x80, 0x49, 0x8b, 0xfc, 0xc0, 0x9e, 0xb0, 0x26,
      0x6f, 0x8c, 0x2b, 0x45, 0x98, 0x8a, 0xe9, 0x8c, 0x1e, 0x54, 0x02, 0xb7,
      0x00, 0x34, 0x63, 0xf2, 0x03, 0x59, 0x47, 0x01, 0x59, 0xc0, 0x50, 0x9f,
      0xa9, 0x71, 0x15, 0x34, 0x43, 0xce, 0x25, 0x80, 0xc0, 0xb2, 0x44, 0x3f,
      0x8a, 0xc2, 0xb0, 0x81, 0x04, 0x01, 0xe7, 0x30, 0x52, 0xd6, 0x26, 0xbf,
      0x58, 0xc6, 0x74, 0xee, 0x48, 0x88, 0x0c, 0x40, 0x8d, 0x1f, 0x31, 0x3a,
      0x94, 0xb1, 0x66, 0x7f, 0x89, 0x76, 0x28, 0xc5, 0x5a, 0x83, 0xe2, 0x86,
      0x34, 0xa2, 0x07, 0x10, 0xd2, 0x5d, 0xa8, 0x8b, 0x2a, 0xc9, 0x0c, 0x0f,
      0x5d, 0x6b, 0x0e, 0xd6, 0xe0, 0x80, 0xfd, 0x2c, 0x24, 0xbb, 0x11, 0x81,
      0x6b, 0x5c, 0x60, 0x79, 0x57, 0x78, 0x1d, 0xb2, 0x28, 0x79, 0x66, 0x71,
      0x7f, 0xfa, 0x50, 0x0a, 0x03, 0x02, 0x58, 0x39, 0x16, 0x41, 0x15, 0xba,
      0x5e, 0xa7, 0xb7, 0x17, 0x34, 0x45, 0x88, 0x16, 0x9e, 0x8a, 0x85, 0xa9,
      0xe8, 0x3c, 0x51, 0x6c, 0xab, 0x5e, 0xe6, 0xe6, 0x3b, 0x7c, 0x73, 0x6c,
      0xe9, 0x06, 0x28, 0xdd, 0xec, 0xa9, 0x9a, 0x6b, 0xc5, 0x47, 0x58, 0x86,
      0x81, 0xbc, 0x22, 0x23, 0xf3, 0x9f, 0xbc, 0x44, 0x64, 0xb4, 0x1c, 0x63,
      0xd9, 0x24, 0xaf, 0xeb, 0xa1, 0x09, 0x89, 0x60, 0xbd, 0x43, 0xc1, 0x2f,
      0xd0, 0x08, 0x0f, 0x49, 0x0c, 0x8c, 0x50, 0xf4, 0x84, 0x32, 0x22, 0x30,
      0x3c, 0x68, 0x21, 0xad, 0x7b, 0x66, 0x3d, 0x88, 0x44, 0xa1, 0x36, 0x8a,
      0x19, 0x54, 0x9e, 0x5c, 0xfc, 0x83, 0xd1, 0x4a, 0x84, 0x91, 0x20, 0x5c,
      0x44, 0xf5, 0x49, 0x2b, 0xf6, 0x09, 0xb6, 0x9c, 0x52, 0x6c, 0x95, 0xaa,
      0x5e, 0x50, 0x8e, 0xcf, 0x71, 0x4b, 0x79, 0x55, 0xb6, 0x1f, 0x6c, 0xab,
      0x34, 0x0b, 0x06, 0x20, 0x04, 0x32, 0xce, 0x3b, 0x1e, 0xcd, 0x0a, 0x07,
      0x5a, 0x83, 0x32, 0x04, 0x68, 0x65, 0x95, 0x9a, 0x25, 0x69, 0x8a, 0x08,
      0x1d, 0xf3, 0x53, 0x26, 0x91, 0x34, 0x8a, 0x25, 0x81, 0xd6, 0x06, 0xa0,
      0xe0, 0xb2, 0xcf, 0x2e, 0x8b, 0x3c, 0x9a, 0xfc, 0x59, 0x4e, 0x92, 0x6e,
      0xb0, 0xd7, 0xc7, 0x02, 0xc0, 0xc3, 0xe5, 0xb5, 0x4e, 0xe2, 0xc8, 0x6c,
      0xa2, 0x0c, 0xc4, 0x76, 0xaa, 0xac, 0xa3, 0x20, 0x8f, 0x32, 0xa2, 0x01,
      0x90, 0x56, 0x2c, 0x27, 0x33, 0x82, 0x02, 0x07, 0x1e, 0x11, 0x66, 0x5f,
      0x13, 0x42, 0x73, 0xbd, 0xaa, 0xcb, 0xc9, 0x52, 0xb9, 0x94, 0xba, 0x94,
      0x62, 0x67, 0x12, 0x26, 0x51, 0x4b, 0x44, 0x61, 0x13, 0xe7, 0xab, 0x7e,
      0xdb, 0x9c, 0x54, 0xc3, 0x11, 0xc4, 0xda, 0x94, 0x10, 0x4d, 0x26, 0x2a,
      0x80, 0x28, 0x0e, 0x39, 0x20, 0x1e, 0x75, 0x51, 0x91, 0x76, 0x39, 0x83,
      0xc4, 0x39, 0xa9, 0x5a, 0xea, 0xaf, 0xa7, 0x67, 0xc2, 0xcb, 0x59, 0x48,
      0x29, 0xe6, 0x31, 0x3e, 0x38, 0x69, 0x82, 0xd6, 0x62, 0x1a, 0xcc, 0x4b,
      0xb0, 0x99, 0x1a, 0x60, 0x79, 0x0a, 0x2b, 0x0c, 0x5f, 0x31, 0x39, 0xaa,
      0xdd, 0x70, 0x45, 0xf7, 0xd8, 0x49, 0xaa, 0x20, 0x9b, 0xf6, 0x0b, 0xc1,
      0x5e, 0xd8, 0x26, 0x67, 0x41, 0x4b, 0x70, 0xb7, 0x1a, 0x7e, 0x18, 0x4e,
      0x16, 0x42, 0x80, 0xaf, 0x00, 0xbf, 0x95, 0xa9, 0xad, 0x3d, 0xe4, 0x1d,
      0xcf, 0x19, 0x62, 0x36, 0x10, 0xcf, 0xb3, 0x06, 0x87, 0xa5, 0xa0, 0x82,
      0xa2, 0x45, 0x88, 0x70, 0xeb, 0x33, 0xd6, 0x49, 0xb3, 0xfc, 0xe3, 0x31,
      0x7e, 0x03, 0x62, 0xee, 0x61, 0x75, 0xfb, 0x81, 0x1c, 0x7f, 0xe3, 0x64,
      0x7c, 0xa2, 0x10, 0x20, 0x67, 0x3a, 0xed, 0xd2, 0x3b, 0xf0, 0x47, 0x0c,
      0xd8, 0x73, 0x5c, 0xe5, 0x3a, 0x78, 0x08, 0x26, 0x68, 0xe4, 0x9a, 0x51,
      0x32, 0xd1, 0x33, 0x6a, 0x08, 0x43, 0x60, 0x18, 0x7f, 0xcf, 0xf9, 0x22,
      0x7c, 0xdc, 0x0d, 0x7f, 0x20, 0x5b, 0x2a, 0xf2, 0xc8, 0x8d, 0xc6, 0xbc,
      0x9a, 0x04, 0xb4, 0x1f, 0x42, 0x9f, 0xa9, 0xa3, 0x86, 0xf5, 0x8b, 0x8f,
      0x21, 0x6c, 0xbf, 0x67, 0x14, 0xa6, 0x19, 0x59, 0x7c, 0xaa, 0x5a, 0x9b,
      0xf6, 0x68, 0x04, 0x8d, 0x71, 0x35, 0x10, 0x5a, 0x94, 0x25, 0xcb, 0x46,
      0xfe, 0x80, 0x7a, 0x40, 0x64, 0x59, 0xb0, 0x67, 0xb7, 0x16, 0xf5, 0xa4,
      0x77, 0x30, 0x97, 0x4b, 0x20, 0x78, 0x26, 0x71, 0x6f, 0xe3, 0xf7, 0x47,
      0xfd, 0x74, 0xcd, 0x8c, 0x02, 0x57, 0x20, 0xb1, 0x00, 0x3f, 0x27, 0xb2,
      0xde, 0xa6, 0x95, 0xf2, 0x21, 0x97, 0x1c, 0xd5, 0xaf, 0x81, 0xa1, 0xa7,
      0x95, 0x00, 0x1f, 0x12, 0x5c, 0x28, 0x5c, 0x15, 0x0d, 0xdc, 0x99, 0x59,
      0xe9, 0xe0, 0x22, 0x0a, 0x69, 0x45, 0x56, 0xbc, 0x0a, 0x4b, 0x5a, 0xad,
      0xe0, 0x32, 0xb2, 0xaa, 0x87, 0x7c, 0x0d, 0xa8, 0x16, 0xef, 0x32, 0xa8,
      0x4d, 0x77, 0xbb, 0xff, 0x46, 0x56, 0xe6, 0x80, 0x85, 0x4e, 0x69, 0x61,
      0xa4, 0x17, 0x08, 0x2b, 0xd4, 0x6f, 0x50, 0x23, 0xc4, 0x7c, 0xea, 0x09,
      0xbb, 0xa2, 0x14, 0x5d, 0x41, 0x51, 0x88, 0xe1, 0x38, 0x03, 0xbb, 0x5e,
      0x7f, 0xc1, 0x9d, 0x39, 0x81, 0x9e, 0x59, 0xca, 0x7f, 0xdc, 0x38, 0x05,
      0x1d, 0x96, 0x94, 0x01, 0xd4, 0xc4, 0x11, 0xda, 0xc0, 0x0a, 0x47, 0xab,
      0x57, 0x52, 0x7d, 0xaa, 0x89, 0x7a, 0x37, 0x75, 0x93, 0x51, 0xb4, 0xb1,
      0x18, 0x5b, 0x14, 0x96, 0xd7, 0x9a, 0xa3, 0x32, 0x29, 0xbe, 0xa3, 0xc0,
      0xe8, 0xdb, 0xcf, 0x1d, 0x7a, 0x64, 0x73, 0x09, 0x28, 0x77, 0x5c, 0x74,
      0xc9, 0x46, 0x18, 0x1b, 0xb8, 0x8e, 0x0a, 0x38, 0xa1, 0x65, 0xeb, 0xbd,
      0xc7, 0x86, 0x0b, 0x01, 0x4c, 0x59, 0x89, 0x4c, 0x72, 0x11, 0xbc, 0x23,
      0xd3, 0x90, 0x77, 0xe6, 0x29, 0x44, 0xd3, 0xc4, 0x8c, 0x43, 0x4b, 0x73,
      0x7b, 0x1a, 0x64, 0xb5, 0xf9, 0xa1, 0x64, 0x6a, 0x18, 0xa5, 0x19, 0x5a,
      0x54, 0x60, 0x3f, 0xfc, 0x28, 0x0b, 0xf5, 0xea, 0x1b, 0xf3, 0xd8, 0xcb,
      0x41, 0x12, 0xaa, 0x82, 0xfa, 0xb8, 0x9e, 0x26, 0x33, 0x66, 0x55, 0x97,
      0xbe, 0xd3, 0x0d, 0xe6, 0xc8, 0x89, 0x7a, 0x96, 0x13, 0xea, 0x8c, 0x11,
      0x73, 0x65, 0x24, 0xa7, 0x08, 0x62, 0x66, 0x52, 0x3f, 0xba, 0xb8, 0x87,
      0x46, 0x54, 0x94, 0xeb, 0xfc, 0x96, 0xfb, 0xe4, 0x46, 0x67, 0x75, 0xbc,
      0x0d, 0x70, 0x7e, 0x62, 0x4b, 0xb3, 0xde, 0x70, 0x0e, 0xaa, 0xc9, 0xb3,
      0x7c, 0x2b, 0xc1, 0x2b, 0x52, 0xaa, 0x2d, 0xfb, 0x73, 0xb7, 0x00, 0x6b,
      0xd9, 0x56, 0xc0, 0xb9, 0x32, 0x4c, 0x47, 0x29, 0x5e, 0x4e, 0x4c, 0x4c,
      0xc4, 0x7b, 0xa9, 0xb5, 0xa8, 0x99, 0x69, 0x4b, 0x38, 0xa6, 0x4a, 0x29,
      0x12, 0x06, 0x84, 0xbf, 0x64, 0xc6, 0x48, 0x21, 0x1b, 0x18, 0xd0, 0x17,
      0xce, 0x34, 0x84, 0x9c, 0x4a, 0x09, 0x2b, 0x24, 0xbc, 0xb8, 0x96, 0x0d,
      0x87, 0x04, 0x22, 0xeb, 0xd9, 0x54, 0xbc, 0x72, 0x64, 0xba, 0x95, 0xa7,
      0x27, 0xa3, 0x19, 0x26, 0xf3, 0x1c, 0x12, 0x38, 0x1d, 0x0d, 0x05, 0x1e,
      0x1e, 0x04, 0x19, 0x0b, 0x94, 0x93, 0x22, 0x44, 0x28, 0x45, 0xf3, 0xaa,
      0x6f, 0xb1, 0x79, 0x40, 0x44, 0x2a, 0x9c, 0x1a, 0x18, 0x3b, 0x23, 0x97,
      0xdc, 0xc0, 0x77, 0x0c, 0xb2, 0xb4, 0xa0, 0x7a, 0xaf, 0x0a, 0xf5, 0x26,
      0x36, 0x77, 0x8a, 0xc9, 0x94, 0x9d, 0x2d, 0xb4, 0x2f, 0x3d, 0x03, 0x6b,
      0xc8, 0x42, 0xb2, 0xf3, 0xb0, 0xb1, 0x1d, 0x27, 0x8e, 0xa0, 0x25, 0xcc,
      0x16, 0xac, 0x8b, 0x69, 0x86, 0xa9, 0xb1, 0x8b, 0x33, 0x56, 0x79, 0x0d,
      0xbc, 0x82, 0x1d, 0x0a, 0xc7, 0xb5, 0xea, 0xe9, 0x65, 0xfc, 0xf1, 0x4b,
      0x0e, 0x20, 0x2b, 0x00, 0xec, 0x5d, 0x70, 0x7c, 0x45, 0xaf, 0x52, 0xbe,
      0x2d, 0x97, 0x71, 0xbc, 0xc5, 0x3b, 0x31, 0xf2, 0x52, 0x5a, 0xb5, 0x3f,
      0x95, 0xa3, 0xb0, 0x02, 0x42, 0xc1, 0x1c, 0x49, 0x4e, 0x25, 0x54, 0x43,
      0x8b, 0x0a, 0x4f, 0x68, 0x19, 0x40, 0x19, 0x18, 0x1b, 0xbd, 0xb1, 0x94,
      0x91, 0x03, 0x07, 0xbc, 0x19, 0xaf, 0xae, 0x55, 0x3d, 0xd6, 0x66, 0x56,
      0x43, 0x32, 0xc0, 0x78, 0x60, 0x4f, 0xbd, 0x8b, 0xc4, 0xfd, 0xc9, 0x2f,
      0x92, 0x01, 0x80, 0x9a, 0xb4, 0x95, 0xc1, 0xb5, 0x35, 0x53, 0x0b, 0xad,
      0x2e, 0xb3, 0x57, 0x53, 0xf9, 0x19, 0x56, 0xb5, 0x8b, 0xfb, 0xf6, 0x2d,
      0x1e, 0x33, 0xc8, 0x61, 0xf7, 0x38, 0x46, 0x33, 0x4c, 0x4a, 0x2a, 0xc5,
      0x57, 0x69, 0x84, 0xef, 0x11, 0x63, 0xa5, 0x04, 0x54, 0xa3, 0xe4, 0xce,
      0x83, 0x88, 0x98, 0xf2, 0x25, 0x86, 0x81, 0x89, 0xbd, 0xae, 0x5a, 0x03,
      0xef, 0xd8, 0x00, 0x1d, 0x2a, 0x9f, 0x55, 0x7c, 0xb6, 0xcd, 0x93, 0xa4,
      0x6f, 0xbb, 0x13, 0x79, 0xd8, 0x16, 0x45, 0xb8, 0x53, 0xcb, 0x98, 0xb4,
      0x7b, 0xa8, 0x7a, 0x66, 0xfc, 0xbc, 0x0a, 0xda, 0x76, 0x50, 0x6b, 0x8d,
      0xcd, 0x65, 0x90, 0x42, 0x64, 0x0e, 0xf6, 0x16, 0x82, 0x0f, 0xf2, 0x0b,
      0x26, 0x43, 0x33, 0x99, 0x37, 0x1d, 0xc4, 0xc5, 0x02, 0x87, 0x34, 0x9c,
      0xa2, 0x76, 0x3e, 0x30, 0x73, 0x86, 0x5c, 0x5b, 0xc9, 0xaa, 0xa1, 0x43,
      0x7b, 0x86, 0xb2, 0x0b, 0x25, 0xb6, 0xf0, 0x88, 0xbd, 0x6a, 0x4c, 0x94,
      0x83, 0x84, 0x2d, 0x7c, 0x65, 0x25, 0x7c, 0xc4, 0x2a, 0xd7, 0xc7, 0xb2,
      0x6f, 0x58, 0xc8, 0xed, 0x19, 0x9b, 0x84, 0x8b, 0x8c, 0x6f, 0x40, 0x8d,
      0x48, 0x76, 0x37, 0x75, 0x18, 0x88, 0xa9, 0x5a, 0x44, 0x9c, 0xab, 0x47,
      0xb7, 0xa2, 0xac, 0xe6, 0xa8, 0x2c, 0x7d, 0x46, 0x91, 0x05, 0x26, 0x41,
      0x4a, 0xb1, 0x49, 0xb1, 0x13, 0x43, 0xb0, 0xea, 0x32, 0xd9, 0x69, 0xa5,
      0x08, 0xa8, 0x1c, 0xf8, 0x96, 0xb0, 0x81, 0x08, 0x7f, 0x4d, 0x52, 0xa2,
      0xa9, 0xe0, 0x77, 0x87, 0x8a, 0x43, 0xe5, 0x97, 0x1d, 0x74, 0x86, 0x3a,
      0x7b, 0xf2, 0x00, 0x37, 0xb2, 0x97, 0x47, 0x03, 0x14, 0x0e, 0xa8, 0x16,
      0x0a, 0x5b, 0xa7, 0x22, 0x27, 0xc1, 0x5b, 0xec, 0x5d, 0xf1, 0x71, 0x14,
      0x01, 0x98, 0x89, 0x6d, 0x31, 0x73, 0x2a, 0x24, 0x1f, 0x5b, 0x93, 0x8f,
      0xf2, 0x33, 0xcb, 0xa2, 0xdc, 0x82, 0xde, 0x91, 0xc6, 0xb4, 0x93, 0x78,
      0x2b, 0x61, 0x63, 0xbc, 0x12, 0x82, 0xdf, 0x85, 0x5e, 0xe6, 0xa6, 0xa6,
      0x59, 0x75, 0xb3, 0x3c, 0xb6, 0xa2, 0x7d, 0x25, 0x8b, 0xd3, 0x17, 0xd0,
      0x4c, 0xef, 0x8e, 0xb5, 0x57, 0xba, 0x02, 0xd4, 0x94, 0x71, 0x92, 0x3c,
      0xd6, 0x0e, 0x99, 0x17, 0x96, 0x6b, 0xe9, 0x0f, 0x56, 0xd2, 0xd5, 0x3f,
      0xa3, 0xbc, 0x26, 0x29, 0x74, 0x0a, 0xbb, 0xd1, 0x67, 0x20, 0xa9, 0xa7,
      0x06, 0x9a, 0x64, 0xde, 0x7a, 0x26, 0x32, 0x88, 0x17, 0xd7, 0x4a, 0x7c,
      0x2f, 0x55, 0xa2, 0x46, 0x1b, 0x07, 0x4b, 0xff, 0x80, 0x44, 0x44, 0x5e,
      0x11, 0x66, 0x0b, 0x1b, 0x6b, 0x26, 0xdf, 0x24, 0x2b, 0x8f, 0xc0, 0x2b,
      0x9e, 0x8d, 0xf5, 0x38, 0xdb, 0x17, 0xa6, 0x39, 0xd7, 0xc4, 0x61, 0x32};

  uint8_t keygen_decaps[MLKEM512_SECRET_KEY_BYTES] = {0};
  uint8_t keygen_encaps[MLKEM512_PUBLIC_KEY_BYTES] = {0};

  if (ml_kem_512_keypair_deterministic_no_self_test(
          keygen_encaps, keygen_decaps, kKeyGenSeed) ||
      !check_test(kKeyGenDK, keygen_decaps, sizeof(keygen_decaps),
                  "ML-KEM keyGen decaps") ||
      !check_test(kKeyGenEK, keygen_encaps, sizeof(keygen_encaps),
                  "ML-KEM keyGen encaps")) {
    goto err;
  }

  const uint8_t kEncapEK[MLKEM512_PUBLIC_KEY_BYTES] = {
      0x57, 0xc3, 0xba, 0x4c, 0xd7, 0x81, 0xd8, 0x69, 0x0b, 0x4c, 0x39, 0x0d,
      0x9a, 0x58, 0xb3, 0x5d, 0x69, 0xa5, 0x2d, 0x52, 0xcd, 0x19, 0x01, 0x2a,
      0x25, 0xe1, 0x58, 0xa2, 0xc1, 0x9b, 0x75, 0x47, 0x0a, 0x03, 0x9a, 0x05,
      0xc5, 0x98, 0x33, 0xc5, 0xd2, 0xa8, 0x28, 0x0b, 0x33, 0xde, 0x95, 0xb6,
      0x0c, 0x1b, 0xb5, 0xc6, 0x33, 0xaf, 0x71, 0x13, 0x8f, 0xb0, 0x64, 0x28,
      0x14, 0x37, 0x12, 0xb7, 0xa0, 0x33, 0xeb, 0xaa, 0x3c, 0x0d, 0x6a, 0x84,
      0x12, 0x11, 0x20, 0x4c, 0x67, 0x56, 0x4b, 0xc5, 0x16, 0xf3, 0x43, 0x73,
      0xb3, 0x04, 0x70, 0xa1, 0xa5, 0x03, 0x3b, 0x85, 0x2a, 0x2c, 0x3a, 0x80,
      0xad, 0xfc, 0x8f, 0x8c, 0x17, 0x62, 0xfb, 0x06, 0x29, 0x0e, 0x55, 0x75,
      0x9d, 0x46, 0x6d, 0x67, 0xd2, 0x7e, 0xfd, 0x3b, 0xb8, 0xf6, 0x5c, 0x7f,
      0x1a, 0xd0, 0x11, 0xb9, 0x9a, 0xc6, 0x96, 0xe0, 0xba, 0x78, 0xf4, 0x56,
      0xc1, 0x91, 0x03, 0x30, 0x43, 0x0d, 0xa6, 0xcc, 0x91, 0x86, 0xcb, 0x21,
      0x9e, 0x47, 0xa8, 0xce, 0xfb, 0xc2, 0xbe, 0x58, 0x05, 0x52, 0xc3, 0x0a,
      0xf1, 0xe8, 0x73, 0x4c, 0x57, 0x2f, 0x48, 0x90, 0xc1, 0xec, 0x14, 0x22,
      0x5b, 0x2a, 0xc6, 0x41, 0xe6, 0xc2, 0x3c, 0x94, 0x94, 0x29, 0x84, 0x32,
      0x3f, 0xc5, 0x4a, 0xf9, 0x29, 0x9a, 0x72, 0xe9, 0x2d, 0x05, 0x68, 0x37,
      0xc4, 0x50, 0x54, 0x5a, 0x88, 0x49, 0x41, 0x7b, 0x40, 0xa3, 0x23, 0x68,
      0x62, 0xb4, 0x07, 0xd0, 0x9a, 0x70, 0x22, 0xdc, 0x54, 0x47, 0x55, 0x15,
      0x5b, 0x23, 0xaf, 0x0f, 0x30, 0x6d, 0xac, 0x83, 0x37, 0x34, 0x78, 0x03,
      0x41, 0x2c, 0xcd, 0x57, 0xdb, 0x4b, 0xeb, 0xf9, 0x78, 0x70, 0xd5, 0xb8,
      0x61, 0xdb, 0xac, 0x6c, 0x4a, 0x8d, 0x7f, 0xdb, 0xc9, 0x26, 0x92, 0x3f,
      0xae, 0xa1, 0x71, 0xbe, 0x17, 0x23, 0x54, 0xf5, 0x4c, 0x11, 0xb6, 0x8c,
      0x9f, 0xf6, 0xc0, 0xfe, 0x05, 0x86, 0x93, 0x24, 0x61, 0x16, 0x8b, 0xa9,
      0x01, 0x88, 0xc8, 0x6c, 0xba, 0xce, 0x80, 0x49, 0x82, 0xe1, 0x88, 0x12,
      0x16, 0x40, 0x11, 0x66, 0xc6, 0x5a, 0x55, 0x64, 0x25, 0xab, 0x21, 0x45,
      0x51, 0x6c, 0x76, 0x99, 0x03, 0x1d, 0xf5, 0x6b, 0x14, 0x78, 0x1b, 0xa2,
      0x44, 0x57, 0x29, 0x47, 0xf6, 0x3f, 0x7c, 0xe9, 0x35, 0x62, 0xb1, 0x09,
      0x01, 0x78, 0x04, 0x01, 0xe2, 0x44, 0x17, 0x47, 0x97, 0xcf, 0xe6, 0xab,
      0x5d, 0x5a, 0xca, 0xfd, 0x07, 0x48, 0x6d, 0x13, 0x36, 0x30, 0x63, 0x1e,
      0x3e, 0x00, 0x9f, 0xda, 0x23, 0xc4, 0x27, 0xf5, 0xb4, 0xc2, 0x05, 0x64,
      0x31, 0x47, 0xc0, 0xef, 0x17, 0xcc, 0x4a, 0x67, 0x4c, 0x96, 0x91, 0x77,
      0xb5, 0x0a, 0x37, 0x24, 0x25, 0xad, 0xbe, 0x65, 0x4a, 0x74, 0xca, 0xa8,
      0x87, 0xf8, 0x7c, 0xdd, 0xe4, 0x92, 0xec, 0x50, 0x27, 0x05, 0x60, 0x9b,
      0x9e, 0x31, 0x8a, 0xe6, 0xbb, 0x5c, 0xe4, 0xa9, 0xa2, 0x82, 0xf1, 0x24,
      0x44, 0x77, 0x9b, 0xba, 0xe0, 0x32, 0x0d, 0x12, 0x97, 0xd9, 0xcb, 0xbb,
      0x87, 0x61, 0x00, 0x03, 0x07, 0x89, 0xc9, 0x05, 0x21, 0x5f, 0x38, 0x69,
      0x03, 0xa9, 0x4e, 0xe6, 0x8c, 0xbc, 0x97, 0x73, 0x10, 0xcb, 0xa0, 0x2b,
      0x79, 0x37, 0x5c, 0x4a, 0x8a, 0x5e, 0xec, 0x55, 0x69, 0x52, 0x8a, 0x10,
      0x02, 0xc0, 0x8a, 0x90, 0x9b, 0xca, 0x05, 0xf1, 0x32, 0x3a, 0x34, 0xae,
      0xf2, 0x08, 0x1f, 0x66, 0x76, 0x93, 0x9b, 0x82, 0x6e, 0xb7, 0x74, 0x8b,
      0x80, 0xd4, 0x5a, 0x1e, 0xa1, 0x5a, 0x6f, 0xb3, 0x19, 0x63, 0xf6, 0x18,
      0x19, 0x96, 0x30, 0xa2, 0xd3, 0x30, 0x1a, 0xf1, 0x13, 0xde, 0xf4, 0x28,
      0xbf, 0x10, 0x75, 0x18, 0xfc, 0x30, 0x55, 0xf2, 0x4d, 0x28, 0xa3, 0x8c,
      0xb4, 0x54, 0x43, 0x48, 0x16, 0x0b, 0xc6, 0x30, 0x1d, 0xd3, 0xe7, 0x60,
      0x82, 0x03, 0x9c, 0x12, 0xb2, 0x92, 0x51, 0xe9, 0x03, 0xc6, 0x8c, 0x7b,
      0xd8, 0x41, 0xa4, 0x53, 0x5c, 0x5d, 0xd7, 0xc9, 0x5a, 0xac, 0xfa, 0x80,
      0x66, 0xf7, 0x3b, 0xa8, 0xc9, 0xac, 0xa4, 0xd3, 0x03, 0xa9, 0xf1, 0xb7,
      0x6f, 0xc0, 0x36, 0x92, 0xa9, 0xbf, 0x11, 0x15, 0xb8, 0x51, 0x86, 0x62,
      0x48, 0x66, 0x1c, 0xfa, 0xf5, 0x39, 0xc1, 0x39, 0x9e, 0x5d, 0xe0, 0x3b,
      0x86, 0xb0, 0x2c, 0xb9, 0x9b, 0x2d, 0xa7, 0xd0, 0x07, 0xce, 0x35, 0x43,
      0xd4, 0xbb, 0x94, 0xfb, 0x1c, 0xc9, 0x9e, 0x40, 0xcd, 0xe8, 0x71, 0x76,
      0x75, 0x29, 0x14, 0x4d, 0xa3, 0x29, 0x79, 0x47, 0x55, 0x56, 0x45, 0x33,
      0x3e, 0xf0, 0x58, 0x37, 0x70, 0x98, 0xb2, 0xa1, 0x51, 0xd5, 0x11, 0x0f,
      0xce, 0x09, 0x21, 0x7d, 0x5a, 0x00, 0x55, 0xa2, 0x08, 0xf3, 0xb5, 0x83,
      0xe2, 0xc7, 0x13, 0x67, 0x72, 0x24, 0x1b, 0x85, 0xb5, 0xbb, 0x36, 0x7d,
      0xb4, 0x3c, 0x07, 0xe5, 0x48, 0x09, 0xb7, 0xa3, 0x0c, 0x82, 0xd9, 0xaf,
      0x74, 0x79, 0xab, 0x37, 0xd8, 0x49, 0xd7, 0x78, 0x91, 0x39, 0xd6, 0xbc,
      0xfa, 0xf8, 0x31, 0x46, 0x77, 0x5f, 0x6f, 0x1a, 0x63, 0x2d, 0x17, 0x91,
      0x0f, 0x0a, 0x49, 0xf6, 0x74, 0x72, 0xc1, 0x8a, 0x34, 0x39, 0xcc, 0x03,
      0x7c, 0xa4, 0x2d, 0x7e, 0xb8, 0x61, 0x09, 0xe3, 0xc4, 0x7b, 0xd9, 0x4e,
      0xc7, 0xbc, 0x99, 0xd5, 0xb8, 0x2a, 0x8f, 0x4c, 0x4d, 0x1d, 0x72, 0x47,
      0xc8, 0x47, 0x99, 0xa9, 0x05, 0x8d, 0x01, 0x84, 0x2f, 0xed, 0x69, 0x57,
      0xce, 0xb4, 0x8d, 0xd2, 0x02, 0x91, 0x22, 0xaa, 0x8f, 0xf4, 0x38, 0x84,
      0x10, 0xe1, 0xb3, 0xbb, 0xa2, 0x76, 0xc2, 0xaa, 0x54, 0x4e, 0xf9, 0x91,
      0x97, 0x3f, 0x15, 0xe7, 0xc5, 0x9b, 0x89, 0x4a, 0xab, 0x27, 0xf5, 0xce,
      0xad, 0xdd, 0xe8, 0x25, 0x33, 0x3f, 0x44, 0x03, 0x9b, 0x02, 0xe7, 0xbd,
      0xec, 0x21, 0xfc, 0x0d, 0x8f, 0x9b, 0x3a, 0x22};
  const uint8_t kEncapM[MLKEM512_ENCAPS_SEED_LEN] = {
      0x2c, 0x87, 0xaa, 0x8b, 0x11, 0x76, 0x75, 0x54, 0x74, 0xdf, 0x76,
      0x3b, 0x2a, 0xe0, 0x46, 0x35, 0x39, 0xe9, 0x53, 0xe0, 0x04, 0xc4,
      0x6a, 0x11, 0x83, 0xfd, 0x53, 0xcf, 0x84, 0xef, 0x81, 0x03};
  const uint8_t kEncapCiphertext[MLKEM512_CIPHERTEXT_BYTES] = {
      0x43, 0x1a, 0x4f, 0x1b, 0x2d, 0x2c, 0x6c, 0x00, 0xf1, 0x69, 0x0b, 0xbe,
      0x48, 0x25, 0x41, 0xef, 0x3d, 0x56, 0x37, 0x74, 0xda, 0xff, 0x83, 0x20,
      0x7f, 0x96, 0xde, 0x7e, 0x5e, 0x4a, 0x59, 0xd5, 0xd9, 0x36, 0xd9, 0x44,
      0x3a, 0xd4, 0x22, 0xe6, 0x45, 0x79, 0x3e, 0x7a, 0x60, 0xa9, 0xb0, 0xa7,
      0x6c, 0xd6, 0x72, 0xd2, 0x0c, 0x69, 0xb8, 0x2a, 0x55, 0x63, 0xdf, 0x52,
      0xd9, 0x6f, 0x9a, 0x6c, 0xdf, 0xc5, 0x6f, 0xbd, 0x4f, 0xd8, 0xd5, 0xa8,
      0xaf, 0xeb, 0x2a, 0x09, 0xd9, 0x2e, 0xc8, 0x54, 0x09, 0x47, 0x94, 0xb4,
      0xed, 0x2d, 0xb3, 0x81, 0xf0, 0x4c, 0x68, 0x43, 0x96, 0x08, 0xaa, 0x99,
      0x02, 0xa4, 0xd1, 0x68, 0x9e, 0x2e, 0xb1, 0xe5, 0xf0, 0x7a, 0x4a, 0x1c,
      0x70, 0x92, 0x62, 0xd7, 0xc2, 0xff, 0x2f, 0x81, 0xf6, 0xee, 0xaa, 0xb2,
      0xa8, 0x6a, 0x41, 0xba, 0x21, 0x0e, 0xb1, 0xbf, 0x8e, 0x75, 0xfe, 0xbc,
      0xcd, 0x1a, 0x15, 0xb4, 0xd7, 0xa7, 0xb6, 0x02, 0x57, 0xc8, 0x9d, 0x00,
      0xbd, 0x81, 0xd3, 0x9f, 0xcb, 0x8d, 0x1c, 0xe3, 0x27, 0x81, 0x02, 0x59,
      0x5d, 0xd6, 0x52, 0xf7, 0xfb, 0x7d, 0x55, 0x84, 0x87, 0x4f, 0x33, 0x27,
      0xb1, 0x74, 0x04, 0x3b, 0x35, 0x0e, 0xbd, 0x4d, 0x41, 0xfe, 0x08, 0xbd,
      0x0e, 0x85, 0x4d, 0x41, 0xcb, 0xb0, 0x27, 0xc4, 0x81, 0xda, 0x64, 0xdc,
      0x61, 0x51, 0xb8, 0x8d, 0xec, 0xec, 0xcf, 0x02, 0x2d, 0xda, 0xc2, 0xe2,
      0x27, 0x36, 0xc1, 0x47, 0xe0, 0x77, 0x32, 0x94, 0x23, 0x1c, 0x05, 0x89,
      0x96, 0x71, 0x54, 0xc5, 0x26, 0xb0, 0xb7, 0xcd, 0xd5, 0x95, 0x68, 0xee,
      0xff, 0x57, 0x49, 0xa4, 0x0c, 0xb1, 0x00, 0xc6, 0x0c, 0x64, 0x80, 0x89,
      0x76, 0x55, 0xd9, 0x6e, 0x9f, 0x64, 0xd6, 0x16, 0x84, 0xc0, 0xb3, 0x15,
      0x06, 0x46, 0x73, 0x2c, 0x19, 0x40, 0x9f, 0xe5, 0x65, 0x54, 0x0a, 0x31,
      0x89, 0x47, 0x03, 0xcf, 0x01, 0x79, 0xca, 0xe8, 0x5b, 0xc8, 0xc1, 0xa5,
      0x73, 0x26, 0x49, 0x83, 0x6e, 0x48, 0xe6, 0x76, 0x40, 0x5b, 0x95, 0x91,
      0xb6, 0x5b, 0xa2, 0x5f, 0x9b, 0x48, 0x9b, 0x9e, 0x57, 0x72, 0xaa, 0x1e,
      0xd5, 0xa0, 0x01, 0x43, 0xcb, 0x9f, 0x54, 0x49, 0xfd, 0x01, 0x34, 0x57,
      0xa3, 0xc1, 0x38, 0x74, 0xcb, 0x58, 0xc7, 0x5b, 0x52, 0xc9, 0xb6, 0xa9,
      0xae, 0x49, 0x5c, 0xcb, 0x50, 0x4a, 0x89, 0xcb, 0x5f, 0x14, 0x56, 0x95,
      0xb9, 0x21, 0x63, 0x2f, 0xb8, 0x5b, 0x03, 0x16, 0xb3, 0x0d, 0x4a, 0xd1,
      0x7f, 0xef, 0x08, 0x62, 0xd6, 0xb1, 0xe6, 0xca, 0x6a, 0x61, 0x1c, 0x8a,
      0x6a, 0x72, 0x34, 0xb4, 0x36, 0x2c, 0x5c, 0xa0, 0xad, 0x9f, 0x76, 0x97,
      0x68, 0x77, 0x98, 0xcf, 0x62, 0x4d, 0xc9, 0xf3, 0x5f, 0xbb, 0x37, 0x6e,
      0x09, 0x95, 0x31, 0x56, 0x53, 0x2a, 0x90, 0x33, 0x70, 0x9d, 0xf7, 0x55,
      0xb4, 0x6c, 0xc6, 0xd8, 0x3d, 0xe3, 0xa1, 0x11, 0xe1, 0x9a, 0x76, 0xb3,
      0x61, 0xe0, 0xef, 0x14, 0xc9, 0x1d, 0xb8, 0xd9, 0x1c, 0x6c, 0x6d, 0x9e,
      0x3e, 0x46, 0xf4, 0x22, 0x91, 0xfd, 0x6c, 0xbf, 0x5c, 0xfd, 0x12, 0x27,
      0x16, 0xfb, 0x06, 0x75, 0x69, 0x8e, 0x60, 0x2a, 0xb3, 0x9e, 0xe9, 0x8e,
      0x0d, 0x81, 0x45, 0xee, 0xba, 0xaa, 0x93, 0x74, 0xf5, 0xb3, 0xbb, 0x0d,
      0xf4, 0xd0, 0xfd, 0x83, 0xa4, 0x0e, 0x0d, 0x25, 0x03, 0x8c, 0x39, 0xe9,
      0xbe, 0xe0, 0x1c, 0xf7, 0x9c, 0x86, 0xf3, 0x08, 0x61, 0x58, 0xd0, 0x31,
      0xd5, 0xc5, 0xe8, 0x6b, 0xc7, 0xe7, 0xeb, 0x16, 0xe6, 0x22, 0x50, 0x5f,
      0x28, 0x88, 0x21, 0x38, 0x84, 0xc0, 0xb5, 0x25, 0x22, 0x89, 0xb1, 0x1f,
      0xce, 0x5b, 0xfe, 0xeb, 0xfb, 0xef, 0x0a, 0x32, 0xce, 0xaf, 0x9c, 0x14,
      0xc6, 0x25, 0x00, 0x90, 0x02, 0x84, 0x63, 0xdb, 0x6f, 0x8d, 0x19, 0x68,
      0x4f, 0x54, 0x11, 0x08, 0xfe, 0x93, 0x4d, 0x88, 0xe7, 0xef, 0x5c, 0xce,
      0x9d, 0xae, 0xbb, 0x32, 0x70, 0x0b, 0x93, 0x97, 0x69, 0x1a, 0x68, 0x42,
      0x98, 0xc9, 0xbf, 0x1b, 0x7c, 0x22, 0xd1, 0xbc, 0xec, 0x3f, 0xca, 0xcf,
      0xbb, 0x17, 0xf2, 0xed, 0x2b, 0x98, 0xb8, 0x5e, 0x6a, 0x8f, 0xe2, 0x48,
      0x29, 0x96, 0xb5, 0xe0, 0x99, 0xe9, 0xd0, 0x21, 0x1c, 0xb9, 0x41, 0x26,
      0x14, 0xde, 0x87, 0xdc, 0x18, 0xd2, 0x36, 0x13, 0xed, 0x7f, 0x6c, 0x29,
      0xcc, 0x37, 0xb7, 0x27, 0x11, 0x6d, 0xd9, 0x01, 0xc2, 0x81, 0x79, 0x38,
      0xc2, 0x9f, 0xcd, 0x02, 0x60, 0x89, 0x33, 0x6a, 0xdd, 0xc0, 0x9e, 0xca,
      0x90, 0xde, 0x9a, 0x25, 0xa6, 0x37, 0x4f, 0xee, 0x86, 0xbc, 0xdd, 0x06,
      0xae, 0x3d, 0xaa, 0xf0, 0xb1, 0xbc, 0x5b, 0x3b, 0x27, 0x90, 0xd4, 0xd9,
      0xf7, 0x59, 0xbe, 0xf8, 0xac, 0x74, 0x36, 0x12, 0xa2, 0xbb, 0xf6, 0xe4,
      0x5d, 0xe8, 0xb2, 0x2e, 0xfa, 0x61, 0x22, 0x66, 0x25, 0xd4, 0xc3, 0x9f,
      0x34, 0x6b, 0x84, 0x4c, 0x5e, 0xbe, 0xc5, 0x35, 0x58, 0x66, 0xc0, 0x0b,
      0x72, 0x6c, 0xc1, 0x64, 0x0c, 0xb2, 0x37, 0xc3, 0x4a, 0x20, 0xa7, 0xc6,
      0x03, 0xd2, 0x51, 0xf4, 0x6e, 0x6b, 0x3b, 0x0f, 0xa7, 0x1b, 0x32, 0x76,
      0x83, 0x5e, 0x3e, 0x9d, 0xa5, 0xb9, 0x48, 0x5e, 0x78, 0x96, 0x14, 0xaf,
      0x49, 0xf1, 0xe9, 0x50, 0x4d, 0xb2, 0x52, 0x86, 0x31, 0xfb, 0xe1, 0xcd,
      0x7d, 0xbe, 0xe8, 0x51, 0x64, 0xe4, 0xc0, 0x99, 0xa2, 0x7a, 0x45, 0x83,
      0xe9, 0x24, 0x7d, 0x07, 0x8f, 0x88, 0x30, 0xb4, 0x68, 0x74, 0xc1, 0xb0,
      0x10, 0xbf, 0x3c, 0xd9, 0x0e, 0xb0, 0x77, 0x49, 0x61, 0xf2, 0x39, 0xba};
  const uint8_t kEncapSharedSecret[MLKEM512_SHARED_SECRET_LEN] = {
      0xa7, 0x72, 0xdf, 0x2d, 0xe2, 0x50, 0xac, 0x7d, 0x89, 0x6b, 0xbb,
      0x82, 0x0b, 0x57, 0xf2, 0xae, 0x05, 0xf9, 0xa4, 0x12, 0xab, 0x55,
      0xba, 0xa4, 0x21, 0xd4, 0xaf, 0x6d, 0xac, 0x62, 0x66, 0x2a};

  uint8_t ciphertext[MLKEM512_CIPHERTEXT_BYTES] = {0};
  uint8_t shared_secret[MLKEM512_SHARED_SECRET_LEN] = {0};

  if (ml_kem_512_encapsulate_deterministic_no_self_test(
          ciphertext, shared_secret, kEncapEK, kEncapM) ||
      !check_test(kEncapCiphertext, ciphertext, sizeof(kEncapCiphertext),
                  "ML-KEM encapsulate ciphertext") ||
      !check_test(kEncapSharedSecret, shared_secret, sizeof(kEncapSharedSecret),
                  "ML-KEM encapsulate shared secret")) {
    goto err;
  }

  const uint8_t kDecapDK[MLKEM512_SECRET_KEY_BYTES] = {
      0x73, 0x9b, 0x8b, 0x1f, 0x6a, 0x57, 0x66, 0x31, 0x0b, 0x06, 0x19, 0x04,
      0x02, 0x14, 0x38, 0xbb, 0xd6, 0x1a, 0x14, 0xf0, 0x85, 0xfd, 0xe0, 0x29,
      0xb5, 0x33, 0x86, 0xec, 0x37, 0x61, 0xaa, 0xe7, 0x78, 0x28, 0xfb, 0x19,
      0xde, 0xdc, 0x50, 0xdd, 0xc1, 0xc3, 0x2d, 0x3a, 0x44, 0x4a, 0x15, 0x4b,
      0xf8, 0x33, 0xa8, 0x25, 0x71, 0x31, 0x5a, 0x56, 0x55, 0xfd, 0x3b, 0x06,
      0x65, 0x1c, 0x20, 0xf7, 0xb0, 0x37, 0x8d, 0x78, 0x24, 0x72, 0x68, 0x0c,
      0xfb, 0xb9, 0x1d, 0x07, 0xd2, 0x31, 0x10, 0xd8, 0xb1, 0x58, 0xe3, 0xa8,
      0xf3, 0x29, 0x5d, 0x92, 0x97, 0x64, 0x6d, 0xe5, 0x81, 0x4c, 0xe8, 0x2a,
      0xe8, 0x74, 0xa6, 0x45, 0x4c, 0x61, 0xf6, 0x5b, 0x0b, 0xed, 0x79, 0x77,
      0xce, 0x01, 0xc7, 0x69, 0x99, 0xce, 0x86, 0x75, 0x6d, 0xa2, 0x6a, 0x2f,
      0x5a, 0x09, 0x3c, 0x05, 0x70, 0x73, 0x16, 0xd9, 0x2a, 0xc2, 0xb2, 0x5e,
      0x72, 0xc2, 0x16, 0x0e, 0x60, 0x4c, 0x90, 0xc3, 0x39, 0xd8, 0xd0, 0x3f,
      0xa8, 0x45, 0x4d, 0x40, 0xc7, 0x07, 0x81, 0x33, 0x05, 0xe6, 0x0a, 0x48,
      0x89, 0x2c, 0x6d, 0x48, 0xfb, 0x18, 0x19, 0xf8, 0xa0, 0x92, 0x18, 0x62,
      0x49, 0xe5, 0x02, 0x1d, 0x83, 0x8b, 0xac, 0xe8, 0x19, 0xa1, 0xc7, 0xc7,
      0x2c, 0x5b, 0x45, 0xcf, 0x83, 0x98, 0xc5, 0x2a, 0x4b, 0xed, 0xd3, 0xa5,
      0x9e, 0xa4, 0x5e, 0x48, 0xf7, 0x6e, 0xc3, 0xe0, 0x78, 0x68, 0x71, 0x43,
      0x13, 0x72, 0xb3, 0x6f, 0xa7, 0xb5, 0x63, 0x97, 0xcb, 0x95, 0xe9, 0x20,
      0x0a, 0x99, 0x0b, 0x71, 0xbb, 0x4e, 0x10, 0xba, 0x3e, 0x7c, 0x08, 0xc4,
      0xa6, 0x28, 0x28, 0xb5, 0x49, 0xcc, 0xe4, 0xf5, 0x08, 0x7f, 0xd2, 0xa4,
      0xeb, 0x78, 0x64, 0xfd, 0xdc, 0x5f, 0x92, 0xa8, 0x68, 0xd5, 0xbb, 0x70,
      0xab, 0x51, 0x51, 0x0c, 0xb0, 0x43, 0xc1, 0x45, 0xb7, 0x4e, 0xb0, 0x5f,
      0x72, 0xe5, 0x96, 0x4b, 0xc7, 0x05, 0x00, 0xb0, 0xc8, 0x93, 0xe0, 0x1e,
      0xa1, 0xab, 0x77, 0xc4, 0x73, 0x7f, 0x23, 0x6a, 0xa8, 0xdd, 0x1c, 0x61,
      0xf2, 0xd7, 0xae, 0x5c, 0x93, 0x10, 0x24, 0x28, 0x12, 0xaa, 0x03, 0x4a,
      0xaa, 0xd3, 0xbc, 0xdf, 0x22, 0x4d, 0x42, 0xf3, 0x47, 0x09, 0xc0, 0x5f,
      0x12, 0x38, 0xb2, 0x18, 0xec, 0x33, 0xdd, 0x80, 0x4c, 0xb3, 0x31, 0x32,
      0xf3, 0x64, 0x7c, 0xbd, 0x48, 0xa6, 0xf5, 0xf5, 0x4a, 0x68, 0x66, 0xb2,
      0x4b, 0x51, 0x83, 0xeb, 0x66, 0x91, 0xd0, 0xa0, 0x84, 0x6b, 0x34, 0x6f,
      0x87, 0x79, 0x87, 0xf2, 0x6b, 0x76, 0xbe, 0xb3, 0x05, 0x50, 0x7b, 0x34,
      0xf2, 0xe1, 0xcc, 0x95, 0xcb, 0x81, 0x85, 0xbb, 0x5e, 0x35, 0x60, 0x67,
      0x6f, 0x30, 0x27, 0x52, 0xc3, 0x82, 0x4e, 0x90, 0xbd, 0x84, 0x08, 0x1b,
      0x28, 0x89, 0x1f, 0xcd, 0xda, 0x31, 0x3f, 0x06, 0x37, 0x65, 0x59, 0x4e,
      0x41, 0xe0, 0x85, 0x2a, 0x65, 0x55, 0x1f, 0xf2, 0x50, 0xb3, 0x18, 0x8a,
      0x2a, 0x61, 0xbc, 0x5f, 0x76, 0x50, 0x3b, 0x39, 0x16, 0xf0, 0x64, 0x61,
      0xca, 0x01, 0x1f, 0xe1, 0xd1, 0x91, 0x53, 0xe2, 0x64, 0x29, 0xb0, 0x37,
      0x3a, 0x45, 0x39, 0xdf, 0x32, 0x47, 0x47, 0x76, 0xbe, 0xcb, 0xec, 0xa6,
      0x08, 0xe9, 0x70, 0x6f, 0x65, 0xad, 0x92, 0x42, 0x2b, 0xbb, 0x28, 0xb6,
      0xb9, 0xc9, 0x6c, 0xa0, 0x2b, 0x61, 0x3d, 0x5c, 0x06, 0xee, 0xe3, 0x1f,
      0x9f, 0x32, 0x22, 0x33, 0x00, 0xb2, 0x22, 0xe3, 0x95, 0x5c, 0xb0, 0x5a,
      0x1b, 0xab, 0xac, 0x60, 0x2a, 0xb3, 0x1e, 0x18, 0x74, 0x15, 0x4a, 0x83,
      0xd4, 0xbc, 0x54, 0xdc, 0xdc, 0x94, 0xd6, 0xd0, 0x5a, 0xdf, 0xbc, 0x6f,
      0x46, 0x61, 0x3d, 0x85, 0x81, 0x45, 0xed, 0xeb, 0x63, 0xf9, 0x20, 0x44,
      0x55, 0x1b, 0xb9, 0xe8, 0xbb, 0x63, 0x07, 0x10, 0x1b, 0xef, 0x5b, 0xc2,
      0x68, 0x5b, 0xcb, 0x04, 0x35, 0x3c, 0x94, 0x84, 0x3e, 0xc9, 0x94, 0x28,
      0x69, 0x89, 0x99, 0xf9, 0x95, 0x80, 0x33, 0x6a, 0x90, 0x3e, 0xb3, 0x08,
      0x08, 0xe3, 0x01, 0xa1, 0x88, 0x79, 0xb0, 0x9b, 0x8b, 0x20, 0x59, 0x68,
      0xe6, 0x24, 0x7f, 0x3a, 0x14, 0xb9, 0x00, 0x93, 0x5d, 0x80, 0x8a, 0x7c,
      0xf8, 0x21, 0x85, 0x15, 0xf3, 0xb8, 0xc1, 0x3b, 0x54, 0x07, 0x29, 0xaa,
      0x74, 0x1b, 0x42, 0x46, 0x0b, 0x41, 0x17, 0xe0, 0x55, 0x56, 0x19, 0x46,
      0x0a, 0x5a, 0x48, 0x4c, 0x52, 0x15, 0x0f, 0x3b, 0x1b, 0x60, 0x00, 0x95,
      0xd2, 0x25, 0x9f, 0x45, 0x56, 0x7a, 0x3e, 0x03, 0x4e, 0x14, 0x39, 0xaf,
      0x5e, 0xd3, 0xbc, 0x41, 0x61, 0x93, 0x52, 0x77, 0x53, 0x05, 0x29, 0x24,
      0x26, 0x18, 0x49, 0x13, 0xc2, 0x53, 0xd9, 0x54, 0x46, 0xfc, 0x45, 0x33,
      0xfe, 0x12, 0x79, 0xf5, 0x5a, 0x04, 0x9b, 0xcc, 0x70, 0xad, 0x50, 0x86,
      0xad, 0x72, 0xc7, 0x40, 0xd5, 0x3d, 0x58, 0x5b, 0x4b, 0x93, 0x20, 0x42,
      0xa2, 0x84, 0x3b, 0x69, 0x5c, 0x21, 0x9f, 0xb1, 0x3c, 0xa3, 0xc0, 0xcf,
      0x62, 0x56, 0xca, 0x82, 0xe5, 0x8a, 0x98, 0xfb, 0x79, 0x88, 0x06, 0x91,
      0x27, 0x1c, 0x1a, 0xc5, 0x30, 0x38, 0x95, 0x3c, 0x0e, 0xc6, 0x07, 0x92,
      0x06, 0xc5, 0x0c, 0x5d, 0x24, 0xcd, 0xfe, 0x80, 0x23, 0x43, 0x91, 0xba,
      0x17, 0x62, 0x19, 0x7d, 0xf1, 0x2c, 0xcd, 0x01, 0x04, 0x7b, 0x74, 0x3b,
      0x9c, 0xab, 0x2d, 0xc8, 0xab, 0xb8, 0x61, 0x19, 0x0f, 0xfc, 0x92, 0xc6,
      0xa2, 0x6b, 0x86, 0x50, 0xf6, 0xa0, 0x0d, 0xf4, 0x40, 0x4d, 0xb2, 0x54,
      0x24, 0x68, 0x6a, 0x34, 0x55, 0xc6, 0x45, 0x90, 0x51, 0x1a, 0x34, 0x3a,
      0x8c, 0x73, 0x4c, 0x55, 0x8b, 0x42, 0x74, 0xb4, 0x39, 0x77, 0x12, 0x13,
      0xed, 0x5b, 0xc6, 0x28, 0x58, 0x5f, 0x75, 0x0b, 0x67, 0xd3, 0xc0, 0x42,
      0x96, 0xdc, 0x4b, 0xda, 0x56, 0x44, 0xec, 0xd5, 0x84, 0xee, 0x87, 0x0b,
      0x46, 0xa7, 0x04, 0xe5, 0xd1, 0x9f, 0xa1, 0x37, 0x6e, 0x66, 0x88, 0x09,
      0x73, 0xab, 0x56, 0xa5, 0x16, 0x0d, 0x8b, 0x99, 0xc2, 0xb4, 0xba, 0x4e,
      0x17, 0x7c, 0x25, 0xb2, 0x4a, 0x9b, 0xdf, 0x80, 0xba, 0x1b, 0x48, 0xc2,
      0x7c, 0x88, 0xbd, 0x2c, 0xab, 0xbd, 0x98, 0x80, 0x34, 0xf3, 0x5c, 0xa7,
      0x92, 0x54, 0x9e, 0x4d, 0x57, 0x9b, 0x20, 0xa5, 0x07, 0x45, 0x47, 0x43,
      0x11, 0x97, 0xa3, 0x62, 0x7b, 0x5d, 0x7c, 0x75, 0x73, 0xfa, 0x93, 0x06,
      0x52, 0xfa, 0x99, 0x9a, 0x12, 0x89, 0x15, 0xcc, 0x01, 0x73, 0x7a, 0x08,
      0x70, 0xa3, 0x02, 0xa1, 0x51, 0x38, 0xb3, 0x85, 0x48, 0xb5, 0x9a, 0x65,
      0xd3, 0x97, 0xb6, 0xa0, 0xfc, 0xb6, 0x61, 0x23, 0xad, 0x93, 0x47, 0xc8,
      0x9c, 0xf5, 0xbe, 0x90, 0xa9, 0xac, 0xcf, 0x4a, 0x61, 0x6f, 0x42, 0x36,
      0x9a, 0x83, 0x99, 0xcf, 0xc4, 0x6e, 0x9c, 0xc6, 0xa1, 0x01, 0xfc, 0x35,
      0x3f, 0xf2, 0x27, 0xc6, 0xf2, 0x62, 0xb7, 0x95, 0xa2, 0x15, 0x70, 0x49,
      0x7a, 0x9b, 0xad, 0x2d, 0x7c, 0xbf, 0x2e, 0x87, 0x7a, 0xde, 0x82, 0xc8,
      0x37, 0xea, 0x68, 0xbc, 0x42, 0x43, 0x8d, 0xf7, 0x9e, 0xc5, 0x96, 0x31,
      0x8b, 0x57, 0x98, 0x1c, 0x35, 0x96, 0x87, 0x15, 0xa7, 0x59, 0xf4, 0x53,
      0x83, 0x5c, 0xbc, 0x27, 0xd7, 0x59, 0x4c, 0xb8, 0x65, 0xdd, 0x51, 0x15,
      0x21, 0x72, 0xb0, 0xa1, 0x36, 0xbe, 0x2a, 0x4a, 0x95, 0xed, 0x0a, 0x80,
      0x27, 0xe1, 0x5f, 0xca, 0x37, 0x35, 0x13, 0xe9, 0x8a, 0x2a, 0x32, 0xc8,
      0xe2, 0x24, 0x62, 0xff, 0xb4, 0x46, 0x43, 0x54, 0x1e, 0x8c, 0x45, 0x83,
      0x5e, 0x67, 0x81, 0xd1, 0x99, 0x8e, 0x97, 0xa9, 0x4e, 0xf4, 0x98, 0xcf,
      0x41, 0xa0, 0x80, 0x20, 0xb0, 0x9f, 0x9c, 0xac, 0x36, 0xac, 0x42, 0x28,
      0x89, 0x45, 0x58, 0xf3, 0x29, 0x61, 0x0d, 0x52, 0x88, 0x20, 0x8a, 0x9b,
      0x44, 0xf6, 0x14, 0xd8, 0x26, 0x44, 0x0a, 0x87, 0xb9, 0x20, 0xe0, 0x7c,
      0x54, 0x70, 0xc1, 0x01, 0x24, 0x5a, 0xd7, 0x78, 0x5b, 0x30, 0xa4, 0xc5,
      0x4c, 0xd3, 0xa0, 0xec, 0x08, 0x8c, 0xf6, 0xb2, 0x7d, 0x43, 0x10, 0x4c,
      0x63, 0xf0, 0x56, 0x75, 0x68, 0x26, 0x38, 0xfb, 0xaf, 0x16, 0xa8, 0xaa,
      0xb8, 0x91, 0x04, 0x58, 0xb5, 0xc1, 0xe1, 0xdb, 0x60, 0xb3, 0xa2, 0x06,
      0xb7, 0xf2, 0x7f, 0x50, 0x44, 0x0b, 0xc2, 0x26, 0x09, 0x58, 0x45, 0x38,
      0xb3, 0x53, 0x70, 0x09, 0x01, 0x98, 0x8b, 0x44, 0x56, 0xda, 0xd6, 0x0c,
      0x98, 0xfa, 0x54, 0xc2, 0xba, 0xab, 0xd2, 0xbb, 0x38, 0x79, 0xb7, 0x2f,
      0xbc, 0xd5, 0x03, 0x15, 0x3a, 0x68, 0xa5, 0x28, 0x3e, 0x17, 0x60, 0x89,
      0x6e, 0x19, 0xcf, 0xdb, 0xbc, 0xb4, 0x3e, 0x86, 0x45, 0x88, 0x9a, 0x2a,
      0xe1, 0x1c, 0x0f, 0x9c, 0x74, 0xaa, 0x38, 0xb5, 0xa8, 0x13, 0x12, 0x0a,
      0x3f, 0xf5, 0x08, 0xf7, 0x53, 0xc8, 0x28, 0xf6, 0x46, 0x65, 0xf1, 0x1e,
      0xa6, 0x1c, 0x54, 0xc2, 0xa8, 0xb0, 0xaa, 0xb8, 0xc3, 0xd8, 0x35, 0x34,
      0xaf, 0x09, 0x38, 0xdc, 0x98, 0x59, 0x8d, 0xf0, 0x15, 0xc8, 0x37, 0xc6,
      0x9f, 0xb5, 0x53, 0x0d, 0x70, 0x01, 0xf5, 0x3b, 0x42, 0x9c, 0x0a, 0xbe,
      0x42, 0x30, 0x60, 0x87, 0xe5, 0x38, 0xdc, 0x27, 0xc8, 0xd6, 0x34, 0x80,
      0x96, 0xd7, 0xcf, 0x19, 0x41, 0x2e, 0x3e, 0x32, 0xb3, 0xd2, 0x53, 0xc8,
      0x23, 0x38, 0x19, 0x83, 0xe0, 0x88, 0xc1, 0xe0, 0xa1, 0x9f, 0x11, 0x51,
      0x07, 0x33, 0xaa, 0x07, 0xf6, 0x16, 0x02, 0xda, 0x8d, 0xbd, 0x20, 0x6a,
      0x62, 0xb6, 0xa9, 0xe1, 0x90, 0xad, 0x8d, 0xd9, 0xc1, 0x91, 0x52, 0x6d,
      0x26, 0xd7, 0x27, 0xfd, 0x15, 0x95, 0x46, 0x54, 0x71, 0xd2, 0xf0, 0x55,
      0xb1, 0x41, 0x74, 0xe7, 0x02, 0x1c, 0x84, 0x0a, 0x62, 0x17, 0x22, 0xac,
      0x0b, 0x76, 0xc3, 0xf0, 0xe4, 0x1e, 0x24, 0x9c, 0x8a, 0xa1, 0x06, 0x9c,
      0x52, 0x55, 0x38, 0x8d, 0xbb, 0xc8, 0xa9, 0xd6, 0xbd, 0xcb, 0x3b, 0x8a,
      0x2c, 0x19, 0x08, 0xf5, 0x40, 0x74, 0x54, 0x51, 0x70, 0x6d, 0x08, 0xbe,
      0xc0, 0x90, 0x2d, 0xbe, 0xca, 0x2e, 0xe9, 0x89, 0x0c, 0x2a, 0x24, 0x1d,
      0xde, 0x47, 0xc3, 0x62, 0x66, 0x16, 0xb4, 0x3c, 0x7d, 0x39, 0xf6, 0x3f,
      0x98, 0x9c, 0x62, 0x77, 0xc3, 0x0f, 0x70, 0xf9, 0x7b, 0xad, 0x09, 0x89,
      0xcb, 0x64, 0xac, 0xbe, 0xb4, 0xa4, 0x03, 0xd2, 0xa4, 0xb6, 0x57, 0x06,
      0x33, 0x89, 0x86, 0xc9, 0xa0, 0x79, 0xb1, 0x8b, 0x7d, 0x43, 0xf1, 0x19,
      0xa5, 0x14, 0xaa, 0x59, 0x06, 0x6c, 0xdc, 0x39, 0x11, 0xcc, 0x3c, 0xc4,
      0x8b, 0xec, 0x56, 0x45, 0x92, 0x7b, 0x25, 0x63, 0xa6, 0x51, 0x37, 0xb5,
      0x02, 0x56, 0xb0, 0xf3, 0xda, 0x25, 0xd5, 0xe4, 0x10, 0x92, 0x00, 0x9d,
      0x0c, 0x7b, 0xc5, 0x39, 0x8b, 0xb7, 0xcf, 0x15, 0x3c, 0xcb, 0x85, 0x4e,
      0xaf, 0xb2, 0x14, 0x40, 0x93, 0x73, 0x9b, 0x7a, 0x7c, 0xcd, 0x03, 0xc8,
      0xcb, 0xa8, 0xb2, 0xc2, 0x89, 0x2e, 0x57, 0xf9, 0x9c, 0xbd, 0x03, 0x23,
      0x4f, 0x09, 0x97, 0x52, 0xd7, 0xb1, 0x05, 0xec, 0x45, 0xc1, 0xa1, 0x53,
      0xbb, 0x30, 0x0d, 0xb7, 0x75, 0x3b, 0x3e, 0xb3, 0x0b, 0x76, 0x59, 0x56,
      0xe8, 0x37, 0xbb, 0xaa, 0x8c, 0x51, 0xab, 0xd8, 0xbe, 0x54, 0xeb, 0x64,
      0xab, 0x52, 0x29, 0x61, 0x1b, 0x4b, 0x8c, 0xee, 0xd4, 0xd7, 0x7a, 0xe3,
      0xb3, 0x93, 0x35, 0xa4, 0xa4, 0x7a, 0x68, 0xaa, 0x38, 0x50, 0x42, 0x50,
      0x82, 0x98, 0x74, 0xd0, 0xec, 0xc6, 0xdf, 0xf4, 0x98, 0x6d, 0x1a, 0xe5,
      0x8c, 0xfe, 0x53, 0x7a, 0xfd, 0x2e, 0x02, 0xb2, 0xc4, 0x4b, 0xa0, 0x88,
      0xe7, 0xf3, 0xdb, 0xd2, 0x1f, 0x9d, 0x1c, 0x02, 0x37, 0x17, 0x85, 0x74,
      0x6e, 0x66, 0x72, 0x63, 0xb5, 0x0d, 0x48, 0xc4, 0x85, 0x5a, 0xe2, 0x49,
      0xab, 0x4c, 0x80, 0xc9, 0xb4, 0x93, 0xd2, 0x2b, 0xf9, 0x59, 0xcd, 0x6e,
      0x85, 0xfd, 0x28, 0xc0, 0xcd, 0x9f, 0xae, 0xce, 0xd7, 0x0f, 0xa7, 0x90};
  const uint8_t kDecapCiphertext[MLKEM512_CIPHERTEXT_BYTES] = {
      0xaf, 0x4a, 0x00, 0x68, 0xc3, 0x73, 0x44, 0xd7, 0xe1, 0x06, 0xa9, 0xcd,
      0x39, 0x77, 0x9c, 0xf8, 0xc7, 0x67, 0xd5, 0xb8, 0x1c, 0xb3, 0x44, 0x34,
      0x40, 0xda, 0xde, 0x14, 0xc2, 0xc4, 0x83, 0x27, 0xba, 0x34, 0x28, 0x45,
      0xdc, 0x8c, 0x58, 0x8b, 0xd2, 0x5d, 0x9b, 0x55, 0xe2, 0x7f, 0x33, 0x13,
      0x42, 0xe2, 0x46, 0xbd, 0xd5, 0x6d, 0x18, 0x3a, 0xc1, 0xe0, 0x78, 0x84,
      0x20, 0x57, 0x3f, 0x37, 0xe2, 0x29, 0xd4, 0x49, 0x0b, 0xf4, 0x17, 0xb3,
      0xfe, 0xda, 0x42, 0x20, 0xd7, 0x76, 0xa5, 0x29, 0xd9, 0x6f, 0x7c, 0xdd,
      0xfa, 0x1e, 0xce, 0x84, 0x81, 0x51, 0x56, 0x93, 0x54, 0x8b, 0xb7, 0x6f,
      0x5f, 0xb7, 0xda, 0xa6, 0x5d, 0xfb, 0x13, 0xbf, 0x84, 0xdd, 0x1c, 0xa4,
      0xe0, 0xef, 0x7e, 0x49, 0xe0, 0xd1, 0xe8, 0xa3, 0x91, 0x8e, 0x3c, 0xe9,
      0xa7, 0x84, 0xb3, 0x7d, 0xc2, 0xa4, 0xd2, 0xd2, 0xd3, 0x11, 0xf7, 0x06,
      0xe5, 0x05, 0xa1, 0xd9, 0x3e, 0x55, 0x23, 0x69, 0xe5, 0x10, 0xa0, 0xd2,
      0xca, 0x34, 0x7a, 0xec, 0xf8, 0x84, 0x62, 0xe2, 0xfd, 0xe0, 0xa6, 0xc9,
      0xfe, 0xb4, 0x95, 0x06, 0xa4, 0xfc, 0xeb, 0xdf, 0x98, 0x77, 0xab, 0xd1,
      0x8c, 0xa8, 0x1f, 0xea, 0x64, 0x11, 0x5f, 0xb1, 0x79, 0xe3, 0x35, 0xe0,
      0x06, 0x90, 0x5b, 0x72, 0x2a, 0x58, 0x88, 0xf7, 0x3d, 0x70, 0xed, 0x77,
      0x26, 0xe0, 0x72, 0x2f, 0x55, 0x24, 0x0e, 0xe2, 0xc2, 0x05, 0xd3, 0xe2,
      0xb2, 0xc5, 0x37, 0xa5, 0x2d, 0xf0, 0x2b, 0xb6, 0x93, 0xf4, 0xd7, 0x21,
      0x01, 0x17, 0x35, 0xce, 0x11, 0xb8, 0x07, 0x43, 0x76, 0x16, 0xff, 0x21,
      0x3d, 0x71, 0xc3, 0xa7, 0x32, 0x48, 0x4c, 0x5b, 0xc3, 0xda, 0xed, 0xea,
      0xf4, 0xfb, 0xe0, 0x2f, 0xc5, 0xe1, 0xb9, 0xb0, 0xcc, 0x82, 0xf6, 0xa6,
      0xe5, 0x1c, 0x0b, 0xca, 0x4f, 0x6d, 0x66, 0xfe, 0x19, 0x82, 0xcf, 0xdc,
      0x48, 0x04, 0xfb, 0xc8, 0xa7, 0x67, 0xbc, 0x44, 0x2e, 0x85, 0x34, 0xf6,
      0x3a, 0xf3, 0xb0, 0xd0, 0x57, 0xbc, 0x6c, 0x95, 0x0b, 0x5e, 0x0a, 0x07,
      0xae, 0x9f, 0x03, 0x85, 0x51, 0x0a, 0xe7, 0x8c, 0x11, 0xa2, 0xa1, 0x1e,
      0xbb, 0x84, 0x9c, 0x13, 0xe1, 0x77, 0xd9, 0x82, 0xdb, 0x7c, 0xd4, 0x7d,
      0x55, 0xf8, 0x08, 0x6c, 0x14, 0xe5, 0xc1, 0xc3, 0xe6, 0xc8, 0x20, 0x49,
      0x41, 0xbc, 0xa3, 0x79, 0x16, 0xfe, 0x20, 0x15, 0xc2, 0x99, 0x0d, 0x00,
      0xbd, 0x98, 0x64, 0x1c, 0xe6, 0x15, 0x04, 0x53, 0x86, 0x81, 0x29, 0x39,
      0xd0, 0xbc, 0xb7, 0x42, 0x77, 0xfc, 0xb7, 0x18, 0x34, 0xed, 0x29, 0x7e,
      0xda, 0x87, 0xdb, 0x1d, 0xf9, 0x14, 0x97, 0x85, 0x48, 0x95, 0xf6, 0xcd,
      0x8d, 0x94, 0xcf, 0xcb, 0x41, 0xed, 0xc1, 0xbe, 0x15, 0x1d, 0xf9, 0x14,
      0x73, 0xe3, 0x7e, 0xba, 0x54, 0x6e, 0x15, 0x62, 0x7a, 0x6d, 0xbd, 0x58,
      0x3d, 0x9b, 0xa0, 0xed, 0x34, 0xee, 0x51, 0x1a, 0x08, 0x31, 0xeb, 0xa1,
      0x35, 0x68, 0x29, 0x75, 0xa2, 0x39, 0xf4, 0x95, 0xe3, 0x09, 0x84, 0x2b,
      0xab, 0xee, 0xf6, 0xf4, 0x0e, 0x7b, 0xb4, 0xd6, 0xcd, 0x45, 0x09, 0x5e,
      0x3f, 0x91, 0xf9, 0xb6, 0x1b, 0x86, 0x35, 0x9c, 0xdd, 0x05, 0xd7, 0x9b,
      0xb7, 0x2f, 0x5e, 0xaa, 0x2e, 0xb9, 0x85, 0x4e, 0x21, 0xa0, 0x19, 0x4c,
      0x46, 0x8d, 0x9f, 0xe7, 0xe8, 0x9f, 0x3c, 0x0e, 0x74, 0xf5, 0x70, 0xf8,
      0x8b, 0x5b, 0x50, 0x15, 0xd4, 0xbb, 0x4c, 0x8b, 0xcb, 0x9e, 0xa6, 0x43,
      0xed, 0xce, 0x57, 0xba, 0x72, 0x11, 0x4c, 0xf5, 0x86, 0xcd, 0x6f, 0xb8,
      0xa4, 0xea, 0x5b, 0x05, 0x04, 0x47, 0xe0, 0x6b, 0xb7, 0x13, 0x89, 0x1e,
      0x13, 0x46, 0xcd, 0x22, 0x76, 0xf2, 0xb8, 0x63, 0x96, 0xe4, 0xcf, 0x42,
      0x41, 0x27, 0x6f, 0xec, 0xce, 0xec, 0x79, 0xe7, 0x38, 0xdc, 0xb4, 0x74,
      0x5e, 0x2d, 0xcd, 0x7d, 0x68, 0x98, 0x86, 0x25, 0x29, 0xf2, 0x4c, 0x38,
      0x79, 0xeb, 0xee, 0x42, 0xd8, 0x79, 0x67, 0x92, 0xdb, 0x7a, 0x7c, 0x86,
      0xd2, 0x8a, 0xb1, 0x8f, 0x26, 0xb9, 0x9b, 0x6c, 0xfc, 0x82, 0xf3, 0x9f,
      0xe7, 0xc0, 0xe5, 0x70, 0x73, 0x6d, 0x33, 0xa1, 0x77, 0x0b, 0xe8, 0x20,
      0x03, 0x66, 0x6e, 0x4f, 0x69, 0xb1, 0x18, 0x6b, 0x3d, 0x55, 0x83, 0x8a,
      0x85, 0x1b, 0x87, 0xbe, 0xb6, 0xe7, 0x22, 0x01, 0xfe, 0x3f, 0x8a, 0x25,
      0xf4, 0x86, 0x7e, 0xc6, 0x3e, 0xe8, 0xfe, 0xab, 0x05, 0xff, 0xf6, 0x08,
      0x39, 0x0f, 0xa4, 0x8d, 0x18, 0x60, 0x69, 0x80, 0x9c, 0xa6, 0x42, 0xa0,
      0xe7, 0x44, 0x8f, 0x3a, 0xd0, 0xc2, 0xd9, 0x11, 0xb3, 0x3c, 0x17, 0x5c,
      0x66, 0xa2, 0xea, 0xdf, 0xba, 0x79, 0xe2, 0xbc, 0x14, 0x0c, 0x03, 0xbf,
      0x96, 0x60, 0xec, 0x28, 0xb4, 0x1d, 0x07, 0x1c, 0x0f, 0xfb, 0xc7, 0x3c,
      0xe2, 0x58, 0x5f, 0x3d, 0xe5, 0xe1, 0x98, 0xf0, 0x25, 0xe7, 0xaf, 0xd4,
      0xbd, 0x12, 0x67, 0xe1, 0xf5, 0x4e, 0x06, 0x34, 0x96, 0xfe, 0xf3, 0xe9,
      0x66, 0xc5, 0xb7, 0x1a, 0x56, 0x5b, 0xf9, 0xbc, 0xfc, 0x70, 0x34, 0x21,
      0x6b, 0x5b, 0xf6, 0x21, 0xa2, 0xc2, 0x17, 0x9b, 0x52, 0xe9, 0xc9, 0x92,
      0x87, 0x1b, 0xd6, 0x61, 0x87, 0x39, 0x6d, 0xc6, 0x9f, 0x23, 0x94, 0xab,
      0x20, 0xc4, 0x17, 0x00, 0x2c, 0x83, 0x0e, 0x70, 0x5a, 0x08, 0xac, 0x20,
      0x5c, 0x61, 0x38, 0x04, 0xaf, 0x15, 0xac, 0x2c, 0x65, 0x55, 0x3b, 0x69,
      0xd5, 0xd0, 0x12, 0xcd, 0x90, 0xe5, 0x0a, 0x40, 0xfd, 0xdb, 0x8e, 0xda,
      0xa8, 0x33, 0x13, 0xb0, 0x08, 0x9b, 0x5d, 0x69, 0x97, 0x19, 0x3f, 0x21};
  const uint8_t kDecapSharedSecret[MLKEM512_SHARED_SECRET_LEN] = {
      0xbe, 0x41, 0xc0, 0x7a, 0xd0, 0x59, 0x67, 0xab, 0xf7, 0x00, 0xb8,
      0xb2, 0xe6, 0x97, 0x5e, 0x3c, 0x1e, 0x87, 0x50, 0x3c, 0xa0, 0xb5,
      0x83, 0xe1, 0xd7, 0x30, 0x46, 0x23, 0x0e, 0x26, 0x5e, 0x5a};
  const uint8_t kDecapCiphertextRejection[MLKEM512_CIPHERTEXT_BYTES] = {
      0x18, 0x1a, 0xf9, 0xb9, 0xf1, 0x63, 0xa3, 0x04, 0x9d, 0x97, 0xd8, 0x19,
      0x4b, 0x5c, 0x26, 0x35, 0x5f, 0xc9, 0xf9, 0xdc, 0xbf, 0x12, 0x05, 0x76,
      0xab, 0xe4, 0x4d, 0x71, 0xa0, 0xd9, 0x91, 0x06, 0xd5, 0x57, 0x5d, 0xb4,
      0xd1, 0x11, 0x89, 0xc0, 0xeb, 0xbd, 0x3b, 0x30, 0x9e, 0x2b, 0x7f, 0xb8,
      0x6b, 0xb2, 0x51, 0xfe, 0xbd, 0x0c, 0xb5, 0x1b, 0x34, 0x72, 0x89, 0x52,
      0xde, 0x93, 0xad, 0xf1, 0x52, 0xb9, 0xb2, 0x39, 0x27, 0x79, 0x04, 0x93,
      0x93, 0x74, 0xcd, 0x63, 0x89, 0xc9, 0x9e, 0xd2, 0x6d, 0xc9, 0xb7, 0xd7,
      0xa4, 0x82, 0x82, 0xe9, 0x1c, 0x8f, 0x51, 0x75, 0xca, 0xd4, 0xcb, 0x0c,
      0x49, 0xfb, 0x5b, 0x1d, 0x5b, 0x32, 0xfa, 0x89, 0x04, 0xd4, 0x84, 0x49,
      0xe1, 0x0a, 0x18, 0xa3, 0x99, 0x60, 0x25, 0xb1, 0x4e, 0xc7, 0x13, 0x7c,
      0x10, 0x49, 0xe8, 0x6f, 0xa0, 0xd1, 0x25, 0xb2, 0xb5, 0xe1, 0xa6, 0xd1,
      0xe8, 0xfa, 0x8d, 0x6d, 0xc1, 0xdb, 0x67, 0x25, 0x3c, 0xba, 0x44, 0xb2,
      0xa6, 0xb5, 0xf1, 0xa5, 0xe8, 0x7f, 0x05, 0x29, 0x1a, 0x03, 0xdb, 0xd2,
      0x75, 0x70, 0x70, 0x2c, 0xb7, 0x0c, 0x46, 0xe3, 0xe2, 0xd2, 0x60, 0x98,
      0x40, 0x65, 0xc9, 0x69, 0x18, 0x84, 0xd8, 0x86, 0x62, 0x32, 0x68, 0x8b,
      0x77, 0x43, 0xf2, 0x7f, 0x46, 0x92, 0xc9, 0x3e, 0x94, 0xe1, 0x05, 0x67,
      0xf6, 0x22, 0x6c, 0xef, 0xe8, 0x20, 0x30, 0xfd, 0x0a, 0x83, 0x54, 0x7a,
      0x8c, 0x0a, 0x1f, 0xf7, 0xca, 0x81, 0x26, 0x08, 0x76, 0x05, 0x68, 0xec,
      0x69, 0x28, 0xe0, 0xe6, 0x30, 0xd7, 0xe9, 0x37, 0xa9, 0x66, 0x99, 0x14,
      0x7e, 0x34, 0xbe, 0x04, 0xbb, 0xb7, 0xeb, 0xb5, 0x55, 0x07, 0xeb, 0x51,
      0x61, 0xd0, 0xe9, 0xdb, 0x13, 0xbe, 0x22, 0x15, 0xb7, 0x21, 0xe5, 0x47,
      0x41, 0xc7, 0x53, 0xe5, 0x0a, 0x50, 0xb6, 0x9c, 0x15, 0x97, 0x0c, 0xb1,
      0x2c, 0x92, 0xc4, 0x05, 0xa5, 0x08, 0xae, 0xa3, 0xce, 0x17, 0xb3, 0xa8,
      0x4e, 0x25, 0x36, 0x14, 0xfa, 0x67, 0x8b, 0x06, 0xf3, 0x0e, 0x19, 0x7c,
      0xff, 0x1f, 0x41, 0x24, 0x38, 0xe8, 0x00, 0x46, 0x70, 0xd8, 0x88, 0xde,
      0x0d, 0xf2, 0xc1, 0x13, 0x9c, 0xc2, 0x93, 0xee, 0x3a, 0xb9, 0x2f, 0xa9,
      0x8e, 0xdd, 0x4a, 0xbe, 0x6b, 0xe6, 0xca, 0x7d, 0xcd, 0x72, 0xe7, 0x27,
      0x30, 0xad, 0x9c, 0xaf, 0xc0, 0x93, 0x59, 0x4d, 0x57, 0x42, 0xdd, 0xf5,
      0x15, 0x4b, 0x63, 0x17, 0x34, 0x29, 0xdc, 0x5e, 0xf0, 0x33, 0x99, 0xc9,
      0xd1, 0x39, 0x51, 0x08, 0xc8, 0x94, 0xb2, 0x43, 0x59, 0xd0, 0xb1, 0xff,
      0x44, 0x9f, 0xb8, 0xde, 0xb0, 0xcb, 0x58, 0x45, 0xc7, 0x14, 0x75, 0xc5,
      0xc9, 0xa2, 0x4c, 0x7b, 0x77, 0x5e, 0x07, 0x99, 0x21, 0x49, 0xe4, 0xe5,
      0x7c, 0x79, 0x17, 0xc0, 0x1a, 0xed, 0x40, 0x29, 0x78, 0xa0, 0xf0, 0xa7,
      0xb5, 0x90, 0xca, 0xcf, 0xbf, 0x21, 0xa6, 0xcf, 0x59, 0x0a, 0x7d, 0x0d,
      0x0a, 0xd7, 0xa4, 0x55, 0x98, 0x2b, 0xe3, 0x2f, 0x94, 0xec, 0xb5, 0x42,
      0x9c, 0x24, 0x16, 0x2e, 0x13, 0x0d, 0x92, 0x13, 0x04, 0x8a, 0x65, 0x63,
      0x39, 0x0f, 0x63, 0xfe, 0x8f, 0xfb, 0x1a, 0xaf, 0x2d, 0x51, 0xdf, 0xee,
      0x47, 0x76, 0x06, 0xa3, 0x85, 0xf9, 0xa1, 0x6a, 0x00, 0x98, 0x06, 0x7f,
      0xf7, 0x89, 0x10, 0x2a, 0xac, 0xd5, 0x99, 0xee, 0x98, 0x79, 0xef, 0x4d,
      0xb7, 0xeb, 0xc0, 0xf1, 0xcc, 0x1e, 0x7d, 0xe5, 0xd2, 0x5b, 0x67, 0x65,
      0x15, 0x99, 0x4a, 0x00, 0x96, 0x91, 0xf4, 0xe2, 0xaf, 0x31, 0x1f, 0xf7,
      0x3d, 0x2b, 0xea, 0xcb, 0xdb, 0x2b, 0x28, 0x72, 0xd2, 0x7e, 0xe0, 0x2f,
      0x70, 0x50, 0xe8, 0x11, 0x98, 0xf4, 0x7f, 0xe7, 0x5e, 0x0a, 0xa0, 0x42,
      0x81, 0xf0, 0x9e, 0x5d, 0x5d, 0x19, 0x8b, 0x08, 0xbe, 0xc5, 0x61, 0x32,
      0x8e, 0xcb, 0x8d, 0x94, 0x32, 0x4e, 0xa6, 0xe1, 0xad, 0x9f, 0xad, 0x7a,
      0x8f, 0xbf, 0x4f, 0x4b, 0x1b, 0x91, 0xcc, 0x15, 0x0b, 0x0f, 0x6c, 0x86,
      0x40, 0xaf, 0x2e, 0x11, 0x2b, 0x6e, 0xeb, 0xbe, 0x8e, 0xa0, 0x33, 0x81,
      0x20, 0x09, 0x3f, 0x58, 0xbb, 0xe8, 0xae, 0xae, 0x7d, 0xf7, 0x28, 0x8f,
      0xef, 0x1d, 0xe8, 0xab, 0x32, 0x5a, 0x92, 0xb4, 0x2c, 0x0a, 0xa8, 0xed,
      0x3a, 0x62, 0xed, 0x9a, 0x86, 0xe9, 0x46, 0xff, 0xdd, 0xdc, 0x87, 0xaf,
      0x80, 0x15, 0xb4, 0xee, 0x64, 0xdf, 0x5f, 0xf4, 0x8e, 0x94, 0x11, 0x54,
      0x5a, 0x00, 0x78, 0xcf, 0x16, 0xd1, 0x49, 0xfb, 0xf3, 0x19, 0x38, 0xae,
      0xb1, 0x10, 0x92, 0x72, 0x7a, 0x45, 0x2c, 0xa8, 0x4b, 0xab, 0xb7, 0x03,
      0x89, 0x39, 0xa8, 0xdb, 0x3f, 0xa1, 0x04, 0x48, 0x17, 0xec, 0x03, 0x83,
      0x2f, 0x6c, 0x0a, 0x76, 0xe2, 0xb6, 0x16, 0xbb, 0x40, 0xb2, 0xe7, 0x62,
      0x11, 0xb9, 0x1f, 0xf9, 0x69, 0xdc, 0xe0, 0xad, 0x06, 0x9f, 0xb7, 0x29,
      0x47, 0x1e, 0x95, 0xba, 0xce, 0x35, 0x8e, 0x1b, 0xbf, 0xf2, 0xbd, 0xab,
      0xd7, 0x42, 0x98, 0x57, 0xdb, 0xab, 0x72, 0x21, 0x45, 0x47, 0xaf, 0x1a,
      0xbc, 0xdb, 0x08, 0x89, 0x6e, 0xe7, 0x7f, 0x13, 0xf2, 0xd7, 0x5b, 0x17,
      0x17, 0x44, 0x6f, 0xca, 0x7d, 0xf3, 0x2e, 0xfe, 0x1c, 0x2d, 0x09, 0xdc,
      0xfd, 0x5b, 0xfa, 0xff, 0xd1, 0x9e, 0xde, 0x7e, 0x50, 0x2b, 0x63, 0x64,
      0xab, 0xe3, 0x2e, 0x84, 0x49, 0x99, 0xb4, 0x47, 0x7c, 0x99, 0x8a, 0x9f,
      0xb3, 0xc9, 0xba, 0xbb, 0xe8, 0x3c, 0x6e, 0xc6, 0x13, 0x74, 0x0c, 0x2b,
      0x04, 0x75, 0xec, 0xb7, 0x32, 0xde, 0x51, 0x64, 0x38, 0x68, 0xeb, 0xb7};
  const uint8_t kDecapSharedSecretRejection[MLKEM512_SHARED_SECRET_LEN] = {
      0x98, 0xed, 0x60, 0x0f, 0xfd, 0x9e, 0x01, 0x9f, 0x35, 0x0e, 0x0a,
      0x15, 0xd4, 0x69, 0x5b, 0xa0, 0x96, 0xce, 0x2b, 0x32, 0xc3, 0x75,
      0x24, 0x4f, 0x79, 0xa5, 0x74, 0xda, 0x06, 0xb4, 0xb1, 0xbd};

  if (ml_kem_512_decapsulate_no_self_test(shared_secret, kDecapCiphertext,
                                          kDecapDK) ||
      !check_test(kDecapSharedSecret, shared_secret, sizeof(kDecapSharedSecret),
                  "ML-KEM decapsulate non-rejection") ||
      ml_kem_512_decapsulate_no_self_test(
          shared_secret, kDecapCiphertextRejection, kDecapDK) ||
      !check_test(kDecapSharedSecretRejection, shared_secret,
                  sizeof(kDecapSharedSecretRejection),
                  "ML-KEM decapsulate implicit rejection")) {
    goto err;
  }

  ret = 1;
err:
  return ret;
}

static int boringssl_self_test_eddsa(void) {
  int ret = 0;

  const uint8_t kEd25519PrivateKey[ED25519_PRIVATE_KEY_SEED_LEN] = {
      0xb3, 0x99, 0x05, 0xbf, 0x43, 0x0b, 0x2a, 0xd2, 0x1d, 0xb6, 0x5d,
      0x49, 0xa6, 0xab, 0x03, 0xc1, 0x7d, 0xdb, 0x72, 0xe7, 0xa9, 0x8e,
      0xb9, 0x8f, 0xae, 0x59, 0x91, 0x7a, 0xe2, 0x5f, 0x92, 0x14};
  const uint8_t kEd25519PublicKey[ED25519_PUBLIC_KEY_LEN] = {
      0xe7, 0x75, 0xcf, 0x0e, 0x33, 0x48, 0x52, 0xa7, 0xe6, 0x99, 0xbe,
      0xba, 0x13, 0xbc, 0x24, 0xf8, 0x32, 0xf3, 0xc2, 0xa3, 0xa0, 0x3d,
      0xc9, 0x3c, 0x42, 0xb5, 0x92, 0x76, 0x15, 0xa5, 0x46, 0xba};
  const uint8_t kEd25519Signature[ED25519_SIGNATURE_LEN] = {
      0x30, 0x1a, 0x4c, 0x56, 0xe0, 0x37, 0x0b, 0x57, 0x2f, 0x7d, 0x8c,
      0x75, 0x1b, 0x5c, 0xfa, 0xb6, 0xc3, 0x98, 0x7c, 0x6f, 0x5d, 0xe8,
      0x7c, 0xac, 0x4d, 0x71, 0x16, 0x73, 0xda, 0x8c, 0xb2, 0x19, 0x86,
      0x03, 0xcd, 0x91, 0x82, 0x73, 0xa5, 0x34, 0x24, 0x93, 0xf1, 0xc1,
      0xad, 0x0e, 0x8a, 0x78, 0x45, 0x15, 0xa7, 0xfe, 0xc8, 0xc9, 0xbe,
      0xa2, 0xa3, 0xf1, 0xcf, 0x7b, 0x3a, 0x89, 0x10, 0x0f};
  const uint8_t kEd25519Message[128] = {
      0x13, 0x1d, 0x2a, 0xa9, 0x8f, 0x46, 0xfd, 0x5a, 0xca, 0xef, 0x8e, 0x92,
      0xfa, 0x8c, 0x50, 0xd4, 0x8b, 0xda, 0xdf, 0xfe, 0x13, 0xd7, 0x9c, 0xc7,
      0x1b, 0x95, 0x85, 0x5f, 0xaf, 0xa4, 0x84, 0x66, 0x50, 0x2a, 0x1c, 0x61,
      0x4d, 0xb7, 0x85, 0xfc, 0xc9, 0x4c, 0x50, 0x61, 0x65, 0x23, 0x93, 0x42,
      0xcb, 0x9b, 0x3e, 0xe6, 0x3b, 0x35, 0xdc, 0x2f, 0x7e, 0x78, 0x61, 0x15,
      0x42, 0xc7, 0xa6, 0x1b, 0x50, 0xf3, 0xb6, 0x8e, 0xcf, 0x1b, 0x70, 0xca,
      0xc0, 0x1b, 0x34, 0xef, 0x06, 0x1b, 0x3f, 0x7c, 0xaa, 0xc8, 0x26, 0x56,
      0xbf, 0xd5, 0x5a, 0x06, 0xb8, 0xeb, 0x7d, 0xbe, 0x82, 0x45, 0x17, 0xfe,
      0x3c, 0x56, 0x7d, 0xa5, 0xa0, 0x3e, 0x0b, 0xf2, 0xf1, 0xfe, 0xbb, 0x96,
      0x3c, 0x94, 0x1a, 0xfc, 0x36, 0xe4, 0x5a, 0x5a, 0xc5, 0xe2, 0x71, 0xcd,
      0x99, 0x56, 0xcc, 0xda, 0x0d, 0x62, 0xc8, 0x7c};
  uint8_t ed25519_private_key[ED25519_PRIVATE_KEY_LEN] = {0};
  OPENSSL_memcpy(ed25519_private_key, kEd25519PrivateKey,
                 ED25519_PRIVATE_KEY_SEED_LEN);
  OPENSSL_memcpy(ed25519_private_key + ED25519_PRIVATE_KEY_SEED_LEN,
                 kEd25519PublicKey, ED25519_PUBLIC_KEY_LEN);
  uint8_t ed25519_out_sig[ED25519_SIGNATURE_LEN] = {0};
  if (!ED25519_sign_no_self_test(&ed25519_out_sig[0], &kEd25519Message[0],
                                 sizeof(kEd25519Message),
                                 ed25519_private_key) ||
      !check_test(&kEd25519Signature[0], &ed25519_out_sig[0],
                  ED25519_SIGNATURE_LEN, "ED25519 sign")) {
    fprintf(stderr, "ED25519 sign failed.\n");
    goto err;
  }
  if (!ED25519_verify_no_self_test(&kEd25519Message[0], sizeof(kEd25519Message),
                                   ed25519_out_sig, kEd25519PublicKey)) {
    fprintf(stderr, "ED25519 verify failed.\n");
    goto err;
  }

  ret = 1;
err:
  return ret;
}

#if defined(BORINGSSL_FIPS)

static void run_self_test_rsa(void) {
  if (!boringssl_self_test_rsa()) {
    BORINGSSL_FIPS_abort();
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_rsa);

void boringssl_ensure_rsa_self_test(void) {
  CRYPTO_once(g_self_test_once_rsa_bss_get(), run_self_test_rsa);
}

static void run_self_test_ecc(void) {
  if (!boringssl_self_test_ecc()) {
    BORINGSSL_FIPS_abort();
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_ecc);

void boringssl_ensure_ecc_self_test(void) {
  CRYPTO_once(g_self_test_once_ecc_bss_get(), run_self_test_ecc);
}

static void run_self_test_ffdh(void) {
  if (!boringssl_self_test_ffdh()) {
    BORINGSSL_FIPS_abort();
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_ffdh);

void boringssl_ensure_ffdh_self_test(void) {
  CRYPTO_once(g_self_test_once_ffdh_bss_get(), run_self_test_ffdh);
}

static void run_self_test_ml_kem(void) {
  if (!boringssl_self_test_ml_kem()) {
    BORINGSSL_FIPS_abort();
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_ml_kem);

void boringssl_ensure_ml_kem_self_test(void) {
  CRYPTO_once(g_self_test_once_ml_kem_bss_get(), run_self_test_ml_kem);
}

static void run_self_test_eddsa(void) {
  if (!boringssl_self_test_eddsa()) {
    BORINGSSL_FIPS_abort();
  }
}

DEFINE_STATIC_ONCE(g_self_test_once_eddsa);

void boringssl_ensure_eddsa_self_test(void) {
  CRYPTO_once(g_self_test_once_eddsa_bss_get(), run_self_test_eddsa);
}

#endif  // BORINGSSL_FIPS


// Startup self tests.
//
// These tests are run at process start when in FIPS mode. Note that the SHA256
// and HMAC-SHA256 tests are also used from bcm.c, so they can't be static.

int boringssl_self_test_sha256(void) {
  static const uint8_t kInput[16] = {
      0xff, 0x3b, 0x85, 0x7d, 0xa7, 0x23, 0x6a, 0x2b,
      0xaa, 0x0f, 0x39, 0x6b, 0x51, 0x52, 0x22, 0x17,
  };
  static const uint8_t kPlaintextSHA256[32] = {
      0x7f, 0xe4, 0xd5, 0xf1, 0xa1, 0xe3, 0x82, 0x87, 0xd9, 0x58, 0xf5,
      0x11, 0xc7, 0x1d, 0x5e, 0x27, 0x5e, 0xcc, 0xd2, 0x66, 0xcf, 0xb9,
      0xc8, 0xc6, 0x60, 0xd8, 0x92, 0x1e, 0x57, 0xfd, 0x46, 0x75,
  };
  uint8_t output[SHA256_DIGEST_LENGTH];

  // SHA-256 KAT
  SHA256(kInput, sizeof(kInput), output);
  return check_test(kPlaintextSHA256, output, sizeof(kPlaintextSHA256),
                    "SHA-256 KAT");
}

static int boringssl_self_test_sha512(void) {
  static const uint8_t kInput[16] = {
      0x21, 0x25, 0x12, 0xf8, 0xd2, 0xad, 0x83, 0x22,
      0x78, 0x1c, 0x6c, 0x4d, 0x69, 0xa9, 0xda, 0xa1,
  };
  static const uint8_t kPlaintextSHA512[64] = {
      0x29, 0x3c, 0x94, 0x35, 0x4e, 0x98, 0x83, 0xe5, 0xc2, 0x78, 0x36,
      0x7a, 0xe5, 0x18, 0x90, 0xbf, 0x35, 0x41, 0x01, 0x64, 0x19, 0x8d,
      0x26, 0xeb, 0xe1, 0xf8, 0x2f, 0x04, 0x8e, 0xfa, 0x8b, 0x2b, 0xc6,
      0xb2, 0x9d, 0x5d, 0x46, 0x76, 0x5a, 0xc8, 0xb5, 0x25, 0xa3, 0xea,
      0x52, 0x84, 0x47, 0x6d, 0x6d, 0xf4, 0xc9, 0x71, 0xf3, 0x3d, 0x89,
      0x4c, 0x3b, 0x20, 0x8c, 0x5b, 0x75, 0xe8, 0xf8, 0x7c,
  };
  uint8_t output[SHA512_DIGEST_LENGTH];

  // SHA-512 KAT
  SHA512(kInput, sizeof(kInput), output);
  return check_test(kPlaintextSHA512, output, sizeof(kPlaintextSHA512),
                    "SHA-512 KAT");
}

int boringssl_self_test_hmac_sha256(void) {
  static const uint8_t kInput[16] = {
      0xda, 0xd9, 0x12, 0x93, 0xdf, 0xcf, 0x2a, 0x7c,
      0x8e, 0xcd, 0x13, 0xfe, 0x35, 0x3f, 0xa7, 0x5b,
  };
  static const uint8_t kPlaintextHMACSHA256[32] = {
      0x36, 0x5f, 0x5b, 0xd5, 0xf5, 0xeb, 0xfd, 0xc7, 0x6e, 0x53, 0xa5,
      0x73, 0x6d, 0x73, 0x20, 0x13, 0xaa, 0xd3, 0xbc, 0x86, 0x4b, 0xb8,
      0x84, 0x94, 0x16, 0x46, 0x88, 0x9c, 0x48, 0xee, 0xa9, 0x0e,
  };
  uint8_t output[EVP_MAX_MD_SIZE];

  unsigned output_len = 0;
  HMAC(EVP_sha256(), kInput, sizeof(kInput), kInput, sizeof(kInput), output,
       &output_len);
  return output_len == sizeof(kPlaintextHMACSHA256) &&
         check_test(kPlaintextHMACSHA256, output, sizeof(kPlaintextHMACSHA256),
                    "HMAC-SHA-256 KAT");
}

static int boringssl_self_test_hkdf_sha256(void) {
  const uint8_t kHKDF_ikm_tc1[] = {   // RFC 5869 Test Case 1
      0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
      0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b
  };
  const uint8_t kHKDF_salt_tc1[] = {
      0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0a, 0x0b,
      0x0c
  };
  const uint8_t kHKDF_info_tc1[] = {
      0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7, 0xf8, 0xf9
  };
  const uint8_t kHKDF_okm_tc1_sha256[] = {
      0x3c, 0xb2, 0x5f, 0x25, 0xfa, 0xac, 0xd5, 0x7a, 0x90, 0x43, 0x4f, 0x64,
      0xd0, 0x36, 0x2f, 0x2a, 0x2d, 0x2d, 0x0a, 0x90, 0xcf, 0x1a, 0x5a, 0x4c,
      0x5d, 0xb0, 0x2d, 0x56, 0xec, 0xc4, 0xc5, 0xbf, 0x34, 0x00, 0x72, 0x08,
      0xd5, 0xb8, 0x87, 0x18, 0x58, 0x65
  };

  uint8_t output[sizeof(kHKDF_okm_tc1_sha256)];
  HKDF(output, sizeof(output), EVP_sha256(),
       kHKDF_ikm_tc1, sizeof(kHKDF_ikm_tc1),
       kHKDF_salt_tc1, sizeof(kHKDF_salt_tc1),
       kHKDF_info_tc1, sizeof(kHKDF_info_tc1));
  return check_test(kHKDF_okm_tc1_sha256, output, sizeof(output),
                    "HKDF-SHA-256 KAT");
}

static int boringssl_self_test_sha3_256(void) {
  // From: SHA3_256ShortMsg.txt
  // Len = 128
  // Msg = d83c721ee51b060c5a41438a8221e040
  // MD  = b87d9e4722edd3918729ded9a6d03af8256998ee088a1ae662ef4bcaff142a96
  static const uint8_t kInput[16] = {
    0xd8, 0x3c, 0x72, 0x1e, 0xe5, 0x1b, 0x06, 0x0c,
    0x5a, 0x41, 0x43, 0x8a, 0x82, 0x21, 0xe0, 0x40,
};
  static const uint8_t kPlaintextSHA3_256[SHA3_256_DIGEST_LENGTH] = {
    0xb8, 0x7d, 0x9e, 0x47, 0x22, 0xed, 0xd3, 0x91, 0x87, 0x29, 0xde,
    0xd9, 0xa6, 0xd0, 0x3a, 0xf8, 0x25, 0x69, 0x98, 0xee, 0x08, 0x8a,
    0x1a, 0xe6, 0x62, 0xef, 0x4b, 0xca, 0xff, 0x14, 0x2a, 0x96,
};
  uint8_t output[SHA3_256_DIGEST_LENGTH];

  // SHA3-256 KAT
  SHA3_256(kInput, sizeof(kInput), output);
  return check_test(kPlaintextSHA3_256, output, sizeof(kPlaintextSHA3_256),
                    "SHA3-256 KAT");
}

static int boringssl_self_test_fast(void) {
  static const uint8_t kAESKey[16] = "BoringCrypto Key";
  // Older versions of the gcc release build on ARM will optimize out the
  // assembly label for kAESIV, if we define it with {0}. The assembler
  // will set the value of kAESIV to another static constant in the
  // fipsmodule, kZeroIV, since they have the same value, then skip creating
  // a label for kAESIV. This does not mesh well with delocate.go, since we
  // use these labels to determine if the symbol is "known". Expanding the
  // definition of kAESIV gets around the unwanted assembler optimization.
  static const uint8_t kAESIV[16] = {
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
  };

  EVP_AEAD_CTX aead_ctx;
  EVP_AEAD_CTX_zero(&aead_ctx);
  int ret = 0;

  AES_KEY aes_key;
  uint8_t aes_iv[16];
  uint8_t output[256];

  // AES-CBC Encryption KAT
  static const uint8_t kAESCBCEncPlaintext[32] = {
      0x07, 0x86, 0x09, 0xa6, 0xc5, 0xac, 0x25, 0x44, 0x69, 0x9a, 0xdf,
      0x68, 0x2f, 0xa3, 0x77, 0xf9, 0xbe, 0x8a, 0xb6, 0xae, 0xf5, 0x63,
      0xe8, 0xc5, 0x6a, 0x36, 0xb8, 0x4f, 0x55, 0x7f, 0xad, 0xd3,
  };
  static const uint8_t kAESCBCEncCiphertext[sizeof(kAESCBCEncPlaintext)] = {
      0x56, 0x46, 0xc1, 0x41, 0xf4, 0x13, 0xd6, 0xff, 0x62, 0x92, 0x41,
      0x7a, 0x26, 0xc6, 0x86, 0xbd, 0x30, 0x5f, 0xb6, 0x57, 0xa7, 0xd2,
      0x50, 0x3a, 0xc5, 0x5e, 0x8e, 0x93, 0x40, 0xf2, 0x10, 0xd8,
  };
  memcpy(aes_iv, kAESIV, sizeof(kAESIV));
  if (AES_set_encrypt_key(kAESKey, 8 * sizeof(kAESKey), &aes_key) != 0) {
    fprintf(stderr, "AES_set_encrypt_key failed.\n");
    goto err;
  }
  AES_cbc_encrypt(kAESCBCEncPlaintext, output, sizeof(kAESCBCEncPlaintext),
                  &aes_key, aes_iv, AES_ENCRYPT);
  if (!check_test(kAESCBCEncCiphertext, output, sizeof(kAESCBCEncCiphertext),
                  "AES-CBC-encrypt KAT")) {
    goto err;
  }

  // AES-CBC Decryption KAT
  static const uint8_t kAESCBCDecCiphertext[32] = {
      0x34, 0x7a, 0xa5, 0xa0, 0x24, 0xb2, 0x82, 0x57, 0xb3, 0x65, 0x10,
      0xbe, 0x58, 0x3d, 0x4f, 0x47, 0xad, 0xb7, 0xbb, 0xee, 0xdc, 0x60,
      0x05, 0xbb, 0xbd, 0x0d, 0x0a, 0x9f, 0x06, 0xbb, 0x7b, 0x10,
  };
  static const uint8_t kAESCBCDecPlaintext[sizeof(kAESCBCDecCiphertext)] = {
      0x51, 0xa7, 0xa0, 0x1f, 0x6b, 0x79, 0x6c, 0xcd, 0x48, 0x03, 0xa1,
      0x41, 0xdc, 0x56, 0xa6, 0xc2, 0x16, 0xb5, 0xd1, 0xd3, 0xb7, 0x06,
      0xb2, 0x25, 0x6f, 0xa6, 0xd0, 0xd2, 0x0e, 0x6f, 0x19, 0xb5,
  };
  memcpy(aes_iv, kAESIV, sizeof(kAESIV));
  if (AES_set_decrypt_key(kAESKey, 8 * sizeof(kAESKey), &aes_key) != 0) {
    fprintf(stderr, "AES_set_decrypt_key failed.\n");
    goto err;
  }
  AES_cbc_encrypt(kAESCBCDecCiphertext, output, sizeof(kAESCBCDecCiphertext),
                  &aes_key, aes_iv, AES_DECRYPT);
  if (!check_test(kAESCBCDecPlaintext, output, sizeof(kAESCBCDecPlaintext),
                  "AES-CBC-decrypt KAT")) {
    goto err;
  }

  size_t out_len;
  uint8_t nonce[EVP_AEAD_MAX_NONCE_LENGTH];
  OPENSSL_memset(nonce, 0, sizeof(nonce));
  if (!EVP_AEAD_CTX_init(&aead_ctx, EVP_aead_aes_128_gcm(), kAESKey,
                         sizeof(kAESKey), 0, NULL)) {
    fprintf(stderr, "EVP_AEAD_CTX_init for AES-128-GCM failed.\n");
    goto err;
  }

  // AES-GCM Encryption KAT
  static const uint8_t kAESGCMEncPlaintext[32] = {
      0x8f, 0xcc, 0x40, 0x99, 0x80, 0x8e, 0x75, 0xca, 0xaf, 0xf5, 0x82,
      0x89, 0x88, 0x48, 0xa8, 0x8d, 0x80, 0x8b, 0x55, 0xab, 0x4e, 0x93,
      0x70, 0x79, 0x7d, 0x94, 0x0b, 0xe8, 0xcc, 0x1d, 0x78, 0x84,
  };
  static const uint8_t kAESGCMCiphertext[sizeof(kAESGCMEncPlaintext) + 16] = {
      0x87, 0x7b, 0xd5, 0x8d, 0x96, 0x3e, 0x4b, 0xe6, 0x64, 0x94, 0x40, 0x2f,
      0x61, 0x9b, 0x7e, 0x56, 0x52, 0x7d, 0xa4, 0x5a, 0xf9, 0xa6, 0xe2, 0xdb,
      0x1c, 0x63, 0x2e, 0x97, 0x93, 0x0f, 0xfb, 0xed, 0xb5, 0x9e, 0x1c, 0x20,
      0xb2, 0xb0, 0x58, 0xda, 0x48, 0x07, 0x2d, 0xbd, 0x96, 0x0d, 0x34, 0xc6,
  };
  if (!EVP_AEAD_CTX_seal(&aead_ctx, output, &out_len, sizeof(output), nonce,
                         EVP_AEAD_nonce_length(EVP_aead_aes_128_gcm()),
                         kAESGCMEncPlaintext, sizeof(kAESGCMEncPlaintext), NULL,
                         0) ||
      !check_test(kAESGCMCiphertext, output, sizeof(kAESGCMCiphertext),
                  "AES-GCM-encrypt KAT")) {
    fprintf(stderr, "EVP_AEAD_CTX_seal for AES-128-GCM failed.\n");
    goto err;
  }

  // AES-GCM Decryption KAT
  static const uint8_t kAESGCMDecCiphertext[48] = {
      0x35, 0xf3, 0x05, 0x8f, 0x87, 0x57, 0x60, 0xff, 0x09, 0xd3, 0x12, 0x0f,
      0x70, 0xc4, 0xbc, 0x9e, 0xd7, 0xa8, 0x68, 0x72, 0xe1, 0x34, 0x52, 0x20,
      0x21, 0x76, 0xf7, 0x37, 0x1a, 0xe0, 0x4f, 0xaa, 0xe1, 0xdd, 0x39, 0x19,
      0x20, 0xf5, 0xd1, 0x39, 0x53, 0xd8, 0x96, 0x78, 0x59, 0x94, 0x82, 0x3c,
  };
  static const uint8_t kAESGCMDecPlaintext[sizeof(kAESGCMDecCiphertext) - 16] =
      {
          0x3d, 0x44, 0x90, 0x9b, 0x91, 0xe7, 0x5e, 0xd3, 0xc2, 0xb2, 0xd0,
          0xa9, 0x99, 0x17, 0x6a, 0x45, 0x05, 0x5e, 0x99, 0x83, 0x56, 0x01,
          0xc0, 0x82, 0x40, 0x81, 0xd2, 0x48, 0x45, 0xf2, 0xcc, 0xc3,
      };
  if (!EVP_AEAD_CTX_open(&aead_ctx, output, &out_len, sizeof(output), nonce,
                         EVP_AEAD_nonce_length(EVP_aead_aes_128_gcm()),
                         kAESGCMDecCiphertext, sizeof(kAESGCMDecCiphertext),
                         NULL, 0) ||
      !check_test(kAESGCMDecPlaintext, output, sizeof(kAESGCMDecPlaintext),
                  "AES-GCM-decrypt KAT")) {
    fprintf(stderr,
            "AES-GCM-decrypt KAT failed because EVP_AEAD_CTX_open failed.\n");
    goto err;
  }

  // SHA-1 KAT
  static const uint8_t kSHA1Input[16] = {
      0x13, 0x2f, 0xd9, 0xba, 0xd5, 0xc1, 0x82, 0x62,
      0x63, 0xba, 0xfb, 0xb6, 0x99, 0xf7, 0x07, 0xa5,
  };
  static const uint8_t kSHA1Digest[20] = {
      0x94, 0x19, 0x55, 0x93, 0x0a, 0x58, 0x29, 0x38, 0xeb, 0xf5,
      0x09, 0x11, 0x6d, 0x1a, 0xfd, 0x0f, 0x1e, 0x11, 0xe3, 0xcb,
  };
  SHA1(kSHA1Input, sizeof(kSHA1Input), output);
  if (!check_test(kSHA1Digest, output, sizeof(kSHA1Digest),
                  "SHA-1 KAT")) {
    goto err;
  }

  if (!boringssl_self_test_sha512() ||
      !boringssl_self_test_sha3_256() ||
      !boringssl_self_test_hkdf_sha256()) {
    goto err;
  }

  // DBRG KAT
  static const uint8_t kDRBGEntropy[48] = {
      0xc4, 0xda, 0x07, 0x40, 0xd5, 0x05, 0xf1, 0xee, 0x28, 0x0b, 0x95, 0xe5,
      0x8c, 0x49, 0x31, 0xac, 0x6d, 0xe8, 0x46, 0xa0, 0x15, 0x2f, 0xbb, 0x4a,
      0x3f, 0x17, 0x4c, 0xf4, 0x78, 0x7a, 0x4f, 0x1a, 0x40, 0xc2, 0xb5, 0x0b,
      0xab, 0xe1, 0x4a, 0xae, 0x53, 0x0b, 0xe5, 0x88, 0x6d, 0x91, 0x0a, 0x27,
  };
  static const uint8_t kDRBGPersonalization[18] = "BCMPersonalization";
  static const uint8_t kDRBGAD[16] = "BCM DRBG KAT AD ";
  static const uint8_t kDRBGOutput[64] = {
      0x19, 0x1f, 0x2b, 0x49, 0x76, 0x85, 0xfd, 0x51, 0xb6, 0x56, 0xbc,
      0x1c, 0x7d, 0xd5, 0xdd, 0x44, 0x76, 0xa3, 0x5e, 0x17, 0x9b, 0x8e,
      0xb8, 0x98, 0x65, 0x12, 0xca, 0x35, 0x6c, 0xa0, 0x6f, 0xa0, 0x22,
      0xe4, 0xf6, 0xd8, 0x43, 0xed, 0x4e, 0x2d, 0x97, 0x39, 0x43, 0x3b,
      0x57, 0xfc, 0x23, 0x3f, 0x71, 0x0a, 0xe0, 0xed, 0xfe, 0xd5, 0xb8,
      0x67, 0x7a, 0x00, 0x39, 0xb2, 0x6e, 0xa9, 0x25, 0x97,
  };
  static const uint8_t kDRBGEntropy2[48] = {
      0xc7, 0x16, 0x1c, 0xa3, 0x6c, 0x23, 0x09, 0xb7, 0x16, 0xe9, 0x85, 0x9b,
      0xb9, 0x6c, 0x6d, 0x49, 0xbd, 0xc8, 0x35, 0x21, 0x03, 0xa1, 0x8c, 0xd2,
      0x4e, 0xf4, 0x2e, 0xc9, 0x7e, 0xf4, 0x6b, 0xf4, 0x46, 0xeb, 0x1a, 0x45,
      0x76, 0xc1, 0x86, 0xe9, 0x35, 0x18, 0x03, 0x76, 0x3a, 0x79, 0x12, 0xfe,
  };
  static const uint8_t kDRBGReseedOutput[64] = {
      0x00, 0xf2, 0x05, 0xaa, 0xfd, 0x11, 0x6c, 0x77, 0xbc, 0x81, 0x86,
      0x99, 0xca, 0x51, 0xcf, 0x80, 0x15, 0x9f, 0x02, 0x9e, 0x0b, 0xcd,
      0x26, 0xc8, 0x4b, 0x87, 0x8a, 0x15, 0x1a, 0xdd, 0xf2, 0xf3, 0xeb,
      0x94, 0x0b, 0x08, 0xc8, 0xc9, 0x57, 0xa4, 0x0b, 0x4b, 0x0f, 0x13,
      0xde, 0x7c, 0x0c, 0x6a, 0xac, 0x34, 0x4a, 0x9a, 0xf2, 0xd0, 0x83,
      0x02, 0x05, 0x17, 0xc9, 0x81, 0x8f, 0x2a, 0x81, 0x92,
  };
  CTR_DRBG_STATE drbg;
  if (!CTR_DRBG_init(&drbg, kDRBGEntropy, kDRBGPersonalization,
                     sizeof(kDRBGPersonalization)) ||
      !CTR_DRBG_generate(&drbg, output, sizeof(kDRBGOutput), kDRBGAD,
                         sizeof(kDRBGAD)) ||
      !check_test(kDRBGOutput, output, sizeof(kDRBGOutput),
                  "DRBG Generate KAT") ||
      !CTR_DRBG_reseed(&drbg, kDRBGEntropy2, kDRBGAD, sizeof(kDRBGAD)) ||
      !CTR_DRBG_generate(&drbg, output, sizeof(kDRBGReseedOutput), kDRBGAD,
                         sizeof(kDRBGAD)) ||
      !check_test(kDRBGReseedOutput, output, sizeof(kDRBGReseedOutput),
                  "DRBG-reseed KAT")) {
    fprintf(stderr, "CTR-DRBG failed.\n");
    goto err;
  }
  CTR_DRBG_clear(&drbg);

  CTR_DRBG_STATE kZeroDRBG;
  memset(&kZeroDRBG, 0, sizeof(kZeroDRBG));
  if (!check_test(&kZeroDRBG, &drbg, sizeof(drbg), "DRBG Clear KAT")) {
    goto err;
  }

  // TLS KDF KAT
  static const uint8_t kTLSSecret[32] = {
      0xab, 0xc3, 0x65, 0x7b, 0x09, 0x4c, 0x76, 0x28, 0xa0, 0xb2, 0x82,
      0x99, 0x6f, 0xe7, 0x5a, 0x75, 0xf4, 0x98, 0x4f, 0xd9, 0x4d, 0x4e,
      0xcc, 0x2f, 0xcf, 0x53, 0xa2, 0xc4, 0x69, 0xa3, 0xf7, 0x31,
  };
  static const char kTLSLabel[] = "FIPS self test";
  static const uint8_t kTLSSeed1[16] = {
      0x8f, 0x0d, 0xe8, 0xb6, 0x90, 0x8f, 0xb1, 0xd2,
      0x6d, 0x51, 0xf4, 0x79, 0x18, 0x63, 0x51, 0x65,
  };
  static const uint8_t kTLSSeed2[16] = {
      0x7d, 0x24, 0x1a, 0x9d, 0x3c, 0x59, 0xbf, 0x3c,
      0x31, 0x1e, 0x2b, 0x21, 0x41, 0x8d, 0x32, 0x81,
  };
  static const uint8_t kTLSOutput[32] = {
      0xe2, 0x1d, 0xd6, 0xc2, 0x68, 0xc7, 0x57, 0x03, 0x2c, 0x2c, 0xeb,
      0xbb, 0xb8, 0xa9, 0x7d, 0xe9, 0xee, 0xe6, 0xc9, 0x47, 0x83, 0x0a,
      0xbd, 0x11, 0x60, 0x5d, 0xd5, 0x2c, 0x47, 0xb6, 0x05, 0x88,
  };
  uint8_t tls_output[sizeof(kTLSOutput)];
  if (!CRYPTO_tls1_prf(EVP_sha256(), tls_output, sizeof(tls_output), kTLSSecret,
                       sizeof(kTLSSecret), kTLSLabel, sizeof(kTLSLabel),
                       kTLSSeed1, sizeof(kTLSSeed1), kTLSSeed2,
                       sizeof(kTLSSeed2)) ||
      !check_test(kTLSOutput, tls_output, sizeof(kTLSOutput), "TLS-KDF KAT")) {
    fprintf(stderr, "TLS KDF failed.\n");
    goto err;
  }

  // PBKDF2 KAT - password/salt data from RFC 6070, derived key generated by
  // Python's cryptography module
  static const uint8_t kPBKDF2Password[] = {
    'p', 'a', 's', 's', 'w', 'o', 'r', 'd', 'P', 'A', 'S', 'S', 'W', 'O', 'R',
    'D', 'p', 'a', 's', 's', 'w', 'o', 'r', 'd'
  };
  static const uint8_t kPBKDF2Salt[] = {
    's', 'a', 'l', 't', 'S', 'A', 'L', 'T', 's', 'a', 'l', 't', 'S', 'A', 'L',
    'T', 's', 'a', 'l', 't', 'S', 'A', 'L', 'T', 's', 'a', 'l', 't', 'S', 'A',
    'L', 'T', 's', 'a', 'l', 't'
  };
  const unsigned kPBKDF2Iterations = 2;
  static const uint8_t kPBKDF2DerivedKey[] = {
    0x13, 0xdc, 0x8a, 0x7c, 0x13, 0xd3, 0x72, 0xc9, 0x03, 0x82, 0x82, 0x2d,
    0x2d, 0xc4, 0x92, 0xf2, 0xed, 0x52, 0x46, 0x7f, 0xb7, 0x82, 0x8e, 0xa8,
    0x64    // 25 bytes
  };
  uint8_t pbkdf2_output[sizeof(kPBKDF2DerivedKey)];
  if (!PKCS5_PBKDF2_HMAC((const char *)kPBKDF2Password, sizeof(kPBKDF2Password),
                         kPBKDF2Salt, sizeof(kPBKDF2Salt), kPBKDF2Iterations,
                         EVP_sha256(), sizeof(kPBKDF2DerivedKey),
                         pbkdf2_output) ||
      !check_test(kPBKDF2DerivedKey, pbkdf2_output, sizeof(kPBKDF2DerivedKey),
                  "PBKDF2 KAT")) {
    fprintf(stderr, "PBKDF2 failed.\n");
    goto err;
  }

  static const uint8_t kSSKDFDigestInfo[] = {
      0xaf, 0x85, 0xce, 0x3e, 0xa3, 0x03, 0x35, 0x0a, 0x2a, 0xa1, 0x31,
      0xb8, 0x79, 0xea, 0x95, 0x4e, 0x1e, 0xe1, 0xe6, 0x85, 0xd2, 0xbf,
      0x9d, 0xb7, 0x95, 0x42, 0x02, 0xe2, 0xf5, 0xec, 0x30, 0x96};
  static const uint8_t kSSKDFDigestSharedSecret[] = {
      0x39, 0xa1, 0xe2, 0xb3, 0x89, 0x9e, 0x87, 0xef, 0xec, 0xf6, 0x27,
      0x12, 0x82, 0xd8, 0xf8, 0x00, 0x8f, 0x25, 0x26, 0x86, 0xdd, 0x35,
      0xbf, 0xc3, 0x9a, 0x0f, 0x71, 0x47, 0x8d, 0xa4, 0x8c, 0x69, 0x15,
      0x65, 0xce, 0xe4, 0x31, 0x25, 0x4d, 0xd5, 0x0c, 0xab, 0x74, 0x62,
      0xc6, 0xcf, 0x19, 0x9b, 0xe9, 0xbf, 0x5c};
  static const uint8_t kSSKDFDigestDerivedKey[] = {
      0x5a, 0x2e, 0x26, 0x64, 0x4d, 0x16, 0x22, 0x2c, 0xd6, 0x36, 0xa1, 0xfd,
      0xb5, 0x7b, 0xfa, 0xa1, 0x7f, 0x94, 0x44, 0x91, 0x27, 0x61, 0x2b, 0xcd,
      0x7b, 0xe1, 0xbb, 0x39, 0xcc, 0x18, 0xf3, 0x28, 0x93, 0xd3, 0xc6, 0x48,
      0xc1, 0x63, 0x72, 0xfb, 0x6e, 0x9c, 0x63, 0xde, 0x54, 0x33, 0xb1, 0xcc,
      0xde, 0xb5, 0x1b, 0xb5, 0xf1, 0x53, 0x68, 0xc8, 0xa8, 0x49, 0xa1, 0xe5,
      0xa4, 0xef, 0xc6, 0x66, 0xfd, 0x33, 0xee, 0xb9, 0xf6, 0x72, 0x8b, 0x04,
      0x79, 0xf7, 0x66, 0x68, 0xcf, 0xaf, 0xc1, 0x3a, 0x91, 0x36, 0x70, 0x74,
      0xde, 0xf2, 0xb5, 0x0e, 0x9d, 0x9a, 0x91, 0x8a, 0x12, 0x02, 0x10, 0x82,
      0x41, 0x65, 0xd5, 0x96, 0xad, 0x4f, 0x94, 0xa3, 0x23, 0x6e, 0xf7, 0xcf,
      0x58, 0x43, 0x28, 0x2a, 0x0a, 0x57, 0xa4, 0x83, 0x81, 0x9f, 0x63, 0xe0,
      0xcf, 0xb2, 0x08, 0x1d, 0xaf, 0x9c, 0xcf, 0x35, 0xc6, 0x6a, 0x03, 0xe7,
      0xa0, 0x2d, 0x38, 0x91, 0xf4, 0x50, 0x22, 0xe1, 0xc8, 0x9d, 0x88, 0x8a,
      0xa8, 0x08, 0x7e, 0x08, 0xf4, 0x5b, 0xab, 0xbc, 0x52, 0x06, 0x2b, 0x18,
      0xe6, 0xfb, 0x70, 0xc1, 0x2d, 0xcb, 0x29, 0xa1, 0x94, 0xd2, 0x3a, 0xbc,
      0x35, 0x1c, 0xfb, 0x3c, 0xf4, 0xf1, 0x61, 0xcc, 0x77, 0x5a, 0x3e, 0x71,
      0x1b, 0xb1, 0x50, 0x2d, 0x69, 0x01, 0xf6, 0x93, 0x14, 0x07, 0xa9, 0xae,
      0x86, 0x84, 0x76, 0xf9, 0x98, 0xd1, 0xca, 0x4c, 0xca, 0x29, 0x6a, 0x9f,
      0x14, 0x75, 0x2d, 0x14, 0xf4, 0x74, 0x27, 0xe6, 0x66, 0x28, 0x9f, 0x80,
      0x89, 0x2a, 0x3d, 0x14, 0xa8, 0x4f, 0xe3, 0x43, 0xfd, 0x78, 0xd0, 0xda,
      0xdb, 0xde, 0x18, 0x19, 0xac, 0xa9, 0x15, 0xf7, 0xc0, 0xc0, 0x24, 0x37,
      0x6b, 0x40, 0xcb, 0x34, 0xba, 0xe2, 0xd2, 0x6e, 0x9f, 0x45, 0x52, 0xb6,
      0xb1, 0xa2, 0x6f, 0xa5};  // 256 bytes
  uint8_t sskdf_digest_output[sizeof(kSSKDFDigestDerivedKey)];
  if (!SSKDF_digest(&sskdf_digest_output[0], sizeof(kSSKDFDigestDerivedKey),
                    EVP_sha256(), &kSSKDFDigestSharedSecret[0],
                    sizeof(kSSKDFDigestSharedSecret), &kSSKDFDigestInfo[0],
                    sizeof(kSSKDFDigestInfo)) ||
      !check_test(kSSKDFDigestDerivedKey, sskdf_digest_output,
                  sizeof(kSSKDFDigestDerivedKey), "SSKDF_digest KAT")) {
    fprintf(stderr, "SSKDF_digest failed.\n");
    goto err;
  }

  ret = 1;

err:
  EVP_AEAD_CTX_cleanup(&aead_ctx);

  return ret;
}

int BORINGSSL_self_test(void) {
  if (!boringssl_self_test_fast() ||
      // When requested to run self tests, also run the lazy tests.
      !boringssl_self_test_rsa() ||
      !boringssl_self_test_ecc() ||
      !boringssl_self_test_ffdh() ||
      !boringssl_self_test_ml_kem() ||
      !boringssl_self_test_eddsa()) {
    return 0;
  }

  return 1;
}

#if defined(BORINGSSL_FIPS)
int boringssl_self_test_startup(void) {
  return boringssl_self_test_fast();
}
#endif
