#!/usr/bin/env bash
# stop script on error
set -e

# The files generated by this script are used in unit tests that create TLS
# connections between a localhost server and client. Run this script if any
# certificates used for tests are expired.

# Files generated by this script:
# ca_root.crt:        root certificate authority
# server.crt:         certificate for server signed by ca_root.crt
# server.key:         private key for server.crt
# server_chain.crt:   certificate chain containing server.crt and ca_root.crt
# unittests.crt:      self-signed certificate
# unittests.p8:       private key, pkcs#8 syntax
# unittests.p12:      pkcs#12 file bundling the certificate and private key. Password is "1234"

# Create directory for use with certificate generation
mkdir certGeneration
# Copy files needed to generate new certificates
cp unittests.key certGeneration/unittests.key
cp unittests.conf certGeneration/unittests.conf
cp ca_root.cnf certGeneration/ca_root.cnf
cd certGeneration

# index.txt and serial are required for use with openssl config files
touch index.txt
echo 1000 > serial

# Generating a key for the new ca_root
openssl genrsa -out ca_root.key 2048

# Generate the ca_root certificate
openssl req -config ca_root.cnf \
            -key ca_root.key \
            -new -x509 -days 824 -sha256 -extensions v3_ca \
            -out ca_root.crt \
            -set_serial 00 \
            -subj '/C=US/ST=Washington/L=Seattle/O=Amazon/OU=SDKs/CN=localhostCA/emailAddress=aws-sdk-common-runtime@amazon.com'

# Generate a private key for the server
openssl genrsa -out server.key 2048

# Generate a certificate signing request for the server
openssl req -new -sha256 \
            -key server.key \
            -out server.csr \
            -set_serial 02 \
            -subj '/C=US/ST=Washington/L=Seattle/O=Amazon/OU=SDKs/CN=localhost/emailAddress=aws-sdk-common-runtime@amazon.com'

# Sign the server signing request with ca_root
yes | openssl ca -config ca_root.cnf \
            -extensions server_cert \
            -days 824 -notext -md sha256 \
            -in server.csr \
            -out server.crt

# Generate a certificate chain containing the ca_root and server certificates
cat server.crt ca_root.crt > server_chain.crt

openssl req -x509 -new \
            -key unittests.key \
            -config unittests.conf \
            -out unittests.crt \
            -days 824

openssl pkcs8 -topk8 \
            -out unittests.p8 \
            -in unittests.key \
            -nocrypt

openssl pkcs12 -export \
            -out unittests.p12 \
            -inkey unittests.key \
            -in unittests.crt \
            -password pass:1234

# Copy the generated certificates and keys to the resouces folder
cd ..
cp certGeneration/ca_root.crt ./ca_root.crt
cp certGeneration/server.crt ./server.crt
cp certGeneration/server.key ./server.key
cp certGeneration/server_chain.crt ./server_chain.crt
cp certGeneration/server.crt ./server.crt

cp certGeneration/unittests.crt ./unittests.crt
cp certGeneration/unittests.p8 ./unittests.p8
cp certGeneration/unittests.p12 ./unittests.p12

# Clean up the certGeneration folder
rm -r certGeneration
