/*
 * Copyright 1999,2004 The Apache Software Foundation.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */ 


package org.apache.naming;

import java.util.Hashtable;

/**
 * Handles the access control on the JNDI contexts.
 *
 * @author Remy Maucherat
 * @version $Revision: 1.4 $ $Date: 2004/08/29 16:46:15 $
 */

public class ContextAccessController {


    // -------------------------------------------------------------- Variables


    /**
     * Catalina context names on which writing is not allowed.
     */
    private static Hashtable readOnlyContexts = new Hashtable();


    /**
     * Security tokens repository.
     */
    private static Hashtable securityTokens = new Hashtable();


    // --------------------------------------------------------- Public Methods


    /**
     * Set a security token for a context. Can be set only once.
     * 
     * @param name Name of the context
     * @param token Security token
     */
    public static void setSecurityToken(Object name, Object token) {
        if ((!securityTokens.containsKey(name)) && (token != null)) {
            securityTokens.put(name, token);
        }
    }


    /**
     * Remove a security token for a context.
     * 
     * @param name Name of the context
     * @param token Security token
     */
    public static void unsetSecurityToken(Object name, Object token) {
        if (checkSecurityToken(name, token)) {
            securityTokens.remove(name);
        }
    }


    /**
     * Check a submitted security token. The submitted token must be equal to
     * the token present in the repository. If no token is present for the 
     * context, then returns true.
     * 
     * @param name Name of the context
     * @param token Submitted security token
     */
    public static boolean checkSecurityToken
        (Object name, Object token) {
        Object refToken = securityTokens.get(name);
        if (refToken == null)
            return (true);
        if ((refToken != null) && (refToken.equals(token)))
            return (true);
        return (false);
    }


    /**
     * Allow writing to a context.
     * 
     * @param name Name of the context
     * @param token Security token
     */
    public static void setWritable(Object name, Object token) {
        if (checkSecurityToken(name, token))
            readOnlyContexts.remove(name);
    }


    /**
     * Set whether or not a context is writable.
     * 
     * @param name Name of the context
     */
    public static void setReadOnly(Object name) {
        readOnlyContexts.put(name, name);
    }


    /**
     * Returns if a context is writable.
     * 
     * @param name Name of the context
     */
    public static boolean isWritable(Object name) {
        return !(readOnlyContexts.containsKey(name));
    }


}

