/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is the reusable ccl java library
 * (http://www.kclee.com/clemens/java/ccl/).
 *
 * The Initial Developer of the Original Code is
 * Chr. Clemens Lee.
 * Portions created by Chr. Clemens Lee are Copyright (C) 2002
 * Chr. Clemens Lee. All Rights Reserved.
 *
 * Contributor(s): Chr. Clemens Lee <clemens@kclee.com>
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

package ccl.platform;

import ccl.util.ClassPathUtil;
import ccl.util.FileUtil;
import ccl.util.Util;
import java.awt.Dialog;
import java.awt.Frame;
import java.io.File;
import java.io.IOException;
import java.util.zip.ZipException;

/**
 * Linux native methods are invoked in this class. But users
 * will use the Native class to use these methods.
 * 
 * @author  Chr. Clemens Lee
 * @version $Id: Linux.java,v 1.9 2003/05/01 16:43:57 clemens Exp clemens $
 */
public class Linux extends Native 
{
    private static boolean _bLibLoaded = false;

    static 
    {
        _bLibLoaded = false;

        // libLinux.so is expected to be in $CCL_HOME/lib/linux/libLinux.so.
        // So we first search for ccl.jar and expect it to be in $CCL_HOME/lib.
        // Should the jar be unpacked, we search for classes/ccl/platform/Linux.class
        // and from there expect to find the lib/libLinux.so.
        // As a last try, it also looks directly inside the lib directory.
        // Then make sure absolute path is used.
        String sLibFullName = FileUtil.getPackagePath( "ccl.platform" );
        Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
        if ( !Util.isEmpty( sLibFullName ) ) 
        {
            // we found Linux.class, it's not in a jar file
            sLibFullName = FileUtil.concatPath( sLibFullName
                                                , ".."
                                                  + File.separator
                                                  + ".."
                                                  + File.separator
                                                  + ".."
                                                  + File.separator
                                                  + "lib" );
            Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
            if ( FileUtil.existsFile
                 ( FileUtil.concatPath( sLibFullName
                                        , "libLinux.so" ) ) )
            {
                // found lib/libLinux.so
                sLibFullName = FileUtil.concatPath( sLibFullName
                                                    , "libLinux.so" );
                Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
            } 
            else if ( FileUtil.existsFile
                      ( FileUtil.concatPath( sLibFullName
                                             , "linux"
                                             + File.separator
                                             + "libLinux.so" ) ) )
            {
                sLibFullName = FileUtil.concatPath( sLibFullName
                                                    , "linux"
                                                    + File.separator
                                                    + "libLinux.so" );
                Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
            }
            else 
            {
                sLibFullName = "";
            }
        }
        Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
        if ( Util.isEmpty( sLibFullName ) ) 
        {
            try 
            {
                sLibFullName = ClassPathUtil.getJarFileName( "ccl.platform.Linux" );
                Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
            }
            catch( ZipException zipException ) 
            {
                Util.printlnErr( "ccl.platform.Linux.<static>.zipException: "
                                 + zipException );
                sLibFullName = null;
            }
            catch( IOException pIOException ) 
            {
                Util.printlnErr( "ccl.platform.Linux.<static>.pIOException: "
                                 + pIOException );
                sLibFullName = null;
            }
            if ( !Util.isEmpty( sLibFullName ) ) 
            {
                // found ccl.jar
                String sLibDirectory = new File( sLibFullName ).getParent();
                sLibFullName = FileUtil.concatPath( sLibDirectory
                                                    , "libLinux.so" );
                Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
                if ( !FileUtil.existsFile( sLibFullName ) ) 
                {
                    sLibFullName = FileUtil.concatPath( sLibDirectory
                                                        , "linux"
                                                        + File.separator
                                                        + "libLinux.so" );
                    Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
                    if ( !FileUtil.existsFile( sLibFullName ) ) 
                    {
                        sLibFullName = "";
                    }
                }
            }
        }
        Util.debug( "ccl.platform.Linux.<static>.sLibFullName: " + sLibFullName );
        if ( Util.isEmpty( sLibFullName ) ) 
        {
            Util.printlnErr( "libLinux.so not found in ccl.platform.Linux!\n"
                             + "Put libLinux.so in the same directory where ccl.jar is,\n"
                             + "or underneath a linux directory." );
        }
        else 
        {
            Runtime.getRuntime().load
                   ( FileUtil.getAbsoluteFileName
                     ( sLibFullName ) );
            _bLibLoaded = true;
        }
        Util.debug( "ccl.platform.Linux.<static>._bLibLoaded: " + _bLibLoaded );
    }
    
    /**
     * This is a singleton class. Use 'Native.getInstance()'
     * to create a Native or Linux object.
     */
    protected Linux() 
    {
        super();
    }
    
    /**
     * Create a Linux symbolic link implemented in C.
     */
    private static native int _symlink( String sOldPath_, 
                                        String sNewPath_ );

    /**
     * Remove a Linux symbolic link implemented in C.
     */
    private static native int _unlink( String sPath_ );
    
    /**
     * Get pid in C.
     */
    private static native int _getpid();
    
    /**
     * Get process group id in C.
     */
    private static native int _getpgid( int pid_ );

    /**
     * Get unix windows id.
     */
    public static String getWindowID( String sFrameName_ )
    {
        return getWindowID( sFrameName_, Util.isDebug()?1:0 );
    }
    
    /**
     * Get unix windows id.
     */
    public static native String getWindowID( String sFrameName_, int debug );
    
    /**
     * Set focus on Unix window's id.
     */
    public static native void requestFocus( String sFrameID_ );
    
    /**
     * Moves an X window.
     */
    public static native void moveWindow( String sFrameID_, int x, int y );
    
    /**
     * Set focus on Unix window with given title.
     */
    public void requestInputFocus( String sWindowTitle_ ) 
    {
        requestFocus( getWindowID( sWindowTitle_ ) );
    }

    /**
     * Create a symbolic link.
     */
    public int symlink( String sOldPath_, 
                        String sNewPath_ )
    {
        return _symlink( sOldPath_, sNewPath_ );
    }
    
    /**
     * Remove a symbolic link.
     */
    public boolean unlink( String sPath_ ) 
    {
        return (_unlink( sPath_ ) != 0);
    }
    
    /**
     * Is the libLinux.so library really loaded?
     */
    public static boolean isLibLoaded() 
    {
        return _bLibLoaded;
    }
    
    /**
     * Uses native POSIX 'getpid()' to get the process ID of the current Java program/process.
     */
    public int getPIDViaPOSIX() 
    {
        return _getpid();
    }
    
    /**
     * Uses the /proc system to get the process ID of the current Java program/process.
     * This method exists only in the Linux class, not in the Native super class
     * (makes no sense outside of Linux).
     */
    public int getPIDViaProc() 
    {
        String sPid = FileUtil.normalizeFileName( "/proc/self" );
        sPid = sPid.substring( 6 );
        int pid = Util.atoi( sPid );
        /*
        if (5==5) return Util.atoi( sPid );

        int pid = -1;
        
        String sCatOutput = "<no ls output>";
        try 
        {
            String sPSOutput = Util.systemAndWait
                   ( Util.objectsToVector( new String[] { "ps", "-f" } ) );
            Util.debug( this, "getPIDViaProc().ps \n"
                        + sPSOutput );
            sCatOutput = Util.systemAndWait
                   ( Util.objectsToVector( new String[] { "cat", "/proc/self/stat" } ) );
            Vector vStat = Util.stringToLines( sCatOutput, ' ' );
            / *sLSOutput = sLSOutput.substring( sLSOutput.lastIndexOf( " -> " ) + 4,
              sLSOutput.length() -1 );* /
            sCatOutput = (String)vStat.elementAt( 3 );
            Util.debug( this, "getPIDViaProc().sCatOutput: "
                        + sCatOutput );
            
            pid = Util.atoi( sCatOutput );
            / *
            sPSOutput = Util.systemAndWait
                   ( Util.objectsToVector( new String[] { "ps", "-p", sCatOutput, "-o", "pid", "-o", "pgid", "-o", "cmd" } ) );
            Util.println( "proc ps\n"
                          + sPSOutput );
            * /
        }
        catch( Exception exception ) 
        {
            Util.printlnErr( "External cat /proc/self/stat command or output parsing failed: " + sCatOutput );
            Util.printlnErr( exception );
        }
        */

        return pid;
    }

    /**
     * Get Process ID of the current Java program/process.
     */
    public int getPID() 
    {
        return getPIDViaPOSIX();
    }
    
    /**
     * E.g. in IBM's JDK 1.3 for Linux multiple processes are
     * started. getPID() returns the current process ID.
     * This method returns the main Java process ID.
     */
    public int getMainJavaPID() 
    {
        return getPIDViaProc();
    }

    /**
     * Get process group id for a given pid.
     * Use pid = 0 to get the pgrpid of the current process.
     */
    public int getPGID( int pid_ ) 
    {
        return _getpgid( pid_ );
    }
    
    /**
     * Sets the location of the frame using X Window.
     * The wrong window might be change if there are two
     * with the same title.
     */
    public void setLocation( Frame frame, int x, int y )
    {
        String sTitle = frame.getTitle();
        String sWindowId = getWindowID( sTitle );
        if ( sWindowId == null )
        {
            super.setLocation( frame, x, y );
        }
        else
        {
            Util.debug( this, "setLocation(..).sWindowId: " + sWindowId );
            moveWindow( sWindowId, x, y );
        }
    }
  
    /**
     * Sets the location of the dialog using X Window.
     * The wrong window might be change if there are two
     * with the same title.
     */
    public void setLocation( Dialog dialog, int x, int y )
    {
        String sTitle = dialog.getTitle();
        String sWindowId = getWindowID( sTitle );
        if ( sWindowId == null )
        {
            super.setLocation( dialog, x, y );
        } 
        else
        {
            Util.debug( this, "setLocation(..).sWindowId: " + sWindowId );
            moveWindow( sWindowId, x, y );
        }
    }
}
