/*
 * Copyright (C) 2004-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <X11/Xlib.h>
#include <X11/cursorfont.h>

#include "_xitk.h"
#include "_backend.h"
#include "window.h"
#include "cursors_x11.h"

static const struct cursors_init_s {
#if defined(XC_num_glyphs) && (XC_num_glyphs < 256)
  char name[23]; /** << name[0] == '_' means xitk embedded, plain x otherwise. */
  uint8_t x_shape;
#else
  char name[24]; /** << name[0] == '_' means xitk embedded, plain x otherwise. */
  unsigned int x_shape;
#endif
  const char *map; /** << (fg / bg / trans) normal: (* / . /  ) hotspot: (# / + / -) */
} cursors_init[xitk_cursor_num_glyphs] = {
  [xitk_cursor_invisible]           = { "_invisible",           0,
    "|--------"
    "|--------"
    "|--------"
    "|--------"
    "|--------"
    "|--------"
    "|--------"
    "|--------",
  },
  [xitk_cursor_X_cursor]            = {  "X_cursor",            XC_X_cursor,            NULL },
  [xitk_cursor_arrow]               = {  "arrow",               XC_arrow,               NULL },
  [xitk_cursor_based_arrow_down]    = {  "based_arrow_down",    XC_based_arrow_down,    NULL },
  [xitk_cursor_based_arrow_up]      = {  "based_arrow_up",      XC_based_arrow_up,      NULL },
  [xitk_cursor_boat]                = {  "boat",                XC_boat,                NULL },
  [xitk_cursor_bogosity]            = {  "bogosity",            XC_bogosity,            NULL },
  [xitk_cursor_bottom_left_corner]  = { "_bottom_left_corner",  XC_bottom_left_corner,
    "|       ********"
    "|       *......*"
    "|        *.....*"
    "|         *....*"
    "|        *.....*"
    "|       *...*..*"
    "|      *...* *.*"
    "|**   *...*   **"
    "|*.* *...*"
    "|*..*...*"
    "|*.....*"
    "|*....*"
    "|*.....*"
    "|*......*"
    "|#*******"
  },
  [xitk_cursor_bottom_right_corner] = { "_bottom_right_corner", XC_bottom_right_corner,
    "|********"
    "|*......*"
    "|*.....*"
    "|*....*"
    "|*.....*"
    "|*..*...*"
    "|*.* *...*"
    "|**   *...*   **"
    "|      *...* *.*"
    "|       *...*..*"
    "|        *.....*"
    "|         *....*"
    "|        *.....*"
    "|       *......*"
    "|       *******#"
  },
  [xitk_cursor_bottom_side]         = { "_bottom_side",         XC_bottom_side,
    "|    **"
    "|   *..*"
    "|  *....*"
    "| *......*"
    "|*........*"
    "|****..****"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|****..****"
    "|*........*"
    "| *......*"
    "|  *....*"
    "|   *..*"
    "|    #*"
  },
  [xitk_cursor_bottom_tee]          = {  "bottom_tee",          XC_bottom_tee,          NULL },
  [xitk_cursor_box_spiral]          = {  "box_spiral",          XC_box_spiral,          NULL },
  [xitk_cursor_center_ptr]          = { "_center_ptr",          XC_center_ptr,
    "|    **"
    "|   *+.*"
    "|  *....*"
    "| *......*"
    "|*........*"
    "|****..****"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   ****"
  },
  [xitk_cursor_circle]              = { "_circle",              XC_circle,
    "| *"
    "| #*"
    "| *.*"
    "| *..*"
    "| *...*"
    "| *....*"
    "| *.....*"
    "| *......*"
    "| *.......*"
    "| *........*"
    "| *.....*****"
    "| *..*..*"
    "| *.* *..*"
    "| **  *..*       ...."
    "| *    *..*    ........"
    "|      *..*   ...    ..."
    "|       *..*  ....    .."
    "|       *..* .. ...    .."
    "|        *** ..  ...   .."
    "|            ..   ...  .."
    "|            ..    ... .."
    "|             ..    ...."
    "|             ...    ..."
    "|              ........"
    "|                ...."
  },
  [xitk_cursor_clock]               = {  "clock",               XC_clock,               NULL },
  [xitk_cursor_coffee_mug]          = {  "coffee_mug",          XC_coffee_mug,          NULL },
  [xitk_cursor_cross]               = { "_cross",               XC_cross,
    "|          *"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|..........*........"
    "|**********#**********"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|         .*"
    "|          *"
  },
  [xitk_cursor_cross_reverse]       = {  "cross_reverse",       XC_cross_reverse,       NULL },
  [xitk_cursor_crosshair]           = {  "crosshair",           XC_crosshair,           NULL },
  [xitk_cursor_diamond_cross]       = {  "diamond_cross",       XC_diamond_cross,       NULL },
  [xitk_cursor_dot]                 = {  "dot",                 XC_dot,                 NULL },
  [xitk_cursor_dotbox]              = { "_dotbox",              XC_dotbox,
    "|*** ** ** ** ** ***"
    "|*                 *"
    "|*                 *"
    "|"
    "|*                 *"
    "|*       ....      *"
    "|      .....**"
    "|*     .....***    *"
    "|*    ....*****    *"
    "|     *.**#****"
    "|*    *********    *"
    "|*     ******.*    *"
    "|      ****.**"
    "|*       ***       *"
    "|*                 *"
    "|"
    "|*                 *"
    "|*                 *"
    "|*** ** ** ** ** ***"
  },
  [xitk_cursor_double_arrow]        = { "_double_arrow",        XC_double_arrow,
    "|    **"
    "|   *..*"
    "|  *....*"
    "| *......*"
    "|*........*"
    "|****..****"
    "|   *..*"
    "|   *..*"
    "|   *+.*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|****..****"
    "|*........*"
    "| *......*"
    "|  *....*"
    "|   *..*"
    "|    **"
  },
  [xitk_cursor_draft_large]         = {  "draft_large",         XC_draft_large,         NULL },
  [xitk_cursor_draft_small]         = {  "draft_small",         XC_draft_small,         NULL },
  [xitk_cursor_draped_box]          = { "_draped_box",          XC_draped_box,
    "|         ...*.."
    "|       .********"
    "|      .**********"
    "|      ***********"
    "|      ***********"
    "|      .*********."
    "|      .*   *   *."
    "|      ..   *   *."
    "|      .*.......**"
    "|       ...*....-"
    "|        ...**.."
    "|        ......."
    "|  .*    ......*    *."
    "|  **     *...**  *.."
    "|  ***     *.**  .**.*"
    "| *.***.    _   **...*"
    "|.***.***..   .**..*"
    "|      .**..***.."
    "| **    ..***..***   .*"
    "| .*******.*  ..**.****"
    "| .**.**.        ....*"
    "| .*..             .**"
    "|  ..                ."
  },
  [xitk_cursor_exchange]            = {  "exchange",            XC_exchange,            NULL },
  [xitk_cursor_fleur]               = { "_fleur",               XC_fleur,
    "|          **"
    "|         *..*"
    "|        *....*"
    "|       *......*"
    "|      *........*"
    "|      ****..****"
    "|    **   *..*   **"
    "|   *.*   *..*   *.*"
    "|  *..*   *..*   *..*"
    "| *...*****..*****...*"
    "|*.........+..........*"
    "|*....................*"
    "| *...*****..*****...*"
    "|  *..*   *..*   *..*"
    "|   *.*   *..*   *.*"
    "|    **   *..*   **"
    "|      ****..****"
    "|      *........*"
    "|       *......*"
    "|        *....*"
    "|         *..*"
    "|          **"
  },
  [xitk_cursor_gobbler]             = {  "gobbler",             XC_gobbler,             NULL },
  [xitk_cursor_gumby]               = {  "gumby",               XC_gumby,               NULL },
  [xitk_cursor_hand1]               = {  "hand1",               XC_hand1,               NULL },
  [xitk_cursor_hand2]               = { "_hand2",               XC_hand2,
    "|    **"
    "|   *+.*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..***"
    "|   *..*..***"
    "|   *..*..*..**"
    "| ***..*..*..*.*"
    "|**.*..*..*..*..*"
    "|*..*........*..*"
    "|*..*...........*"
    "|*..*...........*"
    "|*..............*"
    "|*..............*"
    "|*..............*"
    "| *.............*"
    "| *............**"
    "|  *...........*"
    "|  *..........**"
    "|  ************"
  },
  [xitk_cursor_drag]                = { "_drag",                XC_fleur,
    "|      ***"
    "|    ***..***"
    "|   *+.*..*..**"
    "| ***..*..*..*.*"
    "|**.*..*..*..*..*"
    "|*..*........*..*"
    "|*..*...........*"
    "|*..*...........*"
    "|*..............*"
    "|*..............*"
    "|*..............*"
    "| *.............*"
    "| *............**"
    "|  *...........*"
    "|  *..........**"
    "|  ************"
  },
  [xitk_cursor_heart]               = {  "heart",               XC_heart,               NULL },
  [xitk_cursor_icon]                = {  "icon",                XC_icon,                NULL },
  [xitk_cursor_iron_cross]          = {  "iron_cross",          XC_iron_cross,          NULL },
  [xitk_cursor_left_ptr]            = { "_left_ptr",            XC_left_ptr,
    "|*"
    "|#*"
    "|*.*"
    "|*..*"
    "|*...*"
    "|*....*"
    "|*.....*"
    "|*......*"
    "|*.......*"
    "|*........*"
    "|*.....*****"
    "|*..*..*"
    "|*.* *..*"
    "|**  *..*"
    "|*    *..*"
    "|     *..*"
    "|      *..*"
    "|      *..*"
    "|       ***"
  },
  [xitk_cursor_left_side]           = { "_left_side",           XC_left_side,
    "|    **       **"
    "|   *.*       *.*"
    "|  *..*       *..*"
    "| *...*********...*"
    "|*+................*"
    "|*.................*"
    "| *...*********...*"
    "|  *..*       *..*"
    "|   *.*       *.*"
    "|    **       **"
  },
  [xitk_cursor_left_tee]            = {  "left_tee",            XC_left_tee,            NULL },
  [xitk_cursor_leftbutton]          = {  "leftbutton",          XC_leftbutton,          NULL },
  [xitk_cursor_ll_angle]            = {  "ll_angle",            XC_ll_angle,            NULL },
  [xitk_cursor_lr_angle]            = {  "lr_angle",            XC_lr_angle,            NULL },
  [xitk_cursor_man]                 = {  "man",                 XC_man,                 NULL },
  [xitk_cursor_middlebutton]        = {  "middlebutton",        XC_middlebutton,        NULL },
  [xitk_cursor_mouse]               = {  "mouse",               XC_mouse,               NULL },
  [xitk_cursor_pencil]              = { "_pencil",              XC_pencil,
    "|#"
    "| **"
    "| ****"
    "|  **.*"
    "|  ***.*"
    "|   **..*"
    "|    **..*"
    "|     **..*"
    "|      **..*"
    "|       **..*"
    "|        **..*"
    "|         ***.*  *"
    "|          ***.* **"
    "|           ***.****"
    "|            ***.*.**"
    "|             ***.*.**"
    "|              ***.*.**"
    "|               ***.*.*"
    "|                ***.**"
    "|                 ***.*"
    "|                  ****"
    "|                   **"
  },
  [xitk_cursor_pirate]              = {  "pirate",              XC_pirate,              NULL },
  [xitk_cursor_plus]                = {  "plus",                XC_plus,                NULL },
  [xitk_cursor_question_arrow]      = { "_question_arrow",      XC_question_arrow,
    "|*         *******"
    "|#*       ***  ****"
    "|*.*     ***    ****"
    "|*..*    ***    ****"
    "|*...*   ***    ****"
    "|*....*  ***    ***"
    "|*.....*       ***"
    "|*......*     ***"
    "|*.......*   ***"
    "|*........*  ***"
    "|*.....***** ***"
    "|*..*..*"
    "|*.* *..*    ***"
    "|**  *..*   *****"
    "|*    *..*   ***"
    "|     *..*"
    "|      *..*"
    "|      *..*"
    "|       ***"
  },
  [xitk_cursor_right_ptr]           = { "_right_ptr",           XC_right_ptr,
    "|          *"
    "|         *#"
    "|        *.*"
    "|       *..*"
    "|      *...*"
    "|     *....*"
    "|    *.....*"
    "|   *......*"
    "|  *.......*"
    "| *........*"
    "|*****.....*"
    "|    *..*..*"
    "|   *..* *.*"
    "|   *..*  **"
    "|  *..*    *"
    "|  *..*"
    "| *..*"
    "| *..*"
    "| ***"
  },
  [xitk_cursor_right_side]          = { "_right_side",          XC_right_side,
    "|    **       **"
    "|   *.*       *.*"
    "|  *..*       *..*"
    "| *...*********...*"
    "|*................+*"
    "|*.................*"
    "| *...*********...*"
    "|  *..*       *..*"
    "|   *.*       *.*"
    "|    **       **"
  },
  [xitk_cursor_right_tee]           = {  "right_tee",           XC_right_tee,           NULL },
  [xitk_cursor_rightbutton]         = {  "rightbutton",         XC_rightbutton,         NULL },
  [xitk_cursor_rtl_logo]            = {  "rtl_logo",            XC_rtl_logo,            NULL },
  [xitk_cursor_sailboat]            = {  "sailboat",            XC_sailboat,            NULL },
  [xitk_cursor_sb_down_arrow]       = { "_sb_down_arrow",       XC_sb_down_arrow,
    "|   ****"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|****..****"
    "|*........*"
    "| *......*"
    "|  *....*"
    "|   *+.*"
    "|    **"
  },
  [xitk_cursor_sb_h_double_arrow]   = { "_sb_h_double_arrow",   XC_sb_h_double_arrow,
    "|    **       **"
    "|   *.*       *.*"
    "|  *..*       *..*"
    "| *...*********...*"
    "|*........+........*"
    "|*.................*"
    "| *...*********...*"
    "|  *..*       *..*"
    "|   *.*       *.*"
    "|    **       **"
  },
  [xitk_cursor_sb_left_arrow]       = { "_sb_left_arrow",       XC_sb_left_arrow,
    "|    **"
    "|   *.*"
    "|  *..*"
    "| *...************"
    "|*+..............*"
    "|*...............*"
    "| *...************"
    "|  *..*"
    "|   *.*"
    "|    **"
  },
  [xitk_cursor_sb_right_arrow]      = { "_sb_right_arrow",      XC_sb_right_arrow,
    "|           **"
    "|           *.*"
    "|           *..*"
    "|************...*"
    "|*..............+*"
    "|*...............*"
    "|************...*"
    "|           *..*"
    "|           *.*"
    "|           **"
  },
  [xitk_cursor_sb_up_arrow]         = { "_sb_up_arrow",         XC_sb_up_arrow,
    "|    **"
    "|   *+.*"
    "|  *....*"
    "| *......*"
    "|*........*"
    "|****..****"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   ****"
  },
  [xitk_cursor_sb_v_double_arrow]   = { "_sb_v_double_arrow",   XC_sb_v_double_arrow,
    "|    **"
    "|   *..*"
    "|  *....*"
    "| *......*"
    "|*........*"
    "|****..****"
    "|   *..*"
    "|   *..*"
    "|   *+.*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|****..****"
    "|*........*"
    "| *......*"
    "|  *....*"
    "|   *..*"
    "|    **"
  },
  [xitk_cursor_shuttle]             = {  "shuttle",             XC_shuttle,             NULL },
  [xitk_cursor_sizing]              = {  "sizing",              XC_sizing,              NULL },
  [xitk_cursor_spider]              = {  "spider",              XC_spider,              NULL },
  [xitk_cursor_spraycan]            = {  "spraycan",            XC_spraycan,            NULL },
  [xitk_cursor_star]                = {  "star",                XC_star,                NULL },
  [xitk_cursor_target]              = {  "target",              XC_target,              NULL },
  [xitk_cursor_tcross]              = {  "tcross",              XC_tcross,              NULL },
  [xitk_cursor_top_left_arrow]      = {  "top_left_arrow",      XC_top_left_arrow,      NULL },
  [xitk_cursor_top_left_corner]     = { "_top_left_corner",     XC_top_left_corner,
    "|#*******"
    "|*......*"
    "|*.....*"
    "|*....*"
    "|*.....*"
    "|*..*...*"
    "|*.* *...*"
    "|**   *...*   **"
    "|      *...* *.*"
    "|       *...*..*"
    "|        *.....*"
    "|         *....*"
    "|        *.....*"
    "|       *......*"
    "|       ********"
  },
  [xitk_cursor_top_right_corner]    = { "_top_right_corner",    XC_top_right_corner,
    "|       *******#"
    "|       *......*"
    "|        *.....*"
    "|         *....*"
    "|        *.....*"
    "|       *...*..*"
    "|      *...* *.*"
    "|**   *...*   **"
    "|*.* *...*"
    "|*..*...*"
    "|*.....*"
    "|*....*"
    "|*.....*"
    "|*......*"
    "|********"
  },
  [xitk_cursor_top_side]            = { "_top_side",            XC_top_side,
    "|    #*"
    "|   *..*"
    "|  *....*"
    "| *......*"
    "|*........*"
    "|****..****"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|   *..*"
    "|****..****"
    "|*........*"
    "| *......*"
    "|  *....*"
    "|   *..*"
    "|    **"
  },
  [xitk_cursor_top_tee]             = {  "top_tee",             XC_top_tee,             NULL },
  [xitk_cursor_trek]                = {  "trek",                XC_trek,                NULL },
  [xitk_cursor_ul_angle]            = {  "ul_angle",            XC_ul_angle,            NULL },
  [xitk_cursor_umbrella]            = {  "umbrella",            XC_umbrella,            NULL },
  [xitk_cursor_ur_angle]            = {  "ur_angle",            XC_ur_angle,            NULL },
  [xitk_cursor_watch]               = {  "watch",               XC_watch,               NULL },
  [xitk_cursor_xterm]               = { "_xterm",               XC_xterm,
    "|*******"
    "|...*.."
    "|  .*"
    "|  .*"
    "|  .*"
    "|  .*"
    "|  .*"
    "|  .#"
    "|  .*"
    "|  .*"
    "|  .*"
    "|  .*"
    "|  .*"
    "|  .*"
    "|  .*"
    "|...*.."
    "|*******"
  }
};

struct cursors_s {
  Cursor cursor;
  Pixmap p;
  Pixmap mask;
};

struct xitk_x11_cursors_s {
  int verbosity;
  xitk_be_display_t *display;
  struct cursors_s cursors[xitk_cursor_num_glyphs];
};

static void _cursors_create_cursor (xitk_x11_cursors_t *c, xitk_cursors_t idx) {
  Display *display = (Display *)c->display->id;
  struct cursors_s *cursor = &c->cursors[idx];
  const struct cursors_init_s *cursor_init = &cursors_init[idx];

  cursor->cursor = None;
  /* if both the x and xitk versions exitst, follow user pref. */
  if ((xitk_get_cfg_num (c->display->be->xitk, XITK_CURSORS_FEATURE) && (cursor_init->name[0] == '_')) ||
      (cursor_init->x_shape == 0)) {
    XColor bg = { .red = 255 << 8, .green = 255 << 8, .blue  = 255 << 8 },
           fg = { .red = 0, .green = 0, .blue = 0 };
    char img[1536], mask[1536];
    const char *p;
    uint8_t m;
    uint32_t wx = 0, w = 0, h = 0, hx = 0, hy = 0;

    for (p = cursor_init->map; *p; p++) {
      if (*p == '|') {
        h += 1;
        if (hx > w)
          w = hx;
        hx = wx = 0;
      } else if (*p == ' ') {
        wx += 1;
      } else {
        wx += 1;
        hx = wx;
      }
    }
    if (hx > w)
      w = hx;

    if (w && h) {
      int32_t x, y;

      wx = (w + 7) >> 3;
      if (wx * h > sizeof (img))
        h = sizeof (img) / wx;

      memset (img, 0, wx * h);
      memset (mask, 0, wx * h);

      m = 0x01;
      x = 0;
      y = -1;
      for (p = cursor_init->map; *p; p++) {
        uint32_t u;

        switch (*p) {
          case '-':
            hx = x;
            hy = y;
            /* fall through */
          default:
          case ' ':
            x += 1;
            m = (m << 1) | (m >> 7);
            break;
          case '+':
            hx = x;
            hy = y;
            /* fall through */
          case '.':
            u = wx * y + (x >> 3);
            mask[u] |= m;
            x += 1;
            m = (m << 1) | (m >> 7);
            break;
          case '|':
            x = 0;
            y += 1;
            m = 0x01;
            break;
          case '#':
            hx = x;
            hy = y;
            /* fall through */
          case '*':
            u = wx * y + (x >> 3);
            mask[u] |= m;
            img[u] |= m;
            x += 1;
            m = (m << 1) | (m >> 7);
            break;
        }
      }
      c->display->lock (c->display);
      cursor->p       = XCreateBitmapFromData (display, DefaultRootWindow (display), img, w, h);
      cursor->mask    = XCreateBitmapFromData (display, DefaultRootWindow (display), mask, w, h);
      cursor->cursor  = XCreatePixmapCursor (display, cursor->p, cursor->mask, &fg, &bg, hx, hy);
      c->display->unlock (c->display);
    }
  }

  if (cursor->cursor == None) {
    c->display->lock (c->display);
    cursor->cursor = XCreateFontCursor (display, cursor_init->x_shape);
    c->display->unlock (c->display);
  }
}

xitk_x11_cursors_t *xitk_x11_cursors_init (xitk_be_display_t *display) {
  int   i;
  xitk_x11_cursors_t *c = calloc (1, sizeof (*c));

  if (!c)
    return NULL;

  c->verbosity = display->be->verbosity;
  c->display = display;

  for (i = 1; i < xitk_cursor_num_glyphs; i++) {
    c->cursors[i].p = None;
    c->cursors[i].mask = None;
    c->cursors[i].cursor = None;
  }

  return c;
}

void xitk_x11_cursors_deinit (xitk_x11_cursors_t **p) {
  xitk_x11_cursors_t *c = *p;
  int i;

  if (!c)
    return;
  *p = NULL;

  c->display->lock (c->display);

  for (i = 0; i < xitk_cursor_num_glyphs; i++) {
    if (c->cursors[i].cursor != None) {
      if (c->verbosity >= 2)
        printf ("xitk.x11.cursor.delete (%s).\n", cursors_init[i].name);
      XFreeCursor ((Display *)c->display->id, c->cursors[i].cursor);
    }
    if (c->cursors[i].p != None)
      XFreePixmap ((Display *)c->display->id, c->cursors[i].p);
    if (c->cursors[i].mask != None)
      XFreePixmap ((Display *)c->display->id, c->cursors[i].mask);
  }

  c->display->unlock (c->display);

  free(c);
}

/* Public */
void xitk_x11_cursors_define_window_cursor (xitk_x11_cursors_t *c, xitk_be_window_t *window, xitk_cursors_t cursor) {
  if (c && window && (cursor < xitk_cursor_num_glyphs)) {
    if (cursor != xitk_cursor_default) {
      if (c->cursors[cursor].cursor == None) {
        if (c->verbosity >= 2)
          printf ("xitk.x11.cursor.new (%s).\n", cursors_init[cursor].name);
        _cursors_create_cursor (c, cursor);
      }
      c->display->lock (c->display);
      if (c->cursors[cursor].cursor != None)
        XDefineCursor ((Display *)c->display->id, (Window)window->id, c->cursors[cursor].cursor);
    } else {
      c->display->lock (c->display);
      XUndefineCursor ((Display *)c->display->id, (Window)window->id);
    }
    XSync ((Display *)c->display->id, False);
    c->display->unlock (c->display);
  }
}
