/*
 * Copyright (C) 2002 Stefan Holst
 * Copyright (C) 2025 the xine project
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA.
 *
 * utilities
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <errno.h>
#include <pthread.h>

#include "../xine-toolkit/xitk.h"
#include "list.h"
#include "utils.h"


/*
 * heap management
 */

/*
 * Heap objects are aligned on sizeof(int) boundaries
 */

#define ALIGNMENT (sizeof(int))
#define DOALIGN(num) (((num)+ALIGNMENT-1)&~(ALIGNMENT-1))

/*
 * tag datastructures
 */

typedef struct prefix_tag_s  prefix_tag_t;
typedef struct postfix_tag_s postfix_tag_t;

struct prefix_tag_s {
  prefix_tag_t *prev;         /* previous object in heap      */
  prefix_tag_t* next;         /* next object in heap          */
  postfix_tag_t* postfix;     /* ptr to postfix object        */
  const char* filename;       /* filename ptr or NULL         */
  long line;                  /* line number or 0             */
  size_t size;                /* size of allocated block      */
  void* content;              /* _gen_malloc() ptr of object  */
  char* tag;                  /* description string or NULL   */
};

struct postfix_tag_s {
  prefix_tag_t* prefix;
};

/*
 * GLOBAL: Points to first object in linked list of heap objects
 */

static prefix_tag_t* heap_head=NULL;


static void AddToLinkedList      ( prefix_tag_t* );
static void RemoveFromLinkedList ( prefix_tag_t* );
static void RenderDesc           ( prefix_tag_t*, char*, size_t );


void *_gen_malloc(size_t wSize, const char* tag, const char* lpFile, int nLine) {

  prefix_tag_t* prefix;

  wSize = DOALIGN(wSize);
  prefix=(prefix_tag_t*)malloc(sizeof(prefix_tag_t)+wSize+sizeof(postfix_tag_t));
  if (prefix) {
    AddToLinkedList( prefix );
    prefix->postfix = (postfix_tag_t*)((char*)(prefix+1)+wSize);
    prefix->postfix->prefix = prefix;
    prefix->filename = lpFile;
    prefix->line = nLine;
    prefix->content = prefix+1;
    prefix->size = wSize;
    prefix->tag = NULL;
    if (tag) prefix->tag = strdup(tag);
    memset( prefix->content, 0, wSize );
    }
  else {
    printf("utils: failed to alloc memory\n");
    abort();
  }

  return(prefix ? prefix+1 : NULL);
}


void *_gen_free(void* content) {

  if (ho_verify(content)) {

    prefix_tag_t* prefix=(prefix_tag_t*)content-1;
    size_t        wSize=(char*)(prefix->postfix+1)-(char*)prefix;

    RemoveFromLinkedList( prefix );
    free(prefix->tag);
    memset( prefix, 0, wSize );
    free(prefix);
  }

  return (NULL);
}


void *_gen_strdup(const char* lpS, const char* lpFile, int nLine) {

  void* lpReturn=NULL;

  if (lpS) {
    size_t wSize = (size_t)(strlen(lpS)+1);

    lpReturn = _gen_malloc( wSize, "strdup'ed string", lpFile, nLine );
    if (lpReturn) {
      memcpy( lpReturn, lpS, wSize );
    }
  }
  return(lpReturn);

}


void *_gen_realloc(void* lpOld, size_t wSize, const char* lpFile, int nLine) {

  void* lpNew=NULL;

  /*--- Try to realloc ---*/
  if (lpOld) {
    if (ho_verify(lpOld)) {
      prefix_tag_t* prefix=(prefix_tag_t*)lpOld-1;
      prefix_tag_t* lpNewPrefix;
      prefix_tag_t* lpPre;

      /*--- Try to reallocate block ---*/
      RemoveFromLinkedList( prefix );
      memset( prefix->postfix, 0, sizeof(postfix_tag_t) );
      wSize = DOALIGN(wSize);
      lpNewPrefix=(prefix_tag_t*)realloc(prefix,
          sizeof(prefix_tag_t)+wSize+sizeof(postfix_tag_t));

      /*--- Add new (or failed old) back in ---*/
      lpPre=(lpNewPrefix?lpNewPrefix:prefix);
      AddToLinkedList( lpPre );
      lpPre->postfix = (postfix_tag_t*)((char*)(lpPre+1)+wSize);
      lpPre->postfix->prefix = lpPre;
      lpPre->content = lpPre+1;
      lpPre->size = wSize;

      /*--- Finish ---*/
      lpNew = (lpNewPrefix ? &lpNewPrefix[1] : NULL);
      if (!lpNew) {
	printf("utils: failed to alloc memory\n");
	abort();
      }
    }
  }

  /*--- Else try new allocation ---*/
  else {
    lpNew = _gen_malloc( wSize, NULL, lpFile, nLine );
    }

  /*--- Return address to object ---*/
  return(lpNew);

}


void heapstat(void) {

  unsigned long total = 0;
  unsigned long chunks = 0;
  if (heap_head) {
    prefix_tag_t* lpCur=heap_head;

    while (ho_verify(&lpCur[1])) {
      char buffer[256];

      RenderDesc (lpCur, buffer, sizeof (buffer));
      /*--- print out buffer ---*/
      printf( "heapstat: %s\n", buffer );
      total+=lpCur->size;
      chunks++;
      lpCur = lpCur->next;
      if (lpCur==heap_head) {
        break;
      }
    }
    if (total)
      printf("heapstat: memory usage: %lu words in %lu chunks\n", total, chunks);
  }
}


static void AddToLinkedList(prefix_tag_t* lpAdd) {
    /*--- Add before current head of list ---*/
    if (heap_head) {
        lpAdd->prev = heap_head->prev;
        (lpAdd->prev)->next = lpAdd;
        lpAdd->next = heap_head;
        (lpAdd->next)->prev = lpAdd;
        }

    /*--- Else first node ---*/
    else {
        lpAdd->prev = lpAdd;
        lpAdd->next = lpAdd;
        }

    /*--- Make new item head of list ---*/
    heap_head = lpAdd;
}

static void RemoveFromLinkedList(prefix_tag_t* lpRemove) {

    /*--- Remove from doubly linked list ---*/
    (lpRemove->prev)->next = lpRemove->next;
    (lpRemove->next)->prev = lpRemove->prev;

    /*--- Possibly correct head pointer ---*/
    if (lpRemove==heap_head) {
        heap_head = ((lpRemove->next==lpRemove) ? NULL :
            lpRemove->next);
        }
}

static int ho_is_ok(void* content)
{
    return ((content) && (!((long)content&(ALIGNMENT-1))));
}


int ho_verify(void *content) {
  int bOk=0;

  if (content) {
    if (ho_is_ok(content)) {
      prefix_tag_t* prefix=(prefix_tag_t*)content-1;
      if(prefix->content==content) {
        if(prefix->postfix->prefix==prefix) {
          bOk = 1;
	}
      }
    }
  }
  return (bOk);
}

static void RenderDesc (prefix_tag_t *prefix, char *lpBuffer, size_t lpbsize) {
  if (prefix->content==&prefix[1]) {
    size_t n = snprintf (lpBuffer, lpbsize, "%zu words @ %p:", prefix->size, prefix->content);
    if (prefix->filename)
      n += snprintf (lpBuffer + n, lpbsize - n, "%s:%ld ", prefix->filename, prefix->line);
    if (prefix->tag)
      snprintf (lpBuffer + n, lpbsize - n, "%s", prefix->tag);
  }
  else {
    snprintf (lpBuffer, lpbsize, "(bad pointer given)");
  }
}
