%feature("docstring") OT::CovarianceMatrix
"Covariance (real symmetric positive definite) matrix.

Parameters
----------
size : int, :math:`n > 0`, optional
    Matrix size.
    Default is 1.
values : sequence of float with size :math:`n^2`, optional
    Values. OpenTURNS uses **column-major** ordering (like Fortran) for
    reshaping the flat list of values.
    Default creates an identity matrix.

Raises
------
TypeError : If the matrix is not symmetric.

Examples
--------
Create a matrix

>>> import openturns as ot
>>> C = ot.CovarianceMatrix(2, [1.0, 0.5, 0.5, 1.0])
>>> print(C)
[[ 1   0.5 ]
 [ 0.5 1   ]]

Get or set terms

>>> print(C[0, 1])
0.5
>>> C[0, 1] = 0.6
>>> print(C[0, 1])
0.6
>>> print(C[:, 0])
[[ 1   ]
 [ 0.6 ]]

Create an openturns matrix from a **symmetric** numpy 2d-array (or matrix, or
2d-list)...

>>> import numpy as np
>>> np_2d_array = np.array([[1.0, 0.5], [0.5, 1.0]])
>>> ot_matrix = ot.CovarianceMatrix(np_2d_array)

and back

>>> np_matrix = np.matrix(ot_matrix)"

// ---------------------------------------------------------------------

%feature("docstring") OT::CovarianceMatrix::computeCholesky
R"RAW(Compute the Cholesky factor.

The Cholesky factor of a covariance (real symmetric positive definite) matrix
:math:`\mat{C}` is the lower triangular matrix :math:`\mat{L}` such that:

.. math::

    \mat{C} = \mat{L} \Tr{\mat{L}}

Notes
-----
This uses LAPACK's `DPOTRF <http://www.netlib.org/lapack/lapack-3.1.1/html/dpotrf.f.html>`_.

Returns
-------
cholesky_factor : :class:`~openturns.TriangularMatrix`
    The left (lower) Cholesky factor.)RAW"

// ---------------------------------------------------------------------

%feature("docstring") OT::CovarianceMatrix::computeCholeskyInPlace
"Compute the Cholesky factor in place.

Similar to :meth:`computeCholesky` but modifies the matrix in place to avoid a copy.
"

// ---------------------------------------------------------------------

%feature("docstring") OT::CovarianceMatrix::computeRegularizedCholesky
"Compute the regularized Cholesky factor.

Similar to :meth:`computeCholesky` but with a regularization loop according
to the largest eigenvalue and keys `Matrix-StartingScaling` and `Matrix-MaximalScaling`.

Returns
-------
cholesky_factor : :class:`~openturns.TriangularMatrix`
    The left (lower) Cholesky factor."

// ---------------------------------------------------------------------

%feature("docstring") OT::CovarianceMatrix::isPositiveDefinite
R"RAW(Test whether the matrix is positive definite or not.

A matrix :math:`\mat{M}` is positive definite if :math:`\Tr{\vect{z}} \mat{M} \vect{z}`
is positive for every compatible non-zero column vector :math:`\vect{z}`.

Notes
-----
This uses LAPACK's `DPOTRF <http://www.netlib.org/lapack/lapack-3.1.1/html/dpotrf.f.html>`_.

Returns
-------
test : bool
    Answer.)RAW"
