/*************************************************************************
**
**    This file is part of the 'forwords' educational programm.
**    Copyright (C) 2024  Alexander Fomin
**
**    This program is free software: you can redistribute it and/or modify
**    it under the terms of the GNU General Public License as published by
**    the Free Software Foundation, either version 3 of the License, or
**    (at your option) any later version.
**
**    This program is distributed in the hope that it will be useful,
**    but WITHOUT ANY WARRANTY; without even the implied warranty of
**    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**    GNU General Public License for more details.
**
**    You should have received a copy of the GNU General Public License
**    along with this program.  If not, see <https://www.gnu.org/licenses/>.
**
**    Contact: fomin_alex@yahoo.com
**
**************************************************************************/

#include "dlgsettings.h"
#include "qmessagebox.h"
#include "ui_dlgsettings.h"
#include "common.h"

dlgSettings::dlgSettings(QWidget *parent) :
    QDialog(parent),
    ui(new Ui::dlgSettings)
{
    vVoiceChangedActionOff = true;
    ui->setupUi(this);
}

dlgSettings::~dlgSettings()
{
    delete ui;
}

void dlgSettings::Open(QJsonObject config, const QStringList *ttsEng)
{
    ui->lbWorkfolder->setText(config.value("Path").toString());
    vConfig = config;
    if(ttsEng!=nullptr)
        setTtsEngines(ttsEng);
    fSetLanguages();
    on_cbSpeechEngine_currentTextChanged(ui->cbSpeechEngine->currentText());
    this->show();
}

void dlgSettings::fSetLanguages()
{
    ui->cbLanguage->clear();
    ui->cbLanguage->addItems(fDirList(QDir(ui->lbWorkfolder->text())));
}

void dlgSettings::fSetTopics(QString s)
{
    ui->cbTopic->clear();
    ui->cbTopic->addItems(fFileList(ui->lbWorkfolder->text()+"/"+s,"tof"));
}

void dlgSettings::GetUiText(QJsonObject *uitext)
{
    QJsonObject jobj;
    jobj.insert("workFolder",QJsonValue(ui->lbWorkFoderHead->text()));
    jobj.insert("language",QJsonValue(ui->lbLanguage->text()));
    jobj.insert("topic",QJsonValue(ui->lbTopic->text()));
    jobj.insert("speechEngine",QJsonValue(ui->lbSpeechEngine->text()));
    jobj.insert("speechSettings",QJsonValue(ui->lbSpeechSettings->text()));
    jobj.insert("voice",QJsonValue(ui->lbVoice->text()));
    jobj.insert("delLanguage",QJsonValue(ui->pbDeleteLanguage->text()));
    jobj.insert("delTopic",QJsonValue(ui->pbDeleteTopic->text()));
    jobj.insert("speechTest",QJsonValue(ui->pbSpeechTest->text()));
    jobj.insert("save",QJsonValue(ui->pbSave->text()));
    jobj.insert("exit",QJsonValue(ui->pbExit->text()));
    uitext->insert("dlgSettings",jobj);
}

void dlgSettings::SetUiText(QJsonObject *uitext)
{
    QJsonObject jobj = uitext->value("dlgSettings").toObject();
    ui->lbWorkFoderHead->setText(jobj.value("workFolder").toString());
    ui->lbLanguage->setText(jobj.value("language").toString());
    ui->lbTopic->setText(jobj.value("topic").toString());
    ui->lbSpeechEngine->setText(jobj.value("speechEngine").toString());
    ui->lbSpeechSettings->setText(jobj.value("speechSettings").toString());
    ui->lbVoice->setText(jobj.value("voice").toString());
    ui->pbDeleteLanguage->setText(jobj.value("delLanguage").toString());
    ui->pbDeleteTopic->setText(jobj.value("delTopic").toString());
    ui->pbSpeechTest->setText(jobj.value("speechTest").toString());
    ui->pbSave->setText(jobj.value("save").toString());
    ui->pbExit->setText(jobj.value("exit").toString());
}

void dlgSettings::setTtsEngines(const QStringList *engine)
{
    ui->cbSpeechEngine->clear();
    ui->cbSpeechEngine->addItems(*engine);
}

void dlgSettings::SetTtsVoices(const QStringList *voices)
{
    vVoiceChangedActionOff = true;
    ui->cbSpeechVoice->clear();
    vVoiceChangedActionOff = true;
    ui->cbSpeechVoice->addItems(*voices);
    ui->cbSpeechVoice->setCurrentText(vConfig.value("SpeechConfig").toObject().
                                      value(ui->cbLanguage->currentText()).toObject().
                                      value("voice").toString());
}

void dlgSettings::on_pbWorkFolder_clicked()
{
    QFileDialog fdlg;
    QString s = vConfig.value("Path").toString();
    if(!QDir(s).exists() || s == "")
        s = QDir().homePath();
    fdlg.setDirectory(s);
    fdlg.setFileMode(QFileDialog::Directory);
    fdlg.setOptions(QFileDialog::ShowDirsOnly);
    if (fdlg.exec())
    {
        s = fdlg.directory().path();
        if(s.startsWith(QDir().homePath()))
            ui->lbWorkfolder->setText(s);
    }
}

void dlgSettings::on_pbDeleteLanguage_clicked()
{
    if(!QDir(ui->lbWorkfolder->text()).rmdir(ui->cbLanguage->currentText()))
    {
        fShowMsg("Folder isn't empty.\nDelete all topics inside first.");
        return;
    }
    else
    {
        fSetLanguages();
        emit config_changed(vConfig);
    }
}

void dlgSettings::on_pbDeleteTopic_clicked()
{
    QString s = ui->lbWorkfolder->text()+"/"+ui->cbLanguage->currentText();
    QDir(s).remove(ui->cbTopic->currentText()+".tof");
    fSetTopics(ui->cbLanguage->currentText());
    emit config_changed(vConfig);
}

 void dlgSettings::on_cbLanguage_currentTextChanged(const QString &arg1)
{
    if(ui->cbLanguage->count()>0)
    {
        //topics
        fSetTopics(arg1);

        //speech engine
        QJsonObject speechConf = vConfig.value("SpeechConfig").toObject().value(arg1).toObject();
        QString s = speechConf.value("engine").toString();
        ui->cbSpeechEngine->setCurrentText(s);

        //speech voice
        if(ui->cbSpeechEngine->currentText()==s)
        {
            s = speechConf.value("voice").toString();
            vVoiceChangedActionOff = true;
            ui->cbSpeechVoice->setCurrentText(s);
        }
    }
}

void dlgSettings::on_cbSpeechEngine_currentTextChanged(const QString &arg1)
{
    if(ui->cbSpeechEngine->count()>0)
        emit ttsEngineChanged(arg1);
}

void dlgSettings::on_pbSpeechTest_clicked()
{
    QString s = ui->leSpeechTest->text();
    emit textToSpeech(s, ui->cbSpeechEngine->currentText(), ui->cbSpeechVoice->currentText());
}

void dlgSettings::on_pbSave_clicked()
{
    vConfig.insert("Path",QJsonValue(ui->lbWorkfolder->text()));
    emit config_changed(vConfig);
    this->hide();
}

void dlgSettings::on_pbExit_clicked()
{
    this->hide();
}

void dlgSettings::on_cbSpeechVoice_currentTextChanged(const QString &arg1)
{
    if(!vVoiceChangedActionOff)
    {
        QJsonObject josc = vConfig.value("SpeechConfig").toObject();
        QJsonObject jobj = josc.value(ui->cbLanguage->currentText()).toObject();
        jobj.insert("engine", QJsonValue(ui->cbSpeechEngine->currentText()));
        jobj.insert("voice", QJsonValue(arg1));
        josc.insert(ui->cbLanguage->currentText(), QJsonValue(jobj));
        vConfig.insert("SpeechConfig",QJsonValue(josc));
    }
    vVoiceChangedActionOff = false;
}

