/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

/*
 * file-to-tape. This reads a file created by dump and writes it to a faketape
 * recreating the blocking as if it had been written by dump
 */

#include <config.h>
#include "faketape-lib.h"

#include <optional>
#include <algorithm>
#include <fcntl.h>
#include <getopt.h>

using Response = FakeTape::Response;

struct CMD {
	std::string cmd;
	Response (*callback)(FakeTape&, const std::string&, std::optional<off_t>);
};

static Response asf(FakeTape&, const std::string&, std::optional<off_t>);
static Response bsf(FakeTape&, const std::string&, std::optional<off_t>);
static Response bsfm(FakeTape&, const std::string&, std::optional<off_t>);
static Response bsr(FakeTape&, const std::string&, std::optional<off_t>);
static Response bss(FakeTape&, const std::string&, std::optional<off_t>);
static Response compression(FakeTape&, const std::string&, std::optional<off_t>);
static Response datasize(FakeTape&, const std::string&, std::optional<off_t>);
static Response defblksize(FakeTape&, const std::string&, std::optional<off_t>);
static Response defcompression(FakeTape&, const std::string&, std::optional<off_t>);
static Response defdensity(FakeTape&, const std::string&, std::optional<off_t>);
static Response defdrvbuffer(FakeTape&, const std::string&, std::optional<off_t>);
static Response densities(FakeTape&, const std::string&, std::optional<off_t>);
static Response drvbuffer(FakeTape&, const std::string&, std::optional<off_t>);
static Response eject(FakeTape&, const std::string&, std::optional<off_t>);
static Response eod(FakeTape&, const std::string&, std::optional<off_t>);
static Response eof(FakeTape&, const std::string&, std::optional<off_t>);
static Response erase(FakeTape&, const std::string&, std::optional<off_t>);
static Response fsf(FakeTape&, const std::string&, std::optional<off_t>);
static Response fsfm(FakeTape&, const std::string&, std::optional<off_t>);
static Response fsr(FakeTape&, const std::string&, std::optional<off_t>);
static Response fss(FakeTape&, const std::string&, std::optional<off_t>);
static Response load(FakeTape&, const std::string&, std::optional<off_t>);
static Response lock(FakeTape&, const std::string&, std::optional<off_t>);
static Response mkpartition(FakeTape&, const std::string&, std::optional<off_t>);
static Response offline(FakeTape&, const std::string&, std::optional<off_t>);
static Response partseek(FakeTape&, const std::string&, std::optional<off_t>);
static Response readdata(FakeTape&, const std::string&, std::optional<off_t>);
static Response retension(FakeTape&, const std::string&, std::optional<off_t>);
static Response rewind(FakeTape&, const std::string&, std::optional<off_t>);
static Response rewoffl(FakeTape&, const std::string&, std::optional<off_t>);
static Response seek(FakeTape&, const std::string&, std::optional<off_t>);
static Response seod(FakeTape&, const std::string&, std::optional<off_t>);
static Response setblk(FakeTape&, const std::string&, std::optional<off_t>);
static Response setdensity(FakeTape&, const std::string&, std::optional<off_t>);
static Response setpartition(FakeTape&, const std::string&, std::optional<off_t>);
static Response status(FakeTape&, const std::string&, std::optional<off_t>);
static Response stclearoptions(FakeTape&, const std::string&, std::optional<off_t>);
static Response stlongtimeout(FakeTape&, const std::string&, std::optional<off_t>);
static Response stoptions(FakeTape&, const std::string&, std::optional<off_t>);
static Response stsetcln(FakeTape&, const std::string&, std::optional<off_t>);
static Response stsetoptions(FakeTape&, const std::string&, std::optional<off_t>);
static Response stshowoptions(FakeTape&, const std::string&, std::optional<off_t>);
static Response sttimeout(FakeTape&, const std::string&, std::optional<off_t>);
static Response stwrthreshold(FakeTape&, const std::string&, std::optional<off_t>);
static Response tell(FakeTape&, const std::string&, std::optional<off_t>);
static Response unlock(FakeTape&, const std::string&, std::optional<off_t>);
static Response weof(FakeTape&, const std::string&, std::optional<off_t>);
static Response writedata(FakeTape&, const std::string&, std::optional<off_t>);
static Response wset(FakeTape&, const std::string&, std::optional<off_t>);

const std::vector<CMD> cmds = {
	{"asf", asf },
	{"bsf", bsf },
	{"bsfm", bsfm },
	{"bsr", bsr },
	{"bss", bss },
	{"compression", compression },
	{"datasize", datasize },
	{"defblksize", defblksize },
	{"defcompression", defcompression },
	{"defdensity", defdensity },
	{"defdrvbuffer", defdrvbuffer },
	{"densities", densities },
	{"drvbuffer", drvbuffer },
	{"eject", eject },
	{"eod", eod },
	{"eof", eof },
	{"erase", erase },
	{"fsf", fsf },
	{"fsfm", fsfm },
	{"fsr", fsr },
	{"fss", fss },
	{"load", load },
	{"lock", lock },
	{"mkpartition", mkpartition },
	{"offline", offline },
	{"partseek", partseek },
	{"readdata", readdata },
	{"retension", retension },
	{"rewind", rewind },
	{"rewoffl", rewoffl },
	{"seek", seek },
	{"seod", seod },
	{"setblk", setblk },
	{"setdensity", setdensity },
	{"setpartition", setpartition },
	{"status", status },
	{"stclearoptions", stclearoptions },
	{"stlongtimeout", stlongtimeout },
	{"stoptions", stoptions },
	{"stsetcln", stsetcln },
	{"stsetoptions", stsetoptions },
	{"stshowoptions", stshowoptions },
	{"sttimeout", sttimeout },
	{"stwrthreshold", stwrthreshold },
	{"tell", tell },
	{"unlock", unlock },
	{"weof", weof },
	{"writedata", writedata },
	{"wset", wset },
};

void usage(const char* prog) {
	std::cout << std::endl;
	std::cout << "Usage: " << prog << "[--debug/-d] [--version/-v] [-h] -f <tape> command [ count ]" << std::endl;
	std::cout << "commands: ";
	size_t len = 0;
	for(const auto& c : cmds) {
		if (len == 0) {
			std::cout << c.cmd;
			len=10 + c.cmd.size();
		} else if (len + c.cmd.size() + 3 > 80) {
			std::cout << std::endl << "          " << c.cmd;
			len=10 + c.cmd.size();
		} else {
			std::cout << ", " << c.cmd;
			len += 3 + c.cmd.size();
		}
	}
	std::cout << std::endl;
	exit(1);
}

int main(int argc, char* argv[]) {

	const char* prog = argv[0];

	int ch;

	const char* opts = "dhf:v";
	struct option lopts[] = {
		{ .name = "version", .has_arg = 0, .flag = nullptr, .val = 'v' },
		{ .name = "debug", .has_arg = 0, .flag = nullptr, .val = 'd' },
		{},
	};
	std::string tapedev;
	FakeTape tape;

	while (( ch = getopt_long(argc, argv, opts, lopts, nullptr)) != -1)
		switch(ch) {
			case 'd':
				tape.debug = std::make_unique<std::ostream>(std::cerr.rdbuf());
				tape.debug->copyfmt(std::cerr);
				tape.debug->clear(std::cout.rdstate());
				break;
			case 'v':
				std::cout << prog << " version 1.0" << std::endl;
				break;
			case 'f':
				tapedev = optarg;
				break;
			default:
				usage(prog);
		}
	argc -= optind;
	argv += optind;

	if (argc < 1)
		usage(prog);

	std::optional<off_t> count = std::nullopt;
	if (argc == 2)
		count = atoll(argv[1]);

	auto it = std::find_if(std::begin(cmds), std::end(cmds), [argv](const CMD& s){ return std::string(s.cmd)==argv[0]; });
	if (it == std::end(cmds)) {
		std::cerr << "unrecognised command: " << argv[0] << std::endl;
		usage(prog);
	}
	if (it->callback(tape, tapedev, count) == Response::ERROR)
		std::cerr << prog << " " << argv[0] << ":Error" << std::endl;
}

static Response unimplemented(const std::string& cmd) {
	std::cerr << "'" << cmd << "' unimplemented'." << std::endl;
	return Response::ERROR;
}

static Response extraarg(const std::string& cmd) {
	std::cerr << "Too many arguments for the command '" << cmd << "'." << std::endl;
	return Response::ERROR;
}

#define TAPECMDNOFATAL(x) do { if(x == Response::FATAL) return Response::ERROR; } while(0)
#define TAPECMDNOERROR(x) do { if(x != Response::OK) return Response::ERROR; } while(0)

static Response fsf_impl(FakeTape& tape, off_t count) {
	while(count--) {
		uint32_t fno = tape.getFileNo();
		do {
			TAPECMDNOERROR(tape.fsr(1));
		} while(fno == tape.getFileNo());
	}
	return Response::OK;
}

static Response bsf_impl(FakeTape& tape, off_t count) {
	while(count--) {
		uint32_t fno = tape.getFileNo();
		do {
			TAPECMDNOERROR(tape.bsr(1));
		} while(fno == tape.getFileNo());
	}
	return Response::OK;
}

static Response asf(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	TAPECMDNOERROR(tape.rewind());
	TAPECMDNOERROR(fsf_impl(tape, count.has_value()?*count:1));
	return Response::OK;
}

static Response bsf(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	return bsf_impl(tape, count.has_value()?*count:1);
}

static Response bsfm(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	TAPECMDNOERROR(bsf_impl(tape, count.has_value()?*count:1));
	TAPECMDNOERROR(tape.fsr(1));
	return Response::OK;
}

static Response bsr(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	TAPECMDNOERROR(tape.bsr(count.has_value()?*count:1));
	return Response::OK;
}

static Response bss(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	return bsr(tape, tapedev, count);
}

static Response compression(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("compression");
}

static Response datasize(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("datasize");
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	std::vector<uint8_t> data(FakeTape::getMaxBlockSize());
	size_t rl;
	TAPECMDNOERROR(tape.readData(data.data(), data.size(), &rl));
	std::cout << rl << std::endl;
	TAPECMDNOERROR(tape.bsr(1));
	return Response::OK;
}

static Response defblksize(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("defblksize");
}

static Response defcompression(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("defcompression");
}

static Response defdensity(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("defdensity");
}

static Response defdrvbuffer(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("defdrvbuffer");
}

static Response densities(FakeTape&, const std::string&, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("densities");

	std::cout << "Some SCSI tape density codes:" << std::endl;
	std::cout << "code   explanation                   code   explanation" << std::endl;
	std::cout << "0x00   default                       0x33   SLR6" << std::endl;
	std::cout << "0x01   NRZI (800 bpi)                0x34   SLR100" << std::endl;
	std::cout << "0x02   PE (1600 bpi)                 0x40   DLT1 40 GB, or Ultrium" << std::endl;
	std::cout << "0x03   GCR (6250 bpi)                0x41   DLT 40GB, or Ultrium2" << std::endl;
	std::cout << "0x04   QIC-11                        0x42   LTO-2" << std::endl;
	std::cout << "0x05   QIC-45/60 (GCR, 8000 bpi)     0x44   LTO-3" << std::endl;
	std::cout << "0x06   PE (3200 bpi)                 0x45   QIC-3095-MC (TR-4)" << std::endl;
	std::cout << "0x07   IMFM (6400 bpi)               0x46   LTO-4" << std::endl;
	std::cout << "0x08   GCR (8000 bpi)                0x47   DDS-5 or TR-5" << std::endl;
	std::cout << "0x09   GCR (37871 bpi)               0x48   SDLT220" << std::endl;
	std::cout << "0x0a   MFM (6667 bpi)                0x49   SDLT320" << std::endl;
	std::cout << "0x0b   PE (1600 bpi)                 0x4a   SDLT600, T10000A" << std::endl;
	std::cout << "0x0c   GCR (12960 bpi)               0x4b   T10000B" << std::endl;
	std::cout << "0x0d   GCR (25380 bpi)               0x4c   T10000C" << std::endl;
	std::cout << "0x0f   QIC-120 (GCR 10000 bpi)       0x4d   T10000D" << std::endl;
	std::cout << "0x10   QIC-150/250 (GCR 10000 bpi)   0x51   IBM 3592 J1A" << std::endl;
	std::cout << "0x11   QIC-320/525 (GCR 16000 bpi)   0x52   IBM 3592 E05" << std::endl;
	std::cout << "0x12   QIC-1350 (RLL 51667 bpi)      0x53   IBM 3592 E06" << std::endl;
	std::cout << "0x13   DDS (61000 bpi)               0x54   IBM 3592 E07" << std::endl;
	std::cout << "0x14   EXB-8200 (RLL 43245 bpi)      0x55   IBM 3592 E08" << std::endl;
	std::cout << "0x15   EXB-8500 or QIC-1000          0x58   LTO-5" << std::endl;
	std::cout << "0x16   MFM 10000 bpi                 0x5a   LTO-6" << std::endl;
	std::cout << "0x17   MFM 42500 bpi                 0x5c   LTO-7" << std::endl;
	std::cout << "0x18   TZ86                          0x5d   LTO-7-M8" << std::endl;
	std::cout << "0x19   DLT 10GB                      0x5e   LTO-8" << std::endl;
	std::cout << "0x1a   DLT 20GB                      0x71   IBM 3592 J1A, encrypted" << std::endl;
	std::cout << "0x1b   DLT 35GB                      0x72   IBM 3592 E05, encrypted" << std::endl;
	std::cout << "0x1c   QIC-385M                      0x73   IBM 3592 E06, encrypted" << std::endl;
	std::cout << "0x1d   QIC-410M                      0x74   IBM 3592 E07, encrypted" << std::endl;
	std::cout << "0x1e   QIC-1000C                     0x75   IBM 3592 E08, encrypted" << std::endl;
	std::cout << "0x1f   QIC-2100C                     0x80   DLT 15GB uncomp. or Ecrix" << std::endl;
	std::cout << "0x20   QIC-6GB                       0x81   DLT 15GB compressed" << std::endl;
	std::cout << "0x21   QIC-20GB                      0x82   DLT 20GB uncompressed" << std::endl;
	std::cout << "0x22   QIC-2GB                       0x83   DLT 20GB compressed" << std::endl;
	std::cout << "0x23   QIC-875                       0x84   DLT 35GB uncompressed" << std::endl;
	std::cout << "0x24   DDS-2                         0x85   DLT 35GB compressed" << std::endl;
	std::cout << "0x25   DDS-3                         0x86   DLT1 40 GB uncompressed" << std::endl;
	std::cout << "0x26   DDS-4 or QIC-4GB              0x87   DLT1 40 GB compressed" << std::endl;
	std::cout << "0x27   Exabyte Mammoth               0x88   DLT 40GB uncompressed" << std::endl;
	std::cout << "0x28   Exabyte Mammoth-2             0x89   DLT 40GB compressed" << std::endl;
	std::cout << "0x29   QIC-3080MC, IBM 3590 B        0x8c   EXB-8505 compressed" << std::endl;
	std::cout << "0x2a   IBM 3590 E                    0x90   SDLT110 uncompr/EXB-8205 compr" << std::endl;
	std::cout << "0x30   AIT-1 or MLR3                 0x91   SDLT110 compressed" << std::endl;
	std::cout << "0x31   AIT-2                         0x92   SDLT160 uncompressed" << std::endl;
	std::cout << "0x32   AIT-3 or SLR7                 0x93   SDLT160 comprssed" << std::endl;
	return Response::OK;
}

static Response drvbuffer(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("drvbuffer");
}

static Response eject(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("eject");
	return rewind(tape, tapedev, count);
}

static Response eod(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("eod");
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	size_t seek = (std::numeric_limits<size_t>::max()>>1)+1;
	while (seek) {
		TAPECMDNOFATAL(tape.fsr(seek));
		seek >>= 1;
	}
	return Response::OK;
}

static Response eof(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	int c = 1;
	if(count.has_value())
		c = *count;
	TAPECMDNOERROR(tape.open(tapedev, O_WRONLY));
	while(c--)
		TAPECMDNOERROR(tape.writeData(nullptr, 0));
	return Response::OK;
}

static Response erase(FakeTape& tape, const std::string& tapedev, std::optional<off_t>) {
	TAPECMDNOERROR(tape.open(tapedev, O_WRONLY|O_CREAT));
	TAPECMDNOERROR(tape.erase());
	return Response::OK;
}

static Response fsf(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	return fsf_impl(tape, count.has_value()?*count:1);
}

static Response fsfm(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	TAPECMDNOERROR(fsf_impl(tape, count.has_value()?*count:1));
	TAPECMDNOERROR(tape.bsr(1));
	return Response::OK;
}

static Response fsr(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	TAPECMDNOERROR(tape.fsr(count.has_value()?*count:1));
	return Response::OK;
}

static Response fss(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	return fsr(tape, tapedev, count);
}

static Response load(FakeTape& tape, const std::string& tapedev, std::optional<off_t>) {
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	return Response::OK;
}

static Response lock(FakeTape&, const std::string&, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("lock");
	return unimplemented("lock");
}

static Response mkpartition(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("mkpartition");
}

static Response offline(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("offline");
	return rewind(tape, tapedev, count);
}

static Response readdata(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("readdata");
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	std::vector<uint8_t> data(FakeTape::getMaxBlockSize());
	size_t rl;
	TAPECMDNOERROR(tape.readData(data.data(), data.size(), &rl));
	if (rl == 0)
		return Response::OK;
	if (write(1, data.data(), rl) != (ssize_t)rl)
		return Response::ERROR;
	return Response::OK;
}

static Response partseek(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("partseek");
}

static Response retension(FakeTape&, const std::string&, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("retension");
	return unimplemented("retension");
}

static Response rewind(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("rewind");
	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	TAPECMDNOERROR(tape.rewind());
	return Response::OK;
}

static Response rewoffl(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("rewoffl");
	return rewind(tape, tapedev, count);
}

static Response seek(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("seek");
}

static Response seod(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("seod");
	return eod(tape, tapedev, count);
}

static Response setblk(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("setblk");
}

static Response setdensity(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("setdensity");
}

static Response setpartition(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("setpartition");
}

static Response status(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("status");

	TAPECMDNOERROR(tape.open(tapedev, O_RDONLY));
	std::cout << "SCSI 2 tape drive:" << std::endl;
	std::cout << "File number=" << tape.getFileNo() << ", block number=" << tape.getFileBlock() << ", partition=0." << std::endl;
	std::cout << "Tape block size 0 bytes. Density code 0x5e (LTO-8)." << std::endl;
	std::cout << "Soft error count since last status=0" << std::endl;
	std::cout << "General status bits on (81010000):" << std::endl;
	std::cout << " EOF ONLINE IM_REP_EN" << std::endl;
	return Response::OK;
}

static Response stclearoptions(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("stclearoptions");
}

static Response stlongtimeout(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("stlongtimeout");
}

static Response stoptions(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("stoptions");
}

static Response stsetcln(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("stsetcln");
}

static Response stsetoptions(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("stsetoptions");
}

static Response stshowoptions(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("stshowoptions");
}

static Response sttimeout(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("sttimeout");
}

static Response stwrthreshold(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("stwrthreshold");
}

static Response tell(FakeTape&, const std::string&, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("tell");
	return unimplemented("tell");
}

static Response unlock(FakeTape&, const std::string&, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("unlock");
	return unimplemented("unlock");
}

static Response weof(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	return eof(tape, tapedev, count);
}

static Response writedata(FakeTape& tape, const std::string& tapedev, std::optional<off_t> count) {
	if(count.has_value()) return extraarg("writedata");
	TAPECMDNOERROR(tape.open(tapedev, O_WRONLY));
	std::vector<uint8_t> data(FakeTape::getMaxBlockSize());
	ssize_t rl;
	rl = read(0, data.data(), data.size());
	if (rl <= 0)
		return Response::ERROR;
	TAPECMDNOERROR(tape.writeData(data.data(), rl));
	return Response::OK;
}

static Response wset(FakeTape&, const std::string&, std::optional<off_t>) {
	return unimplemented("wset");
}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
